//==============================================================================================//
//              DSP.C                                                                           //
//=============================================================================================c//

#ifndef dsp_C_
#define dsp_C_

#include "dsp.h"

//==============================================================================================//
//              FIRST-ORDER LOW-PASS FILTER                                                     //
//=============================================================================================c//

_Bool DSP_lp1Init(volatile DSP_lp1_S *lp1)
{
    // Check parameters
    if (lp1->K==0 || lp1->WC<=0 || lp1->TS<=0)
        return ERR_SET;

    // Reset filter
    DSP_lp1Reset(lp1);

    switch (lp1->DTYPE)
    {
        case DSP_DTYPE_ZOH :
            lp1->A1 = -expf(-lp1->WC*lp1->TS);
            lp1->B0 = 0;
            lp1->B1 = lp1->K * (1.0f + lp1->A1);
            return ERR_CLEAR;

        case DSP_DTYPE_TUSTIN :
            lp1->B0 = lp1->K / (1.0f + 2.0f/(lp1->WC*lp1->TS));
            lp1->B1 = lp1->B0;
            lp1->A1 = (1.0f - 2.0f/(1.0f + (lp1->WC*lp1->TS)/2.0f));
            return ERR_CLEAR;

        default :
            return ERR_SET;
    }
}

void DSP_lp1Reset(volatile DSP_lp1_S *lp1)
{
    lp1->u1 = 0.0f;
    lp1->y1 = 0.0f;
}

//==============================================================================================//
//              SECOND-ORDER LOW-PASS BUTTERWORTH FILTER                                        //
//=============================================================================================c//

_Bool DSP_bw2Init(volatile DSP_bw2_S *bw2)
{
    // Check parameters
    if (bw2->K==0 || bw2->WC<=0 || bw2->TS<=0)
        return ERR_SET;

    // Reset filter
    DSP_bw2Reset(bw2);

    // Calculate temporary variables
    fp32_t WC_TS_1 = bw2->WC * bw2->TS;
    fp32_t WC_TS_2 = WC_TS_1 * WC_TS_1;
    fp32_t DEN = 4.0f + 2.0f*UTILS_SQRT2*WC_TS_1 + WC_TS_2;

    switch (bw2->DTYPE)
    {
        case DSP_DTYPE_TUSTIN :
            bw2->B0 = bw2->K * WC_TS_2 / DEN;
            bw2->B1 = 2.0f*bw2->B0;
            bw2->B2 = bw2->B0;
            bw2->A1 = (2.0*WC_TS_2 - 8.0f) / DEN;
            bw2->A2 = (DEN - 4.0f*UTILS_SQRT2*WC_TS_1) / DEN;
            return ERR_CLEAR;

        default :
            return ERR_SET;
    }
}

void DSP_bw2Reset(volatile DSP_bw2_S *bw2)
{
    bw2->u1 = 0.0f;
    bw2->u2 = 0.0f;
    bw2->y1 = 0.0f;
    bw2->y2 = 0.0f;
}

//==============================================================================================//
//              MOVING AVERAGE FILTER                                                           //
//=============================================================================================c//

_Bool DSP_mavInit(volatile DSP_mav_S *mav)
{
    // Check parameters
    if (mav->bufferSize==0 || mav->buffer==0)
        return ERR_SET;

    // Reset moving average filter
    DSP_mavReset(mav);

    return ERR_CLEAR;
}

void DSP_mavReset(volatile DSP_mav_S *mav)
{
    // Circular buffer iterator
    uint16_t i;

    // Reset the circular buffer
    for (i=0; i<mav->bufferSize; i++)
    {
        *(mav->buffer + i) = 0.0f;
    }

    // Reset position of the oldest element
    mav->indOldest = 0;

    // Reset moving average sum
    mav->sum = 0;
}

//==============================================================================================//
//              RATE LIMITER                                                                    //
//=============================================================================================c//

_Bool DSP_rlimInit(volatile DSP_rlim_S *rlim)
{
    // Check parameters
    if (rlim->RATE_POS<=0 || rlim->RATE_NEG<=0 || rlim->TS<=0)
        return ERR_SET;

    // Calculate positive and negative rate of change per sample time, i.e.,
    // maximum allowed increments and decrements per sample time period
    rlim->INC_POS = rlim->RATE_POS * rlim->TS;
    rlim->INC_NEG = rlim->RATE_NEG * rlim->TS;

    // Reset filter
    DSP_rlimReset(rlim);

    return ERR_CLEAR;
}

void DSP_rlimReset(volatile DSP_rlim_S *rlim)
{
    rlim->y1 = 0.0f;
}

//==============================================================================================//
//              MEASUREMENTS STATISTICS                                                         //
//=============================================================================================c//

_Bool DSP_statInit(volatile DSP_stat_S *stat)
{
    // Check parameters
    if (stat->N <= 1)
        return ERR_SET;

    // Reset statistics structure
    DSP_statReset(stat);

    return ERR_CLEAR;
}

void DSP_statReset(volatile DSP_stat_S *stat)
{
    stat->sum    = 0;
    stat->sumSqr = 0;
    stat->mean   = 0.0f;
    stat->var    = 0.0f;
    stat->std    = 0.0f;
}

#endif /* DSP_C_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
