//==============================================================================================//
//              DSP.H                                                                           //
//=============================================================================================c//

#ifndef DSP_H_
#define DSP_H_

// TODO consider to use const for some parameters

#include <math.h>

#include "utils.h"

#include "error.h"

//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

//
// DISCRETIZATION TYPE FLAGS
//

#define DSP_DTYPE_IMPULSE           1
#define DSP_DTYPE_ZOH               2
#define DSP_DTYPE_EULER_FORWARD     3
#define DSP_DTYPE_EULER_BACKWARD    4
#define DSP_DTYPE_TUSTIN            5

//==============================================================================================//
//              FIRST-ORDER LOW-PASS FILTER                                                     //
//=============================================================================================c//

typedef struct DSP_lp1_STRUCT
{
    uint16_t DTYPE; // Discretization type flag
    fp32_t K;       // Filter gain (?)
    fp32_t WC;      // Filter cutoff frequency (rad/s)
    fp32_t TS;      // Filter sample time (s)
    fp32_t B0;      // Filter parameter for u[k]
    fp32_t B1;      // Filter parameter for u[k-1]
    fp32_t A1;      // Filter parameter for y[k-1]
    fp32_t u1;      // Filter state u[k-1]
    fp32_t y1;      // Filter state y[k-1]
} DSP_lp1_S;

_Bool DSP_lp1Init(volatile DSP_lp1_S *lp1);
void DSP_lp1Reset(volatile DSP_lp1_S *lp1);
UTILS_INLINE fp32_t DSP_lp1Update(fp32_t u0, volatile DSP_lp1_S *lp1);

UTILS_INLINE fp32_t DSP_lp1Update(fp32_t u0, volatile DSP_lp1_S *lp1)
{
    fp32_t y0;

    switch (lp1->DTYPE)
    {
        case DSP_DTYPE_ZOH :
            y0 = lp1->B1*lp1->u1 - lp1->A1*lp1->y1;
            break;

        case DSP_DTYPE_TUSTIN :
            y0 = lp1->B0*(u0+lp1->u1) - lp1->A1*lp1->y1;
            break;

        default :
            return u0;
    }

    // Update filter states
    lp1->u1 = u0;
    lp1->y1 = y0;

    // Return filtered value
    return y0;
}

//==============================================================================================//
//              SECOND-ORDER LOW-PASS BUTTERWORTH FILTER                                        //
//=============================================================================================c//

// TODO check equations one more time!

typedef struct DSP_bw2_STRUCT
{
    uint16_t DTYPE; // Discretization type flag
    fp32_t K;       // Filter gain
    fp32_t WC;      // Filter cutoff frequency (rad/s)
    fp32_t TS;      // Filter sample time (s)
    fp32_t B0;      // Filter parameter for u[k]
    fp32_t B1;      // Filter parameter for u[k-1]
    fp32_t B2;      // Filter parameter for u[k-2]
    fp32_t A1;      // Filter parameter for y[k-1]
    fp32_t A2;      // Filter parameter for y[k-2]
    fp32_t u1;      // Filter state u[k-1]
    fp32_t u2;      // Filter state u[k-2]
    fp32_t y1;      // Filter state y[k-1]
    fp32_t y2;      // Filter state y[k-2]
} DSP_bw2_S;

_Bool DSP_bw2Init(volatile DSP_bw2_S *bw2);
void DSP_bw2Reset(volatile DSP_bw2_S *bw2);
UTILS_INLINE fp32_t DSP_bw2Update(fp32_t u0, volatile DSP_bw2_S *bw2);

UTILS_INLINE fp32_t DSP_bw2Update(fp32_t u0, volatile DSP_bw2_S *bw2)
{
    fp32_t y0;

    switch (bw2->DTYPE)
    {
        case DSP_DTYPE_TUSTIN :
            y0 = bw2->B0*(u0+2*bw2->u1+bw2->u2) - bw2->A1*bw2->y1 - bw2->A2*bw2->y2;
            break;

        default :
            return u0;
    }

    // Update filter states
    bw2->y2 = bw2->y1;
    bw2->y1 = y0;
    bw2->u2 = bw2->u1;
    bw2->u1 = u0;

    // Return filtered value
    return y0;
}

//==============================================================================================//
//              MOVING AVERAGE FILTER                                                           //
//=============================================================================================c//

typedef struct DSP_mav_STRUCT
{
    volatile fp32_t *buffer;    // Circular buffer
    uint16_t bufferSize;        // Circular buffer size
    uint16_t indOldest;         // Position of the oldest element in the circular buffer
    fp32_t sum;                 // Sum of all elements in the circular buffer
} DSP_mav_S;

_Bool DSP_mavInit(volatile DSP_mav_S *mav);
void DSP_mavReset(volatile DSP_mav_S *mav);
UTILS_INLINE fp32_t DSP_mavUpdate(fp32_t u0, volatile DSP_mav_S *mav);

UTILS_INLINE fp32_t DSP_mavUpdate(fp32_t u0, volatile DSP_mav_S *mav)
{
    // Pointer to the oldest element in the circular buffer
    volatile fp32_t *oldest = (mav->buffer + mav->indOldest);

    // Update the moving average sum
    mav->sum += (u0 - *oldest);

    // Replace the oldest element with the newest element
    *oldest = u0;

    // Update the position of the oldest element in the circular buffer
    mav->indOldest++;
    if (mav->indOldest >= mav->bufferSize)  { mav->indOldest = 0; }

    // Return filtered value
    return (mav->sum / mav->bufferSize);
}

//==============================================================================================//
//              RATE LIMITER                                                                    //
//=============================================================================================c//

typedef struct DSP_rlim_STRUCT
{
    fp32_t RATE_POS;    // Absolute value of the positive rate of change (-/s)
    fp32_t RATE_NEG;    // Absolute value of the negative rate of change (-/s)
    fp32_t TS;          // Filter sample time (s)
    fp32_t INC_POS;     // Positive rate of change per sample time (-/TS)
    fp32_t INC_NEG;     // Negative rate of change per sample time (-/TS)
    fp32_t y1;          // Filter state y[k-1]
} DSP_rlim_S;

_Bool DSP_rlimInit(volatile DSP_rlim_S *rlim);
void DSP_rlimReset(volatile DSP_rlim_S *rlim);
UTILS_INLINE fp32_t DSP_rlimUpdate(fp32_t u0, volatile DSP_rlim_S *rlim);

UTILS_INLINE fp32_t DSP_rlimUpdate(fp32_t u0, volatile DSP_rlim_S *rlim)
{
    // Limit rate of change of the input value only if the rate-limiter
    // state is not equal to the input value
    if (rlim->y1 != u0)
    {
        // Positive change
        if (u0 > rlim->y1)
        {
            rlim->y1 += rlim->INC_POS;
            if (rlim->y1 > u0)  { rlim->y1 = u0; }
        }

        // Negative change
        else // (u0 < rlim->y1)
        {
            rlim->y1 -= rlim->INC_NEG;
            if (rlim->y1 < u0)  { rlim->y1 = u0; }
        }
    }

    // Return filtered value
    return rlim->y1;
}

//==============================================================================================//
//              MEASUREMENTS STATISTICS                                                         //
//=============================================================================================c//

typedef struct DSP_stat_STRUCT
{
    uint16_t N;         // Number of samples in population
    int64_t sum;        // Sum of all samples in population
    int64_t sumSqr;     // Sum of squares of all samples in population
    fp32_t mean;        // Population mean value
    fp32_t var;         // Population unbiased variance
    fp32_t std;         // Population unbiased standard deviation
} DSP_stat_S;

_Bool DSP_statInit(volatile DSP_stat_S *stat);
void DSP_statReset(volatile DSP_stat_S *stat);
UTILS_INLINE void DSP_statUpdate(int16_t oldest, int16_t newest, volatile DSP_stat_S *stat);

UTILS_INLINE void DSP_statUpdate(int16_t oldest, int16_t newest, volatile DSP_stat_S *stat)
{
    // Update sum and sumSqr fields in the calibration structure
    stat->sum    += (newest - oldest);
    stat->sumSqr += ((int32_t)newest*newest - (int32_t)oldest*oldest);

    // Recalculate population mean, unbiased variance, and standard deviation
    stat->mean = (fp32_t) stat->sum / stat->N;
    stat->var  = (stat->sumSqr - stat->mean*stat->sum) / (stat->N-1);
    stat->std  = sqrtf(stat->var); // (100ns with fp_mode=relaxed, and 2.6us with fp_mode=strict)
}

#endif /* DSP_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
