//==============================================================================================//
//              INVERTER.C                                                                      //
//=============================================================================================c//

#ifndef inverter_C_
#define inverter_C_

#include "inverter.h"

#include "mcu.h"

// Inverter handler
volatile INV_pack_S *INV;

//==============================================================================================//
//              EPWM4 INTERRUPT FUNCTION DEFINITION                                             //
//=============================================================================================c//

// CAUTION: You must update this table if you add new ISR commands
const uint32_t INV_COMMAND_TABLE[6][4] =
{
    { SYS_RELAY_CONTACTOR_AC_DISCONNECT, SYS_RELAY_PRECHARGE_DC_LINK_ON,  SYS_RELAY_DISCHARGE_DC_LINK_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 0 DEFAULT_STATE
    { SYS_RELAY_CONTACTOR_AC_DISCONNECT, SYS_RELAY_PRECHARGE_DC_LINK_ON,  SYS_RELAY_DISCHARGE_DC_LINK_ON,  MCU_EPWM_OUTPUT_DISABLE },  // 1 DISCHARGE_START
    { SYS_RELAY_CONTACTOR_AC_DISCONNECT, SYS_RELAY_PRECHARGE_DC_LINK_ON,  SYS_RELAY_DISCHARGE_DC_LINK_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 2 DISCHARGE_STOP
    { SYS_RELAY_CONTACTOR_AC_CONNECT,    SYS_RELAY_PRECHARGE_DC_LINK_ON,  SYS_RELAY_DISCHARGE_DC_LINK_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 3 PRECHARGE_START
    { SYS_RELAY_CONTACTOR_AC_CONNECT,    SYS_RELAY_PRECHARGE_DC_LINK_OFF, SYS_RELAY_DISCHARGE_DC_LINK_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 4 PRECHARGE_STOP
    { SYS_RELAY_CONTACTOR_AC_CONNECT,    SYS_RELAY_PRECHARGE_DC_LINK_OFF, SYS_RELAY_DISCHARGE_DC_LINK_OFF, MCU_EPWM_OUTPUT_ENABLE  }   // 5 INVERTER_START
};

interrupt void INV_isrEPwm4(void)
{
    //
    // GET STATE-MACHINE AND ERROR HANDLERS
    //

    // Pointer to state machine structure
    volatile CTRL_sm_S *sm = &INV->state_machine;

    // Pointer to error structure
    volatile ERR_pack_S *err = INV->error;

    // Pointer to stopwatch structure
    volatile SWC_tictoc_S *swc = &INV->stopwatch;

#ifdef PLECS
    printf("INVERTER:   %2d : %2d\n", sm->state, sm->substate);
    printf("ERROR:      %08X %08X %08X %08X %08X %08X %08X\n",
           err->CONFIG.all, err->INIT.all, err->TIMEOUT.all,
           err->MEAS_LOW.all, err->MEAS_HIGH.all,
           err->STAT_LOW.all, err->STAT_HIGH.all);
#endif //#ifdef PLECS

    //
    // GET DIGITAL MEASUREMENTS
    //

    // Wait until measurements triggered on TBCTR=0x00 of ePWM4 are ready, but do not wait forever
    // as we will stuck in an infinite loop and block resources for other ISRs with lower or same
    // priority level. The ADC flags will be latched on the corresponding EOC15 event.
    while   (   (AdcbRegs.ADCINTFLG.bit.ADCINT1 == 0) ||
                (AdccRegs.ADCINTFLG.bit.ADCINT1 == 0) ||
                (AdcdRegs.ADCINTFLG.bit.ADCINT1 == 0)
            )
    {
        // Do not wait forever, check timeout condition
        if (EPwm4Regs.TBCTR > CONST_TMO_ADC_MEASUREMENTS)
        {
            // Timeout has occurred, raise error flag
            ERR_FIELD_SET(TIMEOUT, ADC_EPWM4);

            // Do not wait for measurements any more
            break;
        }
    }

    // Get measurements from SOC channels if there is no timeout error
    if (ERR_FIELD_GET(TIMEOUT, ADC_EPWM4) == ERR_CLEAR)
    {
        // Secondary (phase) current
        INV->raw->i_sec_abcn.an  = (fp32_t) AdcbResultRegs.ADCRESULT12;     // B4
        INV->raw->i_sec_abcn.bn  = (fp32_t) AdccResultRegs.ADCRESULT12;     // C4
        INV->raw->i_sec_abcn.cn  = (fp32_t) AdcdResultRegs.ADCRESULT12;     // D2

        // Secondary (line) voltage
        INV->raw->v_sec_abc.ab   = (fp32_t) AdcbResultRegs.ADCRESULT13;     // B2
        INV->raw->v_sec_abc.bc   = (fp32_t) AdccResultRegs.ADCRESULT13;     // C2
        INV->raw->v_sec_abc.ca   = (fp32_t) AdcdResultRegs.ADCRESULT13;     // D0

        // Grid (phase) voltage
        INV->raw->v_grid_abcn.an = (fp32_t) AdcbResultRegs.ADCRESULT14;     // B5
        INV->raw->v_grid_abcn.bn = (fp32_t) AdccResultRegs.ADCRESULT14;     // C5
        INV->raw->v_grid_abcn.cn = (fp32_t) AdcdResultRegs.ADCRESULT14;     // D3

        // Temperature on AC an DC coldplate side
        INV->raw->temp_ac        = (fp32_t) AdcbResultRegs.ADCRESULT15;     // B3
        INV->raw->temp_dc        = (fp32_t) AdccResultRegs.ADCRESULT15;     // C3

        // DC link voltage
        INV->raw->v_dc           = (fp32_t) AdcdResultRegs.ADCRESULT15;     // D1

        // Clear all EOC trigger flags
        AdcbRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
        AdccRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
        AdcdRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
    }

    //
    // MEASUREMENTS CALIBRATION
    //

#ifdef CONST_CALIBRATION_BUFFER_SIZE

    // Start the stopwatch
    SWC_start(swc);

    // Limit the input channel number
    if (INV->meas_cal->channel >= INV->meas_cal->numOfChannels)
    {
        INV->meas_cal->channel = INV->meas_cal->numOfChannels-1;
    }

    // Load the newest measurement.
    // There is no round-off error when converting whole numbers:
    //  Steven W. Smith, "Digital Signal Processing", Ch. 4: "DSP Software", 2nd, 1999.
    int16_t cal_newest = (int16_t) *(INV->meas_cal->inputs+INV->meas_cal->channel);

    // Replace the oldest measurement with the newest measurement in the circular buffer
    int16_t cal_oldest = INV->meas_cal->buffer[INV->meas_cal->indOldest];
    INV->meas_cal->buffer[INV->meas_cal->indOldest] = cal_newest;

    // Update index of the oldest element in the circular buffer
    INV->meas_cal->indOldest = (INV->meas_cal->indOldest==(CONST_CALIBRATION_BUFFER_SIZE-1)) ?
            0 : (INV->meas_cal->indOldest+1);

    // Recalculate statistics for the selected input channel
    DSP_statUpdate(cal_oldest, cal_newest, &INV->meas_cal->stats);

    // Stop the stopwatch
    SWC_stop(swc);

#else

    //
    // INVERTER CONTROL ALGORITHM
    //

    // Update GLOBAL error flag
    if (   (ERR_REGISTER_GET(CONFIG)    |
            ERR_REGISTER_GET(INIT)      |
            ERR_REGISTER_GET(TIMEOUT)   |
            ERR_REGISTER_GET(MEAS_LOW)  |
            ERR_REGISTER_GET(MEAS_HIGH) |
            ERR_REGISTER_GET(STAT_LOW)  |
            ERR_REGISTER_GET(STAT_HIGH)) != 0)
    {
        ERR_FIELD_SET(CONFIG, GLOBAL);
        INV->usr_command = CTRL_SM_COMMAND_STOP;
    }

    // START or STOP the inverter state machine
    sm->command = INV->usr_command;
    CTRL_smErrorStopStart(sm, ERR_FIELD_GET(CONFIG, GLOBAL));

    // Inverter control algorithm
    SWC_start(swc);
    INV_doControl();
    SWC_stop(swc);

    // Update inverter duty cycles
    MCU_epwmSetDutyCycle(SYS_PWM_INV_L1, INV->duty->d1);
    MCU_epwmSetDutyCycle(SYS_PWM_INV_L2, INV->duty->d2);
    MCU_epwmSetDutyCycle(SYS_PWM_INV_L3, INV->duty->d3);

    //
    // UPDATE RELAY AND PWM ENABLE STATES
    //

    // Update relay and PWM output-enable states, but only if the ISR command from the
    // state machine has changed since the last interrupt call
    if (INV->isr_command != INV_ISR_COMMAND)
    {
        // Update new state-machine command
        INV->isr_command = INV_ISR_COMMAND;

        // Update relay states
        GPIO_WritePin(SYS_RELAY_CONTACTOR_AC,      INV_COMMAND_TABLE[INV->isr_command][0]);
        GPIO_WritePin(SYS_RELAY_PRECHARGE_DC_LINK, INV_COMMAND_TABLE[INV->isr_command][1]);
        GPIO_WritePin(SYS_RELAY_DISCHARGE_DC_LINK, INV_COMMAND_TABLE[INV->isr_command][2]);

        // Enable or disable PWM outputs
        MCU_epwmEnableOutputs(SYS_PWM_INV_L1, INV_COMMAND_TABLE[INV->isr_command][3]);
        MCU_epwmEnableOutputs(SYS_PWM_INV_L2, INV_COMMAND_TABLE[INV->isr_command][3]);
        MCU_epwmEnableOutputs(SYS_PWM_INV_L3, INV_COMMAND_TABLE[INV->isr_command][3]);
    }

#endif //#ifdef CONST_CALIBRATION_BUFFER_SIZE

    //
    // RETURN FROM INTERRUPT SERVICE ROUTINE
    //

    // Reset interrupt flags and return
    EPwm4Regs.ETCLR.bit.INT = 1;
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

//==============================================================================================//
//              INITIALIZATION FUNCTION DEFINITION                                              //
//=============================================================================================c//

void INV_init(volatile INV_pack_S *INV_PACK)
{
    // Initialize handler
    INV = INV_PACK;

    // Pointer to error structure
    volatile ERR_pack_S *err = INV->error;

    // Initialize PI controllers
    ERR_FIELD_GET(INIT, PI_V_DC)    = CTRL_piInit(&INV->pi_v_dc);
    ERR_FIELD_GET(INIT, PI_I_INV_D) = CTRL_piInit(&INV->pi_i_inv_d);
    ERR_FIELD_GET(INIT, PI_I_INV_Q) = CTRL_piInit(&INV->pi_i_inv_q);

#ifdef CONST_MAV_BUFFER_SIZE
    // Initialize moving-average filters
    ERR_FIELD_GET(INIT, MAV_V_GRID_D) = DSP_mavInit(&INV->mav_v_grid_d);
    ERR_FIELD_GET(INIT, MAV_V_GRID_Q) = DSP_mavInit(&INV->mav_v_grid_q);
    ERR_FIELD_GET(INIT, MAV_V_SEC_D)  = DSP_mavInit(&INV->mav_v_sec_d);
    ERR_FIELD_GET(INIT, MAV_V_SEC_Q)  = DSP_mavInit(&INV->mav_v_sec_q);
#else
    // Initialize second-order Butterworth filters
    ERR_FIELD_GET(INIT, BW2_V_GRID_D) = DSP_bw2Init(&INV->bw2_v_grid_d);
    ERR_FIELD_GET(INIT, BW2_V_GRID_Q) = DSP_bw2Init(&INV->bw2_v_grid_q);
    ERR_FIELD_GET(INIT, BW2_V_SEC_D)  = DSP_bw2Init(&INV->bw2_v_sec_d);
    ERR_FIELD_GET(INIT, BW2_V_SEC_Q)  = DSP_bw2Init(&INV->bw2_v_sec_q);
#endif //#ifdef CONST_MAV_BUFFER_SIZE

    // Reset calibration structures
    ERR_FIELD_GET(INIT, STAT_V_GRID_AN) = DSP_statInit(&INV->stat_v_grid_an);
    ERR_FIELD_GET(INIT, STAT_V_GRID_BN) = DSP_statInit(&INV->stat_v_grid_bn);
    ERR_FIELD_GET(INIT, STAT_V_GRID_CN) = DSP_statInit(&INV->stat_v_grid_cn);
    ERR_FIELD_GET(INIT, STAT_V_SEC_AB)  = DSP_statInit(&INV->stat_v_sec_ab);
    ERR_FIELD_GET(INIT, STAT_V_SEC_BC)  = DSP_statInit(&INV->stat_v_sec_bc);
    ERR_FIELD_GET(INIT, STAT_V_SEC_CA)  = DSP_statInit(&INV->stat_v_sec_ca);
    ERR_FIELD_GET(INIT, STAT_I_SEC_AN)  = DSP_statInit(&INV->stat_i_sec_an);
    ERR_FIELD_GET(INIT, STAT_I_SEC_BN)  = DSP_statInit(&INV->stat_i_sec_bn);
    ERR_FIELD_GET(INIT, STAT_I_SEC_CN)  = DSP_statInit(&INV->stat_i_sec_cn);

    // Initialize PLL structures
    ERR_FIELD_GET(INIT, PLL_GRID) = AC3PH_pllInit(&INV->pll_grid_abcn);
    ERR_FIELD_GET(INIT, PLL_SEC)  = AC3PH_pllInit(&INV->pll_sec_abc);

    // Initialize stopwatch
    SWC_init(
        &INV->stopwatch,
        (uint32_t *) &CpuTimer2.RegsAddr->TIM.all,
        (uint32_t) CpuTimer2.RegsAddr->PRD.all,
        (fp32_t) CpuTimer2.PeriodInUSec
    );
}

//==============================================================================================//
//              CONTROL FUNCTION DEFINITION                                                     //
//=============================================================================================c//

void INV_doControl(void)
{
#ifndef DO_NOT_PROGRAM_CONTORL_FUNCTIONS
    //
    // GET STATE MACHINE AND ERROR HANDLERS
    //

    // Pointer to inverter state machine
    volatile CTRL_sm_S *sm = &INV->state_machine;

    // Pointer to error structure
    volatile ERR_pack_S *err = INV->error;

    //
    // CALCULATE PHYSICAL MEASUREMENTS
    //

    // Digital measurements are converted here from raw format (0..4095) to their actual physical
    // values. Note that some measurements have an inherent offset, i.e., 2048 is converted to 0
    // for grid and secondary voltages, and for secondary, two inductor and output currents.
    // These measurements need to be calibrated before we can calculate their physical values.

    // DC link voltage
    INV->meas->v_dc = UTILS_ADC_NORM(INV->raw->v_dc, CONST_MEAS_V_DC, 0);

    // Scale measurements only after the measurements calibration is done. Note that measurements
    // calibration is done in two parts: first, grid voltage and secondary current are calibrated,
    // and then after grid is connected and transformer core is magnetized, secondary voltage is
    // calibrated. The calibration flag is set only after the secondary voltage is calibrated.
    if (INV_CAL_FLAG == CONST_MEAS_CALIBRATED_YES)
    {
        // Grid (phase) voltage
        INV->meas->v_grid_abcn.an = UTILS_ADC_NORM(INV->raw->v_grid_abcn.an, CONST_MEAS_V_GRID_AN, INV->stat_v_grid_an.mean);
        INV->meas->v_grid_abcn.bn = UTILS_ADC_NORM(INV->raw->v_grid_abcn.bn, CONST_MEAS_V_GRID_BN, INV->stat_v_grid_bn.mean);
        INV->meas->v_grid_abcn.cn = UTILS_ADC_NORM(INV->raw->v_grid_abcn.cn, CONST_MEAS_V_GRID_CN, INV->stat_v_grid_cn.mean);

        // Secondary (line) voltage
        INV->meas->v_sec_abc.ab   = UTILS_ADC_NORM(INV->raw->v_sec_abc.ab,   CONST_MEAS_V_SEC_AB,  INV->stat_v_sec_ab.mean );
        INV->meas->v_sec_abc.bc   = UTILS_ADC_NORM(INV->raw->v_sec_abc.bc,   CONST_MEAS_V_SEC_BC,  INV->stat_v_sec_bc.mean );
        INV->meas->v_sec_abc.ca   = UTILS_ADC_NORM(INV->raw->v_sec_abc.ca,   CONST_MEAS_V_SEC_CA,  INV->stat_v_sec_ca.mean );

        // Secondary (phase) current
        INV->meas->i_sec_abcn.an  = UTILS_ADC_NORM(INV->raw->i_sec_abcn.an,  CONST_MEAS_I_SEC_AN,  INV->stat_i_sec_an.mean );
        INV->meas->i_sec_abcn.bn  = UTILS_ADC_NORM(INV->raw->i_sec_abcn.bn,  CONST_MEAS_I_SEC_BN,  INV->stat_i_sec_bn.mean );
        INV->meas->i_sec_abcn.cn  = UTILS_ADC_NORM(INV->raw->i_sec_abcn.cn,  CONST_MEAS_I_SEC_CN,  INV->stat_i_sec_cn.mean );
    }

    // The calibration flag is set after the secondary voltage is calibrated, which in the state
    // machine comes after the phase order detection on the input AC connector (grid side). Since
    // we need grid voltage measurements at this phase, we scale measurements here even before the
    // calibration flag is set.
    else if (sm->state == INV_SM_STATE_PHASE_ORDER_DETECTION_I)
    {
        // Grid (phase) voltage
        INV->meas->v_grid_abcn.an = UTILS_ADC_NORM(INV->raw->v_grid_abcn.an, CONST_MEAS_V_GRID_AN, INV->stat_v_grid_an.mean);
        INV->meas->v_grid_abcn.bn = UTILS_ADC_NORM(INV->raw->v_grid_abcn.bn, CONST_MEAS_V_GRID_BN, INV->stat_v_grid_bn.mean);
        INV->meas->v_grid_abcn.cn = UTILS_ADC_NORM(INV->raw->v_grid_abcn.cn, CONST_MEAS_V_GRID_CN, INV->stat_v_grid_cn.mean);
    }

    //
    // SOFTWARE PROTECTION
    //

    // TODO comment
    // TODO think about including additional protection signals (PLL, etc.)
    // TODO implement unexpected state, substate and ISR command protection

    // DC link overvoltage protection
    if (INV->meas->v_dc > CONST_PROT_V_DC_HIGH)
    {
        ERR_FIELD_SET(MEAS_HIGH, V_DC);
    }

    // Some variables are monitored only in the STARTED state
    if (sm->state == CTRL_SM_STATE_STARTED)
    {
        // DC link undervoltage protection
        if (INV->meas->v_dc < CONST_PROT_V_DC_LOW)
        {
            ERR_FIELD_SET(MEAS_LOW, V_DC);
        }

        // TODO add other protection signals (frequency etc.)
    }

    //
    // STATE MACHINE
    //

    // TODO ADD COMMENTS FOR STATE MACHINE

    // STATE MACHINE
    switch (sm->state)
    {
        // State machine is STOPPED,
        // immediately break the control algorithm.
        case CTRL_SM_STATE_STOPPED :
            return;

        // State machine is STARTED,
        // proceed with the control algorithm.
        case CTRL_SM_STATE_STARTED :
            break;

        // Reset inverter to default state
        case CTRL_SM_STATE_RESET :
            switch (sm->substate)
            {
                // RESET TO DEFAULT STATE
                case 0 :
                    // Set default state for inverter relays and PWM generators
                    INV_ISR_COMMAND = INV_ISR_COMMAND_DEFAULT_STATE;

                    // Reset measurements that need to be calibrated
                    INV->meas->v_grid_abcn.an = 0.0f;
                    INV->meas->v_grid_abcn.bn = 0.0f;
                    INV->meas->v_grid_abcn.cn = 0.0f;
                    INV->meas->v_sec_abc.ab   = 0.0f;
                    INV->meas->v_sec_abc.bc   = 0.0f;
                    INV->meas->v_sec_abc.ca   = 0.0f;
                    INV->meas->i_sec_abcn.an  = 0.0f;
                    INV->meas->i_sec_abcn.bn  = 0.0f;
                    INV->meas->i_sec_abcn.cn  = 0.0f;

                    // Reset inverter duty cycles
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = 0.0f;

                    // Reset measurements calibration flag
                    INV_CAL_FLAG = CONST_MEAS_CALIBRATED_NO;

                    // There is no CRITICAL ERROR in the system,
                    // state machine can start with the discharge routine,
                    // and after that go to STARTING or STOPPING procedure.
                    if (ERR_FIELD_GET(CONFIG, CRITICAL) == ERR_CLEAR)
                    {
                        CTRL_smGotoState(sm, INV_SM_STATE_DISCHARGE_DC_LINK);
                        return;
                    }

                    // There is CRITICAL ERROR in the system,
                    // state machine should be STOPPED without going into the discharge routine.
                    else
                    {
                        CTRL_smGotoState(sm, CTRL_SM_STATE_STOPPED);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // In this routine we discharge the DC link. It must be noted that the output capacitors
        // will also be discharged through high-side diodes of the buck-boost bridge. If DC link
        // is already discharged, this routine is skipped, even if output capacitors are still
        // charged. The output capacitors will be discharged in the buck-boost state machine.
        case INV_SM_STATE_DISCHARGE_DC_LINK :
            switch (sm->substate)
            {
                // START OR SKIP DISCHARGE ROUTINE
                case 0 :
                    // Skip the discharge routine if DC link is already discharged
                    if (INV->meas->v_dc < CONST_INV_DISCHARGE_V_DC)
                    {
                        CTRL_smGotoState(sm, CTRL_SM_STATE_START_OR_STOP);
                        return;
                    }

                    // Start the discharge routine
                    else // (INV->meas->v_dc >= CONST_INV_DISCHARGE_V_DC)
                    {
                        // Connect the DC link discharge resistor
                        INV_ISR_COMMAND = INV_ISR_COMMAND_DISCHARGE_START;

                        CTRL_smGotoNextSubstate(sm);
                        return;
                    }

                // WAIT UNTIL DC LINK IS DISCHARGED
                case 1 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->meas->v_dc < CONST_INV_DISCHARGE_V_DC,
                        CONST_TMO_INV_DISCHARGE_DC_LINK,
                        TIMEOUT, DISCHARGE_DC_LINK
                    );

                    // Set critical error flag if DC link cannot be discharged. DC link discharge
                    // resistors will be disconnected in the CTRL_SM_STATE_RESET state.
                    if (ERR_FIELD_GET(TIMEOUT, DISCHARGE_DC_LINK) == ERR_SET)
                    {
                        ERR_FIELD_SET(CONFIG, CRITICAL);
                    }

                    return;

                // STOP THE DISCHARGE ROUTINE
                case 2 :
                    // Disconnect the DC link discharge resistor
                    INV_ISR_COMMAND = INV_ISR_COMMAND_DISCHARGE_STOP;

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // WAIT UNTIL DISCHARGE RESISTOR DISCONNECTS
                case 3 :
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_INV_RELAY);
                    return;

                // DISCHARGE ROUTINE IS DONE
                case 4 :
                    CTRL_smGotoState(sm, CTRL_SM_STATE_START_OR_STOP);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // START or STOP the state machine
        case CTRL_SM_STATE_START_OR_STOP :
            switch (sm->substate)
            {
                // START OR STOP THE STATE MACHINE
                case 0 :
                    // State machine is in the STARTING mode,
                    // we need to calibrate measurements and detect grid phase order before we can
                    // start with the passive and active precharge routines.
                    if (sm->mode == CTRL_SM_MODE_STARTING)
                    {
                        CTRL_smGotoState(sm, INV_SM_STATE_CALIBRATION_I);
                        return;
                    }

                    // State machine is in the STOPPING mode,
                    // go straight to the STOPPED state.
                    else // CTRL_SM_MODE_STOPPING
                    {
                        // If the PHASE_ORDER_V_SEC error flag is raised, raise the critical error
                        // flag in order not to allow user to clear this error flag until it turns
                        // off the inverter and rewires the transformer. We raise critical error
                        // flag here in order to allow the inverter to go into discharge routine.
                        if (ERR_FIELD_GET(CONFIG, PHASE_ORDER_SEC) == ERR_SET)
                        {
                            ERR_FIELD_SET(CONFIG, CRITICAL);
                        }

                        CTRL_smGotoState(sm, CTRL_SM_STATE_STOPPED);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // It must be noted that here we calibrate only the grid (phase) voltage and secondary
        // (phase) current, but not the secondary (line) voltage. The secondary voltage cannot
        // be calibrated at this point as the AC contactor is still disconnected.
        case INV_SM_STATE_CALIBRATION_I :
            switch (sm->substate)
            {
                // RESET CALIBRATION STRUCTURES
                case 0 :
                    // Reset calibration structures
                    //  Grid (phase) voltage
                    DSP_statReset(&INV->stat_v_grid_an);
                    DSP_statReset(&INV->stat_v_grid_bn);
                    DSP_statReset(&INV->stat_v_grid_cn);

                    // Reset calibration structures
                    //  Secondary (phase) current
                    DSP_statReset(&INV->stat_i_sec_an);
                    DSP_statReset(&INV->stat_i_sec_bn);
                    DSP_statReset(&INV->stat_i_sec_cn);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATE MEASUREMENTS
                case 1 :
                    // Update statistics
                    //  Grid (phase) voltage
                    DSP_statUpdate(0, (int16_t) INV->raw->v_grid_abcn.an, &INV->stat_v_grid_an);
                    DSP_statUpdate(0, (int16_t) INV->raw->v_grid_abcn.bn, &INV->stat_v_grid_bn);
                    DSP_statUpdate(0, (int16_t) INV->raw->v_grid_abcn.cn, &INV->stat_v_grid_cn);

                    // Update statistics
                    //  Secondary (phase) current
                    DSP_statUpdate(0, (int16_t) INV->raw->i_sec_abcn.an,  &INV->stat_i_sec_an );
                    DSP_statUpdate(0, (int16_t) INV->raw->i_sec_abcn.bn,  &INV->stat_i_sec_bn );
                    DSP_statUpdate(0, (int16_t) INV->raw->i_sec_abcn.cn,  &INV->stat_i_sec_cn );

                    // Ensure that we collect enough calibration samples. Since voltage waveform
                    // is a sine wave, the CALIBRATION timer constant needs to match the voltage
                    // fundamental frequency (i.e., period). Otherwise, the calculated mean will
                    // be wrong.
                    // TODO explain why exactly one sine wave period!
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_INV_CALIBRATION);
                    return;

                // CHECK MEASUREMENT OFFSET
                case 2 :
                    // Check offset
                    //  Grid (phase) voltage
                    ERR_CHECK_OFFSET(INV->stat_v_grid_an.mean, CONST_PROT_OFFSET, V_GRID_AN);
                    ERR_CHECK_OFFSET(INV->stat_v_grid_bn.mean, CONST_PROT_OFFSET, V_GRID_BN);
                    ERR_CHECK_OFFSET(INV->stat_v_grid_cn.mean, CONST_PROT_OFFSET, V_GRID_CN);

                    // Check offset
                    //  Secondary (phase) current
                    ERR_CHECK_OFFSET(INV->stat_i_sec_an.mean,  CONST_PROT_OFFSET, I_SEC_AN );
                    ERR_CHECK_OFFSET(INV->stat_i_sec_bn.mean,  CONST_PROT_OFFSET, I_SEC_BN );
                    ERR_CHECK_OFFSET(INV->stat_i_sec_cn.mean,  CONST_PROT_OFFSET, I_SEC_CN );

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CHECK RMS VALUE
                case 3 :
                    // Calculate RMS value
                    //  Grid (phase) voltage
                    // TODO explain why is this equal to std
                    INV->v_grid_rms.an = INV->stat_v_grid_an.std * CONST_MEAS_V_GRID_AN;
                    INV->v_grid_rms.bn = INV->stat_v_grid_bn.std * CONST_MEAS_V_GRID_BN;
                    INV->v_grid_rms.cn = INV->stat_v_grid_cn.std * CONST_MEAS_V_GRID_CN;

                    // Check RMS value
                    //  Grid (phase) voltage
                    ERR_CHECK_RMS(INV->v_grid_rms.an, CONST_PROT_V_GRID_RMS, V_GRID_AN);
                    ERR_CHECK_RMS(INV->v_grid_rms.bn, CONST_PROT_V_GRID_RMS, V_GRID_BN);
                    ERR_CHECK_RMS(INV->v_grid_rms.cn, CONST_PROT_V_GRID_RMS, V_GRID_CN);

                    // Note that we do not need to check RMS value for current measurements

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATION IS DONE
                case 4 :
                    CTRL_smGotoState(sm, INV_SM_STATE_PHASE_ORDER_DETECTION_I);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // TODO Implement a functionality so that the user does not need to fix phase order!
        // There are six possible combinations to order phases on the input AC connector. Three of
        // them are valid, while the other three are invalid for the proper inverter operation:
        //      R-S-T (valid) and R-T-S (invalid),
        //      T-R-S (valid) and T-S-R (invalid),
        //      S-T-R (valid) and S-R-T (invalid).
        // To fix invalid phase order, it is enough to swap the second and the third phase, i.e.,
        //      R-T-S -> R-S-T  or  T-S-R -> T-R-S  or  S-R-T -> S-T-R
        //
        // By comparing three-phase grid voltage measurements to zero, we can detect in which
        // of the six possible sectors the grid voltage is currently positioned:
        //      (va>0)  (vb>0)  (vc>0)  =>  sector code         phase 'a' angle
        //      ---------------------------------------------------------------
        //      FALSE   FALSE   FALSE   =>  000 is impossible   -
        //      FALSE   FALSE   TRUE    =>  001 is 1            300..360
        //      FALSE   TRUE    FALSE   =>  010 is 2            180..240
        //      FALSE   TRUE    TRUE    =>  011 is 3            240..300
        //      TRUE    FALSE   FALSE   =>  100 is 4             60..120
        //      TRUE    FALSE   TRUE    =>  101 is 5              0..60
        //      TRUE    TRUE    FALSE   =>  110 is 6            120..180
        //      TRUE    TRUE    TRUE    =>  111 is impossible   -
        //      ---------------------------------------------------------------
        //
        // The above table can be simplified into the following equation:
        //      sector = (int) (((va>0)<<2) | ((vb>0)<<1) | ((vc>0)<<0))
        //
        // All three correct phase order combinations have the same sector transition sequence:
        //      0->N/A, 1->5, 2->3, 3->1, 4->6, 5->4, 6->2, 7->N/A
        // The above transition sequence can be simplified into the following equation:
        //      sector2 = (14-2*sector1) % 7
        // where sector1 is from the left side, and sector2 is from the right side of the ->.
        //
        // All three incorrect phase order combinations have the same sector transition sequence:
        //      0->N/A, 2->6, 6->4, 4->5, 5->1, 1->3, 3->2, 7->N/A
        // which can be simplified into: sector1 = (14-2*sector2) % 7
        //
        // In order to detect if phase order on the input AC connector is correct, we only have to
        // detect two consecutive sectors of the grid voltage.
        case INV_SM_STATE_PHASE_ORDER_DETECTION_I :
            // Copy measurements for phase order detection.
            // Phase order detection algorithm is used as a function, and this is function input.
            INV->v_phase_order.an = INV->meas->v_grid_abcn.an;
            INV->v_phase_order.bn = INV->meas->v_grid_abcn.bn;
            INV->v_phase_order.cn = INV->meas->v_grid_abcn.cn;

            // Make sure that you update the following parts of the nested switch statement:
            //  UPDATE (A)  next state in CTRL_smGotoState() function call
            //  UPDATE (B)  sector code error flag in CONFIG register
            //  UPDATE (C)  sector timeout error flag in TIMEOUT register
            //  UPDATE (D)  phase order error flag in CONFIG register
            switch (sm->substate)
            {
                // SECTOR DETECTION I
                case 0 :
                    // Detect sector
                    INV->sector1 =  ((INV->v_phase_order.an > 0) << 2) |
                                    ((INV->v_phase_order.bn > 0) << 1) |
                                    ((INV->v_phase_order.cn > 0) << 0) ;

                    // Sanity check: sector cannot be 0 or 7
                    if ((INV->sector1==0) || (INV->sector1>=7))
                    {
                        // UPDATE (B)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_GRID_CODE);
                        return;
                    }

                    // Accept detected sector only if all voltage measurements are outside of the
                    // small vicinity around 0 V.
                    // UPDATE (C)
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        ((INV->v_phase_order.an < -CONST_INV_V_EPS) || (INV->v_phase_order.an > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.bn < -CONST_INV_V_EPS) || (INV->v_phase_order.bn > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.cn < -CONST_INV_V_EPS) || (INV->v_phase_order.cn > +CONST_INV_V_EPS)),
                        CONST_TMO_INV_PHASE_ORDER,
                        TIMEOUT, PHASE_ORDER_GRID
                    );

                    return;

                // SECTOR DETECTION II
                case 1 :
                    // Detect sector
                    INV->sector2 =  ((INV->v_phase_order.an > 0) << 2) |
                                    ((INV->v_phase_order.bn > 0) << 1) |
                                    ((INV->v_phase_order.cn > 0) << 0) ;

                    // Sanity check: sector cannot be 0 or 7
                    if ((INV->sector1==0) || (INV->sector1>=7))
                    {
                        // UPDATE (B)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_GRID_CODE);
                        return;
                    }

                    // Accept detected sector only if all voltage measurements are outside of the
                    // small vicinity around 0 V. Additional condition here is that two sectors
                    // must be different, i.e., a sector transition must occur.
                    // UPDATE (C)
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        ((INV->v_phase_order.an < -CONST_INV_V_EPS) || (INV->v_phase_order.an > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.bn < -CONST_INV_V_EPS) || (INV->v_phase_order.bn > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.cn < -CONST_INV_V_EPS) || (INV->v_phase_order.cn > +CONST_INV_V_EPS)) &&
                        (INV->sector2 != INV->sector1),
                        CONST_TMO_INV_PHASE_ORDER,
                        TIMEOUT, PHASE_ORDER_GRID
                    );

                    return;

                // CHECK PHASE ORDER
                case 2 :
                    // Check transition from sector1 to sector2
                    if (((14-2*INV->sector1) % 7) != INV->sector2)
                    {
                        // UPDATE (D)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_GRID);
                        return;
                    }

                    // Phase order is correct
                    else
                    {
                        // UPDATE (A)
                        CTRL_smGotoState(sm, INV_SM_STATE_PRECHARGE_PASSIVE_I);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // Passive precharge is when grid is connected to the DC link only via precharge resistors
        // and bridge diodes. If given enough time, the DC link would charge to secondary (line)
        // voltage amplitude minus two forward voltage drops on a diode which is around 2 V each.
        // It must be noted that the precharge resistors can be disconnected only when the DC link
        // is charged to at least 95% of the voltage amplitude. Otherwise, the voltage difference
        // between the secondary voltage and DC link voltage might be too large, which will cause
        // large current to flow through bridge diodes. However, at this point we cannot estimate
        // secondary (line) voltage amplitude, as the secondary voltage is still not calibrated.
        // That is why we perform passive precharge in two parts: (1) we precharge the DC link
        // to some safe value, and then (2) wait until the DC link reaches at least 95% of the
        // voltage amplitude before we disconnect the precharge resistors.
        case INV_SM_STATE_PRECHARGE_PASSIVE_I :
            switch (sm->substate)
            {
                // CONNECT TO THE GRID
                case 0 :
                    // Connect the precharge resistors and AC contactor, which will start the
                    // DC link passive precharge via precharge resistors and bridge diodes.
                    INV_ISR_COMMAND = INV_ISR_COMMAND_PRECHARGE_START;

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // WAIT UNTIL DC LINK IS PRECHARGED
                // We wait in this substate until the DC link is precharged to some safe voltage
                // level. Once we establish that the DC link is charged to the safe voltage level,
                // we can proceed with the secondary voltage measurements calibration. After the
                // secondary voltage has been calibrated, we can estimate the minimum DC link
                // voltage level (e.g., 95% of the voltage amplitude) at which we can disconnect
                // the precharge resistors.
                case 1 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->meas->v_dc > CONST_INV_PASSIVE_I_V_DC,
                        CONST_TMO_INV_PRECHARGE_PASSIVE_I,
                        TIMEOUT, PRECHARGE_PASSIVE_I
                    );

                    return;

                // DC LINK REACHED SAFE VOLTAGE LEVEL
                // This only indicates that everything might be OK with the grid. We still need to
                // estimate the secondary (line) voltage amplitude, in order to exactly determine
                // DC link voltage level at which it is safe to disconnect precharge resistors.
                case 2 :
                    CTRL_smGotoState(sm, INV_SM_STATE_CALIBRATION_II);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // Secondary (line) voltage should be stable by now, which means that we can start with
        // the measurements calibration procedure (part II). Once secondary voltage has been
        // calibrated, we can set the measurements calibration flag.
        case INV_SM_STATE_CALIBRATION_II :
            switch (sm->substate)
            {
                // RESET CALIBRATION STRUCTURES
                case 0 :
                    // Reset calibration structures
                    //  Secondary (line) voltage
                    DSP_statReset(&INV->stat_v_sec_ab);
                    DSP_statReset(&INV->stat_v_sec_bc);
                    DSP_statReset(&INV->stat_v_sec_ca);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATE MEASUREMENTS
                case 1 :
                    // Update statistics
                    //  Secondary (line) voltage
                    DSP_statUpdate(0, (int16_t) INV->raw->v_sec_abc.ab, &INV->stat_v_sec_ab);
                    DSP_statUpdate(0, (int16_t) INV->raw->v_sec_abc.bc, &INV->stat_v_sec_bc);
                    DSP_statUpdate(0, (int16_t) INV->raw->v_sec_abc.ca, &INV->stat_v_sec_ca);

                    // Ensure that we collect enough calibration samples. Since voltage waveform
                    // is a sine wave, the CALIBRATION timer constant needs to match the voltage
                    // fundamental frequency (i.e., period). Otherwise, the calculated mean will
                    // be wrong.
                    // TODO explain why exactly one sine wave period!
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_INV_CALIBRATION);
                    return;

                // CHECK MEASUREMENT OFFSET
                case 2 :
                    // Check offset
                    //  Secondary (line) voltage
                    ERR_CHECK_OFFSET(INV->stat_v_sec_ab.mean, CONST_PROT_OFFSET, V_SEC_AB);
                    ERR_CHECK_OFFSET(INV->stat_v_sec_bc.mean, CONST_PROT_OFFSET, V_SEC_BC);
                    ERR_CHECK_OFFSET(INV->stat_v_sec_ca.mean, CONST_PROT_OFFSET, V_SEC_CA);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CHECK RMS VALUE
                case 3 :
                    // Calculate RMS value
                    //  Secondary (line) voltage
                    INV->v_sec_rms.ab = INV->stat_v_sec_ab.std * CONST_MEAS_V_SEC_AB;
                    INV->v_sec_rms.bc = INV->stat_v_sec_bc.std * CONST_MEAS_V_SEC_BC;
                    INV->v_sec_rms.ca = INV->stat_v_sec_ca.std * CONST_MEAS_V_SEC_CA;

                    // Check RMS value
                    //  Secondary (line) voltage
                    ERR_CHECK_RMS(INV->v_sec_rms.ab, CONST_PROT_V_SEC_RMS, V_SEC_AB);
                    ERR_CHECK_RMS(INV->v_sec_rms.bc, CONST_PROT_V_SEC_RMS, V_SEC_BC);
                    ERR_CHECK_RMS(INV->v_sec_rms.ca, CONST_PROT_V_SEC_RMS, V_SEC_CA);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATION IS DONE
                case 4 :
                    // All measurements have been successfully calibrated,
                    // set the measurements calibration flag
                    INV_CAL_FLAG = CONST_MEAS_CALIBRATED_YES;

                    CTRL_smGotoState(sm, INV_SM_STATE_PHASE_ORDER_DETECTION_II);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // Test phase order for the secondary (line) voltage
        case INV_SM_STATE_PHASE_ORDER_DETECTION_II :
            // Copy measurements for phase order detection.
            // Phase order detection algorithm is used as a function, and this is function input.
            INV->v_phase_order.an = INV->meas->v_sec_abc.ab;
            INV->v_phase_order.bn = INV->meas->v_sec_abc.bc;
            INV->v_phase_order.cn = INV->meas->v_sec_abc.ca;

            // Make sure that you update the following parts of the nested switch statement:
            //  UPDATE (A)  next state in CTRL_smGotoState() function call
            //  UPDATE (B)  sector code error flag in CONFIG register
            //  UPDATE (C)  sector timeout error flag in TIMEOUT register
            //  UPDATE (D)  phase order error flag in CONFIG register
            switch (sm->substate)
            {
                // SECTOR DETECTION I
                case 0 :
                    // Detect sector
                    INV->sector1 =  ((INV->v_phase_order.an > 0) << 2) |
                                    ((INV->v_phase_order.bn > 0) << 1) |
                                    ((INV->v_phase_order.cn > 0) << 0) ;

                    // Sanity check: sector cannot be 0 or 7
                    if ((INV->sector1==0) || (INV->sector1>=7))
                    {
                        // UPDATE (B)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_SEC_CODE);
                        return;
                    }

                    // Accept detected sector only if all voltage measurements are outside of the
                    // small vicinity around 0 V.
                    // UPDATE (C)
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        ((INV->v_phase_order.an < -CONST_INV_V_EPS) || (INV->v_phase_order.an > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.bn < -CONST_INV_V_EPS) || (INV->v_phase_order.bn > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.cn < -CONST_INV_V_EPS) || (INV->v_phase_order.cn > +CONST_INV_V_EPS)),
                        CONST_TMO_INV_PHASE_ORDER,
                        TIMEOUT, PHASE_ORDER_SEC
                    );

                    return;

                // SECTOR DETECTION II
                case 1 :
                    // Detect sector
                    INV->sector2 =  ((INV->v_phase_order.an > 0) << 2) |
                                    ((INV->v_phase_order.bn > 0) << 1) |
                                    ((INV->v_phase_order.cn > 0) << 0) ;

                    // Sanity check: sector cannot be 0 or 7
                    if ((INV->sector1==0) || (INV->sector1>=7))
                    {
                        // UPDATE (B)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_SEC_CODE);
                        return;
                    }

                    // Accept detected sector only if all voltage measurements are outside of the
                    // small vicinity around 0 V. Additional condition here is that two sectors
                    // must be different, i.e., a sector transition must occur.
                    // UPDATE (C)
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        ((INV->v_phase_order.an < -CONST_INV_V_EPS) || (INV->v_phase_order.an > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.bn < -CONST_INV_V_EPS) || (INV->v_phase_order.bn > +CONST_INV_V_EPS)) &&
                        ((INV->v_phase_order.cn < -CONST_INV_V_EPS) || (INV->v_phase_order.cn > +CONST_INV_V_EPS)) &&
                        (INV->sector2 != INV->sector1),
                        CONST_TMO_INV_PHASE_ORDER,
                        TIMEOUT, PHASE_ORDER_SEC
                    );

                    return;

                // CHECK PHASE ORDER
                case 2 :
                    // Check transition from sector1 to sector2
                    if (((14-2*INV->sector1) % 7) != INV->sector2)
                    {
                        // UPDATE (D)
                        ERR_FIELD_SET(CONFIG, PHASE_ORDER_SEC);
                        return;
                    }

                    // Phase order is correct
                    else
                    {
                        // UPDATE (A)
                        CTRL_smGotoState(sm, INV_SM_STATE_WAIT_PLL_LOCK);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // We can start the PLL algorithm only after the secondary (line) voltage measurements
        // have been calibrated. Wait in this state until both PLLs are in lock. From this point,
        // we break the state machine to reach phase-locked loop routines.
        case INV_SM_STATE_WAIT_PLL_LOCK :
            switch (sm->substate)
            {
                // RESET PLL STRUCTURES
                case 0 :
                    // Reset PLL structures
                    AC3PH_pllReset(&INV->pll_grid_abcn);
                    AC3PH_pllReset(&INV->pll_sec_abc);

                    CTRL_smGotoNextSubstate(sm);
                    break;

                // WAIT UNTIL PLL LOCKS I
                case 1 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->pll_grid_abcn.in_lock == AC3PH_PLL_INLOCK_YES,
                        CONST_TMO_INV_PLL_LOCK,
                        TIMEOUT, PLL_LOCK_GRID
                    );

                    break;

                // WAIT UNTIL PLL LOCKS II
                case 2 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->pll_sec_abc.in_lock == AC3PH_PLL_INLOCK_YES,
                        CONST_TMO_INV_PLL_LOCK,
                        TIMEOUT, PLL_LOCK_SEC
                    );

                    break;

                // PROCEED WITH THE PASSIVE PRECHARGE
                case 3 :
                    CTRL_smGotoState(sm, INV_SM_STATE_PRECHARGE_PASSIVE_II);
                    break;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

            break;

        // Before we disconnect the precharge resistors, the DC link should be charged to at least
        // 95% of the secondary (line) voltage amplitude, in order to avoid large current flowing
        // through bridge diodes. Since secondary (line) voltage measurements have been calibrated
        // at this point, we can estimate its voltage amplitude, i.e., the minimum DC link voltage
        // level at which it is safe to disconnect the precharge resistors.
        case INV_SM_STATE_PRECHARGE_PASSIVE_II :
            switch (sm->substate)
            {
                // WAIT UNTIL DC LINK IS PRECHARGED
                case 0 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->meas->v_dc >= CONST_INV_PASSIVE_II_V_DC*(UTILS_SQRT3*INV->v_sec_dq.d),
                        CONST_TMO_INV_PRECHARGE_PASSIVE_II,
                        TIMEOUT, PRECHARGE_PASSIVE_II
                    );

                    break;

                // DISCONNECT PRECHARGE RESISTORS
                case 1 :
                    // Passive precharge is done, disconnect precharge resistors
                    INV_ISR_COMMAND = INV_ISR_COMMAND_PRECHARGE_STOP;

                    CTRL_smGotoNextSubstate(sm);
                    break;

                // WAIT UNTIL PRECHARGE RESISTORS DISCONNECT
                case 2 :
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_INV_RELAY);
                    break;

                // PASSIVE PRECHARGE ROUTINE IS DONE
                case 3 :
                    CTRL_smGotoState(sm, INV_SM_STATE_PRECHARGE_ACTIVE);
                    break;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

            break;

        // Precharge the DC link to its nominal voltage level, by using PI controllers that will
        // calculate appropriate duty cycles for the three inverter PWM generators.
        case INV_SM_STATE_PRECHARGE_ACTIVE :
            switch (sm->substate)
            {
                // RESET CONTROL VARIABLES
                case 0 :
                    // Reset all PI controllers
                    CTRL_piReset(&INV->pi_v_dc);
                    CTRL_piReset(&INV->pi_i_inv_d);
                    CTRL_piReset(&INV->pi_i_inv_q);

                    // Reset inverter current references
                    INV->i_inv_ref_dq.d = 0.0f;
                    INV->i_inv_ref_dq.q = 0.0f;

                    // Reset inverter duty cycles
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = 0.0f;

                    // Enable inverter PWM generators
                    INV_ISR_COMMAND = INV_ISR_COMMAND_INVERTER_START;

                    CTRL_smGotoNextSubstate(sm);
                    break;

                // WAIT UNTIL CURRENT PI CONTROLLERS SETTLE
                case 1 :
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_INV_PI_SETTLE);
                    break;

                // START WITH THE ACTIVE PRECHARGE
                case 2 :
                    // Set the DC link precharge current
                    INV->i_inv_ref_dq.d = CONST_INV_ACTIVE_I_D;
                    INV->i_inv_ref_dq.q = 0.0f;

                    CTRL_smGotoNextSubstate(sm);
                    break;

                // WAIT UNTIL DC LINK IS PRECHARGED
                case 3 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        INV->meas->v_dc >= CONST_INV_V_DC_NOM,
                        CONST_TMO_INV_PRECHARGE_ACTIVE,
                        TIMEOUT, PRECHARGE_ACTIVE
                    );
                    break;

                // ACTIVE PRECHARGE IS DONE
                case 4 :
                    // Set current reference to 0 A
                    INV->i_inv_ref_dq.d = 0.0f;
                    INV->i_inv_ref_dq.q = 0.0f;

                    CTRL_smGotoState(sm, CTRL_SM_STATE_STARTED);
                    break;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

            break;

        // UNEXPECTED STATE
        default :
            ERR_FIELD_SET(CONFIG, UNEXPECTED_STATE);
            return;
    }

    //
    // PHASE-LOCKED LOOP ROUTINE
    //

    // Update angle trigonometry
    UTILS_calcSinAndCos(&INV->pll_grid_abcn.phase);
    UTILS_calcSinAndCos(&INV->pll_sec_abc.phase  );

    // Clarke transformation
    // [stationary circuit to SRF]
    AC3PH_fwdClarkePhase(&INV->meas->v_grid_abcn, &INV->v_grid_ab);
    AC3PH_fwdClarkeLine (&INV->meas->v_sec_abc,   &INV->v_sec_ab );
    AC3PH_fwdClarkePhase(&INV->meas->i_sec_abcn,  &INV->i_sec_ab );

    // Park transformation
    // [SRF to RRF]
    AC3PH_fwdPark(&INV->v_grid_ab, &INV->v_grid_dq, &INV->pll_grid_abcn.phase);
    AC3PH_fwdPark(&INV->v_sec_ab,  &INV->v_sec_dq,  &INV->pll_sec_abc.phase  );
    AC3PH_fwdPark(&INV->i_sec_ab,  &INV->i_sec_dq,  &INV->pll_sec_abc.phase  );

    // Filter output from the Park transformation
    // [we use moving-average or second-order Butterworth filter]
#ifdef CONST_MAV_BUFFER_SIZE
    INV->v_grid_dq.d = DSP_mavUpdate(INV->v_grid_dq.d, &INV->mav_v_grid_d);
    INV->v_grid_dq.q = DSP_mavUpdate(INV->v_grid_dq.q, &INV->mav_v_grid_q);
    INV->v_sec_dq.d  = DSP_mavUpdate(INV->v_sec_dq.d,  &INV->mav_v_sec_d );
    INV->v_sec_dq.q  = DSP_mavUpdate(INV->v_sec_dq.q,  &INV->mav_v_sec_q );
#else
    INV->v_grid_dq.d = DSP_bw2Update(INV->v_grid_dq.d, &INV->bw2_v_grid_d);
    INV->v_grid_dq.q = DSP_bw2Update(INV->v_grid_dq.q, &INV->bw2_v_grid_q);
    INV->v_sec_dq.d  = DSP_bw2Update(INV->v_sec_dq.d,  &INV->bw2_v_sec_d );
    INV->v_sec_dq.q  = DSP_bw2Update(INV->v_sec_dq.q,  &INV->bw2_v_sec_q );
#endif //#ifdef CONST_MAV_BUFFER_SIZE

    // Phase-locked loop routine
    AC3PH_pllUpdate(INV->v_grid_dq.q, &INV->pll_grid_abcn);
    AC3PH_pllUpdate(INV->v_sec_dq.q,  &INV->pll_sec_abc  );

    //
    // START OR STOP CONTROL ALGORITHM
    //

    // Proceed to the inverter control algorithm only in STARTED and PRECHARGE_ACTIVE states
    if (!((sm->state == CTRL_SM_STATE_STARTED) || (sm->state == INV_SM_STATE_PRECHARGE_ACTIVE)))
        return;

    //
    // INVERTER CONTROL ALGORITHM
    //

    // PI controller: DC link voltage
    // [controller is activated only after the active precharge is done]
    if (sm->state == CTRL_SM_STATE_STARTED)
    {
        INV->i_inv_ref_dq.d = CTRL_piUpdate((CONST_INV_V_DC_NOM - INV->meas->v_dc), &INV->pi_v_dc);
    }

    // PI controller: secondary current in dq frame
    AC3PH_dq_S v_inv_dq_pi;
    v_inv_dq_pi.d = CTRL_piUpdate((INV->i_inv_ref_dq.d - INV->i_sec_dq.d), &INV->pi_i_inv_d);
    v_inv_dq_pi.q = CTRL_piUpdate((INV->i_inv_ref_dq.q - INV->i_sec_dq.q), &INV->pi_i_inv_q);

    // Feed-forward control to compensate inverter model nonlinearity in dq frame
    AC3PH_dq_S v_inv_dq_ff;
    v_inv_dq_ff.d = INV->pll_sec_abc.w_est * CONST_INV_L_DQ * INV->i_sec_dq.q;
    v_inv_dq_ff.q = INV->pll_sec_abc.w_est * CONST_INV_L_DQ * INV->i_sec_dq.d;

    // Inverter voltage in RRF
    AC3PH_dq_S v_inv_ref_dq;
    v_inv_ref_dq.d = INV->v_sec_dq.d - (v_inv_dq_pi.d - v_inv_dq_ff.d);
    v_inv_ref_dq.q = INV->v_sec_dq.q - (v_inv_dq_pi.q + v_inv_dq_ff.q);

    //
    // SPACE-VECTOR MODULATION
    //

    // Inverter voltage in SRF
    AC3PH_ab_S v_inv_ref_ab;
    AC3PH_invPark(&v_inv_ref_dq, &v_inv_ref_ab, &INV->pll_sec_abc.phase);

    // Calculate magnitude and phase of the reference voltage vector
    //  magnitude:  sqrt(Re^2+Im^2)
    //  phase:      atan2(Im/Re)
    fp32_t v_inv_ref_mag = sqrtf((v_inv_ref_ab.alpha*v_inv_ref_ab.alpha)+(v_inv_ref_ab.beta*v_inv_ref_ab.beta));
    fp32_t v_inv_ref_phase = atan2f(v_inv_ref_ab.beta, v_inv_ref_ab.alpha);

    // Transform vector phase interval from (-PI,+PI) to (0,2PI)
    if (v_inv_ref_phase < 0)
    {
        v_inv_ref_phase += UTILS_2PI;
    }

    // Detect sector of the reference voltage vector
    // [must be in interval 0..5]
    // TODO consider using %6 if sector is not in interval 0..5
    uint32_t svm_sector = ((uint32_t) (v_inv_ref_phase/UTILS_PIBY3));

    // Calculate the vector angle relative to its sector start angle
    UTILS_angle_S svm_angle;
    svm_angle.theta = v_inv_ref_phase - ((fp32_t) svm_sector*UTILS_PIBY3);

    // Calculate sine and cosine of the svm_angle
    UTILS_calcSinAndCos(&svm_angle);

    // Calculate the maximum inverter voltage magnitude
    // [limited by the DC link voltage and vector position]
    fp32_t v_inv_ref_mag_lim = (UTILS_2BY3*INV->meas->v_dc) / (svm_angle._cos+UTILS_SQRT3BY3*svm_angle._sin);

    // Limit the inverter reference voltage magnitude
    if (v_inv_ref_mag > v_inv_ref_mag_lim)
    {
        v_inv_ref_mag = v_inv_ref_mag_lim;
    }

    // Calculate temporary variable used to calculate vector ON times. If DC link voltage is 0 V,
    // the inverter will be disabled by setting the null vector ON time to 100%.
    fp32_t svm_time = (INV->meas->v_dc <= 0.0f) ?
            0.0f : (UTILS_SQRT3*v_inv_ref_mag/INV->meas->v_dc);

    // Calculate ON times for vectors that span the SVM sector
//  fp32_t T1 = svm_time*sinf(UTILS_PIBY3-svm_angle.theta); // [use only if MCU has TMU]
    fp32_t T1 = svm_time*(UTILS_SQRT3BY2*svm_angle._cos-UTILS_1BY2*svm_angle._sin);
    fp32_t T2 = svm_time*svm_angle._sin;
    fp32_t T0 = 1.0f-T1-T2;

    // Limit active- and zero-vector ON times:
    //  - active vectors cannot be ON for more than 100% of the switching period, and
    //  - zero vector cannot be ON for less than 0% of the switching period.
    if (T0 < 0.0f)  { T0 = 0.0f; }              // <= 0% ?
    if (T1 > 1.0f)  { T1 = 1.0f; T2 = 0.0f; }   // >= 100% ?
    if (T2 > 1.0f)  { T2 = 1.0f; T1 = 0.0f; }   // >= 100% ?

    // Temporary variable used in symmetrical modulation
    fp32_t T0BY2 = UTILS_1BY2*T0;

    //
    // SPACE-VECTOR MODULATION LOOKUP TABLE
    //

    switch (INV->svm_type)
    {
        // V0 zero-vector is used in all sectors
        case INV_SVM_CONSTANT_ZERO_VECTOR :
            switch (svm_sector)
            {
                case 0 :
                    INV->duty->d1 = T1+T2;
                    INV->duty->d2 = T2;
                    INV->duty->d3 = 0.0f;
                    break;
                case 1 :
                    INV->duty->d1 = T1;
                    INV->duty->d2 = T1+T2;
                    INV->duty->d3 = 0.0f;
                    break;
                case 2 :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = T1+T2;
                    INV->duty->d3 = T2;
                    break;
                case 3 :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = T1;
                    INV->duty->d3 = T1+T2;
                    break;
                case 4 :
                    INV->duty->d1 = T2;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = T1+T2;
                    break;
                case 5 :
                    INV->duty->d1 = T1+T2;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = T1;
                    break;

                // SECTOR ERROR
                // Set zero-vector
                default :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = 0.0f;

                    ERR_FIELD_SET(CONFIG, SVM_SECTOR);
                    break;
            }
            break;

        // V0 zero-vector is used in {0,2,4} sectors, and
        // V7 zero-vector is used in {1,3,5} sectors
        case INV_SVM_ALTERNATING_ZERO_VECTOR :
            switch (svm_sector)
            {
                case 0 :
                    INV->duty->d1 = 1.0f;
                    INV->duty->d2 = T0+T2;
                    INV->duty->d3 = T0;
                    break;
                case 1 :
                    INV->duty->d1 = T1;
                    INV->duty->d2 = T1+T2;
                    INV->duty->d3 = 0.0f;
                    break;
                case 2 :
                    INV->duty->d1 = T0;
                    INV->duty->d2 = 1.0f;
                    INV->duty->d3 = T0+T2;
                    break;
                case 3 :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = T1;
                    INV->duty->d3 = T1+T2;
                    break;
                case 4 :
                    INV->duty->d1 = T0+T2;
                    INV->duty->d2 = T0;
                    INV->duty->d3 = 1.0f;
                    break;
                case 5 :
                    INV->duty->d1 = T1+T2;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = T1;
                    break;

                // SECTOR ERROR
                // Set zero-vector
                default :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = 0.0f;

                    ERR_FIELD_SET(CONFIG, SVM_SECTOR);
                    break;
            }
            break;

        // Both zero-vectors (V0 and V7) are used in all sectors
        case INV_SVM_SYMMETRICAL :
            switch (svm_sector)
            {
                case 0 :
                    INV->duty->d1 = T1+T2+T0BY2;
                    INV->duty->d2 = T2+T0BY2;
                    INV->duty->d3 = T0BY2;
                    break;
                case 1 :
                    INV->duty->d1 = T1+T0BY2;
                    INV->duty->d2 = T1+T2+T0BY2;
                    INV->duty->d3 = T0BY2;
                    break;
                case 2 :
                    INV->duty->d1 = T0BY2;
                    INV->duty->d2 = T1+T2+T0BY2;
                    INV->duty->d3 = T2+T0BY2;
                    break;
                case 3 :
                    INV->duty->d1 = T0BY2;
                    INV->duty->d2 = T1+T0BY2;
                    INV->duty->d3 = T1+T2+T0BY2;
                    break;
                case 4 :
                    INV->duty->d1 = T2+T0BY2;
                    INV->duty->d2 = T0BY2;
                    INV->duty->d3 = T1+T2+T0BY2;
                    break;
                case 5 :
                    INV->duty->d1 = T1+T2+T0BY2;
                    INV->duty->d2 = T0BY2;
                    INV->duty->d3 = T1+T0BY2;
                    break;

                // SECTOR ERROR
                // Set zero-vector
                default :
                    INV->duty->d1 = 0.0f;
                    INV->duty->d2 = 0.0f;
                    INV->duty->d3 = 0.0f;

                    ERR_FIELD_SET(CONFIG, SVM_SECTOR);
                    break;
            }
            break;

        // UNKNOWN MODULATION TYPE
        // Set zero-vector
        default :
            INV->duty->d1 = 0.0f;
            INV->duty->d2 = 0.0f;
            INV->duty->d3 = 0.0f;

            ERR_FIELD_SET(CONFIG, SVM_TYPE);
            break;
    }
#endif
}

#endif /* INVERTER_C_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
