//==============================================================================================//
//              INVERTER.H                                                                      //
//=============================================================================================c//

#ifndef INVERTER_H_
#define INVERTER_H_

#include "utils.h"

#include "system.h"
#include "control.h"
#include "dsp.h"
#include "error.h"

#include "constants.h"

#include "stopwatch.h"
#include "ac3ph.h"

//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

// These macro constants are used not to accidentally mix state machine flags
#define INV_ISR_COMMAND     sm->flag_1
#define INV_CAL_FLAG        sm->flag_2

//
// INVERTER-SPECIFIC STATE-MACHINE STATE FLAGS
//

// It must be noted that lower 3 bytes for the state-machine state flags are reserved, which means
// that state flags should start from at least 8. There are 4 predefined state-machine state flags
// (see control.h for more details):
//      CTRL_SM_STATE_RESET             0
//      CTRL_SM_STATE_START_OR_STOP     1
//      CTRL_SM_STATE_STARTED           2
//      CTRL_SM_STATE_STOPPED           3

#define INV_SM_STATE_DISCHARGE_DC_LINK              8
#define INV_SM_STATE_CALIBRATION_I                  9
#define INV_SM_STATE_PHASE_ORDER_DETECTION_I        10
#define INV_SM_STATE_PRECHARGE_PASSIVE_I            11
#define INV_SM_STATE_CALIBRATION_II                 12
#define INV_SM_STATE_PHASE_ORDER_DETECTION_II       13
#define INV_SM_STATE_WAIT_PLL_LOCK                  14
#define INV_SM_STATE_PRECHARGE_PASSIVE_II           15
#define INV_SM_STATE_PRECHARGE_ACTIVE               16

//
// INVERTER COMMANDS TO ISR
//

// CAUTION: Update INV_COMMAND_TABLE if you add new commands!

#define INV_ISR_COMMAND_DEFAULT_STATE               0
#define INV_ISR_COMMAND_DISCHARGE_START             1
#define INV_ISR_COMMAND_DISCHARGE_STOP              2
#define INV_ISR_COMMAND_PRECHARGE_START             3
#define INV_ISR_COMMAND_PRECHARGE_STOP              4
#define INV_ISR_COMMAND_INVERTER_START              5

//
// SPACE-VECTOR MODULATION TYPE FLAGS
//

// Different SVM types use different zero-vectors:
//  CONSTANT ZERO-VECTOR uses V0 in all sectors
//  ALTERNATING ZERO-VECTOR uses V0 in sectors {0,2,4}, and V7 in {1,3,5}
//  SYMMETRICAL (aka ALTERNATING REVERSING SEQUENCE) uses both zero-vectors in all sectors
// It should be noted that CONSTANT ZERO-VECTOR type should never be used, while ALTERNATING ZERO-
// VECTOR produces 33% less switching losses than SYMMETRICAL modulation, but it also has a worse
// current ripple (i.e., current THD).

#define INV_SVM_CONSTANT_ZERO_VECTOR                1
#define INV_SVM_ALTERNATING_ZERO_VECTOR             2
#define INV_SVM_SYMMETRICAL                         3

//==============================================================================================//
//              HANDLER DEFINITION                                                              //
//=============================================================================================c//

typedef struct INV_pack_STRUCT
{
    // Inverter state machine
    CTRL_sm_S state_machine;

    // State-machine commands interface
    uint32_t usr_command;       // User command for the state machine
    uint32_t isr_command;       // ISR command from the state machine

    // Space-vector modulation type
    uint32_t svm_type;

    // Grid and secondary voltage RMS measurements
    AC3PH_abcn_S v_grid_rms;    // Grid RMS voltage
    AC3PH_abc_S v_sec_rms;      // Secondary RMS voltage

    // Variables for phase order detection
    AC3PH_abcn_S v_phase_order;
    uint32_t sector1;
    uint32_t sector2;

    // Digital PI controllers
    CTRL_pi_S pi_v_dc;          // DC link voltage PI controller
    CTRL_pi_S pi_i_inv_d;       // Inverter (d-axis) current PI controller
    CTRL_pi_S pi_i_inv_q;       // Inverter (q-axis) current PI controller

#ifdef CONST_MAV_BUFFER_SIZE
    // Circular buffers for moving average filters
    // (arrays are saved in the memory row-wise)
    fp32_t buffer[2+2][CONST_MAV_BUFFER_SIZE];

    // Moving average filters
    DSP_mav_S mav_v_grid_d;
    DSP_mav_S mav_v_grid_q;
    DSP_mav_S mav_v_sec_d;
    DSP_mav_S mav_v_sec_q;
#else
    // Second-order Butterworth filters
    DSP_bw2_S bw2_v_grid_d;
    DSP_bw2_S bw2_v_grid_q;
    DSP_bw2_S bw2_v_sec_d;
    DSP_bw2_S bw2_v_sec_q;
#endif //#ifdef CONST_MAV_BUFFER_SIZE

    // Calibration structures
    DSP_stat_S stat_v_grid_an;
    DSP_stat_S stat_v_grid_bn;
    DSP_stat_S stat_v_grid_cn;
    DSP_stat_S stat_v_sec_ab;
    DSP_stat_S stat_v_sec_bc;
    DSP_stat_S stat_v_sec_ca;
    DSP_stat_S stat_i_sec_an;
    DSP_stat_S stat_i_sec_bn;
    DSP_stat_S stat_i_sec_cn;

    // Phase-locked loop
    AC3PH_pll_S pll_grid_abcn;      // PLL for grid (phase) voltage
    AC3PH_pll_S pll_sec_abc;        // PLL for secondary (line) voltage

    // alpha-beta frame variables
    AC3PH_ab_S v_grid_ab;
    AC3PH_ab_S v_sec_ab;
    AC3PH_ab_S i_sec_ab;

    // dq frame variables
    AC3PH_dq_S v_grid_dq;
    AC3PH_dq_S v_sec_dq;
    AC3PH_dq_S i_sec_dq;

    // Inverter current reference in dq frame
    AC3PH_dq_S i_inv_ref_dq;

    // Stopwatch to measure time
    SWC_tictoc_S stopwatch;

    // POINTERS TO OTHER STRUCTURES

    // Pointer to measurement structures
    volatile SYS_meas_S *raw;       // Structure with raw measurements
    volatile SYS_meas_S *meas;      // Structure with physical values

    // Pointer to duty-cycle structure
    volatile SYS_pwm3_S *duty;

    // Pointer to error structure
    volatile ERR_pack_S *error;

#ifdef CONST_CALIBRATION_BUFFER_SIZE
    // Pointer to measurements calibration structure
    volatile SYS_cal_S *meas_cal;
#endif

} INV_pack_S;

//==============================================================================================//
//              FUNCTION DECLARATIONS                                                           //
//=============================================================================================c//

interrupt void INV_isrEPwm4(void);

void INV_init(volatile INV_pack_S *INV_PACK);

void INV_doControl(void);

#endif /* INVERTER_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
