//==============================================================================================//
//              MCU.C                                                                           //
//=============================================================================================c//

#ifndef mcu_C_
#define mcu_C_

#include "mcu.h"

#include "system.h"
#include "constants.h"

#include "inverter.h"
#include "buckboost.h"
#include "task.h"

//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

//
// ADC FLAGS
//

// Burst conversion mode enable or disable flags
#define MCU_ADC_BURST_DISABLE       0
#define MCU_ADC_BURST_ENABLE        1

// ADC burst conversion mode trigger select flags
#define MCU_ADC_BURST_TRIGSEL_DISABLE           0x00

// ADC SOC trigger select flags
#define MCU_ADC_SOC_TRIGSEL_DISABLED            0x00
#define MCU_ADC_SOC_TRIGSEL_EPWM1_SOCA          0x05
#define MCU_ADC_SOC_TRIGSEL_EPWM2_SOCA          0x07
#define MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA          0x0B

//
//
// EPWM FLAGS
//

// EPWM SOC letter flags
#define MCU_EPWM_SOC_A              1
#define MCU_EPWM_SOC_B              2

// EPWM SOC and INT trigger event select flags
#define MCU_EPWM_TRIGSEL_TBCTR_0X00             0x1
#define MCU_EPWM_TRIGSEL_TBCTR_TBPRD            0x2
#define MCU_EPWM_TRIGSEL_TBCTR_0X00_OR_TBPRD    0x3

//==============================================================================================//
//              FUNCTION PROTOTYPES                                                             //
//=============================================================================================c//

static void MCU_initGpio(void);
static void MCU_initTmr(void);
static void MCU_initSci(void);
static void MCU_initAdc(void);
static void MCU_initEPwm(void);
static void MCU_initPie(void);

static void MCU_gpioCfgAsOutput(uint32_t pin, uint32_t value);

static void MCU_sciCfgModule(volatile struct SCI_REGS *SciRegs);
static void MCU_sciCfgTiming(volatile struct SCI_REGS *SciRegs, fp32_t baudrate_bps);

static void MCU_adcCfgModule(volatile struct ADC_REGS *AdcRegs);
static void MCU_adcCfgConvMode(volatile struct ADC_REGS *AdcRegs, uint16_t SOCPRIORITY, uint16_t BURSTSIZE, uint16_t TRIGSEL, uint16_t BURSTEN);
static void MCU_adcCfgSOC(volatile struct ADC_REGS *AdcRegs, uint16_t SOC, uint16_t CHSEL, uint16_t TRIGSEL, uint16_t ACQPS);
static void MCU_adcCfgInt(volatile struct ADC_REGS *AdcRegs, uint16_t INT, uint16_t SEL);

static void MCU_epwmCfgModule(volatile struct EPWM_REGS *EPwmRegs);
static void MCU_epwmCfgTiming(volatile struct EPWM_REGS *EPwmRegs, fp32_t freq_Hz, fp32_t deadtime_us);
static void MCU_epwmCfgPhaseAndSync(volatile struct EPWM_REGS *EPwmRegs, fp32_t phase_pu, uint16_t PHSEN, uint16_t SYNCOSEL, uint16_t SYNCOSELX);
static void MCU_epwmCfgAdcSOC(volatile struct EPWM_REGS *EPwmRegs, uint16_t SOC, uint16_t SEL);
static void MCU_epwmCfgInt(volatile struct EPWM_REGS *EPwmRegs, uint16_t INTSEL);

//==============================================================================================//
//              INITIALIZATION FUNCTIONS                                                        //
//=============================================================================================c//

void MCU_initSys(volatile ERR_pack_S *err)
{
    EALLOW;
        // The maximum EPWMCLK frequency is 100 MHz (see TMS320F28379D datasheet). For SYSCLK
        // above 100 MHz, the EPWMCLK must be half of PLLSYSCLK (i.e., divide-by-2).
        // [divide-by-2 is default on reset]
//      ClkCfgRegs.PERCLKDIVSEL.bit.EPWMCLKDIV = 1; // divide-by-2 (default)

        // Set SPI and SCI clock divider so that the SCI baud-rate error is minimal. Divide-by-1
        // clock prescaler only works with external oscillator (see TMS320F28379D datasheet), but
        // it also consumes more power compared to lower clock frequencies.
        // [divide-by-4 is default on reset]
//      ClkCfgRegs.LOSPCP.bit.LSPCLKDIV = 0x0; // divide-by-1
    EDIS;

    // Initialize system:
    //  - enable pull-ups on all unbounded IOs to reduce power consumption
    //  - if _LAUNCHXL_F28379D is defined, system clock is configured for 200 MHz operation
    //      (check this in "Project > Properties > Build > C2000 Compiler > Predefined Symbols")
    //  - initialize and start all peripheral clocks
    //  - disable the watchdog
    InitSysCtrl();

    // Set basic GPIO configuration:
    //  - configure all IO pins as DIGITAL INPUT
    //  - enable pull-up resistors on all pins
    //  - give control to CPU1 for all IO pins
    //  - disable open-drain and inverse polarity
    //  - set qualification to synchronous
    InitGpio();

    // Configure some GPIO pins as DIGITAL OUTPUT, and set safe state
    MCU_initGpio();

    // CPU timer is used to clock deterministic task scheduler
    MCU_initTmr();

    // Initialize SCI module
    MCU_initSci();

    // Initialize all 4 ADC modules
    MCU_initAdc();

    // Initialize all 5 ePWM modules
    MCU_initEPwm();

    // Initialize PIE module
    MCU_initPie();
}

//==============================================================================================//
//              STATIC FUNCTIONS - INITIALIZATION                                               //
//=============================================================================================c//

static void MCU_initGpio(void)
{
    //
    // CONFIGURE LED
    //

    // Two LEDs on the LaunchPad
    MCU_gpioCfgAsOutput(SYS_LED9,  SYS_LED_OFF);
    MCU_gpioCfgAsOutput(SYS_LED10, SYS_LED_OFF);

    //
    // CONFIGURE ALL RELAYS
    //

    MCU_gpioCfgAsOutput(SYS_RELAY_CONTACTOR_AC,      SYS_RELAY_CONTACTOR_AC_DISCONNECT);
    MCU_gpioCfgAsOutput(SYS_RELAY_CONTACTOR_DC,      SYS_RELAY_CONTACTOR_DC_DISCONNECT);
    MCU_gpioCfgAsOutput(SYS_RELAY_PRECHARGE_DC_LINK, SYS_RELAY_PRECHARGE_DC_LINK_ON   );
    MCU_gpioCfgAsOutput(SYS_RELAY_DISCHARGE_DC_LINK, SYS_RELAY_DISCHARGE_DC_LINK_OFF  );
    MCU_gpioCfgAsOutput(SYS_RELAY_DISCHARGE_OUTPUT,  SYS_RELAY_DISCHARGE_OUTPUT_OFF   );

    //
    // CONFIGURE FPGA PINS
    //

    // TODO one of these pins is ERROR indicator
    MCU_gpioCfgAsOutput(SYS_SPIB_CLK,    0);
    MCU_gpioCfgAsOutput(SYS_SPIB_CS,     0);
    MCU_gpioCfgAsOutput(SYS_SPIB_SIMO,   0);
    MCU_gpioCfgAsOutput(SYS_SPIB_SOMI,   0);
}

static void MCU_initTmr(void)
{
    //
    // CONFIGURE CPU TIMERS
    //

    // Initialize all CPU timers to a known state
    InitCpuTimers();

    // Configure CPU-Timer2 for system tick operation. Note that the ConfigCpuTimer() function
    // always uses divide-by-1 clock prescaler, i.e., timer operates at the system frequency.
    // CpuTimer2 is normally reserved for RTOS operation, but it can be used for other purposes
    // if RTOS is not used within the application.
    ConfigCpuTimer(&CpuTimer2, CONST_SYSTEM_FREQ_HZ/1.e6, CONST_SCHEDULER_CLOCK_US);

    // Enable CpuTimer2 interrupts
    CpuTimer2Regs.TCR.bit.TIE = 1;

    // Start CPU2 timer
    StartCpuTimer2();
}

static void MCU_initSci(void)
{
    // P18 is SCIB-TX, P19 is SCIB-RX
    GPIO_SetupPinMux(18, GPIO_MUX_CPU1, 2);
    GPIO_SetupPinMux(19, GPIO_MUX_CPU1, 2);

    // Configure GPIO
    GPIO_SetupPinOptions(18, GPIO_OUTPUT, GPIO_ASYNC   );
    GPIO_SetupPinOptions(19, GPIO_INPUT,  GPIO_PUSHPULL);

    // Configure SCI module
    MCU_sciCfgModule(&ScibRegs);
}

static void MCU_initAdc(void)
{
    // Set ADC resolution and signal mode before powering up. It must be noted that the resolution
    // and signal mode bits should not be directly configured in the ADCCTL2 registers, but they
    // must be configured strictly using the AdcSetMode() function.
    AdcSetMode(ADC_ADCA, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);
    AdcSetMode(ADC_ADCB, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);
    AdcSetMode(ADC_ADCC, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);
    AdcSetMode(ADC_ADCD, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);

    // Configure all ADC modules with basic settings:
    //  - ADC clock: ADCCLK=SYSCLK/4=50MHz (maximum ADC operating frequency)
    //  - Interrupt pulse generated on the end-of-conversion event
    //  - Powers up the ADC module and waits 1 ms for everything to settle
    MCU_adcCfgModule(&AdcaRegs);
    MCU_adcCfgModule(&AdcbRegs);
    MCU_adcCfgModule(&AdccRegs);
    MCU_adcCfgModule(&AdcdRegs);

    // Configure ePWM SOC trigger pulses.
    // Triggers are configured to occur on every event, instead of every 2nd or 3rd event. Note
    // that ePWM1 and ePWM2 are phase-shifted by 50% of the PWM period, i.e., TBCTR=0x00 of ePWM1
    // occurs on TBCTR=TBPRD of ePWM2, and vice versa.
    MCU_epwmCfgAdcSOC(&EPwm1Regs, MCU_EPWM_SOC_A, MCU_EPWM_TRIGSEL_TBCTR_0X00);
    MCU_epwmCfgAdcSOC(&EPwm2Regs, MCU_EPWM_SOC_A, MCU_EPWM_TRIGSEL_TBCTR_0X00);
    MCU_epwmCfgAdcSOC(&EPwm4Regs, MCU_EPWM_SOC_A, MCU_EPWM_TRIGSEL_TBCTR_0X00);

    // Configure conversion mode for all ADC modules:
    //  - no SOC is in high-priority mode, i.e., they are all part of the round-robin wheel
    //  - disable burst mode conversion, TRIGSEL bit in SOCxCTL registers controls conversion
    MCU_adcCfgConvMode(&AdcaRegs, 0, 1, MCU_ADC_BURST_TRIGSEL_DISABLE, MCU_ADC_BURST_DISABLE);
    MCU_adcCfgConvMode(&AdcbRegs, 0, 1, MCU_ADC_BURST_TRIGSEL_DISABLE, MCU_ADC_BURST_DISABLE);
    MCU_adcCfgConvMode(&AdccRegs, 0, 1, MCU_ADC_BURST_TRIGSEL_DISABLE, MCU_ADC_BURST_DISABLE);
    MCU_adcCfgConvMode(&AdcdRegs, 0, 1, MCU_ADC_BURST_TRIGSEL_DISABLE, MCU_ADC_BURST_DISABLE);

    // Configure buck-boost SOC wrappers
    MCU_adcCfgSOC(&AdcaRegs, 10, 4, MCU_ADC_SOC_TRIGSEL_EPWM2_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcaRegs, 11, 1, MCU_ADC_SOC_TRIGSEL_EPWM2_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcaRegs, 12, 0, MCU_ADC_SOC_TRIGSEL_EPWM2_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcaRegs, 13, 5, MCU_ADC_SOC_TRIGSEL_EPWM1_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcaRegs, 14, 2, MCU_ADC_SOC_TRIGSEL_EPWM1_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcaRegs, 15, 3, MCU_ADC_SOC_TRIGSEL_EPWM1_SOCA, CONST_ADC_SAMPLE_WINDOW);

    // Configure inverter SOC wrappers
    MCU_adcCfgSOC(&AdcbRegs, 12, 4, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcbRegs, 13, 2, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcbRegs, 14, 5, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcbRegs, 15, 3, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdccRegs, 12, 4, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdccRegs, 13, 2, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdccRegs, 14, 5, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdccRegs, 15, 3, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcdRegs, 12, 2, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcdRegs, 13, 0, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcdRegs, 14, 3, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);
    MCU_adcCfgSOC(&AdcdRegs, 15, 1, MCU_ADC_SOC_TRIGSEL_EPWM4_SOCA, CONST_ADC_SAMPLE_WINDOW);

    // All ADC modules are configured to generate EOC pulse on the end-of-conversion event, which
    // is configured in INTPULSEPOS bit in ADCCTL1 register (see MCU_adcCfgModule() function).
    // The alternative is to generate EOC pulse on the end-of-acquisition event.
    // The EOC flag will be latched until cleared.
    // (TODO check if EOC flag is cleared when new conversion starts)
    MCU_adcCfgInt(&AdcaRegs, 1, 15);    // ADCINT1 on EOC15
    MCU_adcCfgInt(&AdcaRegs, 2, 12);    // ADCINT2 on EOC12
    MCU_adcCfgInt(&AdcbRegs, 1, 15);    // ADCINT1 on EOC15
    MCU_adcCfgInt(&AdccRegs, 1, 15);    // ADCINT1 on EOC15
    MCU_adcCfgInt(&AdcdRegs, 1, 15);    // ADCINT1 on EOC15
}

static void MCU_initEPwm(void)
{
    EALLOW;
        // Stop time base for all ePWM modules
        CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0;
    EDIS;

    // Configure ePWM modules with basic settings:
    //  - ePWM time base clock: TBCLK=SYSCLK/2=100MHz
    //  - Up-down-count mode (aka center-aligned mode)
    //  - ePWM channel has two outputs in complementary mode
    //  - Dead-time on rising edge for both outputs
    //  - Enable and disable some shadow registers
    //  - Reset all timing registers
    MCU_epwmCfgModule(SYS_PWM_BUB_L1);      // X2 on DC LINK board (L2)
    MCU_epwmCfgModule(SYS_PWM_BUB_L2);      // X1 on DC LINK board (L1)
    MCU_epwmCfgModule(SYS_PWM_INV_L1);      // X5 on DC LINK board (L1)
    MCU_epwmCfgModule(SYS_PWM_INV_L2);      // X6 on DC LINK board (L2)
    MCU_epwmCfgModule(SYS_PWM_INV_L3);      // X7 on DC LINK board (L3)

    // Set frequency and dead-time for all ePWM modules
    MCU_epwmCfgTiming(SYS_PWM_BUB_L1, CONST_BUB_FREQUENCY_HZ, CONST_BUB_DEADTIME_US);
    MCU_epwmCfgTiming(SYS_PWM_BUB_L2, CONST_BUB_FREQUENCY_HZ, CONST_BUB_DEADTIME_US);
    MCU_epwmCfgTiming(SYS_PWM_INV_L1, CONST_INV_FREQUENCY_HZ, CONST_INV_DEADTIME_US);
    MCU_epwmCfgTiming(SYS_PWM_INV_L2, CONST_INV_FREQUENCY_HZ, CONST_INV_DEADTIME_US);
    MCU_epwmCfgTiming(SYS_PWM_INV_L3, CONST_INV_FREQUENCY_HZ, CONST_INV_DEADTIME_US);

    // Set synchronization input only for EPWM4 generator. Note that our MCU (TMS320F28379D) does
    // not have EPWM7 and EPWM10 generators, and you cannot set synchronization input for EPWM1.
    // SYNCIN lines of other PWM generators (EPWM2, EPWM3, EPWM5, EPWM6) are connected to SYNCOUT
    // line of the PWM generator that is one position earlier in the synchronization chain, e.g.,
    //      EPWM1SYNCOUT->EPWM2SYNCIN,
    //      EPWM2SYNCOUT->EPWM3SYNCIN,
    //      EPWM4SYNCOUT->EPWM5SYNCIN,
    //      EPWM5SYNCOUT->EPWM6SYNCIN.
    // However, it must be noted that synchronization is disabled on EPWM4 generator (PHSEN=0), as
    // we do not have to synchronize inverter to buck-boost PWM generators.
//  SyncSocRegs.SYNCSELECT.bit.EPWM4SYNCIN = 0b000;     // EPWM1OUT->EPWM4IN

    // ePWM module time base synchronization. Phase shift needs to be enabled in order to be able
    // to synchronize the time base of that ePWM module. When synchronization event occurs, the
    // TBPHS register value is loaded to the corresponding TBCTR register.
    // The ePWM1 module is master with independent time base (TB_DISABLE), and generates sync.
    // pulse on its TB_CTR_ZERO, which is propagated to EPWM2SYNCIN. Since ePWM2 has phase shift
    // enabled, as soon as it receives the sync. pulse, it will load its TBPHS register value to
    // its TBCTR register. The ePWM2 module will not generate sync. pulse on its output, since it
    // is configured with SYNCOSEL=TB_SYNC_DISABLE and SYNCOSELX=0x0.
    // The ePWM4 module is master with independent time base (TB_DISABLE), and generates sync.
    // pulse on its TB_CTR_ZERO, which is propagated to EPWM5SYNCIN. ePWM5 module is configured
    // to pass through its SYNCIN signal to its output, which means that ePWM6 module will also
    // receive sync. pulse from the ePWM4 module. The ePWM5 and ePWM6 modules will synchronize to
    // ePWM4 sync. pulse (TBCTR=0x00), since they have phase shift enabled (TB_ENABLE).
    MCU_epwmCfgPhaseAndSync(SYS_PWM_BUB_L2, 0.0f, TB_DISABLE, TB_CTR_ZERO,     0x0);    //     disabled -> ePWM1SYNCIN, ePWM1SYNCOUT <- ePWM1 TBCTR=0x00
    MCU_epwmCfgPhaseAndSync(SYS_PWM_BUB_L1, 0.5f, TB_ENABLE,  TB_SYNC_DISABLE, 0x0);    // ePWM1SYNCOUT -> ePWM2SYNCIN, ePWM2SYNCOUT <- disabled
    MCU_epwmCfgPhaseAndSync(SYS_PWM_INV_L1, 0.0f, TB_DISABLE, TB_CTR_ZERO,     0x0);    //     disabled -> ePWM4SYNCIN, ePWM4SYNCOUT <- ePWM4 TBCTR=0x00
    MCU_epwmCfgPhaseAndSync(SYS_PWM_INV_L2, 0.0f, TB_ENABLE,  TB_SYNC_IN,      0x0);    // ePWM4SYNCOUT -> ePWM5SYNCIN, ePWM5SYNCOUT <- ePWM5SYNCIN (ePWM4SYNCOUT)
    MCU_epwmCfgPhaseAndSync(SYS_PWM_INV_L3, 0.0f, TB_ENABLE,  TB_SYNC_DISABLE, 0x0);    // ePWM5SYNCOUT -> ePWM6SYNCIN, ePWM6SYNCOUT <- disabled

    // Configure interrupt trigger pulses for ePWM2 and ePWM4 modules
    MCU_epwmCfgInt(SYS_PWM_BUB_L1, MCU_EPWM_TRIGSEL_TBCTR_0X00);
    MCU_epwmCfgInt(SYS_PWM_INV_L1, MCU_EPWM_TRIGSEL_TBCTR_0X00);

    // Map ePWM modules to their corresponding GPIO pins. ePWM channels are mapped to GPIO after
    // ePWMx module has been configured to avoid possible unknown states on GPIO output. Even if
    // there is a glitch on GPIO output, nothing is going to happen on the inverter, as FPGA will
    // not let through the glitch, since PWM is globally disabled on FPGA at this point.
    InitEPwm1Gpio();    // SYS_PWM_BUB_L2
    InitEPwm2Gpio();    // SYS_PWM_BUB_L1
    InitEPwm4Gpio();    // SYS_PWM_INV_L1
    InitEPwm5Gpio();    // SYS_PWM_INV_L2
    InitEPwm6Gpio();    // SYS_PWM_INV_L3

    EALLOW;
        // Start ePWM time base for all ePWM modules
        // TODO Check from which value EPWM counters start counting (TBPHS or something else?)
        CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;
    EDIS;
}

static void MCU_initPie(void)
{
    // Disable Global interrupt INTM
    DINT;

    // Clear all interrupts and initialize PIE vector table
    InitPieCtrl();
    IER = 0x0000;
    IFR = 0x0000;
    InitPieVectTable();

    // Map ISR functions
    EALLOW;
        PieVectTable.ADCA1_INT  = &BUB_isrAdca1;
        PieVectTable.EPWM2_INT  = &BUB_isrEPwm2;
        PieVectTable.EPWM4_INT  = &INV_isrEPwm4;
        PieVectTable.TIMER2_INT = &TASK_isrCpuTimer2;
    EDIS;

    // Enable group interrupt flags
    //  - M_INT1    ADCA1
    //  - M_INT3    ePWM2, ePWM4
    //  - M_INT14   CPU-Timer2
    IER |= (M_INT1 | M_INT3 | M_INT14);

    // Enable Global interrupt INTM
    EINT;

    // Enable Global real-time interrupt DBGM
    ERTM;   // (TODO do we really need this?)

    // Enable PIE interrupts
    // [for some peripherals we do this in mainCpu1.c]
//  PieCtrlRegs.PIEIER1.bit.INTx1 = 1;      // ADCA1 on EOC15
//  PieCtrlRegs.PIEIER3.bit.INTx2 = 1;      // ePWM2 on TBCTR=0x00
//  PieCtrlRegs.PIEIER3.bit.INTx4 = 1;      // ePWM4 on TBCTR=0x00
}

//==============================================================================================//
//              STATIC FUNCTIONS - GPIO                                                         //
//=============================================================================================c//

static void MCU_gpioCfgAsOutput(uint32_t pin, uint32_t value)
{
    // Write initial state to GPIO to avoid possible unwanted states during configuration
    GPIO_WritePin(pin, value);

    // Configure GPIO pin as digital output
    GPIO_SetupPinOptions(pin, GPIO_OUTPUT, 0);
}

//==============================================================================================//
//              STATIC FUNCTIONS - SCI                                                          //
//=============================================================================================c//

static void MCU_sciCfgModule(volatile struct SCI_REGS *SciRegs)
{
    // TODO REVIEW

    // Communication control register
    SciRegs->SCICCR.bit.STOPBITS = 0;       // One stop bit
    SciRegs->SCICCR.bit.PARITY = 0;         // Odd parity
    SciRegs->SCICCR.bit.PARITYENA = 0;      // Parity check is disabled
    SciRegs->SCICCR.bit.LOOPBKENA = 0;      // Internal loop back test mode is disabled
    SciRegs->SCICCR.bit.ADDRIDLE_MODE = 0;  // Idle-line mode protocol is selected
    SciRegs->SCICCR.bit.SCICHAR = 0x7;      // 8-bit data length

    // Control register 1
    SciRegs->SCICTL1.bit.RXERRINTENA = 0;   // Disable interrupt on SCI receive error
    // TODO ...
    SciRegs->SCICTL1.bit.TXENA = 1;         // Enable SCI transmitter
    SciRegs->SCICTL1.bit.RXENA = 1;         // Enable SCI receiver

    // Configure SCI baud-rate registers
    MCU_sciCfgTiming(SciRegs, CONST_SCI_BAUDRATE_BPS);
}

static void MCU_sciCfgTiming(volatile struct SCI_REGS *SciRegs, fp32_t baudrate_bps)
{
    if (baudrate_bps<=0)
    {
        // TODO ERROR
        return;
    }

    // Get SPI and SCI clock prescaler
    uint16_t LSPCLKDIV = ClkCfgRegs.LOSPCP.bit.LSPCLKDIV;

    // Calculate SCI clock frequency
    uint16_t lspclkdiv = (LSPCLKDIV==0x0) ? 1 : (2*LSPCLKDIV);
    fp32_t freq_Hz = CONST_SYSTEM_FREQ_HZ / lspclkdiv;

    // Calculate value for the baud-rate register
    int32_t BAUD = (int32_t) (freq_Hz / (baudrate_bps*8.0f) - 1.0f + 0.5f);

    // Minimum baud-rate value is 1
    if (BAUD<=0)    BAUD = 1;

    // Calculate baud-rate error
//  fp32_t baudrate_err = ((freq_Hz / ((BAUD+1)*8)) - baudrate_bps) / baudrate_bps * 100.0;

    // Configure baud-rate registers
    SciRegs->SCILBAUD.bit.BAUD = ((uint16_t) (BAUD>>0) & 0x00FF);
    SciRegs->SCIHBAUD.bit.BAUD = ((uint16_t) (BAUD>>8) & 0x00FF);
}

//==============================================================================================//
//              STATIC FUNCTIONS - ADC                                                          //
//=============================================================================================c//

static void MCU_adcCfgModule(volatile struct ADC_REGS *AdcRegs)
{
    EALLOW;
        // Interrupt pulse generation occurs at the end-of-conversion event
        // [alternative is the end-of-acquisition event]
        AdcRegs->ADCCTL1.bit.INTPULSEPOS = 1;

        // ADC module clock prescaler
        // [ADCCLK=SYSCLK/4=50MHz]
        AdcRegs->ADCCTL2.bit.PRESCALE = 0x6;

        // Power up the ADC
        AdcRegs->ADCCTL1.bit.ADCPWDNZ = 1;
    EDIS;

    // 1-ms delay for ADC to power-up
    DELAY_US(1000);
}

static void MCU_adcCfgConvMode(volatile struct ADC_REGS *AdcRegs, uint16_t SOCPRIORITY, uint16_t BURSTSIZE, uint16_t TRIGSEL, uint16_t BURSTEN)
{
    if (SOCPRIORITY>16 || (BURSTSIZE==0 || BURSTSIZE>16) || TRIGSEL>31)
    {
        // TODO ERROR
        return;
    }

    EALLOW;
        // Set how many SOC wrappers are in high-priority mode, e.g.,
        //  - for SOCPRIORITY=0,  no SOC is in high-priority mode,
        //  - for SOCPRIORITY=1,  only SOC0 is in high-priority mode,
        //  - for SOCPRIORITY=5,  SOC0-SOC4 are in high-priority mode,
        //  - for SOCPRIORITY=16, SOC0-SOC15 are in high-priority mode.
        // SOC wrappers that are in high-priority mode are not part of the round-robin wheel.
        AdcRegs->ADCSOCPRICTL.bit.SOCPRIORITY = SOCPRIORITY;

        // Disable burst mode
        AdcRegs->ADCBURSTCTL.bit.BURSTEN = 0;

        // Configure burst mode
        AdcRegs->ADCBURSTCTL.bit.BURSTSIZE = BURSTSIZE-1;
        AdcRegs->ADCBURSTCTL.bit.BURSTTRIGSEL = TRIGSEL;

        // Disable or enable burst mode
        AdcRegs->ADCBURSTCTL.bit.BURSTEN = BURSTEN;
    EDIS;
}

static void MCU_adcCfgSOC(volatile struct ADC_REGS *AdcRegs, uint16_t SOC, uint16_t CHSEL, uint16_t TRIGSEL, uint16_t ACQPS)
{
    // Sanity check
    if (SOC>15 || CHSEL>15 || TRIGSEL>16 || (ACQPS==0 || ACQPS>512))
    {
        // TODO ERROR
        return;
    }

    // Get pointer to the ADC SOC control register
    volatile struct ADCSOC0CTL_BITS *ADCSOCxCTL = &AdcRegs->ADCSOC0CTL.bit + SOC;

    EALLOW;
        // SOC trigger source select
        ADCSOCxCTL->TRIGSEL = TRIGSEL;

        // SOC channel select
        ADCSOCxCTL->CHSEL   = CHSEL;

        // Acquisition window duration
        ADCSOCxCTL->ACQPS   = ACQPS-1;
    EDIS;
}

static void MCU_adcCfgInt(volatile struct ADC_REGS *AdcRegs, uint16_t INT, uint16_t SEL)
{
    if ((INT==0 || INT>4) || (SEL&0xF)!=SEL)
    {
        // TODO ERROR
        return;
    }

    // Get pointer to the interrupt configuration register
    uint16_t *ADCINTSEL = (uint16_t *) &AdcRegs->ADCINTSEL1N2.all + ((INT-1)>>1);

    // Build code for the ADCINTSEL register:
    //  7   RSVD    Reserved
    //  6   CONT    Continuously generate pulses (TODO check this!)
    //  5   E       Enable interrupt
    //  4   RSVD    Reserved
    //  3:0 SEL     Set EOC source for INT flag
    // (INT-1)&0x1 is 1 when INT is 2 or 4, and 0 when INT is 1 or 3
//  uint16_t CODE = ((0x0040 | SEL) << ((INT-1)&0x1)*8);    // CONT=0, E=1
    uint16_t CODE = ((0x0060 | SEL) << ((INT-1)&0x1)*8);    // CONT=1, E=1

    // Write configuration code to the ADCINTSEL register
    EALLOW;
        *ADCINTSEL |= CODE;
    EDIS;

    // Clear interrupt flag
    AdcRegs->ADCINTFLGCLR.all = (1U<<(INT-1));
}

//==============================================================================================//
//              STATIC FUNCTIONS - EPWM                                                         //
//=============================================================================================c//

static void MCU_epwmCfgModule(volatile struct EPWM_REGS *EPwmRegs)
{

    // The ePWM time base clock is determined by EPWMCLKDIV, CLKDIV, and HSPCLKDIV bits:
    //  EPWMCLK = PLLCLK/EPWMCLKDIV (100 MHz)
    //  TBCLK = EPWMCLK/(HSPCLKDIV*CLKDIV) (100 MHz)

    // Time base control register
    EPwmRegs->TBCTL.bit.FREE_SOFT = 0x3;            // Run free during emulation events
    EPwmRegs->TBCTL.bit.CLKDIV = TB_DIV1;           // Time base clock prescale: divide-by-1
    EPwmRegs->TBCTL.bit.HSPCLKDIV = TB_DIV1;        // High-speed time base clock prescale: divide-by-1
    EPwmRegs->TBCTL.bit.PHSEN = TB_DISABLE;         // Phase shift (and time base synchronization) is disabled by default
    EPwmRegs->TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Counter mode: up-down-count mode
    EPwmRegs->TBCTL.bit.PHSDIR = TB_UP;             // Count up after a synchronization event. This makes sense only for up-down-count mode.

#ifdef MCU_PWM_COUNT_UP
    EPwmRegs->TBCTL.bit.CTRMODE = TB_COUNT_UP;
#endif //#ifdef MCU_PWM_COUNT_UP

    // Time base synchronization will be configured in function MCU_epwmCfgPhaseAndSync(), and
    // here we set only some default settings, i.e., by default this ePWM module will not send
    // a synchronization pulse to the ePWM module that is one position later in the sync. chain.
    // In order for this ePWM module to be able to synchronize its time base to the time base of
    // ePWM module one position earlier in the sync. chain, PHSEN bit of this ePWM module needs
    // to be enabled.
    EPwmRegs->TBCTL.bit.SYNCOSEL = TB_SYNC_DISABLE; // SYNCO is defined by SYNCOSELX
    EPwmRegs->TBCTL2.bit.SYNCOSELX = 0x0;           // SYNCOSELX is set to "Disable"

    // EPWMSYNCINSEL register does not exist. See SYNCSELECT register in SYNCSOCREGS.

    // Do not use shadow register for TBPRD, as we do not plan to change the PWM period once it
    // has been configured.
    EPwmRegs->TBCTL.bit.PRDLD = TB_IMMEDIATE;       // Immediate mode. Shadow register is not used for TBPRD
    EPwmRegs->TBCTL2.bit.PRDLDSYNC = 0x0;           // Has no effect in immediate mode

    // Configure shadow register for CMPA:CMPAHR register
    EPwmRegs->CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow register is used for CMPA:CMPAHR registers
    EPwmRegs->CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Shadow to active CMPA:CMPAHR occurs on TBCTR=0x00
    EPwmRegs->CMPCTL.bit.LOADASYNC = 0x0;           // Shadow to active CMPA:CMPAHR occurs on event defined only by LOADAMODE bits

    // Do not use shadow register for dead-band generator control register
    EPwmRegs->DBCTL2.bit.SHDWDBCTLMODE = 0;         // Immediate mode. Shadow register is not used for DBCTL.
    EPwmRegs->DBCTL2.bit.LOADDBCTLMODE = 0x0;       // Has no effect in immediate mode.

    // Rising edge delay is added to both outputs
    EPwmRegs->DBCTL.bit.HALFCYCLE = 0;              // The dead-band counters are clocked at the TBCLK rate
    EPwmRegs->DBCTL.bit.IN_MODE = DBA_RED_DBB_FED;  // S4: ePWMxA is source for RED; S5: ePWMxB is source for FED
    EPwmRegs->DBCTL.bit.DEDB_MODE = 0;              // S8: not used
    EPwmRegs->DBCTL.bit.POLSEL = DB_ACTV_HIC;       // S2: RED; S3: Invert FED
    EPwmRegs->DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // S1: PathA=RED, S0: PathB=FED
    EPwmRegs->DBCTL.bit.OUTSWAP = 0x0;              // S6: OutA=PathA; S7: OutB=PathB

    // Do not use shadow registers for DBRED and DBFED registers, as we do not plan to change
    // them once they are configured.
    EPwmRegs->DBCTL.bit.SHDWDBREDMODE = 0;          // Immediate mode. Shadow register is not used for DBRED.
    EPwmRegs->DBCTL.bit.SHDWDBFEDMODE = 0;          // Immediate mode. Shadow register is not used for DBFED.
    EPwmRegs->DBCTL.bit.LOADREDMODE = 0x0;          // Has no effect in immediate mode.
    EPwmRegs->DBCTL.bit.LOADFEDMODE = 0x0;          // Has no effect in immediate mode.

    // Do not use shadow registers for AQCTLA and AQCTLB registers, as we do not plan to change
    // them once they are configured.
    EPwmRegs->AQCTL.bit.SHDWAQAMODE = 0;            // Immediate mode. Shadow register is not used for AQCTLA.
    EPwmRegs->AQCTL.bit.SHDWAQBMODE = 0;            // Immediate mode. Shadow register is not used for AQCTLB.
    EPwmRegs->AQCTL.bit.LDAQAMODE = 0x0;            // Has no effect in immediate mode.
    EPwmRegs->AQCTL.bit.LDAQBMODE = 0x0;            // Has no effect in immediate mode.
    EPwmRegs->AQCTL.bit.LDAQASYNC = 0x0;            // Has no effect in immediate mode.
    EPwmRegs->AQCTL.bit.LDAQBSYNC = 0x0;            // Has no effect in immediate mode.

    // These bits determine action on CMPA event for the two outputs (ePWMxA and ePWMxB) from the
    // action qualifier submodule. ePWMxA/B at the output of AQ are defined to be the same:
    //  PWMxA=LOW for CMPA<TBCTR, PWMxA=HIGH for CMPA>TBCTR
    //  PWMxB=LOW for CMPA<TBCTR, PWMxB=HIGH for CMPA>TBCTR
    EPwmRegs->AQCTLA.bit.ZRO = AQ_NO_ACTION;        // Do nothing on TBCTR=0x00
    EPwmRegs->AQCTLA.bit.PRD = AQ_NO_ACTION;        // Do nothing on TBCTR=TBPRD
    EPwmRegs->AQCTLA.bit.CAU = AQ_CLEAR;            // Force ePWMxA to low (up-count)
    EPwmRegs->AQCTLA.bit.CAD = AQ_SET;              // Force ePWMxA to high (down-count)
    EPwmRegs->AQCTLB.bit.ZRO = AQ_NO_ACTION;        // Do nothing on TBCTR=0x00
    EPwmRegs->AQCTLB.bit.PRD = AQ_NO_ACTION;        // Do nothing on TBCTR=TBPRD
    EPwmRegs->AQCTLB.bit.CAU = AQ_CLEAR;            // Force ePWMxB to low (up-count)
    EPwmRegs->AQCTLB.bit.CAD = AQ_SET;              // Force ePWMxB to high (down-count)

#ifdef MCU_PWM_COUNT_UP
    // Used only in up- or down-count mode
    EPwmRegs->AQCTLA.bit.ZRO = AQ_SET;
    EPwmRegs->AQCTLB.bit.ZRO = AQ_SET;
#endif //#ifdef MCU_PWM_COUNT_UP

    // Action qualifier continuous S/W force register
    EPwmRegs->AQSFRC.bit.RLDCSF = 0x0;              // Shadow register is used for AQCSFRC. Shadow to active AQCSFRC occurs on TBCTR=0x00
    EPwmRegs->AQSFRC.bit.ACTSFA = 0x0;              // Disable one-time software trigger for shadow to active for output A
    EPwmRegs->AQSFRC.bit.ACTSFB = 0x0;              // Disable one-time software trigger for shadow to active for output B

    // Force both ePWM outputs to low
    MCU_epwmEnableOutputs(EPwmRegs, MCU_EPWM_OUTPUT_DISABLE);

    // Configure shadow registers for CMPAHR
    // TODO should we use HRPWM?

    // Clear timing registers
    EPwmRegs->TBCTR = 0x0000;           // ePWM clock counter
    EPwmRegs->TBPRD = 0;                // ePWM frequency
    EPwmRegs->CMPA.bit.CMPA = 0;        // ePWM duty-cycle
    EPwmRegs->DBRED.bit.DBRED = 0;      // ePWM rising-edge delay
    EPwmRegs->DBFED.bit.DBFED = 0;      // ePWM falling-edge delay
}

static void MCU_epwmCfgTiming(volatile struct EPWM_REGS *EPwmRegs, fp32_t freq_Hz, fp32_t deadtime_us)
{
    // Get ePWM clock dividers
    uint16_t EPWMCLKDIV = ClkCfgRegs.PERCLKDIVSEL.bit.EPWMCLKDIV;
    uint16_t CLKDIV     = EPwmRegs->TBCTL.bit.CLKDIV;
    uint16_t HSPCLKDIV  = EPwmRegs->TBCTL.bit.HSPCLKDIV;

    // Transform configuration bits to divide coefficients
    uint16_t epwmclkdiv = 1U << EPWMCLKDIV; // 1,2
    uint16_t clkdiv     = 1U << CLKDIV;     // 1,2,4,8,16,32,64,128
    uint16_t hspclkdiv  = (HSPCLKDIV==0x0) ? 1 : (2*HSPCLKDIV); // 1,2,4,6,8,10,12,14

    // ePWM time base clock frequency
    fp32_t tbFreq_hz = (fp32_t) CONST_SYSTEM_FREQ_HZ / (epwmclkdiv*clkdiv*hspclkdiv);

    // Get ePWM count mode
    uint16_t CTRMODE = EPwmRegs->TBCTL.bit.CTRMODE;

    // Calculate ePWM timing registers
    switch (CTRMODE)
    {
        case TB_COUNT_UP:
        case TB_COUNT_DOWN:
            EPwmRegs->TBPRD = (uint16_t)((tbFreq_hz/freq_Hz)-1.0f+0.5f);
            EPwmRegs->DBRED.bit.DBRED = (uint16_t)((freq_Hz*deadtime_us/1e6)*(EPwmRegs->TBPRD+1.0f)+0.5f);
            EPwmRegs->DBFED.bit.DBFED = EPwmRegs->DBRED.bit.DBRED;
            return;

        case TB_COUNT_UPDOWN:
            EPwmRegs->TBPRD = (uint16_t)((tbFreq_hz/freq_Hz)/2.0f+0.5f);
            EPwmRegs->DBRED.bit.DBRED = (uint16_t)((freq_Hz*deadtime_us/1e6)*(EPwmRegs->TBPRD*2.0f)+0.5f);
            EPwmRegs->DBFED.bit.DBFED = EPwmRegs->DBRED.bit.DBRED;
            return;

        case TB_FREEZE:
        default :
            // TODO ERROR
            return;
    }
}

static void MCU_epwmCfgPhaseAndSync(volatile struct EPWM_REGS *EPwmRegs, fp32_t phase_pu, uint16_t PHSEN, uint16_t SYNCOSEL, uint16_t SYNCOSELX)
{
    // Disable phase shift (and time base synchronization)
    EPwmRegs->TBCTL.bit.PHSEN = 0;

    // Check if phase shift should be enabled for the ePWM module. Note that if phase shift is
    // enabled, this will also enable time base synchronization with the ePWM module that is one
    // position earlier in the synchronization chain. If you don't want to synchronize time base
    // of the given ePWM module, make sure to disable the SYNCOUT of the ePWM module that is one
    // position earlier in the synchronization chain.
    if (PHSEN==TB_ENABLE)
    {
        // Get ePWM count mode
        uint16_t CTRMODE = EPwmRegs->TBCTL.bit.CTRMODE;

        // Phase shift
        switch (CTRMODE)
        {
            case TB_COUNT_UP:
            case TB_COUNT_DOWN:
                EPwmRegs->TBPHS.bit.TBPHS = (uint16_t) (phase_pu*(EPwmRegs->TBPRD+1.0f));
                break;
            case TB_COUNT_UPDOWN:
                EPwmRegs->TBPHS.bit.TBPHS = (uint16_t) (phase_pu*(EPwmRegs->TBPRD*2.0f));
                break;
            case TB_FREEZE:
            default :
                return;
        }

        // Enable phase shift. Time base synchronization will also be enabled.
        EPwmRegs->TBCTL.bit.PHSEN = 1;
    }

    // Configure SYNCOUT, which is directly connected to SYNCIN of the ePWM module that is one
    // position later in the synchronization chain. For example, ePWM2SYNCIN is connected to
    // ePWM1SYNCOUT etc. SYNCOUT configuration bits:
    //  -------------------------------------------------
    //  CODE    TBCTL[SYNCOSEL]         TBCTL2[SYNCOSELX]
    //  -------------------------------------------------
    //  0x0     EPWMxSYNCI/SWFSYNC      Disable
    //  0x1     TBCTR=0x00              TBCTR=CMPC
    //  0x2     TBCTR=CMPB              TBCTR=CMPD
    //  0x3    *EPWMxSYNCO              Reserved
    //  -------------------------------------------------
    //  *If SYNCOSEL is defined as EPWMxSYNCO, then SYNCOSELX bits determine SYNCOUT signal.
    EPwmRegs->TBCTL.bit.SYNCOSEL    = SYNCOSEL;
    EPwmRegs->TBCTL2.bit.SYNCOSELX  = SYNCOSELX;
}

static void MCU_epwmCfgAdcSOC(volatile struct EPWM_REGS *EPwmRegs, uint16_t SOC, uint16_t SEL)
{
    // SOC?PRD determines conversion frequency
    // [? is A or B]
    EPwmRegs->ETPS.bit.SOCPSSEL = 0;

    switch (SOC)
    {
        case MCU_EPWM_SOC_A :
            EPwmRegs->ETSEL.bit.SOCAEN   = 0;       // Disable SOC
            EPwmRegs->ETSEL.bit.SOCASEL  = SEL;     // SOC trigger event select
            EPwmRegs->ETPS .bit.SOCAPRD  = 0x1;     // SOC is triggered on every trigger event
            EPwmRegs->ETSEL.bit.SOCAEN   = 1;       // Enable SOC
            return;

        case MCU_EPWM_SOC_B :
            EPwmRegs->ETSEL.bit.SOCBEN   = 0;       // Disable SOC
            EPwmRegs->ETSEL.bit.SOCBSEL  = SEL;     // SOC trigger event select
            EPwmRegs->ETPS .bit.SOCBPRD  = 0x1;     // SOC is triggered on every trigger event
            EPwmRegs->ETSEL.bit.SOCBEN   = 1;       // Enable SOC
            return;

        default :
            return;
    }
}

static void MCU_epwmCfgInt(volatile struct EPWM_REGS *EPwmRegs, uint16_t INTSEL)
{
    EPwmRegs->ETPS. bit.INTPRD   = 0x1;         // Interrupt pulse is generated on every event
    EPwmRegs->ETPS. bit.INTPSSEL = 0;           // INTPRD determines trigger frequency
    EPwmRegs->ETSEL.bit.INTSEL   = INTSEL;      // Interrupt trigger event select
    EPwmRegs->ETSEL.bit.INTEN    = 1;           // Enable ePWM interrupt generation
    EPwmRegs->ETCLR.bit.INT      = 1;           // Clear interrupt flag
}

#endif /* MCU_C_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
