//==============================================================================================//
//              STOPWATCH.H                                                                     //
//=============================================================================================c//

#ifndef STOPWATCH_H_
#define STOPWATCH_H_

#include "utils.h"

#include "error.h"

//==============================================================================================//
//              STRUCTURE DEFINITION                                                            //
//=============================================================================================c//

typedef struct SWC_tictoc_STRUCT
{
    fp32_t PERIOD_US;           // Dedicated timer period in microseconds
    fp32_t TICKS_PER_US;        // Number of timer ticks per microsecond
    volatile uint32_t *tim;     // Pointer to counter register of the dedicated timer
    uint32_t tic;               // Dedicated timer counter value on stopwatch start
    uint32_t toc;               // Dedicated timer counter value on stopwatch stop
    fp32_t time_us;             // Number of ticks translated to microseconds
} SWC_tictoc_S;

//==============================================================================================//
//              FUNCTION PROTOTYPES                                                             //
//=============================================================================================c//

_Bool SWC_init(volatile SWC_tictoc_S *stopwatch, volatile uint32_t *timer, uint32_t PERIOD_REG, fp32_t PERIOD_US);

//==============================================================================================//
//              STOPWATCH FUNCTIONS                                                             //
//=============================================================================================c//

UTILS_INLINE void SWC_start(volatile SWC_tictoc_S *stopwatch);
UTILS_INLINE void SWC_stop(volatile SWC_tictoc_S *stopwatch);

UTILS_INLINE void SWC_start(volatile SWC_tictoc_S *stopwatch)
{
    // Take the timer counter value on stopwatch start
    stopwatch->tic = *stopwatch->tim;
}

UTILS_INLINE void SWC_stop(volatile SWC_tictoc_S *stopwatch)
{
    // Take the timer counter value on stopwatch stop
    stopwatch->toc = *stopwatch->tim;

    // Calculate stopwatch value
    stopwatch->time_us = (stopwatch->tic - stopwatch->toc) / stopwatch->TICKS_PER_US;

    // Account for possible counter overflow
    if (stopwatch->time_us < 0)
    {
        stopwatch->time_us += stopwatch->PERIOD_US;
    }
}

#endif /* STOPWATCH_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
