# run_qat_ti.py
import torch
import torch.nn as nn
import os
import joblib
import importlib.util # <--- For dynamic importing
from tqdm import tqdm

# --- Custom Imports ---
from rampupV6.data import PumpDataModule # Data module is still a project dependency
import tinyml_torchmodelopt.quantization as tinpu_quantization

# ==============================================================================
#      CONFIGURATION
# ==============================================================================
# This is the ONLY path you need to change
ARTIFACT_DIR = "training_runs/FullParamsEmbeddedSmall_20251021-143330/packaged_model"

# QAT settings
DATA_FILE = "dataset/rampupV6_train.npz"
FINE_TUNE_EPOCHS = 15
LEARNING_RATE = 1e-6
BATCH_SIZE = 256
DEVICE = 'cpu'

# These constants must match what was used in train.py
SIM_CONSTANTS_FOR_LOSS = {
    'alpha1': 4.03, 'gamma1': 2.14e-02, 'C': 6e-4,
    'max_rpm': 13000.0, 'max_rpm_acc' : 25400, 'dt' : 0.0025
} 
# ==============================================================================

def dynamically_load_model_class(artifact_dir):
    """Dynamically loads the PumpParameterRegressor class from the package."""
    model_def_path = os.path.join(artifact_dir, "model_definition.py")
    if not os.path.exists(model_def_path):
        raise FileNotFoundError(f"Cannot find 'model_definition.py' in {artifact_dir}")
        
    spec = importlib.util.spec_from_file_location("model_definition", model_def_path)
    model_module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(model_module)
    
    # This assumes your PL module is named PumpParameterRegressor in the .py file
    if not hasattr(model_module, 'PumpParameterRegressor'):
         raise ImportError(f"Could not find 'PumpParameterRegressor' in {model_def_path}")

    print(f"✅ Dynamically loaded model definition from: {model_def_path}")
    return model_module.PumpParameterRegressor

def evaluate_model(model, dataloader, device='cpu'):
    """Helper function to evaluate MSE loss."""
    model.eval()
    loss_fn = nn.MSELoss()
    total_loss = 0
    with torch.no_grad():
        for inputs, labels_scaled, _ in tqdm(dataloader, desc="Evaluating"):
            inputs = inputs.to(device).unsqueeze(1).unsqueeze(3)
            labels_scaled = labels_scaled.to(device)
            predictions = model(inputs)
            loss = loss_fn(predictions, labels_scaled)
            total_loss += loss.item()
    return total_loss / len(dataloader)


if __name__ == '__main__':

    # --- 1. Load Data ---
    print("--- 1. Loading Data ---")
    dm = PumpDataModule(data_path=DATA_FILE, batch_size=BATCH_SIZE, train_ratio=0.9)
    dm.setup()
    train_dataloader = dm.train_dataloader()
    val_dataloader = dm.val_dataloader()
    print("✅ Data loaded.")

    # --- 2. Dynamically Load the "White Box" Model ---
    print("\n--- 2. Loading 'White Box' Model from Artifact Package ---")
    
    CHECKPOINT_PATH = os.path.join(ARTIFACT_DIR, "best_model.ckpt")
    SCALERS_PATH = os.path.join(ARTIFACT_DIR, "scalers.joblib")
    
    PumpParameterRegressor = dynamically_load_model_class(ARTIFACT_DIR)
    scalers = joblib.load(SCALERS_PATH)
    
    model_pl = PumpParameterRegressor.load_from_checkpoint(
        CHECKPOINT_PATH,
        y_scaler=scalers['y_scaler'],
        constants=SIM_CONSTANTS_FOR_LOSS
    )
    model_fp32 = model_pl.model.to(DEVICE).eval() # Get the raw nn.Module
    print("✅ 'White Box' model loaded successfully.")

    # --- 3. Establish FP32 Baseline ---
    print("\n--- 3. Establishing FP32 Performance Baseline ---")
    baseline_loss = evaluate_model(model_fp32, val_dataloader, device=DEVICE)
    print(f"🎯 FP32 Baseline MSE Loss: {baseline_loss:.6f}")

    # --- 4. Prepare and Run TI QAT ---
    print("\n--- 4. Wrapping model for TI QAT ---")
    dummy_input = torch.randn(1, 1, dm.input_size, 1, device=DEVICE)
    ti_qat_model = tinpu_quantization.TINPUTinyMLQATFxModule(
        model_fp32, total_epochs=FINE_TUNE_EPOCHS, example_inputs=dummy_input
    )
    ti_qat_model.to(DEVICE)

    optimizer = torch.optim.SGD(ti_qat_model.parameters(), lr=LEARNING_RATE)
    loss_fn = nn.MSELoss()

    print("\n--- 5. Starting QAT Fine-Tuning ---")
    for epoch in range(FINE_TUNE_EPOCHS):
        ti_qat_model.train()
        progress_bar = tqdm(train_dataloader, desc=f"QAT Epoch {epoch+1}/{FINE_TUNE_EPOCHS}")
        for inputs, labels, _ in progress_bar:
            optimizer.zero_grad()
            predictions = ti_qat_model(inputs.to(DEVICE).unsqueeze(1).unsqueeze(3))
            loss = loss_fn(predictions, labels.to(DEVICE))
            loss.backward()
            optimizer.step()
            progress_bar.set_postfix({'loss': loss.item()})
    print("✅ QAT Fine-tuning complete.")

    # --- 6. Convert, Evaluate, and Export INT8 Model ---
    print("\n--- 6. Converting, Evaluating, and Exporting TI INT8 Model ---")
    ti_qat_model.eval()
    
    quantized_loss = evaluate_model(ti_qat_model, val_dataloader, device=DEVICE)
    
    print("\n--- 📊 TI QAT RESULTS ---")
    print(f"FP32 Baseline Loss:   {baseline_loss:.6f}")
    print(f"INT8 QAT Loss:        {quantized_loss:.6f}")
    print(f"Performance Change:   {quantized_loss - baseline_loss:+.6f}")

    model_int8 = ti_qat_model.convert()
    # Export the new INT8 'black box' model
    try:
        import torch.export 
        print("\nExporting 'black box' (model_int8_ti.ep)...")
        exported_int8_program = torch.export.export(model_int8.to('cpu'), (dummy_input.to('cpu'),))
        
        exported_int8_path = os.path.join(ARTIFACT_DIR, "model_int8_ti.ep")
        torch.export.save(exported_int8_program, exported_int8_path)
        print(f"✅ Exported TI INT8 'black box' to: {exported_int8_path}")

    except Exception as e:
        print(f"!!!!!!!!!!!!!\nFailed to export INT8 model: {e}\n!!!!!!!!!!!!!")