//###########################################################################
//
// FILE:   sci_ex2_loopback_interrupts.c
//
// TITLE:  SCI Digital Loop Back with Interrupts.
//
//! \addtogroup driver_example_list
//! <h1> SCI Digital Loop Back with Interrupts </h1>
//!
//!  This test uses the internal loop back test mode of the peripheral.
//!  Other then boot mode pin configuration, no other hardware configuration
//!  is required. Both interrupts and the SCI FIFOs are used.
//!
//!  A stream of data is sent and then compared to the received stream.
//!  The SCI-A sent data looks like this: \n
//!  00 01 \n
//!  01 02 \n
//!  02 03 \n
//!  .... \n
//!  FE FF \n
//!  FF 00 \n
//!  etc.. \n
//!  The pattern is repeated forever.
//!
//!  \b Watch \b Variables \n
//!  - \b sDataA - Data being sent
//!  - \b rDataA - Data received
//!  - \b rDataPointA - Keep track of where we are in the data stream.
//!    This is used to check the incoming data
//!
//
//###########################################################################
//
//
// $Copyright: $
//###########################################################################

//
// Included Files
//
#include "driverlib.h"
#include "device.h"

//
// Globals
//

//
// Send data for SCI-C
//
uint16_t sDataA[2];

//
// Received data for SCI-C
//
uint16_t rDataA[2];

//
// Used for checking the received data
//
uint16_t rDataPointA;

//
// Function Prototypes
//
__interrupt void scicTXFIFOISR(void);
__interrupt void scicRXFIFOISR(void);
void initSCICFIFO(void);
void error(void);

//
// Main
//
void main(void)
{
    uint16_t i;

    //
    // Initialize device clock and peripherals
    //
    Device_init();

    //
    // Setup GPIO by disabling pin locks and enabling pullups
    //
    Device_initGPIO();

    //
    // GPIO28 is the SCI Rx pin.
    //
    GPIO_setMasterCore(DEVICE_GPIO_PIN_SCIRXDC, GPIO_CORE_CPU1);
    GPIO_setPinConfig(DEVICE_GPIO_CFG_SCIRXDC);
    GPIO_setDirectionMode(DEVICE_GPIO_PIN_SCIRXDC, GPIO_DIR_MODE_IN);
    GPIO_setPadConfig(DEVICE_GPIO_PIN_SCIRXDC, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(DEVICE_GPIO_PIN_SCIRXDC, GPIO_QUAL_ASYNC);

    //
    // GPIO29 is the SCI Tx pin.
    //
    GPIO_setMasterCore(DEVICE_GPIO_PIN_SCITXDC, GPIO_CORE_CPU1);
    GPIO_setPinConfig(DEVICE_GPIO_CFG_SCITXDC);
    GPIO_setDirectionMode(DEVICE_GPIO_PIN_SCITXDC, GPIO_DIR_MODE_OUT);
    GPIO_setPadConfig(DEVICE_GPIO_PIN_SCITXDC, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(DEVICE_GPIO_PIN_SCITXDC, GPIO_QUAL_ASYNC);

    //
    // Initialize PIE and clear PIE registers. Disables CPU interrupts.
    //
    Interrupt_initModule();

    //
    // Initialize the PIE vector table with pointers to the shell Interrupt
    // Service Routines (ISR).
    //
    Interrupt_initVectorTable();

    //
    // Interrupts that are used in this example are re-mapped to
    // ISR functions found within this file.
    //
    Interrupt_register(INT_SCIC_RX, scicRXFIFOISR);
    Interrupt_register(INT_SCIC_TX, scicTXFIFOISR);

    //
    // Initialize the Device Peripherals:
    //
    initSCICFIFO();

    //
    // Init the send data.  After each transmission this data
    // will be updated for the next transmission
    //
    for(i = 0; i < 2; i++)
    {
        sDataA[i] = i;
    }

    rDataPointA = sDataA[0];

    Interrupt_enable(INT_SCIC_RX);
    Interrupt_enable(INT_SCIC_TX);

    Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP9);

    //
    // Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
    //
    EINT;
    ERTM;

    //
    // IDLE loop. Just sit and loop forever (optional):
    //
    for(;;);
}

//
// error - Function to halt debugger on error
//
void error(void)
{
    asm("     ESTOP0"); // Test failed!! Stop!
    for (;;);
}

//
// sciaTXFIFOISR - SCIC Transmit FIFO ISR
//
__interrupt void scicTXFIFOISR(void)
{
//    uint16_t i;
//
//    SCI_writeCharArray(SCIA_BASE, sDataA, 2);
//
//    //
//    // Increment send data for next cycle
//    //
//    for(i = 0; i < 2; i++)
//    {
//        sDataA[i] = (sDataA[i] + 1) & 0x00FF;
//    }
//
//    SCI_clearInterruptStatus(SCIA_BASE, SCI_INT_TXFF);
//
//    //
//    // Issue PIE ACK
//    //
//    Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP9);
}

//
// sciaRXFIFOISR - SCIC_BASE Receive FIFO ISR
//
__interrupt void scicRXFIFOISR(void)
{
    uint16_t i;

    SCI_readCharArray(SCIC_BASE, rDataA, 2);

    //
    // Check received data
    //
//    for(i = 0; i < 2; i++)
//    {
//        if(rDataA[i] != ((rDataPointA + i) & 0x00FF))
//        {
//            error();
//        }
//    }

//    rDataPointA = (rDataPointA + 1) & 0x00FF;

    SCI_clearOverflowStatus(SCIC_BASE);

    SCI_clearInterruptStatus(SCIC_BASE, SCI_INT_RXFF);

    //
    // Issue PIE ack
    //
    Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP9);
}

//
// initSCICFIFO - Configure SCIC FIFO
//
void initSCICFIFO()
{
    //
    // 8 char bits, 1 stop bit, no parity. Baud rate is 921600.
    //
    SCI_setConfig(SCIC_BASE, DEVICE_LSPCLK_FREQ, 921600, (SCI_CONFIG_WLEN_8 |
                                                        SCI_CONFIG_STOP_ONE |
                                                        SCI_CONFIG_PAR_NONE));
    SCI_enableModule(SCIC_BASE);
    //SCI_enableLoopback(SCIC_BASE);
    SCI_resetChannels(SCIC_BASE);
    SCI_enableFIFO(SCIC_BASE);

    //
    // RX and TX FIFO Interrupts Enabled
    //
    SCI_enableInterrupt(SCIC_BASE, (SCI_INT_RXFF | SCI_INT_TXFF));
    SCI_disableInterrupt(SCIC_BASE, SCI_INT_RXERR);

    //
    // The transmit FIFO generates an interrupt when FIFO status
    // bits are less than or equal to 2 out of 16 words
    // The receive FIFO generates an interrupt when FIFO status
    // bits are greater than equal to 2 out of 16 words
    //
    SCI_setFIFOInterruptLevel(SCIC_BASE, SCI_FIFO_TX2, SCI_FIFO_RX2);
    SCI_performSoftwareReset(SCIC_BASE);

    SCI_resetTxFIFO(SCIC_BASE);
    SCI_resetRxFIFO(SCIC_BASE);
}

//
// End of file
//
