//###########################################################################
//
// FILE:   Example_2806xScib_FFDLB.c
//
// TITLE:  SCI Digital Loop Back Example
//
//!  \addtogroup f2806x_example_list
//!  <h1>SCI Digital Loop Back(scib_loopback)</h1>
//!
//!  This program uses the internal loop back test mode of the peripheral.
//!  Other then boot mode pin configuration, no other hardware configuration
//!  is required.
//!  
//!  This test uses the loopback test mode of the SCI module to send
//!  characters starting with 0x00 through 0xFF.  The test will send
//!  a character and then check the receive buffer for a correct match.
//!
//!  \b Watch \b Variables \n
//!  - \b LoopCount , Number of characters sent
//!  - \b ErrorCount , Number of errors detected
//!  - \b SendChar , Character sent
//!  - \b ReceivedChar , Character received
//
//###########################################################################
// $TI Release: F2806x Support Library v2.06.00.00 $
// $Release Date: Fri Feb 12 19:15:11 IST 2021 $
// $Copyright:
// Copyright (C) 2009-2021 Texas Instruments Incorporated - http://www.ti.com/
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions 
// are met:
// 
//   Redistributions of source code must retain the above copyright 
//   notice, this list of conditions and the following disclaimer.
// 
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the 
//   documentation and/or other materials provided with the   
//   distribution.
// 
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// $
//###########################################################################

//
// Included Files
//
#include "DSP28x_Project.h"     // Device Headerfile and Examples Include File
//#include "F2806x_Sci.h"


//
// Function Prototypes
//
void scib_loop_init(void);
void scib_fifo_init(void);
void scib_xmit(int a);
void error();
void serial_read_info(Uint16  *buf, Uint16  rcv, Uint16  count);
__interrupt void scib_rx_isr(void);
__interrupt void scib_tx_isr(void);

//
// Globals
//
Uint16 LoopCount;
Uint16 ErrorCount;
Uint16  i = 0;

//
// Main
//
void main(void)
{
    Uint16 SendChar;
    Uint16 ReceivedChar;
    Uint16  Info_Array[16] = {0};
    Uint16  rx_in;


    //
    // Step 1. Initialize System Control registers, PLL, WatchDog, Clocks to 
    // default state: This function is found in the F2806x_SysCtrl.c file.
    //
    InitSysCtrl();

    //
    // Step 2. Select GPIO for the device or for the specific application:
    // This function is found in the F2806x_Gpio.c file.
    //
    InitSciGpio();

    //
    // Step 3. Initialize PIE vector table:
    // The PIE vector table is initialized with pointers to shell Interrupt
    // Service Routines (ISR).  The shell routines are found in 
    // F2806x_DefaultIsr.c. Insert user specific ISR code in the appropriate 
    // shell ISR routine in the DSP28_DefaultIsr.c file.
    //
    
    // Disable and clear all CPU interrupts
    //
    DINT;
    IER = 0x0000;
    IFR = 0x0000;
    //
    // Initialize Pie Control Registers To Default State:
    // This function is found in the F2806x_PieCtrl.c file.
    //
    //InitPieCtrl();  PIE is not used for this example
    //
    // Initialize the PIE Vector Table To a Known State:
    // This function is found in F2806x_PieVect.c.
    // This function populates the PIE vector table with pointers
    // to the shell ISR functions found in F2806x_DefaultIsr.c.
    //

    InitPieVectTable();
    //
    // Enable CPU and PIE interrupts
    // This example function is found in the F2806x_PieCtrl.c file.
    //

    EnableInterrupts();

    //
    // Step 4. Initialize all the Device Peripherals to a known state:
    // This function is found in F2806x_InitPeripherals.c
    //
    //InitPeripherals(); skip this for SCI tests

    //
    // Step 5. User specific functions, Reassign vectors (optional), 
    // Enable Interrupts:
    //
    LoopCount = 0;
    ErrorCount = 0;
    scib_fifo_init();	   // Initialize the SCI FIFO
    scib_loop_init();  // Initalize SCI for digital loop back
    //
    // Note: Autobaud lock is not required for this example
    //

    //
    // Send a character starting with 0
    //
    SendChar = 0;

    //
    // Step 6.
    //
    for(;;)
    {
        


        //
        // wait for RRDY/RXFFST = 1 for 1 data available in FIFO
        //
        if(ScibRegs.SCIFFRX.bit.RXFFST == 1)
        {
            rx_in = ScibRegs.SCIRXBUF.all;
            serial_read_info(Info_Array, rx_in, 16);

        }


        LoopCount++;
    }
}

//
// Step 7. Insert all local Interrupt Service Routines (ISRs) and functions 
// here:
//

//
// error - 
//
void
error(void)
{
    ErrorCount++;
    //__asm("     ESTOP0");  // Uncomment to stop the test here
    //for (;;);
}

//
// scib_loop_init - Test 1,SCIB  DLB, 8-bit word, baud rate 0x000F, No loopback,
// default, 1 STOP bit, no parity
//
void
scib_loop_init()
{
    //
    // Note: Clocks were turned on to the SCIB peripheral
    // in the InitSysCtrl() function
    //
    
    //
    // 1 stop bit, No loopback, No parity,8 char bits, async mode,
    // idle-line protocol
    //
    ScibRegs.SCICCR.all =0x0007;
    
    //
    // enable TX, RX, internal SCICLK, Disable RX ERR, SLEEP, TXWAKE
    //
    ScibRegs.SCICTL1.all =0x0003;
    ScibRegs.SCICTL2.all =0x0003;
    ScibRegs.SCICTL2.bit.TXINTENA = 0;
    ScibRegs.SCICTL2.bit.RXBKINTENA = 0;
    ScibRegs.SCIHBAUD    =0x0000;
    ScibRegs.SCILBAUD    =0x000F;
    ScibRegs.SCICCR.bit.LOOPBKENA =0; // Disable loop back
    ScibRegs.SCICTL1.all =0x0023;     // Relinquish SCI from Reset
}

//
// scib_xmit - Transmit a character from the SCI'
//
void
scib_xmit(int a)
{
    ScibRegs.SCITXBUF=a;
}

//
// scib_fifo_init - Initalize the SCI FIFO
//
void
scib_fifo_init()
{
    ScibRegs.SCIFFTX.all=0xE040; // Enable Tx FIFO with Enhancements. Empty FIFO. Clear Interrupt Flag.
    ScibRegs.SCIFFRX.all=0x2040; //was 0x2044. Enable Rx FIFO. Clear FIFO Interrupt.
                                 // Rx Interrupt generated when # bytes >= 1.
    ScibRegs.SCIFFCT.all=0x0;// AutoBAUD disabled. Delay = 0.
}

void
serial_read_info(Uint16  *info, Uint16  rcv_in, Uint16  icount)
{
    if (i < icount)
    {
        info[i] = rcv_in;
        i++;
    }
}

//
// End of File
//

