/*
 *  ======== mutex.c ========
 *  This example shows the use of two tasks and one semaphore to perform
 *  mutual exclusive data access.
 */

#include <xdc/std.h>
#include <xdc/runtime/System.h>

#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Semaphore.h>

#include <xdc/cfg/global.h>

#include "F2837xD_gpio.h"
#include "F2837xD_Gpio_defines.h"
#include "F2837xD_sysctrl.h"
#include "F2837xD_ipc.h"

#define  EINT   __asm(" clrc INTM")
#define  DINT   __asm(" setc INTM")
#define  ERTM   __asm(" clrc DBGM")
#define  DRTM   __asm(" setc DBGM")
#define  EALLOW __asm(" EALLOW")
#define  EDIS   __asm(" EDIS")
#define  ESTOP0 __asm(" ESTOP0")

#define C1C2_BROM_BOOTMODE_BOOT_FROM_FLASH                     0x0000000B


//----------------------------------------
#ifdef CPU1
#ifdef __cplusplus
#pragma DATA_SECTION("GpioCtrlRegsFile")
#else
#pragma DATA_SECTION(GpioCtrlRegs,"GpioCtrlRegsFile");
#endif
volatile struct GPIO_CTRL_REGS GpioCtrlRegs;
#endif

//----------------------------------------
#ifdef __cplusplus
#pragma DATA_SECTION("GpioDataRegsFile")
#else
#pragma DATA_SECTION(GpioDataRegs,"GpioDataRegsFile");
#endif
volatile struct GPIO_DATA_REGS GpioDataRegs;
//----------------------------------------
#ifdef __cplusplus
#pragma DATA_SECTION("ClkCfgRegsFile")
#else
#pragma DATA_SECTION(ClkCfgRegs,"ClkCfgRegsFile");
#endif
volatile struct CLK_CFG_REGS ClkCfgRegs;
//----------------------------------------
#ifdef __cplusplus
#pragma DATA_SECTION("CpuSysRegsFile")
#else
#pragma DATA_SECTION(CpuSysRegs,"CpuSysRegsFile");
#endif
volatile struct CPU_SYS_REGS CpuSysRegs;
//----------------------------------------
#ifdef __cplusplus
#pragma DATA_SECTION("DevCfgRegsFile")
#else
#pragma DATA_SECTION(DevCfgRegs,"DevCfgRegsFile");
#endif
volatile struct DEV_CFG_REGS DevCfgRegs;
//----------------------------------------
#ifdef __cplusplus
#pragma DATA_SECTION("IpcRegsFile")
#else
#pragma DATA_SECTION(IpcRegs,"IpcRegsFile");
#endif
volatile struct IPC_REGS_CPU1 IpcRegs;



Void task1(UArg arg0, UArg arg1);
Void task2(UArg arg0, UArg arg1);

extern void GPIO_SetupPinMux(Uint16 pin, Uint16 cpu, Uint16 peripheral);
extern uint16_t IPCBootCPU2(uint32_t ulBootMode);


Int resource = 0;
Semaphore_Handle sem;
Task_Handle tsk1;
Task_Handle tsk2;

Int finishCount = 0;


void initializeGPIOs(void)
{
    //Initialize GPIOs for the LEDs and turn them off
    EALLOW;
    GpioCtrlRegs.GPADIR.bit.GPIO31 = 1;
    GpioCtrlRegs.GPBDIR.bit.GPIO34 = 1;
    GpioDataRegs.GPADAT.bit.GPIO31 = 1;
//    GpioDataRegs.GPBDAT.bit.GPIO34 = 1;

    GPIO_SetupPinMux(34, GPIO_MUX_CPU2, 0);


    EDIS;
}

void doNothing(void)
{
    __asm(" NOP");
}


/*
 *  ======== main ========
 */
Int main()
{ 
    Task_Params taskParams;
        
    /* Create a Semaphore object to be use as a resource lock */
    sem = Semaphore_create(1, NULL, NULL);

    /* Create two tasks that share a resource*/
    Task_Params_init(&taskParams);
    taskParams.priority = 1;
    tsk1 = Task_create (task1, &taskParams, NULL);
    
    Task_Params_init(&taskParams);
    taskParams.priority = 2;
    tsk2 = Task_create (task2, &taskParams, NULL);


    initializeGPIOs();

    GpioDataRegs.GPASET.bit.GPIO31 = 1;
//    GpioDataRegs.GPBCLEAR.bit.GPIO34 = 1;

    //
    // Send boot command to allow the CPU2 application to begin execution
    //
    //   if(C2TOC1BOOTSTS)
    IPCBootCPU2(C1C2_BROM_BOOTMODE_BOOT_FROM_FLASH);

    BIOS_start();    /* does not return */
    return(0);
}

/*
 *  ======== task1 ========
 */
Void task1(UArg arg0, UArg arg1)
{
    UInt32 time;

    for (;;) {
        System_printf("Running task1 function\n");

        if (Semaphore_getCount(sem) == 0) {
            System_printf("Sem blocked in task1\n");
        }

        /* Get access to resource */
        Semaphore_pend(sem, BIOS_WAIT_FOREVER); 

        /* do work by waiting for 2 system ticks to pass */
        time = Clock_getTicks();
        while (Clock_getTicks() <= (time + 1)) {
            ;
        }

        /* do work on locked resource */
        resource += 1;
        /* unlock resource */
        GpioDataRegs.GPATOGGLE.bit.GPIO31 = 1;
//        GpioDataRegs.GPBTOGGLE.bit.GPIO34 = 1;

        Semaphore_post(sem);

        Task_sleep(1000);
    }
}

/*
 *  ======== task2 ========
 */
Void task2(UArg arg0, UArg arg1)
{
    for (;;) {
        System_printf("Running task2 function\n");

        if (Semaphore_getCount(sem) == 0) {
            System_printf("Sem blocked in task2\n");
        }

        /* Get access to resource */
        Semaphore_pend(sem, BIOS_WAIT_FOREVER);

        /* do work on locked resource */
        resource += 1;
        /* unlock resource */
        
        Semaphore_post(sem);

        
        Task_sleep(1000);
        
        finishCount++;
        if (finishCount == 5) {
            System_printf("Calling BIOS_exit from task2\n");
//            BIOS_exit(0);
        }
    }
}

