/**
 *****************************************************************************************
 * @file F28335Adc.cpp
 * @details See header file for module overview.
 * @copyright 2020 Eaton Corporation. All Rights Reserved.
 *
 *****************************************************************************************
 */
#include "F28335Adc.h"

/*
 *****************************************************************************************
 *		Static variables
 *****************************************************************************************
 */
static const float32 ADC_VOLTS_MEDIUM_PLL = 1.5f;
static const float32 ADC_VOLTS_PER_PLL = 123.320f;
static const float32 ADC_OFFSET_PLL = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_HALL = 1.5f;
static const float32 ADC_VOLTS_PER_HALL = 589.931f;
static const float32 ADC_OFFSET_HALL_REC_IAC = 0.0f;
static const float32 ADC_OFFSET_HALL_INV_IAC = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_VOLT = 1.5f;
static const float32 ADC_VOLTS_PER_VOLT = 496.771f;
static const float32 ADC_OFFSET_VOLT_REC_VAC = 0.0f;
static const float32 ADC_OFFSET_VOLT_INV_VAC = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_VDC = 0.0f;
static const float32 ADC_VOLTS_PER_VDC = 287.026f;
static const float32 ADC_OFFSET_VDC = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_CT = 1.5f;
static const float32 ADC_VOLTS_PER_CT = 268.097f;
static const float32 ADC_OFFSET_CT = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_TEMP = 0.0f;
static const float32 ADC_VOLTS_PER_TEMP = 0.02f;
static const float32 ADC_OFFSET_TEMP = 0.0f;

static const float32 ADC_VOLTS_MEDIUM_IFB = 0.0f;
static const float32 ADC_VOLTS_PER_IFB = 1.0f;
static const float32 ADC_OFFSET_IFB_P15 = 0.0f;
static const float32 ADC_OFFSET_IFB_P5 = 0.0f;
static const float32 ADC_OFFSET_IFB_P12 = 0.0f;
static const float32 ADC_OFFSET_IFB_P24 = 0.0f;
static const float32 ADC_OFFSET_IFB_REC = 0.0f;
static const float32 ADC_OFFSET_IFB_INV = 0.0f;

static const float32 PLLA_Fltr_KC = 0;
static const float32 Rec_Iac_Fltr_KC = 0;
static const float32 IFB_Fltr_KC = 0;
static const float32 Temp_Fltr_KC = 0;
static const float32 TapIac_Fltr_KC = 0;
static const float32 Vdc_Fltr_KC = 0;
static const float32 Vac_Fltr_KC = 0;

/*
 *****************************************************************************************
 *		Globals
 *****************************************************************************************
 */
float32 PLL_CL7_Previous = 0;
float32 Rec_Iac_Prev = 0;
float32 Inv_Iac_Prev = 0;
float32 IFB_P15_Prev = 0;
float32 IFB_P5_Prev = 0;
float32 IFB_P24_Prev = 0;
float32 IFB_P12_Prev = 0;
float32 IFB_GD_Rec_Prev = 0;
float32 IFB_GD_Inv_A_Prev = 0;
float32 Rec_Vac_Prev = 0;
float32 Inv_Vac_Prev = 0;
float32 Vdc_Prev = 0;
float32 Tap_P1_Iac_Prev = 0;
float32 Tap_P2_Iac_Prev = 0;

float32 Rec_Temp_Prev = 0;
float32 Inv_Temp_A_Prev = 0;
float32 Amb_Temp_Prev = 0;

float32 PLLA_Fltr_Reset_Cond = 0;
float32 Rec_Iac_Fltr_Reset_Cond = 0;
float32 IFB_Fltr_Reset_Cond = 0;
float32 Temp_Fltr_Reset_Cond = 0;

/* Psedo Signal */
//#define PSEUDO_PLL_SIGNAL	1
#define BYPASS_FIRST_ORDER_LPF  1
float32 gAmplitude = 1.0F;
uint32_t gsine_count = 0;
uint32_t gSineDivisions = 200; //Fs = 10000Hz, Fref = 50Hz 200 = Fs/fref


float32 TapIac_Fltr_Reset_Cond = 0;
float32 Vdc_Fltr_Reset_Cond = 0;
float32 Vac_Fltr_Reset_Cond = 0;

bool_t PLLA_Fltr_Reset = false;
bool_t Rec_Iac_Fltr_Reset = false;
bool_t IFB_Fltr_Reset = false;
bool_t Temp_Fltr_Reset = false;
bool_t TapIac_Fltr_Reset = false;
bool_t Vdc_Fltr_Reset = false;
bool_t Vac_Fltr_Reset = false;

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_PLL_A( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_0()  - ADC_VOLTS_MEDIUM_PLL ) * ADC_VOLTS_PER_PLL;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;

#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, PLLA_Fltr_KC , &PLL_CL7_Previous,
										 PLLA_Fltr_Reset, PLLA_Fltr_Reset_Cond );
#endif

#ifdef PSEUDO_PLL_SIGNAL
	adc_res = gAmplitude * sin( (2* 3.142F * gsine_count)/gSineDivisions);
	gsine_count++;
	if( gsine_count > gSineDivisions )
	{
		gsine_count = 1;
	}
#endif

	adc_res = adc_res - ADC_OFFSET_PLL;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Rec_Iac( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_4() - ADC_VOLTS_MEDIUM_HALL ) * ADC_VOLTS_PER_HALL;

#ifdef BYPASS_FIRST_ORDER_LPF

	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Rec_Iac_Fltr_KC , &Rec_Iac_Prev,
										 Rec_Iac_Fltr_Reset, Rec_Iac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_HALL_REC_IAC;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Inv_Iac_A( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_5() - ADC_VOLTS_MEDIUM_HALL ) * ADC_VOLTS_PER_HALL;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Rec_Iac_Fltr_KC , &Inv_Iac_Prev,
										 Rec_Iac_Fltr_Reset, Rec_Iac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_HALL_INV_IAC;

	return ( adc_res );
}


/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Inv_Vac_A( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_3() - ADC_VOLTS_MEDIUM_VOLT ) * ADC_VOLTS_PER_VOLT;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Vac_Fltr_KC , &Inv_Vac_Prev,
										 Vac_Fltr_Reset, Vac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_VOLT_INV_VAC;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Rec_Vac( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_2() - ADC_VOLTS_MEDIUM_VOLT ) * ADC_VOLTS_PER_VOLT;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Vac_Fltr_KC , &Rec_Vac_Prev,
										 Vac_Fltr_Reset, Vac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_VOLT_REC_VAC;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Vdc( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_3() - ADC_VOLTS_MEDIUM_VDC ) * ADC_VOLTS_PER_VDC;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Vdc_Fltr_KC , &Vdc_Prev,
										 Vdc_Fltr_Reset, Vdc_Fltr_Reset_Cond );
#endif
	adc_res = adc_res - ADC_OFFSET_VDC;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Tap_P1_Iac( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_10() - ADC_VOLTS_MEDIUM_CT ) * ADC_VOLTS_PER_CT;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, TapIac_Fltr_KC , &Tap_P1_Iac_Prev,
										 TapIac_Fltr_Reset, TapIac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_CT;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Tap_P2_Iac( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_6() - ADC_VOLTS_MEDIUM_CT ) * ADC_VOLTS_PER_CT;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, TapIac_Fltr_KC , &Tap_P2_Iac_Prev,
										 TapIac_Fltr_Reset, TapIac_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_CT;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Rec_Temp( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_11() - ADC_VOLTS_MEDIUM_TEMP ) * ADC_VOLTS_PER_TEMP;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Temp_Fltr_KC , &Rec_Temp_Prev,
										 Temp_Fltr_Reset, Temp_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_TEMP;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Inv_Temp_A( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_11() - ADC_VOLTS_MEDIUM_TEMP ) * ADC_VOLTS_PER_TEMP;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Temp_Fltr_KC , &Inv_Temp_A_Prev,
										 Temp_Fltr_Reset, Temp_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_TEMP;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_Amb_Temp( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_11()  - ADC_VOLTS_MEDIUM_TEMP ) * ADC_VOLTS_PER_TEMP;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, Temp_Fltr_KC , &Amb_Temp_Prev,
										 Temp_Fltr_Reset, Temp_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_TEMP;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_P15( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_7() - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_P15_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_P15;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_P5( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_7() - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_P5_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_P5;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_P12( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_7() - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_P12_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_P12;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_P24( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_7()  - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_P24_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_P24;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_GD_Rec( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_9() - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_GD_Rec_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_REC;

	return ( adc_res );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_IFB_GD_Inv_A( void )
{
	float32 temp_val;
	float32 adc_res = 0;

	temp_val = ( Read_ADC_Result_8()  - ADC_VOLTS_MEDIUM_IFB ) * ADC_VOLTS_PER_IFB;

#ifdef BYPASS_FIRST_ORDER_LPF
	adc_res = temp_val;
#else
	adc_res = FirstOrderIIRLPFwithReset( temp_val, IFB_Fltr_KC , &IFB_GD_Inv_A_Prev,
										 IFB_Fltr_Reset, IFB_Fltr_Reset_Cond );
#endif

	adc_res = adc_res - ADC_OFFSET_IFB_INV;

	return ( adc_res );
}


/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A0( adc_channel_a0_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case SPARE_FAST_ADC1:
			adc_val = Read_ADC_Result_0();
			break;

		case PLL_A:
			adc_val = Get_PLL_A();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A1( adc_channel_a1_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case SPARE_FAST_ADC2:
			adc_val = Read_ADC_Result_1();
			break;

		case PLL_B:
			adc_val = Read_ADC_Result_1();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A2( adc_channel_a2_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case REC_VAC:
			adc_val = Get_Rec_Vac();
			break;

		case PLL_C:
			adc_val = Read_ADC_Result_2();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A3( adc_channel_a3_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case INV_VAC_A:
			adc_val = Get_Inv_Vac_A();
			break;

		case VDC:
			adc_val = Get_Vdc();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A4( adc_channel_a4_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case REC_IAC:
			adc_val = Get_Rec_Iac();
			break;

		case FAST_SPARE_G1A4:
			adc_val = Read_ADC_Result_4();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A5( adc_channel_a5_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case INV_IAC_A:
			adc_val = Get_Inv_Iac_A();
			break;

		case FAST_SPARE_G1A5:
			adc_val = Read_ADC_Result_5();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A6( adc_channel_a6_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case SPARE_MED_ADC3:
			adc_val = Read_ADC_Result_6();
			break;

		case SPARE_MED_ADC5:
			adc_val = Read_ADC_Result_6();
			break;

		case MEDIUM_SPARE_G2A6:
			adc_val = Read_ADC_Result_6();
			break;

		case MEDIUM_SPARE_G3A6:
			adc_val = Read_ADC_Result_6();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_A7( adc_channel_a7_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case P1_5V:
			adc_val = Get_IFB_P15();
			break;

		case P5V:
			adc_val = Get_IFB_P5();
			break;

		case P12V:
			adc_val = Get_IFB_P12();
			break;

		case P24V:
			adc_val = Get_IFB_P24();
			break;

		case SLOW_SPARE_G4A7:
			adc_val = Read_ADC_Result_7();
			break;

		case SLOW_SPARE:
			adc_val = Read_ADC_Result_7();
			break;

		case PIB_TEMP_G6A7:
			adc_val = Read_ADC_Result_7();
			break;

		case CALIBRATION_REF1:
			adc_val = Read_ADC_Result_7();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B0( adc_channel_b0_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P1_A:
			adc_val = Get_Tap_P1_Iac();
			break;

		case FAST_SPARE_G1B0:
			adc_val = Read_ADC_Result_8();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B1( adc_channel_b1_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P2_A:
			adc_val = Get_Tap_P2_Iac();
			break;

		case INV_VAC_C:
			adc_val = Read_ADC_Result_9();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B2( adc_channel_b2_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P1_B:
			adc_val = Read_ADC_Result_10();
			break;

		case INV_IAC_C:
			adc_val = Read_ADC_Result_10();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B3( adc_channel_b3_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P2_B:
			adc_val = Read_ADC_Result_11();
			break;

		case GD_PWR_REC:
			adc_val = Get_IFB_GD_Rec();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B4( adc_channel_b4_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P2_B:
			adc_val = Read_ADC_Result_12();
			break;

		case GD_PWR_REC:
			adc_val = Get_IFB_GD_Inv_A();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B5( adc_channel_b5_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case I_P2_C:
			adc_val = Read_ADC_Result_13();
			break;

		case GD_PWR_INV_C:
			adc_val = Read_ADC_Result_13();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B6( adc_channel_b6_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case SPARE_MED_ADC4:
			adc_val = Read_ADC_Result_14();
			break;

		case SPARE_MED_ADC6:
			adc_val = Read_ADC_Result_14();
			break;

		case MEDIUM_SPARE_G2B6:
			adc_val = Read_ADC_Result_14();
			break;

		case MEDIUM_SPARE_G3B6:
			adc_val = Read_ADC_Result_14();
			break;

		default:
			break;
	}
	return ( adc_val );
}

/*
 *****************************************************************************************
 * See header file for function definition.
 *****************************************************************************************
 */
float32 Get_ADC_Channel_B7( adc_channel_b7_t signal )
{
	float32 adc_val = 0;

	switch ( signal )
	{
		case LOGIC_POWER_5V:
			adc_val = Read_ADC_Result_15();
			break;

		case REC_TEMP:
			adc_val = Get_Rec_Temp();
			break;

		case INV_TEMP_A:
			adc_val = Get_Inv_Temp_A();
			break;

		case INV_TEMP_C:
			adc_val = Read_ADC_Result_15();
			break;

		case AMB_TEMP:
			adc_val = Get_Amb_Temp();
			break;

		case SLOW_SPARE_G5B6:
			adc_val = Read_ADC_Result_15();
			break;

		case SLOW_SPARE_G6B7:
			adc_val = Read_ADC_Result_15();
			break;

		case CALIBRATION_REF2:
			adc_val = Read_ADC_Result_15();
			break;

		default:
			break;
	}
	return ( adc_val );
}
