/**
 *****************************************************************************************
 * @file F28335Adc.h
 *
 * @brief Wrapper functions for ADC module.
 *
 * @copyright 2020 Eaton Corporation. All Rights Reserved.
 *
 *****************************************************************************************
 */
#ifndef F28335_ADC_H
   #define F28335_ADC_H

#include "DSP28x_Project.h"
#include "Constants.h"
#include "math.h"

#define ADC_VOLTS_PER_ADC_COUNT  ( 3.0f / 4095.0f )

/**
 * @brief Enum to represent channel A0
 */
enum adc_channel_a0_t
{
	SPARE_FAST_ADC1,
	PLL_A
};

/**
 * @brief Enum to represent channel A1
 */
enum adc_channel_a1_t
{
	SPARE_FAST_ADC2,
	PLL_B
};

/**
 * @brief Enum to represent channel A2
 */
enum adc_channel_a2_t
{
	REC_VAC,		// Rec.Vac / Inv.Vac.B
	PLL_C
};

/**
 * @brief Enum to represent channel A3
 */
enum adc_channel_a3_t
{
	INV_VAC_A,
	VDC
};

/**
 * @brief Enum to represent channel A4
 */
enum adc_channel_a4_t
{
	REC_IAC,		// Rec.Iac  / Inv.Iac.B
	FAST_SPARE_G1A4
};

/**
 * @brief Enum to represent channel A5
 */
enum adc_channel_a5_t
{
	INV_IAC_A,
	FAST_SPARE_G1A5
};

/**
 * @brief Enum to represent channel A6
 */
enum adc_channel_a6_t
{
	SPARE_MED_ADC3,
	SPARE_MED_ADC5,
	MEDIUM_SPARE_G2A6,
	MEDIUM_SPARE_G3A6
};

/**
 * @brief Enum to represent channel A7
 */
enum adc_channel_a7_t
{
	P1_5V,
	P5V,
	P12V,
	P24V,
	SLOW_SPARE_G4A7,
	SLOW_SPARE,
	PIB_TEMP_G6A7,
	CALIBRATION_REF1
};

/**
 * @brief Enum to represent channel B0
 */
enum adc_channel_b0_t
{
	I_P1_A,
	FAST_SPARE_G1B0
};

/**
 * @brief Enum to represent channel B1
 */
enum adc_channel_b1_t
{
	I_P2_A,
	INV_VAC_C
};

/**
 * @brief Enum to represent channel B2
 */
enum adc_channel_b2_t
{
	I_P1_B,
	INV_IAC_C
};

/**
 * @brief Enum to represent channel B3
 */
enum adc_channel_b3_t
{
	I_P2_B,
	GD_PWR_REC
};

/**
 * @brief Enum to represent channel B4
 */
enum adc_channel_b4_t
{
	I_P1_C,
	GD_PWR_INV_A
};

/**
 * @brief Enum to represent channel B5
 */
enum adc_channel_b5_t
{
	I_P2_C,
	GD_PWR_INV_C
};

/**
 * @brief Enum to represent channel B6
 */
enum adc_channel_b6_t
{
	SPARE_MED_ADC4,
	SPARE_MED_ADC6,
	MEDIUM_SPARE_G2B6,
	MEDIUM_SPARE_G3B6
};

/**
 * @brief Enum to represent channel B7
 */
enum adc_channel_b7_t
{
	LOGIC_POWER_5V,
	REC_TEMP,			// Rec.Temp / Inv.Temp.B
	INV_TEMP_A,
	INV_TEMP_C,
	AMB_TEMP,
	SLOW_SPARE_G5B6,
	SLOW_SPARE_G6B7,
	CALIBRATION_REF2
};


/**
 * @brief Function for IIR digital filters
 * @param[in] in
 * @param[in] Kc : Filter cutoff parameter
 * @param[in] prevOut
 * @param[in] resetFilt
 * @param[in] resetCond
 * @return IIR filter
 * @details first order IIR low pass filter with reset
 *			Fc cutoff frequency Hz
 *			Fs filter update rate Hz
 *			Ts = 1 / Fs
 *			Kc = 1.0 - exp(-2.0*pi*Fc*Ts) filter cutoff parameter
 *			difference equation:
 *			y(n) = y(n-1) + Kc*[x(n)-y(n-1)]
 */
float32 FirstOrderIIRLPFwithReset( float32 in, float32 Kc, float32* prevOut, bool resetFilt, float32 resetCond );

/*
 * @brief Function to reset the sequence 1
 * @param[in] value : The value used to reset the ADC
 * @return None
 */
inline void Reset_Seq_1( uint16_t value )
{
	AdcRegs.ADCTRL2.bit.RST_SEQ1 = value;
}

/*
 * @brief Function to clear sequence 1
 * @param[in] value : The value used to clear the sequence 1 of ADC
 * @return None
 */
inline void Clear_Seq_1_Int( uint16_t value )
{
	AdcRegs.ADCST.bit.INT_SEQ1_CLR = value;
}

/*
 * @brief Function to read the ADC Result 0
 * @return ADC Conversion 0 output
 */
inline float32 Read_ADC_Result_0()
{
	return ( ( float32 )AdcMirror.ADCRESULT0 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 1
 * @return ADC Conversion 1 output
 */
inline float32 Read_ADC_Result_1()
{
	return ( ( float32 )AdcMirror.ADCRESULT1 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 2
 * @return ADC Conversion 2 output
 */
inline float32 Read_ADC_Result_2()
{
	return ( ( float32 )AdcMirror.ADCRESULT2 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 3
 * @return ADC Conversion 3 output
 */
inline float32 Read_ADC_Result_3()
{
	return ( ( float32 )AdcMirror.ADCRESULT3 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 4
 * @return ADC Conversion 4 output
 */
inline float32 Read_ADC_Result_4()
{
	return ( ( float32 )AdcMirror.ADCRESULT4 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 5
 * @return ADC Conversion 5 output
 */
inline float32 Read_ADC_Result_5()
{
	return ( ( float32 )AdcMirror.ADCRESULT5 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 6
 * @return ADC Conversion 6 output
 */
inline float32 Read_ADC_Result_6()
{
	return ( ( float32 )AdcMirror.ADCRESULT6 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 7
 * @return ADC Conversion 7 output
 */
inline float32 Read_ADC_Result_7()
{
	return ( ( float32 )AdcMirror.ADCRESULT7 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 8
 * @return ADC Conversion 8 output
 */
inline float32 Read_ADC_Result_8()
{
	return ( ( float32 )AdcMirror.ADCRESULT8 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result 9
 * @return ADC Conversion 9 output
 */
inline float32 Read_ADC_Result_9()
{
	return ( ( float32 )AdcMirror.ADCRESULT9 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 10 output
 */
inline float32 Read_ADC_Result_10()
{
	return ( ( float32 )AdcMirror.ADCRESULT10 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 11 output
 */
inline float32 Read_ADC_Result_11()
{
	return ( ( float32 )AdcMirror.ADCRESULT11 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 12 output
 */
inline float32 Read_ADC_Result_12()
{
	return ( ( float32 )AdcMirror.ADCRESULT12 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 13 output
 */
inline float32 Read_ADC_Result_13()
{
	return ( ( float32 )AdcMirror.ADCRESULT13 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 14 output
 */
inline float32 Read_ADC_Result_14()
{
	return ( ( float32 )AdcMirror.ADCRESULT14 * ADC_VOLTS_PER_ADC_COUNT );
}

/*
 * @brief Function to read the ADC Result register
 * @return ADC Conversion 15 output
 */
inline float32 Read_ADC_Result_15()
{
	return ( ( float32 )AdcMirror.ADCRESULT15 * ADC_VOLTS_PER_ADC_COUNT );
}

/**
 *  @brief Function to get calibrated value of PLLA adc result
 *  @return Calibrated PLLA value
 */
float32 Get_PLL_A( void );

/**
 *  @brief Function to get calibrated value of Rec vac adc result
 *  @return Calibrated value
 */
float32 Get_Rec_Vac( void );

/**
 *  @brief Function to get calibrated value of Inv vac adc result
 *  @return Calibrated value
 */
float32 Get_Inv_Vac_A( void );

/**
 *  @brief Function to get calibrated value of Inv Iac adc result
 *  @return Calibrated value
 */
float32 Get_Inv_Iac_A( void );

/**
 *  @brief Function to get calibrated value of Rec Iac adc result
 *  @return Calibrated value
 */
float32 Get_Rec_Iac( void );

/**
 *  @brief Function to get calibrated value of Rec temp adc result
 *  @return Calibrated value
 */
float32 Get_Rec_Temp( void );

/**
 *  @brief Function to get calibrated value of Inv temp adc result
 *  @return Calibrated value
 */
float32 Get_Inv_Temp_A( void );

/**
 *  @brief Function to get calibrated value of Amb temp adc result
 *  @return Calibrated value
 */
float32 Get_Amb_Temp( void );

/**
 *  @brief Function to get calibrated value of Vdc adc result
 *  @return Calibrated value
 */
float32 Get_Vdc( void );

/**
 *  @brief Function to get calibrated value of P 1.5 adc result
 *  @return Calibrated value
 */
float32 Get_IFB_P15( void );

/**
 *  @brief Function to get calibrated value of P5 adc result
 *  @return Calibrated value
 */
float32 Get_IFB_P5( void );

/**
 *  @brief Function to get calibrated value of P12 adc result
 *  @return Calibrated value
 */
float32 Get_IFB_P12( void );

/**
 *  @brief Function to get calibrated value of P24 adc result
 *  @return Calibrated value
 */
float32 Get_IFB_P24( void );

/**
 *  @brief Function to get calibrated value of Rec Pwr adc result
 *  @return Calibrated value
 */
float32 Get_IFB_GD_Rec( void );

/**
 *  @brief Function to get calibrated value of Inv Pwr adc result
 *  @return Calibrated value
 */
float32 Get_IFB_GD_Inv_A( void );

/**
 *  @brief Function to get calibrated value of CT Iac result
 *  @return Calibrated value
 */
float32 Get_Tap_P1_Iac( void );

/**
 *  @brief Function to get calibrated value of CT Iac result
 *  @return Calibrated value
 */
float32 Get_Tap_P2_Iac( void );

/**
 *  @brief Returns ADC channel A0 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A0( adc_channel_a0_t signal );

/**
 *  @brief Returns ADC channel A1 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A1( adc_channel_a1_t signal );

/**
 *  @brief Returns ADC channel A2 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A2( adc_channel_a2_t signal );

/**
 *  @brief Returns ADC channel A3 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A3( adc_channel_a3_t signal );

/**
 *  @brief Returns ADC channel A4 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A4( adc_channel_a4_t signal );

/**
 *  @brief Returns ADC channel A5 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A5( adc_channel_a5_t signal );

/**
 *  @brief Returns ADC channel A6 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A6( adc_channel_a6_t signal );

/**
 *  @brief Returns ADC channel A7 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_A7( adc_channel_a7_t signal );

/**
 *  @brief Returns ADC channel B0 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B0( adc_channel_b0_t signal );

/**
 *  @brief Returns ADC channel B1 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B1( adc_channel_b1_t signal );

/**
 *  @brief Returns ADC channel B2 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B2( adc_channel_b2_t signal );

/**
 *  @brief Returns ADC channel B3 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B3( adc_channel_b3_t signal );

/**
 *  @brief Returns ADC channel B4 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B4( adc_channel_b4_t signal );

/**
 *  @brief Returns ADC channel B5 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B5( adc_channel_b5_t signal );

/**
 *  @brief Returns ADC channel B6 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B6( adc_channel_b6_t signal );

/**
 *  @brief Returns ADC channel B7 value corresponding to the signal name passed
 *  @param[in] signal: signal/channel whose value to be read
 *  @return ADC output
 */
float32 Get_ADC_Channel_B7( adc_channel_b7_t signal );

#endif
