/**
 *****************************************************************************************
 * @file F28335Epwm.h
 *
 * @brief Wrapper functions for PWM module.
 *
 * @copyright 2020 Eaton Corporation. All Rights Reserved.
 *
 *****************************************************************************************
 */
#ifndef F28335EPWM_H
   #define F28335EPWM_H

#include "DSP28x_Project.h"
#include "Constants.h"

/**
 * @brief Enum to represent EPWM modules
 */
enum pwm_base_t
{
	EPWM1,
	EPWM2,
	EPWM3,
	EPWM4,
	EPWM5,
	EPWM6
};

/**
 * @brief Enum to represent TBCTR counting direction
 */
enum pwm_ctr_dir_t
{
	COUNTER_DIR_DOWN,
	COUNTER_DIR_UP
};

/**
 * @brief Enum to represent EPWM Compare modules
 */
enum epwm_counter_compare_module_t
{
	EPWM_COUNTER_COMPARE_A,
	EPWM_COUNTER_COMPARE_B
};

/**
 * @brief Enum to represent EPWM Event trigger select register
 */
enum epwm_event_trigger_select_t
{
	EPWM_EVT_TRIG_INT_SEL,
	EPWM_EVT_TRIG_CONV_A_SEL,
	EPWM_EVT_TRIG_CONV_B_SEL
};

/**
 * @brief Enum to represent EPWM Event trigger enable register
 */
enum epwm_event_trigger_enable_t
{
	EPWM_EVT_TRIG_INT_ENBL,
	EPWM_EVT_TRIG_CONV_A_ENBL,
	EPWM_EVT_TRIG_CONV_B_ENBL
};

/**
 * @brief Enum to represent Dead Band Delay mode
 */
enum epwm_deadband_delay_mode_t
{
	EPWM_DB_RED,
	EPWM_DB_FED
};

enum epwm_event_triggers_sel_t
{
	//! Time base counter equals zero
	SOCA_PULSE_CTR_ZERO       = 1,
	//! Time base counter equals period
	SOCA_PULSE_CTR_PRD     = 2,
	//! Time base counter up equals COMPA
	EPWM_SOC_TBCTR_U_CMPA    = 4,
	//! Time base counter down equals COMPA
	EPWM_SOC_TBCTR_D_CMPA  = 5,
	//! Time base counter up equals COMPB
	EPWM_SOC_TBCTR_U_CMPB    = 6,
	//! Time base counter down equals COMPB
	EPWM_SOC_TBCTR_D_CMPB  = 7
};


/**
 *  @brief Reads TB Counter value of EPWM1 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM1_TB_Counter_Value( void )
{
	return ( EPwm1Regs.TBCTR );
}

/**
 *  @brief Reads TB Counter value of EPWM2 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM2_TB_Counter_Value( void )
{
	return ( EPwm2Regs.TBCTR );
}

/**
 *  @brief Reads TB Counter value of EPWM3 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM3_TB_Counter_Value( void )
{
	return ( EPwm3Regs.TBCTR );
}

/**
 *  @brief Reads TB Counter value of EPWM4 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM4_TB_Counter_Value( void )
{
	return ( EPwm4Regs.TBCTR );
}

/**
 *  @brief Reads TB Counter value of EPWM5 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM5_TB_Counter_Value( void )
{
	return ( EPwm5Regs.TBCTR );
}

/**
 *  @brief Reads TB Counter value of EPWM6 Module
 *  @details returns the current Time base Counter value
 *  @return TBCTR
 */
inline uint16_t Get_PWM6_TB_Counter_Value( void )
{
	return ( EPwm6Regs.TBCTR );
}

/**
 *  @brief Function to read the TB counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM1_TB_Counter_DIR( void )
{
	return ( ( EPwm1Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}

/**
 *  @brief Function to read the counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM2_TB_Counter_DIR( void )
{
	return ( ( EPwm2Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}

/**
 *  @brief Function to read the counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM3_TB_Counter_DIR( void )
{
	return ( ( EPwm3Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}

/**
 *  @brief Function to read the counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM4_TB_Counter_DIR( void )
{
	return ( ( EPwm4Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}

/**
 *  @brief Function to read the counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM5_TB_Counter_DIR( void )
{
	return ( ( EPwm5Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}

/**
 *  @brief Function to read the counter direction
 *  @return 0:Counting Down 1: Counting UP
 */
inline bool_t Get_PWM6_TB_Counter_DIR( void )
{
	return ( ( EPwm6Regs.TBSTS.bit.CTRDIR ) ? COUNTER_DIR_UP: COUNTER_DIR_DOWN );
}



/**
 *  @brief Function to set 16 bit counter value of the time base counter
 *  @param[in] base: PWM module base whose counter compare value is
 *  @n to be set.
 *  @param[in] prd: PWM Period
 *  @return None
 */
void EPWM_SetTimeBaseCounter( pwm_base_t base, uint16_t prd );


/**
 *  @brief Function to set the count value of Counter Compare
 *  @details This function selects PWM module whose counter compare
 *  @n value for counter compare register is to be set.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] base: PWM module base whose counter compare value is
 *  @n to be set.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void EPWM_SetCounterCompareValue( pwm_base_t base, epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM1
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM1_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM2
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM2_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM3
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM3_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM4
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM4_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM5
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM5_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to set the count value of EPWM6
 *  @details This function sets the counter compare value for counter
 *  @n compare registers.
 *  @n The maximum value for comp_count is 0xFFFF.
 *  @param[in] comp_module: It is the Counter Compare value module.
 *  @param[in] comp_count: It is the counter compare count value.
 *  @return None
 */
void SetPWM6_Cmp_Val( epwm_counter_compare_module_t comp_module, uint16_t comp_count );

/**
 *  @brief Function to enable interrupt.
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] base: PWM module base whose Event Trigger is to be
 *  @n enabled.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void EPWM_SetEventTriggerInterruptEnable( pwm_base_t base, epwm_event_trigger_enable_t evnt_trig_enbl,
										  bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM1
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM1_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM2
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM2_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM3
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM3_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM4
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM4_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM5
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM5_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to enable interrupt of EPWM6
 *  @details This function sets the interrupt enable
 *  @n of event trigger selection register.
 *  @param[in] evnt_trig_enbl: It is the enable register to be set
 *  @param[in] enable_value: Enable or Disable value.
 *  @return None
 */
void SetPWM6_Evnt_Trig_Int_Enable( epwm_event_trigger_enable_t evnt_trig_enbl, bool enable_value );

/**
 *  @brief Function to set the Event Trigger status bit.
 *  @details This function selects PWM module whose Event trigger
 *  @n select register bit is to be set.
 *  @param[in] base: PWM module base whose Event Trigger status
 *  @n has to be set.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void EPWM_SetEventTriggerSel( pwm_base_t base, epwm_event_trigger_select_t evnt_trig_sel,
							  epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM1.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM1_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM2.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM2_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM3.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM3_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM4.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM4_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM5.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM5_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to set the Event Trigger status bit for PWM6.
 *  @details This function sets the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be set.
 *  @param[in] selctn_value: It is the event trigger select register bit value.
 *  @return None
 */
void SetPWM6_Evnt_Trig_Select( epwm_event_trigger_select_t evnt_trig_sel,
							   epwm_event_triggers_sel_t selctn_value );

/**
 *  @brief Function to read the Event Trigger status bit.
 *  @details This function selects PWM module whose event trigger
 *  @n select register bit is to be read.
 *  @param[in] base: PWM module base whose Event Trigger status
 *  @n has to be read.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t EPWM_GetEventTriggerInterruptStatus( pwm_base_t base, epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM1.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM1_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM2.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM2_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM3.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM3_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM4.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM4_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM5.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM5_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to read the Event Trigger status bit for PWM6.
 *  @details This function reads the Event trigger select register bit.
 *  @param[in] evnt_trig_sel: It is the event trigger select register
 *  @n module to be read.
 *  @return Event Trigger status bit.
 */
uint16_t GetPWM6_Evnt_Trig_Status( epwm_event_trigger_select_t evnt_trig_sel );

/**
 *  @brief Function to set the DB delay count.
 *  @details This function selects PWM module whose DB delay
 *  @n count for either rising edge or falling edge is to be set.
 *  @param[in] base: PWM module base whose DB delay count has to
 *  @n be set.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @param[in] deadband_count: It is the delay count to the dead band.
 *  @return None
 */
void EPWM_SetDeadBandDelayCount( pwm_base_t base, epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM1.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @param[in] deadband_count: It is the delay count to the dead band.
 *  @return None
 */
void SetPWM1_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM2.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @param[in] deadband_count: It is the delay count to the dead band.
 *  @return None
 */
void SetPWM2_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM3.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @return None
 */
void SetPWM3_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM4.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @return None
 */
void SetPWM4_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM5.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @param[in] deadband_count: It is the delay count to the dead band.
 */
void SetPWM5_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

/**
 *  @brief Function to set the DB delay count for PWM6.
 *  @details This function sets the DB delay count for particular
 *  @n PWM module for either rising edge or falling edge.
 *  @param[in] delay_mode: It determines if the applied delay is
 *  @n Rising Edge or Falling Edge.
 *  @param[in] deadband_count: It is the delay count to the dead band.
 */
void SetPWM6_DB_Delay_Count( epwm_deadband_delay_mode_t delay_mode, uint16_t deadband_count );

#endif
