// ********************************************************************************************************
// *            SystemTrap.c
// ********************************************************************************************************
// ********************************************************************************************************
// *
// *    THIS INFORMATION IS PROPRIETARY TO EATON
// *
// ********************************************************************************************************
// *
// *    Copyright (c) 2003...2007 EATON
// *    ALL RIGHTS RESERVED
// *
// ********************************************************************************************************
// ********************************************************************************************************
// *    FILE NAME: main.c
// *
// *    DESCRIPTION: Contains ISR for system traps
// *
// *    ORIGINATOR: Mike Westerfield
// *
// *    DATE: 7/17/2003
// *
// *    HISTORY: See revision control system's history.
// *********************************************************************************************************

// *********************************************************************************************************
// *        INCLUDE FILES
// *********************************************************************************************************
#include "DSP28x_Project.h"
#include "SystemTrap.h"


uint16_t  xi2Counter = 0;

extern "C"
{
    void InitWatchdog( void );
    void Illegal_isr( void );
    void Watchdog_Overflow_isr( void );
    void ExternalInt2_isr( void );
}

// *********************************************************************************************************
// *        Global Variables
// *********************************************************************************************************


//INT16U WatchdogFlags;

// ********************************************************************************************************
// *
// * Function: WatchdogReset(void)
// *
// * Purpose:  Create a watchdog reset
// *
// * Description:  Watchdog reset resets all DSP system registers. 
// *
// ********************************************************************************************************
void WatchdogReset( void )
{
    uint16_t wd_reset_delay = 0;
   
    DINT; // disable interrupts
    
    // WD reset
    EALLOW;
    // Enable watchdog module
    SysCtrlRegs.WDCR = 0x0028;
    // Enable Watchdog reset
    SysCtrlRegs.SCSR = 0x0000;
    // Create WD reset
//  SysCtrlRegs.WDKEY = 0x0000;
    EDIS;
        
	while (++wd_reset_delay < 1000)
    {
        asm(" nop");
    }
        
    // Should never reach this. Watch dog should kick in.
    // Soft reset
    //asm(" LB #3FFC00h");
    asm(" LB #33FFF6h");
}

// ********************************************************************************************************
// *
// * Function: FastShutdown(void);
// *
// * Purpose:  Shutdown everything ASAP
// *
// * Description:
// *
// ********************************************************************************************************
void FastShutdown( void )
{
    //Don't call SetIOGood() function here,because all the interrupts have been disabled.

    GpioDataRegs.GPASET.bit.GPIO24 = 1;
    GpioDataRegs.GPCCLEAR.bit.GPIO65 = 0;

    //Disable Rec and Inv gate drivers
    GpioDataRegs.GPBCLEAR.bit.GPIO50 = 1;
    GpioDataRegs.GPBCLEAR.bit.GPIO52 = 1;
}


// ********************************************************************************************************
// *
// * Function: InitWatchdog(void);
// *
// * Purpose: Initialize Watchdog
// *
// *
// * Description:  Set Watchdog up to generate interrupt instead of reset.  Counter overflow at approx. 4.4 ms
// *
// ********************************************************************************************************
void InitWatchdog( void )
{
    // Enable watchdog module
    EALLOW;
    SysCtrlRegs.SCSR = 0x0002; //bit 1 == Enable Watchdog interrupt and module
                               //if bit 0 is set watchdog cannot be turned off until hardware reset
    SysCtrlRegs.WDCR = 0x0028; //bit 2-0 : 000 = prescale 1
                               //bit 5-3 : 101 or else immediate reset
                               //bit 6   : watchdog disable
    // Kick Watchdog
    SysCtrlRegs.WDKEY = 0x55;
    SysCtrlRegs.WDKEY = 0xAA;
    EDIS;
}

void DisableWatchdog(void)
{
    EALLOW;
    SysCtrlRegs.WDKEY = 0x55;
    SysCtrlRegs.WDKEY = 0xAA;
    SysCtrlRegs.WDCR  = 0x68;
    EDIS;
}

// ********************************************************************************************************
// *
// * Function: KickWatchdog(void);
// *
// * Purpose: Kick Watchdog
// *
// *
// * Description:  Reset watchdog timer
// *
// ********************************************************************************************************
#pragma CODE_SECTION("ramfuncs");
void KickWatchdog( void )
{
    EALLOW;
    SysCtrlRegs.WDKEY = 0x55;
    SysCtrlRegs.WDKEY = 0xAA;
    EDIS;
}


// ********************************************************************************************************
// *
// * Function: Watchdog_Overflow_isr(void);
// *
// * Purpose: If a watchdog overflow occurs shutdown the rectifier and inverter and reset the dsp.
// *         
// *
// * Description:  A watchdog timeout will occur whenever certain interrupts stop occuring or when the idle
// *               task hasnt happened in 9 ms.
// *
// ********************************************************************************************************
#pragma CODE_SECTION( "ramfuncs" )
void Watchdog_Overflow_isr( void )
{
    // step1 : disable interrupt
    DINT;
    
    // step2 : fast shut down unit
    FastShutdown();

    // step4 : trigger watchdog to reset dsp 
    WatchdogReset();
}

// ********************************************************************************************************
// *
// * Function: Illegal_Address_isr(void);
// *
// * Purpose: Shutdown the inverter and recrifier, then reset the dsp if the processor to a known state
// *          if it has become "lost"
// *
// * Description:
// *
// ********************************************************************************************************
#pragma CODE_SECTION( "ramfuncs" )
void Illegal_isr( void )
{
    // step 1 : disable interrupt
    DINT;
    
    // step 2 : fast shut down unit   
    FastShutdown();
    
    // step 4 : trigger watchdog to reset dsp 
    WatchdogReset();
}

// ***********************************************************************
/*!
    ISR from External Interrupt 2... TODO: Test this.
 */
// ***********************************************************************
void InitEPLDWatchdogEcho()
{
    //PieCtrlRegs.PIEIER1.bit.INTx5       = 1;

    EALLOW;
    GpioCtrlRegs.GPAMUX2.bit.GPIO22     = 0;    // GPIO
    GpioCtrlRegs.GPADIR.bit.GPIO22      = 0;    // Input
    GpioCtrlRegs.GPAQSEL2.bit.GPIO22    = 2;    // XINT2 Qual using 6 samples
    GpioCtrlRegs.GPACTRL.bit.QUALPRD0   = 0xFF; // Each sampling window is 510 * SYSCLKOUT
    EDIS;

    EALLOW;
    GpioIntRegs.GPIOXINT2SEL.bit.GPIOSEL = 22;  // XINT2 is being configured for GPIO22
    EDIS;

    XIntruptRegs.XINT2CR.bit.POLARITY   = 1;    // Rising Edge Interrupt
    XIntruptRegs.XINT2CR.bit.ENABLE     = 1;    // Enable XINT2

    PieCtrlRegs.PIEIER1.bit.INTx5 = 1;

    //IER |= M_INT1; -- Done elsewhere
}

// ***********************************************************************
/*!
    ISR from External Interrupt 2...
    This should be tied to a 3.6 kHz signal
 */
// ***********************************************************************
#pragma CODE_SECTION( "ramfuncs" )
void ExternalInt2_isr( void )
{
    xi2Counter++;

    // Acknowledge interrupt and return
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;     // Acknowledge interrupt to PIE
}

// *********************************************************************************************************
// *        End of File
// *********************************************************************************************************
