/*
 *  CAN RX FIFOs demo.
 * F28377S, 10MHz crystal, CAN routed to GPIO70/71
 *
 * CAN speed - 1Mbps, only extended ID's.
 * Update CAN_BTR for different speed.
 * Update Xtd/Mxtd in init loops if you need standard ID's
 *
 * To make it easier to collect few messages in FIFO, 1s delay is inserted into ISR to prevent in time
 * message servicing. You may put breakpoint below DELAY_US call and check if number of while() iterations
 * matches with the number of messages sent
 *
 * */


#include "F28x_Project.h"

#include "F2837xS_Can_defines.h"

#define CANM CanaRegs // using CANA


#define FIFOR0 1 // FIFO0 starts fromn MB1
#define FIFOR1 9 // FIFO0 starts fromn MB9
#define FIFOTX 17

// acceptance id and mask for fifo 0
#define Acc0id  0x6400000
#define Acc0msk 0x7E00000

// acceptance id and mask for fifo 1
#define Acc1id  0x0000000
#define Acc1msk 0x7E00000




interrupt void CAN_RX_ISR(void)
{
    // CPU "is busy" for 1s to prevent servicing data in time and
    // let fifo collect few messages.
    DELAY_US(1000000);

    // pull everything from all CAN FIFO queues, IF3 is used to read
    while(CANM.CAN_IF3OBS.bit.IF3Upd)
    {
        volatile unsigned long iddata, dlcdata, dataadata, databdata;

        // IF3 adcances to next mailbox with data after reading all 4 : IF3ARB, IF3MCTL, IF3DATA, IF3DATB
        iddata = CANM.CAN_IF3ARB.bit.ID;

        dlcdata = CANM.CAN_IF3MCTL.bit.DLC;

        dataadata = CANM.CAN_IF3DATA.all;
        databdata = CANM.CAN_IF3DATB.all;
    }


    CANM.CAN_GLB_INT_CLR.bit.INT0_FLG_CLR = 1;
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP9;
}


void CanInit(void)
{
    unsigned int iMsg;

   CANM.CAN_CTL.all = 1; // set Init bit, clear others
   CANM.CAN_CTL.bit.SWR = 1; //
   CANM.CAN_CTL.bit.ABO = 1; //

   CANM.CAN_RAM_INIT.all = 0x1A;
   while(! CANM.CAN_RAM_INIT.bit.RAM_INIT_DONE)
   {}
   CANM.CAN_RAM_INIT.all = 5;


   for(iMsg = FIFOR0; iMsg < FIFOR1; iMsg++)
   {
       CANM.CAN_IF1ARB.bit.MsgVal = 1; // message is used
       CANM.CAN_IF1ARB.bit.Xtd = 1; // extended id
       CANM.CAN_IF1ARB.bit.Dir = 0; // receive mb
       CANM.CAN_IF1ARB.bit.ID = Acc0id;

       CANM.CAN_IF1MSK.bit.MXtd = 1; // mask extended bit
       CANM.CAN_IF1MSK.bit.MDir = 0; // don't accept send messages
       CANM.CAN_IF1MSK.bit.Msk  = Acc0msk;

       CANM.CAN_IF1MCTL.all = (1<<12/*UMask*/)|(1<<10/*RxIE*/);
       if(iMsg == (FIFOR1-1))
           CANM.CAN_IF1MCTL.bit.EoB = 1; // end of fifo


       CANM.CAN_IF1CMD.all =
       (
                0x80 * 1 /*DIR*/
              | 0x40 * 1 /*Mask*/
              | 0x20 * 1 /*Arb*/
              | 0x10 * 1 /*Control*/
              | 0x08 * 1 /*ClrIntPnd*/
              | 0x04 * 0 /*TXRQST*/
              | 0x02 * 0 /*DATA_A*/
              | 0x01 * 0 /*DATA_B*/
              ) * 0x10000ul | iMsg;

       while(CANM.CAN_IF1CMD.bit.Busy)
       {
       }
       CANM.CAN_IF1CMD.all = 0;
   }

   for(iMsg = FIFOR1; iMsg < FIFOTX; iMsg++)
   {
       CANM.CAN_IF1ARB.bit.MsgVal = 1; // message is used
       CANM.CAN_IF1ARB.bit.Xtd = 1; // extended id
       CANM.CAN_IF1ARB.bit.Dir = 0; // receive mb
       CANM.CAN_IF1ARB.bit.ID = Acc1id;

       CANM.CAN_IF1MSK.bit.MXtd = 1; // mask extended bit
       CANM.CAN_IF1MSK.bit.MDir = 0; // don't accept send messages
       CANM.CAN_IF1MSK.bit.Msk  = Acc1msk;

       CANM.CAN_IF1MCTL.all = (1<<12/*UMask*/)|(1<<10/*RxIE*/);
       if(iMsg == (FIFOTX-1))
           CANM.CAN_IF1MCTL.bit.EoB = 1; // end of fifo


       CANM.CAN_IF1CMD.all =
               (
                0x80 * 1 /*DIR*/
              | 0x40 * 1 /*Mask*/
              | 0x20 * 1 /*Arb*/
              | 0x10 * 1 /*Control*/
              | 0x08 * 1 /*ClrIntPnd*/
              | 0x04 * 0 /*TXRQST*/
              | 0x02 * 0 /*DATA_A*/
              | 0x01 * 0 /*DATA_B*/
              ) * 0x10000ul | iMsg;;
       // Wait for busy bit to clear
       while(CANM.CAN_IF1CMD.bit.Busy)
       {
       }
       CANM.CAN_IF1CMD.all = 0;
   }

#if 0
   for(iMsg = FIFOTX; iMsg <= 32; iMsg++)
   {
       CANM.CAN_IF1MSK.all = 0xFFFFFFFF;
       CANM.CAN_IF1ARB.all = 0xFFFFFFFF;

       CANM.CAN_IF1ARB.bit.MsgVal = 1; // message is used
       CANM.CAN_IF1ARB.bit.Xtd = 1; // extended id
       //CANM.CAN_IF1ARB.bit.Dir = 1; // receive mb

       CANM.CAN_IF1MCTL.all = (1<<12/*UMask*/)|(1<<7/*EoB*/);

       CANM.CAN_IF1CMD.all =
               (
                0x80 * 1 /*DIR*/
               | 0x40 * 1 /*Mask*/
               | 0x20 * 1 /*Arb*/
              | 0x10 * 1 /*Control*/
              | 0x08 * 1 /*ClrIntPnd*/
              | 0x04 * 0 /*TXRQST*/
              | 0x02 * 0 /*DATA_A*/
              | 0x01 * 0 /*DATA_B*/
              ) * 0x10000ul | iMsg;

       // Wait for busy bit to clear
       while(CANM.CAN_IF1CMD.bit.Busy)
       {
       }

       CANM.CAN_IF1CMD.all = 0;
   }
#endif



   CANM.CAN_IF3OBS.all = 0x1E;  // DataB, DataA, Ctrl and Arb have to be read to enable next IF3 update

   // update IF3 on FIFO0 (mailboxes FIFOR0..FIFOR1)
   CANM.CAN_IF3UPD |= ((1ul<<(FIFOR1-1))-1) & ~((1ul<<(FIFOR0-1))-1);

   // update IF3 on FIFO1 (mailboxes FIFOR1..FIFOR2)
   CANM.CAN_IF3UPD |= ((1ul<<(FIFOTX-1))-1) & ~((1ul<<(FIFOR1-1))-1);


   // empty IF3 by reading it
   while(CANM.CAN_IF3OBS.bit.IF3Upd)
   {
       unsigned long tmp;

       tmp = CANM.CAN_IF3ARB.all;
       tmp = CANM.CAN_IF3MCTL.all;
       tmp = CANM.CAN_IF3DATA.all;
       tmp = CANM.CAN_IF3DATB.all;
       (void)tmp;
   }

   // Acknowledge any pending status interrupts.
   (void)CANM.CAN_ES.all;

   // select sysclkout
   ClkCfgRegs.CLKSRCCTL2.bit.CANABCLKSEL = 0;

   CANM.CAN_CTL.bit.CCE = 1;

   CANM.CAN_BTR.all = (0ul/*BRPE*/<<16)|((4/*TSEG2*/-1)<<12)|((5/*TSEG1*/-1)<<8)|((2/*SJW*/-1)<<6)|(20/*BRP*/-1);

   CANM.CAN_CTL.bit.CCE = 0;
   CANM.CAN_CTL.bit.Init = 0; // Enable CAN

   CANM.CAN_CTL.bit.IE0 = 1; // interrupt on line 1
   //CANM.CAN_CTL.bit.IE1 = 1; // interrupt on line 1

   CANM.CAN_GLB_INT_EN.bit.GLBINT0_EN = 1;

   EALLOW;  // This is needed to write to EALLOW protected registers
   PieVectTable.CANA0_INT = &CAN_RX_ISR;
   EDIS;    // This is needed to disable write to EALLOW protected registers

   PieCtrlRegs.PIECTRL.bit.ENPIE = 1;   // Enable the PIE block
   PieCtrlRegs.PIEIER9.bit.INTx5 = 1;   // PIE Group 9, INT5
   IER |= 0x100;                         // Enable CPU INT
}


static void InitPIE(void)
{
    //  Clear all interrupts and initialize PIE vector table:
    // Disable CPU interrupts
    DINT;

    // Initialize PIE control registers to their default state.
    // The default state is all PIE interrupts disabled and flags
    // are cleared.
    // This function is found in the F2806x_PieCtrl.c file.
    InitPieCtrl();

    // Disable CPU interrupts and clear all CPU interrupt flags:
    IER = 0x0000;
    IFR = 0x0000;

    // Initialize the PIE vector table with pointers to the shell Interrupt
    // Service Routines (ISR).
    // This will populate the entire table, even if the interrupt
    // is not used in this example.  This is useful for debug purposes.
    // The shell ISR routines are found in F2806x_DefaultIsr.c.
    // This function is found in F2806x_PieVect.c.
    InitPieVectTable();

    // Enable global Interrupts and higher priority real-time debug events:
    EINT;   // Enable Global interrupt INTM
    ERTM;   // Enable Global realtime interrupt DBGM
}


/**
 * main.c
 */
int main(void)
{
    // Initialize System Control:
    // PLL, WatchDog, enable Peripheral Clocks
    // This example function is found in the F2806x_SysCtrl.c file.
    InitSysCtrl();

    InitPIE();

    EALLOW;
    // GPIO 70 and 71 for CanA pins
    GpioCtrlRegs.GPCGMUX1.bit.GPIO70 = 1;
    GpioCtrlRegs.GPCMUX1.bit.GPIO70 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO70 = 0;

    GpioCtrlRegs.GPCQSEL1.bit.GPIO70 = 3; // async

    GpioCtrlRegs.GPCPUD.bit.GPIO70 = 1;


    GpioCtrlRegs.GPCGMUX1.bit.GPIO71 = 1;
    GpioCtrlRegs.GPCMUX1.bit.GPIO71 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO71 = 1;

    GpioCtrlRegs.GPCQSEL1.bit.GPIO71 = 3; // async

    GpioCtrlRegs.GPCPUD.bit.GPIO71 = 0;

    GpioCtrlRegs.GPCODR.bit.GPIO71 = 0;
    EDIS;

    CanInit();

    //DINT; // disable interrupts, enable later from debugger after some messages are send

    for(;;)
    {
    }
}
