
#!/usr/bin/env python3
import argparse, os, socket, threading, time, sys, platform

def set_sndbuf(sock, size):
    try:
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, size)
    except Exception:
        pass

def bind_iface(sock, iface):
    if platform.system() == "Linux":
        SO_BINDTODEVICE = 25
        try:
            sock.setsockopt(socket.SOL_SOCKET, SO_BINDTODEVICE, iface.encode())
            return True
        except PermissionError:
            print(f"[warn] --iface {iface} requires sudo/root for SO_BINDTODEVICE; falling back to routing", file=sys.stderr)
        except Exception as e:
            print(f"[warn] SO_BINDTODEVICE failed on {iface}: {e}; falling back to routing", file=sys.stderr)
    else:
        print(f"[warn] --iface works only on Linux; use --bind-ip instead", file=sys.stderr)
    return False

def worker(dst, port, payload_size, pps, duration, iface, bind_ip, src_port, tos, sndbuf):
    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    if sndbuf:
        set_sndbuf(sock, sndbuf)
    if tos is not None:
        try:
            sock.setsockopt(socket.IPPROTO_IP, socket.IP_TOS, tos)
        except Exception:
            pass

    if bind_ip:
        try:
            sock.bind((bind_ip, src_port if src_port else 0))
        except Exception as e:
            print(f"[warn] bind({bind_ip}) failed: {e}; relying on routing", file=sys.stderr)

    if iface:
        bind_iface(sock, iface)

    payload = os.urandom(payload_size)
    deadline = time.perf_counter() + duration if duration > 0 else float('inf')

    interval = (1.0 / pps) if pps > 0 else 0.0
    sent = 0
    next_send = time.perf_counter()
    try:
        while time.perf_counter() < deadline:
            try:
                sock.sendto(payload, (dst, port))
                sent += 1
            except Exception:
                pass
            if interval:
                next_send += interval
                while True:
                    now = time.perf_counter()
                    dt = next_send - now
                    if dt <= 0:
                        break
                    time.sleep(dt if dt < 0.002 else 0.002)
    finally:
        sock.close()
    return sent

def main():
    ap = argparse.ArgumentParser(description="UDP load generator")
    ap.add_argument("--dst", required=True, help="Destination IPv4 (e.g., 10.3.23.241)")
    ap.add_argument("--port", type=int, default=5000, help="UDP destination port (default: 5000)")
    ap.add_argument("--payload-size", type=int, default=1472, help="UDP payload bytes (default: 1472 ~ 1500B frame)")
    ap.add_argument("--pps", type=float, default=0.0, help="Packets/s per thread (0 = unlimited)")
    ap.add_argument("--duration", type=float, default=60.0, help="Duration in seconds (0 = forever)")
    ap.add_argument("--threads", type=int, default=4, help="Number of threads")
    ap.add_argument("--iface", help="Bind to interface (Linux, root required). Example: eth0")
    ap.add_argument("--bind-ip", help="Bind to specific source IPv4. Example: 10.3.23.100")
    ap.add_argument("--src-port", type=int, help="Source UDP port (when binding IP)")
    ap.add_argument("--tos", type=int, help="IP TOS/DSCP byte (0-255)")
    ap.add_argument("--sndbuf", type=int, help="SO_SNDBUF bytes (try 4194304)")
    args = ap.parse_args()

    print(f"UDP → {args.dst}:{args.port}  size={args.payload_size}B  pps/thread={args.pps}  threads={args.threads}  duration={args.duration}s")
    if args.iface:   print(f"  iface={args.iface} (Linux/root only)")
    if args.bind_ip: print(f"  bind_ip={args.bind_ip} src_port={args.src_port or 0}")
    if args.tos is not None: print(f"  TOS={args.tos}")
    if args.sndbuf: print(f"  sndbuf={args.sndbuf}")

    start = time.perf_counter()

    counts = [0] * args.threads
    def run(i):
        counts[i] = worker(args.dst, args.port, args.payload_size, args.pps, args.duration,
                           args.iface, args.bind_ip, args.src_port, args.tos, args.sndbuf)

    threads = [threading.Thread(target=run, args=(i,), daemon=True) for i in range(args.threads)]
    for t in threads: t.start()
    for t in threads: t.join()

    elapsed = time.perf_counter() - start
    total = sum(counts)
    rate = total / elapsed if elapsed > 0 else 0
    print(f"Done. Sent {total} packets in {elapsed:.2f}s  (~{rate:.0f} pps).")

if __name__ == "__main__":
    main()
