import tkinter as tk
from tkinter import filedialog, ttk, messagebox
import pandas as pd
import matplotlib.pyplot as plt

class DualAxisPlotterApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Dual Axis Log Plotter")

        self.frame = ttk.Frame(root, padding="10")
        self.frame.grid(row=0, column=0, sticky="nsew")

        self.upload_btn = ttk.Button(self.frame, text="Upload Log File", command=self.load_file)
        self.upload_btn.grid(row=0, column=0, pady=5)

        self.group1_label = ttk.Label(self.frame, text="Select columns for Group 1 (Left Y-axis):")
        self.group1_label.grid(row=1, column=0, pady=5)

        self.group1_listbox = tk.Listbox(self.frame, selectmode="multiple", exportselection=False, width=50, height=10)
        self.group1_listbox.grid(row=2, column=0, pady=5)

        self.group2_label = ttk.Label(self.frame, text="Select columns for Group 2 (Right Y-axis):")
        self.group2_label.grid(row=3, column=0, pady=5)

        self.group2_listbox = tk.Listbox(self.frame, selectmode="multiple", exportselection=False, width=50, height=10)
        self.group2_listbox.grid(row=4, column=0, pady=5)

        self.plot_btn = ttk.Button(self.frame, text="Plot Selected Groups", command=self.plot_columns)
        self.plot_btn.grid(row=5, column=0, pady=10)

        self.df = None

    def load_file(self):
        file_path = filedialog.askopenfilename(filetypes=[("Log files", "*.log"), ("CSV files", "*.csv")])
        if file_path:
            try:
                with open(file_path, 'r') as f:
                    lines = f.readlines()
                for i, line in enumerate(lines):
                    if line.startswith("Sample,"):
                        header_line = i
                        break
                data = ''.join(lines[header_line:])
                from io import StringIO
                self.df = pd.read_csv(StringIO(data))
                self.df['DateTime'] = pd.to_datetime(self.df['DateTime'], errors='coerce')
                self.df = self.df.drop(columns=['Sample'], errors='ignore')
                start_time = self.df['DateTime'].iloc[0]
                self.df['Minutes'] = (self.df['DateTime'] - start_time).dt.total_seconds() / 60.0
                self.group1_listbox.delete(0, tk.END)
                self.group2_listbox.delete(0, tk.END)
                for col in self.df.columns:
                    if col not in ['DateTime', 'Minutes']:
                        self.group1_listbox.insert(tk.END, col)
                        self.group2_listbox.insert(tk.END, col)
            except Exception as e:
                messagebox.showerror("Error", f"Failed to load file: {e}")

    def plot_columns(self):
        if self.df is not None:
            group1_indices = self.group1_listbox.curselection()
            group2_indices = self.group2_listbox.curselection()
            group1_columns = [self.group1_listbox.get(i) for i in group1_indices]
            group2_columns = [self.group2_listbox.get(i) for i in group2_indices]
            if not group1_columns and not group2_columns:
                messagebox.showwarning("No Selection", "Please select at least one column to plot.")
                return
            try:
                fig, ax1 = plt.subplots(figsize=(12, 6))
                ax2 = ax1.twinx()

                for col in group1_columns:
                    ax1.plot(self.df['Minutes'], self.df[col], label=f"Group 1: {col}", linestyle='-', marker='')

                for col in group2_columns:
                    ax2.plot(self.df['Minutes'], self.df[col], label=f"Group 2: {col}", linestyle='--', marker='')

                ax1.set_xlabel("Minutes")
                ax1.set_ylabel("Group 1 Values")
                ax2.set_ylabel("Group 2 Values")
                ax1.set_title("Selected Columns Over Time (Minutes from Start)")

                lines_1, labels_1 = ax1.get_legend_handles_labels()
                lines_2, labels_2 = ax2.get_legend_handles_labels()
                ax1.legend(lines_1 + lines_2, labels_1 + labels_2)

                ax1.grid(True)
                fig.tight_layout()
                plt.show()
            except Exception as e:
                messagebox.showerror("Plot Error", f"Failed to plot columns: {e}")

if __name__ == "__main__":
    root = tk.Tk()
    app = DualAxisPlotterApp(root)
    root.mainloop()
