#include "system_defines.h"
#include "Cyclone_Device.h"
#include "pmbus_commands.h"
#include "variables.h"
#include "function_definitions.h"
#include "software_interrupts.h"
#include "pmbus_common.h"
#include "pmbus_topology.h"
#include "pmbus_constants.h"
#include "dcdc_parameters.h"
#include "standard_interrupt.h"
#include "fault.h"
#include <stdlib.h>
#include "led.h"
#include "Calibration.h"

#include "adc_gains.h"


#define ABS(x)		(((x)>0)?(x):(-1*(x)))


const Uint8 status_bit_mask[]    = STATUS_BIT_MASK;


/***********************************************************************
 *Start of MFR message
 ***********************************************************************/
const Uint8 setup_id[]          = SETUP_ID;
const Uint8 mfr_model[]			= MFR_MODEL;
const Uint8 mfr_revision[]		= MFR_REVISION;
const Uint8 mfr_date[]			= MFR_DATE;
const Uint8 mfr_location[]		= MFR_LOCATION;
const Uint8 mfr_serial[]		= MFR_SERIAL;
const Uint8 mfr_ic_device[]			= DEVICE_ID;
const Uint8 mfr_ic_device_rev[]	= MFR_IC_DEVICE_REV;
const Uint8 mfr_id[]			= MFR_ID;

#pragma DATA_SECTION (device_id_string, "FixedDeviceID");
const Uint8  device_id_string[] = DEVICE_ID;


int16_t PMBus_Vout_Command; 		// Vout command received from DSP via PMBus

volatile Uint16 PMBus_HVDC_UV_Command = 0U; 	// HVDC Under Voltage Set Point - Generates a Fault
volatile Uint16 PMBus_HVDC_OV_Command = 0U; 	// HVDC Over Voltage Set Point - Generates a Fault
volatile  int32 pmbus_Vout_offset     = 0; 	// Vout ADC Calibration
volatile  int16 pmbus_Ibat_offset     = 0; 	// Ibat ADC Calibration
volatile  int16 pmbus_Iout_offset     = 0; 	// Iout ADC Calibration
volatile  int16 dac_ibatt_offset      = 0;
volatile  int16 dac_vout_offset       = 0;
volatile  int16 temp_offset           = 0;

//PMBUS_DCDC_READING            pmbus_dcdc_reading[DCDC_PAGE_NUMS];
PMBUS_DCDC_CONFIG             pmbus_dcdc_config[DCDC_PAGE_NUMS];
PMBUS_DCDC_CONFIG             pmbus_dcdc_config_translated[DCDC_PAGE_NUMS];
PMBUS_DCDC_CAL                pmbus_dcdc_cal[DCDC_PAGE_NUMS];

PMBUS_DCDC_CONFIG_NONPAGED    pmbus_dcdc_config_nonpaged;
PMBUS_DCDC_CONFIG_NONPAGED    pmbus_dcdc_config_nonpaged_translated;
PMBUS_DCDC_CAL_NONPAGED       pmbus_dcdc_cal_nonpaged;
Uint16                        dcdc_temperature_1, dcdc_temperature_2, dcdc_temperature_3, dcdc_temperature_4;

FILTER_PMBUS_REGS filter0_pmbus_regs;
FILTER_PMBUS_REGS filter1_pmbus_regs;
FILTER_PMBUS_REGS filter2_pmbus_regs;


//==========================================================================================
//pmbus_write_vout_cmd - Read or write a Vout command over PMBUS
//==========================================================================================
inline Uint8 pmbus_write_vout_cmd(void)
{
    // Put Vout command write function here
    Uint16 tempbuff = 0;
    Uint16 tempVout = 0;

	tempbuff = pmbus_buffer[1]+(pmbus_buffer[2]<<8);
	tempVout = tempbuff * 10;

	if(tempVout > VOUT_SETPOINT_MAX)
	{
		tempVout = VOUT_SETPOINT_MAX;
	}
	else if (tempVout < VOUT_SETPOINT_MIN)
	{
		tempVout = VOUT_SETPOINT_MIN;
	}
	PMBus_Vout_Command = tempVout;

	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vout_cmd(void)
{
    // Put Vout command read function here
    Uint16 tempbuff = 0;
    tempbuff = PMBus_Vout_Command/10;
	pmbus_read_two_byte_handler(tempbuff);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vout_command(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vout_cmd();
	else
		return pmbus_write_vout_cmd();
}

/**********************
 * End of vout command
 *********************/


//==========================================================================================
//pmbus_write_vout_cmd - Read or write a iOut_Setpoint command over PMBUS
//==========================================================================================
inline Uint8 pmbus_write_ioutSetPoint(void)
{
    // Put Vout command write function here
    Uint16 tempbuff = 0;
    Uint16 tempIout = 0;

	tempbuff = pmbus_buffer[1]+(pmbus_buffer[2]<<8);
	tempIout = (Uint16)(DAC_IBAT_MILLIAMPS_TO_COUNTS(((int16)(tempbuff))));
	ISet_mA=tempbuff;

	if(tempIout < DAC_IBAT_MILLIAMPS_TO_COUNTS(0))
	{
		tempIout = DAC_IBAT_MILLIAMPS_TO_COUNTS(0);
	}
	else if (tempIout > IBAT_SETPOINT_MAX)
	{
		tempIout = IBAT_SETPOINT_MAX;
	}

	IBatt_SetPoint = (tempIout);
	LED_Set_Error(0U);

	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_ioutSetPoint(void)
{     
	int32 temp32 = 0;
	temp32 = (int32)(DAC_IBAT_COUNTS_TO_MILLIAMP(IBatt_SetPoint));
	pmbus_read_two_byte_handler(0xFFFF & temp32);
    return PMBUS_SUCCESS;
}


Uint8 pmbus_read_write_ioutSetPoint(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_ioutSetPoint();
	else
		return pmbus_write_ioutSetPoint();
}

/**********************
 * End of iout command
 *********************/

//==========================================================================================
//pmbus_write_PSON - Enable or Disable the DCDC Converter
//==========================================================================================
Uint8 pmbus_write_PSON(Uint8 read)
{

	if(pmbus_buffer[1] & 0x01)
	{
		//PREVIOUS_PSON = PSON;
		PSON = 1;
		//Reset the Global Fault Register when PSON has been Re-asserted by DSP
		// if (!PREVIOUS_PSON)
		// {
		// 	Global_Fault_Reg = 0;
		// }
	}
	else
	{
		//PREVIOUS_PSON = PSON;
		PSON = 0;
	}
	return PMBUS_SUCCESS;

}


//==========================================================================================
//pmbus_read_fault_bytes - Read the Global Fault Register
//==========================================================================================
Uint8 pmbus_read_fault_bytes(Uint8 read)
{
	pmbus_read_two_byte_handler(Global_Fault_Reg);
	// Allow capture of all faults that occured since last read, then refresh fault reg
	GDF_Read = 1U;
	Global_Fault_Reg = 0;
	fault_handler();
	return PMBUS_SUCCESS;

}
/********************************
 * End of pmbus_read_fault_bytes
 ********************************/


//==========================================================================================
//pmbus read revisions commands 
//==========================================================================================
Uint8 pmbus_read_major_rev(Uint8 read)
{
	pmbus_read_one_byte_handler((Uint8) MFR_FIRMWARE_V_MAJOR);
	return PMBUS_SUCCESS;
}


Uint8 pmbus_read_minor_rev(Uint8 read)
{
	
	pmbus_read_one_byte_handler((Uint8)MFR_FIRMWARE_V_MINOR);
	return PMBUS_SUCCESS;

}


Uint8 pmbus_read_build_rev(Uint8 read)
{

	pmbus_read_one_byte_handler((Uint8) MFR_FIRMWARE_V_BUILD);
	return PMBUS_SUCCESS;
}
/*********************************
 * End of read revisions commands
 *********************************/


//==========================================================================================
//pmbus read/write OV Limits for Fault generation
//==========================================================================================
inline Uint8 pmbus_write_vin_ov_fault_limit(void)
{

    Uint16 tempbuff = 0;
    Uint16 tempVout = 0;

	tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	tempVout = (tempbuff * UV_OV_COUNTS_MILIVOLTS_SCALER)/UV_OV_COUNTS_MILIVOLTS_SCALER_G;

	if(tempVout > 0) //Need to addsome real invalid data security here 
		PMBus_HVDC_OV_Command = tempVout;
	else
	//if tempVout is invalid, set up a status fault or consider PMBUS_ERROR
		return PMBUS_INVALID_DATA;

	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vin_ov_fault_limit(void)
{
    Uint16 tempbuff = 0;
    tempbuff = PMBus_HVDC_OV_Command/UV_OV_COUNTS_MILIVOLTS_SCALER*UV_OV_COUNTS_MILIVOLTS_SCALER_G;
	pmbus_read_two_byte_handler(tempbuff);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vin_ov_fault_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin_ov_fault_limit();
	else
		return pmbus_write_vin_ov_fault_limit();
}
/***************************************************
 * End of read/write OV Limits for Fault generation
 **************************************************/



//==========================================================================================
//pmbus read/write UV Limits for Fault generation
//==========================================================================================
inline Uint8 pmbus_write_vin_uv_fault_limit(void)
{
	Uint16 tempbuff = 0;
    Uint16 tempVout = 0;

	tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	tempVout = (tempbuff * UV_OV_COUNTS_MILIVOLTS_SCALER)/UV_OV_COUNTS_MILIVOLTS_SCALER_G;

	if(tempVout > 0) //Need to addsome real invalid data security here 
		PMBus_HVDC_UV_Command = tempVout;
	else
	//if tempVout is invalid, set up a status fault or consider PMBUS_ERROR
		return PMBUS_INVALID_DATA;

	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vin_uv_fault_limit(void)
{
    Uint16 tempbuff = 0;
    tempbuff = PMBus_HVDC_UV_Command/UV_OV_COUNTS_MILIVOLTS_SCALER*UV_OV_COUNTS_MILIVOLTS_SCALER_G;
	pmbus_read_two_byte_handler(tempbuff);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vin_uv_fault_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin_uv_fault_limit();
	else
		return pmbus_write_vin_uv_fault_limit();
}
/***************************************************
 * End of read/write UV Limits for Fault generation
 **************************************************/


//==========================================================================================
//pmbus read device temperature
//==========================================================================================
Uint8 pmbus_read_temp_device(Uint8 read)
{
    Uint16 tempbuff = 0;
    tempbuff = (Uint16)Synchro_Temp;
	pmbus_read_two_byte_handler(tempbuff);
    return PMBUS_SUCCESS;
}
/***************************************************
 * End of PMBus read device temp
 **************************************************/



//******************************************************************************************
//
//----------------------------Begin Calibration Routines------------------------------------
//
//******************************************************************************************


//==========================================================================================
//----Read/Write Calibration offsets
//==========================================================================================

Uint8 pmbus_read_write_hwversion(Uint8 pmbus_read)
{
	Uint16 tempbuff = 0;
	int32 temp32 = 0;

	if(pmbus_read) {
		temp32 = hw_version;
		pmbus_read_two_byte_handler(0xFFFF & temp32);
	}
	else {
		tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
		if (tempbuff < HW_Gains_Count)
		{
			hw_version = tempbuff;
		}
	}
    return PMBUS_SUCCESS;

}

Uint8 pmbus_read_write_vout_offset(Uint8 pmbus_read)
{
	Uint16 tempbuff = 0;
	int32 temp32 = 0;

	if(pmbus_read) {
		temp32 = (int32)(int16)(pmbus_Vout_offset / 10);
		pmbus_read_two_byte_handler(0xFFFF & temp32);
	}
	else {
		tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
		if (ABS((int16)tempbuff) > MAX_VOUT_OFFSET_cV)
		{
			tempbuff = (Uint16)(((int16)tempbuff > 0)?((int16)MAX_VOUT_OFFSET_cV):(-1*(int16)MAX_VOUT_OFFSET_cV));
		}
		temp32 = ((int32)(int16)tempbuff) * 10;
		pmbus_Vout_offset = temp32;
	}
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_ibat_offset(Uint8 pmbus_read)
{
	Uint16 tempbuff = 0;
	int16 temp16 = 0;
	if(pmbus_read) {
		temp16 = pmbus_Ibat_offset;
		pmbus_read_two_byte_handler((Uint16)temp16);
	}
	else {
		tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
		temp16 = ((int16)tempbuff);
		if (ABS(temp16) > MAX_IBATT_OFFSET_mA)
		{
			temp16 = (((int16)tempbuff > 0)?((int16)MAX_IBATT_OFFSET_mA):(-1*(int16)MAX_IBATT_OFFSET_mA));
		}
		pmbus_Ibat_offset = temp16;
	}

	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_iout_offset(Uint8 pmbus_read)
{
	Uint16 tempbuff = 0;

	if(pmbus_read) {
		tempbuff = (Uint16)pmbus_Iout_offset;
		pmbus_read_two_byte_handler(0xFFFF & tempbuff);
	}
	else {
		tempbuff = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
		if (ABS((int16)tempbuff) > MAX_ILOAD_OFFSET_mA)
		{
			tempbuff = (Uint16)(((int16)tempbuff > 0)?((int16)MAX_ILOAD_OFFSET_mA):(-1*(int16)MAX_ILOAD_OFFSET_mA));
		}
		pmbus_Iout_offset = (int16)tempbuff;
	}

	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_dac_ibatt_offset(Uint8 pmbus_read)
{
	Uint8 ans;
	int32 temp32;
	if (pmbus_read)
	{
		pmbus_read_two_byte_handler((Uint16)(dac_ibatt_offset));
		ans = PMBUS_SUCCESS;
	}
	else
	{
		temp32 = (int32)(int16)(pmbus_buffer[1] + (pmbus_buffer[2] << 8));
		temp32 += (int32)(dac_ibatt_offset);
		if (temp32 >= MAX_DAC_IBATT_OFFSET_mA)
		{
			temp32 = MAX_DAC_IBATT_OFFSET_mA;
		}
		if (temp32 <= -1*MAX_DAC_IBATT_OFFSET_mA)
		{
			temp32 = -1*MAX_DAC_IBATT_OFFSET_mA;
		}
		dac_ibatt_offset = (int16)temp32;
		dac_ibat_offset_digits=(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/KG;
		ans = PMBUS_SUCCESS;
	}
	return ans;
}


//==========================================================================================
//----Read back adc values with calculated offsets
//==========================================================================================
Uint8 pmbus_read_vout_value(Uint8 read)
{ 
	int32 temp32 = 0;
    temp32 = (int32)(int16)((vout_mV)/10);    
	pmbus_read_two_byte_handler(0xFFFF & temp32);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_ibat_value(Uint8 read)
{
	int16 temp16;
	if (INT16_MIN > ibatt_mA)
	{
		temp16 = INT16_MIN;
	}
	else if (INT16_MAX < ibatt_mA)
	{
		temp16 = INT16_MAX;
	}
	else
	{
		temp16 = (int16)ibatt_mA;
	}
	pmbus_read_two_byte_handler((Uint16)temp16);
	// pmbus_read_two_byte_handler((Uint16)adc_values.Ibatt);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_iout_value(Uint8 read)
{ 
	int32 temp32 = 0;
	temp32 = (int32)(iout_mA);
	pmbus_read_two_byte_handler(0xFFFF & temp32);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_dac_ibatt_value(Uint8 read)
{
	Uint8 ans;
	if (read)
	{
		pmbus_read_two_byte_handler((Uint16)((int16)(DAC_IBAT_COUNTS_TO_MILLIAMP(IoutCommand))));
		ans = PMBUS_SUCCESS;
	}
	else
	{
		ans = pmbus_invalid_write();
	}
	return ans;
}


//******************************************************************************************
//----------------------------End Calibration Routines------------------------------------
//******************************************************************************************









Uint8 pmbus_read_write_vout_mode(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_one_byte_handler((32 - VOUT_MODE_EXP));
	else
		return pmbus_invalid_write();
}


//==========================================================================================
//pmbus_write_vout_transition_rate
//pmbus_read_vout_transition_rate
//==========================================================================================
inline Uint8 pmbus_write_vout_transition_rate(void)
{
	pmbus_dcdc_config[0].vout_transition_rate = 21; //pmbus_buffer[1] + (pmbus_buffer[2] << 8);

	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vout_transition_rate(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].vout_transition_rate);

	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vout_transition_rate(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vout_transition_rate();
	else
		return pmbus_write_vout_transition_rate();
}


/***********************
 * Start of fault command
 **********************/
inline Uint8 pmbus_write_clear_faults(void)
{
	switch (supply_state)
	{
	case STATE_CURRENT_REGULATED:
//	case STATE_CPCC:
		{
			pmbus_status_word = 0;
			//fault_status = 0;
			//dpwm_fault_status = 0;
			MiscAnalogRegs.GLBIOVAL.all &= ~((Uint32)MASK_FAILURE);
			return PMBUS_SUCCESS;
		}
	case STATE_IDLE:
		{
		//	pmbus_status_word = PMBUS_STATUS_WORD_HIGH_BYTE |
		//						PMBUS_STATUS_WORD_OFF;
			//fault_status = 0;
			//dpwm_fault_status = 0;
			MiscAnalogRegs.GLBIOVAL.all &= ~((Uint32)MASK_FAILURE);
			return PMBUS_SUCCESS;
		}
	default: return PMBUS_SUCCESS;
	}
}

Uint8 pmbus_read_write_clear_faults(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_invalid_read();
	else
		return pmbus_write_clear_faults();
}

Uint8 DbounceCntr = 0;
Uint8 DbounceVal = 0;


Uint8 pmbus_read_write_vin_on_limit(Uint8 pmbus_read)
{
    return PMBUS_SUCCESS;
}


//OV Warn
inline Uint8 pmbus_write_vout_ov_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_vout_ov_warn_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].vout_ov_warn_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vout_ov_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vout_ov_warn_limit();
	else
		return pmbus_write_vout_ov_warn_limit();
}

//UV Warn
inline Uint8 pmbus_write_vout_uv_warn_limit(void)
{
	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vout_uv_warn_limit(void)
{
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vout_uv_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vout_uv_warn_limit();
	else
		return pmbus_write_vout_uv_warn_limit();
}

//OC Fault
inline Uint8 pmbus_write_iout_oc_fault_limit(void)
{
	pmbus_dcdc_config[0].iout_oc_fault_limit = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	configure_iout_ocp();
	configure_fault_levels();
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_iout_oc_fault_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].iout_oc_fault_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_iout_oc_fault_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_iout_oc_fault_limit();
	else
		return pmbus_write_iout_oc_fault_limit();
}

//OC Warn
inline Uint8 pmbus_write_iout_oc_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_iout_oc_warn_limit(void)
{
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_iout_oc_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_iout_oc_warn_limit();
	else
		return pmbus_write_iout_oc_warn_limit();
}


//OT Fault
inline Uint8 pmbus_write_temp_ot_fault_limit(void)
{
	pmbus_dcdc_config[0].temp_ot_fault_limit = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	configure_fault_levels();
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_temp_ot_fault_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].temp_ot_fault_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_temp_ot_fault_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_temp_ot_fault_limit();
	else
		return pmbus_write_temp_ot_fault_limit();
}

//OT Warn
inline Uint8 pmbus_write_temp_ot_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_temp_ot_warn_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].temp_ot_warn_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_temp_ot_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_temp_ot_warn_limit();
	else
		return pmbus_write_temp_ot_warn_limit();
}


//VIN OV Warn
inline Uint8 pmbus_write_vin_ov_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_vin_ov_warn_limit(void)
{
	return pmbus_invalid_read();
}

Uint8 pmbus_read_write_vin_ov_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin_ov_warn_limit();
	else
		return pmbus_write_vin_ov_warn_limit();
}


//VIN UV Warn
inline Uint8 pmbus_write_vin_uv_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_vin_uv_warn_limit(void)
{
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vin_uv_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin_uv_warn_limit();
	else
		return pmbus_write_vin_uv_warn_limit();
}
//IIN OC Fault
inline Uint8 pmbus_write_iin_oc_fault_limit(void)
{
	pmbus_dcdc_config[0].iin_oc_fault_limit = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	configure_ipri_cycle_by_cycle();
	configure_fault_levels();
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_iin_oc_fault_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].iin_oc_fault_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_iin_oc_fault_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_iin_oc_fault_limit();
	else
		return pmbus_write_iin_oc_fault_limit();
}

//IIN OC Warn
inline Uint8 pmbus_write_iin_oc_warn_limit(void)
{
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_iin_oc_warn_limit(void)
{
	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].iin_oc_warn_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_iin_oc_warn_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_iin_oc_warn_limit();
	else
		return pmbus_write_iin_oc_warn_limit();
}

//PGOOD ON
inline Uint8 pmbus_write_pgood_on_limit(void)
{
	//pmbus_dcdc_config[0].pgood_on_limit = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	//configure_pgood_levels();
	return PMBUS_SUCCESS;
}
inline Uint8 pmbus_read_pgood_on_limit(void)
{
	//pmbus_read_two_byte_handler(pmbus_dcdc_config[0].pgood_on_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_pgood_on_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_pgood_on_limit();
	else
		return pmbus_write_pgood_on_limit();
}

Uint8 pmbus_read_write_temp_mosfet(Uint8 pmbus_read)
{


	return PMBUS_SUCCESS;
}
//PGOOD OFF
inline Uint8 pmbus_write_pgood_off_limit(void)
{
	//pmbus_dcdc_config[0].pgood_off_limit = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	//configure_pgood_levels();
	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_pgood_off_limit(void)
{
	//pmbus_read_two_byte_handler(pmbus_dcdc_config[0].pgood_off_limit);
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_pgood_off_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_pgood_off_limit();
	else
		return pmbus_write_pgood_off_limit();
}

/*************************************
 * End of fault command
 *************************************/


/*************************************
 * Start of Vout sequence command
 ************************************/

inline Uint8 pmbus_write_ton_rise(void)
{
  	pmbus_dcdc_config[0].ton_rise = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
    configure_ton_rise();
  	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_ton_rise(void)
{

  	pmbus_read_two_byte_handler(pmbus_dcdc_config[0].ton_rise);

  	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_ton_rise(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_ton_rise();
	else
		return pmbus_write_ton_rise();
}

/*******************************************
 * End of vout sequence command
 *******************************************/


Uint8 pmbus_read_write_vout_uv_fault_limit(Uint8 pmbus_read)
{
		return 1;
}


Uint8 pmbus_read_write_vout_ov_fault_limit(Uint8 pmbus_read)
{

		return 1;
}

/*******************************************
 * Start of read param command
 ******************************************/
#define max(a,b)					(((a)>(b)) ? (a):(b))

//==========================================================================================
// pmbus_read_vout()
//==========================================================================================
inline Uint8 pmbus_read_vout(void)
{
	Uint16 vout_adc12_value = 0;
	//struct qnote vout_adc_scaler = {VOUT_ADC_TO_LITERAL_SCALER, VOUT_ADC_TO_LITERAL_SHIFT - 16};
	//struct qnote vout_adc_scaler =  {11889 , -11}; 	//0.010960514 {24884 , -11}; 	//0.010960514 = 1/(4095/VoutTf*2.5)

	vout_adc12_value = 10000;//(qnote_to_linear11(qnote_scale(vout_adc_scaler, adc_values.Vout )));
	pmbus_read_two_byte_handler(vout_adc12_value);
	return 0;
}

Uint8 pmbus_read_write_vout(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vout();
	else
		return pmbus_invalid_write();
}
/*
The mantissa is calculated by the following relationship:
400V/1.93V
Round(2.5 / 1.93 * 400) = 518.
If the exponent is 0 the available range of VIN that can be sensed is
0V to 518V.
*/
inline Uint8 pmbus_read_vin(void)
{
	Uint16 value;

	struct qnote constant_0_12408 = {22933 , -16};	//0.12408549 = 1/(4095/VinTf*2.5)

	value = qnote_to_linear11(qnote_scale(constant_0_12408, adc_values.Vin ));
	pmbus_read_two_byte_handler(value);

	return 0;
}

Uint8 pmbus_read_write_vin(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin();
	else
		return pmbus_invalid_write();

}


Uint8 pmbus_read_write_temp_device(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_temp_device(pmbus_read);
	else
		return pmbus_invalid_write();
}

inline Uint8 pmbus_read_frequency(void)
{
	Uint16 value;
	struct qnote tempo;
	if((supply_state == STATE_IDLE) /*|| (supply_state == STATE_LATCH)*/)
	{
		value =0;
	}
	else
	{
	  struct qnote constant_0_04 = {20972 , -19}; //This struct represents the floating point number: 0.04
	  value = (Uint16)(100000000/(Uint32)(pmbus_dcdc_config[0].period));
	  tempo = qnote_scale(constant_0_04, value);
	  value = qnote_to_linear11(tempo);
	}
	pmbus_read_two_byte_handler(value);

	return 0;

}
Uint8 pmbus_read_write_frequency(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_frequency();
	else
		return pmbus_invalid_write();
}
/*************************************************
 * End of read param command
 *************************************************/

/**************************************************
 * Start of Communication, Memory, and Logic fault
 **************************************************/
inline Uint8 pmbus_write_restore_default_all(void)
{
    gpio_dpwm_off();
    restore_default_all();
#if FLASH_CONFIG
	// Reconfigure periphearals here
	configure_dpwm_timing();
	configure_cpcc();
#endif
    //configure_vin_on_off_thresholds();
	configure_filter_parameters();
	configure_burst_mode();
	configure_ipri_cycle_by_cycle();
	configure_ovp();
	configure_iout_ocp();
	configure_fault_levels();
	//configure_pgood_levels();
	configure_vout_cmd();
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_restore_default_all(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_invalid_read();
	else
		return pmbus_write_restore_default_all();
}



//==========================================================================================
// pmbus_write_store_default_all()
//	Store all PMBus-configurable variables from RAM to Data Flash.
//==========================================================================================
inline Uint8 pmbus_write_store_default_all(void)
{
    return pmbus_invalid_write();
}

Uint8 pmbus_read_write_store_default_all(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_invalid_read();
	else
		return pmbus_write_store_default_all();
}
/**************************************************
 * End of Communication, Memory, and Logic fault
 **************************************************/

/**************************************************
 * Start of specific command
 **************************************************/
inline Uint8 pmbus_write_deadband_config(void)
{
 	pmbus_dcdc_config[0].dead_time_1 = pmbus_buffer[3] + (pmbus_buffer[2] << 8);
 	pmbus_dcdc_config[0].dead_time_2 = pmbus_buffer[5] + (pmbus_buffer[4] << 8);
 	pmbus_dcdc_config[0].dead_time_3 = pmbus_buffer[7] + (pmbus_buffer[6] << 8);
 	pmbus_dcdc_config[0].dead_time_4 = pmbus_buffer[9] + (pmbus_buffer[8] << 8);

	configure_dpwm_timing();
 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_deadband_config(void)
{
	pmbus_number_of_bytes = 9;

	pmbus_buffer[0] = 8;

	pmbus_buffer[1] = 	pmbus_dcdc_config[0].dead_time_1 >> 8;
	pmbus_buffer[2] = 	pmbus_dcdc_config[0].dead_time_1 & 0xFF;

	pmbus_buffer[3] = 	pmbus_dcdc_config[0].dead_time_2 >> 8;
	pmbus_buffer[4] = 	pmbus_dcdc_config[0].dead_time_2 & 0xFF;

	pmbus_buffer[5] = 	pmbus_dcdc_config[0].dead_time_3 >> 8;
	pmbus_buffer[6] = 	pmbus_dcdc_config[0].dead_time_3 & 0xFF;

	pmbus_buffer[7] = 	pmbus_dcdc_config[0].dead_time_4 >> 8;
	pmbus_buffer[8] = 	pmbus_dcdc_config[0].dead_time_4 & 0xFF;

 	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_deadband_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_deadband_config();
	else
		return pmbus_write_deadband_config();
}

inline Uint8 pmbus_write_mode_switching_config(void)
{
 	pmbus_dcdc_config[0].period = pmbus_buffer[3] + (pmbus_buffer[2] << 8);
// 	pmbus_dcdc_config[0].max_period = pmbus_buffer[5] + (pmbus_buffer[4] << 8);
// 	pmbus_dcdc_config[0].tsrmax 	= pmbus_buffer[7] + (pmbus_buffer[6] << 8);
	configure_ton_rise();
	configure_dpwm_timing();
 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_mode_switching_config(void)
{
	pmbus_number_of_bytes = 7;

	pmbus_buffer[0] = 6;

	pmbus_buffer[1] = 	pmbus_dcdc_config[0].period  >> 8;
	pmbus_buffer[2] = 	pmbus_dcdc_config[0].period & 0xFF;

 	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_mode_switching_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_mode_switching_config();
	else
		return pmbus_write_mode_switching_config();
}


inline Uint8 pmbus_write_light_load_config(void)
{
 	pmbus_dcdc_config[0].ll_turn_on_thresh  = pmbus_buffer[4] + (pmbus_buffer[3] << 8) + (pmbus_buffer[2] << 16);
 	pmbus_dcdc_config[0].ll_turn_off_thresh = pmbus_buffer[7] + (pmbus_buffer[6] << 8) + (pmbus_buffer[5] << 16);
 	pmbus_dcdc_config[0].ll_en 				= pmbus_buffer[8] >> 6;
	configure_burst_mode();
 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_light_load_config(void)
{
	pmbus_number_of_bytes = 8;

	pmbus_buffer[0] = 7;

	pmbus_buffer[1] =  (pmbus_dcdc_config[0].ll_turn_on_thresh >> 16) & 0xFF;
	pmbus_buffer[2] =  (pmbus_dcdc_config[0].ll_turn_on_thresh >> 8) & 0xFF;
	pmbus_buffer[3] = 	pmbus_dcdc_config[0].ll_turn_on_thresh & 0xFF;

	pmbus_buffer[4] =  (pmbus_dcdc_config[0].ll_turn_off_thresh >> 16) & 0xFF;
	pmbus_buffer[5] =  (pmbus_dcdc_config[0].ll_turn_off_thresh >> 8) & 0xFF;
	pmbus_buffer[6] = 	pmbus_dcdc_config[0].ll_turn_off_thresh & 0xFF;

	pmbus_buffer[7] = 	pmbus_dcdc_config[0].ll_en;// For read, there is no need to left shift<< 6;

 	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_light_load_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_light_load_config();
	else
		return pmbus_write_light_load_config();
}

inline Uint8 pmbus_write_cpcc_config(void)
{
 	pmbus_dcdc_config[0].cpcc_pmax   = pmbus_buffer[3] + (pmbus_buffer[2] << 8);
 	pmbus_dcdc_config[0].cpcc_imax   = pmbus_buffer[5] + (pmbus_buffer[4] << 8);
 	pmbus_dcdc_config[0].cpcc_ton 	 = pmbus_buffer[7] + (pmbus_buffer[6] << 8);
	pmbus_dcdc_config[0].cpcc_enable = pmbus_buffer[8];
	configure_cpcc();
 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_cpcc_config(void)
{
	pmbus_number_of_bytes = 9;

	pmbus_buffer[0] = 8;

	pmbus_buffer[1] = 	pmbus_dcdc_config[0].cpcc_pmax  >> 8;
	pmbus_buffer[2] = 	pmbus_dcdc_config[0].cpcc_pmax & 0xFF;

	pmbus_buffer[3] = 	pmbus_dcdc_config[0].cpcc_imax >> 8;
	pmbus_buffer[4] = 	pmbus_dcdc_config[0].cpcc_imax & 0xFF;

	pmbus_buffer[5] = 	pmbus_dcdc_config[0].cpcc_ton >> 8;
	pmbus_buffer[6] = 	pmbus_dcdc_config[0].cpcc_ton & 0xFF;

	pmbus_buffer[7] = 	pmbus_dcdc_config[0].cpcc_enable & 0xFF;
	pmbus_buffer[8] = 	pmbus_dcdc_config[0].cpcc_time_out_en & 0xFF;

 	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_cpcc_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_cpcc_config();
	else
		return pmbus_write_cpcc_config();
}

inline Uint8 pmbus_write_vff_config(void)
{
  pmbus_dcdc_config[0].vff_config = pmbus_buffer[1] ;
  if(pmbus_dcdc_config[0].vff_config & 0x01)
  {
    //Filter0Regs.FILTERCTRL.bit.OUTPUT_MULT_SEL = 2; // Feed-Forward - Multiply by another Filter output
  }
  else
  {
    //Filter0Regs.FILTERCTRL.bit.OUTPUT_MULT_SEL = 0;// Or 1? // Multiply by KCOMP (Or period)
  }
  return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vff_config(void)
{
  	pmbus_number_of_bytes = 1;
	pmbus_buffer[0] = pmbus_dcdc_config[0].vff_config;
	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vff_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vff_config();
	else
		return pmbus_write_vff_config();
}

inline Uint8 pmbus_write_blank_config(void)
{
  	pmbus_dcdc_config[0].blank_time[0] = pmbus_buffer[3] + (pmbus_buffer[2] << 8);
 	pmbus_dcdc_config[0].blank_time[1] = pmbus_buffer[5] + (pmbus_buffer[4] << 8);
 	pmbus_dcdc_config[0].blank_time[2] = pmbus_buffer[7] + (pmbus_buffer[6] << 8);
 	pmbus_dcdc_config[0].blank_time[3] = pmbus_buffer[9] + (pmbus_buffer[8] << 8);

	Dpwm0Regs.DPWMBLKABEG.bit.BLANK_A_BEGIN = pmbus_dcdc_config[0].blank_time[0];
	Dpwm0Regs.DPWMBLKAEND.bit.BLANK_A_END   = pmbus_dcdc_config[0].blank_time[1];
	Dpwm0Regs.DPWMBLKBBEG.bit.BLANK_B_BEGIN = pmbus_dcdc_config[0].blank_time[2];
	Dpwm0Regs.DPWMBLKBEND.bit.BLANK_B_END   = pmbus_dcdc_config[0].blank_time[3];

	Dpwm1Regs.DPWMBLKABEG.bit.BLANK_A_BEGIN = pmbus_dcdc_config[0].blank_time[0];
	Dpwm1Regs.DPWMBLKAEND.bit.BLANK_A_END   = pmbus_dcdc_config[0].blank_time[1];
	Dpwm1Regs.DPWMBLKBBEG.bit.BLANK_B_BEGIN = pmbus_dcdc_config[0].blank_time[2];
	Dpwm1Regs.DPWMBLKBEND.bit.BLANK_B_END   = pmbus_dcdc_config[0].blank_time[3];

	Dpwm2Regs.DPWMBLKABEG.bit.BLANK_A_BEGIN = pmbus_dcdc_config[0].blank_time[0];
	Dpwm2Regs.DPWMBLKAEND.bit.BLANK_A_END   = pmbus_dcdc_config[0].blank_time[1];
	Dpwm2Regs.DPWMBLKBBEG.bit.BLANK_B_BEGIN = pmbus_dcdc_config[0].blank_time[2];
	Dpwm2Regs.DPWMBLKBEND.bit.BLANK_B_END   = pmbus_dcdc_config[0].blank_time[3];

//	Dpwm3Regs.DPWMBLKABEG.bit.BLANK_A_BEGIN = pmbus_dcdc_config[0].blank_time[0];
//	Dpwm3Regs.DPWMBLKAEND.bit.BLANK_A_END   = pmbus_dcdc_config[0].blank_time[1];
//	Dpwm3Regs.DPWMBLKBBEG.bit.BLANK_B_BEGIN = pmbus_dcdc_config[0].blank_time[2];
//	Dpwm3Regs.DPWMBLKBEND.bit.BLANK_B_END   = pmbus_dcdc_config[0].blank_time[3];

 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_blank_config(void)
{
    pmbus_number_of_bytes = 9;
    pmbus_buffer[0] = 8;

	pmbus_buffer[1] = 	pmbus_dcdc_config[0].blank_time[0]  >> 8;
	pmbus_buffer[2] = 	pmbus_dcdc_config[0].blank_time[0] & 0xFF;

	pmbus_buffer[3] = 	pmbus_dcdc_config[0].blank_time[1] >> 8;
	pmbus_buffer[4] = 	pmbus_dcdc_config[0].blank_time[1] & 0xFF;

	pmbus_buffer[5] = 	pmbus_dcdc_config[0].blank_time[2] >> 8;
	pmbus_buffer[6] = 	pmbus_dcdc_config[0].blank_time[2] & 0xFF;

	pmbus_buffer[7] = 	pmbus_dcdc_config[0].blank_time[3] >> 8;
	pmbus_buffer[8] = 	pmbus_dcdc_config[0].blank_time[3] & 0xFF;

    return PMBUS_SUCCESS;
}


Uint8 pmbus_read_write_blank_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_blank_config();
	else
		return pmbus_write_blank_config();
}


inline Uint8 pmbus_write_ide_config(void)
{
  pmbus_dcdc_config[0].ide_config = pmbus_buffer[1] ;
  return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_ide_config(void)
{
  pmbus_number_of_bytes = 1;
  pmbus_buffer[0] = pmbus_dcdc_config[0].ide_config;
  return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_ide_config(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_ide_config();
	else
		return pmbus_write_ide_config();
}
/**************************************************
 * End of specific command
 **************************************************/

/**************************************************
 * Start of pmbus status command
 **************************************************/
inline Uint8 pmbus_read_status_bit_mask(void)
{
	 Uint8 byte;

	 for (byte = 0; byte < sizeof(status_bit_mask); byte++)
	 {
		  pmbus_buffer[byte+1] = (Uint8)status_bit_mask[byte];
	 }
	 pmbus_buffer[0] = sizeof(status_bit_mask);
	 pmbus_number_of_bytes = sizeof(status_bit_mask) + 1;
	 return 0;
}

Uint8 pmbus_read_write_status_bit_mask(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_status_bit_mask();
	else
		return pmbus_invalid_write();
}

Uint8 pmbus_read_write_status_byte(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_one_byte_handler((Uint8) (pmbus_status_word & 0xFF));
	else
		return pmbus_invalid_write();
}

Uint8 pmbus_read_write_status_word(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_two_byte_handler(pmbus_status_word);
	else
		return pmbus_invalid_write();
}
/**************************************************
 * End of pmbus status command
 **************************************************/

/***************************************************
 * Start of other commands
 ***************************************************/

Uint8 pmbus_read_write_dcdc_op_state(Uint8 pmbus_read)
{
	Uint8 ans;
	if (pmbus_read)
	{
		ans = pmbus_read_two_byte_handler(dcdc_op_state);
	}
	else
	{
		ans = pmbus_invalid_write();
	}
	return ans;
}


//==========================================================================================
inline Uint8 pmbus_write_vin_off_limit(void)
{
	//pmbus_dcdc_config[0].vin_off_limit = 12;//pmbus_buffer[1] + (pmbus_buffer[2] << 8);
	//configure_vin_on_off_thresholds();
 	return PMBUS_SUCCESS;
}

inline Uint8 pmbus_read_vin_off_limit(void)
{
    //pmbus_read_two_byte_handler(pmbus_dcdc_config[0].vin_off_limit);
 	return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_vin_off_limit(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_vin_off_limit();
	else
		return pmbus_write_vin_off_limit();
}


inline Uint8 pmbus_write_frequency_switch(void)
{
    Uint32 freq_switch;
    struct qnote freq_switch_qnote;

	//peroid_old = pmbus_dcdc_config[0].period;
    freq_linear11 = pmbus_buffer[1] + (pmbus_buffer[2] << 8);
    freq_switch_qnote = linear11_to_qnote (freq_linear11);
    freq_switch = ((Uint32)freq_switch_qnote.mantissa) << ( 6 + freq_switch_qnote.exponent);
	//peroid_new = (Uint16)((Uint32)255434304/freq_switch);

    return PMBUS_SUCCESS;
}


inline Uint8 pmbus_read_frequency_switch(void)
{
    pmbus_read_two_byte_handler(freq_linear11);
    return PMBUS_SUCCESS;
}

Uint8 pmbus_read_write_frequency_switch(Uint8 pmbus_read)
{
	if(pmbus_read)
		return pmbus_read_frequency_switch();
	else
		return pmbus_write_frequency_switch();
}
/***************************************************
 * End of other commands
 ***************************************************/
