#ifndef PMBUS_TOPOLOGY_H
#define PMBUS_TOPOLOGY_H

#include "dcdc_parameters.h"

#define STRING(x) #x
#define STRING$(x) STRING(x)

//-----------------------
//setup ID description:
//DCDC PFC | CLA 1           CLA2               CLA 3              CLA 4             |  DCDC rail 1 |   DCDC rail 2
//-----------------------------------------------------------------------------------------------------------------
// #   #    rail# loop type   rail# loop type   rail# loop type    rail# loop type      Topology      Topology
// R   R     O       R         O       R         O      R           O       R              O            O
// R: means the field is required to have.
// O: means the field is optional to have
//
//DCDC #: required. choose from 0, 1, 2;  # of DCDC rails
//PFC #:  required. choose from 0, 1;  # of PFC
//rail #: optional. choose from 1, 2 if the associated CLA loop type is either C or V.
//loop type: required. choose from C, V, N. C: current loop. V: voltage loop. N: N/A.
//           defines each CLA's loop type.
//Topology:  optional.DCDC rail topology. choose from: TTIF, PSHFB, ZVSPWM, LLC, HSFB
//-----------------------------------------------------------------------------------------
#if   defined(UCD3138)
  #define DEVICE           	 UCD3100ISO1  				//Device Name
#elif defined(UCD3138A)
  #define DEVICE           	 UCD3138A  				    //Device Name
#elif defined(UCD3138064)
  #define DEVICE           	 UCD310064V1  				//Device Name
#elif defined(UCD3138064A)
  #define DEVICE           	 UCD3138064A  				//Device Name
#elif defined(UCD3138A64)
  #define DEVICE           	 UCD310A64V1  				//Device Name
#elif defined(UCD3138128)
  #define DEVICE           	 UCD310128V1  				//Device Name
#endif
  
#define MFR_ID				"Current Ways, Inc."						///Hardware Manufacturer
#define SETUP_ID            "VERSION1|BUCKSYNC000"			//Generic setup id
#define MFR_REVISION    	"V"	STRING$(MFR_FIRMWARE_V_MAJOR.MFR_FIRMWARE_V_MINOR.MFR_FIRMWARE_V_BUILD)					//Hardware revision
#define MFR_MODEL			"HDOBC_MY20_DCDC"			//Hardware Model/
#define MFR_DATE			"YYMMDD"
#define MFR_LOCATION		"Simi Valley, CA"
#define MFR_SERIAL			"XXXXX"
#define MFR_IC_DEVICE		"DCDC_CONVERTER"
#define MFR_IC_DEVICE_REV	"4"


#if   defined(UCD3138)
  #define DEVICE_ID           	 "UCD3100ISO1|0.0.11.0105|141009"  				//Device id
#elif defined(UCD3138A)
  #define DEVICE_ID           	 "UCD3138A|0.0.11.0105|141009" 				    //Device id
#elif defined(UCD3138064)
  #define DEVICE_ID           	 "UCD310064V1|0.0.11.0105|141009" 				//Device id
#elif defined(UCD3138064A)
  #define DEVICE_ID           	 "UCD3138064A|0.0.11.0105|141009" 				//Device id
#elif defined(UCD3138A64)
  #define DEVICE_ID           	 "UCD310A64V1|0.0.11.0105|141009"   		    //Device id
#elif defined(UCD3138128)
  #define DEVICE_ID           	 "UCD310128V1|0.0.11.0105|141009"   		    //Device id
#endif

#define MFR_ID_LENGTH 				sizeof(MFR_ID)
#define SETUP_ID_LENGTH 			sizeof(SETUP_ID)
#define MFR_MODEL_LENGTH 		sizeof(MFR_MODEL)
#define MFR_REVISION_LENGTH 		sizeof(MFR_REVISION)
#define MFR_DATE_LENGTH				sizeof(MFR_DATE)
#define MFR_LOCATION_LENGTH			sizeof(MFR_LOCATION)
#define MFR_SERIAL_LENGTH			sizeof(MFR_SERIAL)
#define MFR_IC_DEVICE_LENGTH		sizeof(MFR_IC_DEVICE)
#define MFR_IC_DEVICE_REV_LENGTH	sizeof(MFR_IC_DEVICE_REV)
#define DEVICE_ID_LENGTH			sizeof(DEVICE_ID)


#define STATUS_BIT_MASK    {0x00, 0x00, 0x00, 0x80, 0x00, 0x80, 0x80, 0x34, 0xC8, 0x00, 0x00}



#define BIT_MASK(bit)             (1 << (bit))
#define SET_BIT(value,bit)        ((value) |= BIT_MASK(bit))
#define CLEAR_BIT(value,bit)      ((value) &= ~BIT_MASK(bit))



/******************************
 * Start of power parameter
 *****************************/


typedef struct
{
 Uint16 vout_cal_offset;
 Uint16 iout_cal_gain;
 Uint16 iout_cal_offset;
 Uint16 temperature_cal_offset;
 Uint16 temperature_cal_gain;
 Uint16 vout_cal_monitor;
}PMBUS_DCDC_CAL;//must be even number of int16

//first rail
#define DEFAULT_PMBUS_DCDC_1_CAL {0, 0, 0, 0, 0, 0}

//second rail default value
#define DEFAULT_PMBUS_DCDC_2_CAL  {0, 0, 0, 0, 0, 0}

typedef struct
{
 Uint16 vin_scale;
 Uint16 vin_offset;
}PMBUS_DCDC_CAL_NONPAGED; //must be even number of int16

typedef struct
{
 Uint16 vout_cmd;
 Uint16 vout_ov_fault_limit;
 Uint16 vout_ov_warn_limit;
 Uint16 vout_uv_fault_limit;
 Uint16 vout_uv_warn_limit;
 Uint16 iout_oc_fault_limit;
 Uint16 iout_oc_warn_limit;
 Uint16 temp_ot_fault_limit;
 Uint16 temp_ot_warn_limit;
 Uint16 vin_ov_fault_limit;
 Uint16 vin_ov_warn_limit;
 Uint16 vin_uv_fault_limit;
 Uint16 vin_uv_warn_limit;
 Uint16 iin_oc_fault_limit;
 Uint16 iin_oc_warn_limit;
 Uint16 ton_rise;
 Uint16 vout_transition_rate;
 Uint16 dead_time_1;
 Uint16 dead_time_2;
 Uint16 dead_time_3;
 Uint16 dead_time_4;
 Uint16 period;
 Uint32 ll_turn_on_thresh;
 Uint32 ll_turn_off_thresh;
 Uint8  ll_en;
 Uint16 cpcc_pmax;
 Uint16 cpcc_imax;
 Uint16	cpcc_ton;
 Uint8	cpcc_enable;
 Uint8  cpcc_time_out_en;
 Uint8  vff_config;
 Uint8  ide_config;
 Uint16 blank_time[4];
 Uint8	rsvd;
}PMBUS_DCDC_CONFIG; //must be even number of int16

#define VOUT_0 				(6144)
#define VOUT_OV_FAULT_LIMIT (7168)
#define VOUT_OV_WARN_LIMIT 	(6656)
#define VOUT_UV_FAULT_LIMIT (1536)
#define VOUT_UV_WARN_LIMIT 	(5632)
#define IOUT_OC_FAULT_LIMIT (38)
#define IOUT_OC_WARN_LIMIT 	(32)
#define TEMP_OT_FAULT_LIMIT (50)
#define TEMP_OT_WARN_LIMIT 	(40)
#define VIN_OV_FAULT_LIMIT 	(75)
#define VIN_OV_WARN_LIMIT 	(73)
#define VIN_UV_FAULT_LIMIT 	(33)
#define VIN_UV_WARN_LIMIT 	(34)
#define IIN_OC_FAULT_LIMIT	(16)
#define IIN_OC_WARN_LIMIT	(13)
#define PGOOD_ON_LIMIT		(5888)
#define PGOOD_OFF_LIMIT		(5632)
#define VIN_ON_LIMIT		(37)
#define VIN_OFF_LIMIT		(35)
#define TON_RISE 			(20)
#define VOUT_0_TRAN 		(1)
#define DEADTIME1			(420) // OB Fall to OA Rise
#define DEADTIME2			(400) // OA Fall to 0B Rise
#define DEADTIME3			(420) // 1B Fall to 1A Rise
#define DEADTIME4			(400) // 1A Fall to 1B Rise
#define PERIOD1				(20000)// default value, (20000 * 0.25nS) = 5uS = 200KHZ
#define TURN_ON_THRESHOLD	(2000)//(25000)
#define TURN_OFF_THRESHOLD	(1000)//(25000)
#define LL_ENABLE			(0)
#define CPCC_PMAX			(360)   //167 is 21A
#define CPCC_IMAX			(36) //(24) //(40)  //16 is 26A wrong scalling
#define CPCC_TON			(100)
#define CPCC_ENABLE			(1)
#define CPCC_TIME_OUT_EN	(0)
#define VFF_CONFIG          (1)
#define IDE_CONFIG          (1)
#define BLANK_TIME          {1,10,1,10}



//first rail default value
#define DEFAULT_PMBUS_DCDC_1_CONFIG  {\
	VOUT_0, \
	VOUT_OV_FAULT_LIMIT, \
	VOUT_OV_WARN_LIMIT, \
	VOUT_UV_FAULT_LIMIT,\
	VOUT_UV_WARN_LIMIT, \
	IOUT_OC_FAULT_LIMIT,\
	IOUT_OC_WARN_LIMIT,\
	TEMP_OT_FAULT_LIMIT,\
	TEMP_OT_WARN_LIMIT,\
	VIN_OV_FAULT_LIMIT,\
	VIN_OV_WARN_LIMIT, \
	VIN_UV_FAULT_LIMIT,\
	VIN_UV_WARN_LIMIT, \
	IIN_OC_FAULT_LIMIT,\
	IIN_OC_WARN_LIMIT,\
	PGOOD_ON_LIMIT,\
	PGOOD_OFF_LIMIT, \
	VIN_ON_LIMIT,\
	VIN_OFF_LIMIT,\
	TON_RISE, \
	VOUT_0_TRAN,\
	DEADTIME1,\
	DEADTIME2,\
	DEADTIME3,\
	DEADTIME4,\
	PERIOD1,\
	TURN_ON_THRESHOLD,\
	TURN_OFF_THRESHOLD,\
	LL_ENABLE,\
	CPCC_PMAX,\
	CPCC_IMAX,\
	CPCC_TON,\
	CPCC_ENABLE,\
	CPCC_TIME_OUT_EN,\
	VFF_CONFIG,\
	IDE_CONFIG,\
	BLANK_TIME,\
	0}

//second rail default value
#define DEFAULT_PMBUS_DCDC_2_CONFIG  {22222, POUT_MAX_ADC, IOUT_MAX_ADC, 15360, 14848, 13312, 33, 30, 300, 0, 0, 5, 20, 1, 0}

typedef struct
{
 Uint16 ot_limit_DCDC_1;
 Uint16 ot_limit_DCDC_2;
 Uint16 ot_limit_DCDC_3;
 Uint16 ot_limit_DCDC_4;
 int16  deadband_config[8];
 Uint16 vin_ov_fault_limit;
 Uint16 vin_uv_fault_limit;
 Uint16 vin_uv_warn_limit;
 char  mfr_date[14];
 char  rom_password[ROM_PASSWORD_BYTES];
}PMBUS_DCDC_CONFIG_NONPAGED; //must be even number int16

extern int16_t PMBus_Vout_Command; 		// Vout command received from DSP via PMBus


//extern PMBUS_DCDC_READING            pmbus_dcdc_reading[DCDC_PAGE_NUMS];
extern PMBUS_DCDC_CONFIG             pmbus_dcdc_config[DCDC_PAGE_NUMS];
extern PMBUS_DCDC_CONFIG             pmbus_dcdc_config_translated[DCDC_PAGE_NUMS];
extern PMBUS_DCDC_CAL                pmbus_dcdc_cal[DCDC_PAGE_NUMS];

extern PMBUS_DCDC_CONFIG_NONPAGED    pmbus_dcdc_config_nonpaged;
extern PMBUS_DCDC_CONFIG_NONPAGED    pmbus_dcdc_config_nonpaged_translated;
extern PMBUS_DCDC_CAL_NONPAGED       pmbus_dcdc_cal_nonpaged;
extern Uint16                        dcdc_temperature_1, dcdc_temperature_2, dcdc_temperature_3, dcdc_temperature_4;
/******************************
 * End of power parameter
 *****************************/

/***********************************************************************
 *Start of MFR message
 ***********************************************************************/
extern const Uint8 setup_id[];
extern const Uint8 mfr_model[];
extern const Uint8 mfr_revision[];
extern const Uint8 mfr_date[];
extern const Uint8 mfr_location[];
extern const Uint8 mfr_serial[];
extern const Uint8 mfr_ic_device[];
extern const Uint8 mfr_ic_device_rev[];
extern const Uint8 mfr_id[];

extern const Uint8  device_id_string[];

/*************************************
 * Start of filter coeffients
 ************************************/

struct FILTER_MISC_REG_BITS{
    Uint32 rsvd0:31;                // 31:1  Reserved
    Uint32 NL_MODE:1;          		// 0     NL_MODE - stored in Filter Control Register
  };

union FILTER_MISC_REG{         // bits   description
    struct FILTER_MISC_REG_BITS bit;
    Uint32                 all;
  };

  //=============================================================================
  // Filter Module Register File
  //=============================================================================
typedef struct  {
    union COEFCONFIG_REG        COEFCONFIG;         // Coefficient Configuration Register
    union FILTERKPCOEF0_REG     FILTERKPCOEF0;      // Filter KP Coefficient 0 Register
    union FILTERKPCOEF1_REG     FILTERKPCOEF1;      // Filter KP Coefficient 1 Register
    union FILTERKICOEF0_REG     FILTERKICOEF0;      // Filter KI Coefficient 0 Register
    union FILTERKICOEF1_REG     FILTERKICOEF1;      // Filter KI Coefficient 1 Register
    union FILTERKDCOEF0_REG     FILTERKDCOEF0;      // Filter KD Coefficient 0 Register
    union FILTERKDCOEF1_REG     FILTERKDCOEF1;      // Filter KD Coefficient 1 Register
    union FILTERKDALPHA_REG     FILTERKDALPHA;      // Filter KD Alpha Register
    union FILTERNL0_REG         FILTERNL0;          // Filter Non-Linear Limit 0 Register
    union FILTERNL1_REG         FILTERNL1;          // Filter Non-Linear Limit 1 Register
    union FILTERNL2_REG         FILTERNL2;          // Filter Non-Linear Limit 2 Register
    union FILTERKICLPHI_REG     FILTERKICLPHI;      // Filter KI Feedback Clamp High Register
    union FILTERKICLPLO_REG     FILTERKICLPLO;      // Filter KI Feedback Clamp Low Register
    union FILTERYNCLPHI_REG     FILTERYNCLPHI;       // Filter Output Clamp High Register
    union FILTERYNCLPLO_REG     FILTERYNCLPLO;       // Filter Output Clamp Low Register
	union FILTEROCLPHI_REG		FILTEROCLPHI;
	union FILTEROCLPLO_REG		FILTEROCLPLO;
	union FILTER_MISC_REG		FILTER_MISC;
  }FILTER_PMBUS_REGS;

#define DEFAULT_PMBUS_DCDC_CONFIG_NONPAGED {81, 82, 83, 84, {1 ,400 ,400 ,400 ,400 , 0, 0, 0}, 55, 32, 38,{'1','1','/','1','2','/','2','0','0','9',0,0,0,0},{'A','B','C','D'}}

#define DEFAULT_PMBUS_DCDC_CAL_NONPAGED    {0, 0}

//FILTER 0 pmbus controlled coefficients
#define INIT_FILTER_0_BIN0_ALPHA (0)           // 3      Bin0 Alpha Configuration
#define INIT_FILTER_0_BIN0_CONFIG (0)          // 2:0    Bin0 Configuration
#define INIT_FILTER_0_BIN1_ALPHA (0)           // 7      Bin1 Alpha Configuration
#define INIT_FILTER_0_BIN1_CONFIG (0)          // 6:4    Bin1 Configuration
#define INIT_FILTER_0_BIN2_ALPHA (1)           // 11     Bin2 Alpha Configuration
#define INIT_FILTER_0_BIN2_CONFIG (1)          // 10:8   Bin2 Configuration
#define INIT_FILTER_0_BIN3_ALPHA (1)           // 15     Bin3 Alpha Configuration
#define INIT_FILTER_0_BIN3_CONFIG (1)          // 14:12  Bin3 Configuration
#define INIT_FILTER_0_BIN4_ALPHA (1)           // 19     Bin4 Alpha Configuration
#define INIT_FILTER_0_BIN4_CONFIG (1)          // 18:16  Bin4 Configuration
#define INIT_FILTER_0_BIN5_ALPHA (1)           // 23     Bin5 Alpha Configuration
#define INIT_FILTER_0_BIN5_CONFIG (1)          // 22:20  Bin5 Configuration
#define INIT_FILTER_0_BIN6_ALPHA (1)           // 27     Bin6 Alpha Configuration
#define INIT_FILTER_0_BIN6_CONFIG (1)          // 26:24  Bin6 Configuration

#define INIT_FILTER_0_KP_COEF_0 (1500)      //(1500)       // 15:0   KP Coefficient 0
#define INIT_FILTER_0_KP_COEF_1 (5000)             // 31:16  KP Coefficient 1
#define INIT_FILTER_0_KP_COEF_2 (10000)             // 15:0   KP Coefficient 2

#define INIT_FILTER_0_KI_COEF_0 (160)   //(90)          // 15:0   KI Coefficient 0
#define INIT_FILTER_0_KI_COEF_1 (250)             // 31:16  KI Coefficient 1
#define INIT_FILTER_0_KI_COEF_2 (500)             // 15:0   KI Coefficient 2
#define INIT_FILTER_0_KI_COEF_3 (500)             // 31:16  KI Coefficient 3

#define INIT_FILTER_0_KD_COEF_0 (2000)             // 15:0   KD Coefficient 0
#define INIT_FILTER_0_KD_COEF_1 (10000)             // 31:16  KD Coefficient 1
#define INIT_FILTER_0_KD_COEF_2 (5500)             // 15:0   KD Coefficient 2

#define INIT_FILTER_0_KD_ALPHA_0 (150)             // 24:16  KD Alpha 1 Value
#define INIT_FILTER_0_KD_ALPHA_1 (25)             // 8:0    KD Alpha 0 Value

#define INIT_FILTER_0_LIMIT0 (0)              // 8:0    Non-Linear Limit 0 Value
#define INIT_FILTER_0_LIMIT1 (25)              // 24:16  Non-Linear Limit 1 Value
#define INIT_FILTER_0_LIMIT2 (25)              // 8:0    Non-Linear Limit 2 Value
#define INIT_FILTER_0_LIMIT3 (25)              // 24:16  Non-Linear Limit 3 Value
#define INIT_FILTER_0_LIMIT4 (31)              // 8:0    Non-Linear Limit 4 Value
#define INIT_FILTER_0_LIMIT5 (31)              // 24:16  Non-Linear Limit 5 Value


#define INIT_FILTER_0_KI_CLAMP_HIGH (0x4AFFFF)       // 23:0   KI Clamp High
#define INIT_FILTER_0_KI_CLAMP_LOW  (0xD99999)       // 23:0   KI Clamp Low

#define INIT_FILTER_0_YN_CLAMP_HIGH (0x7FFFFF)
#define INIT_FILTER_0_YN_CLAMP_LOW  (0xD99999)

#define INIT_FILTER_0_OUTPUT_CLAMP_HIGH (0x2500)//
#define INIT_FILTER_0_OUTPUT_CLAMP_LOW  (0)

#define INIT_FILTER_0_NL_MODE (1)                // written to FILTERCTRL

//FILTER 1 pmbus controlled coefficients
#define INIT_FILTER_1_BIN0_ALPHA  (0)           // 3      Bin0 Alpha Configuration
#define INIT_FILTER_1_BIN0_CONFIG (0)          // 2:0    Bin0 Configuration
#define INIT_FILTER_1_BIN1_ALPHA  (0)           // 7      Bin1 Alpha Configuration
#define INIT_FILTER_1_BIN1_CONFIG (0)          // 6:4    Bin1 Configuration
#define INIT_FILTER_1_BIN2_ALPHA  (0)           // 11     Bin2 Alpha Configuration
#define INIT_FILTER_1_BIN2_CONFIG (0)          // 10:8   Bin2 Configuration
#define INIT_FILTER_1_BIN3_ALPHA  (0)           // 15     Bin3 Alpha Configuration
#define INIT_FILTER_1_BIN3_CONFIG (0)          // 14:12  Bin3 Configuration
#define INIT_FILTER_1_BIN4_ALPHA  (0)           // 19     Bin4 Alpha Configuration
#define INIT_FILTER_1_BIN4_CONFIG (0)          // 18:16  Bin4 Configuration
#define INIT_FILTER_1_BIN5_ALPHA  (0)           // 23     Bin5 Alpha Configuration
#define INIT_FILTER_1_BIN5_CONFIG (0)          // 22:20  Bin5 Configuration
#define INIT_FILTER_1_BIN6_ALPHA  (0)           // 27     Bin6 Alpha Configuration
#define INIT_FILTER_1_BIN6_CONFIG (0)          // 26:24  Bin6 Configuration

#define INIT_FILTER_1_KP_COEF_0 (100) //(13000)             // 15:0   KP Coefficient 0
#define INIT_FILTER_1_KP_COEF_1 (0)             // 31:16  KP Coefficient 1
#define INIT_FILTER_1_KP_COEF_2 (0)             // 15:0   KP Coefficient 2

#define INIT_FILTER_1_KI_COEF_0 (10) //(168)             // 15:0   KI Coefficient 0
#define INIT_FILTER_1_KI_COEF_1 (0)             // 31:16  KI Coefficient 1
#define INIT_FILTER_1_KI_COEF_2 (0)             // 15:0   KI Coefficient 2
#define INIT_FILTER_1_KI_COEF_3 (0)             // 31:16  KI Coefficient 3

#define INIT_FILTER_1_KD_COEF_0 (0)             // 15:0   KD Coefficient 0
#define INIT_FILTER_1_KD_COEF_1 (0)             // 31:16  KD Coefficient 1
#define INIT_FILTER_1_KD_COEF_2 (0)             // 15:0   KD Coefficient 2

#define INIT_FILTER_1_KD_ALPHA_0 (0)             // 24:16  KD Alpha 1 Value
#define INIT_FILTER_1_KD_ALPHA_1 (0)             // 8:0    KD Alpha 0 Value

#define INIT_FILTER_1_LIMIT0 (0x0)              // 8:0    Non-Linear Limit 0 Value
#define INIT_FILTER_1_LIMIT1 (0x0)              // 24:16  Non-Linear Limit 1 Value
#define INIT_FILTER_1_LIMIT2 (0x0)              // 8:0    Non-Linear Limit 2 Value
#define INIT_FILTER_1_LIMIT3 (0x0)              // 24:16  Non-Linear Limit 3 Value
#define INIT_FILTER_1_LIMIT4 (0x0)              // 8:0    Non-Linear Limit 4 Value
#define INIT_FILTER_1_LIMIT5 (0x0)              // 24:16  Non-Linear Limit 5 Value


#define INIT_FILTER_1_KI_CLAMP_HIGH (0x4FFFFF)       // 23:0   KI Clamp High
#define INIT_FILTER_1_KI_CLAMP_LOW  (0xD00000)       // 23:0   KI Clamp Low

#define INIT_FILTER_1_YN_CLAMP_HIGH (0x7FFFFF)
#define INIT_FILTER_1_YN_CLAMP_LOW  (0x0)

#define INIT_FILTER_1_OUTPUT_CLAMP_HIGH (0x2500)       // 23:0   Output Clamp High
#define INIT_FILTER_1_OUTPUT_CLAMP_LOW   (0)         // 23:0   Output Clamp High

#define INIT_FILTER_1_NL_MODE (1)       // written to FILTERCTRL



//FILTER 2 pmbus controlled coefficients
#define INIT_FILTER_2_BIN0_ALPHA  (1)           // 3      Bin0 Alpha Configuration
#define INIT_FILTER_2_BIN0_CONFIG (0)          // 2:0    Bin0 Configuration
#define INIT_FILTER_2_BIN1_ALPHA  (1)           // 7      Bin1 Alpha Configuration
#define INIT_FILTER_2_BIN1_CONFIG (2)          // 6:4    Bin1 Configuration
#define INIT_FILTER_2_BIN2_ALPHA  (1)           // 11     Bin2 Alpha Configuration
#define INIT_FILTER_2_BIN2_CONFIG (2)          // 10:8   Bin2 Configuration
#define INIT_FILTER_2_BIN3_ALPHA  (1)           // 15     Bin3 Alpha Configuration
#define INIT_FILTER_2_BIN3_CONFIG (2)          // 14:12  Bin3 Configuration
#define INIT_FILTER_2_BIN4_ALPHA  (1)           // 19     Bin4 Alpha Configuration
#define INIT_FILTER_2_BIN4_CONFIG (1)          // 18:16  Bin4 Configuration
#define INIT_FILTER_2_BIN5_ALPHA  (1)           // 23     Bin5 Alpha Configuration
#define INIT_FILTER_2_BIN5_CONFIG (1)          // 22:20  Bin5 Configuration
#define INIT_FILTER_2_BIN6_ALPHA  (1)           // 27     Bin6 Alpha Configuration
#define INIT_FILTER_2_BIN6_CONFIG (1)          // 26:24  Bin6 Configuration

#define INIT_FILTER_2_KP_COEF_0 (0)             // 15:0   KP Coefficient 0
#define INIT_FILTER_2_KP_COEF_1 (8000)             // 31:16  KP Coefficient 1   //12000
#define INIT_FILTER_2_KP_COEF_2 (10000)             // 15:0   KP Coefficient 2 //8000

#define INIT_FILTER_2_KI_COEF_0 (0)             // 15:0   KI Coefficient 0
#define INIT_FILTER_2_KI_COEF_1 (0)             // 31:16  KI Coefficient 1
#define INIT_FILTER_2_KI_COEF_2 (0)             // 15:0   KI Coefficient 2
#define INIT_FILTER_2_KI_COEF_3 (0)             // 31:16  KI Coefficient 3

#define INIT_FILTER_2_KD_COEF_0 (0)             // 15:0   KD Coefficient 0
#define INIT_FILTER_2_KD_COEF_1 (0)             // 31:16  KD Coefficient 1
#define INIT_FILTER_2_KD_COEF_2 (0)             // 15:0   KD Coefficient 2

#define INIT_FILTER_2_KD_ALPHA_0 (0)             // 24:16  KD Alpha 1 Value
#define INIT_FILTER_2_KD_ALPHA_1 (0)             // 8:0    KD Alpha 0 Value

#define INIT_FILTER_2_LIMIT0 (96)              // 8:0    Non-Linear Limit 0 Value
#define INIT_FILTER_2_LIMIT1 (120)              // 24:16  Non-Linear Limit 1 Value
#define INIT_FILTER_2_LIMIT2 (176)              // 8:0    Non-Linear Limit 2 Value
#define INIT_FILTER_2_LIMIT3 (216)              // 24:16  Non-Linear Limit 3 Value
#define INIT_FILTER_2_LIMIT4 (240)              // 8:0    Non-Linear Limit 4 Value
#define INIT_FILTER_2_LIMIT5 (248)              // 24:16  Non-Linear Limit 5 Value


#define INIT_FILTER_2_KI_CLAMP_HIGH (0x7FFFFF)       // 23:0   KI Clamp High
#define INIT_FILTER_2_KI_CLAMP_LOW  (0)            // 23:0   KI Clamp Low


#define INIT_FILTER_2_YN_CLAMP_HIGH (0x7FFFFF)
#define INIT_FILTER_2_YN_CLAMP_LOW  (0xD99999)

#define INIT_FILTER_2_OUTPUT_CLAMP_HIGH (PERIOD1)       // 23:0   Output Clamp High    20000 is the Period  for 200Khz
#define INIT_FILTER_2_OUTPUT_CLAMP_LOW  (PERIOD1 * 0.4)       // 23:0   Output Clamp High     20000 is the Period for 200Khz

#define INIT_FILTER_2_NL_MODE (1)       // written to FILTERCTRL


#define DEFAULT_FILTER_0_PMBUS_REGS \
{\
	{\
		0,\
	    INIT_FILTER_0_BIN6_ALPHA,\
	    INIT_FILTER_0_BIN6_CONFIG,\
	    INIT_FILTER_0_BIN5_ALPHA,\
	    INIT_FILTER_0_BIN5_CONFIG,\
	    INIT_FILTER_0_BIN4_ALPHA,\
	    INIT_FILTER_0_BIN4_CONFIG,\
	    INIT_FILTER_0_BIN3_ALPHA,\
	    INIT_FILTER_0_BIN3_CONFIG,\
	    INIT_FILTER_0_BIN2_ALPHA,\
	    INIT_FILTER_0_BIN2_CONFIG,\
	    INIT_FILTER_0_BIN1_ALPHA,\
	    INIT_FILTER_0_BIN1_CONFIG,\
	    INIT_FILTER_0_BIN0_ALPHA,\
	    INIT_FILTER_0_BIN0_CONFIG\
	},\
	{\
		INIT_FILTER_0_KP_COEF_1,\
	    INIT_FILTER_0_KP_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_0_KP_COEF_2\
	},\
	{\
		INIT_FILTER_0_KI_COEF_1,\
	    INIT_FILTER_0_KI_COEF_0\
	},\
	{\
		INIT_FILTER_0_KI_COEF_3,\
	    INIT_FILTER_0_KI_COEF_2\
	},\
	{\
		INIT_FILTER_0_KD_COEF_1,\
	    INIT_FILTER_0_KD_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_0_KD_COEF_2\
	},\
	{\
		0,\
		INIT_FILTER_0_KD_ALPHA_1,\
		0,\
	    INIT_FILTER_0_KD_ALPHA_0\
	},\
	{\
		0,\
		INIT_FILTER_0_LIMIT1,\
		0,\
	    INIT_FILTER_0_LIMIT0\
	},\
	{\
		0,\
		INIT_FILTER_0_LIMIT3,\
		0,\
	    INIT_FILTER_0_LIMIT2\
	},\
	{\
		0,\
		INIT_FILTER_0_LIMIT5,\
		0,\
	    INIT_FILTER_0_LIMIT4\
	},\
	{\
		0,\
	    INIT_FILTER_0_KI_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_0_KI_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_0_YN_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_0_YN_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_0_OUTPUT_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_0_OUTPUT_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_0_NL_MODE\
	},\
}

#define DEFAULT_FILTER_1_PMBUS_REGS \
{\
	{\
		0,\
	    INIT_FILTER_1_BIN6_ALPHA,\
	    INIT_FILTER_1_BIN6_CONFIG,\
	    INIT_FILTER_1_BIN5_ALPHA,\
	    INIT_FILTER_1_BIN5_CONFIG,\
	    INIT_FILTER_1_BIN4_ALPHA,\
	    INIT_FILTER_1_BIN4_CONFIG,\
	    INIT_FILTER_1_BIN3_ALPHA,\
	    INIT_FILTER_1_BIN3_CONFIG,\
	    INIT_FILTER_1_BIN2_ALPHA,\
	    INIT_FILTER_1_BIN2_CONFIG,\
	    INIT_FILTER_1_BIN1_ALPHA,\
	    INIT_FILTER_1_BIN1_CONFIG,\
	    INIT_FILTER_1_BIN0_ALPHA,\
	    INIT_FILTER_1_BIN0_CONFIG\
	},\
	{\
		INIT_FILTER_1_KP_COEF_1,\
	    INIT_FILTER_1_KP_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_1_KP_COEF_2\
	},\
	{\
		INIT_FILTER_1_KI_COEF_1,\
	    INIT_FILTER_1_KI_COEF_0\
	},\
	{\
		INIT_FILTER_1_KI_COEF_3,\
	    INIT_FILTER_1_KI_COEF_2\
	},\
	{\
		INIT_FILTER_1_KD_COEF_1,\
	    INIT_FILTER_1_KD_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_1_KD_COEF_2\
	},\
	{\
		0,\
		INIT_FILTER_1_KD_ALPHA_1,\
		0,\
	    INIT_FILTER_1_KD_ALPHA_0\
	},\
	{\
		0,\
		INIT_FILTER_1_LIMIT1,\
		0,\
	    INIT_FILTER_1_LIMIT0\
	},\
	{\
		0,\
		INIT_FILTER_1_LIMIT3,\
		0,\
	    INIT_FILTER_1_LIMIT2\
	},\
	{\
		0,\
		INIT_FILTER_1_LIMIT5,\
		0,\
	    INIT_FILTER_1_LIMIT4\
	},\
	{\
		0,\
	    INIT_FILTER_1_KI_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_1_KI_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_1_YN_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_1_YN_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_1_OUTPUT_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_1_OUTPUT_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_1_NL_MODE\
	},\
}


#define DEFAULT_FILTER_2_PMBUS_REGS \
{\
	{\
		0,\
	    INIT_FILTER_2_BIN6_ALPHA,\
	    INIT_FILTER_2_BIN6_CONFIG,\
	    INIT_FILTER_2_BIN5_ALPHA,\
	    INIT_FILTER_2_BIN5_CONFIG,\
	    INIT_FILTER_2_BIN4_ALPHA,\
	    INIT_FILTER_2_BIN4_CONFIG,\
	    INIT_FILTER_2_BIN3_ALPHA,\
	    INIT_FILTER_2_BIN3_CONFIG,\
	    INIT_FILTER_2_BIN2_ALPHA,\
	    INIT_FILTER_2_BIN2_CONFIG,\
	    INIT_FILTER_2_BIN1_ALPHA,\
	    INIT_FILTER_2_BIN1_CONFIG,\
	    INIT_FILTER_2_BIN0_ALPHA,\
	    INIT_FILTER_2_BIN0_CONFIG\
	},\
	{\
		INIT_FILTER_2_KP_COEF_1,\
	    INIT_FILTER_2_KP_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_2_KP_COEF_2\
	},\
	{\
		INIT_FILTER_2_KI_COEF_1,\
	    INIT_FILTER_2_KI_COEF_0\
	},\
	{\
		INIT_FILTER_2_KI_COEF_3,\
	    INIT_FILTER_2_KI_COEF_2\
	},\
	{\
		INIT_FILTER_2_KD_COEF_1,\
	    INIT_FILTER_2_KD_COEF_0\
	},\
	{\
		0,\
	    INIT_FILTER_2_KD_COEF_2\
	},\
	{\
		0,\
		INIT_FILTER_2_KD_ALPHA_1,\
		0,\
	    INIT_FILTER_2_KD_ALPHA_0\
	},\
	{\
		0,\
		INIT_FILTER_2_LIMIT1,\
		0,\
	    INIT_FILTER_2_LIMIT0\
	},\
	{\
		0,\
		INIT_FILTER_2_LIMIT3,\
		0,\
	    INIT_FILTER_2_LIMIT2\
	},\
	{\
		0,\
		INIT_FILTER_2_LIMIT5,\
		0,\
	    INIT_FILTER_2_LIMIT4\
	},\
	{\
		0,\
	    INIT_FILTER_2_KI_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_2_KI_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_2_YN_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_2_YN_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_2_OUTPUT_CLAMP_HIGH\
	},\
	{\
		0,\
	    INIT_FILTER_2_OUTPUT_CLAMP_LOW\
	},\
	{\
		0,\
	    INIT_FILTER_2_NL_MODE\
	},\
}

extern FILTER_PMBUS_REGS filter0_pmbus_regs;
extern FILTER_PMBUS_REGS filter1_pmbus_regs;
extern FILTER_PMBUS_REGS filter2_pmbus_regs;

void copy_coefficients_to_filter(volatile struct FILTER_REGS *dest, const FILTER_PMBUS_REGS *source);
void copy_coefficients_to_ram(volatile FILTER_PMBUS_REGS *dest, volatile struct FILTER_REGS *source);

Uint8 pmbus_write_PSON(Uint8 read);
Uint8 pmbus_read_fault_bytes(Uint8 read);
Uint8 pmbus_read_major_rev(Uint8 read);
Uint8 pmbus_read_minor_rev(Uint8 read);
Uint8 pmbus_read_build_rev(Uint8 read);
Uint8 pmbus_read_write_vout_command(Uint8 pmbus_read);
Uint8 pmbus_read_write_vin_ov_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_vin_uv_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_ioutSetPoint(Uint8 pmbus_read);

Uint8 pmbus_read_write_hwversion(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout_offset(Uint8 pmbus_read);
Uint8 pmbus_read_write_ibat_offset(Uint8 pmbus_read);
Uint8 pmbus_read_write_iout_offset(Uint8 pmbus_read);
Uint8 pmbus_read_write_dac_ibatt_offset(Uint8 pmbus_read);
Uint8 pmbus_read_vout_value(Uint8 read);
Uint8 pmbus_read_ibat_value(Uint8 read);
Uint8 pmbus_read_iout_value(Uint8 read);
Uint8 pmbus_read_dac_ibatt_value(Uint8 read);

Uint8 pmbus_read_write_clear_faults(Uint8 pmbus_read);
Uint8 pmbus_read_write_store_default_all(Uint8 pmbus_read);
Uint8 pmbus_read_write_restore_default_all(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout_mode(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout_transition_rate(Uint8 pmbus_read);
Uint8 pmbus_read_write_frequency_switch(Uint8 pmbus_read);
Uint8 pmbus_read_write_vin_on_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_vin_off_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout_ov_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_iout_oc_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_temp_ot_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_iin_oc_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_ton_rise(Uint8 pmbus_read);
Uint8 pmbus_read_write_status_bit_mask(Uint8 pmbus_read);
Uint8 pmbus_read_write_status_byte(Uint8 pmbus_read);
Uint8 pmbus_read_write_status_word(Uint8 pmbus_read);
Uint8 pmbus_read_write_dcdc_op_state(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout(Uint8 pmbus_read);
Uint8 pmbus_read_write_temp_mosfet(Uint8 pmbus_read);
Uint8 pmbus_read_write_temp_device(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_ic_device(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_ic_device_rev(Uint8 pmbus_read);
Uint8 pmbus_read_write_deadband_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_vff_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_blank_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_ide_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_vout_uv_fault_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_pgood_on_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_pgood_off_limit(Uint8 pmbus_read);
Uint8 pmbus_read_write_vin(Uint8 pmbus_read);
Uint8 pmbus_read_temp_device(Uint8 read);
Uint8 pmbus_read_write_frequency(Uint8 pmbus_read);
Uint8 pmbus_read_write_mode_switching_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_light_load_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_cpcc_config(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_setup_id(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_revision(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_model(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_date(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_location(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_serial(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_ic_device(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_ic_device_rev(Uint8 pmbus_read);
Uint8 pmbus_read_write_mfr_id(Uint8 pmbus_read);
Uint8 pmbus_read_write_device_id(Uint8 pmbus_read);
Uint8 pmbus_read_write_cmd_dcdc_nonpaged(Uint8 pmbus_read);
Uint8 pmbus_read_write_cmd_dcdc_paged(Uint8 pmbus_read);


#define PMBUS_CMD_01_FUNCTION_POINTER pmbus_write_PSON
#define PMBUS_CMD_01_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)


#define PMBUS_CMD_02_FUNCTION_POINTER pmbus_read_fault_bytes
#define PMBUS_CMD_02_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

/*
#define PMBUS_CMD_03_FUNCTION_POINTER pmbus_read_write_clear_faults
#define PMBUS_CMD_03_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)
*/

#define PMBUS_CMD_04_FUNCTION_POINTER pmbus_read_major_rev
#define PMBUS_CMD_04_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_05_FUNCTION_POINTER pmbus_read_minor_rev
#define PMBUS_CMD_05_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_06_FUNCTION_POINTER pmbus_read_build_rev
#define PMBUS_CMD_06_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)


#define PMBUS_CMD_11_FUNCTION_POINTER pmbus_read_write_store_default_all
#define PMBUS_CMD_11_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_12_FUNCTION_POINTER pmbus_read_write_restore_default_all
#define PMBUS_CMD_12_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_20_FUNCTION_POINTER pmbus_read_write_vout_mode
#define PMBUS_CMD_20_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_21_FUNCTION_POINTER pmbus_read_write_vout_command
#define PMBUS_CMD_21_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_27_FUNCTION_POINTER pmbus_read_write_vout_transition_rate
#define PMBUS_CMD_27_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_33_FUNCTION_POINTER pmbus_read_write_frequency_switch
#define PMBUS_CMD_33_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_35_FUNCTION_POINTER pmbus_read_write_vin_on_limit
#define PMBUS_CMD_35_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_36_FUNCTION_POINTER pmbus_read_write_vin_off_limit
#define PMBUS_CMD_36_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_40_FUNCTION_POINTER pmbus_read_write_vout_ov_fault_limit
#define PMBUS_CMD_40_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_44_FUNCTION_POINTER pmbus_read_write_vout_uv_fault_limit
#define PMBUS_CMD_44_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_46_FUNCTION_POINTER pmbus_read_write_iout_oc_fault_limit
#define PMBUS_CMD_46_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_4F_FUNCTION_POINTER pmbus_read_write_temp_ot_fault_limit
#define PMBUS_CMD_4F_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_55_FUNCTION_POINTER pmbus_read_write_vin_ov_fault_limit
#define PMBUS_CMD_55_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_59_FUNCTION_POINTER pmbus_read_write_vin_uv_fault_limit
#define PMBUS_CMD_59_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_5E_FUNCTION_POINTER pmbus_read_write_pgood_on_limit
#define PMBUS_CMD_5E_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_5F_FUNCTION_POINTER pmbus_read_write_pgood_off_limit
#define PMBUS_CMD_5F_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_5B_FUNCTION_POINTER pmbus_read_write_iin_oc_fault_limit
#define PMBUS_CMD_5B_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_61_FUNCTION_POINTER pmbus_read_write_ton_rise
#define PMBUS_CMD_61_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_78_FUNCTION_POINTER pmbus_read_write_status_byte
#define PMBUS_CMD_78_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_79_FUNCTION_POINTER pmbus_read_write_status_word
#define PMBUS_CMD_79_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_80_FUNCTION_POINTER pmbus_read_write_dcdc_op_state
#define PMBUS_CMD_80_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_88_FUNCTION_POINTER pmbus_read_write_vin
#define PMBUS_CMD_88_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_8B_FUNCTION_POINTER pmbus_read_write_vout
#define PMBUS_CMD_8B_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_8C_FUNCTION_POINTER pmbus_read_write_ioutSetPoint
#define PMBUS_CMD_8C_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_8D_FUNCTION_POINTER pmbus_read_write_temp_mosfet
#define PMBUS_CMD_8D_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_8E_FUNCTION_POINTER pmbus_read_temp_device
#define PMBUS_CMD_8E_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_95_FUNCTION_POINTER pmbus_read_write_frequency
#define PMBUS_CMD_95_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_LINEAR)

#define PMBUS_CMD_99_FUNCTION_POINTER pmbus_read_write_mfr_id
#define PMBUS_CMD_99_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_9A_FUNCTION_POINTER pmbus_read_write_mfr_model
#define PMBUS_CMD_9A_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_9B_FUNCTION_POINTER pmbus_read_write_mfr_revision
#define PMBUS_CMD_9B_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_9C_FUNCTION_POINTER pmbus_read_write_mfr_location
#define PMBUS_CMD_9C_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_9D_FUNCTION_POINTER pmbus_read_write_mfr_date
#define PMBUS_CMD_9D_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_9E_FUNCTION_POINTER pmbus_read_write_mfr_serial
#define PMBUS_CMD_9E_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_AD_FUNCTION_POINTER pmbus_read_write_mfr_ic_device
#define PMBUS_CMD_AD_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_AE_FUNCTION_POINTER pmbus_read_write_mfr_ic_device_rev
#define PMBUS_CMD_AE_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_B1_FUNCTION_POINTER pmbus_read_vout_value
#define PMBUS_CMD_B1_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_B2_FUNCTION_POINTER pmbus_read_ibat_value
#define PMBUS_CMD_B2_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_B3_FUNCTION_POINTER pmbus_read_iout_value
#define PMBUS_CMD_B3_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_B5_FUNCTION_POINTER pmbus_read_dac_ibatt_value
#define PMBUS_CMD_B5_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_B8_FUNCTION_POINTER pmbus_read_write_hwversion
#define PMBUS_CMD_B8_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_D1_FUNCTION_POINTER pmbus_read_write_vout_offset
#define PMBUS_CMD_D1_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_D2_FUNCTION_POINTER pmbus_read_write_ibat_offset
#define PMBUS_CMD_D2_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_D3_FUNCTION_POINTER pmbus_read_write_iout_offset
#define PMBUS_CMD_D3_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_D5_FUNCTION_POINTER pmbus_read_write_dac_ibatt_offset
#define PMBUS_CMD_D5_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_E4_FUNCTION_POINTER pmbus_read_write_cmd_dcdc_paged
#define PMBUS_CMD_E4_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_E5_FUNCTION_POINTER pmbus_read_write_cmd_dcdc_nonpaged
#define PMBUS_CMD_E5_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_E7_FUNCTION_POINTER pmbus_read_write_mfr_setup_id
#define PMBUS_CMD_E7_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_EA_FUNCTION_POINTER pmbus_read_write_deadband_config
#define PMBUS_CMD_EA_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_F2_FUNCTION_POINTER pmbus_read_write_status_bit_mask
#define PMBUS_CMD_F2_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_F4_FUNCTION_POINTER pmbus_read_write_cpcc_config
#define PMBUS_CMD_F4_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_FA_FUNCTION_POINTER pmbus_read_write_vff_config
#define PMBUS_CMD_FA_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_FC_FUNCTION_POINTER pmbus_read_write_blank_config
#define PMBUS_CMD_FC_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define PMBUS_CMD_FD_FUNCTION_POINTER pmbus_read_write_device_id
#define PMBUS_CMD_FD_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_NON_NUMERIC)

#define PMBUS_CMD_FE_FUNCTION_POINTER pmbus_read_write_ide_config
#define PMBUS_CMD_FE_QUERY (QUERY_COMMAND_SUPPORTED + QUERY_WRITE_SUPPORTED + QUERY_READ_SUPPORTED + QUERY_MAN_SPEC)

#define	PMBUS_CMD_00_ENABLE	(0)
#define	PMBUS_CMD_01_ENABLE	(1)
#define	PMBUS_CMD_02_ENABLE	(1)
#define	PMBUS_CMD_03_ENABLE	(0)
#define	PMBUS_CMD_04_ENABLE	(1)
#define	PMBUS_CMD_05_ENABLE	(1)
#define	PMBUS_CMD_06_ENABLE	(1)
#define	PMBUS_CMD_07_ENABLE	(0)
#define	PMBUS_CMD_08_ENABLE	(0)
#define	PMBUS_CMD_09_ENABLE	(0)
#define	PMBUS_CMD_0A_ENABLE	(0)
#define	PMBUS_CMD_0B_ENABLE	(0)
#define	PMBUS_CMD_0C_ENABLE	(0)
#define	PMBUS_CMD_0D_ENABLE	(0)
#define	PMBUS_CMD_0E_ENABLE	(0)
#define	PMBUS_CMD_0F_ENABLE	(0)
#define	PMBUS_CMD_10_ENABLE	(0)
#define	PMBUS_CMD_11_ENABLE	(0)
#define	PMBUS_CMD_12_ENABLE	(0)
#define	PMBUS_CMD_13_ENABLE	(0)
#define	PMBUS_CMD_14_ENABLE	(0)
#define	PMBUS_CMD_15_ENABLE	(0)
#define	PMBUS_CMD_16_ENABLE	(0)
#define	PMBUS_CMD_17_ENABLE	(0)
#define	PMBUS_CMD_18_ENABLE	(0)
#define	PMBUS_CMD_19_ENABLE	(0)
#define	PMBUS_CMD_1A_ENABLE	(0)
#define	PMBUS_CMD_1B_ENABLE	(0)
#define	PMBUS_CMD_1C_ENABLE	(0)
#define	PMBUS_CMD_1D_ENABLE	(0)
#define	PMBUS_CMD_1E_ENABLE	(0)
#define	PMBUS_CMD_1F_ENABLE	(0)
#define	PMBUS_CMD_20_ENABLE	(1)
#define	PMBUS_CMD_21_ENABLE	(1)
#define	PMBUS_CMD_22_ENABLE	(0)
#define	PMBUS_CMD_23_ENABLE	(0)
#define	PMBUS_CMD_24_ENABLE	(0)
#define	PMBUS_CMD_25_ENABLE	(0)
#define	PMBUS_CMD_26_ENABLE	(0)
#define	PMBUS_CMD_27_ENABLE	(0)
#define	PMBUS_CMD_28_ENABLE	(0)
#define	PMBUS_CMD_29_ENABLE	(0)
#define	PMBUS_CMD_2A_ENABLE	(0)
#define	PMBUS_CMD_2B_ENABLE	(0)
#define	PMBUS_CMD_2C_ENABLE	(0)
#define	PMBUS_CMD_2D_ENABLE	(0)
#define	PMBUS_CMD_2E_ENABLE	(0)
#define	PMBUS_CMD_2F_ENABLE	(0)
#define	PMBUS_CMD_30_ENABLE	(0)
#define	PMBUS_CMD_31_ENABLE	(0)
#define	PMBUS_CMD_32_ENABLE	(0)
#define	PMBUS_CMD_33_ENABLE	(0)
#define	PMBUS_CMD_34_ENABLE	(0)
#define	PMBUS_CMD_35_ENABLE	(0)
#define	PMBUS_CMD_36_ENABLE	(0)
#define	PMBUS_CMD_37_ENABLE	(0)
#define	PMBUS_CMD_38_ENABLE	(0)
#define	PMBUS_CMD_39_ENABLE	(0)
#define	PMBUS_CMD_3A_ENABLE	(0)
#define	PMBUS_CMD_3B_ENABLE	(0)
#define	PMBUS_CMD_3C_ENABLE	(0)
#define	PMBUS_CMD_3D_ENABLE	(0)
#define	PMBUS_CMD_3E_ENABLE	(0)
#define	PMBUS_CMD_3F_ENABLE	(0)
#define	PMBUS_CMD_40_ENABLE	(0)
#define	PMBUS_CMD_41_ENABLE	(0)
#define	PMBUS_CMD_42_ENABLE	(0)
#define	PMBUS_CMD_43_ENABLE	(0)
#define	PMBUS_CMD_44_ENABLE	(0)
#define	PMBUS_CMD_45_ENABLE	(0)
#define	PMBUS_CMD_46_ENABLE	(0)
#define	PMBUS_CMD_47_ENABLE	(0)
#define	PMBUS_CMD_48_ENABLE	(0)
#define	PMBUS_CMD_49_ENABLE	(0)
#define	PMBUS_CMD_4A_ENABLE	(0)
#define	PMBUS_CMD_4B_ENABLE	(0)
#define	PMBUS_CMD_4C_ENABLE	(0)
#define	PMBUS_CMD_4D_ENABLE	(0)
#define	PMBUS_CMD_4E_ENABLE	(0)
#define	PMBUS_CMD_4F_ENABLE	(0)
#define	PMBUS_CMD_50_ENABLE	(0)
#define	PMBUS_CMD_51_ENABLE	(0)
#define	PMBUS_CMD_52_ENABLE	(0)
#define	PMBUS_CMD_53_ENABLE	(0)
#define	PMBUS_CMD_54_ENABLE	(0)
#define	PMBUS_CMD_55_ENABLE	(1)
#define	PMBUS_CMD_56_ENABLE	(0)
#define	PMBUS_CMD_57_ENABLE	(0)
#define	PMBUS_CMD_58_ENABLE	(0)
#define	PMBUS_CMD_59_ENABLE	(1)
#define	PMBUS_CMD_5A_ENABLE	(0)
#define	PMBUS_CMD_5B_ENABLE	(0)
#define	PMBUS_CMD_5C_ENABLE	(0)
#define	PMBUS_CMD_5D_ENABLE	(0)
#define	PMBUS_CMD_5E_ENABLE	(0)
#define	PMBUS_CMD_5F_ENABLE	(0)
#define	PMBUS_CMD_60_ENABLE	(0)
#define	PMBUS_CMD_61_ENABLE	(0)
#define	PMBUS_CMD_62_ENABLE	(0)
#define	PMBUS_CMD_63_ENABLE	(0)
#define	PMBUS_CMD_64_ENABLE	(0)
#define	PMBUS_CMD_65_ENABLE	(0)
#define	PMBUS_CMD_66_ENABLE	(0)
#define	PMBUS_CMD_67_ENABLE	(0)
#define	PMBUS_CMD_68_ENABLE	(0)
#define	PMBUS_CMD_69_ENABLE	(0)
#define	PMBUS_CMD_6A_ENABLE	(0)
#define	PMBUS_CMD_6B_ENABLE	(0)
#define	PMBUS_CMD_6C_ENABLE	(0)
#define	PMBUS_CMD_6D_ENABLE	(0)
#define	PMBUS_CMD_6E_ENABLE	(0)
#define	PMBUS_CMD_6F_ENABLE	(0)
#define	PMBUS_CMD_70_ENABLE	(0)
#define	PMBUS_CMD_71_ENABLE	(0)
#define	PMBUS_CMD_72_ENABLE	(0)
#define	PMBUS_CMD_73_ENABLE	(0)
#define	PMBUS_CMD_74_ENABLE	(0)
#define	PMBUS_CMD_75_ENABLE	(0)
#define	PMBUS_CMD_76_ENABLE	(0)
#define	PMBUS_CMD_77_ENABLE	(0)
#define	PMBUS_CMD_78_ENABLE	(1)
#define	PMBUS_CMD_79_ENABLE	(1)
#define	PMBUS_CMD_7A_ENABLE	(0)
#define	PMBUS_CMD_7B_ENABLE	(0)
#define	PMBUS_CMD_7C_ENABLE	(0)
#define	PMBUS_CMD_7D_ENABLE	(0)
#define	PMBUS_CMD_7E_ENABLE	(0)
#define	PMBUS_CMD_7F_ENABLE	(0)
#define	PMBUS_CMD_80_ENABLE	(1)
#define	PMBUS_CMD_81_ENABLE	(0)
#define	PMBUS_CMD_82_ENABLE	(0)
#define	PMBUS_CMD_83_ENABLE	(0)
#define	PMBUS_CMD_84_ENABLE	(0)
#define	PMBUS_CMD_85_ENABLE	(0)
#define	PMBUS_CMD_86_ENABLE	(0)
#define	PMBUS_CMD_87_ENABLE	(0)
#define	PMBUS_CMD_88_ENABLE	(0)
#define	PMBUS_CMD_89_ENABLE	(0)
#define	PMBUS_CMD_8A_ENABLE	(0)
#define	PMBUS_CMD_8B_ENABLE	(0)
#define	PMBUS_CMD_8C_ENABLE	(1)
#define	PMBUS_CMD_8D_ENABLE	(0)
#define	PMBUS_CMD_8E_ENABLE	(1)
#define	PMBUS_CMD_8F_ENABLE	(0)
#define	PMBUS_CMD_90_ENABLE	(0)
#define	PMBUS_CMD_91_ENABLE	(0)
#define	PMBUS_CMD_92_ENABLE	(0)
#define	PMBUS_CMD_93_ENABLE	(0)
#define	PMBUS_CMD_94_ENABLE	(0)
#define	PMBUS_CMD_95_ENABLE	(0)
#define	PMBUS_CMD_96_ENABLE	(0)
#define	PMBUS_CMD_97_ENABLE	(0)
#define	PMBUS_CMD_98_ENABLE	(1)
#define	PMBUS_CMD_99_ENABLE	(1)
#define	PMBUS_CMD_9A_ENABLE	(1)
#define	PMBUS_CMD_9B_ENABLE	(1)
#define	PMBUS_CMD_9C_ENABLE	(1)
#define	PMBUS_CMD_9D_ENABLE	(1)
#define	PMBUS_CMD_9E_ENABLE	(1)
#define	PMBUS_CMD_9F_ENABLE	(0)
#define	PMBUS_CMD_A0_ENABLE	(0)
#define	PMBUS_CMD_A1_ENABLE	(0)
#define	PMBUS_CMD_A2_ENABLE	(0)
#define	PMBUS_CMD_A3_ENABLE	(0)
#define	PMBUS_CMD_A4_ENABLE	(0)
#define	PMBUS_CMD_A5_ENABLE	(0)
#define	PMBUS_CMD_A6_ENABLE	(0)
#define	PMBUS_CMD_A7_ENABLE	(0)
#define	PMBUS_CMD_A8_ENABLE	(0)
#define	PMBUS_CMD_A9_ENABLE	(0)
#define	PMBUS_CMD_AA_ENABLE	(0)
#define	PMBUS_CMD_AB_ENABLE	(0)
#define	PMBUS_CMD_AC_ENABLE	(0)
#define	PMBUS_CMD_AD_ENABLE	(1)
#define	PMBUS_CMD_AE_ENABLE	(1)
#define	PMBUS_CMD_AF_ENABLE	(0)
#define	PMBUS_CMD_B0_ENABLE	(0)
#define	PMBUS_CMD_B1_ENABLE	(1)
#define	PMBUS_CMD_B2_ENABLE	(1)
#define	PMBUS_CMD_B3_ENABLE	(1)
#define	PMBUS_CMD_B4_ENABLE	(0)
#define	PMBUS_CMD_B5_ENABLE	(0)
#define	PMBUS_CMD_B6_ENABLE	(0)
#define	PMBUS_CMD_B7_ENABLE	(0)
#define	PMBUS_CMD_B8_ENABLE	(1)
#define	PMBUS_CMD_B9_ENABLE	(0)
#define	PMBUS_CMD_BA_ENABLE	(0)
#define	PMBUS_CMD_BB_ENABLE	(0)
#define	PMBUS_CMD_BC_ENABLE	(0)
#define	PMBUS_CMD_BD_ENABLE	(0)
#define	PMBUS_CMD_BE_ENABLE	(0)
#define	PMBUS_CMD_BF_ENABLE	(0)
#define	PMBUS_CMD_C0_ENABLE	(0)
#define	PMBUS_CMD_C1_ENABLE	(0)
#define	PMBUS_CMD_C2_ENABLE	(0)
#define	PMBUS_CMD_C3_ENABLE	(0)
#define	PMBUS_CMD_C4_ENABLE	(0)
#define	PMBUS_CMD_C5_ENABLE	(0)
#define	PMBUS_CMD_C6_ENABLE	(0)
#define	PMBUS_CMD_C7_ENABLE	(0)
#define	PMBUS_CMD_C8_ENABLE	(0)
#define	PMBUS_CMD_C9_ENABLE	(0)
#define	PMBUS_CMD_CA_ENABLE	(0)
#define	PMBUS_CMD_CB_ENABLE	(0)
#define	PMBUS_CMD_CC_ENABLE	(0)
#define	PMBUS_CMD_CD_ENABLE	(0)
#define	PMBUS_CMD_CE_ENABLE	(0)
#define	PMBUS_CMD_CF_ENABLE	(0)
#define	PMBUS_CMD_D0_ENABLE	(0)
#define	PMBUS_CMD_D1_ENABLE	(1)
#define	PMBUS_CMD_D2_ENABLE	(1)
#define	PMBUS_CMD_D3_ENABLE	(1)
#define	PMBUS_CMD_D4_ENABLE	(0)
#define	PMBUS_CMD_D5_ENABLE	(1)
#define	PMBUS_CMD_D6_ENABLE	(0)
#define	PMBUS_CMD_D7_ENABLE	(0)
#define	PMBUS_CMD_D8_ENABLE	(0)
#define	PMBUS_CMD_D9_ENABLE	(1)
#define	PMBUS_CMD_DA_ENABLE	(0)
#define	PMBUS_CMD_DB_ENABLE	(0)
#define	PMBUS_CMD_DC_ENABLE	(0)
#define	PMBUS_CMD_DD_ENABLE	(0)
#define	PMBUS_CMD_DE_ENABLE	(0)
#define	PMBUS_CMD_DF_ENABLE	(0)
#define	PMBUS_CMD_E0_ENABLE	(0)
#define	PMBUS_CMD_E1_ENABLE	(0)
#define	PMBUS_CMD_E2_ENABLE	(1)
#define	PMBUS_CMD_E3_ENABLE	(1)
#define	PMBUS_CMD_E4_ENABLE	(1)
#define	PMBUS_CMD_E5_ENABLE	(1)
#define	PMBUS_CMD_E6_ENABLE	(0)
#define	PMBUS_CMD_E7_ENABLE	(1)
#define	PMBUS_CMD_E8_ENABLE	(0)
#define	PMBUS_CMD_E9_ENABLE	(0)
#define	PMBUS_CMD_EA_ENABLE	(0)
#define	PMBUS_CMD_EB_ENABLE	(0)
#define	PMBUS_CMD_EC_ENABLE	(0)
#define	PMBUS_CMD_ED_ENABLE	(0)
#define	PMBUS_CMD_EE_ENABLE	(0)
#define	PMBUS_CMD_EF_ENABLE	(0)
#define	PMBUS_CMD_F0_ENABLE	(1)
#define	PMBUS_CMD_F1_ENABLE	(0)
#define	PMBUS_CMD_F2_ENABLE	(0)
#define	PMBUS_CMD_F3_ENABLE	(0)
#define	PMBUS_CMD_F4_ENABLE	(0)
#define	PMBUS_CMD_F5_ENABLE	(0)
#define	PMBUS_CMD_F6_ENABLE	(0)
#define	PMBUS_CMD_F7_ENABLE	(0)
#define	PMBUS_CMD_F8_ENABLE	(0)
#define	PMBUS_CMD_F9_ENABLE	(0)
#define	PMBUS_CMD_FA_ENABLE	(0)
#define	PMBUS_CMD_FB_ENABLE	(0)
#define	PMBUS_CMD_FC_ENABLE	(0)
#define	PMBUS_CMD_FD_ENABLE	(1)
#define	PMBUS_CMD_FE_ENABLE	(0)
#define	PMBUS_CMD_FF_ENABLE	(0)




#endif //PMBUS_TOPOLOGY_H



