#include "system_defines.h"
#include "Cyclone_Device.h"
#include "pmbus_commands.h"
#include "variables.h"
#include "function_definitions.h"
#include "software_interrupts.h"
#include "pmbus_common.h"
#include "pmbus_topology.h"
#include "dcdc_parameters.h"
#include "fault.h"
#include "Calibration.h"

#include "adc_gains.h"

#include "Thermistor_LUT.h"

#include "standard_interrupt.h"
#include "led.h"

#define ABS(x)		(((x)>0)?(x):(-1*(x)))

#define TEMP_SNS_RATE_MS			(1000)
#define TEMP_SNS_INIT_DELAY_MS		(50)

#define STANDARD_INTERRUPT_RATE_MS	(1U)

#define STARTUP_TIMEOUT_MS			(1000U)

#define PEAKI_DELAY_MS				(1500U)

#define TR_RATIO 7
#define INDUCTANCE 5
#define I_SAT (30000U)

const int32_t iout_max=10000;
int32_t vin_mV;

Uint16 dcdc_op_state = 0U;

#define init_KI_val 1
#define init_KP_val 2
#define nominal_KI_val 10
#define nominal_KP_val 100
int16 Synchro_Temp = 25; //Synchronous Rectifier Temperature
Uint32 IoutCommand;
Uint32 VoutCommand = 14500;

Uint16 IBatt_SetPoint = IBAT_SETPOINT_DEFAULT; //Assign to Default
Uint32 Iout_Ramp_Step_Count = 0;
#define Iout_Ramp_Step_Size 1
uint16_t ISet_mA=2000;

//***********************************************************//
//Frequency Dither Parameters 
//For some reasson placing these anywhere else will 
//not let me access these variables via the memory debugger
//**********************************************************//
Uint16 switching_period;
/////////////////////////////////////////

Uint8 VC_Mode = 0U;
Uint8 CC_Mode = 0U;
Uint8 ILoop_Mode = 0U;
Uint8 VLoop_Mode = 0U;

Uint64 Iramp_Stable_Timestamp = 0U;

Uint64 Startup_Timestamp = 0U;
Uint8 adcCompleted=0;
static uint8_t ccmMode=0;
//uint8_t oc_status=0;
//uint32_t oc_timer=0;
//uint32_t batRes=0;
uint8_t narrowPWM=0;
struct PID_V
{
    int32_t iOut;
    int32_t out;
};
struct PID_V opid={0,0};
int32_t xiset=0;

#ifdef CONTROL_STATE
#undef CONTROL_STATE
#endif
#define CONTROL_STATE(x,y) void y(void);
#include "control_states.def"
#undef CONTROL_STATE

typedef void (*State_Table_Handler_d)(void);
const State_Table_Handler_d State_Table[] = 
{
	#ifdef CONTROL_STATE
	#undef CONTROL_STATE
	#endif
	#define CONTROL_STATE(x,y) y,
	#include "control_states.def"
	#undef CONTROL_STATE
};
const Uint32 State_Table_Count = sizeof(State_Table)/sizeof(State_Table_Handler_d);
#define INLINE

void preset_filter0(unsigned int preset_value);
void preset_filter1(signed int preset_value);
//int32 IIR_Filter(int32 RawValue, float* PrevVal);
//int32 IIR_Filter$(int32 RawValue, float* PrevVal, const float CoeffA, const float CoeffB);
void control(void);
//inline void max_current(void);
inline void CCM_Detection();
inline void poll_adc(void);
inline void CCM_Detection(void);
inline void SyncFet_Ctrl(Uint32 Enable);
inline void vPID(int error);

//***********************************************************//
//Clamp duty cycle to a max allowable value to: 
//prevent hardware damage at various HVDC Voltage values during ABS Events
//prevent hardware damage at high temperatures due to large persistant loads
//**********************************************************//
inline void handle_max_duty_output_clamp(void)
{
const uint32_t VTS=((uint32_t)(840*4000));
    //840: volt-time(us): V-us, *1000: V-ns, *4: 250ps/bit
    register uint32_t width0,width1;
	uint8_t Temp_Derating_Factor;
	if(vin_mV>=175000)
	{
	    width0=((VTS)/(vin_mV/1000)); //unit:250ps
	    width1=((I_SAT-iout_mA)*2*INDUCTANCE*4000)/(vin_mV/TR_RATIO-vout_mV);//unit:250ps
	    width0=width0<width1?width0:width1;
	}
	else
	{
	    width0=0;
	}


	//Calculate the Max Duty Cycle for Max High Side Current
	//Duty*(1/Freq)*(Vin/L) = di  ---  don't know what L is in this case yet 7/2/19 
	//For 600W's output (with efficiency) max Iin @ 200VDC = 3A's, @300VDC = 2A's
	//Quick Fix Code
	//Interpolation Equation => y = 0.0232x2 - 58.335x + 47610
	//output_clamp = (Uint32)(((float)(0.0232 * (adc_values.Vin*adc_values.Vin)) - (58.335*adc_values.Vin) + 47450));
	width1 = (Uint32)(((232 * (adc_values.Vin*adc_values.Vin)) - (583350*adc_values.Vin)))/10000 + 47450;

	//We will derate the power output of the converter starting at 100C upto 125C (125C is the max we can measure)
	//At 100C we will output full power
	//At 125C we will only output %10 of the max rated current
	//y = 1-(temp-100)*(0.9/25)  ---> 0.9 is the derating range in percentage, 25 is the number of step from 100-125C

    width1=width1<width0?width1:width0;
	if(Synchro_Temp >= 100)
	{
		//Temp_Derating_Factor = (float)(1-(Synchro_Temp - 100) * (0.9/25));
	    Temp_Derating_Factor = (100-(Synchro_Temp - 100) * (90/25));
	    width1 *= (Temp_Derating_Factor/100);
	}

	//Filter1Regs.FILTEROCLPHI.bit.OUTPUT_CLAMP_HIGH = (int)(((float)switching_period) * output_clamp);
	if(width1 < (switching_period>>1))  //40000 Max Switching Period
	{
		Filter1Regs.FILTEROCLPHI.bit.OUTPUT_CLAMP_HIGH = (uint32_t)((width1));
	    Filter1Regs.FILTERKICLPHI.bit.KI_CLAMP_HIGH = (uint32_t)(width1*100)/switching_period*((uint32_t)(0x800000/100));
	    Filter1Regs.FILTERYNCLPHI.bit.YN_CLAMP_HIGH=Filter1Regs.FILTERKICLPHI.bit.KI_CLAMP_HIGH;

	}
}

// Output in adc counts 
int handle_copper_temperature_compensation(int trace_voltage_drop_count, int temperature)
{
	return ((trace_voltage_drop_count - DROP_COUNT_OFFSET) * ( 1 - (COPPER_TEMP_COEFF*(temperature - CALIB_TEMPERATURE)))) / CURRENT_GAIN_FACTOR;
  // We can switch to a multi segment linearization later on  
}

inline void poll_adc(void)
{

	adc_values.Iout 	    =   AdcRegs.ADCAVGRESULT[0].bit.RESULT; // IOUT_SNS   		 --	Output Current	--	AD01
	adc_values.Iout_fast    =   AdcRegs.ADCRESULT[0].bit.RESULT;
	adc_values.Vout  		=   AdcRegs.ADCAVGRESULT[1].bit.RESULT; // DCDC_V_SNS_SCALED --	Output Voltage	--	AD02
	adc_values.Vout_fast    =   AdcRegs.ADCRESULT[1].bit.RESULT;
	adc_values.Vout_Prot 	= 	AdcRegs.ADCAVGRESULT[2].bit.RESULT; // LVDC_OV_MON		 --	Output Voltage	--	AD03
	adc_values.Ibatt 	    =   AdcRegs.ADCAVGRESULT[3].bit.RESULT;  	// LV_BATT_I_N		 --	Batt Current 	--	AD04
	adc_values.Ibatt_fast   =   AdcRegs.ADCRESULT[3].bit.RESULT;
	adc_values.HV_battI 	=   AdcRegs.ADCAVGRESULT[4].bit.RESULT;  	// HV_BATT_I_MON_DDC --	HV Batt Current --	AD06
	adc_values.Vin 			=   AdcRegs.ADCAVGRESULT[5].bit.RESULT;	// HV_BATT_V_MON_DDC --	Input Voltage   --	AD13
	adc_values.Vin_fast     =   AdcRegs.ADCRESULT[5].bit.RESULT;
	adc_values.temp_sense   =   AdcRegs.ADCRESULT[6].bit.RESULT;  	// LV_DCDC_TEMP_SNS  --	MOSFET TEMP SES	--	AD00

}

///////////////////////////////////////////////////////////////////////////////
// Get's Temperature value in degrees Celcius from look up table
// The Temperature parameter will be automatically updated in this function 
//////////////////////////////////////////////////////////////////////////////
void get_temperature(void)
{
	Uint16 Tval = (Uint16)(adc_values.temp_sense); 
	Uint16 i = 0;	

	if(Tval >= Thermistor_LUT[0].ADC)
	{
		Synchro_Temp = Thermistor_LUT[0].Temp;  //We need to add some negative temps here - for now lowest temperature is 0C
	}
	else if(Tval <= Thermistor_LUT[Thermistor_LUT_Count-1].ADC)
	{
		Synchro_Temp = Thermistor_LUT[Thermistor_LUT_Count-1].Temp; //Highest Temperature the thermistor can reliably read
	}
	else
	{
		for(i=0; i<Thermistor_LUT_Count; i++)
		{
			//Very Rudimentary Temp Lookup table - gets the temperature that's below the reading (off by 1 degree at worst case)
			if(Thermistor_LUT[i].ADC < Tval)
			{
				//Since the table goes from 0 to 125 C with 125 elements
				Synchro_Temp = Thermistor_LUT[i].Temp + temp_offset;
				break;
			}
		}
	}
}

inline void SyncFet_Ctrl(Uint32 Enable)
{
	if(Enable&&(narrowPWM==0))
	{
		Dpwm0Regs.DPWMCTRL1.bit.GPIO_B_EN = 0; //Enable PWM - 0  Disable PWM - 1
		Dpwm1Regs.DPWMCTRL1.bit.GPIO_B_EN = 0; //Enable PWM - 0  Disable PWM - 1

	}
	else
	{
		Dpwm0Regs.DPWMCTRL1.bit.GPIO_B_EN = 1; //Enable PWM - 0  Disable PWM - 1
		Dpwm1Regs.DPWMCTRL1.bit.GPIO_B_EN = 1; //Enable PWM - 0  Disable PWM - 1
	}


}


void clear_filter1(void)
{
	Filter1Regs.FILTERCTRL.bit.FORCE_START =0;
	Filter1Regs.FILTERCTRL.bit.FILTER_EN = 0;
	Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 1;    
	Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0;
	Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1;
}

INLINE void handle_idle_state(void)
{
    init_dpwm_pidctrl_fe1();
	{
		supply_state = STATE_ILOOP_SETUP;
	}

}


INLINE void handle_iloop_setup_state(void)
{

	Filter1Regs.FILTERKICOEF0.bit.KI_COEF_0 = init_KI_val;  //I Gain change Made here - original Val 1
	Filter1Regs.FILTERKPCOEF0.bit.KP_COEF_0 = init_KP_val; //P Gain change Made here - Original Val 10

	//****Ramp Controller Code - Did not work as expected 6/2/2019 - need to debug****//
	// FeCtrl0Regs.RAMPCTRL.bit.RAMP_EN = 1; //Enable Ramp Logic
	// FeCtrl0Regs.RAMPDACEND.bit.RAMP_DAC_VALUE = VoutCommand; 
	//****End Of Ramp Controller Code****//

	IoutCommand = DAC_IBAT_MILLIAMPS_TO_COUNTS(iout_mA);//IBAT_SETPOINT_INITIAL;
	//Iout_Ramp_Step_Size  = 1;
	FeCtrl1Regs.EADCDAC.bit.DAC_VALUE = IoutCommand;
    clear_filter1();

    VoutCommand = 14700;
	Iramp_Stable_Timestamp = RTC_ms;
    Filter1Regs.FILTERCTRL.bit.FILTER_EN = 1;
	supply_state = STATE_ILOOP_RAMP;
}


INLINE void handle_Iloop_ramp_state(void)
{
	VC_Mode = 0U;
	CC_Mode = 1U;
	ILoop_Mode = 1U;
	VLoop_Mode = 0U;

	{
		if(vout_mV >= PMBus_Vout_Command)
		{
			Filter1Regs.FILTERKICOEF0.bit.KI_COEF_0 = nominal_KI_val;
			Filter1Regs.FILTERKPCOEF0.bit.KP_COEF_0 = nominal_KP_val; 
			opid.iOut=0;
			supply_state = STATE_VOLTAGE_CONTROL;
			LED_Set_CMode(0U);
			LED_Set_VMode(1U);
		}
		else if((++Iout_Ramp_Step_Count)>= Iout_Ramp_Step_Size)
		{
			if(IoutCommand < IBatt_SetPoint)
			{
				IoutCommand += 2;
				Iramp_Stable_Timestamp = RTC_ms;
			}
			else if (RTC_ms > Iramp_Stable_Timestamp + IRAMP_STABILITY_DELAY_MS)
			{
				Filter1Regs.FILTERKICOEF0.bit.KI_COEF_0 = nominal_KI_val;
				Filter1Regs.FILTERKPCOEF0.bit.KP_COEF_0 = nominal_KP_val; 
				supply_state = STATE_CURRENT_REGULATED;
				LED_Set_CMode(1U);
				LED_Set_VMode(0U);
			}

			Iout_Ramp_Step_Count = 0;
		}
		FeCtrl1Regs.EADCDAC.bit.DAC_VALUE = IoutCommand + dac_ibat_offset_digits;//(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/1000000;
	}
}


void handle_current_regulated_state(void)
{
	ILoop_Mode = 1U;
	VLoop_Mode = 0U;

    LED_Set_CMode(1U);
    LED_Set_VMode(0U);
    VC_Mode = 0U;
    CC_Mode = 1U;

    register int32_t x=iout_max-iout_mA;
    if(x<0)
    {
        supply_state=STATE_HEAVY_LOAD;
        xiset += x;
        IoutCommand=DAC_IBAT_MILLIAMPS_TO_COUNTS(xiset);
    }
    else
    {
        if(vout_mV >= PMBus_Vout_Command)
        {
			opid.iOut=ISet_mA;
            supply_state = STATE_VOLTAGE_CONTROL;

            LED_Set_CMode(0U);
            LED_Set_VMode(1U);
            VC_Mode = 1U;
            CC_Mode = 0U;
       }
        else
        {
            xiset=ISet_mA;
            IoutCommand = IBatt_SetPoint;

        }
    }

	FeCtrl1Regs.EADCDAC.bit.DAC_VALUE = (IoutCommand + dac_ibat_offset_digits);//(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/1000000);
}


INLINE void Disable_Converter(void)
{
	
	LED_Set_CMode(0U);
	LED_Set_VMode(0U);

	if(PSON)
	{   
		VC_Mode = 0U;
		CC_Mode = 0U;
		ILoop_Mode = 0U;
		VLoop_Mode = 0U;

		///////////////////////////////////////////////////////////////////////
		//Reset Filter 1 Parameters
        Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0; //put a 0
		Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 0; //into I holding register
		Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1; //set bit to make it happen

        Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0; //put a 0
		Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 1; //into I holding register
		Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1; //set bit to make it happen

        Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0; //put a 0
		Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 2; //into I holding register
		Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1; //set bit to make it happen

        Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0; //put a 0
		Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 3; //into I holding register
		Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1; //set bit to make it happen

        Filter1Regs.FILTERPRESET.bit.PRESET_VALUE = 0; //put a 0
		Filter1Regs.FILTERPRESET.bit.PRESET_REG_SEL = 4; //into I holding register
		Filter1Regs.FILTERPRESET.bit.PRESET_EN = 1; //set bit to make it happen
		///////////////////////////////////////////////////////////////////////

		GateDriveFault = 0;		
		Global_Fault_Reg = 0;
    	pmbus_status_word = 0;

		MiscAnalogRegs.GLBIOVAL.all &= ~MASK_FAILURE;
		MiscAnalogRegs.GLBIOVAL.all |= (MASK_PGOOD|MASK_VINOK);

		Dpwm0Regs.DPWMCTRL1.bit.GPIO_A_EN = 0;  //0 = PWM A in DPWM mode (Default)
		Dpwm1Regs.DPWMCTRL1.bit.GPIO_A_EN = 0;  //0 = PWM A in DPWM mode (Default)
		
		Dpwm0Regs.DPWMCTRL1.bit.GPIO_B_EN = 1;  //0 = PWM A in DPWM mode (Default)
		Dpwm1Regs.DPWMCTRL1.bit.GPIO_B_EN = 1;  //0 = PWM A in DPWM mode (Default)
        MiscAnalogRegs.GLBIOEN.all &= 0xfffffff0;


		SyncFet_Ctrl(0);

		BatteryLost = 0;

		supply_state = STATE_IDLE;

		Startup_Timestamp = RTC_ms;

		pmbus_status_word &= UINT16_MAX ^ (PMBUS_STATUS_WORD_OFF);
	}
	else
	{
		pmbus_status_word |= PMBUS_STATUS_WORD_OFF;
			
		Dpwm1Regs.DPWMCTRL1.bit.GPIO_A_EN = 1;  //0 = PWM A in DPWM mode (Default)
		Filter1Regs.FILTERCTRL.bit.FILTER_EN = 0; //Disable the Filter
		Filter0Regs.FILTERCTRL.bit.FILTER_EN = 0; //Disable the Filter

		VC_Mode = 0U;
		CC_Mode = 0U;
		ILoop_Mode = 0U;
		VLoop_Mode = 0U;
	}
}
#pragma INTERRUPT(standard_interrupt,IRQ)
void standard_interrupt(void)
{
    uint32_t stt;
    register uint32_t irq_number = (CimRegs.IRQIVEC.bit.IRQIVEC); // Clear on read

    switch(irq_number)
    {
    case 30:
        stt=Dpwm0Regs.DPWMINT.all;
        if((Filter1Regs.FILTERYNREAD.bit.YN)<=(0x7fffff/20))
        {
            MiscAnalogRegs.GLBIOEN.all |=0xf;
            narrowPWM=1;
        }
        else
        {
            if((GateDriveFault==0)&&(NoFault))
            {
                MiscAnalogRegs.GLBIOEN.all &= 0xfffffff0;
                narrowPWM=0;

            }
        }
        break;
        /*
    case 26:    //Fault Mux Interrupt
        if(FaultMuxRegs.FAULTMUXINTSTAT.bit.ACOMP_C)
        {
            supply_state=STATE_PAUSE;
            oc_status=1;
        }
        break;
        */
    case 25:    //ADC_CONV_INT: ADC completed interrupt

            if(AdcRegs.ADCSTAT.bit.ADC_INT == 1)
            {
                stt=AdcRegs.ADCCTRL.all;
                poll_adc();
                adcCompleted=1;

            }

            stt=AdcRegs.ADCCOMPRESULT.all;
            break;
    case 18:    //PWM0_INT: Timer0 interrupt; period: 1ms
            if (TimerRegs.T16PWM0CMPCTRL.bit.CMP0_INT_FLAG)
            {
                AdcRegs.ADCCTRL.bit.SW_START = 1;   //start ADC
                TimerRegs.T16PWM0CMPCTRL.all |=3; //clear the flag by writing '1';
                if(PSON)
                {
                    RTC_ms++;
                    //if(oc_status)
                    //{
                    //    oc_timer++;
                    //}
                }
            }

            break;

            default:
            break;
    }

}

void control(void)
{
    static int16 read_temp_delay = (int16)TEMP_SNS_RATE_MS - (int16)TEMP_SNS_INIT_DELAY_MS;
    //static int32_t iBatPrv=0;
    //static int32_t vBatPrv=0;
    if(adcCompleted)
    {
        adcCompleted=0;
        //iBatPrv=ibatt_mA;
        //vBatPrv=vout_mV;

        ibatt_mA_fast = (int32)(IBAT_MON_COUNTS_TO_MILLIAMP(adc_values.Ibatt_fast) + pmbus_Ibat_offset);
        iout_mA_fast  = IOUT_MON_COUNTS_TO_MILLIAMP((int32)adc_values.Iout_fast) + (int32)pmbus_Iout_offset;
        vout_mV_fast  = VOUT_COUNTS_TO_MILLIVOLTS(adc_values.Vout_fast) + (int32)pmbus_Vout_offset;

        ibatt_mA = (int32)(IBAT_MON_COUNTS_TO_MILLIAMP(adc_values.Ibatt) + pmbus_Ibat_offset);
        iout_mA  = IOUT_MON_COUNTS_TO_MILLIAMP((int32)adc_values.Iout) + (int32)pmbus_Iout_offset;
        if(iout_mA<0)
        {
            iout_mA=0;
        }
        vout_mV  = VOUT_COUNTS_TO_MILLIVOLTS(adc_values.Vout) + (int32)pmbus_Vout_offset;
        vin_mV=VIN_MON_COUNTS_TO_MILLIVOLTS(adc_values.Vin);

        //int dV=vout_mV-vBatPrv;
        //int dI=ibatt_mA-iBatPrv;
        //register int32_t x;
        /*
        if((dI>=100)||(dI<=-100))
        {
            x=dV/dI;
            if(x>0)
            {
                batRes=((batRes*19)+x)/20;
            }
        }
        */

        NoFault=fault_handler(); //Run Fault Handler

        //max_current();
        //x=iout_max-iout_mA_fast;
        if(iout_max-iout_mA<0)
        {
            supply_state=STATE_HEAVY_LOAD;
            //IoutCommand = DAC_IBAT_MILLIAMPS_TO_COUNTS(x);
            //FeCtrl1Regs.EADCDAC.bit.DAC_VALUE = (IoutCommand + dac_ibat_offset_digits);//(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/1000000);

        }
        //Since Temp data does not need to be calculated every cycle - calculated it every 1Khz
        //And look up tables take up some cycles
        if(read_temp_delay >= TEMP_SNS_RATE_MS)
        {
            get_temperature();
            read_temp_delay = 0;
        }
        else
        {
            read_temp_delay += STANDARD_INTERRUPT_RATE_MS;
        }


        if(GateDriveFault&&FaultReset)
        {
            GateDriveFault = 0;
            FaultReset = 0;
            //Enable_OpenLoop = 0;
            clear_dpwm_faults();
        }
        /*
        if(oc_status)
        {
            if(oc_timer>=10)
            {
                supply_state=STATE_DISABLED;
                oc_timer=0;
                oc_status=0;
                Dpwm0Regs.DPWMCTRL0.bit.PWM_EN = 0;
                Dpwm1Regs.DPWMCTRL0.bit.PWM_EN = 0;
                LoopMuxRegs.GLBEN.all = 0x00;
                Dpwm0Regs.DPWMCTRL0.bit.PWM_EN = 1;
                Dpwm1Regs.DPWMCTRL0.bit.PWM_EN = 1;
                LoopMuxRegs.GLBEN.all = 0x203;
            }
        }
        */
        CCM_Detection();
        SyncFet_Ctrl(ccmMode);
        //Set Frequency Dithering every cycle
        frequency_dithering();
        //Handle output duty cycle clamping
        handle_max_duty_output_clamp();

        if (State_Table_Count > supply_state)
        {
            State_Table[supply_state]();
        }

        if
            (
                (PSON) &&
                (RTC_ms > (Startup_Timestamp + STARTUP_TIMEOUT_MS)) &&
                (0U != Startup_Timestamp) &&
                (STATE_CURRENT_REGULATED != supply_state) &&
                (STATE_VOLTAGE_CONTROL != supply_state)&&
                (STATE_ILOOP_RAMP != supply_state) &&
                (STATE_HEAVY_LOAD != supply_state)
                //(ibatt_mA < 0)
            )
        {
            Startup_Timestamp = RTC_ms;
            PSON = 0;
            Disable_Converter();
            //fault_handler();
            PSON = 1;
            supply_state = STATE_DISABLED;
        }

        dcdc_op_state =
                (DCDC_OP_STATE_ON           * (0U != PSON)) |
                //(DCDC_OP_STATE_STARTUP      * (0U != Startup_Mode)) |
                (DCDC_OP_STATE_CALIBRATED   * (0U != Calibration_Status())) |
                (DCDC_OP_STATE_ILOOP        * (0U != ILoop_Mode)) |
                (DCDC_OP_STATE_VLOOP        * (0U != VLoop_Mode)) |
                //(DCDC_OP_STATE_PEAKV        * (0U != PeakV)) |
                //(DCDC_OP_STATE_PEAKI        * (0U != PeakI)) |
                (DCDC_OP_STATE_SYNCHRO      * (0U != ccmMode)) |
                0U;
    }
    return;
}
void handle_heavy_load_state(void)
{
    xiset += (iout_max-iout_mA);
    if(xiset>=ISet_mA)
    {
        xiset=ISet_mA;
        if(vout_mV>=PMBus_Vout_Command)
        {
            supply_state=STATE_VOLTAGE_CONTROL;
            opid.iOut=ISet_mA;
        }
        else
        {
            supply_state=STATE_CURRENT_REGULATED;
        }
    }
    IoutCommand = DAC_IBAT_MILLIAMPS_TO_COUNTS(xiset);
    if(IoutCommand>=IBatt_SetPoint)
    {
        IoutCommand=IBatt_SetPoint;
    }
    FeCtrl1Regs.EADCDAC.bit.DAC_VALUE = (IoutCommand + dac_ibat_offset_digits);//(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/1000000);

}
inline void CCM_Detection(void)
{
    if(iout_mA>5000)
    {
        //register int32_t x=(int32_t)(((int64_t)((vin_mV/TR_RATIO-vout_mV)/(2*INDUCTANCE)*Filter1Regs.FILTERYNREAD.bit.YN*PERIOD/4000))/((uint32_t)0x7fffff));//PERIOD: unit:250ps; 1/4000: us
        //register int32_t x=((vin_mV/TR_RATIO)-vout_mV)/(2*INDUCTANCE)*((Filter1Regs.FILTERYNREAD.bit.YN*(PERIOD/4000))/0x800000U);//PERIOD: unit:250ps; 1/4000: us
        register int32_t x=((vin_mV/TR_RATIO)-vout_mV)/(2*INDUCTANCE)*((Filter1Regs.FILTERYNREAD.bit.YN*(PERIOD/4000))>>23);//PERIOD: unit:250ps; 1/4000: us
        ccmMode=(iout_mA>(x+(x>>2)))?1:0;
    }
    else
    {
        ccmMode=0;
    }
}


inline void vPID(int error)
{
    const int KP=10;
    const int KI=1;
    opid.iOut += KI*error;
    if(opid.iOut>ISet_mA)
    {
        opid.iOut=ISet_mA;
    }
    else
    {
        if(opid.iOut<0)
        {
            opid.iOut=0;
        }
    }
    opid.out=((KP*error) + opid.iOut);
    if(opid.out>ISet_mA)
    {
        opid.out=ISet_mA;
    }
    else
    {
        if(opid.out<0)
        {
            opid.out=0;
        }
    }
}
void handle_voltage_control_state(void)
{
    ILoop_Mode = 0U;
    VLoop_Mode = 1U;

    VC_Mode = 1U;
    CC_Mode = 0U;

    LED_Set_CMode(0U);
    LED_Set_VMode(1U);

    int32_t x=iout_max-iout_mA;
    if(x<0)
    {
        supply_state=STATE_HEAVY_LOAD;
        xiset += x;
        IoutCommand = DAC_IBAT_MILLIAMPS_TO_COUNTS(xiset);
    }
    else
    {
        if(vout_mV<PMBus_Vout_Command-100)
        {
            xiset=ISet_mA;
            IoutCommand=IBatt_SetPoint;
            supply_state = STATE_CURRENT_REGULATED;

        }
        else
        {
            x=(int32_t)PMBus_Vout_Command-(int32_t)vout_mV;
            vPID(x);
            xiset=opid.out;
            x=DAC_IBAT_MILLIAMPS_TO_COUNTS(opid.out);
            if(x>IBatt_SetPoint)
            {
                x=IBatt_SetPoint;
            }
            IoutCommand=x;
        }
    }
    FeCtrl1Regs.EADCDAC.bit.DAC_VALUE=IoutCommand+dac_ibat_offset_digits;

}
void handle_pause_state(void)
{
    init_dpwm_pidctrl_fe1();

}
/*
inline void max_current(void)
{
    //iout_max=I_SAT-(uint32_t)((uint64_t)((vin_mV/TR_RATIO-vout_mV)/(2*INDUCTANCE)*Filter1Regs.FILTERYNREAD.bit.YN*PERIOD/4000)/(0x800000));//PERIOD: unit:250ps; 1/4000: us


    register uint32_t x=vin_mV/(TR_RATIO);

    register uint16_t d=((uint32_t)(vout_mV*100))/x;
    if(d>50)
    {
        d=50;
    }
    register uint32_t y=I_SAT-((x-vout_mV)*d*(PERIOD/4000))/(2*INDUCTANCE*100);    //PERIOD: unit - 250pF; PERIOD/4000: unit - us
    iout_max=(y*4)/5;

    //iout_max=I_SAT-((vin_mV/(TR_RATIO)-vout_mV)*vout_mV)/(vin_mV/(TR_RATIO))*(PERIOD/4000)/(2*INDUCTANCE);


    FaultMuxRegs.ACOMPCTRL2.bit.ACOMP_E_THRESH = ((uint32_t)IOUT_MON_MILLIAMP_TO_COUNTS(y))>>5;

    //AdcRegs.ADCCOMPLIM[0].bit.UPPER_LIMIT=(i_max<<1)/3; //2/3 of i_max;
}
*/
