/*
 * Main.c - For PMP7605 - MSP430F51x2 based MPPT reference design.
 *
 *  Created on: Feb 8, 2013
 *      Author: a0131604
 */

//#define GUI
#include "msp430f5132.h"
#include "hal_tlv.h"
#include<string.h>
#include<stdio.h>
#define CALTDH0CTL1_256        *((unsigned int *)0x1A36)

unsigned int Is_Load_On=0,
			 GUI_Battery_Charge_Current_State=0,
			 GUI_Battery_Charge_Previous_State=0,
			 Is_Cal_Load_Management=0,
			 Load_I_Limit = 0x02;

unsigned int i;

unsigned int 		ADC_Readings [6];
unsigned int 		Panel_Voltage,
					Battery_Voltage,
					Panel_Current,
					Battery_Current,
					Panel_Current_S,
					Battery_Current_S,
					Prev_Battery_Current;

unsigned int Panel_Voltage_Avg=0,
			 Panel_Current_Avg=0,
			 Battery_Voltage_Avg=0,
			 Battery_Current_Avg=0;

unsigned long 		Panel_Power = 0,
					Prev_Power = 0;
unsigned int 		Duty = 210;

unsigned char 		Reading_Captured = 1,
					Phase_Shifting_Done = 0;
char				POB_Direction  = 1;

unsigned int 		V_negate = 0,
        			I_negate = 0,
        			Idle_Count = 0;

signed long 		PV_Volts_Prev = 0,
        			PV_Amps_Prev = 0;

float           	Delta_PV_Volts = 0,
                	Delta_PV_Amps = 0,
                	Inc_Condctnce = 0,
                	Condctnce = 0;

unsigned int 		Avg_ctr = 0, ctr_th = 64;

unsigned char 		MPPT_Loop = 1,
					CV_Mode_ON = 0,
					MPP_Loop_Exit_Counter = 0,
					CC_Loop_Exit_Counter = 0;

unsigned long		Charging_Voltage_Reqd = 0;

unsigned char 	Hysterisis_ON = 0,
				Cutoff_Counter = 0,
				Reconnect_Counter = 0,
				OC_Trigger_Counter,
				OC_Triggered = 0,
				Battery_Current_Counter = 0,
				Start_Delay = 0,
				Blanking_Time = 0;

#define Cutoff_Counter_Th			10
#define Reconnect_Counter_Th		10
#define OC_Trigger_Counter_Th		100

#define P_V 	5
#define B_V		4
#define P_I		3
#define B_I		2
#define P_I_S	1
#define B_I_S	0

#define MIN_TOLERANCE_VOLTS	     	3    // Needs to be tuned with Hardware
#define MIN_TOLERANCE_AMPS	     	1    // Needs to be tuned with Hardware
#define MIN_TOLERANCE_CONDUCTANCE    0    // Needs to be tuned with Hardware

#define PANEL_DISABLE 		P3OUT |= BIT3
#define PANEL_ENABLE		P3OUT &= ~BIT3
#define LOAD_DISABLE 		P3OUT |= BIT2
#define LOAD_ENABLE			P3OUT &= ~BIT2

#define POWER_GOOD 			P1IN & BIT6


/**********************Needs to be changed with different Batteries************************************/
unsigned int CC_LIMIT		=	300;						// 210 - 10A, 300 - 15A, 350 - 20A
unsigned int CC_TO_CV_LIMIT	=	305	;					// 305 - 14.2	, 610 - 28.4
unsigned int FLOAT_VOLTAGE	=	295	;					// 295 - 13.8	, 590 - 27.6
unsigned int BATTERY_CUTOFF	=	220	;					// 220 - 10.2	, 440 - 20.4
unsigned int BATTERY_RECONNECT = 240;						// 240 - 11.2	, 480 - 22.4
/******************************************************************************************************/

#define LOOP_EXIT_LIMIT		5

void Init_Clocks (void);
void SetVcoreUp (unsigned int level);
void Init_Timer (void);
void Init_Comparator (void);
void Init_IOs (void);
void Init_ADC (void);
void Init_WDT (void);
void MPPT_Tracking (void);
void Battery_Charge_Profiling (void);
void Load_Management (void);



void main (void)
{

	WDTCTL = WDTPW + WDTHOLD;                 		// Stop WDT
	Init_Clocks ();							  		// Initialize clocks for 25 MHz
	while (!POWER_GOOD);

	_delay_cycles (25000000);  // ??


	// Initialize parameters from GUI
	Init_GUI_Parameters();

	Init_Comparator ();
	Init_Timer ();									// Enable PWM outputs for 200KHz
	__bis_SR_register(GIE);        					// Enable Interrupts
	while (Phase_Shifting_Done);                    //wait till phase shifting is done

	TD0CCTL1 = TD1CCTL1 = OUTMOD_0;                // Once phase shifting is done, switch off the timer outputs,
	TD0CCTL2 = TD1CCTL2 = OUTMOD_0;

	Init_IOs ();

	_delay_cycles (500);

	Init_ADC ();									// ADC Initialization
	Init_WDT ();									// Init Watchdog for ADC Sampling Start

	while (1)
	{
		if (Reading_Captured)
		{
			Reading_Captured = 0;
			if (Avg_ctr  == ctr_th)
			{
				/* Here ctr_th = 64 . i.e average of 64 ADC readings is calculated.
				 * Once is every 1.3ms a set of ADC readings is taken
				 * So loop time is approximately 64 * 1.3 = 83ms
				*/
				Panel_Voltage = Panel_Voltage_Avg / ctr_th;
				Battery_Voltage = Battery_Voltage_Avg / ctr_th;
				Battery_Current = Battery_Current_Avg / ctr_th;
				Panel_Current = Panel_Current_Avg / ctr_th;

#ifdef GUI
				GUI_Battery_Charge_Previous_State=GUI_Battery_Charge_Current_State;
#endif

#ifndef GUI
				GUI_Battery_Charge_Current_State =1;
#endif
				/* Charging starts only when
				 * 1. Panel voltage is more then battery voltage
				 * 2. Panel voltage is less than the predefined maximum threshold value
				 * 3. Battery voltage is more than certain value (this is to ensure than battery is not deeply discharged)
				 *  Start_Delay is a variable used to generated 5s delay
				 * */
				if (((Panel_Voltage > Battery_Voltage + 10) && (Panel_Voltage < 1000))&&(!Start_Delay)/* &&(Battery_Voltage > 150)*/ && (GUI_Battery_Charge_Current_State))
				{
					TD0CCTL1 = TD1CCTL1 = OUTMOD_2;
					TD0CCTL2 = TD1CCTL2 = OUTMOD_6;

					PANEL_ENABLE;

//					if (MPPT_Loop == 1)
						MPPT_Tracking ();
//					else
//						Battery_Charge_Profiling();

					TD0CCTL0 &= ~CCIFG; // wait till the timer completes its current cycle
					while(!(TD0CCTL0 & CCIFG));

					TD0CCR1 = TD1CCR1 = Duty; // update Duty cycles
					TD0CCR2 = TD1CCR2 = Duty - 20;
					Prev_Power = Panel_Power;
					Prev_Battery_Current = Battery_Current;

				}

				if (((Battery_Current < 6)&&(!Start_Delay))||(!GUI_Battery_Charge_Current_State))
				{
					Battery_Current_Counter ++;
					if ((Battery_Current_Counter > 10) || (Panel_Voltage > 1000) || (!GUI_Battery_Charge_Current_State))
					{
						PANEL_DISABLE;
						TD0CCTL1 = TD1CCTL1 = OUTMOD_0;
						TD0CCTL2 = TD1CCTL2 = OUTMOD_0;
						Duty = 170;
						Start_Delay = 1;
						Blanking_Time = 0;
					}

				}
				else if (Battery_Current > 5)
					Battery_Current_Counter = 0;

				/* This block generates approximately 5 seconds delay
				 * Whenever battery current goes below a threshold value, panel is switched off
				 * and restarted after 5 seconds */

				if (Start_Delay)
				{
					Blanking_Time++;
					if (Blanking_Time > 50)
					{
						Start_Delay = 0;
						Battery_Current_Counter = 0;
					}
				}

#ifdef GUI
				/* If GUI is enabled, then system variables are updated from GUI*/
				RespndToGUI();
				if(	Is_Cal_Load_Management)
					Load_Management();
#endif

#ifndef GUI
				// If GUI is not used , Load management is done all the time
				Load_Management();
#endif
				// Reset all the ADC values
				Avg_ctr = 0;
				Panel_Voltage_Avg = 0;
				Battery_Voltage_Avg = 0;
				Panel_Current_Avg = 0;
				Battery_Current_Avg = 0;
			}
		}
	}
}

/* This ISR is called when DMA finishes putting all the ADC values into respective variables
 * Once its done, in this ISR corresponding values are summed up for averaging  */

#pragma vector=DMA_VECTOR
__interrupt void DMA0_ISR (void)
{
  switch(__even_in_range(DMAIV,16))
  {
    case  0: break;                          // No interrupt
    case  2:
      // sequence of conversions complete
    ADC10CTL0 &= ~ADC10ENC;
    Panel_Voltage_Avg += ADC_Readings [P_V];
    Battery_Voltage_Avg += ADC_Readings [B_V];
    Panel_Current_Avg += ADC_Readings [P_I_S];
    Battery_Current_Avg += ADC_Readings [B_I_S];
    Reading_Captured = 1;
    Avg_ctr ++;

      break;                                 // DMA0IFG
    case  4: break;                          // DMA1IFG
    case  6: break;                          // DMA2IFG
    case  8: break;                          // Reserved
    case 10: break;                          // Reserved
    case 12: break;                          // Reserved
    case 14: break;                          // Reserved
    case 16: break;                          // Reserved
    default: break;
  }
}
/*Here all the PORT initialization is done*/
void Init_IOs (void)
{
    // Hi-Res freq locked; now configure ports to output PWMs at TD0/TD1
    P1SEL |= BIT7;                  //  P1.7,TD0.1, options select
    P1DIR |= BIT7;                  // Output direction
    P2SEL |= BIT0 + BIT2 + BIT3;    // P2.0/TD0.2,  P2.2/TD1.1, P2.3/TD1.2, options select
    P2DIR |= BIT0 + BIT2 + BIT3;    // Output direction

    PMAPPWD = 0x02D52;      // Enable Write-access to modify port mapping registers
    PMAPCTL = PMAPRECFG;    // Allow reconfiguration during runtime
    P1MAP0|= PM_ANALOG;    // Modify all PxMAPy registers
    P1MAP1|= PM_ANALOG;    // Modify all PxMAPy registers
    P1MAP2|= PM_ANALOG;    // Modify all PxMAPy registers
    P1MAP3|= PM_ANALOG;    // Modify all PxMAPy registers
    P1MAP4|= PM_ANALOG;    // Modify all PxMAPy registers
    P1MAP5|= PM_ANALOG;    // Modify all PxMAPy registers
    PMAPPWD = 0;            // Disable Write-Access to modify port mapping registers by writing incorrect key
    P1SEL |= BIT0 + BIT1 + BIT2 + BIT3 + BIT4 + BIT5;   // setting the port mapping register PxMAPy to PM_ANALOG together with PxSEL.y=1 when applying analog signals

    P3DIR |= BIT2 + BIT3;
    PANEL_DISABLE;
}

void Init_ADC (void)
{
	  // Configure ADC10
	  ADC10CTL0 = ADC10SHT_2 + ADC10MSC + ADC10ON;// 16ADCclks, MSC, ADC ON
	  ADC10CTL1 = ADC10SHP + ADC10CONSEQ_1;     // sampling timer, s/w trig.,single sequence
	  ADC10CTL2 |= ADC10RES;                    // 10-bit conversion results
	  ADC10MCTL0 = ADC10INCH_5 + ADC10SREF_1;   // A0,A1,A2,A3,A4,A5(EoS), Vref
	  // By default, REFMSTR=1 => REFCTL is used to configure the internal reference
	  while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
	  REFCTL0 |= REFVSEL_2+REFON;               // Select internal ref = 2.5V
	  _delay_cycles (1000);
	  // Configure DMA0 (ADC10IFG trigger)
	  DMACTL0 = DMA0TSEL_24;                    // ADC10IFG trigger
	  __data16_write_addr((unsigned short) &DMA0SA,(unsigned long) &ADC10MEM0);
	                                            // Source single address
	  __data16_write_addr((unsigned short) &DMA0DA,(unsigned long) &ADC_Readings[0]);
	                                            // Destination array address
	  DMA0SZ = 0x06;                            // 3 conversions
	  DMA0CTL = DMADT_4 + DMADSTINCR_3 + DMAEN + DMAIE;

}

// Watchdog timer is used to generated interrupts for taking ADC readings
void Init_WDT (void)
{
	  WDTCTL = WDT_MDLY_32;                     // WDT 32ms, for SMCLK = 1Mhz,here SMCLK = 25Mhz so timer interval = 1.3ms
	  SFRIE1 |= WDTIE;                          // Enable WDT interrupt
}

/* Watchdog Timer interrupt service routine
 * Here for each interrupt, it triggers ADC to take readings
 * and also initializes DMA to put ADC readings into corresponding variables*/
#pragma vector=WDT_VECTOR
__interrupt void WDT_ISR(void)
{
	__data16_write_addr((unsigned short) &DMA0DA,(unsigned long) &ADC_Readings[0]);
	ADC10CTL0 |= ADC10ENC + ADC10SC;        // Sampling and conversion start
}

/* Comparator B is used to detect Overload
 * whenever load current goes beyond this limit, it generates an interrupt */
void Init_Comparator (void)
{
	// Setup ComparatorB
	CBCTL0 |= CBIPEN + CBIPSEL_6; // Enable V+, input channel CB6
	CBCTL1 |= CBPWRMD_1;          // normal power mode

	switch(Load_I_Limit & 0x03)
	{
							case 0x00: // 3 Amps
								CBCTL2 = CBRS_1 + CBRSEL + CBREF1_0 + CBREF0_0;
										break;
							case 0x01: // 6 Amps
								CBCTL2 = CBRS_1 + CBRSEL+ CBREF1_1 + CBREF0_1;
										break;
							case 0x02: // 9 Amps
								CBCTL2 = CBRS_1 + CBRSEL + CBREF1_2 + CBREF0_2;
										break;
							case 0x03: // 12 Amps
								CBCTL2 = CBRS_1 + CBRSEL +CBREF1_3 + CBREF0_3;
										break;
							default:
								break;
	}



	CBCTL3 |= BIT6;               // Input Buffer Disable @P1.0/CB6

	__delay_cycles(7500);         // delay for the reference to settle

	CBINT &= ~(CBIFG + CBIIFG);   // Clear any errant interrupts
	CBINT  |= CBIE;               // Enable CompB Interrupt on rising edge of CBIFG (CBIES=0)
	CBCTL1 |= CBON;               // Turn On ComparatorB

}


/* Comp_B ISR
 * This ISR is called whenever overload is detected
 * Here load is disabled and variable OC_Triggered is updated*/
#pragma vector=COMP_B_VECTOR
__interrupt void Comp_B_ISR (void)
{
	LOAD_DISABLE;
	CBINT &= ~CBIFG;              // Clear Interrupt flag
	OC_Triggered = 1;
	OC_Trigger_Counter = 0;
}

/*This block initializes Timer D for interleaved buck stage operation
 * Timer D is operated at 128Mhz using High Resolution Generator
 * TD0.0,TD0.1 operate one buck stage and TD1.0,TD1.1 operate another buck stage
 * Timers are configured such that increasing 'Duty' variable actually decreases the operating duty cycle of that buck stage*/
void Init_Timer (void)
{
	struct s_TLV_Timer_D_Cal_Data * pTD0CAL;  // Structure initialized in tlv.h
	unsigned char bTD0CAL_bytes;
	unsigned int i = 20;

// Configure TimerD in Hi-Res Free Running Mode
	Get_TLV_Info(TLV_TIMER_D_CAL, 0, &bTD0CAL_bytes, (unsigned int **) &pTD0CAL);
	                                               //Get TimerD0 Cal Values (instance 0)
	if(bTD0CAL_bytes == 0x0)
	{
	    // No TimerD free running cal data found
	    while(1);                             // Loop here
	}
	// Configure Master Timer Instance - TimerD0, Hi-Res Calibrated Mode
	TD0CTL0 = TDSSEL_2;                       // TDCLK = SMCLK = 25MHz = Hi-Res input clk select

	TD0HCTL1 = pTD0CAL->TDH0CTL1_128;         // Read the 256Mhz TimerD TLV Data

	TD0CTL1 |= TDCLKM_1;                      // Select Hi-res local clock
	TD0HCTL0 = TDHEN + TDHM_0;                // CALEN=0 => free running mode; enable Hi-res mode
	                                          // TDHM_0 => 128Mhz, TDHM_1 = > 256Mhz
	// Here Timer D is configured to run at 128Mhz

	TD0CCR0 = 700;                            // 128M/700 = 182.8Khz is the operating frequency of the buck stage

	TD0CCTL0 |= CCIE;                     // interrupt is enabled to adjust phase
	TD0CCR1 = Duty ;
	TD0CCTL1 |= OUTMOD_2;                     // TD0CCR1, Reset/Set
	TD0CCR2 = Duty - 20;                             // Duty-20 is done to provide Dead Band
	TD0CCTL2 |= OUTMOD_6;                     // TD0CCR2, Set/reset

	TD1CCR0 = 700;                            //
	TD1CCR1 = Duty ;
	TD1CCTL1 |= OUTMOD_2;                     // TD1CCR1, Reset/Set
	TD1CCR2 = Duty - 20;
	TD1CCTL2 |= OUTMOD_6;                     // TD1CCR2, Set/reset
	TD1CTL1 = TDCLKM_2;                       // TD1 clock = Auxiliary clock source from master timer instance

	// Syncronize master (TD0) and slave (TD1) timer instances
	TEC1XCTL2 |= TECAXCLREN;		    // Enable synchronized clear by enabling Aux clear of slave timer

	TD0CTL0 |= MC_1 + TDCLR;                  // up-mode, clear TDR, Start timer

}
/*This ISR is called when TD0 overflows. Here TD1 is reset to provide 180 degree phase difference
 * between interleaved buck stages*/
#pragma vector=TIMER0_D0_VECTOR
__interrupt void TIMER0_D0_ISR(void)
{
	__delay_cycles(57);						// Delay Adjusted for perfect 180 deg Phase shift
	  TD1CTL0 |= TDCLR;                         // Clear timer counter
	  TD0CCTL0 &= ~CCIE;
	  Phase_Shifting_Done = 1;
}
void Init_Clocks (void)
{

	SetVcoreUp (0x01);
	SetVcoreUp (0x02);
	SetVcoreUp (0x03);

	// Configure DCO = 25Mhz
	UCSCTL3 = SELREF_2;                       // Set DCO FLL reference = REFO
	UCSCTL4 |= SELA_2;                        // Set ACLK = REFO
	__bis_SR_register(SCG0);                  // Disable the FLL control loop
	UCSCTL0 = 0x0000;                         // Set lowest possible DCOx, MODx
	UCSCTL1 = DCORSEL_7;                      // Select DCO range 50MHz operation
	UCSCTL2 = FLLD_1 + 762;                   // Set DCO Multiplier for 25MHz
	                                          // (N + 1) * FLLRef = Fdco
	                                            // (762 + 1) * 32768 = 25MHz
	                                            // Set FLL Div = fDCOCLK/2
	__bic_SR_register(SCG0);                  // Enable the FLL control loop

	// Worst-case settling time for the DCO when the DCO range bits have been
	// changed is n x 32 x 32 x f_MCLK / f_FLL_reference. See UCS chapter in 5xx
	// UG for optimization.
	// 32 x 32 x 25 MHz / 32,768 Hz ~ 780k MCLK cycles for DCO to settle
	__delay_cycles(782000);

	// Loop until Xt1 & DCO stabilizes - In this case only DCO has to stabilize
	do
	{
	    UCSCTL7 &= ~(XT1LFOFFG + XT1HFOFFG + DCOFFG);
	                                            // Clear XT1,DCO fault flags
	    SFRIFG1 &= ~OFIFG;                      // Clear fault flags
	}while (SFRIFG1&OFIFG);                   // Test oscillator fault flag

}

void SetVcoreUp (unsigned int level)
{
  // Open PMM registers for write
  PMMCTL0_H = PMMPW_H;
  // Set SVS/SVM high side new level
  SVSMHCTL = SVSHE + SVSHRVL0 * level + SVMHE + SVSMHRRL0 * level;
  // Set SVM low side to new level
  SVSMLCTL = SVSLE + SVMLE + SVSMLRRL0 * level;
  // Wait till SVM is settled
  while ((PMMIFG & SVSMLDLYIFG) == 0);
  // Clear already set flags
  PMMIFG &= ~(SVMLVLRIFG + SVMLIFG);
  // Set VCore to new level
  PMMCTL0_L = PMMCOREV0 * level;
  // Wait till new level reached
  if ((PMMIFG & SVMLIFG))
    while ((PMMIFG & SVMLVLRIFG) == 0);
  // Set SVS/SVM low side to new level
  SVSMLCTL = SVSLE + SVSLRVL0 * level + SVMLE + SVSMLRRL0 * level;
  // Lock PMM registers for write access
  PMMCTL0_H = 0x00;
}

// Timer0_D1 Interrupt Vector (TDIV) handler
#pragma vector=TIMER0_D1_VECTOR
__interrupt void TIMER0_D1_ISR(void)
{
  switch(__even_in_range(TD0IV,30))
  {
    case  0: break;                          // No interrupt
    case  2: break;                          // CCR1 not used
    case  4: break;                          // CCR2 not used
    case  6: break;                          // reserved
    case  8: break;                          // reserved
    case 10: break;                          // reserved
    case 12: break;                          // reserved
    case 14: break;
    case 16: break;
    case 18:                                 // Clock fail low
      while(1);                              // Input ref clock freq too low; trap here
    case 20:                                 // Clock fail high
      while(1);                              // Input ref clock freq too high; trap here
    case 22:                                 // Hi-res freq locked
      // Hi-Res freq locked; now configure ports to output PWMs at TD0/TD1
      P1SEL |= BIT6 + BIT7;                  // P1.6/TD0.0, P1.7,TD0.1, options select
      P1DIR |= BIT6 + BIT7;                  // Output direction
      P2SEL |= BIT0 + BIT1 + BIT2 + BIT3;    // P2.0/TD0.2, P2.1/TD1.0, P2.2/TD1.1, P2.3/TD1.2, options select
      P2DIR |= BIT0 + BIT1 + BIT2 + BIT3;    // Output direction

      __bic_SR_register_on_exit(LPM0_bits + GIE);  // Exit LPM0 on return to main

      break;
    case 24: break;                          // Hi-res freq unlocked
    case 26: break;                          // reserved
    case 28: break;                          // reserved
    case 30: break;                          // reserved
    default: break;
  }
}

/*Here every time panel power is measured and compared with the previous value.
 * If ther is increase in panel power then tracking is continued in the same direction
 * otherwise tracking direction is reversed.
 * At lower panel current ,change in panel power is very negligible.
 * So battery current is monitored to determine the tracking direction*/
void MPPT_Tracking (void)
{
	Panel_Power = Panel_Voltage * Panel_Current;

	if (Panel_Current < 80)
	{
		if (Battery_Current <  Prev_Battery_Current)
		{
			POB_Direction = POB_Direction * -1;
		}
	}
	else
	{
		if (Panel_Power <  Prev_Power)
		{
			POB_Direction = POB_Direction * -1;
		}
	}


	if (POB_Direction == 1)
	{
			Duty--;
			if (Duty < 30)
				Duty = 30;

	}
	else
	{
			Duty ++;
			if (Duty > 680)
				Duty = 680;

	}

	CV_Mode_ON = 0;

	if ((Battery_Current > CC_LIMIT)||(Battery_Voltage > CC_TO_CV_LIMIT))
	{
		MPP_Loop_Exit_Counter ++;
		if (MPP_Loop_Exit_Counter > LOOP_EXIT_LIMIT)
		{

			MPPT_Loop = 0;
			MPP_Loop_Exit_Counter = 0;
			CC_Loop_Exit_Counter = 0;
			if (Battery_Voltage > CC_TO_CV_LIMIT)
				CV_Mode_ON = 1;

		}
	}
	else
		MPP_Loop_Exit_Counter = 0;
}

/* Once CC limit or CC_To_CV_limit are exceeded,variable MPPT_Loop is reset and there after this function will be called instead of MPPT function
 * Here duty is adjusted such that CC_Limit,CC_To_CV_Limit,Float voltage are all maintained according to the battery state */
void Battery_Charge_Profiling (void)
{
	if ((Battery_Voltage <= CC_TO_CV_LIMIT)&& (!CV_Mode_ON))
	{

		if (Battery_Current < CC_LIMIT)
		{
			Duty--;
			if (Duty < 30)
			{
				Duty = 30;

				CC_Loop_Exit_Counter ++;
				if (CC_Loop_Exit_Counter > LOOP_EXIT_LIMIT)
				{
					MPPT_Loop = 1;
					MPP_Loop_Exit_Counter = 0;
					CC_Loop_Exit_Counter = 0;
					POB_Direction = 1;
					Duty = 210;
				}
			}


		}
		else
		{
			Duty ++;
			if (Duty > 680)	Duty = 680;
			CC_Loop_Exit_Counter = 0;
		}

	}
	else
	{
		CV_Mode_ON = 1;
		Charging_Voltage_Reqd = CC_TO_CV_LIMIT;

		if (Battery_Voltage < Charging_Voltage_Reqd)
		{
			Duty--;
			if (Duty < 30) Duty = 30;
		}
		else
		{
			Duty ++;
			if (Duty > 680) Duty = 680;

		}

		if (Battery_Voltage < FLOAT_VOLTAGE - 20)
		{
			CC_Loop_Exit_Counter ++;
			if (CC_Loop_Exit_Counter > LOOP_EXIT_LIMIT)
			{
				MPPT_Loop = 1;
				MPP_Loop_Exit_Counter = 0;
				CC_Loop_Exit_Counter = 0;
				POB_Direction = 1;
				Duty = 210;
			}
		}
	}
}

/*If the battery voltage goes below the predefined battery cutoff voltage for more than 1 second,
 * then load is disabled and hysterisis mode is entered.
 * If the battery voltage goes above predefined battery reconnect voltage for more than 1 second,
 * then load is enabled and hysterisis mode is exited
 *
 * When load gets disabled due to over current , here load is enabled after approximately 10s */
void Load_Management (void)
{
	if ((Hysterisis_ON == 0)&&(Battery_Voltage < BATTERY_CUTOFF))
	{
		Cutoff_Counter ++;
		if (Cutoff_Counter > Cutoff_Counter_Th)
		{
			LOAD_DISABLE;
			Hysterisis_ON = 1;
			Reconnect_Counter = 0;
		}
	}
	else
		Cutoff_Counter = 0;

	if ((Hysterisis_ON)&&(Battery_Voltage > BATTERY_RECONNECT)&&(!OC_Triggered))
	{
		Reconnect_Counter++;
		if (Reconnect_Counter > Reconnect_Counter_Th)
		{
			LOAD_ENABLE;
			Hysterisis_ON = 0;
			Cutoff_Counter = 0;
		}
	}
	else
		Reconnect_Counter = 0;

	if ((OC_Triggered)&&(!Hysterisis_ON)) // OC triggered at 8.40 A
	{
		OC_Trigger_Counter ++;
		if (OC_Trigger_Counter > OC_Trigger_Counter_Th)
		{
			LOAD_ENABLE;
			OC_Trigger_Counter = 0;
			OC_Triggered = 0;
		}

	}
}


