

#include <stdint.h>
#include <stdio.h>
#include "cyclone_device.h"
#include "function_definitions.h"
#include "variables.h"
#include "pmbus_topology.h"
#include "standard_interrupt.h"

#include "Calibration.h"

#include "dcdc_parameters.h"

#define addressof(x)    ((Uint32)(&(x)))
#define DFLASH_BLOCK_SIZE   (32U)

#define PMBUS_FLUSH //do {init_pmbus(0x58);}while(0)

typedef enum e_Cal_Block 
    {
        CAL_A = 0, 
        CAL_B = 1,
    } Cal_Block_e;

#pragma DATA_SECTION(calibration_A, ".CAL_A_DATA")
volatile const calibration_data_t calibration_A = {0};

#pragma DATA_SECTION(calibration_A_checksum, ".CAL_A_CS")
volatile const Uint32 calibration_A_checksum = 0U;

#pragma DATA_SECTION(calibration_B, ".CAL_B_DATA")
volatile const calibration_data_t calibration_B = {0};

#pragma DATA_SECTION(calibration_B_checksum, ".CAL_B_CS")
volatile const Uint32 calibration_B_checksum = 0U;

/*
extern Uint16 PMBus_HVDC_UV_Command; 	// HVDC Under Voltage Set Point - Generates a Fault
extern Uint16 PMBus_HVDC_OV_Command; 	// HVDC Over Voltage Set Point - Generates a Fault
extern  int16 pmbus_Ibat_offset; 	// Ibat ADC Calibration
extern  int32 pmbus_Vout_offset; 	// Vout ADC Calibration
extern  int32 pmbus_Iout_offset;  // Battery Current Output Setpoint
*/

/*
Uint8 start_erase_task(const void* dest_ptr, Uint16 byte_count)
Uint8 update_data_flash(void* dest_ptr, const void* src_ptr, Uint16 byte_count)
*/

static inline void Calibration_StateMachine(void);
static inline void Write_Calibration(void *dest, void *src, Uint32 size);
static inline void Read_Current_Calibration(calibration_data_t *cur);
static inline void Set_Current_Calibration(const volatile calibration_data_t *cur);
static inline Uint8 Compare_Calibration(const volatile calibration_data_t *A, const volatile calibration_data_t *B);
static inline Uint32 Calc_Checksum(const void *Start, Uint32 Length);

const Uint32 block_size = 
    sizeof(calibration_A) +
    sizeof(calibration_A_checksum);

calibration_data_t calibration_Current = {0};
const volatile calibration_data_t *calibration_config = &calibration_A;
const volatile Uint32 *calibration_checksum = &calibration_A_checksum;

Uint8 Buffer[DFLASH_BLOCK_SIZE] = {0U};

static Cal_Block_e block_used = CAL_A;

static Uint8 calibrated = 0U;

void Calibration_Initialize(void)
{
    while (!Calibration_Initialization())
    {
        if (erase_segment_counter > 0)
		{
		  erase_task();	//  Handle the DFlash segment erases
		}
    }
}

Uint8 Calibration_Initialization(void)
{
    static Uint32 csa, csb;
    static enum {
        checksuma,
        checksumb,
        select,
        complete,
    } state = checksuma;
    Uint8 ans = 0;

	// NOTE: This is a SEND BYTE command that requires no additional data validation.
	// ----- EXECUTE COMMAND -----
	// Check for DFlash erase operations in progress
	if (erase_segment_counter > 1)
	{
		// There are multiple segments that still need to be erased.  There is no way to
		// get them erased in time to avoid a PMBus timeout so we will just report an error
		// and return.
		ans = 0;	// CML - Other memory or logic fault has occurred

	}
    else if (DecRegs.DFLASHCTRL.bit.BUSY != 0)
	{
		; //do nothing while busy erasing DFlash
	}
    else
    {
        switch (state)
        {
            case checksuma:
                csa = Calc_Checksum((const void *)&calibration_A, addressof(calibration_A_checksum)-addressof(calibration_A));
                state = checksumb;
                ans = 0U;
                break;
            case checksumb:
                csb = Calc_Checksum((const void *)&calibration_B, addressof(calibration_B_checksum)-addressof(calibration_B));
                state = select;
                ans = 0U;
                break;
            case select:
                if (calibration_A_checksum == csa)
                {
                    block_used = CAL_A;
                    Set_Current_Calibration(&calibration_A);
                }
                else if (calibration_B_checksum == csb)
                {
                    block_used = CAL_B;
                    Set_Current_Calibration(&calibration_B);
                }
                else
                {
                    block_used = CAL_A;
                }
                PMBUS_FLUSH;
                ans = 1;
                state = complete;
                break;
            case complete:
                ans = 1;
                break;
            default:
                state = checksuma;
                ans = 0U;
                break;
        }
    }
    return ans;
}

void Calibration_Main(void)
{
    static Uint8 init_complete = 0U;
    if (erase_segment_counter != 0)
	{
		// do nothing
	}
    else if (!init_complete)
    {
        init_complete = Calibration_Initialization();
    }
    else
    {
        if (DecRegs.DFLASHCTRL.bit.BUSY == 0)
        {
            Calibration_StateMachine();
        }
    }
}

Uint8 Calibration_Status(void)
{
    return calibrated;
}

static inline void Calibration_StateMachine(void)
{
    Uint32 cs = 0U;
    static enum {
        check,
        erase_new,
        write,
        checksum,
        erase_old,
    } state = check;
    if (erase_segment_counter != 0)
	{
		// do nothing
	}
    else if (!(DecRegs.DFLASHCTRL.bit.BUSY == 0))
    {
        // do nothing
    }
    else
    {
        switch (state)
        {
            case check:
                Read_Current_Calibration(&calibration_Current);
                calibration_config = ((CAL_B == block_used)?(&calibration_B):(&calibration_A));
                if (0 == Compare_Calibration(calibration_config, &calibration_Current))
                {
                    state = erase_old;
                }
                break;
            case erase_old:
                calibration_config = ((CAL_B == block_used)?(&calibration_A):(&calibration_B));
                calibration_checksum = ((CAL_B == block_used)?(&calibration_A_checksum):(&calibration_B_checksum));
                if (UINT32_MAX != *calibration_checksum)
                {
                    start_erase_task(
                        (const void *)calibration_config,
                        addressof(calibration_B_checksum) + sizeof(calibration_B_checksum) - addressof(calibration_B));
                }
                state = write;
                break;
            case write:
                Read_Current_Calibration(&calibration_Current);
                Write_Calibration((void *)calibration_config, &calibration_Current, sizeof(calibration_Current));
                state = checksum;
                break;
            case checksum:
                cs = Calc_Checksum((const void *)calibration_config, addressof(calibration_A_checksum)-addressof(calibration_A));
                Write_Calibration((void *)calibration_checksum, &cs, sizeof(cs));
                state = erase_new;
                break;
            case erase_new:
                start_erase_task(
                    (const void *)((CAL_B == block_used)?(&calibration_B):(&calibration_A)),
                    addressof(calibration_B_checksum) + sizeof(calibration_B_checksum) - addressof(calibration_B));
                block_used = ((CAL_B == block_used)?(CAL_A):(CAL_B));
                PMBUS_FLUSH;
                calibrated = 1U;
                state = check;
                break;
            default:
                state = check;
                break;
        }
    }
}

static inline void Write_Calibration(void *dest, void *src, Uint32 size)
{
    update_data_flash(dest, src, size);
}

static inline void Read_Current_Calibration(calibration_data_t *cur)
{
    cur->StructVersion = CALIBRATION_STRUCT_VERSION;

    cur->hw_version = hw_version;

    cur->vout_cmd = (Uint16)PMBus_Vout_Command;
    cur->ibatt_cmd = (Uint16)IBatt_SetPoint;

    cur->vout_sns_offset = (Uint32)pmbus_Vout_offset;
    cur->iload_sns_offset = (Uint16)pmbus_Iout_offset;
    cur->ibatt_sns_offset = (Uint16)pmbus_Ibat_offset;
    cur->temp_sns_offset = (Uint16)0;

    cur->ibatt_dac_offset = (Uint16)dac_ibatt_offset;

    cur->vin_uv_cmd = (Uint16)PMBus_HVDC_UV_Command;
    cur->vin_ov_cmd = (Uint16)PMBus_HVDC_OV_Command;
}

static inline void Set_Current_Calibration(const volatile calibration_data_t *cur)
{
    if (CALIBRATION_STRUCT_VERSION == cur->StructVersion)
    {
        hw_version = cur->hw_version;

        PMBus_Vout_Command = cur->vout_cmd;
        IBatt_SetPoint = cur->ibatt_cmd;

        pmbus_Vout_offset = (int32)(cur->vout_sns_offset);
        pmbus_Iout_offset = (int16)(cur->iload_sns_offset);
        pmbus_Ibat_offset = (int16)(cur->ibatt_sns_offset);

        dac_ibatt_offset = (int16)(cur->ibatt_dac_offset);
        dac_ibat_offset_digits=(int32)(dac_ibatt_offset * HW_Gains[hw_version].LV_BATTI_SNS.CountsPerUnit)/KG;


        PMBus_HVDC_UV_Command = cur->vin_uv_cmd;
        PMBus_HVDC_OV_Command = cur->vin_ov_cmd;

        calibrated = 1U;
    }
}

static inline Uint8 Compare_Calibration(const volatile calibration_data_t *A, const volatile calibration_data_t *B)
{
    Uint8 ans = 1U;
    Uint32 ndx = 0U;
    for (ndx = 0U; ndx < sizeof(calibration_data_t); ndx++)
    {
        if (((Uint8*)(A))[ndx] != ((Uint8*)(B))[ndx])
        {
            ans = 0U;
            break;
        }
    }
    return ans;
}

static inline Uint32 Calc_Checksum(const void *Start, Uint32 Length)
{
    Uint32 ans = 0x55AA55AAU;
    Uint32 bytes = Length;
    Uint32 ndx = 0U;
    Uint32 acc = 0xA5A5A5A5U;
    for (ndx = 0U; ndx < bytes; ndx++)
    {
        acc <<= 8;
        acc |= ((Uint8*)(Start))[ndx];
        ans += acc;
    }

    ans ^= UINT32_MAX;
    ans += 1U;

    return ans;
}















