# coding=UTF-8

""" 
 * Copyright (C) 2020 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED 
"""
from __future__ import division

import sys

device_path = sys.path[-1]                          #[cust_path, eng_path][eng_mode]
sys.path.insert(0, device_path + r'\ipy_stdlib.zip')    # sys.path.append(r"C:\Program Files (x86)\IronPython 2.7\Lib")

from fractions import Fraction, gcd
import io
import subprocess
import MatlabInterface
import threading
import os
import ast
import operator as op
from math import ceil, floor, log # for some reason there's no log2 for this python version.
import System
from System import Array,Byte
import math
from pllsolver_new import *
from collections import OrderedDict
import itertools
from System.Windows.Media import SolidColorBrush, Color, Colors

class GlobalVariables(object):
    
    def __init__(self):

        self.wizard_page_list = ["Wizard", "Set XO", "Set outputs", "Set reference", "Ref validation", "Set DPLL", "Set DPLL (cont.)", "Save and Evaluate"] # This is used by the NEXT and BACK buttons
        self.channel_num_list = ["0_1", "2_3", "4", "5", "6", "7"]
        self.xo_range = [10e6, 100e6]
        self.ref_range = [1, 800e6]
        self.apll1_pfd_range = [1e6, 50e6]
        self.apll2_pfd_range = [10e6, 150e6]
        self.vco_range = [[2499.75e6, 2500.25e6], [5.5e9, 6.25e9]]   # vco1, vco2 ranges
        self.output_range = [[5.5e9/1792, 800e6], [5.5e9/1792, 800e6], [5.5e9/1792, 800e6], [5.5e9/1792, 800e6], [5.5e9/1792, 800e6], [1, 800e6]] # 1792 = 256*7
        self.dc_buffer_freq_boundary = 5 # Max frquency (in MHz) supported by the DC buffer.

        #----------------------------------For EEPROM--------------------------
        self.vbTab = "\t"
        self.vbCrLf = "\n"

g = GlobalVariables()
gd = globals()

#----------------LMK05318B post APL update April 2020------------------

class Class_initialization_manager():

    def __init__(self):
        pass

    def update_UI_all(self):

        PLL1_PDN_Update_UI()
        PLL2_PDN_Update_UI()

        OUT1_FMT_Update_UI()
        OUT2_FMT_Update_UI()
        OUT3_FMT_Update_UI()
        OUT4_FMT_Update_UI()
        OUT5_FMT_Update_UI()
        OUT6_FMT_Update_UI()
        OUT7_FMT_Update_UI()

        CH0_1_PD_Update_UI()
        CH2_3_PD_Update_UI()
        CH4_PD_Update_UI()
        CH5_PD_Update_UI()
        CH6_PD_Update_UI()
        CH7_PD_Update_UI()

        cb_enable_PRIREF_Update_UI()
        cb_enable_SECREF_Update_UI()
        DPLL_SWITCH_MODE_Update_UI()
        combo_PRIREF_BUF_TYPE_Update_UI()
        combo_SECREF_BUF_TYPE_Update_UI()

        PRIREF_CMOS_SLEW_Update_UI()
        SECREF_CMOS_SLEW_Update_UI()
        PRIREF_PPM_EN_Update_UI()
        SECREF_PPM_EN_Update_UI()
        PRIREF_MISSCLK_EN_Update_UI()
        SECREF_MISSCLK_EN_Update_UI()
        PRIREF_EARLY_DET_EN_Update_UI()
        SECREF_EARLY_DET_EN_Update_UI()
        PRIREF_PH_VALID_EN_Update_UI()
        SECREF_PH_VALID_EN_Update_UI()

        combo_set_max_tdc_freq_Update_UI()
        BAW_LOCKDET_EN_Update_UI()
        DPLL_LOCKDET_PPM_EN_Update_UI()
        DPLL_REF_HIST_EN_Update_UI()

        APLL1_DEN_MODE_Update_UI()
        APLL2_DEN_MODE_Update_UI()

    def show_instructions_all(self):
        
        btn_xo_show_instructions_Update()
        btn_freq_plan_show_instructions_Update()
        btn_refclk_show_instructions_Update()
        btn_ref_validation_show_instructions_Update()
        btn_dpll_show_instructions_Update()
        btn_dpll2_show_instructions_Update()

    def initialize_apll_settings(self):

        frequency_plan_manager.initialize_table()
        cb_manual_override_pll2_rdiv.iValue = 0
        cb_allow_PLL2_prescaler_of_2.iValue = 0
        # bCALC_FREQPLAN_Update()

    def initialize_dpll_settings(self):
        pass

        # cb_show_dpll_registers_Update()

    def update_indirect_controls_all(self):

        combo_dpll_mode_Update()
        combo_ref_priority_Update()
        combo_PRIREF_BUF_TYPE_Update()
        combo_SECREF_BUF_TYPE_Update()
        combo_switching_method_Update()

initialization_manager = Class_initialization_manager()

def System_Update_Run_Once():

    initialization_manager.update_UI_all()
    initialization_manager.show_instructions_all()
    initialization_manager.initialize_apll_settings()
    initialization_manager.initialize_dpll_settings()
    initialization_manager.update_indirect_controls_all()

def After_LoadDefaultConfiguration():

    sXO_freq_Update()
    bCALC_FREQPLAN_Update()
    btn_refclk_show_instructions_Update()
    btn_ref_validation_show_instructions_Update()
    btn_dpll_show_instructions_Update()
    btn_dpll2_show_instructions_Update()

class Class_frequency_plan_manager():

    def __init__(self):

        self.xo_freq = None
        self.baw_pulling_limit = 1000 # This is for XO R-div and doubler calculation
        self.dpll_ratio_limit = 0.0625

        # Below three ordered dictionaries manage the mux, output frequency and output divider limit for all channels
        self.dict_channel_mux = OrderedDict([(channel,"N/A") for channel in g.channel_num_list]) # Manages channel mux for each channel. For example, {"0_1":"APLL1", "2_3":"APLL2", "4":"N/A"}. This dictionary is updated by self.update_dict_channel_mux(). Method self.update_dict_channel_mux(channel_num) is called when the target frequency for that channel is updated.
        self.dict_output_divider = OrderedDict(zip(g.channel_num_list, [[2, 256]]*5 + [[7, 256*2**24]])) # {channel_num: [output_divider_min, output_divider_max]} for all channels
        
        # Below attributes manage the properties of APLL1 and APLL2 that matter to frequency plan calculation. These will be the inputs to solve_possible_vco_frequencies_multiple_prescalers().
        self.vco_frequency_low = [Fraction(g.vco_range[0][0]), Fraction(g.vco_range[1][0])] # [vco1_frequency_low, vco2_frequency_low]
        self.vco_frequency_high = [Fraction(g.vco_range[0][1]), Fraction(g.vco_range[1][1])] # [vco1_frequency_high, vco2_frequency_high]
        self.prescaler_list = [[1], list(range(2, 8))] # [apll1_prescaler_list, apll2_prescaler_list], where apllx_prescaler_list is a list that contains all possible prescaler values. This list can be updated in self.calculate_frequency_plan depending on whether or not prescaler of 2 is allowed for APLL2
        self.prescaler_num = [1, 2] # apll1 has one prescaler; apll2 has two.
        
        # Below attributes manage the channels that are assigned to APLL1 and APLL2, as well as the corresponding frequencies and output divider limit. These will be the inputs to solve_possible_vco_frequencies_multiple_prescalers()
        self.output_channel_list = None # [output_channels_assigned_to_apll1, output_channels_assigned_to_apll2]
        self.output_divider_list = None # [divider_limit_for_output_channels_assigned_to_apll1, divider_limit_for_output_channels_assigned_to_apll2]. The inner list is [[channelx_min_divider, channelx_max_divider], [channely_min_divider, channely_min_divider]]
        self.output_frequency_list = None # [output_frequency_list_apll1, output_frequency_list_apll2]
        self.max_solutions = [10, 50] # max VCO solutions for both PLLs.
        
        # Below attributes store the outputs of function solve_possible_vco_frequencies_multiple_prescalers()
        self.error_code = [None, None] # [error_code_pll1, error_code_pll2]. See docstring of solve_possible_vco_frequencies_multiple_prescalers() for details
        self.error_note = [None, None] # [error_note_pll1, error_note_pll2]
        self.vco_frequency_list = [None, None] # [apll1_vco_frequency_list, apll2_vco_frequency_list]
        self.apll1_vco_prescaler_list = None # This list stores all valid VCO and prescaler combinations. It is [[vco_freq_1, ps1, ps2], [vco_freq_2, ps1, ps2]], where ps1/2 is the value of prescaler1/2. This list is prepared for the table in frequency planner.
        self.apll1_solution_list = None # This list stores all solutions. It is = [solution1, solution2, ...], where solution1 = [(out1_vco_freq, out1_prescaler_index, ou1_prescaler_value, out1_output_divider), (out2_vco_freq, out2_prescaler_index, out2_prescaler_value, out2_output_divider), ...]. This list is used to directly set the prescaler/divider values for all outputs after a solution is selected. It has the same length with self.apllx_vco_prescaler_list
        self.apll2_vco_prescaler_list = None # Same as above    
        self.apll2_solution_list = None # Same as above

        # Below attributes manage the index of selected frequency plan solution, either automatically generated by codes or manually overwritten by user. This index is used find the solution in self.apllx_vco_prescaler_list and self.apllx_solution_list
        self.apll1_final_solution_index = None
        self.apll2_final_solution_index = None

        # Below attributes mange the reference frequencies for APLL1 and APLL2. They are in Fraction() format
        self.apll1_ref_frequency = None
        self.apll2_ref_frequency = None

        # PLL1 and PLL2 enabled/disabled
        self.apll1_enabled = False
        self.apll2_enabled = False
        self.cascaded_apll = False # whether or not APLL2 uses cascaded APLL1 as its input. In that case, APLL1 needs to be enabled even if no output is assigned to APLL1

        # Final selected VCO frequency
        self.vco1_freq = None
        self.vco2_freq = None

        # PLL2 IBS spur frequency list. PLL1 IBS spur doesn't matter that much because the VCO can only be pulled within +/- 100ppm. 
        self.apll2_ibs_frequency = None # This list has the same order with self.apll2_vco_prescaler_list

        # Table objects
        self.table_pll1 = "UIC_table_frequency_plan_pll1"
        self.table_pll2 = "UIC_table_frequency_plan_pll2"

    def autoset_apll1_rdiv(self):
        """This method autosets the doubler and R divider for XO."""

        xo_valid = self.xo_frequency_update()

        if xo_valid:

            temp_xo_freq = float(self.xo_freq) # For R divider and doubler selection float() is sufficient
            temp_r_div = None

            for x in range(1, 2**5 + 1): # iterate through R divider values and check: (1) if PFD frequency is within range; (2) if multiples of PFD frequency are far away from BAW VCO frequency.

                temp_pfd_freq = temp_xo_freq * 2 / x # This is PFD frequency. Enable the doubler first. The doubler will then be disabled if the R divider is an even number
                temp_ibs_freq = min(2500e6 % temp_pfd_freq, temp_pfd_freq - 2500e6 % temp_pfd_freq) # Integer Boundary Spur frequency
                temp_pulled_ppm = temp_ibs_freq / 2500e6 * 1e6 # This is the ppm value pulled away from 2500e6. The integer boundary cannot be within +/-100ppm of BAW VCO frequency because the DPLL cannot make the APLL cross the integer boundary
                temp_actual_ratio = 2500e6 / temp_pfd_freq - floor(2500e6 / temp_pfd_freq)

                if g.apll1_pfd_range[0] <= temp_pfd_freq <= g.apll1_pfd_range[1] and temp_pulled_ppm >= self.baw_pulling_limit and self.dpll_ratio_limit < temp_actual_ratio < 1 - self.dpll_ratio_limit:
                    
                    temp_r_div = x # Solution found
                    message = "XO frequency set successfully!\n\nInteger boundary spur = {}\n\n".format(ee(temp_ibs_freq, "Hz", 2))
                    # message += "IBS_freq / VCO1_freq = {} ppm (need to be larger than {} ppm)\n\n".format(temp_pulled_ppm, self.baw_pulling_limit)
                    # message += "Fractional part of VCO1_freq / PFD_freq = {} (need to be between {} and {})\n\n".format(temp_actual_ratio, self.dpll_ratio_limit, 1 - self.dpll_ratio_limit)
                    break # break the loop if a solution is found. The R divider value is the smaller the better for largest PFD frequency
                
            if not temp_r_div and combo_dpll_mode.iValue == 1: # If no solution's found and DPLL is disabled. 0 = Enable DPLL, 1 = Disable DPLL

                for x in range(1, 2**5 + 1): 

                    temp_pfd_freq = temp_xo_freq * 2 / x # This is PFD frequency. Enable the doubler first

                    if g.apll1_pfd_range[0] <= temp_pfd_freq <= g.apll1_pfd_range[1] and 2500e6 % temp_pfd_freq == 0: 
                        
                        temp_r_div = x # solution found
                        message = "XO frequency set successfully!\n\n"
                        break

            if not temp_r_div: # if still no solution's found, then there's no solution

                message = "Error!\n\nInvalid XO frequency. Two conditions must be met:\n\n(1) The fractional part of BAW VCO frequency (2.5 GHz) divided by PFD frequency must be within {} and {}.\n\n(2) The minimum difference between BAW VCO frequency (2.5 GHz) and any multiple of PFD frequency must be beyond 2.5 GHz * {} ppm.\n\nSelect another XO frequency".format(self.dpll_ratio_limit, 1 - self.dpll_ratio_limit, self.baw_pulling_limit)
            
            if temp_r_div: # If a solution is found

                if temp_r_div % 2 == 0: # Since we enabled the XO doubler at the begining, we need to disable it if the calculated R divider is an even number.

                    OSCIN_RDIV.iValue = temp_r_div / 2 - 1 # OSCIN_RDIV.iValue starts from 0 but R divider value starts from 1
                    OSCIN_DBLR_EN.iValue = 0

                else:

                    OSCIN_RDIV.iValue = temp_r_div - 1
                    OSCIN_DBLR_EN.iValue = 1

                temp_pfd_freq_final = temp_xo_freq * (OSCIN_DBLR_EN.iValue + 1) / (OSCIN_RDIV.iValue + 1)
                
                message += "XO R divider = {}\nXO doubler is {}\nPFD frequency = {}".format(OSCIN_RDIV.iValue + 1, ["disabled", "enabled"][OSCIN_DBLR_EN.iValue], ee(temp_pfd_freq_final, "Hz", 10))  
            
            s_wizard_XO_message_box.sValue = message

    def xo_frequency_update(self):
        """This method retrieves XO frequency. Returned xo_valid is either True or False depending on whether or not the XO frequency is entered correctly. 
        Be very careful when changing this method, because this method is also used outside of the scope of this class"""

        try:
            make_fraction(sXO_freq.sValue) # if the input string is "" then it will raise an exception. So "" is not handled separately.
        except:
            s_wizard_XO_message_box.sValue = "Error!\n\nInvalid input format. Type numbers or expressions such as:\n\n48e6\n100e6/3\n48e6 * (1 + 100e-6)"
            return False
        else:
            self.xo_freq = make_fraction(sXO_freq.sValue)

            if self.xo_freq: # if the XO frequency is not 0     

                if g.xo_range[0] <= self.xo_freq <= g.xo_range[1]: # XO frequency needs to be within range
                    xo_valid = True
                    
                else: # XO frequency exceeds XO range
                    xo_valid = False
                    s_wizard_XO_message_box.sValue = "Error!\n\nXO frequency must be between {} and {}.".format(ee(g.xo_range[0], "Hz", 2), ee(g.xo_range[1], "Hz", 2))

            else: # XO frequency = 0
                xo_valid = False
                s_wizard_XO_message_box.sValue = "Error!\n\nXO frequency cannot be 0."

            return xo_valid

    def xo_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "1. Set XO frequency in Hz. Example frequency formats:\n\n48e6\n100e6 / 3\n48e6 * (1 + 100e-6)\n\nIf DPLL is disabled, then XO frequency can be 25 MHz or 50 MHz for APLL1 to work in integer mode.\n\nIf DPLL is enabled then recommended XO frequencies are 12.8 MHz, 19.2 MHz, 24 MHz, 30.72 MHz, 38.88 MHz, 48 MHz and 48.0048 MHz. For 1-pps reference input, low frequency and high stability XO is recommended. For example, 12.8 MHz TCXO or OCXO.\n\n"
        str1 += "2. The XO doubler and R divider are automatically set. To manually set the R divider and XO doubler, go to tab 'Advanced' -> 'APLL1'. If the DPLL is disabled, then there's no restriction on PFD frequency, as long as it's within the PFD frequency range ({} to {}). If the DPLL is enabled, however, two conditions must be met:\n\n(1) The fractional part of BAW VCO frequency (2.5 GHz) divided by PFD frequency must be within {} and {}.\n\n(2) The minimum difference between BAW VCO frequency (2.5 GHz) and any multiple of PFD frequency must be beyond 2.5 GHz * {} ppm.\n\nThe wizard selects the highest PFD frequency that meets both requirements.\n\n".format(ee(g.xo_range[0], "Hz", 2), ee(g.xo_range[1], "Hz", 2), self.dpll_ratio_limit, 1 - self.dpll_ratio_limit, self.baw_pulling_limit)
        str1 += "3. Select the XO interface type according to the selection tips."
        s_wizard_XO_message_box.sValue = str1

    def update_dict_channel_mux(self, channel_num):
        """
        This method is used to automatically assign channel MUX to an output channel according to target output frequencies. 
        It updates self.dict_channel_mux[channel_num] with "APLL1", "APLL2" or "N/A".

        (1) "APLL1". If the target frequency of channel_num can be generated by APLL1 VCO and it can coexist with other channels that also use APLL1 - in other words, dict_channel_mux[other_channel] = "APLL1"
        (2) "APLL2". If (1) is not met, the target frequency of channel_num can be generated by APLL2 VCO and it can coexist with other channels that use APLL2 - in other words, dict_channel_mux[other_channel] = "APLL2"
        (3) "N/A". Both (1) and (2) are not met.
        """
        temp_bool_ch_assign = False # This is boolean variable to be returned at the end to indicate if the channel has been successfully assigned.
        temp_s_out_freq = gd["sCH{}_IN_freq".format(channel_num)].sValue # string format of target frequency for this output channel

        if temp_s_out_freq != "0" and temp_s_out_freq != "":     

            try:
                temp_out_freq = make_fraction(temp_s_out_freq)
            except:
                s_wizard_freqplan_message_box.sValue = "Error!\n\nInvalid input format. Type numbers or expressions such as:\n8000,\n156.25e6,\n156.25e6 * (1 + 70e-6)\n100e6 / 3.\n\nTo disable and power down a channel, either type '0' or clear the input box then hit 'Enter'."
                return False
            else: # now there's a valid output frequency
                temp_out_freq = make_fraction(temp_s_out_freq) # output frequency in Fraction() format

                temp_out_freq_low = g.output_range[g.channel_num_list.index(channel_num)][0] # output frequency range - lower limit
                temp_out_freq_high = g.output_range[g.channel_num_list.index(channel_num)][1] # output frequency range - upper limit

                temp_vco1_freq_low = g.vco_range[0][0] # VCO1 frequency lower limit
                temp_vco1_freq_high = g.vco_range[0][1] # VCO1 frequency higher limit

                temp_vco2_freq_low = g.vco_range[1][0] # VCO2 frequency lower limit
                temp_vco2_freq_high = g.vco_range[1][1] # VCO2 frequency higher limit

                if temp_out_freq_low <= temp_out_freq <= temp_out_freq_high: # if output frequency doesn't exceed supported frequency range

                    temp_output_supported_by_vco1 = ceil(temp_vco1_freq_low / temp_out_freq) <= floor(temp_vco1_freq_high / temp_out_freq) # True if the output frequency can be generated by APLL1 VCO. This is only the first pass channel mux assignment and it doesn't take prescalers into account.
                    temp_output_supported_by_vco2 = ceil(temp_vco2_freq_low / temp_out_freq) <= floor(temp_vco2_freq_high / temp_out_freq) # Same as above for APLL2
                    
                    temp_output_can_coexist_with_others_apll1 = self.validate_mux_selection(0, channel_num)[0] # True if this output can coexist with other output channels that are assigned to APLL1
                    temp_output_can_coexist_with_others_apll2 = self.validate_mux_selection(1, channel_num)[0] # True if this output can coexist with other output channels that are assigned to APLL2

                    def temp_update_message_box_pass(): # What to display if the channel mux is successfully assigned to this channel
                        s_wizard_freqplan_message_box.sValue = "CH{} MUX is set successfully!".format(channel_num)

                    def temp_update_message_box_fail(): # What to display if the channel mux cannot be succesfully assigned because there is no possible VCO frequency for PLL2. Again, this does not handle prescalers otherwise it will be too complex. The prescalers are handled when "Calculate frequency plan" button is clicked.
                        s_wizard_freqplan_message_box.sValue = "Error!\n\nCH{} and CH{} frequencies cannot be generated at the same time. No common multiple of the two falls within APLL1 VCO range (2.5 GHz +/- 100 ppm) or APLL2 VCO range ({} to {}).".format(channel_num, self.validate_mux_selection(1, channel_num)[1], ee(temp_vco2_freq_low, "Hz", 4), ee(temp_vco2_freq_high, "Hz", 4))

                    if temp_output_supported_by_vco1: # If the output is supported by VCO 1 then there's no reason to choose VCO 2.

                        if temp_output_can_coexist_with_others_apll1: # It must not conflict with other APLL1 channels.

                            self.dict_channel_mux[channel_num] = "APLL1" # Assign this channel to APLL1
                            temp_update_message_box_pass()
                            temp_bool_ch_assign = True

                        elif temp_output_supported_by_vco2: # If it has conflict with other "APLL1" channels - for example, both 156.25MHz and 156.25MHz + 20ppm can be generated by APLL1 but not at the same time - then check if it can be generated by VCO2
                            
                            if temp_output_can_coexist_with_others_apll2: # It must not conflict with other "APLL2" channels
                                
                                self.dict_channel_mux[channel_num] = "APLL2" # Assign this channel to APLL2
                                temp_update_message_box_pass()
                                temp_bool_ch_assign = True
                            
                            else: # It conflicts with other channels that are assigned to APLL2

                                self.dict_channel_mux[channel_num] = "N/A"
                                temp_update_message_box_fail()
                                temp_bool_ch_assign = False

                        else: # This channel is hopeless and cannot be generated by either APLL

                            self.dict_channel_mux[channel_num] = "N/A"
                            s_wizard_freqplan_message_box.sValue = "Error!\n\nCH{} cannot be generated by either APLL1 or APLL2.".format(channel_num)
                            temp_bool_ch_assign = False

                    elif temp_output_supported_by_vco2: # If the output cannot be generated by APLL1, check if it can be generated by APLL2

                        if temp_output_can_coexist_with_others_apll2: # It must be able to share the same channel mux with existing channels that are assigned to APLL2

                            self.dict_channel_mux[channel_num] = "APLL2"
                            temp_update_message_box_pass()
                            temp_bool_ch_assign = True

                        else: # If it conflicts with other channels that are assigned to APLL2

                            self.dict_channel_mux[channel_num] = "N/A"
                            temp_update_message_box_fail()
                            temp_bool_ch_assign = False

                    else: # in case both VCO 1 and VCO 2 don't support this output frequency - I'm not sure if this situation exists.

                        self.dict_channel_mux[channel_num] = "N/A"
                        s_wizard_freqplan_message_box.sValue = "Error!\n\nCH{} cannot be generated by either APLL1 or APLL2.".format(channel_num)
                        temp_bool_ch_assign = False

                else: # frequency out of range

                    s_wizard_freqplan_message_box.sValue = "Frequency out of range. Supported range for CH{} is from {} to {}.".format(channel_num, ee(temp_out_freq_low, "Hz", 4), ee(temp_out_freq_high, "Hz", 4))
                    temp_bool_ch_assign = False

        else: # if the user input is either empty or 0, then the output channel will be disabled and powered down.
            
            self.dict_channel_mux[channel_num] = "N/A"

            # Power down this channel and disable the corresponding outputs
            if channel_num == "0_1":

                CH0_1_PD.iValue = 1 # power down channel
                OUT0_FMT.iValue = 0 # change output format to "disabled"
                OUT1_FMT.iValue = 0

            elif channel_num == "2_3":

                CH2_3_PD.iValue = 1
                OUT2_FMT.iValue = 0
                OUT3_FMT.iValue = 0

            else:
                gd["CH{}_PD".format(channel_num)].iValue = 1
                gd["OUT{}_FMT".format(channel_num)].iValue = 0

            s_wizard_freqplan_message_box.sValue = "CH{} powered down and corresponding output(s) disabled.".format(channel_num)
            temp_bool_ch_assign = True

        for channel_num in g.channel_num_list:

            gd["sCH{}_MUX".format(channel_num)].sValue = self.dict_channel_mux[channel_num] # Update the displayed channel mux information
            output_page_manager.output_powerdown_update_ui(channel_num)

        return temp_bool_ch_assign 

    def validate_mux_selection(self, apll_num, channel_num):
        """
        This method is used to check if the current output channel can share the same APLL with other defined channels by examining whether the least
        common multiple of the two has an integer multiple that's within VCO range. This method is used inside of self.update_dict_channel_mux()

        Inputs: apll_num = 0 for APLL1 and apll_num = 1 for APLL2. channel_num is an item in g.channel_num_list. 

        Outputs: returns bool, channel. 
        (1) bool. If this channel can share the same APLL with any other channel, bool = True. Otherwise, bool = False
        (2) channel. If bool = False, then the current channel cannot be generated together with this returned channel. If bool = True, this is None.       

        Warning: this method assumes that make_fraction(sCHx_IN_freq.sValue) will not raise any exception. The exception needs to be handled outside of this method.
        """

        temp_apll = ["APLL1", "APLL2"][apll_num]

        temp_vco_freq_low = g.vco_range[apll_num][0]
        temp_vco_freq_high = g.vco_range[apll_num][1]

        temp_out_freq = make_fraction(gd["sCH{}_IN_freq".format(channel_num)].sValue) # The Fraction format is not needed here. All outputs will be parsed again for frequency plan calculation

        temp_other_channels_dict = OrderedDict([(k, v) for k, v in self.dict_channel_mux.items() if v == temp_apll and k != channel_num]) # A dict of other channels that share the same apll - not including its past self.

        if temp_other_channels_dict: # if this is not an empty dictionary

            for channel, mux in temp_other_channels_dict.items(): # iterate through all other channels that share the same apll
                
                temp_other_out_freq = make_fraction(gd["sCH{}_IN_freq".format(channel)].sValue) # The output frequency of that channel
                temp_out_freq_lcm = lcm_Fraction(temp_out_freq, temp_other_out_freq) # Least common multiple of the two
                
                if floor(temp_vco_freq_high / temp_out_freq_lcm) < ceil(temp_vco_freq_low / temp_out_freq_lcm): # If no VCO frequency can generate the lcm of the two channels, then return False
                    
                    return False, channel
            
            return True, None # If the current channel can coexist with every other channel, then return True

        else:

            return True, None # If there's no other channel that shares the same apll, then return True

    def calculate_frequency_plan(self):
        """
        This method has below steps:
            1. Pass
            2. Grab the output frequency information from the GUI for the channels that have a non-"N/A" mux assigned to them.
               self.output_channel_list and self.output_frequency_list.
            3. Update the apll2 prescaler list according to cb_allow_PLL2_prescaler_of_2.iValue.
            4. Run solve_possible_vco_frequencies_multiple_prescalers() and update error_code, error_note, apllx_vco_prescaler_list and apllx_solution_list based on the returned solutions.
            5. Report errors if any, otherwise report success.
        """
        # Step 1: Pass

        # Step 2: Grab the user-input output frequencies. Update self.output_channel_list, self.output_frequency_list and self.output_divider_list. These lists will be used as input to solve_possible_vco_frequencies_multiple_prescalers().
        self.output_channel_list = [[], []] # clear the lists
        self.output_frequency_list = [[], []]
        self.output_divider_list = [[], []]

        for channel_num, mux in self.dict_channel_mux.items(): # Iterate through output channels and check which MUX it's assigned to.

            if mux != "N/A":

                temp_out_freq = make_fraction(gd["sCH{}_IN_freq".format(channel_num)].sValue) # Exception raised by make_fraction() has already been handled in self.update_dict_channel_mux when user types in the frequencies
                
                if mux == "APLL1":

                    self.output_channel_list[0].append(channel_num)
                    self.output_frequency_list[0].append(temp_out_freq)
                    self.output_divider_list[0].append(self.dict_output_divider[channel_num]) # The divider range for this output

                elif mux == "APLL2":

                    self.output_channel_list[1].append(channel_num)
                    self.output_frequency_list[1].append(temp_out_freq)
                    self.output_divider_list[1].append(self.dict_output_divider[channel_num])

            else:
                gd["sCH{}_IN_freq".format(channel_num)].sValue = "" # empty the input box if that output is not assigned to either of the APLLs

        # Step 3: Update the apll2 prescaler list according to cb_allow_PLL2_prescaler_of_2.iValue
        if cb_allow_PLL2_prescaler_of_2.iValue == 0:
            self.prescaler_list[1] = list(range(3,8))
        else:
            self.prescaler_list[1] = list(range(2,8))

        # Step 4: Run solve_possible_vco_frequencies_multiple_prescalers() and update the class attributes with returned results.
        def run_pllsolver(apll_num):
            self.error_code[apll_num], self.error_note[apll_num], self.vco_frequency_list[apll_num], self.__dict__["apll{}_solution_list".format(apll_num + 1)], self.__dict__["apll{}_vco_prescaler_list".format(apll_num + 1)] = solve_possible_vco_frequencies_multiple_prescalers(vco_frequency_low = self.vco_frequency_low[apll_num], vco_frequency_high = self.vco_frequency_high[apll_num], prescaler_list = self.prescaler_list[apll_num], prescaler_num = self.prescaler_num[apll_num], output_frequency_list = self.output_frequency_list[apll_num], output_divider_list = self.output_divider_list[apll_num], max_solutions = self.max_solutions[apll_num])

        for apll_num in range(2): # iterate through the two APLLs

            if self.output_frequency_list[apll_num]: # if this is not an empty list, i.e., if there's some output assigned to this APLL

                self.__dict__["apll{}_enabled".format(apll_num + 1)] = True # Then enable this APLL and run pllsolver
                run_pllsolver(apll_num)

            else:
                
                self.__dict__["apll{}_enabled".format(apll_num + 1)] = False # Otherwise disable this APLL
                self.error_code[apll_num] = 0 # Update the error code so it doens't get confused when updating status.

        # Step 5: Update status
        if self.error_code == [0, 0]:
            message = ""
        elif self.error_code[0] == 1: # No possible VCO frequency error for APLL1. This error shouldn't occur because the self.channel_mux_dic_update() is run right before this method.
            message = "Error! (error code = {})\n\n{} frequencies cannot be generated at the same time, because no common multiple of them is within the VCO1 frequency range. Reassign the output frequencies.".format(self.error_code, ["CH" + item for item in self.output_channel_list[0]])
        elif self.error_code[1] == 1: # No possible VCO frequency error for APLL2. This error shouldn't occur either. The reason is same as above
            message = "Error! (error code = {})\n\n{} frequencies cannot be generated at the same time, because no common multiple of them is within the VCO2 frequency range. Reassign the output frequencies.".format(self.error_code, ["CH" + item for item in self.output_channel_list[1]])
        elif self.error_code[0] == 2: # A frequency cannot be generated by existing prescaler or output divider range. Since APLL1 has no prescaler, it can only be output divider. This problem shouldn't normally occur
            message = "Error! (error code = {})\n\nCH{} frequency cannot be generated due to limited output divider range. Try using channel 7 as it has a large output divider".format(self.error_code, self.output_channel_list[0][self.error_note[0]])
        elif self.error_code[1] == 2: 
            message = "Error! (error code = {})\n\nCH{} frequency cannot be generated due to limited APLL2 prescaler range. If output synchronization is not required, then check 'Allow PLL2 prescaler of 2'. If the error still occurs after doing so, then this frequency cannot be generated.".format(self.error_code, self.output_channel_list[1][self.error_note[1]])
        elif self.error_code[0] == 3: # This error should never occur because PLL1_P1 is 1 and there's no limitation to PLL1 prescalers.
            message = "Error! (error code = {})\n\n".format(self.error_code) 
        elif self.error_code[1] == 3:
            message = "Error! (error code = {})\n\nNo valid prescaler combination exists to generate {} at the same time. If output synchronization is not required, then check 'Allow PLL2 prescaler of 2'. If the error still occurs after doing so, then this frequency combination cannot be generated.".format(self.error_code, ["CH" + item for item in self.output_channel_list[1]])
    
        s_wizard_freqplan_message_box.sValue = message

    def set_apll2_reference_for_table_update(self):
        """This method calculates apll2 reference assuming that VCO1 is 2500MHz. This is purely for IBS spur calculation for table update. """

        # Update PLL2 reference frequency
        if PLL2_RCLK_SEL.iValue == 0: # PLL2 reference is PLL1 VCO output

            self.cascaded_apll = True

            temp_vco1_freq = Fraction(2500e6)

            temp_pll2_pfd_freq = temp_vco1_freq / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1) # PLL2 PFD frequency

            if temp_pll2_pfd_freq > g.apll2_pfd_range[1] or temp_pll2_pfd_freq < g.apll2_pfd_range[0] or cb_manual_override_pll2_rdiv.iValue == 0: # if the apll2 pfd frequency is out of range, or it is not allowed to manually override apll2 R divider, then reset the divider value to 18
                
                PLL2_RDIV_PRE.iValue = 0 # primary divider = 3
                PLL2_RDIV_SEC.iValue = 5 # secondary divider = 6

            self.apll2_ref_frequency = temp_vco1_freq / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1) # update PLL2 reference frequency 

        else:

            self.cascaded_apll = False

            OSCIN_DBLR_EN.iValue = 1

            if self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) > g.apll2_pfd_range[1]: # If pll2 pfd frequency is out of range then disable the doubler. (it is impossible that pll2 pfd frequency is lower than the limit when XO is used as reference)
                OSCIN_DBLR_EN.iValue = 0

            self.apll2_ref_frequency = self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) # update PLL2 reference frequency

    def update_table(self):
        """Populate the solutions on the table."""

        self.set_apll2_reference_for_table_update() # update apll2 reference for IBS spur calculation
        
        # predefine the lists for tables' ItemSource
        temp_table_source_pll1 = [] # a list of Table_data_item_pll1 class instances for pll1
        temp_table_source_pll2 = [] # a list of Table_data_item_pll2 class instances for pll2

        def temp_fraction_to_float(vco_ps_list):
            """The input should be [vco_freq, ps1, ps2] or [vco_freq, ps1], where vco_freq is in Fraction() format and cannot be displayed on the GUI.
            Need to change it to float. Also changing the frequency to MHz"""
            temp_vco_ps_list = list(vco_ps_list) # to create a copy in order not to overwrite the original list
            temp_vco_ps_list[0] = round(float(temp_vco_ps_list[0]/1e6), 10)
            return temp_vco_ps_list

        # Create the ItemSource for APLL1 table
        if self.apll1_enabled: # if apll1 is enabled, then directly grab self.apll1_vco_prescaler_list and pupulate the solutions on the table

            for num, item in enumerate(self.apll1_vco_prescaler_list):
                temp_table_source_pll1.append(Table_data_item_pll1([num] + (temp_fraction_to_float(item)))) # One item of the apll1_table.ItemSource is [solution_index, vco1_freq, apll1_prescaler1], where solution_index is the index of self.apll1_vco_prescaler_list
        
        else: # if apll1 is disabled, then update the table with VCO1 frequency = 0
            temp_table_source_pll1.append(Table_data_item_pll1([0, 0, "N/A"]))

        # Create the ItemSource for APLL2 table
        if self.apll2_enabled: # if apll2 is enabled  
            
            self.apll2_ibs_frequency = [] # predefine a list for Integer Boundary Spur offset frequency.
            
            for num, item in enumerate(self.apll2_vco_prescaler_list): # self.apll2_vco_prescaler_list = [vco_freq, apll2_p1, apll2_p2]
                
                temp_ibs_freq = float(min(item[0] % self.apll2_ref_frequency, self.apll2_ref_frequency - item[0] % self.apll2_ref_frequency))
                self.apll2_ibs_frequency.append(temp_ibs_freq) # IBS spur frequency
                temp_table_source_pll2.append(Table_data_item_pll2([num] + (temp_fraction_to_float(item)) + [ee(temp_ibs_freq, "Hz", 1)])) # One item of the apll2_table.ItemSource is [solution_index, vco2_freq, apll2_p1, apll2_p2], where solution_index is the index of self.apll2_vco_prescaler_list
        
        else:
            temp_table_source_pll2.append(Table_data_item_pll2([0, 0, "N/A", "N/A", "N/A"]))

        if not self.apll1_vco_prescaler_list and self.apll2_enabled and self.cascaded_apll: # if no output is assigned to PLL1, but PLL2 is active and PLL2 uses PLL1 VCO as its input, then we need to turn PLL1 back on.
            
            self.apll1_enabled = True
            temp_table_source_pll1 = [Table_data_item_pll1([0, 2500, "N/A"])]

        # Update tables
        gd[self.table_pll1].ItemSource = temp_table_source_pll1
        gd[self.table_pll2].ItemSource = temp_table_source_pll2

    def select_solution_auto(self):
        """This method automatically selects the solutions for PLL1 and PLL2 and updates self.apllx_final_solution_index"""

        if self.apll1_vco_prescaler_list: # Only need to select an option if the solution is not an empty list

            if 2500e6 in self.vco_frequency_list[0]: # if 2500e6 is an option, then use it.
                self.apll1_final_solution_index = [item[0] for item in self.apll1_vco_prescaler_list].index(2500e6)
            else: # Otherwise, it doesn't really matter. 
                self.apll1_final_solution_index = 0

        if self.apll2_vco_prescaler_list: # Only need to select an option if the solution is not an empty list. 
            self.apll2_final_solution_index = self.apll2_ibs_frequency.index(max(self.apll2_ibs_frequency)) # Choose the solution that has the farthest away integer boundary spur

    def select_solution_manual(self):
        """This method is to update the solution index after manual selection"""

        self.apll1_final_solution_index = gd[self.table_pll1].SelectedItem.index
        self.apll2_final_solution_index = gd[self.table_pll2].SelectedItem.index

    def update_apll_references(self):
        """This method reads and calculates the reference frequencies for apll1 and apll2 and update self.apllx_ref_frequency. """

        self.apll1_ref_frequency = self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) / (int(OSCIN_RDIV.iValue) + 1) # APLL1 reference frequency is always this.

        if self.apll1_enabled:

            if not self.apll1_vco_prescaler_list: # If no solution is available for APLL1 but APLL2 needs APLL1 as reference
                self.vco1_freq = Fraction(2500e6) # Then set VCO1 frequency to 2.5GHz
            else: # Otherwise if there's a solution for APLL1 then use selected solution.
                self.vco1_freq = self.apll1_vco_prescaler_list[self.apll1_final_solution_index][0]

        else:
            self.vco1_freq = 0

        # Update PLL2 reference frequency
        if PLL2_RCLK_SEL.iValue == 0: # PLL2 reference is PLL1 VCO output

            temp_pll2_pfd_freq = self.vco1_freq / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1) # PLL2 PFD frequency

            if temp_pll2_pfd_freq > g.apll2_pfd_range[1] or temp_pll2_pfd_freq < g.apll2_pfd_range[0] or cb_manual_override_pll2_rdiv.iValue == 0: # if the apll2 pfd frequency is out of range, or it is not allowed to manually override apll2 R divider, then reset the divider value to 18
                
                PLL2_RDIV_PRE.iValue = 0 # primary divider = 3
                PLL2_RDIV_SEC.iValue = 5 # secondary divider = 6

            self.apll2_ref_frequency = self.vco1_freq / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1) # update PLL2 reference frequency 

        else: # PLL2 reference is XO

            OSCIN_DBLR_EN.iValue = 1

            if self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) > g.apll2_pfd_range[1]: # If pll2 pfd frequency is out of range then disable the doubler. (it is impossible that pll2 pfd frequency is lower than the limit when XO is used as reference)
                OSCIN_DBLR_EN.iValue = 0

            self.apll2_ref_frequency = self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) # update PLL2 reference frequency

    def implement_frequency_plan(self):

        if combo_dpll_mode.iValue == 0 or combo_backward_compatible.iValue == 1: # DPLL enabled or needs to be backward compatible
            APLL1_DEN_MODE.iValue = 0 # fixed 40-bit den
        else: # DPLL disabled
            APLL1_DEN_MODE.iValue = 1 # programmable 24-bit den

        if self.apll1_enabled:

            PLL1_PDN.iValue = 0 # Power up APLL1
            self.compute_apll1_fraction(self.vco1_freq, self.apll1_ref_frequency) # VCO1 frequency is already updated in self.update_all_references

        else:
            PLL1_PDN.iValue = 1 # Power down APLL1

        if self.apll2_enabled:

            PLL2_PDN.iValue = 0 # Power up APLL2
            self.vco2_freq = self.apll2_vco_prescaler_list[self.apll2_final_solution_index][0]
            self.compute_apll2_fraction(self.vco2_freq, self.apll2_ref_frequency)
        
        else:
            PLL2_PDN.iValue = 1 # Power down APLL2
            self.vco2_freq = 0

        self.update_output_configs()

    def compute_apll1_fraction(self, vco_freq, ref_freq):

        temp_fraction = vco_freq / ref_freq

        if self.apll1_enabled:

            PLL1_PDN.iValue = 0
            PLL1_NDIV.iValue = int(temp_fraction)

            if APLL1_DEN_MODE.iValue == 0: # fixed 40-bit denominator and programmable 40-bit numerator

                PLL1_NUM.iValue = round(int(PLL1_DEN.iValue) * (temp_fraction - int(temp_fraction)))
                apll_page_manager.pll1_num_update()

            else: # programmable 24-bit numerator and 24-bit denominator

                if temp_fraction.denominator < 2**24: # If the calculated denominator doesn't exceed the max value of PLL1_24b_DEN

                    den = temp_fraction.denominator

                    while den < 10000: # Set the denominator to be larger than 10000 for dithering
                        den *= 10

                    num = int(den / temp_fraction.denominator) * temp_fraction.numerator

                    PLL1_24b_NUM.iValue = num - den * int(temp_fraction)
                    PLL1_24b_DEN.iValue = den
                    apll_page_manager.pll1_24b_num_den_update()

                else: # Otherwise we better use the larger denominator for higher accuracy

                    APLL1_DEN_MODE.iValue = 0
                    PLL1_NUM.iValue = round(int(PLL1_DEN.iValue) * (temp_fraction - int(temp_fraction)))
                    apll_page_manager.pll1_num_update()

            APLL1_DEN_MODE_Update_UI()
    
        else:
            PLL1_PDN.iValue = 1

    def compute_apll2_fraction(self, vco_freq, ref_freq):

        temp_fraction = vco_freq / ref_freq

        if self.apll2_enabled:

            PLL2_PDN.iValue = 0
            PLL2_NDIV.iValue = int(temp_fraction)

            if temp_fraction.denominator < 2**24 and combo_backward_compatible.iValue == 0: # If the calculated denominator doesn't exceed the max value of PLL2_DEN and it doesn't need to be backward compatible

                APLL2_DEN_MODE.iValue = 1

                den = temp_fraction.denominator

                while den < 10000:
                    den *= 10

                num = int(den / temp_fraction.denominator) * temp_fraction.numerator

                PLL2_NUM.iValue = num - den * int(temp_fraction)
                PLL2_DEN.iValue = den

            else: # othewise, use fixed denominator

                APLL2_DEN_MODE.iValue = 0
                PLL2_NUM.iValue = round(int(PLL2_DEN_fixed.iValue) * (temp_fraction - int(temp_fraction)))
                APLL2_DEN_MODE_Update_UI()

        else:                    
            PLL2_PDN.iValue = 1

    def update_output_configs(self):

        if self.apll1_vco_prescaler_list: # If solution is not an empty list

            selected_solution = self.apll1_solution_list[self.apll1_final_solution_index] # selected_solution = [(channel1_vco_freq, channel1_prescaler_index, channel1_prescaler_value, chanel1_output_divider), (same for channel 2), ...]

            for num, channel_num in enumerate(self.output_channel_list[0]): # Iterate through the channels that are assigned to APLL1

                out_div = selected_solution[num][3] # Set output divider value
                gd["CH{}_MUX".format(channel_num)].iValue = 0 # Because PLL1 only has one prescaler, choose "APLL1 P1" as channel mux for all channels.

                if channel_num == "7": # Channel 7 is special because it has two stages of output dividers
                    sOUT7_DIV.iValue = out_div
                    output_page_manager.out7_divider_update() # update the two stages of output 7 divider according to the sOUT7_DIV
                else: # For all other channels, directly update the output divider value
                    gd["OUT{}_DIV".format(channel_num)].iValue = out_div - 1

        if self.apll2_vco_prescaler_list: # if solution is not an empty list

            selected_solution = self.apll2_solution_list[self.apll2_final_solution_index] # selected solution = [(channel1_vco_freq, channel1_prescaler_index, channel1_prescaler_value, channel1_output_divider), (same for channel 2), ...]

            PLL2_P1.iValue = self.apll2_vco_prescaler_list[self.apll2_final_solution_index][1] - 1 # Update APLL2 prescaler 1 from solution

            # Update APLL2 prescaler 2. If P2 is not used, then set it to the same value as P1. Otherwise, update P2 value from solution
            temp_p2 = self.apll2_vco_prescaler_list[self.apll2_final_solution_index][2]

            if temp_p2 == "N/A":
                PLL2_P2.iValue = PLL2_P1.iValue
            else:
                PLL2_P2.iValue = temp_p2 - 1

            # Update output channel mux and channel divider settings from the solution.
            for num, channel_num in enumerate(self.output_channel_list[1]):

                ps_index, out_div = selected_solution[num][1], selected_solution[num][3] # get the prescaler index and output divider value

                if channel_num == "7": # Channel 7 is special because it has two stages of output dividers
                    sOUT7_DIV.iValue = out_div
                    output_page_manager.out7_divider_update() # update the two stages of output 7 divider according to the sOUT7_DIV
                    gd["CH{}_MUX".format(channel_num)].iValue = ps_index + 2 # CHx_MUX: 2 = "APLL2 P1"; 3 = "APLL2 P2", while ps_index is either 0 or 1.
                else:
                    gd["OUT{}_DIV".format(channel_num)].iValue = out_div - 1 # update output divider values for all other channels
                    gd["CH{}_MUX".format(channel_num)].iValue = ps_index + 2 # update channel mux selection for all other channels

    def update_message_box(self):

        # update message box
        message = "Frequency plan completed!\n\n"

        # get selected frequency plan
        message += "Selected frequency plan:\n\n"
        message += "VCO1 frequency = {} MHz\n".format(VCO1_freq.dValue)
        message += "VCO2 frequency = {} MHz\n\n".format(VCO2_freq.dValue)

        if PLL1_PDN.iValue == 0:

            message += "APLL1 settings:\n\n"
            message += "PFD freq = {} Hz\n".format(frequency_update_manager.apll1_ref_frequency)
            message += "N divider = {}\n".format(PLL1_NDIV.iValue)
            message += "Numerator = {}\n".format([PLL1_NUM.iValue, PLL1_24b_NUM.iValue][APLL1_DEN_MODE.iValue])
            message += "Denominator = {}\n".format([PLL1_DEN.iValue, PLL1_24b_DEN.iValue][APLL1_DEN_MODE.iValue])
            message += "Post divider = 1\n\n"

        if PLL2_PDN.iValue == 0:

            message += "APLL2 settings:\n\n"
            message += "APLL2 reference source is {}\n".format(["VCO1", "XO"][PLL2_RCLK_SEL.iValue])
            message += "PFD freq = {} Hz\n".format(frequency_update_manager.apll2_ref_frequency)
            message += "N divider = {}\n".format(PLL2_NDIV.iValue)
            message += "Numerator = {}\n".format(PLL2_NUM.iValue)
            message += "Denominator = {}\n".format([PLL2_DEN_fixed.iValue, PLL2_DEN.iValue][APLL2_DEN_MODE.iValue])
            message += "Post divider 1 = {}\n".format(PLL2_P1.iValue + 1)
            message += "Post divider 2 = {}\n\n".format(PLL2_P2.iValue + 1)

        # calculate frequency error
        if PLL1_PDN.iValue == 0 and combo_dpll_mode.iValue == 1: # It only makes sense to show the ppb error if PLL1 is enabled and the DPLL is disabled

            vco1_freq_target = self.vco1_freq
            vco1_freq_actual = frequency_update_manager.apll1_vco_frequency
            vco1_ppb_error = float((vco1_freq_actual - vco1_freq_target) / vco1_freq_target * 1e9)

            if vco1_ppb_error == 0:
                vco1_ppb_error = "0"
            else:
                vco1_ppb_error = "{:.2e}".format(vco1_ppb_error)

            message += "APLL1 frequency error update:\n\n"
            message += "Target VCO1 frequency = {} Hz\n".format(vco1_freq_target)
            message += "Actual VCO1 frequency = {} Hz\n".format(vco1_freq_actual)
            message += "APLL1 frequency error = {} ppb\n\n".format(vco1_ppb_error)

        if PLL2_PDN.iValue == 0 and combo_dpll_mode.iValue == 1: # PLL2 enabled and DPLL is disabled

            vco2_freq_target = self.vco2_freq
            vco2_freq_actual = frequency_update_manager.apll2_vco_frequency
            vco2_ppb_error = float((vco2_freq_actual - vco2_freq_target) / vco2_freq_target * 1e9)

            if vco2_ppb_error == 0:
                vco2_ppb_error = "0"
            else:
                vco2_ppb_error = "{:.2e}".format(vco2_ppb_error)

            message += "APLL2 frequency error update:\n\n"
            message += "Target VCO2 frequency = {} Hz\n".format(vco2_freq_target)
            message += "Actual VCO2 frequency = {} Hz\n".format(vco2_freq_actual)
            message += "APLL2 frequency error = {} ppb\n\n".format(vco2_ppb_error)

        s_wizard_freqplan_message_box.sValue = message

    def initialize_table(self):
        """Initialize the frequency planner table - set columns and properties"""
        gd[self.table_pll1].AddColumnWithBinding("VCO1 Freq (MHz)", "vco1_freq")
        gd[self.table_pll1].AddColumnWithBinding("APLL1 P1", "apll1_p1")
        gd[self.table_pll1].FitColumnWidthToHeader()
        gd[self.table_pll1].AutoGenerateColumns = False
        gd[self.table_pll1].CanUserAddRows = False
        gd[self.table_pll1].CanUserSortColumns = True
        gd[self.table_pll1].IsReadOnly = True 

        gd[self.table_pll2].AddColumnWithBinding("VCO2 Freq (MHz)", "vco2_freq")
        gd[self.table_pll2].AddColumnWithBinding("APLL2 P1", "apll2_p1")
        gd[self.table_pll2].AddColumnWithBinding("APLL2 P2", "apll2_p2")
        # gd[self.table_pll2].AddColumnWithBinding("IBS spur", "apll2_spur_freq")
        gd[self.table_pll2].FitColumnWidthToHeader()
        gd[self.table_pll2].AutoGenerateColumns = False
        gd[self.table_pll2].CanUserAddRows = False
        gd[self.table_pll2].CanUserSortColumns = True
        gd[self.table_pll2].IsReadOnly = True 

    def freq_plan_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "1. Type desired output frequencies in Hz and hit 'Enter'. To power down and disable a channel, clear the input box and hit 'Enter'. Example frequency entries:\n\n8000\n156.25e6\n156.25e6 * (1 + 25e-6)\n100e6 / 3\n\nIt is recommended to place the same output frequencies next to each other, and place the close frequencies (for example, 156.25 MHz and 155.52 MHz) physically far away from each other in order to reduce channel crosstalk. As an example, OUT0 and OUT4 are physically far away from each other according to the pinout in the datasheet.\n\n"
        str1 += "2. An output is automatically assigned to a source - either APLL1 or APLL2 - after the frequency is entered. An output is assigned to APLL1 whenever possible. This is because the BAW VCO for APLL1 provides much better phase noise and integrated jitter. In order for an output to be generated by APLL1, one of its multiples must be within VCO1 range (2.5 GHz - 100 ppm to 2.5 GHz + 100 ppm). An output is assigned to APLL2 if (1) This output cannot be generated by APLL1. OR (2) This output and another existing output assigned to APLL1 cannot be generated by APLL1 at the same time. When both APLL1 and APLL2 are used, it is recommended to use OUT0 ~ OUT3 to generate APLL1 frequencies and use OUT4 ~ OUT7 to generate APLL2 frequencies, in order to reduce VCO crosstalk. Channel 7 has a very large output divider and can generate frequencies as low as 1 Hz. 1-pps output can only be generated by OUT7.\n\n"
        str1 += "3. Checkbox 'Allow PLL2 prescaler of 2' should be unchecked unless the frequency plan cannot be generated and the error message suggests allowing it. This is because output synchronization may not work properly when PLL2 prescaler is set to 2.\n\n"
        str1 += "4. If it is desired to manually set PLL2 R divider, check 'Manually set PLL2 R divider'. Otherwise the PLL2 R divider will be forced to 3 and 6. The actual R divider value is the product of the two stages. It divides down VCO1 frequency (2.5 GHz) and sets the PLL2 reference. When the two stages of R divider are set to 3 and 6, maximum allowed PFD frequency is achieved (2500 / 18 MHz). If PLL2 is not used or PLL2 uses XO as reference, then this selection can be ignored. If the PLL2 R divider is set manually but the resulting APLL2 PFD frequency is out of range, then the R divider values will also be forced to 3 and 6.\n\n"
        str1 += "5. Select output formats. Availabe selections are: LVDS, CML or LVPECL that requires external AC coupling, HCSL that requires external 50 Ohm to ground, HCSL that has internal 50 Ohm to ground, or 1.8-V LVCMOS. The recommended termination for AC-LVDS, AC-CML and AC-LVPECL can be found in datasheet '9.3.14 Clock Output Interfacing and Termination'. Load termination for HCSL is recommended, whenever it's possible. Therefore, choose HCSL (ext.50R) and place the 50 Ohm to ground close to the receiver end. Choose HCSL (int.50R) for source termination if the receiver is on a daughter card and not always connected to the transmitter. For various CMOS formats, CMOS (+/-), for example, means that the OUTx_P channel has positive polarity, and the OUTx_N channel has negative polarity. Similarly, CMOS (+/Hi-z) means that the OUTx_P channel has positive polarity and the OUTx_N channel is set to tri-state.\n\n"
        str1 += "6. Click 'Calculate frequency plan'.\n\n"
        str1 += "7. The wizard automatically selects the VCO frequency with the farthest Integer Boundar Spur. To manually select another frequency plan, choose the frequency plans in the tables and click 'Apply selected solution'.\n\n"
        s_wizard_freqplan_message_box.sValue = str1

class Class_frequency_update_manager():

    def __init__(self):

        # XO frequency
        self.xo_freq = None

        # apll1 and apll2 PFD frequencies
        self.apll1_ref_frequency = None
        self.apll2_ref_frequency = None

        # vco frequencies
        self.apll1_vco_frequency = None
        self.apll2_vco_frequency = None

        # output frequencies. These are in Fraction() format
        self.output_frequency_actual = OrderedDict([(channel,None) for channel in g.channel_num_list])

    def update_all_frequencies(self):

        xo_valid = frequency_plan_manager.xo_frequency_update()

        if xo_valid:

            self.xo_freq = frequency_plan_manager.xo_freq
            self.update_apll1_reference()
            self.update_vco1_frequency()
            self.update_apll2_reference()
            self.update_vco2_frequency()
            self.update_output_frequencies()

        else:

            self.xo_freq = 0
            UpdateStatusBar("Invalid XO frequency. Frequencies not updated.")

    def update_apll1_reference(self):

        self.apll1_ref_frequency = self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) / (int(OSCIN_RDIV.iValue) + 1)

        XO_freq.dValue = round(float(self.xo_freq)/1e6, 10) # update XO_freq in MHz

        PLL1_PFD_freq.dValue = round(float(self.apll1_ref_frequency)/1e6, 10)

    def update_vco1_frequency(self):

        if PLL1_PDN.iValue == 0: # PLL1 enabled

            if APLL1_DEN_MODE.iValue == 0: # fixed 40-bit denominator, programmable 40-bit numerator
                temp_fraction = make_fraction("{} + {} / {}".format(PLL1_NDIV.iValue, PLL1_NUM.iValue, PLL1_DEN.iValue))
            else: # programmable 24-bit numerator and denominator
                temp_fraction = make_fraction("{} + {} / {}".format(PLL1_NDIV.iValue, PLL1_24b_NUM.iValue, PLL1_24b_DEN.iValue))

            self.apll1_vco_frequency = self.apll1_ref_frequency * temp_fraction
            vco1_ppb_error = (self.apll1_vco_frequency - 2500e6) / 2500e6 * 1e9

            if abs(vco1_ppb_error) < 2e-5: # If less than 0.0002 ppb then round it to 2500 MHz. This is because 1 / 2**40 * 50e6 / 2500e6 * 1e9 = 1.82e-5, where 50e6 is max PFD frequency. Meaning that the maximum rounding error of one numerator value is 1.82e-5 ppb.
                self.apll1_vco_frequency = Fraction("2500e6")

        else:
            self.apll1_vco_frequency = 0

        VCO1_freq.dValue = round(float(self.apll1_vco_frequency)/1e6, 17)

    def update_apll2_reference(self):

        self.xo_freq = make_fraction(sXO_freq.sValue)

        if PLL2_RCLK_SEL.iValue == 0: # VCO1 cascaded

            temp_pll2_pfd_freq = self.apll1_vco_frequency / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1)

            if temp_pll2_pfd_freq > g.apll2_pfd_range[1] or temp_pll2_pfd_freq < g.apll2_pfd_range[0]: # if the apll2 pfd frequency is out of range, then reset the divider value to 18. Reset the divider to 18 unless manual_override checkbox is 1 (this control is in "apll2" page)
                
                PLL2_RDIV_PRE.iValue = 0 # primary divider = 3
                PLL2_RDIV_SEC.iValue = 5 # secondary divider = 6

            self.apll2_ref_frequency = self.apll1_vco_frequency / (int(PLL2_RDIV_PRE.iValue) + 3) / (int(PLL2_RDIV_SEC.iValue) + 1) 

        else: # PLL2 uses XO as reference

            OSCIN_DBLR_EN.iValue = 1

            if self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1) > g.apll2_pfd_range[1]: # If pll2 pfd frequency is out of range then disable the doubler. (it is impossible that pll2 pfd frequency is lower than the limit when XO is used as reference)
                
                OSCIN_DBLR_EN.iValue = 0

            self.apll2_ref_frequency = self.xo_freq * (int(OSCIN_DBLR_EN.iValue) + 1)

        PLL2_PFD_freq.dValue = round(float(self.apll2_ref_frequency)/1e6, 10)

    def update_vco2_frequency(self):

        if PLL2_PDN.iValue == 0: # PLL2 enabled

            if APLL2_DEN_MODE.iValue == 0: # fixed 24-bit denominator:
                temp_fraction = make_fraction("{} + {} / {}".format(PLL2_NDIV.iValue, PLL2_NUM.iValue, PLL2_DEN_fixed.iValue))
            else: # programmble 24-bit denominator
                temp_fraction = make_fraction("{} + {} / {}".format(PLL2_NDIV.iValue, PLL2_NUM.iValue, PLL2_DEN.iValue))

            self.apll2_vco_frequency = self.apll2_ref_frequency * temp_fraction

        else:
            self.apll2_vco_frequency = 0

        VCO2_freq.dValue = round(float(self.apll2_vco_frequency)/1e6, 17)

    def update_output_frequencies(self):

        PLL2PDIV1_freq.dValue = round(float(self.apll2_vco_frequency) / (PLL2_P1.iValue + 1) / 1e6, 10)
        PLL2PDIV2_freq.dValue = round(float(self.apll2_vco_frequency) / (PLL2_P2.iValue + 1) / 1e6, 10)

        temp_prescaler_list = [1, 1, int(PLL2_P1.iValue) + 1, int(PLL2_P2.iValue) + 1] # Prescaler value. This corresponds to CHx_MUX, which is ["APLL1 P1", "APLL1 P1 inverted", "APLL2 P1", "APLL2 P2"]
        temp_vco_freq_list = [self.apll1_vco_frequency]*2 + [self.apll2_vco_frequency]*2 # VCO frequency. This corresponds to CHx_MUX, which is ["APLL1 P1", "APLL1 P1 inverted", "APLL2 P1", "APLL2 P2"]

        for channel_num in g.channel_num_list:

            temp_prescaler = temp_prescaler_list[gd["CH{}_MUX".format(channel_num)].iValue]
            temp_vco_freq = temp_vco_freq_list[gd["CH{}_MUX".format(channel_num)].iValue]

            if channel_num == "7":

                temp_out_div = int(sOUT7_DIV.iValue)
                temp_out_freq = temp_vco_freq / temp_prescaler / temp_out_div
                OUT7_freq.sValue = ee(temp_out_freq, "Hz", 5)

            else:
                temp_out_div = int(gd["OUT{}_DIV".format(channel_num)].iValue) + 1
                temp_out_freq = temp_vco_freq / temp_prescaler / temp_out_div

                if channel_num == "0_1":

                    OUT0_freq.sValue = ee(temp_out_freq, "Hz", 5)
                    OUT1_freq.sValue = ee(temp_out_freq, "Hz", 5)

                elif channel_num == "2_3":

                    OUT2_freq.sValue = ee(temp_out_freq, "Hz", 5)
                    OUT3_freq.sValue = ee(temp_out_freq, "Hz", 5)

                else:
                    gd["OUT{}_freq".format(channel_num)].sValue = ee(temp_out_freq, "Hz", 6)

            self.output_frequency_actual[channel_num] = temp_out_freq

class Class_dpll_reference_manager():

    def __init__(self):

        self.ref_freq = [None, None] # If a reference is disabled then set the frequency to 0
        self.previous_ref_freq = ["25e6", "25e6"] # To store the previously entered PRIREF/SECREF frequency
        self.ref_type = {"diff_no_term": 1, "diff_100": 3, "diff_50": 5, "se_no_term": 8, "se_50": 12} # This is for XO_TYPE, PRIREF_TYPE and SECREF_TYPE. Sine these three controls use smart list, key = the index that user sees. value = the actual value of that 4-bit register

    def get_refclk_freq(self, x):
        """Update self.ref_freq. self.ref_freq = [0, 0] if both references are invalid"""

        temp_ref = ["PRI", "SEC"][x] # Convert 0 or 1 to "PRI" or "SEC"

        temp_ref_freq_s = gd["s{}REF_freq".format(temp_ref)].sValue

        if temp_ref_freq_s in ["", "0"]: # if the user types nothing or 0 in the reference frequency box, then set the reference frequency to 0
            
            self.ref_freq[x] = 0
            gd["cb_enable_{}REF".format(temp_ref)].iValue = 0

        else:

            try: 
                temp_ref_freq = make_fraction(temp_ref_freq_s)

            except:
                s_wizard_refclk_message_box.sValue = "Invalid input. Example frequency formats:\n\n8000\n25e6\n100e6/3\n\n"
                self.ref_freq[x] = 0
            
            else:

                gd["cb_enable_{}REF".format(temp_ref)].iValue = 1

                if g.ref_range[0] <= temp_ref_freq <= g.ref_range[1]: # if reference frequency range

                    self.ref_freq[x] = temp_ref_freq
                    self.set_reference_instructions()

                else:

                    s_wizard_refclk_message_box.sValue = "{}REF frequency out of range. Supported frequency range is from {} to {}.".format(temp_ref, ee(g.ref_range[0], "Hz", 2), ee(g.ref_range[1], "Hz", 2))
                    self.ref_freq[x] = 0

                if self.ref_freq[0] and self.ref_freq[1]: # if both frequencies are not 0 or None

                    temp_gcd = gcd(self.ref_freq[0], self.ref_freq[1])
                    max_r_div = max(self.ref_freq[0] / temp_gcd, self.ref_freq[1] / temp_gcd)

                    if max_r_div > 2**16 - 1:
                        s_wizard_refclk_message_box.sValue = "Error!\n\nMaximum R divider value for both PRIREF and SECREF is 65535. Since TDC_freq = PRIREF_freq / PRIREF_R_div = SECREF_freq / SECREF_R_div, PRIREF or SECREF divided by the greatest common divisor of the two (which is {}) cannot be larger than 65535.".format(temp_gcd)
 
    def ref_priority_update(self):

        if combo_ref_priority.iValue == 0: # prioritize PRIREF

            DPLL_PRIREF_AUTO_PRTY.iValue = 1 # PRIREF set to first priority
            DPLL_SECREF_AUTO_PRTY.iValue = 2 # SECREF set to second priority

        else: # prioritize SECREF

            DPLL_PRIREF_AUTO_PRTY.iValue = 2 # PRIREF set to first priority
            DPLL_SECREF_AUTO_PRTY.iValue = 1 # SECREF set to second priority

    def ref_enable_disable_update(self, x):
        """Defines what happens when the SECREF/PRIREF is enabled/disabled"""

        temp_ref = ["PRI", "SEC"][x]

        if gd["cb_enable_{}REF".format(temp_ref)].iValue == 1: # reference is enabled
            
            if gd["s{}REF_freq".format(temp_ref)].sValue == "": # if xREF is enabled but the frequency field is empty

                gd["s{}REF_freq".format(temp_ref)].sValue = self.previous_ref_freq[x] # restore previous reference frequency

        else: # xREF is disabled

            self.previous_ref_freq[x] = gd["s{}REF_freq".format(temp_ref)].sValue # store the reference frequency as history
            gd["s{}REF_freq".format(temp_ref)].sValue = ""

        if cb_enable_PRIREF.iValue == 1 and cb_enable_SECREF.iValue == 0: # if only PRIREF is enabled

            DPLL_SWITCH_MODE.iValue = 3 # enters manual holdover mode
            combo_ref_priority.iValue = 0 # prioritize PRIREF
            DPLL_REF_MAN_SEL.iValue = 0 # manually select by register
            DPLL_REF_MAN_REG_SEL.iValue = 0 # manually select PRIREF

        elif cb_enable_PRIREF.iValue == 0 and cb_enable_SECREF.iValue == 1: # if only SECREF is enabled

            DPLL_SWITCH_MODE.iValue = 3 # enters manual holdover mode
            combo_ref_priority.iValue = 1 # prioritize SECREF
            DPLL_REF_MAN_SEL.iValue = 0 # manually select by register
            DPLL_REF_MAN_REG_SEL.iValue = 1 # manually select SECREF

        elif cb_enable_PRIREF.iValue == 1 and cb_enable_SECREF.iValue == 1: # if both references are enabled

            DPLL_SWITCH_MODE.iValue = 0 # enters auto non-revertive mode
            combo_ref_priority.iValue = 0 # prioritize PRIREF

        self.ref_priority_update()

    def autoset_ref_buffer_mode(self, x):

        self.get_refclk_freq(x)

        temp_ref = ["PRI", "SEC"][x]

        if self.ref_freq[x] >= g.dc_buffer_freq_boundary * 1e6 or combo_backward_compatible.iValue == 1:
            gd["combo_{}REF_BUF_TYPE".format(temp_ref)].iValue = 1 # AC buffer
        elif 0 < self.ref_freq[x] < g.dc_buffer_freq_boundary * 1e6:
            gd["combo_{}REF_BUF_TYPE".format(temp_ref)].iValue = 2 # DC buffer

        combo_PRIREF_BUF_TYPE_Update_UI()
        combo_SECREF_BUF_TYPE_Update_UI()

    def ref_buffer_type_selection(self, x):

        temp_ref = ["PRI", "SEC"][x]

        switcher = {0: [0, 0], 1: [0, 1], 2: [1, 0], 3: [1, 1]} # 0: AC, hyst = 50 mV; 1: AC, hyst = 200 mV; 2: DC, hyst enabled; 3: DC, hyst disabled.
        gd["{}REF_DC_MODE".format(temp_ref)].iValue = switcher[gd["combo_{}REF_BUF_TYPE".format(temp_ref)].iValue][0]
        gd["{}REF_BUF_MODE".format(temp_ref)].iValue = switcher[gd["combo_{}REF_BUF_TYPE".format(temp_ref)].iValue][1]

    def autoset_val_timer(self, x):

        temp_ref = ["PRI", "SEC"][x]

        if self.ref_freq[x] == 0:
            gd["{}REF_VALTMR_EN".format(temp_ref)].iValue = 0
        else:
            gd["{}REF_VALTMR_EN".format(temp_ref)].iValue = 1

    def autoset_amp_det(self, x):

        temp_ref = ["PRI", "SEC"][x]

        gd["{}REF_AMPDET_EN".format(temp_ref)].iValue = 1

        if self.ref_freq[x] <= 1: # Either disabled or 1pps.

            gd["{}REF_AMPDET_EN".format(temp_ref)].iValue = 0

        elif self.ref_freq[x] < 5e6: # if frequency < 5 MHz, always use slew rate detector

            gd["{}REF_CMOS_SLEW".format(temp_ref)].iValue = 1 # choose CMOS slew rate detect
            gd["DETECT_MODE_{}REF".format(temp_ref)].iValue = 3 # choose VIH / VIL by default

        elif self.ref_freq[x] >= 5e6 and gd["{}REF_TYPE".format(temp_ref)].iValue in [self.ref_type["se_no_term"], self.ref_type["se_50"]]: # if the input type is SE, always use slew rate detector

            gd["{}REF_CMOS_SLEW".format(temp_ref)].iValue = 1 # choose CMOS slew rate detect
            gd["DETECT_MODE_{}REF".format(temp_ref)].iValue = 3 # choose VIH / VIL by default

        else:

            gd["{}REF_CMOS_SLEW".format(temp_ref)].iValue = 0 # choose amplitude detect
            gd["{}REF_LVL_SEL".format(temp_ref)].iValue = 0 # choose 400 mVpp threshold

        PRIREF_CMOS_SLEW_Update_UI()
        SECREF_CMOS_SLEW_Update_UI()

    def autoset_freq_det(self, x):

        temp_ref = ["PRI", "SEC"][x]

        gd["{}REF_PPM_EN".format(temp_ref)].iValue = 0 # For now, always disable the PPM detector until the holdover bug is fixed.

        # if self.ref_freq[x] < 2e3: # if reference is either disabled (0Hz) or less than 2 kHz, then disable ppm detector
        #     gd["{}REF_PPM_EN".format(temp_ref)].iValue = 0
        # else:
        #     gd["{}REF_PPM_EN".format(temp_ref)].iValue = 1

        reference_validation_manager.ref_freq_detect_update(x, bypass_warning = True)

        PRIREF_PPM_EN_Update_UI()
        SECREF_PPM_EN_Update_UI()

    def autoset_early_late_det(self, x):

        temp_ref = ["PRI", "SEC"][x]

        if self.ref_freq[x] < 2e3:

            gd["{}REF_EARLY_DET_EN".format(temp_ref)].iValue = 0
            gd["{}REF_MISSCLK_EN".format(temp_ref)].iValue = 0

        else:

            gd["{}REF_EARLY_DET_EN".format(temp_ref)].iValue = 1
            gd["{}REF_MISSCLK_EN".format(temp_ref)].iValue = 1

            # gd["s{}REF_EARLY_MARGIN".format(temp_ref)].iValue = 1
            # gd["s{}REF_LATE".format(temp_ref)].iValue = 0
            # gd["s{}REF_LATE_MARGIN".format(temp_ref)].iValue = 1

        reference_validation_manager.ref_early_late_det_update(x, bypass_warning = True)

        PRIREF_EARLY_DET_EN_Update_UI()
        PRIREF_MISSCLK_EN_Update_UI()
        SECREF_EARLY_DET_EN_Update_UI()
        SECREF_MISSCLK_EN_Update_UI()

    def autoset_ph_det(self, x):

        temp_ref = ["PRI", "SEC"][x]

        if 0 < self.ref_freq[x] < 2e3:
            gd["{}REF_PH_VALID_EN".format(temp_ref)].iValue = 1
            gd["{}REF_PH_VALID_THR".format(temp_ref)].iValue = 63
        else:
            gd["{}REF_PH_VALID_EN".format(temp_ref)].iValue = 0
            gd["{}REF_PH_VALID_THR".format(temp_ref)].iValue = 0
            
        reference_validation_manager.ref_ph_det_update(x, bypass_warning = True)

        PRIREF_PH_VALID_EN_Update_UI()
        SECREF_PH_VALID_EN_Update_UI()

    def autoset_reference_validation(self, x):

        self.get_refclk_freq(x)
        self.autoset_val_timer(x)
        self.autoset_amp_det(x)
        self.autoset_freq_det(x)
        self.autoset_early_late_det(x)
        self.autoset_ph_det(x)

    def set_reference_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "1. Enable or disable PRIREF and SECREF as needed. If DPLL is not used, then disable both references and skip this page.\n\n"
        str1 += "2. Type the frequencies of PRIREF and / or SECREF in Hz. Example frequency formats:\n\n1\n25e6\n100e6 / 3\n\n"
        str1 += "3. Select interface type. AC or DC buffer is auto-seletected based on reference frequency. If reference frequency is below {0} MHz, then use DC buffer. Otherwise, use AC buffer. To select interface types for AC buffer, refer to the 'Interface Type Selection Tips' in the XO wizard page. The same can be applied to PRIREF and SECREF.\n\n".format(g.dc_buffer_freq_boundary)
        str1 += "4. Select the input switching mode. The input switching mode is auto-seletected based on the states of PRIREF and SECREF enable. When both references are enabled, then Auto non-revertive is selected. If only one reference is enabled, then manual holdover is selected. However, it is highly recommended to read through the 'Input Switching Mode Selection Guide' in this wizard page and make the decision.\n\n"
        str1 += "5. If manual fallback or manual holdover is selected, then choose between manually select by register and manually select by REFSEL pin."
        s_wizard_refclk_message_box.sValue = str1

class Class_reference_validation_manager():

    def __init__(self):

        self.force_ppm_div_16 = True

        # XO, reference and VCO frequencies
        self.xo_freq = None # This is in Fraction() format. self.xo_freq = 0 if the XO input is invalid.
        self.vco1_freq = None # This is in Fraction() format
        self.ref_freq = [None, None] # If a reference is disabled then set the frequency to 0
       
        # Frequency detect
        self.ref_ppm_valid = [None, None]
        self.ref_ppm_invalid = [None, None]
        self.ref_accuracy_ppm = [None, None]
        self.ref_avg_count = [None, None]

        # Early and late clock detect
        self.ref_early_margin = [None, None]
        self.ref_late = [None, None]
        self.ref_late_margin = [None, None]
        self.ppm_meas_time = [0, 0]

        # 1pps phase detect
        self.ref_ph_valid_thr = [None, None]

        # Used by early/late detector
        self.early_clk_div_req_margin = 3
        self.early_clk_div_min = 3
        self.miss_clk_div_req_margin = 3

    def get_refclk_freq(self, x):

        dpll_reference_manager.get_refclk_freq(x)
        self.ref_freq[x] = dpll_reference_manager.ref_freq[x]

    def get_all_reference_configs(self, x):

        temp_ref = ["PRI", "SEC"][x] # Convert 0 or 1 to "PRI" or "SEC"

        # Get VCO1 frequency
        self.vco1_freq = frequency_plan_manager.vco1_freq

        if not self.vco1_freq: # if it's not None or 0, then use it. Otherwise just use 2500MHz
            self.vco1_freq = Fraction("2500e6")

        # Frequency detect
        self.ref_ppm_valid[x] = Fraction(str(gd["s{}REF_PPM_VALID".format(temp_ref)].dValue))
        self.ref_ppm_invalid[x] = Fraction(str(gd["s{}REF_PPM_INVALID".format(temp_ref)].dValue))
        self.ref_accuracy_ppm[x] = Fraction(str(gd["s{}REF_ACCURACY_PPM".format(temp_ref)].dValue))
        self.ref_avg_count[x] = int(gd["s{}REF_AVG_COUNT".format(temp_ref)].iValue)

        # Early and miss clock detect
        self.ref_early_margin[x] = int(gd["s{}REF_EARLY_MARGIN".format(temp_ref)].iValue)
        self.ref_late[x] = int(gd["s{}REF_LATE".format(temp_ref)].iValue)
        self.ref_late_margin[x] = int(gd["s{}REF_LATE_MARGIN".format(temp_ref)].iValue)

        # 1pps phase detect
        self.ref_ph_valid_thr[x] = int(gd["{}REF_PH_VALID_THR".format(temp_ref)].iValue)

    def ref_freq_detect_update(self, x, bypass_warning = False):
        """Set registers and update measurement time. Set measurement time to 0 if the calculation is invalid."""

        temp_ref = ["PRI", "SEC"][x]

        xo_valid = frequency_plan_manager.xo_frequency_update()
        self.get_refclk_freq(x)
        self.get_all_reference_configs(x)

        if gd["{}REF_PPM_EN".format(temp_ref)].iValue == 1:

            if xo_valid:

                self.xo_freq = frequency_plan_manager.xo_freq

                if self.ref_freq[x]: # if the reference frequency is valid

                    if self.ref_ppm_invalid[x] <= 0 or self.ref_ppm_valid[x] <= 0:

                        if not bypass_warning: UpdateStatusBar("{}REF ppm valid / invalid threshold must be positive number.".format(temp_ref))
                        ppm_det_valid = False

                    elif self.ref_ppm_invalid[x] < self.ref_ppm_valid[x]:

                        if not bypass_warning: UpdateStatusBar("{}REF ppm invalid threshold cannot be lower than the valid threshold".format(temp_ref))
                        ppm_det_valid = False
                    
                    elif self.ref_freq[x] < 2e3:

                        if not bypass_warning: UpdateStatusBar("Frequency detection not supported for frequency below 2 kHz. Use 1-pps phase detector for {}REF.".format(temp_ref))
                        ppm_det_valid = False

                    else:
                        ppm_det_valid = True
                        self.ppm_meas_time[x] = self.compute_ppm_settings(x)
                else:
                    if not bypass_warning: UpdateStatusBar("Invalid {}REF frequency. Frequency detector counters not computed.".format(temp_ref))
                    ppm_det_valid = False

            else:
                if not bypass_warning: UpdateStatusBar("Invalid XO frequency. Frequency detector counters not computed.")
                ppm_det_valid = False

        else:
            ppm_det_valid = False

        if not ppm_det_valid:

            self.ppm_meas_time[x] = 0
            gd['s{}REF_PPM_TIMER'.format(temp_ref)].sValue = 'n/a'
        
    def compute_ppm_settings(self, x):                                
        """ Compute & return ref ppm register settings"""

        # Re-organizing the equations in equations doc, according to Derek's function REF_PPM2CNT (hold is the same thing with feedback)
        # in_freq = ppm_freq = ref_freq / ppm_div # Agreed
        # t_meas = hold_cntstrt / hold_freq = 1 / (accuracy * 1e-6) * in_freq / hold_freq * hold_freq / in_freq / hold_freq = 1 / (accuracy * 1e-6 * hold_freq) # Agreed
        # hold_cntstrt = t_meas * hold_freq # Agreed
        # in_cntstrt = ref_cntstrt = hold_cntstrt * in_freq / hold_freq = t_meas * in_freq # Agreed
        # lock_cnt = ref_ppm_min = 1 / (accuracy * 1e-6) * lock_ppm / accuracy = # 

        temp_ref = ["PRI", "SEC"][x]

        if not self.force_ppm_div_16 and self.ref_freq[x] <= 50e6:

            temp_ppm_div = 0     # div-1
            temp_div_refclk_freq = self.ref_freq[x]        # This is Ref*_ppm_freq

        elif not self.force_ppm_div_16 and self.ref_freq[x] <= 200e6:

            temp_ppm_div = 1     # div-4
            temp_div_refclk_freq = self.ref_freq[x] / 4     # This is Ref*_ppm_freq

        else:

            temp_ppm_div = 3     # div-16
            temp_div_refclk_freq = self.ref_freq[x] / 16    # This is Ref*_ppm_freq

        gd['{}REF_PPMDIV'.format(temp_ref)].iValue = temp_ppm_div

        lock_ppm = self.ref_ppm_valid[x]
        unlk_ppm = self.ref_ppm_invalid[x]
        avg_count = self.ref_avg_count[x]
        accuracy_ppm = self.ref_accuracy_ppm[x]
        in_freq = temp_div_refclk_freq
        hold_freq = self.xo_freq * int(OSCIN_DBLR_EN.iValue + 1) 

        in_cntstrt, hold_cntstrt, lock_cnt, unlk_cnt, ppm_meas_time = self.REFPPM2CNT(lock_ppm,unlk_ppm,avg_count,accuracy_ppm,in_freq,hold_freq)

        gd['{}REF_CNTSTRT'.format(temp_ref)].iValue = in_cntstrt
        gd['{}REF_HOLD_CNTSTRT'.format(temp_ref)].iValue = hold_cntstrt
        gd['{}REF_PPM_MIN'.format(temp_ref)].iValue = lock_cnt
        gd['{}REF_PPM_MAX'.format(temp_ref)].iValue = unlk_cnt
        gd['s{}REF_PPM_TIMER'.format(temp_ref)].sValue = ee(ppm_meas_time, "s", 2)

        return ppm_meas_time

    def REFPPM2CNT(self,lock_ppm,unlk_ppm,avg_count,accuracy_ppm,in_freq,hold_freq,counter_width=28,thresh_width=15):
        """
        this one trips when the XO input (hold_freq) counter hits zero

        Inputs:
        lock_ppm:       float   lock detect threshold
        unlk_ppm:       float   unlock detect threshold
        avg_count:      int     # averages
        accuracy_ppm:   float   lock detect required accuracy
        in_freq:        float   frequency lock target (Hz)
        hold_freq:        float   comparison frequency (Hz)   

        Returns:
        (in_cntstrt, hold_cntstrt, lock_cnt, unlk_cnt, lock_ppm, unlk_ppm, ppm_meas_time)
        """
        hold_cntstrt_tmp = Fraction("1e6") / accuracy_ppm * avg_count
        in_cntstrt = ceil(hold_cntstrt_tmp * in_freq / hold_freq)
        hold_cntstrt = round(in_cntstrt * hold_freq / in_freq)
        lock_cnt = round(avg_count * lock_ppm / accuracy_ppm)
        unlk_cnt = round(avg_count * unlk_ppm / accuracy_ppm)
        ppm_meas_time = hold_cntstrt / hold_freq
        
        return in_cntstrt, hold_cntstrt, lock_cnt, unlk_cnt, ppm_meas_time
        
    def ref_early_late_det_update(self, x, bypass_warning = False):

        temp_ref = ["PRI", "SEC"][x]

        self.get_refclk_freq(x)
        self.get_all_reference_configs(x)

        if self.ref_freq[x]:

            vco_div_freq = self.vco1_freq / 4 

            # Set 1.25 GHz REF clock if required.  TT, 2018-12-09
            high_freq = False

            for xx in [0, 1]: #['PRI', 'SEC']:

                if floor(vco_div_freq / self.ref_freq[x]) - self.early_clk_div_req_margin < self.early_clk_div_min:
                    high_freq = True

            if high_freq:
                REF_CMOS_CLK_1p25G_EN.iValue = 1
                vco_div_freq *= 2
            else:
                REF_CMOS_CLK_1p25G_EN.iValue = 0

            ## EARLY
            if gd["{}REF_EARLY_DET_EN".format(temp_ref)].iValue == 1:

                if self.ref_freq[x] >= 2e3: # Early detector only supports down to 2kHz

                    temp_early_cntr = floor(vco_div_freq / self.ref_freq[x]) - self.ref_early_margin[x] - self.early_clk_div_req_margin

                    if floor(vco_div_freq / self.ref_freq[x]) - self.early_clk_div_req_margin < self.early_clk_div_min: # If the actual clk div value is below the min, then disable the early detector
                        
                        early_det_valid = False
                        max_ref_freq = vco_div_freq / (self.early_clk_div_min + self.early_clk_div_req_margin) # For BAW VCO, the max ref frequency is 2500e6 / 12
                        if not bypass_warning: UpdateStatusBar("{}REF frequency is too high. Maximum reference frequency for early detect = {}.".format(temp_ref, ee(max_ref_freq, "Hz", 3)))
                        
                    elif temp_early_cntr < self.early_clk_div_min: # if reference frequency is low enough but the early detect margin counter is too high, then limit the counter
                        
                        early_det_valid = False
                        max_early_margin = floor(vco_div_freq / self.ref_freq[x]) - self.early_clk_div_min - self.early_clk_div_req_margin
                        if not bypass_warning: UpdateStatusBar("The early margin counter for {}REF is too high. Maxinum value = {}.".format(temp_ref, max_early_margin))

                    elif temp_early_cntr > 2**22 - 1: # REF_EARLY_CLK_DIV has 22 bits. This should never occur because that requires ref_freq = 150 Hz.
                        
                        early_det_valid = False
                        min_early_margin = floor(vco_div_freq / self.ref_freq[x]) - (2**22 - 1) - self.early_clk_div_req_margin
                        if not bypass_warning: UpdateStatusBar("The early margin counter for {}REF is too low. Minimum value = {}.".format(temp_ref, min_early_margin))

                    else:

                        early_det_valid = True
                        gd['{}REF_EARLY_CLK_DIV'.format(temp_ref)].iValue = temp_early_cntr
                        early_calc_s = 1 / self.ref_freq[x] - temp_early_cntr / vco_div_freq
                        gd["s{}REF_EARLY_calc".format(temp_ref)].sValue = ee(early_calc_s, "s", 2)

                else:

                    early_det_valid = False
                    if not bypass_warning: UpdateStatusBar("Early Detector is not available for {}REF frequency below 2 kHz.".format(temp_ref))

            else:
                early_det_valid = False

            ## MISSING (or LATE)
            if gd["{}REF_MISSCLK_EN".format(temp_ref)].iValue == 1:

                if self.ref_freq[x] >= 2e3: # Late detector only supports down to 2kHz

                    temp_late_cntr = floor(vco_div_freq / self.ref_freq[x]) * (self.ref_late[x] + 1) + self.miss_clk_div_req_margin + self.ref_late_margin[x]

                    if temp_late_cntr > 2**22 - 1:
                        
                        late_det_valid = False
                        max_late_cntr = floor((2**22 - 1 - self.miss_clk_div_req_margin) / floor(vco_div_freq / self.ref_freq[x]) ) - 1
                        max_late_margin = (2**21 - 1) - (max_late_cntr + 1) * floor(vco_div_freq / self.ref_freq[x]) - self.miss_clk_div_req_margin
                        if not bypass_warning: UpdateStatusBar("The number of missing clocks and / or the late margin for {}REF is too high.\nLimit the number of missing clocks to {} and limit the late margin to {}.".format(temp_ref, max_late_cntr, max_late_margin)                   )
                    
                    else:

                        late_det_valid = True
                        gd['{}REF_MISSCLK_DIV'.format(temp_ref)].iValue = temp_late_cntr
                        late_calc_s = temp_late_cntr / vco_div_freq - 1 / self.ref_freq[x]
                        gd["s{}REF_LATE_calc".format(temp_ref)].sValue = ee(late_calc_s, "s", 2)

                else:

                    late_det_valid = False
                    if not bypass_warning: UpdateStatusBar("Late / Missing Detector is not available for {}REF frequency below 2 kHz.".format(temp_ref))

            else:
                late_det_valid = False

        else:
            early_det_valid = False
            late_det_valid = False
            if not bypass_warning: UpdateStatusBar("Invalid {}REF frequency. Early and late clock detector counters not computed.".format(temp_ref))

        if not early_det_valid:
            gd["s{}REF_EARLY_calc".format(temp_ref)].sValue = 'n/a'

        if not late_det_valid:
            gd["s{}REF_LATE_calc".format(temp_ref)].sValue = 'n/a'

    def ref_ph_det_update(self, x, bypass_warning = False):

        temp_ref = ["PRI", "SEC"][x]

        xo_valid = frequency_plan_manager.xo_frequency_update()
        self.get_refclk_freq(x)
        self.get_all_reference_configs(x)

        if gd["{}REF_PH_VALID_EN".format(temp_ref)].iValue == 1:

            if xo_valid:

                self.xo_freq = frequency_plan_manager.xo_freq

                if self.ref_freq[x]:
                    
                    if self.ref_freq[x] <= 2e3:

                        ph_det_valid = True

                        ph_valid_cnt = self.xo_freq * int(OSCIN_DBLR_EN.iValue + 1) / self.ref_freq[x]
                        gd["{}REF_PH_VALID_CNT".format(temp_ref)].iValue = int(ph_valid_cnt)
                        gd['{}REF_PH_VALID_THR'.format(temp_ref)].iValue = self.ref_ph_valid_thr[x]

                        max_jitter = self.ref_ph_valid_thr[x] / (self.xo_freq * int(OSCIN_DBLR_EN.iValue + 1)) 
                        gd['s{}REF_PH_VALID_calc'.format(temp_ref)].sValue = ee(max_jitter, "s", 2)

                    else:
                        ph_det_valid = False
                        if not bypass_warning: UpdateStatusBar("1 PPS Phase Detector is not available for {}REF over 2 kHz.".format(temp_ref))

                else:
                    ph_det_valid = False
                    if not bypass_warning: UpdateStatusBar("Invalid {}REF frequency. 1-pps phase detector counters not computed.".format(temp_ref))

            else:
                ph_det_valid = False
                if not bypass_warning: UpdateStatusBar("Invalid XO frequency. 1-pps phase detector counters not computed.")

        else:
            ph_det_valid = False

        if not ph_det_valid:
            gd['s{}REF_PH_VALID_calc'.format(temp_ref)].sValue = 'n/a'

    def reference_validation_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "If DPLL is disabled, then skip this page. All reference validation methods have been enabled or disabled automatically based on reference frequency and interface type. However, it is highly recommended to read through the instructions and loose or tighten the thresholds according to application needs.\n\n"
        str1 += "Frequency detection and early / late window detection are only valid for reference frequencies >= 2 kHz. 1-pps phase detector is only valid for reference frequencies < 2 kHz. For 1-pps input, only enable the 1-pps phase detector and disable all other detectors.\n\n"
        str1 += "1. Validation timer. The reference must stay valid for 'validation timer' amount of time before it's considered valid. It is recommended to set the validation timer to more than twice of the total reference validation measurement time. The frequency detection measurement time is displayed on the wizard. Measurement time of amplitude detection, early / late clock detection as well as 1-pps phase detection is roughly one cycle of reference clock. Therefore, if the reference frequency is >= 2 kHz, total measurement time is approximately the frequency detection measurement time. If the reference frequency is < 2 kHz, total measuremnet time is approximately 1 cycle of reference clock.\n\n"
        str1 += "2. Amplitude detector. There are two modes: amplitude detector mode and CMOS slew rate detector mode. In amplitude detector mode, the reference is considered valid if the signal swing is higher than the selected threshold. In CMOS slew rate detector mode, the detection method can be either slew rate detection or VIH / VIL detection. For slew rate detection, the input slew rate must be faster than 0.2 V/ns. For VIH / VIL detection, the input high level must be above 1.8 V and the low level must be below 0.6 V. The amplitude detection mode cannot be used for reference frequencies less than 5 MHz. If the reference frequency is above 5 MHz, then amplitude detection mode is recommended for differential input and the CMOS slew rate detection mode is recommended for single-ended input. If the input swing is too low (for example, the LVDS voltage swing is 400 mV, very marginal compared to the mininum threshold of amplitude detection mode), then amplitude detector can be disabled.\n\n"
        str1 += "3. Frequency detector. This detector is only valid for PRIREF / SECREF frequencies >= 2 kHz. Frequency detection needs 4 parameters: valid threshold in ppm, invalid threshold in ppm, accuracy in ppm and average count. The PRIREF or SECREF is considered valid if the frequency error between PRIREF / SECREF and XO is within the valid threshold. While it's frequency valid, it's considered as frequency invalid if the frequency error exceeds the invalid threshold. The accuracy in ppm indicates how accurate the valid and invalid threshold can be. In other words, this is the resolution of valid and invalid threshold counters. The minimum average count is 2. Keep it as 2 unless the reference clock has too much wander and the DPLL loop bandwidth is too narrow. In that case, raise the average count to no more than 10. As mentioned, the '0-error' reference for frequency detection is the XO. In reality, of course, the XO frequency is not '0-error'. Therefore, the valid and invalid thresholds must take the XO ppm error into account. The minimum valid threshold should be max XO frequency error + max PRIREF / SECREF frequency error + accuracy in ppm. The minimum invalid threshold should be valid threshold + accuracy in ppm.\n\n"
        str1 += "4. Early and late clock window detector. This detector is only valid for PRIREF / SECREF frequencies >= 2 kHz. 3 parameters are needed: early counter, late counter and number of missing clocks. After setting early and late counters, the T_early and T_late are calculated accordingly. As shown in the timing diagram, the PRIREF / SECREF is considered valid if its next clock edge falls within ideal next edge - T_early and ideal next edge + T_late. Setting the number of missing clocks to x is equivalent to adding x * reference_clock_period to T_late. So the number of missing clocks is typically set to 0 unless gapped clock needs to be supported. The early and late clock detector uses divided down BAW VCO frequency as its '0-error' reference. However, since this is a very coarse detection method (resolution of T_early and T_late counter is roughly 1.6 ns), the ppm error of the BAW VCO itself is not of concern.\n\n"
        str1 += "5. 1-pps phase detector. This detector is only valid for PRIREF / SECREF frequencies < 2 kHz. T_jitter is auto-calculated according to the phase detector counter. As shown in the timing diagram, the PRIREF / SECREF is considered valid if the next clock edge falls within ideal next edge - T_jitter and ideal next edge + T_jitter. Note that the '0-error' clock reference for 1-pps phase detector is the XO, so T_jitter must be greater than the sum of: (1) XO phase error accumulated through one PRIREF / SECREF clock period (for example, 1 second for 1-pps input). This includes the phase error caused by frequency inaccuracy, accumulated jitter as well as wander. (2) The max period jitter of PRIREF / SECREF. Therefore, low frequency and high stability XO is recommended for 1-pps input. This is because for XOs with the same frequency stability, the one with the lower frequency accumulates less phase error over fixed period of time. 12.8 MHz TCXO / OCXO is recommended for 1-pps input.\n\n"
        s_wizard_reference_validation_message_box.sValue = str1

class Class_dpll_calculation_manager():

    def __init__(self):
        
        self.success = None # If there's an error in any step then this will become False. The DPLL script will only be run if this is True
        self.old_matlab = 0

        #### Below blocks are the attributes to be passed to the mat_obj
        # These are fixed attributes   
        self.dpll_type = "1" # 1 = Type 1 for 1PPS input
        self.dpll_type_include = 0 # 1 = include dpll_type parameter in matlab_inputs file # 5/13/2020 - According to JJ, do not include self.dpll_type = "1"
        self.dpll_mode = "0" # 0 = 2-loop, 1 = 3-loop type2, 2 = 3-loop type3, 3 = APLL DCO, 4=TCXO DCO
        self.dpll_pdiv_pri = "2" # 2 if VCO1 is BAW, 4 otherwise.
        self.sref_dpll = [1, 1] # 1 means assigned to DPLL1.  For LMK05x18, this is always the case for both inputs.
        self.dpll_ref_common_div = "1e9" # Don't touch it.
        # These require use input
        self.refclk_freq = [None, None] # [PRIREF frequency, SECREF frequency], 0 if that reference is disabled
        self.dpll_xo_freq = None 
        self.apll_freq = None # APLL1 PFD frequency
        self.dpll_vco_freq = None
        self.dpll_max_tdc_freq = None
        self.dpll_bandwidth = None
        self.dpll_bw_accuracy = "0.01" # Looks like this is always set to 1% in the previous script
        self.dpll_peak_tr = None # Allowed peaking error for DPLL transfer function. Default to 0.1 dB
        self.dpll_peak_er = None # Allowed peaking error for DPLL error function. Default to 1 dB
        self.tr_fn_offset = "100" # This feature is hidden from customer
        self.err_fn_offset = "1" # This feature is hidden from customer
        self.dpll_refclk_pri = None #"[DPLL_PRIREF_AUTO_PRTY.iValue, DPLL_SECREF_AUTO_PRTY.iValue]"

        # 5/13/2020 - JJ confirmed that these three entries will not affect anything. So they can be defaulted to any value.
        self.dpll_dco_ppb = "0.001"
        self.dpll_ppm_settings = "[1, 10]"
        self.dpll_max_meas_time = str(0.05)

        # File name and path
        self.old_matlab = False # Whether or not old matlab gen.exe is used.
        self.exefile = device_path + r'\LMK05X18_ROM_Gen.exe'
        self.exefile_old = device_path + r'\LMK05X18_ROM_Gen_old.exe'
        self.matfile = device_path + r'\matlab_inputs_lmk05x18.m'
        self.outfile = device_path + r'\romdump.txt'
        self.outfile_old = device_path + r'\romdump_old.txt'

        # Returned by matlab script
        self.dpll_vco_freq_returned = None

        # Actual TDC frequency
        self.tdc_freq = None

        # Written DPLL registers
        self.written_dpll_reg = OrderedDict()

    def get_refclk_freq(self):
        """Use the reference_validation_manager's method to get reference frequencies."""

        for x in range(2):
            dpll_reference_manager.get_refclk_freq(x)

        self.refclk_freq = dpll_reference_manager.ref_freq

        if self.refclk_freq == [0, 0]:
            return False

        elif self.refclk_freq[0] and self.refclk_freq[1]: # if both frequencies are not 0 or None

            temp_gcd = gcd(self.refclk_freq[0], self.refclk_freq[1])
            max_r_div = max(self.refclk_freq[0] / temp_gcd, self.refclk_freq[1] / temp_gcd)

            if max_r_div > 2**16 - 1:
                return False

        return True

    def get_dpll_vco_freq(self):

        self.dpll_vco_freq = frequency_plan_manager.vco1_freq  

        return bool(self.dpll_vco_freq) # False if frequency_plan_manager.vco1_freq is None or 0

    def max_tdc_freq_update(self):

        if combo_set_max_tdc_freq.iValue == 0: # If max TDC frequency is set by default, then force the value to 26MHz
            sMAX_TDC_freq.dValue = 26e6
        else: # if manually override max TDC frequency, then set the limit the frequency range to 1Hz ~ 26MHz

            if sMAX_TDC_freq.dValue < 1:
                sMAX_TDC_freq.dValue = 1

            elif sMAX_TDC_freq.dValue > 26e6:
                sMAX_TDC_freq.dValue = 26e6

    def dpll_loop_bandwidth_update(self):

        # Limit the bw to 0.01Hz ~ 4kHz
        if sDPLL_LBW.dValue < 0.01: # min loop bandwidth = 0.01 Hz
            sDPLL_LBW.dValue = 0.01
        elif sDPLL_LBW.dValue > 4000: # max loop bandwidth = 4 kHz
            sDPLL_LBW.dValue = 4000

        for x in self.refclk_freq:

            if x != 0 and sDPLL_LBW.dValue >= x:

                s_wizard_dpll_message_box.sValue = "Error!\n\nDPLL loop bandwidth must be less than PRIREF/SECREF frequency."
                return False 

        # For 1pps recommended bw = 0.01Hz
        if 1 in self.refclk_freq and sDPLL_LBW.dValue != 0.01:
            s_wizard_dpll_message_box.sValue = "Warning!\n\nIt is recommended to set DPLL loop bandwidth = 0.01 Hz for 1-pps reference input."

        return True 

    def calculate_tdc_freq(self):
        """Reference frequency must be obtained outside of this scope"""

        for x in range(2):

            if self.refclk_freq[x]:
                self.tdc_freq = self.refclk_freq[x] / int(gd["DPLL_{}REF_RDIV".format(["PRI", "SEC"][x])].iValue)

    def calc_fastlock_time(self, x):
        """x = 1: fastlock 1. x = 2: fastlock 2
        Make sure that there's at least one valid reference every time this method is called.
        TDC frequency also needs to be calculated outside of this method"""
        
        mantissa = gd['DPLL_REF_TMR_FL{}'.format(x)].iValue >> 5
        exponent = gd['DPLL_REF_TMR_FL{}'.format(x)].iValue & ((2**5) -1)

        locktime = mantissa * (2**exponent)
        locktime = locktime / self.tdc_freq

        return locktime

    def get_user_input(self):
        """Return True if all inputs are valid. False otherwise"""

        s_wizard_dpll_message_box.sValue = ""

        # Get reference frequencies
        ref_valid = self.get_refclk_freq()

        if not ref_valid: # if both references are invalid

            s_wizard_dpll_message_box.sValue = "Error!\n\nInvalid reference. Go back to the reference page and set PRIREF and / or SECREF correctly."
            return False

        # Get XO, PFD and VCO1 frequency
        xo_valid = frequency_plan_manager.xo_frequency_update()

        if xo_valid:

            self.xo_freq = frequency_plan_manager.xo_freq
            self.apll_freq = self.xo_freq * int(OSCIN_DBLR_EN.iValue + 1) / int(OSCIN_RDIV.iValue + 1)

            vco_valid = self.get_dpll_vco_freq()

            if not vco_valid:

                str1 = "Error!\n\nInvalid VCO1 frequency. Possible reasons:\n\n"
                str1 += "1. Frequency plan has been skipped. Frequency plan must be set before running DPLL script. This is because the DPLL script needs the exact VCO1 frequency generated by frequency plan.\n\n"
                str1 += "2. If the frequency plan was set but this message still shows up, it is likely that the APLL1 is powered down because no output is assigned to APLL1, and APLL2 uses XO as reference. In that case, go to the first wizard page and change the PLL2 reference to 'VCO1 - Cascaded Mode'."
                s_wizard_dpll_message_box.sValue = str1
                return False

        else:
            s_wizard_dpll_message_box.sValue = "Error!\n\nInvalid XO frequency."
            return False

        # Get max TDC frequency
        self.max_tdc_freq_update() # Update max TDC frequency
        self.dpll_max_tdc_freq = sMAX_TDC_freq.dValue

        # Get reference priority setting
        self.dpll_refclk_pri = "[{}, {}]".format(DPLL_PRIREF_AUTO_PRTY.iValue, DPLL_SECREF_AUTO_PRTY.iValue)

        # Get DPLL loop bandwidth and peaking
        bw_valid = self.dpll_loop_bandwidth_update()

        if bw_valid:

            self.dpll_bandwidth = sDPLL_LBW.dValue
            self.dpll_peak_tr = sDPLL_PEAK.dValue 
            self.dpll_peak_er = sDPLL_PEAK_ERROR.dValue  
        
        else:
            return False

        return True

    def generate_matlab_input_file(self):

        mat_obj.dpll_type[0] = self.dpll_type
        mat_obj.dpll_type_include[0] = self.dpll_type_include
        mat_obj.dpll_mode[0] = self.dpll_mode
        mat_obj.dpll_pdiv_pri[0] = self.dpll_pdiv_pri
        mat_obj.dpll_ref_common_div[0] = self.dpll_ref_common_div

        mat_obj.dpll_xo_freq[0] = str(self.xo_freq)
        mat_obj.apll_freq[0] = str(self.apll_freq)
        mat_obj.dpll_vco_freq[0] = str(self.dpll_vco_freq)
        mat_obj.dpll_max_tdc_freq[0] = str(self.dpll_max_tdc_freq)
        mat_obj.dpll_bandwidth[0] = str(self.dpll_bandwidth)
        mat_obj.dpll_bw_accuracy[0] = str(self.dpll_bw_accuracy)
        mat_obj.dpll_peak[0] = str(self.dpll_peak_tr)
        mat_obj.dpll_peak_error[0] = str(self.dpll_peak_er)
        mat_obj.dpll_refclk_pri[0] = self.dpll_refclk_pri
        
        mat_obj.tr_fn_offset[0] = self.tr_fn_offset
        mat_obj.err_fn_offset[0] = self.err_fn_offset
   
        mat_obj.dpll_dco_ppb[0] = self.dpll_dco_ppb
        mat_obj.dpll_ppm_settings[0] = self.dpll_ppm_settings
        mat_obj.dpll_max_meas_time[0] = self.dpll_max_meas_time
        
        mat_obj.set_dpll_refclk(refclk_freq = [str(x) for x in self.refclk_freq], refclk_asst = self.sref_dpll) 
        mat_obj.WriteMatlab(filename = self.matfile, old_matlab = self.old_matlab)

    def import_rom(self, filename):
        """Update registers and GUI files from the matlab_gen.exe output file. ref_valid needs to be checked outside of the scope"""

        if os.path.exists(filename):
            
            global romdata
            romdata = rom_obj()             # initialize new rom object
            romdata.ReadRomFile(filename)      # Read imported ROM file
            
            # Get DPLL loop bandwidth            
            dpll_lbw = romdata.rom_header.get('DPLL1_BW3DB_ACT')

            if dpll_lbw:
                dpll_lbw = '{:.4g}'.format(float(dpll_lbw))
            else:
                dpll_lbw = '0'
                
            sDPLL_LBW_ACT.dValue = float(dpll_lbw)       
            
            # Write registers from rom file
            unwanted_list = ["DPLL_PRIREF_AUTO_PRTY", "DPLL_SECREF_AUTO_PRTY", "DPLL_REF_HIST_EN", "DPLL_REF_HIST_HOLD", "DPLL_REF_HISTCNT", "DPLL_REF_HISTDLY", "DPLL_REF_HIST_INTMD", "DPLL_TUNING_FREE_RUN"]

            for key, value in romdata.rom_regdata.items():

                if "DPLL1" in key:
                    key = re.sub("DPLL1", "DPLL", key)

                if "REF0" in key:
                    key = re.sub("REF0", "PRIREF", key)

                if "REF1" in key:
                    key = re.sub("REF1", "SECREF", key)

                if key not in unwanted_list:

                    gd[key].iValue = value
                    self.written_dpll_reg[key] = value

            # Calculate TDC frequency
            self.calculate_tdc_freq()
                
            if self.tdc_freq >= 4.0e6:        # Was >, 2018-09-12
                DPLL_REF_TDC_HI_SPD_MODE.iValue = 1 # Added this register to sep.ini - HZ 5-10-2020
            else:
                DPLL_REF_TDC_HI_SPD_MODE.iValue = 0
            
            cycslip_offset = (self.dpll_vco_freq * 2 / self.tdc_freq)
            DPLL_REF_CYCSLIP_OFFSET.iValue = int(cycslip_offset)

            DPLL_TDC_REF_DLY_GEN_EN.iValue = 1 # 5/12/2020. According to JJ, always set this bit to 1.

            RAMP_CNTR_EN.iValue = 1 # Added 5/13/2020.

            DPLL_TDC_DLY_STEPSIZE2X_EN.iValue = 1

            self.written_dpll_reg["DPLL_REF_TDC_HI_SPD_MODE"] = DPLL_REF_TDC_HI_SPD_MODE.iValue
            self.written_dpll_reg["DPLL_REF_CYCSLIP_OFFSET"] = DPLL_REF_CYCSLIP_OFFSET.iValue
            self.written_dpll_reg["DPLL_TDC_REF_DLY_GEN_EN"] = DPLL_TDC_REF_DLY_GEN_EN.iValue
            self.written_dpll_reg["RAMP_CNTR_EN"] = RAMP_CNTR_EN.iValue
            self.written_dpll_reg["DPLL_TDC_DLY_STEPSIZE2X_EN"] = DPLL_TDC_DLY_STEPSIZE2X_EN.iValue

            DPLL_REF_AVOID_SLIP.iValue = 1     # always set

            self.written_dpll_reg["DPLL_REF_AVOID_SLIP"] = DPLL_REF_AVOID_SLIP.iValue
             
            # Check to ensure all DPLL_REF_FILT_GAIN values are zero or non-zero.  If not, make all non-zero.
            isZero0 = 0 == DPLL_REF_FILT_GAIN.iValue
            isZero1 = 0 == DPLL_REF_FILT_GAIN_FL1.iValue
            isZero2 = 0 == DPLL_REF_FILT_GAIN_FL2.iValue
             
            if isZero0 + isZero1 + isZero2 == 1 or isZero0 + isZero1 + isZero2 == 2:
                # Fail condition above.  Need to make some values non-zero.
                
                if isZero0:
                    DPLL_REF_FILT_GAIN.iValue = 31
                    DPLL_REF_QUANT.iValue += 1
                    
                if isZero1:
                    DPLL_REF_FILT_GAIN_FL1.iValue = 31
                    DPLL_REF_QUANT_FL1.iValue += 1
            
                if isZero2:
                    DPLL_REF_FILT_GAIN_FL2.iValue = 31
                    DPLL_REF_QUANT_FL2.iValue += 1

            self.written_dpll_reg["DPLL_REF_FILT_GAIN"] = DPLL_REF_FILT_GAIN.iValue
            self.written_dpll_reg["DPLL_REF_QUANT"] = DPLL_REF_QUANT.iValue
            self.written_dpll_reg["DPLL_REF_FILT_GAIN_FL1"] = DPLL_REF_FILT_GAIN_FL1.iValue
            self.written_dpll_reg["DPLL_REF_QUANT_FL1"] = DPLL_REF_QUANT_FL1.iValue
            self.written_dpll_reg["DPLL_REF_FILT_GAIN_FL2"] = DPLL_REF_FILT_GAIN_FL2.iValue
            self.written_dpll_reg["DPLL_REF_QUANT_FL2"] = DPLL_REF_QUANT_FL2.iValue
                
            # Now check for DPLL_REF_LOOP_GAIN, TT and CS 2019-07-03.
            # Note: This is impacting (increasing) loop bandwidth and is not accounted for in the calculations.
            # Check to ensure all DPLL_REF_FILT_GAIN values are zero or non-zero.  If not, make all non-zero.
            isZero0 = 0 == DPLL_REF_LOOP_GAIN.iValue
            isZero1 = 0 == DPLL_REF_LOOP_GAIN_FL1.iValue
            isZero2 = 0 == DPLL_REF_LOOP_GAIN_FL2.iValue
             
            if isZero0 + isZero1 + isZero2 == 1 or isZero0 + isZero1 + isZero2 == 2:
                # Fail condition above.  Need to make some values non-zero.
                
                if isZero0:
                    DPLL_REF_LOOP_GAIN.iValue = 31
                    
                if isZero1:
                    DPLL_REF_LOOP_GAIN_FL1.iValue = 31
            
                if isZero2:
                    DPLL_REF_LOOP_GAIN_FL2.iValue = 31

            self.written_dpll_reg["DPLL_REF_LOOP_GAIN"] = DPLL_REF_LOOP_GAIN.iValue
            self.written_dpll_reg["DPLL_REF_LOOP_GAIN_FL1"] = DPLL_REF_LOOP_GAIN_FL1.iValue
            self.written_dpll_reg["DPLL_REF_LOOP_GAIN_FL2"] = DPLL_REF_LOOP_GAIN_FL2.iValue

            DPLL_REF_ORDER.iValue = 3
            DPLL_REF_MASHSEED.iValue = 0

            self.written_dpll_reg["DPLL_REF_ORDER"] = DPLL_REF_ORDER.iValue
            self.written_dpll_reg["DPLL_REF_MASHSEED"] = DPLL_REF_MASHSEED.iValue

            DPLL_REF_NUM_calculated.sValue = str(DPLL_REF_NUM.iValue)

            DPLL_REF_LPF0_GAIN2_FL.iValue = 30
            DPLL_REF_LPF1_GAIN2_FL.iValue = 30

            self.written_dpll_reg["DPLL_REF_LPF0_GAIN2_FL"] = DPLL_REF_LPF0_GAIN2_FL.iValue
            self.written_dpll_reg["DPLL_REF_LPF1_GAIN2_FL"] = DPLL_REF_LPF1_GAIN2_FL.iValue

            # set fastlock timers to 0 for 1pps
            if 1 in self.refclk_freq:
                combo_disable_fastlock.iValue = 1

            self.disable_fastlock_update()

            combo_switching_method_Update()

            self.written_dpll_reg["DPLL_FASTLOCK_ALWAYS"] = DPLL_FASTLOCK_ALWAYS.iValue
            self.written_dpll_reg["DPLL_SWITCHOVER_ALWAYS"] = DPLL_SWITCHOVER_ALWAYS.iValue

    def disable_fastlock_update(self):

        if combo_disable_fastlock.iValue: # fastlock is disabled

            DPLL_REF_TMR_FL1.iValue = 0
            DPLL_REF_TMR_FL2.iValue = 0

            DPLL_REF_FILT_GAIN_FL1.iValue = DPLL_REF_FILT_GAIN.iValue
            DPLL_REF_FILT_GAIN_FL2.iValue = DPLL_REF_FILT_GAIN.iValue

            DPLL_REF_LOOP_GAIN_FL1.iValue = DPLL_REF_LOOP_GAIN.iValue
            DPLL_REF_LOOP_GAIN_FL2.iValue = DPLL_REF_LOOP_GAIN.iValue

            DPLL_REF_LPF0_GAIN_FL1.iValue = DPLL_REF_LPF0_GAIN.iValue
            DPLL_REF_LPF0_GAIN_FL2.iValue = DPLL_REF_LPF0_GAIN.iValue

            DPLL_REF_LPF1_GAIN_FL1.iValue = DPLL_REF_LPF1_GAIN.iValue
            DPLL_REF_LPF1_GAIN_FL2.iValue = DPLL_REF_LPF1_GAIN.iValue

            DPLL_REF_QUANT_FL1.iValue = DPLL_REF_QUANT.iValue
            DPLL_REF_QUANT_FL2.iValue = DPLL_REF_QUANT.iValue

        else:
            pass

        self.written_dpll_reg["DPLL_REF_TMR_FL1"] = DPLL_REF_TMR_FL1.iValue
        self.written_dpll_reg["DPLL_REF_TMR_FL2"] = DPLL_REF_TMR_FL2.iValue

        self.written_dpll_reg["DPLL_REF_FILT_GAIN_FL1"] = DPLL_REF_FILT_GAIN_FL1.iValue
        self.written_dpll_reg["DPLL_REF_FILT_GAIN_FL2"] = DPLL_REF_FILT_GAIN_FL2.iValue

        self.written_dpll_reg["DPLL_REF_LOOP_GAIN_FL1"] = DPLL_REF_LOOP_GAIN_FL1.iValue
        self.written_dpll_reg["DPLL_REF_LOOP_GAIN_FL2"] = DPLL_REF_LOOP_GAIN_FL2.iValue

        self.written_dpll_reg["DPLL_REF_LPF0_GAIN_FL1"] = DPLL_REF_LPF0_GAIN_FL1.iValue
        self.written_dpll_reg["DPLL_REF_LPF0_GAIN_FL2"] = DPLL_REF_LPF0_GAIN_FL2.iValue

        self.written_dpll_reg["DPLL_REF_LPF1_GAIN_FL1"] = DPLL_REF_LPF1_GAIN_FL1.iValue
        self.written_dpll_reg["DPLL_REF_LPF1_GAIN_FL2"] = DPLL_REF_LPF1_GAIN_FL2.iValue

        self.written_dpll_reg["DPLL_REF_QUANT_FL1"] = DPLL_REF_QUANT_FL1.iValue
        self.written_dpll_reg["DPLL_REF_QUANT_FL2"] = DPLL_REF_QUANT_FL2.iValue

    def update_message_box(self):

        # Get VCO frequency. I'm not sure if the VCO frequency calculated by the DPLL script is accurate. So I'll redo the calc under Fraction() system         
        fixed_div = 2
        pre_div = int(DPLL_REF_FB_PRE_DIV.iValue + 2)
        int_div = int(DPLL_REF_FB_DIV.iValue)
        num = int(DPLL_REF_NUM.iValue)

        if DPLL_REF_DEN.iValue != 0:
            den = int(DPLL_REF_DEN.iValue)
        else:
            den = 2**40

        self.dpll_vco_freq_returned = self.tdc_freq * fixed_div * pre_div * (int_div + Fraction(num, den))
        VCO1_freq.dValue = round(self.dpll_vco_freq_returned / 1e6, 15)

        # calculate PLL1 frequency error
        vco1_freq_error = float((self.dpll_vco_freq_returned - self.dpll_vco_freq) / self.dpll_vco_freq * 1e9)

        if vco1_freq_error == 0:
            vco1_freq_error = "0"
        else:
            vco1_freq_error = "{:.2e}".format(vco1_freq_error)

        # calculate lock time
        fl1_locktime = self.calc_fastlock_time(1)
        fl2_locktime = self.calc_fastlock_time(2)

        # update message box
        message = "DPLL calculation completed!\n\n"
        message += "TDC frequency = {}\n\n".format(self.tdc_freq)
        message += "Planned VCO1 frequency = {}\n".format(self.dpll_vco_freq)
        message += "Actual VCO1 frequency = {}\n".format(self.dpll_vco_freq_returned)
        message += "PLL1 frequency error = {} ppb\n\n".format(vco1_freq_error)

        # calculate PLL2 frequency error
        if PLL2_PDN.iValue == 0: # if PLL2 is enabled

            frequency_update_manager.apll1_vco_frequency = self.dpll_vco_freq_returned
            frequency_update_manager.update_apll2_reference()
            frequency_update_manager.update_vco2_frequency()

            vco2_freq_target = frequency_plan_manager.vco2_freq
            vco2_freq_actual = frequency_update_manager.apll2_vco_frequency
            vco2_freq_error = float((vco2_freq_actual - vco2_freq_target) / vco2_freq_target * 1e9)

            if vco2_freq_error == 0:
                vco2_freq_error = "0"
            else:
                vco2_freq_error = "{:.2e}".format(vco2_freq_error)

            message += "Planned VCO2 frequency = {}\n".format(vco2_freq_target)
            message += "Actual VCO2 frequency = {}\n".format(vco2_freq_actual)
            message += "PLL2 frequency error = {} ppb\n\n".format(vco2_freq_error)

        message += "DPLL fastlock time = {} + {} = {}\n\n".format(ee(fl1_locktime, "s", 2), ee(fl2_locktime, "s", 2), ee(fl1_locktime + fl2_locktime, "s", 2))
        message += "Total DPLL frequency lock time ~= DPLL fastlock time + 2 * DPLL frequency lock measurement time\n\n"
        message += "Total DPLL phase lock time ~= DPLL fastlock time + 2 * DPLL phase lock measurement time"

        if cb_show_dpll_registers.iValue == 1:

            message = "Written DPLL registers:\n\n"

            for key, value in self.written_dpll_reg.items():

                message += "{0} = {1}\n".format(key, value)

        s_wizard_dpll_message_box.sValue = message

        self.update_dpll_block_diagram()

    def update_dpll_block_diagram(self):

        ref_valid = self.get_refclk_freq()

        if ref_valid:

            temp_tdc_freq = [0, 0]

            for x in range(2):

                temp_ref = ["PRI", "SEC"][x]

                if self.refclk_freq[x]:
                    temp_tdc_freq[x] = self.refclk_freq[x] / int(gd["DPLL_{}REF_RDIV".format(temp_ref)].iValue)

            if self.refclk_freq[0] and self.refclk_freq[1] and temp_tdc_freq[0] != temp_tdc_freq[1]:
                UpdateStatusBar("PRIREF divided by PRIREF_R_DIV must be equal to SECREF divided by SECREF_R_div.\nDPLL block diagram not updated.") 
            else:
                self.tdc_freq = max(temp_tdc_freq)

                sPRIREF_freq_display.sValue = ee(self.refclk_freq[0], "Hz", 4)
                sSECREF_freq_display.sValue = ee(self.refclk_freq[1], "Hz", 4)
                sTDC_freq.sValue = ee(self.tdc_freq, "Hz", 6)

                fixed_div = 2
                pre_div = int(DPLL_REF_FB_PRE_DIV.iValue + 2)
                int_div = int(DPLL_REF_FB_DIV.iValue)
                num = int(DPLL_REF_NUM.iValue)

                if DPLL_REF_DEN.iValue != 0:
                    den = int(DPLL_REF_DEN.iValue)
                else:
                    den = 2**40

                self.dpll_vco_freq_returned = self.tdc_freq * fixed_div * pre_div * (int_div + Fraction(num, den))

                DPLL_VCO_freq.dValue = round(self.dpll_vco_freq_returned/1e6, 17)

                temp_ref = ["PRI", "SEC"]

                for x in range(2):

                    dim_or_undim = UnDim

                    if not self.refclk_freq[x]:

                        gd["s{}REF_freq_display".format(temp_ref[x])].sValue = "Disabled"
                        dim_or_undim = Dim

                    dim_or_undim(gd["DPLL_{}REF_RDIV".format(temp_ref[x])])

        else:
            UpdateStatusBar("Invalid PRIREF / SECREF. Go back to the reference page for error messages.")

    def run_script_update(self):

        if combo_dpll_mode.iValue == 0: # if DPLL is enabled

            input_valid = self.get_user_input()

            if input_valid:

                self.generate_matlab_input_file()

                if self.old_matlab:
                    cmd_args = [self.exefile_old, self.matfile, self.outfile]
                else:
                    cmd_args = [self.exefile, self.matfile, self.outfile]
                
                subprocess.call(cmd_args)
                self.import_rom(self.outfile)
                self.update_message_box()

                self.default_baw_lock_controls()
                self.default_dpll_frequency_lock_controls()
                self.recommend_dpll_phase_lock_controls()
                self.recommend_dpll_hist_controls()


        else:
            s_wizard_dpll_message_box.sValue = "Error!\n\nThe DPLL has been disabled. Enable DPLL first."

    def compute_frequency_lock_settings(self, threshold_ppm, threshold_ppm_num_bits, avg_count, reqd_ppm_accy, lockdet_meas_time, ref_freq, vco_div_freq, threshold_ppm_max_reg=None, cntstrt_ctrl=None, cntstrt_vco_ctrl=None):
        """
        threshold_ppm:      float FlexControl:  control specifying the threshold.
        threshold_ppm_num_bits:  int            how many bits are in  the threshold ppm.
        avg_count:          int FlexControl:    control specifying how many averages.  (SpinButton is a good recommendation)
        reqd_ppm_accy:      float FlexControl:  control specifying required accuracy.
        lockdet_meas_time:  string FlexControl: control showing resulting time for measurement.
        ref_freq (Hz):      float               the frequency that PPM calculation is made with respect to
        vco_div_freq (Hz)   float               update frequency rate


        If not None, then this control will be updated.
        threshold_ppm_max_reg: int FlexCtrl:    control of this bits. (Often a user control)
        cntstrt_ctrl:       int FlexCtrl:       
        cntstrt_vco_ctrl:   int FlexCtrl:       
        
        Returns:
        (cntstrt, cntstrt_vco)
        """
    
        cntstrt_vco_tmp = Fraction("1e6") / Fraction(str(reqd_ppm_accy.dValue)) * int(avg_count.iValue)
        cntstrt = ceil(cntstrt_vco_tmp * ref_freq / vco_div_freq)
        cntstrt_vco = round(cntstrt * vco_div_freq / ref_freq)
        threshold_ppm_max = min(round(avg_count.iValue * threshold_ppm.dValue / reqd_ppm_accy.dValue), 2**threshold_ppm_num_bits)
        threshold_ppm.dValue = threshold_ppm_max / int(avg_count.iValue) * reqd_ppm_accy.dValue
        lockdet_meas_time.sValue = ee(cntstrt_vco / vco_div_freq, "s", 4)

        if threshold_ppm_max_reg:
            threshold_ppm_max_reg.iValue = int(threshold_ppm_max)
        
        if cntstrt_ctrl:
            cntstrt_ctrl.iValue = cntstrt
        
        if cntstrt_vco_ctrl:
            cntstrt_vco_ctrl.iValue = cntstrt_vco
        
        return threshold_ppm_max, cntstrt, cntstrt_vco  

    def update_baw_lock_controls(self):

        xo_valid = frequency_plan_manager.xo_frequency_update()
        vco_valid = self.get_dpll_vco_freq() # updates self.dpll_vco_freq

        if vco_valid: # If a VCO1 frequency is calculated then use that
            temp_vco_freq = self.dpll_vco_freq
        else: # Otherwise just use 2.5GHz for calculation.
            temp_vco_freq = 2500e6

        vco_div_freq = temp_vco_freq / 24           # 24 is the internal fixed divider value for BAW

        if xo_valid:

            apll1_pfd_freq = frequency_plan_manager.xo_freq * int(1 + OSCIN_DBLR_EN.iValue)

            self.compute_frequency_lock_settings(BAW_LOCK_PPM_THRESH, 15, sBAW_LOCK_AVG, sBAW_LOCK_ACCURACY_PPM, sBAW_TMEAS_LOCK_calc, apll1_pfd_freq, vco_div_freq, BAW_LOCK_PPM_MAX, BAW_LOCK_CNTSTRT, BAW_LOCK_VCO_CNTSTRT)
            self.compute_frequency_lock_settings(BAW_UNLK_PPM_THRESH, 15, sBAW_LOCK_AVG, sBAW_LOCK_ACCURACY_PPM, sBAW_TMEAS_LOCK_calc, apll1_pfd_freq, vco_div_freq, BAW_UNLK_PPM_MAX, BAW_UNLK_CNTSTRT, BAW_UNLK_VCO_CNTSTRT)

        else:
            sBAW_TMEAS_LOCK_calc.sValue = "n/a"
            UpdateStatusBar("Invalid XO frequency. BAW lock detect not calculated.")

    def default_baw_lock_controls(self):
        
        BAW_LOCK_PPM_THRESH.dValue = 5
        BAW_UNLK_PPM_THRESH.dValue = 10
        sBAW_LOCK_AVG.iValue = 2
        sBAW_LOCK_ACCURACY_PPM.dValue = 1

        if combo_dpll_mode.iValue == 0:
            BAW_LOCKDET_EN.iValue = 0
        else:
            BAW_LOCKDET_EN.iValue = 1

        BAW_LOCKDET_EN_Update_UI()

        self.update_baw_lock_controls()

    def update_dpll_frequency_lock_controls(self):

        ref_valid = self.get_refclk_freq() # updates self.refclk_freq
        vco_valid = self.get_dpll_vco_freq() # updates self.dpll_vco_freq

        if vco_valid: # If a VCO1 frequency is calculated then use that
            temp_vco_freq = self.dpll_vco_freq
        else: # Otherwise just use 2.5GHz for calculation.
            temp_vco_freq = Fraction("2500e6")

        if ref_valid: # if reference is valid, then do the calculation

            self.calculate_tdc_freq() # If ref_valid is true then the TDC frequency will also be valid.

            vco_div_freq = temp_vco_freq / 24           # 24 is the internal fixed divider value for BAW
            self.compute_frequency_lock_settings(sDPLL_LOCK_PPM, 15, sDPLL_LOCK_AVG, sDPLL_LOCK_ACCURACY_PPM, sDPLL_TMEAS_LOCK_calc, self.tdc_freq, vco_div_freq, DPLL_REF_LOCKDET_PPM_MAX, DPLL_REF_LOCKDET_CNTSTRT, DPLL_REF_LOCKDET_VCO_CNTSTRT)
            self.compute_frequency_lock_settings(sDPLL_UNLK_PPM, 15, sDPLL_LOCK_AVG, sDPLL_LOCK_ACCURACY_PPM, sDPLL_TMEAS_LOCK_calc, self.tdc_freq, vco_div_freq, DPLL_REF_UNLOCKDET_PPM_MAX) # CNSTRT and VCO_CNTSTRT are used for both lock and unlock in LMK05318B

            ## This is for backward compatibility: 
            # DPLL_REF_UNLOCKDET_VCO_CNTSTRT[29:8] = DPLL_REF_UNLOCKDET_VCO_CNTSTRT_MSB_22, DPLL_REF_UNLOCKDET_VCO_CNTSTRT[7:0] = PLL1_24b_NUM_MSB
            # DPLL_REF_UNLOCKDET_CNTSTRT[29:24] = DPLL_REF_UNLOCKDET_CNTSTRT_MSB_6, DPLL_REF_UNLOCKDET_CNTSTRT[23:0] = PLL2_DEN

            if combo_backward_compatible.iValue == 1: # if backward compatible

                DPLL_REF_UNLOCKDET_CNTSTRT_MSB_6.iValue = DPLL_REF_LOCKDET_CNTSTRT.iValue >> 24
                PLL2_DEN.iValue = DPLL_REF_LOCKDET_CNTSTRT.iValue & 2**24 - 1
                DPLL_REF_UNLOCKDET_VCO_CNTSTRT_MSB_22.iValue = DPLL_REF_LOCKDET_VCO_CNTSTRT.iValue >> 8
                PLL1_24b_NUM_MSB.iValue = DPLL_REF_LOCKDET_VCO_CNTSTRT.iValue & 2**8 - 1

        else: 
            DPLL_TMEAS_LOCK_calc.sValue = "n/a"
            UpdateStatusBar("Invalid PRIREF and SECREF. DPLL frequency lock calculation requires at least one valid reference.")

    def default_dpll_frequency_lock_controls(self):

        sDPLL_LOCK_PPM.dValue = 1
        sDPLL_UNLK_PPM.dValue = 10
        sDPLL_LOCK_AVG.iValue = 10
        sDPLL_LOCK_ACCURACY_PPM.dValue = 1

        DPLL_LOCKDET_PPM_EN.iValue = 1

        self.update_dpll_frequency_lock_controls()

    def update_dpll_phase_lock_controls(self):

        vco_valid = self.get_dpll_vco_freq() # updates self.dpll_vco_freq
        ref_valid = self.get_refclk_freq()
        self.calculate_tdc_freq()

        if vco_valid: # If a VCO1 frequency is calculated then use that
            temp_vco_freq = self.dpll_vco_freq
        else: # Otherwise just use 2.5GHz for calculation.
            temp_vco_freq = Fraction("2500e6")

        pl_lock = int(DPLL_PL_LOCK_THRESH.iValue)
        pl_unlk = int(DPLL_PL_UNLK_THRESH.iValue)

        if pl_unlk <= pl_lock:

            DPLL_PL_UNLK_THRESH.iValue = pl_lock + 1
            pl_unlk = int(DPLL_PL_UNLK_THRESH.iValue)

        scalar = int(DPLL_REF_FILT_SCALAR.iValue)

        if scalar == 0:

            DPLL_PL_LOCK_calc.sValue = 'n/a'
            DPLL_PL_UNLK_calc.sValue = 'n/a'
            UpdateStatusBar('DPLL REF loop is not configured.  Cannot compute phase lock threshold.')
        
        else:

            if ref_valid:

                x_pl_lock = (2**pl_lock) / (2**int(32 - DPLL_REF_LOOP_GAIN.iValue) * scalar *  2 * temp_vco_freq)
                x_pl_unlk = (2**pl_unlk) / (2**int(32 - DPLL_REF_LOOP_GAIN.iValue) * scalar *  2 * temp_vco_freq)
                
                # calculate lock / unlock threshold in seconds
                DPLL_PL_LOCK_calc.sValue = ee(x_pl_lock, "s", 2)
                DPLL_PL_UNLK_calc.sValue = ee(x_pl_unlk, "s", 2)

                # calculate phase lock timer values
                DPLL_Loop_Filter_Freq = self.tdc_freq / int(DPLL_REF_DECIMATION.iValue + 1)
                Timer = DPLL_Loop_Filter_Freq / sDPLL_LBW_ACT.dValue

                exponent = max(0, floor(log(Timer, 2)) - 4)
                mantissa = round(Timer / 2**exponent)
                DPLL_REF_TMR_LCK.iValue = mantissa * 2**5 + exponent

                sDPLL_PL_MEAS_TIME.sValue = ee(Timer / DPLL_Loop_Filter_Freq, "s", 2)

            else:

                DPLL_PL_LOCK_calc.sValue = 'n/a'
                DPLL_PL_UNLK_calc.sValue = 'n/a'
                UpdateStatusBar("No valid reference. DPLL phase lock not computed.")

    def recommend_dpll_phase_lock_controls(self):

        DPLL_PL_LPF_COEFF.iValue = 0 # Set the coefficient to 0 unless it's decided that this needs to be disclosed to customers.

        # calculate margin for 1pps
        vco_valid = self.get_dpll_vco_freq() # updates self.dpll_vco_freq
        ref_valid = self.get_refclk_freq()
        self.calculate_tdc_freq()

        if vco_valid: # If a VCO1 frequency is calculated then use that
            temp_vco_freq = self.dpll_vco_freq
        else: # Otherwise just use 2.5GHz for calculation.
            temp_vco_freq = Fraction("2500e6")

        scalar = int(DPLL_REF_FILT_SCALAR.iValue)
        gain = int(DPLL_REF_LOOP_GAIN.iValue)

        if scalar == 0:

            DPLL_PL_LOCK_calc.sValue = 'n/a'
            DPLL_PL_UNLK_calc.sValue = 'n/a'
            UpdateStatusBar('DPLL REF loop is not configured.  Cannot compute phase lock threshold.')
        
        else:

            if ref_valid:

                if 1 in self.refclk_freq:

                    lock_thresh = 40e-9 # in second
                    unlock_thresh = 200e-9 # in second

                    DPLL_PL_LOCK_THRESH.iValue = ceil(log(lock_thresh * 2**(32 - gain) * scalar * 2 * temp_vco_freq, 2))
                    DPLL_PL_UNLK_THRESH.iValue = ceil(log(unlock_thresh * 2**(32 - gain) * scalar * 2 * temp_vco_freq, 2))

                else:

                    margin = 0
                    
                    DPLL_PL_UNLK_THRESH.iValue = DPLL_REF_ORDER.iValue + ceil(log(DPLL_REF_FB_PRE_DIV.iValue + 2, 2) + log(scalar, 2)) + (32 - gain) + margin            
                    DPLL_PL_LOCK_THRESH.iValue = ceil(log(scalar, 2)) + (32 - gain) + margin
                    
                self.update_dpll_phase_lock_controls()

            else:
                UpdateStatusBar("Invalid PRIREF and SECREF. DPLL phase lock calculation requires at least one valid reference.")

    def update_dpll_hist_controls(self):

        ref_valid = self.get_refclk_freq() # updates self.refclk_freq
        self.calculate_tdc_freq() 

        if ref_valid:

            iCNT = int(DPLL_REF_HISTCNT.iValue)

            if DPLL_REF_HISTDLY.iValue >= 2**iCNT:
                DPLL_REF_HISTDLY.iValue = 2**iCNT - 1      # limit HISTDLY max value 

            iDLY = int(DPLL_REF_HISTDLY.iValue)

            dCNT = 2**iCNT * 2**10 * int(DPLL_REF_DECIMATION.iValue + 1) / self.tdc_freq
            dDLY = (iDLY + 1) * 2**10 * int(DPLL_REF_DECIMATION.iValue + 1) / self.tdc_freq  # Added the +1 to iDLY+1, 2018-11-13

            sDPLL_HISTCNT_calc.sValue = ee(dCNT, 's', 2)
            sDPLL_HISTDLY_calc.sValue = ee(dDLY, 's', 2)

        else:

            DPLL_REF_HISTCNT.sValue = "n/a"
            DPLL_REF_HISTDLY.sValue = "n/a"
            UpdateStatusBar("Invalid PRIREF and SECREF. Tuning word history not computed.")

    def recommend_dpll_hist_controls(self):

        ref_valid = self.get_refclk_freq() # updates self.refclk_freq

        if ref_valid:

            self.calculate_tdc_freq()

            for x in range(2): # calculate ppm detect measurement time
                reference_validation_manager.ref_freq_detect_update(x, bypass_warning = True)

            ref_ppm_meas_time = reference_validation_manager.ppm_meas_time

            DPLL_TUNING_FREE_RUN.iValue = 0
            DPLL_REF_HIST_EN.iValue = 1

            if 1 in self.refclk_freq:

                DPLL_REF_HISTCNT.iValue = 1
                DPLL_REF_HISTDLY.iValue = 0
                DPLL_REF_HIST_INTMD.iValue = 15

            elif ref_ppm_meas_time == [0, 0]: # if ppm detector is not enabled, then use EEPROM default

                DPLL_REF_HISTCNT.iValue = 8
                DPLL_REF_HISTDLY.iValue = 44
                DPLL_REF_HIST_INTMD.iValue = 0

            else:

                max_ppm_meas_time = max(ref_ppm_meas_time)

                for iCNT in range(31): # DPLL_REF_HISTCNT ranges from 0 to 30

                    for iDLY in range(2**iCNT): # DLY counter cannot be more than 2**iCNT - 1

                        dDLY = (iDLY + 1) * 2**10 * int(DPLL_REF_DECIMATION.iValue + 1) / self.tdc_freq

                        if dDLY >= 8 * max_ppm_meas_time:
                                
                            iCNT_recommended = iCNT
                            iDLY_recommended = iDLY
                            break

                    if iDLY != 2**iCNT - 1:
                        break

                DPLL_REF_HISTCNT.iValue = iCNT
                DPLL_REF_HISTDLY.iValue = iDLY
                DPLL_REF_HIST_INTMD.iValue = 0
            
            self.update_dpll_hist_controls() # Update the timing values.

        else:

            DPLL_REF_HISTCNT.sValue = "n/a"
            DPLL_REF_HISTDLY.sValue = "n/a"
            UpdateStatusBar("Invalid PRIREF and SECREF. Tuning word history calculation requires at least one valid reference.")

    def dpll_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "1. Set DPLL loop bandwidth. DPLL loop bandwidth needs to be lower than both PRIREF and SECREF frequencies. Some applications have specific DPLL loop bandwidth requirements. If there are not, recommendations are: If any of the reference is 1 Hz, then set the loop bandwidth to 0.01 Hz; Otherwise, set it to 100 Hz.\n\n"
        str1 += "2. Set Tr peaking (dB) and Er peaking (dB). 'Tr peaking' is the maximum peaking (in dB) allowed for DPLL transfer function. 'Er peaking' is the maximum peaking (in dB) allowed for DPLL error function. If there's no specific requirement, set them to default: Tr peaking = 0.1 dB, Er peaking = 1 dB.\n\n"
        str1 += "3. Set max TDC frequency. If no specific TDC frequency is required, set the max to 26 MHz, which is the maximum TDC frequency supported by this device.\n\n"
        str1 += "4. Fastlock should only be disabled for 1pps input. For most use cases, switchover method should always be set to 'hitless switching'. It is very rare that fastlock is needed during switchover because it will disturb phase and frequency at the outputs.\n\n"
        str1 += "5. Click 'Run Script' button and wait for 1 ~ 2 minutes while the script is running. Make sure that the MatLab runtime is installed as instructed in the first wizard page.\n\n"
        str1 += "The DPLL block diagram is for information only. Do not set the DPLL registers manually."
        s_wizard_dpll_message_box.sValue = str1

    def dpll2_instructions(self):

        str1 = "INSTRUCTIONS:\n\n"
        str1 += "All lock detect settings are set to either default or recommended values after the DPLL script is run. Still, it is highly recommended to go through the instructions and make the adjustments.\n\n"
        str1 += "1. BAW frequency lock detect. Disable this if DPLL is enabled. This detector is only useful if the DPLL is disabled and the device works in free-running mode. Enter lock threshold in ppm, unlock threshold in ppm, average count (min value = 2) and accuracy in ppm. The BAW (VCO1) is considered to be locked if the frequency error between the BAW and the XO is within lock threshold. Once the BAW is locked, it's considered to be unlocked if the frequency error exceeds the unlock threshold. The step size of lock and unlock threshold in ppm = accuracy / average. If there's no specific requirement for BAW lock detect, click 'Set Default'.\n\n"
        str1 += "2. DPLL frequency lock detect. Enter lock and unlock thresholds in ppm, average count (min value = 2) as well as accuracy in ppm. The DPLL is considered to be frequency locked if the frequency error between the VCO1 and the references is within the lock threshold. While the DPLL is frequency locked, it's considered to be frequency unlocked if the frequency error exceeds the unlock threshold. The step size of lock and unlock threshold =  accuracy / average. If there's no specific requirement for DPLL frequency lock detect, click 'Set Default'.\n\n"
        str1 += "3. DPLL phase lock detect. Set lock and unlock threshold counters. The actual lock and unlock thresholds in second are then calculated accordingly. The DPLL is considered as phase locked if the phase difference between the two inputs of TDC (divided reference and divided VCO1) is within the lock threshold. While the DPLL is phase locked, it's considered as phase unlocked if the phase difference exceeds the unlock threshold. Use recommended values for this. The lock and unlock counters should not be manually set, and they are only used for engineering debugging purposes.\n\n"
        str1 += "4. Tuning word history. This block sets the tuning word history for holdover. Set history counter and delay counter. The averaging time and delay time are then auto-calculated. Refer to datasheet section '9.3.7.4 Tuning Word History' for details. If there's no specific requirement for holdover tuning word, click 'Min Values Required'. This makes sure that the delay time is more than 8 times of the reference frequency validation measurement time, and that the averaging time is longer than the delay time.\n\n"
        s_wizard_dpll2_message_box.sValue = str1

class Class_dco_manager():

    def __init__(self):
        
        self.dpll_dco_ppb_range = [Fraction("0.000001"), Fraction("200000")]   # range of dco step size
        self.dpll_den = None
        self.vco1_freq = None
        self.dpll_prescaler = None
        self.tdc_freq = None

    def get_dco_configs(self):

        # Get DPLL denominator
        if DPLL_REF_DEN.iValue == 0:
            self.dpll_den = 2**40
        else:
            self.dpll_den = int(DPLL_REF_DEN.iValue)

        # Get VCO1 frequency
        self.vco1_freq = frequency_plan_manager.vco1_freq

        if not self.vco1_freq:
            self.vco1_freq = Fraction("2500e6")

        # Get DPLL prescaler
        self.dpll_prescaler = int(DPLL_REF_FB_PRE_DIV.iValue + 2)

        # Get PRIREF / SECREF frequency
        ref_valid = dpll_calculation_manager.get_refclk_freq()
    
        if ref_valid:

            dpll_calculation_manager.calculate_tdc_freq()
            self.tdc_freq = dpll_calculation_manager.tdc_freq

        else:
            UpdateStatusBar("Invalid PRIREF and SECREF. DCO controls not computed.")
            return False

        return True

    def calculate_fdv_from_ppb_step(self):

        # Get DCO ppb step size
        if sDPLL_DCO_STEP.dValue < self.dpll_dco_ppb_range[0]:
            sDPLL_DCO_STEP.dValue = float(self.dpll_dco_ppb_range[0])
        elif sDPLL_DCO_STEP.dValue > self.dpll_dco_ppb_range[1]:
            sDPLL_DCO_STEP.dValue = float(self.dpll_dco_ppb_range[1])

        ppb_step = Fraction(str(sDPLL_DCO_STEP.dValue))

        fdev = self.ppb_to_num_delta(ppb_step)

        if fdev:
            DPLL_FDEV.iValue = fdev

        self.calculate_ppb_step_from_fdv()

    def calculate_ppb_step_from_fdv(self):

        fdev = int(DPLL_FDEV.iValue)

        ppb_step = self.num_delta_to_ppb(fdev)

        DPLL_DCO_STEP_ACT.dValue = round(ppb_step, 16)

    def ppb_to_num_delta(self, ppb):

        dco_valid = self.get_dco_configs()

        if dco_valid:

            fdev = ppb * Fraction("1e-9") * self.dpll_den * self.vco1_freq / (2 * self.dpll_prescaler * self.tdc_freq)
            return round(fdev)

        else:
            return False

    def num_delta_to_ppb(self, num_delta):

        dco_valid = self.get_dco_configs()

        if dco_valid:

            actual_ppb = num_delta / self.dpll_den / (self.vco1_freq / (2 * self.dpll_prescaler * self.tdc_freq)) * Fraction("1e9")
            return actual_ppb

        else:
            return False

    def calculate_num_from_ppb_error(self):

        try:
            dpll_num_original = int(DPLL_REF_NUM_calculated.sValue)
        except:
            UpdateStatusBar("Original DPLL numerator not calculated. Run DPLL script first.")
        else:

            if AbsDCO_ppb_error.dValue < self.dpll_dco_ppb_range[0]:
                AbsDCO_ppb_error.dValue = float(self.dpll_dco_ppb_range[0])
            elif AbsDCO_ppb_error.dValue > self.dpll_dco_ppb_range[1]:
                AbsDCO_ppb_error.dValue = float(self.dpll_dco_ppb_range[1])

            ppb_error = Fraction(str(AbsDCO_ppb_error.dValue))
            num_delta = self.ppb_to_num_delta(ppb_error)

            if num_delta:
                DPLL_REF_NUM.iValue = num_delta + dpll_num_original

            self.calculate_ppb_error_from_num()

    def calculate_ppb_error_from_num(self):

        try:
            dpll_num_original = int(DPLL_REF_NUM_calculated.sValue)
        except:
            UpdateStatusBar("Original DPLL numerator not calculated. Run DPLL script first.")
        else:

            num_delta = int(DPLL_REF_NUM.iValue) - dpll_num_original
            ppb_error = self.num_delta_to_ppb(num_delta)

            DPLL_DCO_FREQ_PPB_ACT.dValue = round(ppb_error, 16)

class Class_zdm_manager():

    def __init__(self):
        pass

    def DPLL_ZDM_SYNC_EN_Update(self):

        if DPLL_ZDM_SYNC_EN.iValue == 1:

            DPLL_ZDM_NDIV_RST_DIS.iValue = 1
            CH7_SYNC_EN.iValue = 1

        else:
            CH7_SYNC_EN.iValue = 0

    def sSYNC_PHASE_OFFSET_Update(self):

        res = self.ph_offset_reg(phase_offset = float(sSYNC_PHASE_OFFSET.sValue), scalar = DPLL_REF_FILT_SCALAR.iValue, gain = DPLL_REF_FILT_GAIN.iValue)
        
        DPLL_REF_SYNC_PH_OFFSET.iValue = res
        self.DPLL_REF_SYNC_PH_OFFSET_Update()

    def DPLL_REF_SYNC_PH_OFFSET_Update(self):
        
        res = self.ph_offset(phase_offset_reg = DPLL_REF_SYNC_PH_OFFSET.iValue, scalar = DPLL_REF_FILT_SCALAR.iValue)
        res *= 1e-12
        res = ee(res, "s")
        res = res.replace(" Gs", "e+9")
        res = res.replace(" Ms", "e+6")
        res = res.replace(" ks", "e+3")
        res = res.replace(" ms", "e-3")
        res = res.replace(" us", "e-6")
        res = res.replace(" ns", "e-9")
        res = res.replace(" ps", "e-12")
        res = res.replace(" fs", "e-15")
        res = res.replace(" s", "")
        
        sSYNC_PHASE_OFFSET.sValue = res
        
    def ph_offset_reg(self, phase_offset = -40e-9, scalar = 791, gain = 0):

        # 200e-12 is from BAW (double edged from 2.5 GHz)
        number = phase_offset / 200e-12

        if gain == 0:
            number = int(round(number * scalar))
        else:
            number = int(round(number * scalar * (2**(32 - gain)) ))
           
        if number < 0:
            number = (2**45) + number

        return number

    def ph_offset(self, phase_offset_reg = 35184371983431, scalar = 598):

        if phase_offset_reg & (2**44) > 0:
            number = -(2**45 - phase_offset_reg)
        else:
            number = phase_offset_reg
           
        number /= scalar
        offset_ps = number * 200
        return offset_ps

class Class_output_page_manager():

    def __init__(self):

        pass

    def out7_divider_update(self):

        out7_div = sOUT7_DIV.iValue
        divisors = divisors_gen(out7_div) # divisor_gen is from pllsolver_new
        error_msg = ""
        
        if out7_div >= 2: # output 7 divider 

            o7_s2 = None

            for i in divisors:
                if i <= 2**24 and 2 <= out7_div / i <= 256:
                    o7_s2 = i
                    break

            if o7_s2 == None:
                error_msg = "Illegal divide for OUT7 divider. Divide value must be a product of a number below 256 and a number below 2**24."
            else:
                o7 = int(out7_div / o7_s2)
                
        else:
            error_msg = "Illegal divide for OUT7 divider, divide value must be greater than or equal to 2."
            
        if "" == error_msg:     
            OUT7_DIV.iValue = o7 - 1
            OUT7_STG2_DIV.iValue = o7_s2 - 1
        else:
            UpdateStatusBar(error_msg)   

    def output_format_update_ui(self, out_num):

        if out_num == 0 or out_num == 1:
            self.output_powerdown_update_ui("0_1")
        elif out_num == 2 or out_num == 3:
            self.output_powerdown_update_ui("2_3")
        else:
            self.output_powerdown_update_ui(out_num)

    def output_powerdown_update_ui(self, channel_num):

        temp_pd = gd["CH{}_PD".format(channel_num)].iValue

        if channel_num == "0_1":
            temp_output_num_list = [0, 1]

        elif channel_num == "2_3":
            temp_output_num_list = [2, 3]

        else:
            temp_output_num_list = [int(channel_num)]

        for item in temp_output_num_list:

            if not temp_pd: # if that channel is not powered down - i.e., if CHx_PD = 0 - then hide the output frequency if output format is "disabled"

                if gd["OUT{}_FMT".format(item)].iValue == 0:
                    Hide(gd["OUT{}_freq".format(item)])
                else:
                    Show(gd["OUT{}_freq".format(item)])

            else: # If it is powered down, then always hide the output frequency
                Hide(gd["OUT{}_freq".format(item)])

class Class_apll_page_manager():
    """This class manages functions used by _Update() or _Update_UI() for control objects in APLL1 and APLL2 page"""
    def __init__(self):
        pass

    def pll1_num_update(self):
        """Defines what happens to PLL1_24b_NUM and PLL1_24b_DEN when PLL1_NUM changes"""
        PLL1_24b_DEN.iValue = PLL1_NUM.iValue & 2**24 - 1
        PLL1_24b_NUM.iValue = (PLL1_NUM.iValue >> 24) + PLL1_24b_NUM_MSB.iValue * 2**16

    def pll1_24b_num_den_update(self):
        """Defines what happens to PLL1_NUM and PLL1_24b_NUM_MSB when PLL1_24b_NUM or PLL1_24b_DEN changes."""
        PLL1_NUM.iValue = ((PLL1_24b_NUM.iValue & 2**16 - 1) << 24) + PLL1_24b_DEN.iValue
        PLL1_24b_NUM_MSB.iValue = PLL1_24b_NUM.iValue >> 16 

class Table_data_item_pll1():
    """data object for pll1 table"""
    def __init__(self, data_list):

        self.index = data_list[0] # This is to quickly retrieve the solution index.
        self.vco1_freq = data_list[1]
        self.apll1_p1 = data_list[2]

class Table_data_item_pll2():
    """data object for pll2 table"""
    def __init__(self, data_list):
        
        self.index = data_list[0] # This is to quickly retrieve the solution index
        self.vco2_freq = data_list[1]
        self.apll2_p1 = data_list[2]
        self.apll2_p2 = data_list[3]
        self.apll2_spur_freq = data_list[4]

class rom_obj():

    # Init values below will be overwritten by FLEX script with user's values from FLEX pages
    def __init__(self):

        self.refcount = 2
        
        # Frequencies
        self.xo_freq = '0'
        self.priref_freq = '0'
        self.secref_freq = '0'
        self.vco1_freq = '0'
        self.vco2_freq = '0'
        self.dpll_lbw = '0'       # in Hz
        
        # DPLL mode values
        self.dpll_mode = 0                 # 0 to 4 
        
        self.dpll_ref_sel = ['0','3']              # comma separated list of REF index integer values assigned to DPLL
        self.dpll_ref_clk = ['155.52','156.25']    # comma separated list of REF frequencies assigned to DPLL
        
        # Results
        self.rom_rawdata = OrderedDict()    # dict of raw data from rom file *before* parsing/formatting to separate dicts: rom_header and rom_regdata
        self.rom_header = OrderedDict()     # dict of header data (str value)
        self.rom_regdata = OrderedDict()    # dict of register data (int value as string); field name without "_[" or "[" and formatted to upper

    # ReadRomFile: Reads Rom file and saves to ordered dictionaries to update GUI
    def ReadRomFile(self, filename=r'reg_rom0.txt'):
        f = open(filename, "r")
        romfile = f.readlines()
        f.close()
             
        rom_rawdata = OrderedDict()
        rom_header = OrderedDict()
        rom_header_list = []
        rom_regdata = OrderedDict()
        self.dpll_ref_sel = []
                
        for line in romfile:
            if line.count("=") >= 2:
                pass     #
                prev_line = ""
                for line in line.split(","):
                    key, value = line.split("=")
                    if prev_line != "":
                        key = prev_line + "_" + key.strip()
                    else:
                        prev_line = key.strip()
                    value = value.strip()
                    rom_rawdata[key] = value                                    # store raw data with leading/trailing whitespaces removed
                    
                    key = key.split("_[")[0].split("[")[0].upper()              # removes "_[" or "[" from key

                    if key.startswith(";"):
                        key = key.split(";")[1].strip()
                        value = value.split(";")[0].strip()                     # remove any inline comment (;) from value
                        if (key == 'DPLL_REFSEL'):
                            self.dpll_ref_sel.append(value)                    # add REF to list in order of priority
                            i1 = len(self.dpll_ref_sel)-1
                            key = key + str(i1)
                            rom_header[key] = value                             # new unique key name "DPLL1_REFSEL<i>" where i = priority (0=highest)
                            rom_header_list.append((key, value))
                        else:
                            rom_header[key] = value                             # store header line (without ;)
                            rom_header_list.append((key, value))
                        print ("3=Header: %s = %s") % (key, value)
                        UpdateStatusBar ("Header: %s = %s" % (key, value))       # dbg2
            
            elif line.count("=") == 1:
                key, value = line.split("=")
                key = key.strip()
                value = value.strip()
                rom_rawdata[key] = value                                    # store raw data with leading/trailing whitespaces removed
                
                key = key.split("_[")[0].split("[")[0].upper()              # removes "_[" or "[" from key
                
                key2 = re.sub("DPLL1", "DPLL", key)
                key2 = re.sub("REF0", "PRIREF", key2)
                key2 = re.sub("REF1", "SECREF", key2)
                #UpdateStatusBar("key {} to key2 {}".format(key, key2))
                
               # check/add header data
                if key.startswith(";"):
                    key = key.split(";")[1].strip()
                    value = value.split(";")[0].strip()                     # remove any inline comment (;) from value
                    if (key == 'DPLL_REFSEL'):
                        self.dpll_ref_sel.append(value)                    # add REF to list in order of priority
                        i1 = len(self.dpll_ref_sel)-1
                        key = key + str(i1)
                        rom_header[key] = value                             # new unique key name "DPLL_REFSEL<i>" where i = priority (0=highest)
                    else:
                        rom_header[key] = value                             # store header line (without ;)
                    print ("Header: %s = %s") % (key, value)
                # check/add valid register data
                elif (key not in rom_regdata) and (value != ''):
                    value = value.split(";")[0].strip()                     # remove any inline comment (;) from value
                    rom_regdata[key] = int(value, 16)                       # store valid register field, hex-to-int
                    # print ("Register: %s = %s = %s" % (key, value, rom_regdata[key]))
                # exceptions
                elif (key not in rom_regdata) and (value != ''):
                    value = value.split(";")[0].strip()                     # remove any inline comment (;) from value
                    rom_regdata[key] = int(value, 16)                       # store valid register field, hex-to-int
                    # print ("Register: %s = %s = %s" % (key, value, rom_regdata[key]))
                # exceptions
                elif (key in rom_regdata):
                    print ("Skip! Duplicate: %s = 0x%x" % (key, rom_regdata[key]))
                    pass
                else:
                    print ("Skip! Invalid or masked: %s") % line
                    pass
            else:
                print("Failed split '=': %s" % line)
                pass
        """
        print "dpll_ref_sel"
        print self.dpll_ref_sel
        """
        self.rom_rawdata = rom_rawdata
        self.rom_header = rom_header
        self.rom_header_list = rom_header_list
        self.rom_regdata = rom_regdata
        # return rom_rawdata, rom_header, rom_regdata

def ee(num, suffix="", fix=None):
    """Convert floating numbers to strings for Hz, s and etc."""

    suffix_scaler_low = { -15:"f", -12:"p", -9:"n", -6:"u", -3: "m", 0:"" }
    suffix_scaler_high = { 0: "", 3:"k", 6:"M", 9:"G" }
    scaler_index = 0

    if (num >= 1) or (num <= -1):
        while (num >= 1e3 or num <= -1e3) and scaler_index<9:
            num /= 1e3
            scaler_index += 3

        if fix:
            num = round(num, fix)
        #retstr = "%f %s%s" % (num, suffix_scaler_high[scaler_index], suffix)
        retstr = "{} {}{}".format(num, suffix_scaler_high[scaler_index], suffix)
    else:
        while (num < 1 and num > -1) and scaler_index > -15:
            num *= 1e3
            scaler_index -= 3

        if fix:
            #convstr = "%%0.%df %%s%%s" % fix
            convstr = "{:.%df} {}{}" % fix
        else:
            #convstr = "%f %s%s"
            convstr = "{} {}{}"
            
        ##retstr = "%f %s%s" % (num, suffix_scaler_low[scaler_index], suffix)
        #retstr = convstr % (num, suffix_scaler_low[scaler_index], suffix)
        retstr = convstr.format(num, suffix_scaler_low[scaler_index], suffix)

    return retstr

frequency_plan_manager = Class_frequency_plan_manager()
frequency_update_manager = Class_frequency_update_manager()
output_page_manager = Class_output_page_manager()
apll_page_manager = Class_apll_page_manager()
dpll_reference_manager = Class_dpll_reference_manager()
reference_validation_manager = Class_reference_validation_manager()
dpll_calculation_manager = Class_dpll_calculation_manager()
dco_manager = Class_dco_manager()
zdm_manager = Class_zdm_manager()

mat_obj = MatlabInterface.MatlabInterface(total_num_dpll = 1, total_num_inputs = 2)
romdata = rom_obj()

#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ define "_Update()" functions @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ 

#------------------------------------------------wizard pages--------------------------------------------------------

### Generic
def combo_dpll_mode_Update():

    if combo_dpll_mode.iValue == 0: # Enable DPLL

        PLL1_MODE.iValue = 1 # DPLL mode
        DPLL_LOOP_EN.iValue = 1 # Enable DPLL

    else: # disable DPLL

        PLL1_MODE.iValue = 0 # free-running mode
        DPLL_LOOP_EN.iValue = 0 # Disable DPLL

def combo_backward_compatible_Update():

    if combo_backward_compatible.iValue:
        UpdateStatusBar("Warning! Improvement made in LMK05318B will be disabled for backward compatibility.")

# Next buttons
def btn_page0_next_Update():
    SelectPage(g.wizard_page_list[1])

def btn_page1_next_Update():
    SelectPage(g.wizard_page_list[2])

def btn_page2_next_Update():
    SelectPage(g.wizard_page_list[3])

def btn_page3_next_Update():
    SelectPage(g.wizard_page_list[4])

def btn_page4_next_Update():
    SelectPage(g.wizard_page_list[5])

def btn_page5_next_Update():
    SelectPage(g.wizard_page_list[6])

def btn_page6_next_Update():
    SelectPage(g.wizard_page_list[7])

def btn_page7_next_Update():
    SelectPage(g.wizard_page_list[8])

def btn_page8_next_Update():
    SelectPage(g.wizard_page_list[9])

# Back buttons
def btn_page1_back_Update():
    SelectPage(g.wizard_page_list[0])

def btn_page2_back_Update():
    SelectPage(g.wizard_page_list[1])

def btn_page3_back_Update():
    SelectPage(g.wizard_page_list[2])

def btn_page4_back_Update():
    SelectPage(g.wizard_page_list[3])

def btn_page5_back_Update():
    SelectPage(g.wizard_page_list[4])

def btn_page6_back_Update():
    SelectPage(g.wizard_page_list[5])

def btn_page7_back_Update():
    SelectPage(g.wizard_page_list[6])

def btn_page8_back_Update():
    SelectPage(g.wizard_page_list[7])

def btn_page9_back_Update():
    SelectPage(g.wizard_page_list[8])

# Frequency planner
def sXO_freq_Update():
    frequency_plan_manager.autoset_apll1_rdiv()

def btn_xo_show_instructions_Update():

    frequency_plan_manager.xo_instructions()

def btn_freq_plan_show_instructions_Update():

    frequency_plan_manager.freq_plan_instructions()

def cb_show_frequency_planner_table_Update():

    if cb_show_frequency_planner_table.iValue == 0:
        Hide(table_frequency_plan_pll1)
        Hide(table_frequency_plan_pll2)
        Hide(btn_frequency_plan_apply_solution)
    else:
        Show(table_frequency_plan_pll1)
        Show(table_frequency_plan_pll2)
        Show(btn_frequency_plan_apply_solution)

def bCALC_FREQPLAN_Update():

    frequency_plan_manager.__init__()

    # clear the tables
    gd[frequency_plan_manager.table_pll1].Clear()
    gd[frequency_plan_manager.table_pll2].Clear()

    temp_bool_ch_assign = False # All channels must be successfully assigned before frequency plan can be calculated.
    
    for channel_num in g.channel_num_list:

        # First power up all channels (because the next step will power down channels that are assigend to no channel mux. This prevents channels from always being powered down from history)
        gd["CH{}_PD".format(channel_num)].iValue = 0
        # Then update the channel mux selection for all channels. This is because when loading a .tcs file, the input frequencies are there but the frequency_plan_manager.dict_channel_mux is not updated.
        temp_bool_ch_assign = frequency_plan_manager.update_dict_channel_mux(channel_num)

        if not temp_bool_ch_assign:
            break

    if temp_bool_ch_assign: # If all channels are either successfully assigned to a source or powered down by user

        xo_valid = frequency_plan_manager.xo_frequency_update()

        if xo_valid:

            frequency_plan_manager.calculate_frequency_plan() # run frequency plan calculation

            if frequency_plan_manager.error_code == [0,0]: # only continue to do the rest if there's no error for both PLLs

                frequency_plan_manager.update_table() 
                frequency_plan_manager.select_solution_auto()
                frequency_plan_manager.update_apll_references() 
                frequency_plan_manager.implement_frequency_plan()
                frequency_update_manager.update_all_frequencies()
                frequency_plan_manager.update_message_box()     
            else:
                pass

        else:
            s_wizard_freqplan_message_box.sValue = "Invalid XO frequency! Frequency plan not calculated."

def btn_frequency_plan_apply_solution_Update():

    if UIC_table_frequency_plan_pll1.SelectedItem == None or UIC_table_frequency_plan_pll2.SelectedItem == None:
        s_wizard_freqplan_message_box.sValue = "Error!\n\nNo frequency plan for APLL1 and/or APLL2 is selected."
    else:

        frequency_plan_manager.select_solution_manual()
        frequency_plan_manager.update_apll_references()
        frequency_plan_manager.implement_frequency_plan()
        frequency_update_manager.update_all_frequencies()
        frequency_plan_manager.update_message_box()

# def sCHx_IN_freq_Update(): for all channels.
def sCH0_1_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("0_1")

def sCH2_3_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("2_3")

def sCH4_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("4")

def sCH5_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("5")

def sCH6_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("6")

def sCH7_IN_freq_Update():
    frequency_plan_manager.update_dict_channel_mux("7")

# DPLL reference
def cb_enable_PRIREF_Update_UI():

    dim_or_undim = [Dim, UnDim][cb_enable_PRIREF.iValue]
    dim_or_undim(sPRIREF_freq)

def cb_enable_SECREF_Update_UI():

    dim_or_undim = [Dim, UnDim][cb_enable_SECREF.iValue]
    dim_or_undim(sSECREF_freq)

def cb_enable_PRIREF_Update():

    dpll_reference_manager.ref_enable_disable_update(0)
    dpll_reference_manager.autoset_ref_buffer_mode(0)
    dpll_reference_manager.autoset_reference_validation(0)

def cb_enable_SECREF_Update():

    dpll_reference_manager.ref_enable_disable_update(1)
    dpll_reference_manager.autoset_ref_buffer_mode(1)
    dpll_reference_manager.autoset_reference_validation(1)

def sPRIREF_freq_Update():

    dpll_reference_manager.autoset_ref_buffer_mode(0)
    dpll_reference_manager.autoset_reference_validation(0)

def sSECREF_freq_Update():

    dpll_reference_manager.autoset_ref_buffer_mode(1)
    dpll_reference_manager.autoset_reference_validation(1)

def combo_PRIREF_BUF_TYPE_Update_UI():

    if combo_PRIREF_BUF_TYPE.iValue in [0, 1]:
        Show(PRIREF_TYPE)
    else:
        Hide(PRIREF_TYPE)

def combo_SECREF_BUF_TYPE_Update_UI():

    if combo_SECREF_BUF_TYPE.iValue in [0, 1]:
        Show(SECREF_TYPE)
    else:
        Hide(SECREF_TYPE)

def combo_PRIREF_BUF_TYPE_Update():

    if combo_backward_compatible.iValue and combo_PRIREF_BUF_TYPE.iValue != 1:

        combo_PRIREF_BUF_TYPE.iValue = 1
        UpdateStatusBar("PRIREF buffer type has been forced to 'AC, hyst = 200 mV' to be backward compatible to LMK05318.")

    dpll_reference_manager.ref_buffer_type_selection(0)
    dpll_reference_manager.autoset_reference_validation(0)

def combo_SECREF_BUF_TYPE_Update():

    if combo_backward_compatible.iValue and combo_SECREF_BUF_TYPE.iValue != 1:

        combo_SECREF_BUF_TYPE.iValue = 1
        UpdateStatusBar("SECREF buffer type has been forced to 'AC, hyst = 200 mV' to be backward compatible to LMK05318.")

    dpll_reference_manager.ref_buffer_type_selection(1)
    dpll_reference_manager.autoset_reference_validation(1)

def PRIREF_TYPE_Update():

    dpll_reference_manager.autoset_reference_validation(0)

def SECREF_TYPE_Update():

    dpll_reference_manager.autoset_reference_validation(1)

def combo_ref_priority_Update():

    dpll_reference_manager.ref_priority_update()

def DPLL_SWITCH_MODE_Update_UI():

    if DPLL_SWITCH_MODE.iValue in [0, 1]: # auto reference selection

        Hide(DPLL_REF_MAN_SEL)
        Hide(refclk_pin_select)
        Hide(DPLL_REF_MAN_REG_SEL)

    else: # manual reference selection
        Show(DPLL_REF_MAN_SEL)
        DPLL_REF_MAN_SEL_Update_UI()

    if DPLL_SWITCH_MODE.iValue == 3: # priority doesn't matter for manual holdover
        show_or_hide = Hide
    else:
        show_or_hide = Show

    show_or_hide(combo_ref_priority)

def DPLL_REF_MAN_SEL_Update_UI():

    if DPLL_REF_MAN_SEL.iValue == 0: # manual select by register

        Hide(refclk_pin_select)
        Show(DPLL_REF_MAN_REG_SEL)

    else: # manual select by pin

        Show(refclk_pin_select)
        Hide(DPLL_REF_MAN_REG_SEL)

def btn_refclk_show_instructions_Update():

    dpll_reference_manager.set_reference_instructions()

# Reference validation
def btn_ref_validation_show_instructions_Update():

    reference_validation_manager.reference_validation_instructions()

def PRIREF_PPM_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][PRIREF_PPM_EN.iValue]

    dim_or_undim(sPRIREF_PPM_VALID)
    dim_or_undim(sPRIREF_PPM_INVALID)
    dim_or_undim(sPRIREF_ACCURACY_PPM)
    dim_or_undim(sPRIREF_AVG_COUNT)

def SECREF_PPM_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][SECREF_PPM_EN.iValue]

    dim_or_undim(sSECREF_PPM_VALID)
    dim_or_undim(sSECREF_PPM_INVALID)
    dim_or_undim(sSECREF_ACCURACY_PPM)
    dim_or_undim(sSECREF_AVG_COUNT)

def PRIREF_EARLY_DET_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][PRIREF_EARLY_DET_EN.iValue]

    dim_or_undim(sPRIREF_EARLY_MARGIN)

def SECREF_EARLY_DET_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][SECREF_EARLY_DET_EN.iValue]

    dim_or_undim(sSECREF_EARLY_MARGIN)

def PRIREF_MISSCLK_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][PRIREF_MISSCLK_EN.iValue]

    dim_or_undim(sPRIREF_LATE)
    dim_or_undim(sPRIREF_LATE_MARGIN)

def SECREF_MISSCLK_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][SECREF_MISSCLK_EN.iValue]

    dim_or_undim(sSECREF_LATE)
    dim_or_undim(sSECREF_LATE_MARGIN)

def PRIREF_CMOS_SLEW_Update_UI():

    if PRIREF_CMOS_SLEW.iValue == 0: # Amplitude detector mode

        Show(PRIREF_LVL_SEL)
        Hide(DETECT_MODE_PRIREF)

    else: # CMOS slew rate detector mode

        Hide(PRIREF_LVL_SEL)
        Show(DETECT_MODE_PRIREF)

def SECREF_CMOS_SLEW_Update_UI():

    if SECREF_CMOS_SLEW.iValue == 0: # Amplitude detector mode

        Show(SECREF_LVL_SEL)
        Hide(DETECT_MODE_SECREF)

    else: # CMOS slew rate detector mode

        Hide(SECREF_LVL_SEL)
        Show(DETECT_MODE_SECREF)

def PRIREF_PH_VALID_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][PRIREF_PH_VALID_EN.iValue]

    dim_or_undim(PRIREF_PH_VALID_THR)

def SECREF_PH_VALID_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][SECREF_PH_VALID_EN.iValue]

    dim_or_undim(SECREF_PH_VALID_THR)

def PRIREF_PPM_EN_Update():

    reference_validation_manager.ref_freq_detect_update(0)

    if PRIREF_PPM_EN.iValue:
        UpdateStatusBar("Frequency detector is not available for the current LMK05318B silicon.")

def sPRIREF_PPM_VALID_Update():
    reference_validation_manager.ref_freq_detect_update(0)

def sPRIREF_PPM_INVALID_Update():
    reference_validation_manager.ref_freq_detect_update(0)

def sPRIREF_ACCURACY_PPM_Update():
    reference_validation_manager.ref_freq_detect_update(0)

def sPRIREF_AVG_COUNT_Update():

    if sPRIREF_AVG_COUNT.iValue < 2:
        sPRIREF_AVG_COUNT.iValue = 2

    reference_validation_manager.ref_freq_detect_update(0)

def SECREF_PPM_EN_Update():

    reference_validation_manager.ref_freq_detect_update(1)

    if SECREF_PPM_EN.iValue:
        UpdateStatusBar("Frequency detector is not available for the current LMK05318B silicon.")

def sSECREF_PPM_VALID_Update():
    reference_validation_manager.ref_freq_detect_update(1)

def sSECREF_PPM_INVALID_Update():
    reference_validation_manager.ref_freq_detect_update(1)

def sSECREF_ACCURACY_PPM_Update():
    reference_validation_manager.ref_freq_detect_update(1)

def sSECREF_AVG_COUNT_Update():

    if sSECREF_AVG_COUNT.iValue < 2:
        sSECREF_AVG_COUNT.iValue = 2

    reference_validation_manager.ref_freq_detect_update(1)

def PRIREF_EARLY_DET_EN_Update():
    reference_validation_manager.ref_early_late_det_update(0)

def sPRIREF_EARLY_MARGIN_Update():
    reference_validation_manager.ref_early_late_det_update(0)

def PRIREF_MISSCLK_EN_Update():
    reference_validation_manager.ref_early_late_det_update(0)

def sPRIREF_LATE_Update():
    reference_validation_manager.ref_early_late_det_update(0)

def sPRIREF_LATE_MARGIN_Update():
    reference_validation_manager.ref_early_late_det_update(0)

def SECREF_EARLY_DET_EN_Update():
    reference_validation_manager.ref_early_late_det_update(1)

def sSECREF_EARLY_MARGIN_Update():
    reference_validation_manager.ref_early_late_det_update(1)

def SECREF_MISSCLK_EN_Update():
    reference_validation_manager.ref_early_late_det_update(1)

def sSECREF_LATE_Update():
    reference_validation_manager.ref_early_late_det_update(1)

def sSECREF_LATE_MARGIN_Update():
    reference_validation_manager.ref_early_late_det_update(1)

def PRIREF_PH_VALID_EN_Update():
    reference_validation_manager.ref_ph_det_update(0)

def PRIREF_PH_VALID_THR_Update():
    reference_validation_manager.ref_ph_det_update(0)

def SECREF_PH_VALID_EN_Update():
    reference_validation_manager.ref_ph_det_update(1)

def SECREF_PH_VALID_THR_Update():
    reference_validation_manager.ref_ph_det_update(1)



# DPLL calculation
def combo_set_max_tdc_freq_Update():

    if combo_set_max_tdc_freq.iValue == 0: # Set default
        sMAX_TDC_freq.dValue = 26e6

def combo_set_max_tdc_freq_Update_UI():

    dim_or_undim = [Dim, UnDim][combo_set_max_tdc_freq.iValue]
    dim_or_undim(sMAX_TDC_freq)

def combo_switching_method_Update():

    if combo_switching_method.iValue == 0: # hitless switching

        DPLL_FASTLOCK_ALWAYS.iValue = 0
        DPLL_SWITCHOVER_ALWAYS.iValue = 1

    else:
        DPLL_FASTLOCK_ALWAYS.iValue = 1
        DPLL_SWITCHOVER_ALWAYS.iValue = 0

def sMAX_TDC_freq_Update():

    dpll_calculation_manager.max_tdc_freq_update()

def sDPLL_LBW_Update():

    dpll_calculation_manager.dpll_loop_bandwidth_update()

def bRUN_SCRIPT_Update():

    dpll_calculation_manager.run_script_update()

# DPLL block diagram
def DPLL_PRIREF_RDIV_Update():

    dpll_calculation_manager.update_dpll_block_diagram()

def DPLL_SECREF_RDIV_Update():

    dpll_calculation_manager.update_dpll_block_diagram()

def DPLL_REF_FB_PRE_DIV_Update():

    dpll_calculation_manager.update_dpll_block_diagram()

def DPLL_REF_FB_DIV_Update():

    dpll_calculation_manager.update_dpll_block_diagram()

def DPLL_REF_DEN_Update():

    dpll_calculation_manager.update_dpll_block_diagram()

def cb_show_dpll_registers_Update():

    if combo_dpll_mode.iValue == 0: # if DPLL is enabled

        input_valid = dpll_calculation_manager.get_user_input()

        if input_valid:

            if os.path.exists(dpll_calculation_manager.outfile):

                dpll_calculation_manager.import_rom(dpll_calculation_manager.outfile)
                dpll_calculation_manager.update_message_box()

            else:
                s_wizard_dpll_message_box.sValue = "Error!\n\nNo stored DPLL register settings. Run DPLL script first."

    else:
        s_wizard_dpll_message_box.sValue = "Error!\n\nThe DPLL has been disabled. Enable DPLL first."

def btn_restore_dpll_reg_Update():

    if combo_dpll_mode.iValue == 0: # if DPLL is enabled

        input_valid = dpll_calculation_manager.get_user_input()

        if input_valid:

            if os.path.exists(dpll_calculation_manager.outfile):

                dpll_calculation_manager.import_rom(dpll_calculation_manager.outfile)
                dpll_calculation_manager.update_message_box()

            else:
                UpdateStatusBar("No stored DPLL settings. Run DPLL script first.")

        else:
            UpdateStatusBar("Invalid user input. DPLL block diagram not updated.")

    else:
        UpdateStatusBar("DPLL has been disabled in the first wizard page. DPLL block diagram not updated.")


def btn_dpll_show_instructions_Update():

    dpll_calculation_manager.dpll_instructions()

# BAW frequency lock detect
def BAW_LOCK_PPM_THRESH_Update():

    dpll_calculation_manager.update_baw_lock_controls()

def sBAW_LOCK_AVG_Update():

    dpll_calculation_manager.update_baw_lock_controls()

def sBAW_LOCK_ACCURACY_PPM_Update():

    dpll_calculation_manager.update_baw_lock_controls()

def BAW_UNLK_PPM_THRESH_Update():

    dpll_calculation_manager.update_baw_lock_controls()

def sBAW_UNLK_AVG_Update():

    dpll_calculation_manager.update_baw_lock_controls()

def sBAW_UNLK_ACCURACY_PPM_Update():

    dpll_calculation_manager.update_baw_lock_controls()

# DPLL frequency lock detect
def sDPLL_LOCK_PPM_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

def sDPLL_LOCK_AVG_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

def sDPLL_LOCK_ACCURACY_PPM_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

def sDPLL_UNLK_PPM_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

def sDPLL_UNLK_AVG_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

def sDPLL_UNLK_ACCURACY_PPM_Update():

    dpll_calculation_manager.update_dpll_frequency_lock_controls()

# DPLL phase lock detect
def DPLL_PL_LOCK_THRESH_Update():

    dpll_calculation_manager.update_dpll_phase_lock_controls()

def DPLL_PL_UNLK_THRESH_Update():

    dpll_calculation_manager.update_dpll_phase_lock_controls()

def btn_ph_threshold_recommend_Update():

    dpll_calculation_manager.recommend_dpll_phase_lock_controls()

# history tuning word
def DPLL_REF_HISTCNT_Update():

    dpll_calculation_manager.update_dpll_hist_controls()

def DPLL_REF_HISTDLY_Update():

    dpll_calculation_manager.update_dpll_hist_controls()

def btn_history_recommend():

    dpll_calculation_manager.recommend_dpll_hist_controls()

# update UI
def BAW_LOCKDET_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][BAW_LOCKDET_EN.iValue]
    dim_or_undim(BAW_LOCK_PPM_THRESH)
    dim_or_undim(BAW_UNLK_PPM_THRESH)
    dim_or_undim(sBAW_LOCK_AVG)
    dim_or_undim(sBAW_LOCK_ACCURACY_PPM)


def DPLL_LOCKDET_PPM_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][DPLL_LOCKDET_PPM_EN.iValue]
    dim_or_undim(sDPLL_LOCK_PPM)
    dim_or_undim(sDPLL_UNLK_PPM)
    dim_or_undim(sDPLL_LOCK_AVG)
    dim_or_undim(sDPLL_LOCK_ACCURACY_PPM)

def DPLL_REF_HIST_EN_Update_UI():

    dim_or_undim = [Dim, UnDim][DPLL_REF_HIST_EN.iValue]
    dim_or_undim(DPLL_REF_HISTCNT)
    dim_or_undim(DPLL_REF_HISTDLY)

def btn_dpll2_show_instructions_Update():

    dpll_calculation_manager.dpll2_instructions()

def btn_baw_lock_detect_default_Update():

    dpll_calculation_manager.default_baw_lock_controls()

def btn_dpll_freq_lock_detect_default_Update():

    dpll_calculation_manager.default_dpll_frequency_lock_controls()

def btn_ph_threshold_recommend_Update():

    dpll_calculation_manager.recommend_dpll_phase_lock_controls()

def btn_history_recommend_Update():

    dpll_calculation_manager.recommend_dpll_hist_controls()


# DCO
def sDPLL_DCO_STEP_Update():

    dco_manager.calculate_fdv_from_ppb_step()

def DPLL_FDEV_Update():

    dco_manager.calculate_ppb_step_from_fdv()

def AbsDCO_ppb_error_Update():

    dco_manager.calculate_num_from_ppb_error()

def DPLL_REF_NUM_Update(): # This is updating both DPLL block diagram and DCO

    dpll_calculation_manager.update_dpll_block_diagram()

    dco_manager.calculate_ppb_error_from_num()

def bResetNumerator0ppb_Update():

    try:
        dpll_num_original = int(DPLL_REF_NUM_calculated.sValue)
    except:
        UpdateStatusBar("Original DPLL numerator not calculated. Run DPLL script first.")
    else:
        DPLL_REF_NUM.iValue = dpll_num_original

def cbDCO_MODE_Update():

    if 0 == cbDCO_MODE.iValue:                     # Select Relative GPIO

        GPIO_FDEV_EN.iValue = 1
        DPLL_IGNORE_GPIO_PIN.iValue = 0
        DPLL_FDEV_EN.iValue = 1

    elif 1 == cbDCO_MODE.iValue:                   # Select Relative Registers

        GPIO_FDEV_EN.iValue = 0
        DPLL_IGNORE_GPIO_PIN.iValue = 1
        DPLL_FDEV_EN.iValue = 1

    elif 2 == cbDCO_MODE.iValue:                   # Select Absolute

        GPIO_FDEV_EN.iValue = 0
        DPLL_IGNORE_GPIO_PIN.iValue = 1
        DPLL_FDEV_EN.iValue = 0

def bDPLL_FINCR_Update():

    if cbDCO_MODE.iValue == 0:

        SetPin('GPIO2', 1)
        SetPin('GPIO2', 0)
        UpdateStatusBar('GPIO2/FINC pin pulsed')

    elif cbDCO_MODE.iValue == 1:
        WriteParameter("DPLL_FDEV_REG_UPDATE", 0)

    else:
        UpdateStatusBar("No effect --> Select DCO Freq Control to be a relative mode.")
    
def bDPLL_FDECR_Update():

    if cbDCO_MODE.iValue == 0:

        SetPin('Status1', 1)
        SetPin('Status1', 0)
        UpdateStatusBar('STATUS1/FDEC pin pulsed')

    elif cbDCO_MODE.iValue == 1:
        WriteParameter("DPLL_FDEV_REG_UPDATE", 1)

    else:
        UpdateStatusBar("No effect --> Select DCO Freq Control to be a relative mode.")

# ZDM
def DPLL_ZDM_SYNC_EN_Update():
    zdm_manager.DPLL_ZDM_SYNC_EN_Update()

def sSYNC_PHASE_OFFSET_Update():
    zdm_manager.sSYNC_PHASE_OFFSET_Update()

def DPLL_REF_SYNC_PH_OFFSET_Update():
    zdm_manager.DPLL_REF_SYNC_PH_OFFSET_Update()

#-------------------------------------------------------Reference page-------------------------------------------------
def PRI_BUF_type_Update():

    switcher = {0: [0, 0], 1: [0, 1], 2: [1, 0], 3: [1, 1]}
    PRIREF_DC_MODE.iValue = switcher[PRI_BUF_type.iValue][0]
    PRIREF_BUF_MODE.iValue = switcher[PRI_BUF_type.iValue][1]

def SEC_BUF_type_Update():

    switcher = {0: [0, 0], 1: [0, 1], 2: [1, 0], 3: [1, 1]}
    SECREF_DC_MODE.iValue = switcher[SEC_BUF_type.iValue][0]
    SECREF_BUF_MODE.iValue = switcher[SEC_BUF_type.iValue][1]


#-------------------------------------------------------APLL pages------------------------------------------------------


def PLL1_NUM_Update():

    apll_page_manager.pll1_num_update()
    frequency_update_manager.update_all_frequencies()

def PLL1_24b_NUM_Update():

    apll_page_manager.pll1_24b_num_den_update()
    frequency_update_manager.update_all_frequencies()

def PLL1_24b_DEN_Update():

    apll_page_manager.pll1_24b_num_den_update()
    frequency_update_manager.update_all_frequencies()

def APLL1_DEN_MODE_Update_UI():

    if APLL1_DEN_MODE.iValue == 0:
        Show(PLL1_NUM)
        Show(PLL1_DEN)
        Hide(PLL1_24b_NUM)
        Hide(PLL1_24b_DEN)
    else:
        Hide(PLL1_NUM)
        Hide(PLL1_DEN)
        Show(PLL1_24b_NUM)
        Show(PLL1_24b_DEN)

def APLL2_DEN_MODE_Update_UI():

    if APLL2_DEN_MODE.iValue == 0:
        Show(PLL2_DEN_fixed)
        Hide(PLL2_DEN)
    else:
        Hide(PLL2_DEN_fixed)
        Show(PLL2_DEN)

def OSCIN_DBLR_EN_Update():
    frequency_update_manager.update_all_frequencies()

def OSCIN_RDIV_Update():
    frequency_update_manager.update_all_frequencies()

def PLL1_NDIV_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_RDIV_PRE_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_RDIV_SEC_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_NDIV_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_NUM_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_DEN_Update():
    frequency_update_manager.update_all_frequencies()

def PLL1_PDN_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_PDN_Update():
    frequency_update_manager.update_all_frequencies()    

def PLL1_PDN_Update_UI():

    dim_or_undim = [UnDim, Dim][PLL1_PDN.iValue]

    dim_or_undim(PLL1_NDIV)
    dim_or_undim(PLL1_24b_NUM)
    dim_or_undim(PLL1_24b_DEN)
    dim_or_undim(PLL1_NUM)
    dim_or_undim(APLL1_DEN_MODE)

def PLL2_PDN_Update_UI():

    dim_or_undim = [UnDim, Dim][PLL2_PDN.iValue]

    dim_or_undim(PLL2_NDIV)
    dim_or_undim(PLL2_NUM)
    dim_or_undim(PLL2_DEN)
    dim_or_undim(APLL2_DEN_MODE)

def PLL2_RCLK_SEL_Update():

    frequency_update_manager.update_all_frequencies()

#----------------------------------------------------output page--------------------------------------------------------        
def PLL2_P1_Update():
    frequency_update_manager.update_all_frequencies()

def PLL2_P2_Update():
    frequency_update_manager.update_all_frequencies()

def CH0_1_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def CH2_3_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def CH4_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def CH5_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def CH6_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def CH7_MUX_Update(): 
    frequency_update_manager.update_all_frequencies()

def OUT0_1_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def OUT2_3_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def OUT4_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def OUT5_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def OUT6_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def OUT7_DIV_Update():
    frequency_update_manager.update_all_frequencies()

def sOUT7_DIV_Update():

    output_page_manager.out7_divider_update()
    frequency_update_manager.update_all_frequencies()

def OUT7_DIV_Update():

    sOUT7_DIV.iValue = (OUT7_DIV.iValue + 1) * (OUT7_STG2_DIV.iValue + 1)
    frequency_update_manager.update_all_frequencies()

def OUT7_STG2_DIV_Update():
    OUT7_DIV_Update()

# CHx_FMT update UI
def OUT0_FMT_Update_UI():
    output_page_manager.output_format_update_ui(0)

def OUT1_FMT_Update_UI():
    output_page_manager.output_format_update_ui(1)

def OUT2_FMT_Update_UI():
    output_page_manager.output_format_update_ui(2)

def OUT3_FMT_Update_UI():
    output_page_manager.output_format_update_ui(3)

def OUT4_FMT_Update_UI():
    output_page_manager.output_format_update_ui(4)

def OUT5_FMT_Update_UI():
    output_page_manager.output_format_update_ui(5)

def OUT6_FMT_Update_UI():
    output_page_manager.output_format_update_ui(6)

def OUT7_FMT_Update_UI():
    output_page_manager.output_format_update_ui(7)

# CHx_PD_Update_UI 
def CH0_1_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("0_1")

def CH2_3_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("2_3")

def CH4_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("4")

def CH5_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("5")

def CH6_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("6")

def CH7_PD_Update_UI():
    output_page_manager.output_powerdown_update_ui("7")


