""" 
 * Copyright (C) 2018 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED 
"""
from __future__ import division
# from __future__ import generators

import math
import fractions
import re
import itertools
from fractions import Fraction

try:
    gcd = math.gcd
except:
    gcd = fractions.gcd

def make_simple_fraction(input):
    # Updated make_fraction with Fred Owens update for increased flexibility.  TT 2019-07-03
    if not isinstance(input, str):
        ans = Fraction(input)
    elif "*" in input:
        ans = Fraction(1)
        for x in input.split("*"):
            ans *= make_fraction(x)
    elif "+" in input:
        if 'ans' not in locals():
            ans = Fraction(0)
        for x in input.split("+"):
            ans += make_fraction(x)
    elif len(re.split("[^e]\+", input)) == 2:
        whole, frac = re.split("[^e]\+", input)
        ans = Fraction(whole)
        
        if len(frac.split("/")) == 2:
            num, den = frac.split("/")
            
            ans = ans + Fraction(num) / Fraction(den)
    
    elif len(input.split("/")) == 2:
        num, den = input.split("/")
        
        ans = Fraction(num) / Fraction(den)
    
    else:
        ans = Fraction(input)

    return ans


def make_fraction(input):
    """
    Processes an input string like '74.25/1.001 + 1' or '1 + 2/3' or '(4.3 + 5 + -1) / 5) + 5.5")'  or a combination to return a Fraction.
    """
    # Updated make_fraction with Fred Owens update for increased flexibility.  TT 2019-07-03
    
    if not isinstance(input, str):
        final_fraction = make_simple_fraction(input)
    else:
        if input.count("(") != input.count(")"):
            raise ValueError("Must have matching parantheses")
        
        final_fraction = "" 
        
        # breaks long string into small groups
        
        for simple_group in re.split("\)", input):
            if "(" in simple_group:
                simple_group = simple_group.replace("(", "")
        
            if final_fraction == "":
                final_fraction = make_simple_fraction(simple_group)
           
            else:
                if "*" in simple_group:
                    simple_group = simple_group.replace("*", "")
                    final_fraction = final_fraction * make_simple_fraction(simple_group)
            
                elif "+" in simple_group: 
                    simple_group = simple_group.replace("+", "")
                    final_fraction = final_fraction + make_simple_fraction(simple_group)
                
                elif "/" in simple_group:
                    simple_group = simple_group.replace("/", "")
                    final_fraction = final_fraction / make_simple_fraction(simple_group)
 
    return final_fraction

def pmf(fract):
    """Prints a fraction as a mixed fraction"""
    whole = (fract.numerator // fract.denominator)
    print(whole, "+", fract-whole)

def solve_possible_vco_frequencies(vco_frequency_low, vco_frequency_high, output_frequency_list, max_solutions):

    ## Find the LCM frequency of all outputs.
    lcm_outputs_frequency = Fraction(output_frequency_list[0])
    if len(output_frequency_list) > 1:    
        for freq in output_frequency_list[1:]:
            lcm_outputs_frequency = lcm_Fraction(lcm_outputs_frequency, freq)

    divider_low = int(math.ceil(vco_frequency_low / lcm_outputs_frequency)) # I don't know why Ticspro throws an error if I don't add int(). ceil() and floor() are supposed to return int already.
    divider_high = int(math.floor(vco_frequency_high / lcm_outputs_frequency))

    if divider_low <= divider_high:
        if divider_high - divider_low + 1 > max_solutions: # return the middle portion of solutions
            divider_mid = int((divider_low + divider_high) / 2)
            return [lcm_outputs_frequency * divider for divider in range(divider_mid - int(max_solutions/2), divider_mid + int(max_solutions/2))]
        else:
            return [lcm_outputs_frequency * divider for divider in range(divider_low, divider_high + 1)] 
    else:
        return []

def solve_possible_vco_frequencies_multiple_prescalers(vco_frequency_low = 2920e6, vco_frequency_high = 3080e6, prescaler_list = [1], prescaler_num = 1, output_frequency_list = [40e6], output_divider_list = [[2,257]], max_solutions = 10):
    """
    Notes for inputs:
    
        1. vco_frequency_low, vco_frequency_high and items in output_frequency_list must be Fraction() compatible.
           This function does not handle exceptions from Fraction()
           
        2. If a PLL has multiple prescalers, this function assumes that all prescalers have the same list of values.
           Update needed if not.
           
        3. prescaler_num means the number of prescalers that this PLL has.
        
        3. output_divider_list is a list of list because each output frequency may have a different output divider. 
           The inner list is: [output_divider_min, output_divider_max]. output_divider_list should have the same order with output_frequency_list
        
    Notes for outputs:
    This function returns error_code, error_note, vco_freq_list, solutions, prescaler_selection. 
    
        1. error_code = 0, 1, 2 or 3
            0 - no error. 
            1 - no common multiple of output frequencies is in the VCO range. 
            2 - the output cannot be generated by existing prescaler/divider values. The first frequency that cannot be generated is returned in the error_note. 
            3 - no valid prescaler combination exists
        
        2. error_note. If error_code = 2, then it returns index of output_frequency_list, where output_frequency_list[index] = faulty output frequency. Otherwise, returns None.

        3. vco_frequency_list = a list of all valid VCO frequencies

        4. solutions = a list of list of tuples. This is used to implement prescaler index, prescaler value and output divider value for all output channels. From inside out:
            (1) A tuple = (vco_freq, prescaler_index, prescaler_value, output_divider) is one solution for one output frequency. precaler_index starts from 0. If there's only only one prescaler then ignore this one.
            (2) A list of (1). This list contains one possible solution for all output frequencies. This list has the same order with output_frequency_list
            (3) A list of (2). This list contains all possible solutions
    
        5. prescaler_selection is a list of lists. This is to be displayed on the frequency planner table. From inside out:
            (1) A list = [vco_freq, P1, P2, ...]. For example, if there are two prescalers but only one is used, then P2 is = 0.
            (2) A list of (1), which include all solutions. len(prescaler_selection) = len(solutions), so is the order
    """
    vco_frequency_low = Fraction(vco_frequency_low)
    vco_frequency_high = Fraction(vco_frequency_high)
    output_frequency_list = [Fraction(x) for x in output_frequency_list]
    
    possible_vco_frequencies = solve_possible_vco_frequencies(vco_frequency_low = vco_frequency_low, vco_frequency_high = vco_frequency_high, output_frequency_list = output_frequency_list, max_solutions = max_solutions)

    solutions = []
    prescaler_selection = []

    all_prescaler_combinations = [list(itertools.combinations(prescaler_list, x+1)) for x in range(prescaler_num)] # This includes all possible prescaler combinations. We will get the valid prescaler combinations later

    if possible_vco_frequencies: # if this is not an empty list

        which_frequency_cannot_be_generated = [0]*len(output_frequency_list) # A list that keeps track of frequency solution. It has output_frequency_list number of items. Whenever an output frequency cannot be generated from one VCO frequency in possible_vco_frequencies, there'll be an add-one to that component. In the end, we'll find out which output collects all the stars - an output frequency cannot be generated by any VCO frequency if the number is equal to len(possible_vco_frequencies). This is for error_code 2.
        valid_vco_mask = [True]*len(possible_vco_frequencies) # Not all possible VCOs are valid VCOs. Refer to error_code 2 and 3.

        for num_vco, vco_freq in enumerate(possible_vco_frequencies): 
            prescaler_solution_list = [[] for x in range(len(output_frequency_list))] # This is a temperory helper list. Inner list contains all prescaler solutions for one output frequency. Outer list contains all output frequencies for current VCO
            prescaler_num_needed = None # number of prescalers needed. Not all prescalers are used. For example, if there are two prescalers, we may only end up using one of them.
            prescaler_combination_list_verified = [] # A list that contains all verified prescaler combinations for current VCO. For example, if prescaler_num_needed = 1, this can be [(3,), (7,)]. If prescaler_num_needed = 2, this can be [(2, 3), (2, 5), (2, 7), (3, 7)]

            # The purpose of below block of codes is to find out prescaler_num_needed and prescaler_combination_list_verified
            # Step 1: find out all possible prescaler values for each output frequency. These values are stored in prescaler_solution_list
            for num_freq, freq in enumerate(output_frequency_list): 
                divider = int(vco_freq / freq)

                for factor in divisors_gen(divider): # divisor_generator is a generator that yields ALL divisors of the input. Notice that this is different than factorize. I don't use factorize because it's possible that 2 is not in prescaler list but 4 is, for example. If I use factorize then I'll miss 4. 
                    if factor in prescaler_list and output_divider_list[num_freq][0] <= divider / factor <= output_divider_list[num_freq][1]: # prescaler must be a divisor of vco_freq/freq and output divider must be within the range.
                        prescaler_solution_list[num_freq].append(factor) # One possible prescaler solution for this output frequency

            if [] in prescaler_solution_list: # if one output frequency cannot be generated by this VCO frequency, then this VCO frequency is useless. If one output frequency cannot be generated by any VCO frequency, then this output frequency is hopeless. We need to tell users which output frequency this is. See error_code 2 associated with error_note.
                for num, item in enumerate(prescaler_solution_list):
                    if item == []: which_frequency_cannot_be_generated[num] += 1;
                valid_vco_mask[num_vco] = False
                continue

            # Step 2: iterate through prescaler_num and find out valid prescaler combinations as well as prescaler_num_needed. 
            # By "valid" I mean, for example:
            # (1) If all output frequencies accept "4" as their prescalers, then (4, ) is a valid prescaler combination.
            # (2) If all output frequencies accept either "3" or "4" as their prescalers, then (3, 4) is a valid prescaler combination. 
            # Now, if one prescaler can cover all output frequencies, then there's no need to use two.
            for ps_index in range(prescaler_num):

                for ps_combination in all_prescaler_combinations[ps_index]: # Iterate through the the prescaler combinations for a certain number of prescalers. ps_combination is an iterator even if there's only one prescaler
                    if all([any([y in prescaler_solution_list[x] for y in ps_combination]) for x in range(len(output_frequency_list))]): # This is the key line. The purpose is to check if this prescaler combination can take care of all output frequencies uisng current VCO.
                        prescaler_combination_list_verified.append(ps_combination)
                if prescaler_combination_list_verified: # if this is not an empty list
                    prescaler_num_needed = ps_index + 1 # ps_index starts from 0, but prescaler_num_needed is at least 1.
                    break # if there's a solution with one prescaler, then there's no need to use two. So prescaler_num_needed only takes the lowest number.

            if prescaler_combination_list_verified == []: # if no prescaler combination is valid then this VCO is useless. Mark it up in the mask list.
                valid_vco_mask[num_vco] = False
                continue
            else:
                for ps_combination in prescaler_combination_list_verified:
                    template = ["N/A"]*(prescaler_num + 1) # template for "N/A" padding
                    item = [vco_freq] + [item for item in ps_combination]
                    template[:len(item)] = item
                    prescaler_selection.append(template) # append([vco_freq, ps1, ps2, ...]). If there are two prescalers but ps2 is not used, then it's padded with "N/A".

            # Now that we know how many prescalers are needed and the valid prescaler combinations, we can easily generate all the solutions
            for ps_combination in prescaler_combination_list_verified:
                # These 4 pre-defined lists will be zipped together to generate solutions.
                temp_vco_freq_list = []
                temp_prescaler_index_list = []
                temp_prescaler_value_list = []
                temp_output_divider_list = []
                
                for freq in output_frequency_list:
                    for ps_index in range(prescaler_num_needed):
                        prescaler = ps_combination[ps_index]
                        if vco_freq/freq % prescaler == 0:
                            temp_vco_freq_list.append(vco_freq)
                            temp_prescaler_index_list.append(ps_index)
                            temp_prescaler_value_list.append(prescaler)
                            temp_output_divider_list.append(int(vco_freq/freq/prescaler))
                            break # if one prescaler can generate this output frequency, then we need to stop checking the others.
                            
                solutions.append(list(zip(temp_vco_freq_list, temp_prescaler_index_list, temp_prescaler_value_list, temp_output_divider_list)))

        # Solutions have been generated. Now we need to update the error codes and return stuff.
        if len(possible_vco_frequencies) in which_frequency_cannot_be_generated: # If one output frequency cannot be generated by any possible VCO frequencies.
            return 2, which_frequency_cannot_be_generated.index(len(possible_vco_frequencies)), None, None, None
        elif prescaler_selection == []: # if no prescaler combination is valid.
            return 3, None, None, None, None
        else:
            vco_frequency_list = list(itertools.compress(possible_vco_frequencies, valid_vco_mask))
            return 0, None, vco_frequency_list, solutions, prescaler_selection

    else: # possible_vco_frequencies is an empty list
        return 1, None, None, None, None

def lcm_Fraction(self, other):
    scaler = (self.denominator * other.denominator) // gcd(self.denominator, other.denominator)
    left = self.numerator * (scaler // self.denominator)
    right = other.numerator * (scaler // other.denominator)
    
    ans = Fraction(left * right // gcd(left, right))
    ans /= scaler
    #ans.reduce()
    return ans

def ee(num, suffix="", fix=None):
    suffix_scaler_low = { -15:"f", -12:"p", -9:"n", -6:"u", -3: "m", 0:"" }
    suffix_scaler_high = { 0: "", 3:"k", 6:"M", 9:"G" }
    scaler_index = 0

    num = float(num)

    if (num >= 1) or (num <= -1):
        while (num >= 1e3 or num <= -1e3) and scaler_index<9:
            num /= 1e3
            scaler_index += 3

        if fix:
            num = round(num, fix)
        #retstr = "%f %s%s" % (num, suffix_scaler_high[scaler_index], suffix)
        retstr = "{} {}{}".format(num, suffix_scaler_high[scaler_index], suffix)
    else:
        while (num < 1 and num > -1) and scaler_index > -15:
            num *= 1e3
            scaler_index -= 3

        if fix:
            #convstr = "%%0.%df %%s%%s" % fix
            convstr = "{:.%df} {}{}" % fix
        else:
            #convstr = "%f %s%s"
            convstr = "{} {}{}"
            
        ##retstr = "%f %s%s" % (num, suffix_scaler_low[scaler_index], suffix)
        #retstr = convstr % (num, suffix_scaler_low[scaler_index], suffix)
        retstr = convstr.format(num, suffix_scaler_low[scaler_index], suffix)

    return retstr

def quick_factorize(N):
    factors = []
    n=2
    while n<=N:
        if N%n==0:
            factors.append(n)
            N/=n
        else:
            n+=1

    return factors

def reduce_list(alist):
    new_list = []
    for x in alist:
        if x not in new_list:
            new_list.append(x)
            
    return new_list

def reduce_list_with_count(alist):
    new_list = []
    alist = alist[:]
    while len(alist) > 0:
        val = alist[0]
        count = alist.count(val)
        new_list.append((val, count))
        for i in range(count):
            alist.pop(alist.index(val))
            
    return new_list

def lcm(a,b): 
    return (a*b) / gcd(a,b)

def divisors_gen(n):
    divs = [1]
    for i in range(2,int(math.sqrt(n))+1):
        if n%i == 0:
            divs.extend([i,int(n/i)])
    divs.extend([n])
    return list(set(divs))
    