/**
 * @file      sx126x_hal.h
 *
 * @brief     Hardware Abstraction Layer for SX126x
 *
 * The Clear BSD License
 * Copyright Semtech Corporation 2021. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the Semtech corporation nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY
 * THIS LICENSE. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SEMTECH CORPORATION BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sx126x_hal.h>
#include <stdbool.h>
#include <ti/drivers/GPIO.h>
#include <ti/drivers/Spi.h>
#include <HAL_Delay.h>
#include "board.h"

SPI_Handle      masterSpi;
/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC MACROS -----------------------------------------------------------
 */

/*
 * -----------------------------------------------------------------------------
 * --- PUBLIC CONSTANTS --------------------------------------------------------
 */

/**
 * @brief Write this to SPI bus while reading data, or as a dummy/placeholder
 */

sx126x_hal_status_t sx126x_hal_write(const void *context, const uint8_t *command, const uint16_t command_length,
                                     const uint8_t *data, const uint16_t data_length)
{    SPI_Transaction transaction;
     bool            transferOK;
     unsigned char masterRxBuffer[]={0x00};
     unsigned char masterTxBuffer[]={0x00};
	uint16_t i = 0;

	SX126x_hal_WaitOnBusy( );

    GPIO_write(NSS, 0);

    transaction.count = command_length;
    transaction.txBuf = (void *)command;
    transaction.rxBuf =  (void *) masterRxBuffer;
    transferOK = SPI_transfer(masterSpi, &transaction);

    transaction.count = data_length;
    transaction.txBuf = (void *)data;
    transaction.rxBuf =  (void *) masterRxBuffer;
    transferOK = SPI_transfer(masterSpi, &transaction);

    GPIO_write(NSS, 1);

    SX126x_hal_WaitOnBusy( );
	return SX126X_HAL_STATUS_OK;
}


/**
 * Radio data transfer - read
 *
 * @remark Shall be implemented by the user
 *
 * @param [in] context          Radio implementation parameters
 * @param [in] command          Pointer to the buffer to be transmitted
 * @param [in] command_length   Buffer size to be transmitted
 * @param [in] data             Pointer to the buffer to be received
 * @param [in] data_length      Buffer size to be received
 *
 * @returns Operation status
 */
sx126x_hal_status_t sx126x_hal_read(const void *context, const uint8_t *command, const uint16_t command_length,
                                    uint8_t *data, const uint16_t data_length)
{    SPI_Transaction transaction;
     bool            transferOK;
     unsigned char masterRxBuffer[]={0x00};
     unsigned char masterTxBuffer[]={0x00};
    uint16_t i = 0;

    SX126x_hal_WaitOnBusy( );
    GPIO_write(NSS, 0);

    transaction.count = command_length;
    transaction.txBuf = (void *)command;
    transaction.rxBuf =  (void *) masterRxBuffer;
    transferOK = SPI_transfer(masterSpi, &transaction);

    transaction.count = data_length;
    transaction.txBuf = (void *) masterTxBuffer;
    transaction.rxBuf =  (void *)data;
    transferOK = SPI_transfer(masterSpi, &transaction);

    GPIO_write(NSS, 1);

    SX126x_hal_WaitOnBusy( );
    return SX126X_HAL_STATUS_OK;
}

/**
 * Reset the radio
 *
 * @remark Shall be implemented by the user
 *
 * @param [in] context Radio implementation parameters
 *
 * @returns Operation status
 */
void sx126x_hal_reset(void)
{  GPIO_write(RST, 0);
   delay_us(100);
   GPIO_write(RST, 1);
   delay_us(100);
}

/**
 * Wake the radio up.
 *
 * @remark Shall be implemented by the user
 *
 * @param [in] context Radio implementation parameters
 *
 * @returns Operation status
 */
sx126x_hal_status_t sx126x_hal_wakeup(const void *context) {return SX126X_HAL_STATUS_OK;}




void SX126x_hal_WaitOnBusy(void)
{
   uint32_t u32_count = 0;

   while (GPIO_read(BUSY) == 1)
   {
      if (u32_count++ > 1000)
      {
         u32_count = 0;
      }
      delay_us(1);
   }
}

void SPI0_init()
{    SPI_Params      spiParams;
SPI_Params_init(&spiParams);
spiParams.dataSize = 8;
spiParams.frameFormat = SPI_POL0_PHA0;
spiParams.bitRate = 400000;
masterSpi = SPI_open(Board_SPI0, &spiParams);
}
