/*
 * Copyright (c) 2017-2018, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ====================== IPC_V2.c ===================================
 *  This file is responsible for setting up the board specific items for the
 *  IPC_V2 board.
 */

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <ti/devices/cc13x2_cc26x2/driverlib/ioc.h>
#include <ti/devices/cc13x2_cc26x2/driverlib/udma.h>
#include <ti/devices/cc13x2_cc26x2/inc/hw_ints.h>
#include <ti/devices/cc13x2_cc26x2/inc/hw_memmap.h>

#include "IPC_V2.h"

/*
 *  =============================== ADCBuf ===============================
 */
#include <ti/drivers/ADCBuf.h>
#include <ti/drivers/adcbuf/ADCBufCC26X2.h>

ADCBufCC26X2_Object adcBufCC26xxObjects[IPC_V2_ADCBUFCOUNT];

/*
 *  This table converts a virtual adc channel into a dio and internal analogue
 *  input signal. This table is necessary for the functioning of the adcBuf
 *  driver. Comment out unused entries to save flash. Dio and internal signal
 *  pairs are hardwired. Do not remap them in the table. You may reorder entire
 *  entries. The mapping of dio and internal signals is package dependent.
 */
const ADCBufCC26X2_AdcChannelLutEntry ADCBufCC26X2_adcChannelLut[IPC_V2_ADCBUF0CHANNELCOUNT] = {
    {IPC_V2_DIO23_ANALOG, ADC_COMPB_IN_AUXIO7},
    {IPC_V2_DIO24_ANALOG, ADC_COMPB_IN_AUXIO6},
    {IPC_V2_DIO25_ANALOG, ADC_COMPB_IN_AUXIO5},
    {IPC_V2_DIO26_ANALOG, ADC_COMPB_IN_AUXIO4},
    {IPC_V2_DIO27_ANALOG, ADC_COMPB_IN_AUXIO3},
    {IPC_V2_DIO28_ANALOG, ADC_COMPB_IN_AUXIO2},
    {IPC_V2_DIO29_ANALOG, ADC_COMPB_IN_AUXIO1},
    {PIN_UNASSIGNED, ADC_COMPB_IN_VDDS},
    {PIN_UNASSIGNED, ADC_COMPB_IN_DCOUPL},
    {PIN_UNASSIGNED, ADC_COMPB_IN_VSS},
};

const ADCBufCC26X2_HWAttrs adcBufCC26xxHWAttrs[IPC_V2_ADCBUFCOUNT] = {
    {
        .intPriority       = ~0,
        .swiPriority       = 0,
        .adcChannelLut     = ADCBufCC26X2_adcChannelLut,
        .gpTimerUnit       = IPC_V2_GPTIMER0A,
    }
};

const ADCBuf_Config ADCBuf_config[IPC_V2_ADCBUFCOUNT] = {
    {
        &ADCBufCC26X2_fxnTable,
        &adcBufCC26xxObjects[IPC_V2_ADCBUF0],
        &adcBufCC26xxHWAttrs[IPC_V2_ADCBUF0]
    },
};

const uint_least8_t ADCBuf_count = IPC_V2_ADCBUFCOUNT;

/*
 *  =============================== ADC ===============================
 */
#include <ti/drivers/ADC.h>
#include <ti/drivers/adc/ADCCC26XX.h>

ADCCC26XX_Object adcCC26xxObjects[IPC_V2_ADCCOUNT];

const ADCCC26XX_HWAttrs adcCC26xxHWAttrs[IPC_V2_ADCCOUNT] = {
    {
        .adcDIO              = IPC_V2_DIO27_ANALOG,
        .adcCompBInput       = ADC_COMPB_IN_AUXIO3,
        .refSource           = ADCCC26XX_FIXED_REFERENCE,
        .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_10P9_MS,
        .inputScalingEnabled = true,
        .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
        .returnAdjustedVal   = false
    },
    {
        .adcDIO              = IPC_V2_DIO28_ANALOG,
        .adcCompBInput       = ADC_COMPB_IN_AUXIO2,
        .refSource           = ADCCC26XX_FIXED_REFERENCE,
        .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_10P9_MS,
        .inputScalingEnabled = true,
        .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
        .returnAdjustedVal   = false
    },
};

const ADC_Config ADC_config[IPC_V2_ADCCOUNT] = {
    {&ADCCC26XX_fxnTable, &adcCC26xxObjects[IPC_V2_ADC4], &adcCC26xxHWAttrs[IPC_V2_ADC4]},
    {&ADCCC26XX_fxnTable, &adcCC26xxObjects[IPC_V2_ADC5], &adcCC26xxHWAttrs[IPC_V2_ADC5]},
};

const uint_least8_t ADC_count = IPC_V2_ADCCOUNT;

/*
 *  =============================== ECDH ===============================
 */
#include <ti/drivers/ECDH.h>
#include <ti/drivers/ecdh/ECDHCC26X2.h>

ECDHCC26X2_Object ecdhCC26X2Objects[IPC_V2_ECDHCOUNT];

const ECDHCC26X2_HWAttrs ecdhCC26X2HWAttrs[IPC_V2_ECDHCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const ECDH_Config ECDH_config[IPC_V2_ECDHCOUNT] = {
    {
         .object  = &ecdhCC26X2Objects[IPC_V2_ECDH0],
         .hwAttrs = &ecdhCC26X2HWAttrs[IPC_V2_ECDH0]
    },
};

const uint_least8_t ECDH_count = IPC_V2_ECDHCOUNT;

/*
 *  =============================== ECDSA ===============================
 */
#include <ti/drivers/ECDSA.h>
#include <ti/drivers/ecdsa/ECDSACC26X2.h>

ECDSACC26X2_Object ecdsaCC26X2Objects[IPC_V2_ECDSACOUNT];

const ECDSACC26X2_HWAttrs ecdsaCC26X2HWAttrs[IPC_V2_ECDSACOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const ECDSA_Config ECDSA_config[IPC_V2_ECDSACOUNT] = {
    {
         .object  = &ecdsaCC26X2Objects[IPC_V2_ECDSA0],
         .hwAttrs = &ecdsaCC26X2HWAttrs[IPC_V2_ECDSA0]
    },
};

const uint_least8_t ECDSA_count = IPC_V2_ECDSACOUNT;

/*
 *  =============================== ECJPAKE ===============================
 */
#include <ti/drivers/ECJPAKE.h>
#include <ti/drivers/ecjpake/ECJPAKECC26X2.h>

ECJPAKECC26X2_Object ecjpakeCC26X2Objects[IPC_V2_ECJPAKECOUNT];

const ECJPAKECC26X2_HWAttrs ecjpakeCC26X2HWAttrs[IPC_V2_ECJPAKECOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const ECJPAKE_Config ECJPAKE_config[IPC_V2_ECJPAKECOUNT] = {
    {
         .object  = &ecjpakeCC26X2Objects[IPC_V2_ECJPAKE0],
         .hwAttrs = &ecjpakeCC26X2HWAttrs[IPC_V2_ECJPAKE0]
    },
};

const uint_least8_t ECJPAKE_count = IPC_V2_ECJPAKECOUNT;


/*
 *  =============================== SHA2 ===============================
 */
#include <ti/drivers/SHA2.h>
#include <ti/drivers/sha2/SHA2CC26X2.h>

SHA2CC26X2_Object sha2CC26X2Objects[IPC_V2_SHA2COUNT];

const SHA2CC26X2_HWAttrs sha2CC26X2HWAttrs[IPC_V2_SHA2COUNT] = {
    {
        .intPriority       = ~0,
    }
};

const SHA2_Config SHA2_config[IPC_V2_SHA2COUNT] = {
    {
         .object  = &sha2CC26X2Objects[IPC_V2_SHA20],
         .hwAttrs = &sha2CC26X2HWAttrs[IPC_V2_SHA20]
    },
};

const uint_least8_t SHA2_count = IPC_V2_SHA2COUNT;

/*
 *  =============================== AESCCM ===============================
 */
#include <ti/drivers/AESCCM.h>
#include <ti/drivers/aesccm/AESCCMCC26XX.h>

AESCCMCC26XX_Object aesccmCC26XXObjects[IPC_V2_AESCCMCOUNT];

const AESCCMCC26XX_HWAttrs aesccmCC26XXHWAttrs[IPC_V2_AESCCMCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const AESCCM_Config AESCCM_config[IPC_V2_AESCCMCOUNT] = {
    {
         .object  = &aesccmCC26XXObjects[IPC_V2_AESCCM0],
         .hwAttrs = &aesccmCC26XXHWAttrs[IPC_V2_AESCCM0]
    },
};

const uint_least8_t AESCCM_count = IPC_V2_AESCCMCOUNT;

/*
 *  =============================== AESGCM ===============================
 */
#include <ti/drivers/AESGCM.h>
#include <ti/drivers/aesgcm/AESGCMCC26XX.h>

AESGCMCC26XX_Object aesgcmCC26XXObjects[IPC_V2_AESGCMCOUNT];

const AESGCMCC26XX_HWAttrs aesgcmCC26XXHWAttrs[IPC_V2_AESGCMCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const AESGCM_Config AESGCM_config[IPC_V2_AESGCMCOUNT] = {
    {
         .object  = &aesgcmCC26XXObjects[IPC_V2_AESGCM0],
         .hwAttrs = &aesgcmCC26XXHWAttrs[IPC_V2_AESGCM0]
    },
};

const uint_least8_t AESGCM_count = IPC_V2_AESGCMCOUNT;

/*
 *  =============================== AESCBC ===============================
 */
#include <ti/drivers/AESCBC.h>
#include <ti/drivers/aescbc/AESCBCCC26XX.h>

AESCBCCC26XX_Object aescbcCC26XXObjects[IPC_V2_AESCBCCOUNT];

const AESCBCCC26XX_HWAttrs aescbcCC26XXHWAttrs[IPC_V2_AESCBCCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const AESCBC_Config AESCBC_config[IPC_V2_AESCBCCOUNT] = {
    {
         .object  = &aescbcCC26XXObjects[IPC_V2_AESCBC0],
         .hwAttrs = &aescbcCC26XXHWAttrs[IPC_V2_AESCBC0]
    },
};

const uint_least8_t AESCBC_count = IPC_V2_AESCBCCOUNT;

/*
 *  =============================== AESCTR ===============================
 */
#include <ti/drivers/AESCTR.h>
#include <ti/drivers/aesctr/AESCTRCC26XX.h>

AESCTRCC26XX_Object aesctrCC26XXObjects[IPC_V2_AESCTRCOUNT];

const AESCTRCC26XX_HWAttrs aesctrCC26XXHWAttrs[IPC_V2_AESCTRCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const AESCTR_Config AESCTR_config[IPC_V2_AESCTRCOUNT] = {
    {
         .object  = &aesctrCC26XXObjects[IPC_V2_AESCTR0],
         .hwAttrs = &aesctrCC26XXHWAttrs[IPC_V2_AESCTR0]
    },
};

const uint_least8_t AESCTR_count = IPC_V2_AESCTRCOUNT;

/*
 *  =============================== AESECB ===============================
 */
#include <ti/drivers/AESECB.h>
#include <ti/drivers/aesecb/AESECBCC26XX.h>

AESECBCC26XX_Object aesecbCC26XXObjects[IPC_V2_AESECBCOUNT];

const AESECBCC26XX_HWAttrs aesecbCC26XXHWAttrs[IPC_V2_AESECBCOUNT] = {
    {
        .intPriority       = ~0,
    }
};

const AESECB_Config AESECB_config[IPC_V2_AESECBCOUNT] = {
    {
         .object  = &aesecbCC26XXObjects[IPC_V2_AESECB0],
         .hwAttrs = &aesecbCC26XXHWAttrs[IPC_V2_AESECB0]
    },
};

const uint_least8_t AESECB_count = IPC_V2_AESECBCOUNT;

/*
 *  =============================== AESCTRDRBG ===============================
 */
#include <ti/drivers/AESCTRDRBG.h>
#include <ti/drivers/aesctrdrbg/AESCTRDRBGXX.h>

AESCTRDRBGXX_Object aesctrdrbgXXObjects[IPC_V2_AESCTRDRBGCOUNT];

const AESCTRDRBGXX_HWAttrs aesctrdrbgXXHWAttrs[IPC_V2_AESCTRDRBGCOUNT] = {
    {
        .aesctrIndex       = IPC_V2_AESCTR0,
    }
};

const AESCTRDRBG_Config AESCTRDRBG_config[IPC_V2_AESCTRDRBGCOUNT] = {
    {
         .object  = &aesctrdrbgXXObjects[IPC_V2_AESCTRDRBG0],
         .hwAttrs = &aesctrdrbgXXHWAttrs[IPC_V2_AESCTRDRBG0]
    },
};

const uint_least8_t AESCTRDRBG_count = IPC_V2_AESCTRDRBGCOUNT;

/*
 *  =============================== TRNG ===============================
 */
#include <ti/drivers/TRNG.h>
#include <ti/drivers/trng/TRNGCC26XX.h>

TRNGCC26XX_Object trngCC26XXObjects[IPC_V2_TRNGCOUNT];

const TRNGCC26XX_HWAttrs trngCC26X2HWAttrs[IPC_V2_TRNGCOUNT] = {
    {
        .intPriority       = ~0,
        .swiPriority       = 0,
        .samplesPerCycle   = 240000,
    }
};

const TRNG_Config TRNG_config[IPC_V2_TRNGCOUNT] = {
    {
         .object  = &trngCC26XXObjects[IPC_V2_TRNG0],
         .hwAttrs = &trngCC26X2HWAttrs[IPC_V2_TRNG0]
    },
};

const uint_least8_t TRNG_count = IPC_V2_TRNGCOUNT;

/*
 *  =============================== Display ===============================
 */
#include <ti/display/Display.h>

const Display_Config *Display_config = NULL;
const uint_least8_t Display_count = 0;

/*
 *  =============================== GPIO ===============================
 */
#include <ti/drivers/GPIO.h>
#include <ti/drivers/gpio/GPIOCC26XX.h>

/*
 * Array of Pin configurations
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in IPC_V2.h
 * NOTE: Pins not used for interrupts should be placed at the end of the
 *       array. Callback entries can be omitted from callbacks array to
 *       reduce memory usage.
 */
GPIO_PinConfig gpioPinConfigs[] = {
    /* Input pins */
    GPIOCC26XX_DIO_15 | GPIO_DO_NOT_CONFIG,  /* Button 0 */
    GPIOCC26XX_DIO_14 | GPIO_DO_NOT_CONFIG,  /* Button 1 */

    GPIOCC26XX_DIO_15 | GPIO_DO_NOT_CONFIG,  /* IPC_V2_SPI_MASTER_READY */
    GPIOCC26XX_DIO_21 | GPIO_DO_NOT_CONFIG,  /* IPC_V2_SPI_SLAVE_READY */

    /* Output pins */
    GPIOCC26XX_DIO_07 | GPIO_DO_NOT_CONFIG,  /* Green LED */
    GPIOCC26XX_DIO_06 | GPIO_DO_NOT_CONFIG,  /* Red LED */
    GPIOCC26XX_DIO_30 | GPIO_DO_NOT_CONFIG,  /* TMP116_EN */

    /* SPI Flash CSN */
    GPIOCC26XX_DIO_20 | GPIO_DO_NOT_CONFIG,

    /* SD CS */
    GPIOCC26XX_DIO_21 | GPIO_DO_NOT_CONFIG,

    /* Sharp Display - GPIO configurations will be done in the Display files */
    GPIOCC26XX_DIO_24 | GPIO_DO_NOT_CONFIG, /* SPI chip select */
    GPIOCC26XX_DIO_22 | GPIO_DO_NOT_CONFIG, /* LCD power control */
    GPIOCC26XX_DIO_23 | GPIO_DO_NOT_CONFIG, /* LCD enable */

};

/*
 * Array of callback function pointers
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in CC1352R1_LAUNCH.h
 * NOTE: Pins not used for interrupts can be omitted from callbacks array to
 *       reduce memory usage (if placed at end of gpioPinConfigs array).
 */
GPIO_CallbackFxn gpioCallbackFunctions[] = {
    NULL,  /* Button 0 */
    NULL,  /* Button 1 */
    NULL,  /* IPC_V2_SPI_MASTER_READY */
    NULL,  /* IPC_V2_SPI_SLAVE_READY */
};

const GPIOCC26XX_Config GPIOCC26XX_config = {
    .pinConfigs         = (GPIO_PinConfig *)gpioPinConfigs,
    .callbacks          = (GPIO_CallbackFxn *)gpioCallbackFunctions,
    .numberOfPinConfigs = IPC_V2_GPIOCOUNT,
    .numberOfCallbacks  = sizeof(gpioCallbackFunctions)/sizeof(GPIO_CallbackFxn),
    .intPriority        = (~0)
};

/*
 *  =============================== GPTimer ===============================
 *  Remove unused entries to reduce flash usage both in Board.c and Board.h
 */
#include <ti/drivers/timer/GPTimerCC26XX.h>

GPTimerCC26XX_Object gptimerCC26XXObjects[IPC_V2_GPTIMERCOUNT];

const GPTimerCC26XX_HWAttrs gptimerCC26xxHWAttrs[IPC_V2_GPTIMERPARTSCOUNT] = {
    { .baseAddr = GPT0_BASE, .intNum = INT_GPT0A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT0, .pinMux = GPT_PIN_0A, },
    { .baseAddr = GPT0_BASE, .intNum = INT_GPT0B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT0, .pinMux = GPT_PIN_0B, },
    { .baseAddr = GPT1_BASE, .intNum = INT_GPT1A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT1, .pinMux = GPT_PIN_1A, },
    { .baseAddr = GPT1_BASE, .intNum = INT_GPT1B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT1, .pinMux = GPT_PIN_1B, },
    { .baseAddr = GPT2_BASE, .intNum = INT_GPT2A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT2, .pinMux = GPT_PIN_2A, },
    { .baseAddr = GPT2_BASE, .intNum = INT_GPT2B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT2, .pinMux = GPT_PIN_2B, },
    { .baseAddr = GPT3_BASE, .intNum = INT_GPT3A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT3, .pinMux = GPT_PIN_3A, },
    { .baseAddr = GPT3_BASE, .intNum = INT_GPT3B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT3, .pinMux = GPT_PIN_3B, },
};

const GPTimerCC26XX_Config GPTimerCC26XX_config[IPC_V2_GPTIMERPARTSCOUNT] = {
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER0], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER0A], GPT_A },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER0], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER0B], GPT_B },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER1], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER1A], GPT_A },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER1], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER1B], GPT_B },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER2], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER2A], GPT_A },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER2], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER2B], GPT_B },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER3], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER3A], GPT_A },
    { &gptimerCC26XXObjects[IPC_V2_GPTIMER3], &gptimerCC26xxHWAttrs[IPC_V2_GPTIMER3B], GPT_B },
};

/*
 *  =============================== I2C ===============================
*/
#include <ti/drivers/I2C.h>
#include <ti/drivers/i2c/I2CCC26XX.h>

I2CCC26XX_Object i2cCC26xxObjects[IPC_V2_I2CCOUNT];

const I2CCC26XX_HWAttrsV1 i2cCC26xxHWAttrs[IPC_V2_I2CCOUNT] = {
    {
        .baseAddr    = I2C0_BASE,
        .powerMngrId = PowerCC26XX_PERIPH_I2C0,
        .intNum      = INT_I2C_IRQ,
        .intPriority = ~0,
        .swiPriority = 0,
        .sdaPin      = IPC_V2_I2C0_SDA0,
        .sclPin      = IPC_V2_I2C0_SCL0,
    }
};

const I2C_Config I2C_config[IPC_V2_I2CCOUNT] = {
    {
        .fxnTablePtr = &I2CCC26XX_fxnTable,
        .object      = &i2cCC26xxObjects[IPC_V2_I2C0],
        .hwAttrs     = &i2cCC26xxHWAttrs[IPC_V2_I2C0]
    },
};

const uint_least8_t I2C_count = IPC_V2_I2CCOUNT;

/*
 *  =============================== I2S ===============================
*/
#include <ti/drivers/I2S.h>
#include <ti/drivers/i2s/I2SCC26XX.h>

I2SCC26XX_Object i2sCC26XXObjects[IPC_V2_I2SCOUNT];

const I2SCC26XX_HWAttrs i2sCC26XXHWAttrs[IPC_V2_I2SCOUNT] = {
    {
        .pinSD1      =  IPC_V2_I2S_ADI,
        .pinSD0      =  IPC_V2_I2S_ADO,
        .pinSCK      =  IPC_V2_I2S_BCLK,
        .pinMCLK     =  IPC_V2_I2S_MCLK,
        .pinWS       =  IPC_V2_I2S_WCLK,
        .intPriority = ~0,
    }
};

const I2S_Config I2S_config[IPC_V2_I2SCOUNT] = {
    {
        .object      = &i2sCC26XXObjects[IPC_V2_I2S0],
        .hwAttrs     = &i2sCC26XXHWAttrs[IPC_V2_I2S0]
    },
};

const uint_least8_t I2S_count = IPC_V2_I2SCOUNT;

/*
 *  =============================== NVS ===============================
 */
#include <ti/drivers/NVS.h>
#include <ti/drivers/nvs/NVSSPI25X.h>
#include <ti/drivers/nvs/NVSCC26XX.h>

#define NVS_REGIONS_BASE 0x52000 // Used by BLE stack, we are using the addresses below it (consult the memory map)
#define SECTORSIZE       0x2000
#define REGIONSIZE       (SECTORSIZE * 2)

#define REGIONSIZE_TEST1   (SECTORSIZE) // Twice, once for config and once for backup
#define REGIONSIZE_MEMCPY  (2 * SECTORSIZE)
#define REGIONSIZE_TEST3   (2 * SECTORSIZE)

#ifndef Board_EXCLUDE_NVS_INTERNAL_FLASH

/*
 * Reserve flash sectors for NVS driver use by placing an uninitialized byte
 * array at the desired flash address.
 */
#if defined(__TI_COMPILER_VERSION__)

/*
 * Place uninitialized array at NVS_REGIONS_BASE
 */
#pragma LOCATION(flashBuf, NVS_REGIONS_BASE);
#pragma NOINIT(flashBuf);
static char flashBuf[REGIONSIZE];

#pragma DATA_SECTION(flashBufTest1, ".test1");
static char flashBufTest1[REGIONSIZE_TEST1];

#pragma DATA_SECTION(flashBufTest2, ".test2");
static char flashBufTest2[REGIONSIZE_TEST1];

#pragma DATA_SECTION(flashBufMemcpy, ".memcpyRegion");
static char flashBufMemcpy[REGIONSIZE_MEMCPY];

#pragma DATA_SECTION(flashBufTest3, ".test3");
static char flashBufTest3[REGIONSIZE_TEST3];

#elif defined(__IAR_SYSTEMS_ICC__)

/*
 * Place uninitialized array at NVS_REGIONS_BASE
 */
static __no_init char flashBuf[REGIONSIZE] @ NVS_REGIONS_BASE;

#elif defined(__GNUC__)

/*
 * Place the flash buffers in the .nvs section created in the gcc linker file.
 * The .nvs section enforces alignment on a sector boundary but may
 * be placed anywhere in flash memory.  If desired the .nvs section can be set
 * to a fixed address by changing the following in the gcc linker file:
 *
 * .nvs (FIXED_FLASH_ADDR) (NOLOAD) : AT (FIXED_FLASH_ADDR) {
 *      *(.nvs)
 * } > REGION_TEXT
 */
__attribute__ ((section (".nvs")))
static char flashBuf[REGIONSIZE];

#endif

/* Allocate objects for NVS Internal Regions */
NVSCC26XX_Object nvsCC26xxObjects[IPC_V2_NVSCOUNT];

/* Hardware attributes for NVS Internal Regions */
const NVSCC26XX_HWAttrs nvsCC26xxHWAttrs[IPC_V2_NVSCOUNT] = {
    {
        .regionBase = (void *)flashBuf,
        .regionSize = REGIONSIZE,
    },
    {
        .regionBase = (void *)flashBufTest1,
        .regionSize = REGIONSIZE_TEST1,
    },
    {
        .regionBase = (void *)flashBufTest2,
        .regionSize = REGIONSIZE_TEST1,
    },
    {
        .regionBase = (void *)flashBufMemcpy,
        .regionSize = REGIONSIZE_MEMCPY,
    },
    {
        .regionBase = (void *)flashBufTest3,
        .regionSize = REGIONSIZE_TEST3,
    },
};

#endif /* Board_EXCLUDE_NVS_INTERNAL_FLASH */

/* NVS Region index 0 and 1 refer to NVS and NVS SPI respectively */
const NVS_Config NVS_config[IPC_V2_NVSCOUNT] = {
#ifndef Board_EXCLUDE_NVS_INTERNAL_FLASH
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[0],
        .hwAttrs = &nvsCC26xxHWAttrs[0],
    },
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[1],
        .hwAttrs = &nvsCC26xxHWAttrs[1],
    },
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[2],
        .hwAttrs = &nvsCC26xxHWAttrs[2],
    },
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[3],
        .hwAttrs = &nvsCC26xxHWAttrs[3],
    },
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[4],
        .hwAttrs = &nvsCC26xxHWAttrs[4],
    },
#endif
};

const uint_least8_t NVS_count = IPC_V2_NVSCOUNT;

/*
 *  =============================== PIN ===============================
 */
#include <ti/drivers/PIN.h>
#include <ti/drivers/pin/PINCC26XX.h>

const PIN_Config BoardGpioInitTable[] = {

    IPC_V2_PIN_WDC_LED | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MAX,       /* LED initially off */
    IPC_V2_PIN_SW1 | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_BOTHEDGES | PIN_HYSTERESIS,              /* Button is active low */
    IPC_V2_PIN_JMP1 | PIN_INPUT_EN | PIN_PULLUP | PIN_HYSTERESIS,
    IPC_V2_SPI_FLASH_CS | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_DRVSTR_MIN,     /* External flash chip select */
    IPC_V2_UART0_RX | PIN_INPUT_EN | PIN_PULLDOWN,                                                /* UART RX via debugger back channel */
    IPC_V2_UART0_TX | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL,                           /* UART TX via debugger back channel */
    IPC_V2_UART_RS485_EN_TX | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL,                   /* RS485 TX enable */

    IPC_V2_GPRS_PWR | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL,                           /* GPRS modem power enable circuit */
    IPC_V2_GPRS_EN | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL,                            /* GPRS modem power on pin */
    IPC_V2_UART1_RX | PIN_INPUT_EN | PIN_PULLDOWN,                                                /* UART RX via debugger back channel */
    IPC_V2_UART1_TX | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL,

    IPC_V2_SPI0_MOSI | PIN_INPUT_EN | PIN_PULLDOWN,                                               /* SPI master out - slave in */
    IPC_V2_SPI0_MISO | PIN_INPUT_EN | PIN_PULLDOWN,                                               /* SPI master in - slave out */
    IPC_V2_SPI0_CLK | PIN_INPUT_EN | PIN_PULLDOWN,                                                /* SPI clock */
    PIN_TERMINATE
};

const PINCC26XX_HWAttrs PINCC26XX_hwAttrs = {
    .intPriority = ~0,
    .swiPriority = 0
};

/*
 *  =============================== Power ===============================
 */
#include <ti/drivers/Power.h>
#include <ti/drivers/power/PowerCC26X2.h>

const PowerCC26X2_Config PowerCC26X2_config = {
    .policyInitFxn      = NULL,
    .policyFxn          = &PowerCC26XX_standbyPolicy,
    .calibrateFxn       = &PowerCC26XX_calibrate,
    .enablePolicy       = true,
    .calibrateRCOSC_LF  = true,
    .calibrateRCOSC_HF  = true,
};

/*
 *  =============================== PWM ===============================
 *  Remove unused entries to reduce flash usage both in Board.c and Board.h
 */
#include <ti/drivers/PWM.h>
#include <ti/drivers/pwm/PWMTimerCC26XX.h>

PWMTimerCC26XX_Object pwmtimerCC26xxObjects[IPC_V2_PWMCOUNT];

const PWMTimerCC26XX_HwAttrs pwmtimerCC26xxHWAttrs[IPC_V2_PWMCOUNT] = {
    { .pwmPin = IPC_V2_PWMPIN0, .gpTimerUnit = IPC_V2_GPTIMER0A },
    { .pwmPin = IPC_V2_PWMPIN1, .gpTimerUnit = IPC_V2_GPTIMER0B },
    { .pwmPin = IPC_V2_PWMPIN2, .gpTimerUnit = IPC_V2_GPTIMER1A },
    { .pwmPin = IPC_V2_PWMPIN3, .gpTimerUnit = IPC_V2_GPTIMER1B },
    { .pwmPin = IPC_V2_PWMPIN4, .gpTimerUnit = IPC_V2_GPTIMER2A },
    { .pwmPin = IPC_V2_PWMPIN5, .gpTimerUnit = IPC_V2_GPTIMER2B },
    { .pwmPin = IPC_V2_PWMPIN6, .gpTimerUnit = IPC_V2_GPTIMER3A },
    { .pwmPin = IPC_V2_PWMPIN7, .gpTimerUnit = IPC_V2_GPTIMER3B },
};

const PWM_Config PWM_config[IPC_V2_PWMCOUNT] = {
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM0], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM0] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM1], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM1] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM2], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM2] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM3], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM3] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM4], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM4] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM5], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM5] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM6], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM6] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[IPC_V2_PWM7], &pwmtimerCC26xxHWAttrs[IPC_V2_PWM7] },
};

const uint_least8_t PWM_count = IPC_V2_PWMCOUNT;

/*
 *  =============================== RF Driver ===============================
 */
#include <ti/drivers/rf/RF.h>

const RFCC26XX_HWAttrsV2 RFCC26XX_hwAttrs = {
    .hwiPriority        = ~0,     /* Lowest HWI priority */
    .swiPriority        = 0,      /* Lowest SWI priority */
    .xoscHfAlwaysNeeded = true,   /* Keep XOSC dependency while in standby */
    .globalCallback     = NULL,   /* No board specific callback */
    .globalEventMask    = 0       /* No events subscribed to */
};

/*
 *  =============================== SD ===============================
 */
#include <ti/drivers/SD.h>
#include <ti/drivers/sd/SDSPI.h>

SDSPI_Object sdspiObjects[IPC_V2_SDCOUNT];

const SDSPI_HWAttrs sdspiHWAttrs[IPC_V2_SDCOUNT] = {
    {
        .spiIndex = IPC_V2_SPI0,
        .spiCsGpioIndex = IPC_V2_SDSPI_CS
    }
};

const SD_Config SD_config[IPC_V2_SDCOUNT] = {
    {
        .fxnTablePtr = &SDSPI_fxnTable,
        .object = &sdspiObjects[IPC_V2_SDSPI0],
        .hwAttrs = &sdspiHWAttrs[IPC_V2_SDSPI0]
    },
};

const uint_least8_t SD_count = IPC_V2_SDCOUNT;

/*
 *  =============================== SPI DMA ===============================
 */
#include <ti/drivers/SPI.h>
#include <ti/drivers/spi/SPICC26X2DMA.h>

SPICC26X2DMA_Object spiCC26X2DMAObjects[IPC_V2_SPICOUNT];

/*
 * NOTE: The SPI instances below can be used by the SD driver to communicate
 * with a SD card via SPI.  The 'defaultTxBufValue' fields below are set to 0xFF
 * to satisfy the SDSPI driver requirement.
 */
const SPICC26X2DMA_HWAttrs spiCC26X2DMAHWAttrs[IPC_V2_SPICOUNT] = {
    {
        .baseAddr           = SSI0_BASE,
        .intNum             = INT_SSI0_COMB,
        .intPriority        = ~0,
        .swiPriority        = 0,
        .powerMngrId        = PowerCC26XX_PERIPH_SSI0,
        .defaultTxBufValue  = 0xFF,
        .rxChannelBitMask   = 1<<UDMA_CHAN_SSI0_RX,
        .txChannelBitMask   = 1<<UDMA_CHAN_SSI0_TX,
        .mosiPin            = IPC_V2_SPI0_MOSI,
        .misoPin            = IPC_V2_SPI0_MISO,
        .clkPin             = IPC_V2_SPI0_CLK,
        .csnPin             = IPC_V2_SPI0_CSN,
        .minDmaTransferSize = 10
    },
    {
        .baseAddr           = SSI1_BASE,
        .intNum             = INT_SSI1_COMB,
        .intPriority        = ~0,
        .swiPriority        = 0,
        .powerMngrId        = PowerCC26XX_PERIPH_SSI1,
        .defaultTxBufValue  = 0xFF,
        .rxChannelBitMask   = 1<<UDMA_CHAN_SSI1_RX,
        .txChannelBitMask   = 1<<UDMA_CHAN_SSI1_TX,
        .mosiPin            = IPC_V2_SPI1_MOSI,
        .misoPin            = IPC_V2_SPI1_MISO,
        .clkPin             = IPC_V2_SPI1_CLK,
        .csnPin             = IPC_V2_SPI1_CSN,
        .minDmaTransferSize = 10
    }
};

const SPI_Config SPI_config[IPC_V2_SPICOUNT] = {
    {
         .fxnTablePtr = &SPICC26X2DMA_fxnTable,
         .object      = &spiCC26X2DMAObjects[IPC_V2_SPI0],
         .hwAttrs     = &spiCC26X2DMAHWAttrs[IPC_V2_SPI0]
    },
    {
         .fxnTablePtr = &SPICC26X2DMA_fxnTable,
         .object      = &spiCC26X2DMAObjects[IPC_V2_SPI1],
         .hwAttrs     = &spiCC26X2DMAHWAttrs[IPC_V2_SPI1]
    },
};

const uint_least8_t SPI_count = IPC_V2_SPICOUNT;

/*
 *  =============================== UART ===============================
 */
#include <ti/drivers/UART.h>
#include <ti/drivers/uart/UARTCC26XX.h>

UARTCC26XX_Object uartCC26XXObjects[IPC_V2_UARTCOUNT];

uint8_t uartCC26XXRingBuffer[IPC_V2_UARTCOUNT][32];

const UARTCC26XX_HWAttrsV2 uartCC26XXHWAttrs[IPC_V2_UARTCOUNT] = {
    {
        .baseAddr       = UART0_BASE,
        .powerMngrId    = PowerCC26XX_PERIPH_UART0,
        .intNum         = INT_UART0_COMB,
        .intPriority    = ~0,
        .swiPriority    = 0,
        .txPin          = IPC_V2_UART0_TX,
        .rxPin          = IPC_V2_UART0_RX,
        .ctsPin         = PIN_UNASSIGNED,
        .rtsPin         = PIN_UNASSIGNED,
        .ringBufPtr     = uartCC26XXRingBuffer[IPC_V2_UART0],
        .ringBufSize    = sizeof(uartCC26XXRingBuffer[IPC_V2_UART0]),
        .txIntFifoThr   = UARTCC26XX_FIFO_THRESHOLD_1_8,
        .rxIntFifoThr   = UARTCC26XX_FIFO_THRESHOLD_4_8,
        .errorFxn       = NULL
    },
    {
        .baseAddr       = UART1_BASE,
        .powerMngrId    = PowerCC26X2_PERIPH_UART1,
        .intNum         = INT_UART1_COMB,
        .intPriority    = ~0,
        .swiPriority    = 0,
        .txPin          = IPC_V2_UART1_TX,
        .rxPin          = IPC_V2_UART1_RX,
        .ctsPin         = PIN_UNASSIGNED,
        .rtsPin         = PIN_UNASSIGNED,
        .ringBufPtr     = uartCC26XXRingBuffer[IPC_V2_UART1],
        .ringBufSize    = sizeof(uartCC26XXRingBuffer[IPC_V2_UART1]),
        .txIntFifoThr   = UARTCC26XX_FIFO_THRESHOLD_1_8,
        .rxIntFifoThr   = UARTCC26XX_FIFO_THRESHOLD_4_8,
        .errorFxn       = NULL
    }
};

const UART_Config UART_config[IPC_V2_UARTCOUNT] = {
    {
        .fxnTablePtr = &UARTCC26XX_fxnTable,
        .object      = &uartCC26XXObjects[IPC_V2_UART0],
        .hwAttrs     = &uartCC26XXHWAttrs[IPC_V2_UART0]
    },
    {
        .fxnTablePtr = &UARTCC26XX_fxnTable,
        .object      = &uartCC26XXObjects[IPC_V2_UART1],
        .hwAttrs     = &uartCC26XXHWAttrs[IPC_V2_UART1]
    },
};

const uint_least8_t UART_count = IPC_V2_UARTCOUNT;

/*
 *  =============================== UDMA ===============================
 */
#include <ti/drivers/dma/UDMACC26XX.h>

UDMACC26XX_Object udmaObjects[IPC_V2_UDMACOUNT];

const UDMACC26XX_HWAttrs udmaHWAttrs[IPC_V2_UDMACOUNT] = {
    {
        .baseAddr    = UDMA0_BASE,
        .powerMngrId = PowerCC26XX_PERIPH_UDMA,
        .intNum      = INT_DMA_ERR,
        .intPriority = ~0
    }
};

const UDMACC26XX_Config UDMACC26XX_config[IPC_V2_UDMACOUNT] = {
    {
         .object  = &udmaObjects[IPC_V2_UDMA0],
         .hwAttrs = &udmaHWAttrs[IPC_V2_UDMA0]
    },
};



/*
 *  =============================== Watchdog ===============================
 */
#include <ti/drivers/Watchdog.h>
#include <ti/drivers/watchdog/WatchdogCC26XX.h>

WatchdogCC26XX_Object watchdogCC26XXObjects[IPC_V2_WATCHDOGCOUNT];

const WatchdogCC26XX_HWAttrs watchdogCC26XXHWAttrs[IPC_V2_WATCHDOGCOUNT] = {
    {
        .baseAddr    = WDT_BASE,
        .reloadValue = 1000 /* Reload value in milliseconds */
    },
};

const Watchdog_Config Watchdog_config[IPC_V2_WATCHDOGCOUNT] = {
    {
        .fxnTablePtr = &WatchdogCC26XX_fxnTable,
        .object      = &watchdogCC26XXObjects[IPC_V2_WATCHDOG0],
        .hwAttrs     = &watchdogCC26XXHWAttrs[IPC_V2_WATCHDOG0]
    },
};

const uint_least8_t Watchdog_count = IPC_V2_WATCHDOGCOUNT;

/*
 *  Board-specific initialization function to disable external flash.
 *  This function is defined in the file IPC_V2_fxns.c
 */
extern void Board_initHook(void);

/*
 *  ======== IPC_V2_initGeneral ========
 */
void IPC_V2_initGeneral(void)
{
    Power_init();

    if (PIN_init(BoardGpioInitTable) != PIN_SUCCESS) {
        /* Error with PIN_init */
        while (1);
    }

    /* Perform board-specific initialization */
    Board_initHook();
}

/*
 *  ======== Board_init ========
 */
void Board_init(void)
{
    IPC_V2_initGeneral();
}
