/*
 * Copyright (c) 2015-2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== empty.c ========
 */

/* For usleep() */
#include <unistd.h>
#include <stdint.h>
#include <stddef.h>

/* Driver Header files */
#include <ti/drivers/GPIO.h>

#include <ti/drivers/apps/Button.h>

#include <ti/display/Display.h>
#include <ti/display/DisplayUart2.h>
#include <ti/display/AnsiColor.h>

#include <ti/drivers/dpl/EventP.h>

#include <ti/devices/cc27xx/driverlib/hapi.h>

/* Driver configuration */
#include "ti_drivers_config.h"

#define PRIMARY_SLOT_ADDR   0x7000
#define SECONDARY_SLOT_ADDR 0x39000

#define TRUE 1
#define FALSE 0

/* EventP */
#define EVENT_BUTTON_LEFT_CLICKED     (0x00000001 << 0)
#define EVENT_BUTTON_RIGHT_CLICKED    (0x00000001 << 1)
#define EVENT_BUTTON_LEFT_LONGPRESS   (0x00000001 << 2)
#define EVENT_BUTTON_RIGHT_LONGPRESS  (0x00000001 << 3)
#define EVENT_ALL   EVENT_BUTTON_LEFT_CLICKED    | \
                    EVENT_BUTTON_RIGHT_CLICKED   | \
                    EVENT_BUTTON_LEFT_LONGPRESS  | \
                    EVENT_BUTTON_RIGHT_LONGPRESS

EventP_Handle e_handle;
uint32_t e_mask = EVENT_ALL;
uint32_t events;

Display_Handle d_handle;
Display_Params d_params;

Button_Handle b_right_handle;
Button_Handle b_left_handle;
Button_Params b_params;

void button_callback_left(Button_Handle handle, Button_EventMask events);
void button_callback_right(Button_Handle handle, Button_EventMask events);

static __attribute__((naked)) void start_app(uint32_t *vector_table)
{
    __asm volatile ("ldr r1, [r0, #0]\n"
    "msr msp, r1\n"
    "ldr r1, [r0, #4]\n"
    "mov pc, r1\n");
}

/*
 *  ======== mainThread ========
 */
void *mainThread(void *arg0)
{
    GPIO_init();
    Button_init();
    Display_init();

    /* Open a display */
    Display_Params_init(&d_params);
    d_handle = Display_open(Display_Type_UART, &d_params);

    Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "============= SSB ============");
    Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "Right Button: Primary Image.");
    Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "Left Button : Secondary Image.");
    Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "==============================");

    e_handle = EventP_create();
    events = 0;

    Button_Params_init(&b_params);
    b_params.buttonEventMask = Button_EV_CLICKED | Button_EV_LONGPRESSED;
    b_params.buttonCallback = button_callback_left;
    Button_open(BUTTON_LEFT, &b_params);
    b_params.buttonCallback = button_callback_right;
    Button_open(BUTTON_RIGHT, &b_params);

    while(TRUE)
    {
        events = EventP_pend(e_handle, e_mask, FALSE, EventP_WAIT_FOREVER);
        switch (events)
        {
            case EVENT_BUTTON_RIGHT_CLICKED:
            {
                Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "Starting Primary...");
                start_app((uint32_t *)PRIMARY_SLOT_ADDR);
                break;
            }
            case EVENT_BUTTON_LEFT_CLICKED:
            {
                Display_printf(d_handle, DisplayUart2_SCROLLING, 0, "Starting Secondary...");
                start_app((uint32_t *)SECONDARY_SLOT_ADDR);
                break;
            }

            default: break;
        }
    }
}

/*
 * ==== button_callback_left ==== 
 */
void button_callback_left(Button_Handle handle, Button_EventMask events)
{
    if (events & Button_EV_CLICKED)
    {
        EventP_post(e_handle, EVENT_BUTTON_LEFT_CLICKED);
    }
    if (events & Button_EV_LONGPRESSED)
    {
        EventP_post(e_handle, EVENT_BUTTON_LEFT_LONGPRESS);
    }
}

/*
 * ==== button_callback_right ==== 
 */
void button_callback_right(Button_Handle handle, Button_EventMask events)
{
    if (events & Button_EV_CLICKED)
    {
        EventP_post(e_handle, EVENT_BUTTON_RIGHT_CLICKED);
    }
    if (events & Button_EV_LONGPRESSED)
    {
        EventP_post(e_handle, EVENT_BUTTON_RIGHT_LONGPRESS);
    }
}