/*****< hcitrans.c >***********************************************************/
/*      Copyright 2012 - 2014 Stonestreet One.                                */
/*      All Rights Reserved.                                                  */
/*                                                                            */
/*  HCITRANS - HCI Transport Layer for use with Bluetopia.                    */
/*                                                                            */
/*  Author:  Marcus Funk                                                      */
/*                                                                            */
/*** MODIFICATION HISTORY *****************************************************/
/*                                                                            */
/*   mm/dd/yy  F. Lastname    Description of Modification                     */
/*   --------  -----------    ------------------------------------------------*/
/*   11/08/12  M. Funk        Initial creation.                               */
/******************************************************************************/

#include "BTPSKRNL.h"       /* Bluetooth Kernel Prototypes/Constants.         */
#include "HCITRANS.h"       /* HCI Transport Prototypes/Constants.            */
#include "HCITRCFG.h"       /* HCI Transport configuration.                   */

#define INPUT_BUFFER_SIZE        1056
#define OUTPUT_BUFFER_SIZE       1056

#define ClearReset()             BT_PWR_EN(1)
#define SetReset()               BT_PWR_EN(0)

#define DisableInterrupts()      __set_PRIMASK(1)
#define EnableInterrupts()       __set_PRIMASK(0)

#define INTERRUPT_PRIORITY       5

#define TRANSPORT_ID             1

#define DEBUG_PRINT              BTPS_OutputMessage

typedef enum
{
   hssNormal,
   hssSuspendWait,
   hssSuspendWaitInterrupted,
   hssSuspended
} SuspendState_t;

typedef struct _tagUartContext_t
{
#ifdef HCITR_ENABLE_DEBUG_LOGGING

   Boolean_t                DebugEnabled;

#endif

   SuspendState_t           SuspendState;

   HCITR_COMDataCallback_t  COMDataCallbackFunction;
   unsigned long            COMDataCallbackParameter;

   unsigned short           RxInIndex;
   unsigned short           RxOutIndex;
   volatile unsigned short  RxBytesFree;
   unsigned char            RxBuffer[INPUT_BUFFER_SIZE];

   unsigned short           TxInIndex;
   unsigned short           TxOutIndex;
   volatile unsigned short  TxBytesFree;
   unsigned char            TxBuffer[OUTPUT_BUFFER_SIZE];
} UartContext_t;

extern UART_HandleTypeDef huart2;
extern void MX_USART2_UART_Init(void);
extern  void BT_PWR_EN(int8_t val);

static UartContext_t              UartContext;
static int                        HCITransportOpen        = 0;

   /* Local Function Prototypes.                                        */
void HCI_TxInterrupt(void);
void HCI_RxInterrupt(void);

   /* The following function will reconfigure the BAUD rate without     */
   /* reconfiguring the entire port.  This function is also potentially */
   /* more accurate than the method used in the ST standard peripheral  */
   /* libraries.                                                        */
static void SetBaudRate(unsigned int BaudRate)
{
   unsigned int      SourceFrequency;
   unsigned int      Divider;
   unsigned int      TempDiv;


   SourceFrequency = HAL_RCC_GetPCLK1Freq();

   Divider = (((SourceFrequency * 2) / BaudRate) + 1) / 2;

   if(Divider < 16)
   {
      TempDiv = Divider & 0x7;
      Divider = (Divider & ~0x07) << 1;
      Divider |= TempDiv;

      /* Disable the UART while updating the baud rate.                 */
      huart2.Instance->CR1 &= ~USART_CR1_UE;
      huart2.Instance->CR1 |= USART_CR1_OVER8;
   }
   else
   {
      /* For lower frequencies, 16bit oversampling may be used.         */

      /* Disable the UART while updating the baud rate.                 */
      huart2.Instance->CR1 &= ~USART_CR1_UE;
      huart2.Instance->CR1 &= ~USART_CR1_OVER8;
   }

   huart2.Instance->BRR = Divider;
   huart2.Instance->CR1 |= USART_CR1_UE;
}

void HCI_TxInterrupt(void)
{
   while((UartContext.TxBytesFree != OUTPUT_BUFFER_SIZE) && (huart2.Instance->SR & USART_SR_TXE))
   {
      huart2.Instance->DR = (UartContext.TxBuffer[UartContext.TxOutIndex] & (uint8_t)0xFFU);

      UartContext.TxBytesFree ++;
      UartContext.TxOutIndex ++;
      if(UartContext.TxOutIndex == OUTPUT_BUFFER_SIZE)
         UartContext.TxOutIndex = 0;
   }

   if(UartContext.TxBytesFree == OUTPUT_BUFFER_SIZE)
   {
      CLEAR_BIT(huart2.Instance->CR1, USART_CR1_TXEIE);
        /* Disable the UART Transmit Complete Interrupt */
      CLEAR_BIT(huart2.Instance->CR1, USART_CR1_TCIE);
   }
}

void HCI_RxInterrupt(void)
{
   while((UartContext.RxBytesFree) && (huart2.Instance->SR & (USART_SR_RXNE | USART_SR_ORE)))
   {
      if(huart2.Instance->SR & USART_SR_ORE)
         DBG_MSG(DBG_ZONE_GENERAL, ("Receive Overflow\r\n"));

      UartContext.RxBuffer[UartContext.RxInIndex] = (unsigned char)(huart2.Instance->DR & (uint16_t)0x01FF);

      UartContext.RxBytesFree --;
      UartContext.RxInIndex ++;
      if(UartContext.RxInIndex == INPUT_BUFFER_SIZE)
         UartContext.RxInIndex = 0;
   }

   if(!UartContext.RxBytesFree)
   {
      CLEAR_BIT(huart2.Instance->CR1, (USART_CR1_RXNEIE | USART_CR1_PEIE));
      CLEAR_BIT(huart2.Instance->CR3, USART_CR3_EIE);
   }
}

int BTPSAPI HCITR_COMOpen(HCI_COMMDriverInformation_t *COMMDriverInformation, HCITR_COMDataCallback_t COMDataCallback, unsigned long CallbackParameter)
{
   int ret_val;

   /* First, make sure that the port is not already open and make sure  */
   /* that valid COMM Driver Information was specified.                 */
   if((!HCITransportOpen) && (COMMDriverInformation) && (COMDataCallback))
   {
      /* Initialize the return value for success.                       */
      ret_val                              = TRANSPORT_ID;

      /* Flag that the HCI Transport is open.                           */
      HCITransportOpen                     = 1;

      /* Initialize the context structure.                              */
      BTPS_MemInitialize(&UartContext, 0, sizeof(UartContext_t));

      UartContext.COMDataCallbackFunction  = COMDataCallback;
      UartContext.COMDataCallbackParameter = CallbackParameter;
      UartContext.TxBytesFree              = OUTPUT_BUFFER_SIZE;
      UartContext.RxBytesFree              = INPUT_BUFFER_SIZE;
      UartContext.SuspendState             = hssNormal;

      SetReset();

      SetBaudRate(COMMDriverInformation->BaudRate);

      __HAL_UART_ENABLE_IT(&huart2, UART_IT_PE);
      __HAL_UART_ENABLE_IT(&huart2, UART_IT_ERR);
      __HAL_UART_ENABLE_IT(&huart2, UART_IT_RXNE);

      /* Enable the UART Parity Error Interrupt */

      /* Enable the UART Error Interrupt: (Frame error, noise error, overrun error) */

      /* Enable the UART Data Register not empty Interrupt */

      /* Clear the reset.                                               */
      BTPS_Delay(10);
      ClearReset();
      BTPS_Delay(250);
   }
   else
      ret_val = HCITR_ERROR_UNABLE_TO_OPEN_TRANSPORT;

   return(ret_val);
}

   /* The following function is responsible for closing the specific HCI*/
   /* Transport layer that was opened via a successful call to the      */
   /* HCITR_COMOpen() function (specified by the first parameter).      */
   /* Bluetopia makes a call to this function whenever an either        */
   /* Bluetopia is closed, or an error occurs during initialization and */
   /* the driver has been opened (and ONLY in this case).  Once this    */
   /* function completes, the transport layer that was closed will no   */
   /* longer process received data until the transport layer is         */
   /* Re-Opened by calling the HCITR_COMOpen() function.                */
   /* * NOTE * This function *MUST* close the specified COM Port.  This */
   /*          module will then call the registered COM Data Callback   */
   /*          function with zero as the data length and NULL as the    */
   /*          data pointer.  This will signify to the HCI Driver that  */
   /*          this module is completely finished with the port and     */
   /*          information and (more importantly) that NO further data  */
   /*          callbacks will be issued.  In other words the very last  */
   /*          data callback that is issued from this module *MUST* be a*/
   /*          data callback specifying zero and NULL for the data      */
   /*          length and data buffer (respectively).                   */
void BTPSAPI HCITR_COMClose(unsigned int HCITransportID)
{
   HCITR_COMDataCallback_t COMDataCallback;


   /* Check to make sure that the specified Transport ID is valid.      */
   if((HCITransportID == TRANSPORT_ID) && (HCITransportOpen))
   {
      /* Flag that the HCI Transport is no longer open.                 */
      HCITransportOpen = 0;


      /* Appears to be valid, go ahead and close the port.              */
            /* Disable the UART Parity Error Interrupt and RXNE interrupt*/
      __HAL_UART_DISABLE_IT(&huart2, UART_IT_RXNE);
      __HAL_UART_DISABLE_IT(&huart2, UART_IT_PE);

      /* Disable the UART Error Interrupt: (Frame error, noise error, overrun error) */
      __HAL_UART_DISABLE_IT(&huart2, UART_IT_ERR);
      __HAL_UART_DISABLE_IT(&huart2, UART_IT_TXE);
        /* Disable the UART Transmit Complete Interrupt */
      __HAL_UART_DISABLE_IT(&huart2, UART_IT_TC);

      /* Place the Bluetooth Device in Reset.                           */
      SetReset();


      /* Note the Callback information.                                 */
      COMDataCallback   = UartContext.COMDataCallbackFunction;

      UartContext.COMDataCallbackFunction = NULL;

      /* All finished, perform the callback to let the upper layer know */
      /* that this module will no longer issue data callbacks and is    */
      /* completely cleaned up.                                         */
      if(COMDataCallback)
         (*COMDataCallback)(HCITransportID, 0, NULL, UartContext.COMDataCallbackParameter);

      UartContext.COMDataCallbackParameter = 0;
   }
}

   /* The following function is responsible for instructing the         */
   /* specified HCI Transport layer (first parameter) that was opened   */
   /* via a successful call to the HCITR_COMOpen() function to          */
   /* reconfigure itself with the specified information.                */
   /* * NOTE * This function does not close the HCI Transport specified */
   /*          by HCI Transport ID, it merely reconfigures the          */
   /*          transport.  This means that the HCI Transport specified  */
   /*          by HCI Transport ID is still valid until it is closed via*/
   /*          the HCI_COMClose() function.                             */
void BTPSAPI HCITR_COMReconfigure(unsigned int HCITransportID, HCI_Driver_Reconfigure_Data_t *DriverReconfigureData)
{
   HCI_COMMReconfigureInformation_t *ReconfigureInformation;

   /* Check to make sure that the specified Transport ID is valid.      */
   if((HCITransportID == TRANSPORT_ID) && (HCITransportOpen) && (DriverReconfigureData))
   {
      if((DriverReconfigureData->ReconfigureCommand == HCI_COMM_DRIVER_RECONFIGURE_DATA_COMMAND_CHANGE_COMM_PARAMETERS) && (DriverReconfigureData->ReconfigureData))
      {
         ReconfigureInformation = (HCI_COMMReconfigureInformation_t *)(DriverReconfigureData->ReconfigureData);

         /* Check if the baud rate needs to change.                     */
         if(ReconfigureInformation->ReconfigureFlags & (HCI_COMM_RECONFIGURE_INFORMATION_RECONFIGURE_FLAGS_CHANGE_BAUDRATE | HCI_COMM_RECONFIGURE_INFORMATION_RECONFIGURE_FLAGS_CHANGE_PROTOCOL))
         {
            DisableInterrupts();
            SetBaudRate(ReconfigureInformation->BaudRate);
            EnableInterrupts();
         }
      }
   }
}

   /* The following function is provided to allow a mechanism for       */
   /* modules to force the processing of incoming COM Data.             */
   /* * NOTE * This function is only applicable in device stacks that   */
   /*          are non-threaded.  This function has no effect for device*/
   /*          stacks that are operating in threaded environments.      */
void BTPSAPI HCITR_COMProcess(unsigned int HCITransportID)
{
   unsigned int MaxLength;
   unsigned int TotalLength;

#ifdef HCITR_ENABLE_DEBUG_LOGGING

   unsigned int Index;

#endif

   /* Check to make sure that the specified Transport ID is valid.      */
   if((HCITransportID == TRANSPORT_ID) && (HCITransportOpen))
   {
      /* Loop until the receive buffer is empty.                        */
      while((TotalLength = (INPUT_BUFFER_SIZE - UartContext.RxBytesFree)) != 0)
      {
         /* Determine the maximum number of characters that we can send */
         /* before we reach the end of the buffer.  We need to process  */
         /* the smaller of the max characters or the number of          */
         /* characters that are in the buffer.                          */
         MaxLength   = (INPUT_BUFFER_SIZE - UartContext.RxOutIndex);
         if(TotalLength > MaxLength)
            TotalLength = MaxLength;

#ifdef HCITR_ENABLE_DEBUG_LOGGING

         if(UartContext.DebugEnabled)
         {
            DEBUG_PRINT(">");

            for(Index = 0; Index < TotalLength; Index ++)
               DEBUG_PRINT(" %02X", UartContext.RxBuffer[UartContext.RxOutIndex + Index]);

            DEBUG_PRINT("\r\n");
         }

#endif

         /* Call the upper layer back with the data.                    */
         if(UartContext.COMDataCallbackFunction)
            (*UartContext.COMDataCallbackFunction)(TRANSPORT_ID, TotalLength, &(UartContext.RxBuffer[UartContext.RxOutIndex]), UartContext.COMDataCallbackParameter);

         /* Adjust the Out Index and handle any looping.                */
         UartContext.RxOutIndex += TotalLength;
         if(UartContext.RxOutIndex == INPUT_BUFFER_SIZE)
            UartContext.RxOutIndex = 0;

         /* Credit the amount that was processed and make sure the      */
         /* receive interrupt is enabled.                               */
         DisableInterrupts();
         UartContext.RxBytesFree += TotalLength;
         __HAL_UART_ENABLE_IT(&huart2, UART_IT_RXNE);
         EnableInterrupts();
      }
   }
}

   /* The following function is responsible for actually sending data   */
   /* through the opened HCI Transport layer (specified by the first    */
   /* parameter).  Bluetopia uses this function to send formatted HCI   */
   /* packets to the attached Bluetooth Device.  The second parameter to*/
   /* this function specifies the number of bytes pointed to by the     */
   /* third parameter that are to be sent to the Bluetooth Device.  This*/
   /* function returns a zero if the all data was transferred           */
   /* successfully or a negative value if an error occurred.  This      */
   /* function MUST NOT return until all of the data is sent (or an     */
   /* error condition occurs).  Bluetopia WILL NOT attempt to call this */
   /* function repeatedly if data fails to be delivered.  This function */
   /* will block until it has either buffered the specified data or sent*/
   /* all of the specified data to the Bluetooth Device.                */
   /* * NOTE * The type of data (Command, ACL, SCO, etc.) is NOT passed */
   /*          to this function because it is assumed that this         */
   /*          information is contained in the Data Stream being passed */
   /*          to this function.                                        */
int BTPSAPI HCITR_COMWrite(unsigned int HCITransportID, unsigned int Length, unsigned char *Buffer)
{
   int ret_val;
   int Count;
   int BytesFree;

#ifdef HCITR_ENABLE_DEBUG_LOGGING

   unsigned int Index;

#endif

   /* Check to make sure that the specified Transport ID is valid and   */
   /* the output buffer appears to be valid as well.                    */
   if((HCITransportID == TRANSPORT_ID) && (HCITransportOpen) && (Length) && (Buffer))
   {
      /* If the UART is suspended, resume it.                           */
      if(UartContext.SuspendState == hssSuspended)
      {
         DisableInterrupts();

         __HAL_RCC_USART2_CLK_DISABLE();
         UartContext.SuspendState = hssNormal;

         EnableInterrupts();
      }

#ifdef HCITR_ENABLE_DEBUG_LOGGING

      if(UartContext.DebugEnabled)
      {
         DEBUG_PRINT("<");

         for(Index = 0; Index < Length; Index ++)
            DEBUG_PRINT(" %02X", Buffer[Index]);

         DEBUG_PRINT("\r\n");
      }

#endif

      /* Process all of the data.                                       */
      while(Length)
      {
         /* Wait for space in the transmit buffer.                      */
         while(!UartContext.TxBytesFree) {}

         /* The data may have to be copied in 2 phases.  Calculate the  */
         /* number of character that can be placed in the buffer before */
         /* the buffer must be wrapped.                                 */
         BytesFree = UartContext.TxBytesFree;
         Count = Length;
         Count = (BytesFree < Count) ? BytesFree : Count;
         Count = ((OUTPUT_BUFFER_SIZE - UartContext.TxInIndex) < Count) ? (OUTPUT_BUFFER_SIZE - UartContext.TxInIndex) : Count;

         BTPS_MemCopy(&(UartContext.TxBuffer[UartContext.TxInIndex]), Buffer, Count);

         /* Adjust the index values.                                    */
         Buffer                  += Count;
         Length                  -= Count;
         UartContext.TxInIndex   += Count;
         if(UartContext.TxInIndex == OUTPUT_BUFFER_SIZE)
            UartContext.TxInIndex = 0;

         /* Update the bytes free and make sure the transmit interrupt  */
         /* is enabled.                                                 */
         DisableInterrupts();
         UartContext.TxBytesFree -= Count;
         __HAL_UART_ENABLE_IT(&huart2, UART_IT_TXE);
         EnableInterrupts();
      }

      ret_val = 0;
   }
   else
      ret_val = HCITR_ERROR_WRITING_TO_PORT;

   return(ret_val);
}

   /* The following function is responsible for suspending the HCI COM  */
   /* transport.  It will block until the transmit buffers are empty and*/
   /* all data has been sent then put the transport in a suspended      */
   /* state.  This function will return a value of 0 if the suspend was */
   /* successful or a negative value if there is an error.              */
   /* * NOTE * An error will occur if the suspend operation was         */
   /*          interrupted by another event, such as data being received*/
   /*          before the transmit buffer was empty.                    */
   /* * NOTE * The calling function must lock the Bluetooth stack with a*/
   /*          call to BSC_LockBluetoothStack() before this function is */
   /*          called.                                                  */
   /* * NOTE * This function should only be called when the baseband    */
   /*          low-power protocol in use has indicated that it is safe  */
   /*          to sleep.  Also, this function must be called            */
   /*          successfully before any clocks necessary for the         */
   /*          transport to operate are disabled.                       */
int BTPSAPI HCITR_COMSuspend(unsigned int HCITransportID)
{
   int ret_val;

   ret_val = HCITR_ERROR_INVALID_PARAMETER;

   return(ret_val);
}

   /* The following function is used to enable or disable debug logging */
   /* within HCITRANS.  The function accepts as its parameter a flag    */
   /* which indicates if debugging should be enabled.  It returns zero  */
   /* if successful or a negative value if there was an error.          */
int BTPSAPI HCITR_EnableDebugLogging(Boolean_t Enable)
{
   int ret_val;

#ifdef HCITR_ENABLE_DEBUG_LOGGING

   UartContext.DebugEnabled = Enable;

   ret_val = 0;

#else

   ret_val = HCITR_ERROR_INVALID_PARAMETER;

#endif

   return(ret_val);
}
