/*****< Main.c >***************************************************************/
/*      Copyright 2016 Texas Instruments Incorporated.                        */
/*      All Rights Reserved.                                                  */
/*                                                                            */
/*  Main.c - Main application implementation for MSP432 transparent mode      */
/*                                                                            */
/*  ________________                __________              __________        */
/* |                |              |          |            |          |       */
/* |      PC        |<-Debug UART->|  MSP432  |<-HCI UART->|  CC256x  |       */
/* |   HCITester    |              |          |            |          |       */
/* |________________|              |__________|            |__________|       */
/*                                                                            */
/*  Author:  Vihang Parmar                                                    */
/*                                                                            */
/*** MODIFICATION HISTORY *****************************************************/
/*                                                                            */
/*   mm/dd/yy  F. Lastname    Description of Modification                     */
/*   --------  -----------    ------------------------------------------------*/
/*   08/31/16  V. Parmar      Initial creation.                               */
/******************************************************************************/

/* DriverLib Includes */
#include "driverlib.h"

#include <msp.h>

/* Hardware definitions Includes */
#include "HRDWCFG.h"

/* Standard Includes */
#include <stdint.h>

#include <stdbool.h>

/* UART Configuration Parameter. These are the configuration parameters to
 * make the eUSCI A UART module to operate with a 115200 baud rate. These
 * values were calculated using the online calculator that TI provides
 * at:
 * http://software-dl.ti.com/msp430/msp430_public_sw/mcu/msp430/MSP430BaudRateConverter/index.html
 */
const eUSCI_UART_Config uartConfig =
{
        EUSCI_A_UART_CLOCKSOURCE_SMCLK,          		// SMCLK Clock Source = 24MHz
        13,                                     		// BRDIV = 13
        0,                                       		// UCxBRF = 0
        37,                                       		// UCxBRS = 37
        EUSCI_A_UART_NO_PARITY,                  		// No Parity
        EUSCI_A_UART_MSB_FIRST,                  		// MSB First
        EUSCI_A_UART_ONE_STOP_BIT,               		// One stop bit
        EUSCI_A_UART_MODE,                       		// UART mode
		EUSCI_A_UART_OVERSAMPLING_BAUDRATE_GENERATION  	// Oversampling
};

int main(void)
{
    /* Halting WDT  */
    WDT_A_holdTimer();

    /* Selecting P1.2 and P1.3 in UART mode */
    GPIO_setAsPeripheralModuleFunctionInputPin(HRDWCFG_DEBUG_UART_TX_PORT_NUM,
    		HRDWCFG_DEBUG_UART_RX_PIN_NUM | HRDWCFG_DEBUG_UART_TX_PIN_NUM, GPIO_PRIMARY_MODULE_FUNCTION);

    /* Selecting P3.2 and P3.3 in UART mode */
    GPIO_setAsPeripheralModuleFunctionInputPin(HRDWCFG_HCI_UART_TX_PORT_NUM,
    		HRDWCFG_HCI_UART_RX_PIN_NUM | HRDWCFG_HCI_UART_TX_PIN_NUM, GPIO_PRIMARY_MODULE_FUNCTION);

    /* Selecting P2.5 as output (CC256x_nSHUTDOWN) */
    GPIO_setAsOutputPin(HRDWCFG_NSHUTD_PORT_NUM, HRDWCFG_NSHUTD_PIN_NUM);
    GPIO_setOutputLowOnPin(HRDWCFG_NSHUTD_PORT_NUM, HRDWCFG_NSHUTD_PIN_NUM);

    /* Selecting BLUE LED as output */
    GPIO_setAsOutputPin(HRDWCFG_LED_BLUE_PORT_NUM, HRDWCFG_LED_BLUE_PIN_NUM);
    GPIO_setOutputLowOnPin(HRDWCFG_LED_BLUE_PORT_NUM, HRDWCFG_LED_BLUE_PIN_NUM);

    /* Setting DCO to 24MHz (upping Vcore) */
    PCM_setCoreVoltageLevel(PCM_VCORE1);
    CS_setDCOCenteredFrequency(CS_DCO_FREQUENCY_24);

    /* Configuring UART A0 Module */
    UART_initModule(HRDWCFG_DEBUG_UART_MODULE, &uartConfig);
    /* Configuring UART A2 Module */
    UART_initModule(HRDWCFG_HCI_UART_MODULE, &uartConfig);

    /* Enable UART A0 module */
    UART_enableModule(HRDWCFG_DEBUG_UART_MODULE);
    /* Enable UART A2 module */
    UART_enableModule(HRDWCFG_HCI_UART_MODULE);

    /* Set the CTS flow control line as input with Pull-down resistor */
    GPIO_setAsInputPinWithPullDownResistor(HRDWCFG_HCI_CTS_PORT_NUM, HRDWCFG_HCI_CTS_PIN_NUM);

    /* Set the RTS flow control line as output. RTS = Low */
    GPIO_setOutputLowOnPin(HRDWCFG_HCI_RTS_PORT_NUM, HRDWCFG_HCI_RTS_PIN_NUM);
    GPIO_setAsOutputPin(HRDWCFG_HCI_RTS_PORT_NUM, HRDWCFG_HCI_RTS_PIN_NUM);

    /* Set P2.5 high (CC256x_nSHUTDOWN) */
    GPIO_setOutputHighOnPin(HRDWCFG_NSHUTD_PORT_NUM, HRDWCFG_NSHUTD_PIN_NUM);
    // Testing
    GPIO_setOutputHighOnPin(HRDWCFG_LED_BLUE_PORT_NUM, HRDWCFG_LED_BLUE_PIN_NUM);

    /* Enabling interrupts */
    UART_enableInterrupt(HRDWCFG_DEBUG_UART_MODULE, EUSCI_A_UART_RECEIVE_INTERRUPT);
    Interrupt_enableInterrupt(HRDWCFG_DEBUG_UART_INT_NUM);
    UART_enableInterrupt(HRDWCFG_HCI_UART_MODULE, EUSCI_A_UART_RECEIVE_INTERRUPT);
    Interrupt_enableInterrupt(HRDWCFG_HCI_UART_INT_NUM);
    Interrupt_enableSleepOnIsrExit();

    while(1)
    {
        PCM_gotoLPM0();
    }
}

/* Debug_UART_eUSCI_A_ISR - Dispatches data to HCI_UART */
void Debug_UART_eUSCI_A_ISR(void)
{
    uint32_t status = UART_getEnabledInterruptStatus(HRDWCFG_DEBUG_UART_MODULE);

    UART_clearInterruptFlag(HRDWCFG_DEBUG_UART_MODULE, status);

    if(status & EUSCI_A_UART_RECEIVE_INTERRUPT)
    {

        GPIO_toggleOutputOnPin(HRDWCFG_LED_BLUE_PORT_NUM, HRDWCFG_LED_BLUE_PIN_NUM);

        UART_transmitData(HRDWCFG_HCI_UART_MODULE, UART_receiveData(HRDWCFG_DEBUG_UART_MODULE));

        Interrupt_disableSleepOnIsrExit();
    }

}

/* HCI_UART_eUSCI_A_ISR - Dispatches data to Application Uart (PC UART) */
void HCI_UART_eUSCI_A_ISR(void)
{
    uint32_t status = UART_getEnabledInterruptStatus(HRDWCFG_HCI_UART_MODULE);

    UART_clearInterruptFlag(HRDWCFG_HCI_UART_MODULE, status);

    if(status & EUSCI_A_UART_RECEIVE_INTERRUPT)
    {

        GPIO_toggleOutputOnPin(HRDWCFG_LED_BLUE_PORT_NUM, HRDWCFG_LED_BLUE_PIN_NUM);

        UART_transmitData(HRDWCFG_DEBUG_UART_MODULE, UART_receiveData(HRDWCFG_HCI_UART_MODULE));

        Interrupt_disableSleepOnIsrExit();
    }

}

/* SysTick interrupt handler. To avoid declaration errors. Not used. */
void SysTick_ISR(void)
{
    ; //Do nothing
}
/* CTS GPIO interrupt handler. To avoid declaration errors. Not used. */
void CTS_Port_ISR(void)
{
    ; //Do nothing
}
