
/*********************************************************************
 * INCLUDES
 */
#include <string.h>
#include <icall.h>
/* This Header file contains all BLE API and icall structure definition */
#include "icall_ble_api.h"

#include <ti/bleapp/profiles/simple_gatt/simple_gatt_profile.h>
#include <ti/bleapp/ble_app_util/inc/bleapputil_api.h>

/*********************************************************************
 * MACROS
 */

/*********************************************************************
 * CONSTANTS
 */

/*********************************************************************
 * TYPEDEFS
 */
void SimpleGattProfile_callback( uint8 paramID  );
void SimpleGattProfile_invokeFromFWContext( char *pData );

/*********************************************************************
 * GLOBAL VARIABLES
 */
// Simple GATT Profile Service UUID: 0xFFF0
GATT_BT_UUID(simpleGattProfile_ServUUID, SIMPLEGATTPROFILE_SERV_UUID);

// Characteristic 1 UUID: 0xFFF1
GATT_BT_UUID(simpleGattProfile_char1UUID, SIMPLEGATTPROFILE_CHAR1_UUID);

// Characteristic 2 UUID: 0xFFF2
GATT_BT_UUID(simpleGattProfile_char2UUID, SIMPLEGATTPROFILE_CHAR2_UUID);

// Characteristic 3 UUID: 0xFFF3
GATT_BT_UUID(simpleGattProfile_char3UUID, SIMPLEGATTPROFILE_CHAR3_UUID);

// Characteristic 4 UUID: 0xFFF4
GATT_BT_UUID(simpleGattProfile_char4UUID, SIMPLEGATTPROFILE_CHAR4_UUID);

// Characteristic 5 UUID: 0xFFF5
GATT_BT_UUID(simpleGattProfile_char5UUID, SIMPLEGATTPROFILE_CHAR5_UUID);

/*********************************************************************
 * EXTERNAL VARIABLES
 */

/*********************************************************************
 * EXTERNAL FUNCTIONS
 */

/*********************************************************************
 * LOCAL VARIABLES
 */

static SimpleGattProfile_CBs_t *simpleGattProfile_appCBs = NULL;

/*********************************************************************
 * Profile Attributes - variables
 */

// Simple GATT Profile Service attribute
static CONST gattAttrType_t simpleGattProfile_Service = { ATT_BT_UUID_SIZE, simpleGattProfile_ServUUID };


// Simple GATT Profile Characteristic 1 Properties
static uint8 simpleGattProfile_Char1Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;

// Characteristic 1 Value
static uint8 simpleGattProfile_Char1 = 0;

// Simple GATT Profile Characteristic 1 User Description
static uint8 simpleGattProfile_Char1UserDesp[17] = "Characteristic 1";


// Simple GATT Profile Characteristic 2 Properties
static uint8 simpleGattProfile_Char2Props = GATT_PROP_READ | GATT_PROP_NOTIFY;

// Characteristic 2 Value
static uint8 simpleGattProfile_Char2 = 0;

// Simple Profile Characteristic 2 User Description
static uint8 simpleGattProfile_Char2UserDesp[17] = "Characteristic 2";


// Simple GATT Profile Characteristic 3 Properties
static uint8 simpleGattProfile_Char3Props = GATT_PROP_WRITE;

// Characteristic 3 Value
static uint8 simpleGattProfile_Char3 = 0;

// Simple GATT Profile Characteristic 3 User Description
static uint8 simpleGattProfile_Char3UserDesp[17] = "Characteristic 3";


// Simple GATT Profile Characteristic 4 Properties
static uint8 simpleGattProfile_Char4Props = GATT_PROP_NOTIFY;

// Characteristic 4 Value
static uint8 simpleGattProfile_Char4 = 0;

// Simple GATT Profile Characteristic 4 Configuration Each client has its own
// instantiation of the Client Characteristic Configuration. Reads of the
// Client Characteristic Configuration only shows the configuration for
// that client and writes only affect the configuration of that client.
static gattCharCfg_t *simpleGattProfile1_Char2Config;
static gattCharCfg_t *simpleGattProfile1_Char1Config;

// Simple GATT Profile Characteristic 4 User Description
static uint8 simpleGattProfile_Char4UserDesp[17] = "Characteristic 4";


// Simple GATT Profile Characteristic 5 Properties
static uint8 simpleGattProfile_Char5Props = GATT_PROP_READ;

// Characteristic 5 Value
static uint8 simpleGattProfile_Char5[SIMPLEGATTPROFILE_CHAR5_LEN] = { 0, 0, 0, 0, 0 };

// Simple GATT Profile Characteristic 5 User Description
static uint8 simpleGattProfile_Char5UserDesp[17] = "Characteristic 5";

/*********************************************************************
 * Profile Attributes - Table
 */

static gattAttribute_t simpleGattProfile_attrTbl[] =
{
 /*------------------type-----------------*/ /*-----------permissions-----------*/ /*-----------------pValue----------------*/
 // Simple Profile Service
 GATT_BT_ATT( primaryServiceUUID,           GATT_PERMIT_READ,                      (uint8 *) &simpleGattProfile_Service ),

 // Characteristic 1 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile_Char1Props ),
 // Characteristic Value 1
 GATT_BT_ATT( simpleGattProfile_char1UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile_Char1 ),
 // Characteristic 1 configuration
  GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile1_Char1Config ),

 // Characteristic 1 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile_Char1UserDesp ),

 // Characteristic 2 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile_Char2Props ),
 // Characteristic Value 2
 GATT_BT_ATT( simpleGattProfile_char2UUID,  GATT_PERMIT_READ,                      &simpleGattProfile_Char2 ),
 // Characteristic 3 configuration
 GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile1_Char2Config ),
 // Characteristic 2 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile_Char2UserDesp ),
};

/******************************************SERVICE 2***********************************************************************/


bStatus_t SimpleGattProfile2_readAttrCB(uint16_t connHandle,
                                       gattAttribute_t *pAttr,
                                       uint8_t *pValue, uint16_t *pLen,
                                       uint16_t offset, uint16_t maxLen,
                                       uint8_t method)
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)pLen;
    (void)offset;
    (void)maxLen;
    (void)method;
    return SUCCESS;
}

bStatus_t SimpleGattProfile2_writeAttrCB( uint16_t connHandle,
                                     gattAttribute_t *pAttr,
                                     uint8_t *pValue, uint16_t len,
                                     uint16_t offset, uint8_t method )
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)offset;
    (void)len;
    (void)method;
    return SUCCESS;
}
CONST gattServiceCBs_t simpleGattProfile2_CBs =
{
  SimpleGattProfile2_readAttrCB,  // Read callback function pointer
  SimpleGattProfile2_writeAttrCB, // Write callback function pointer
  NULL                       // Authorization callback function pointer
};

#define SIMPLEGATTPROFILE2_SERV_UUID               0xABCD
#define SIMPLEGATTPROFILE2_CHAR1_UUID              0xABCE
#define SIMPLEGATTPROFILE2_CHAR2_UUID              0xABCF


GATT_BT_UUID(simpleGattProfile2_ServUUID, SIMPLEGATTPROFILE2_SERV_UUID); // Service
GATT_BT_UUID(simpleGattProfile2_char1UUID, SIMPLEGATTPROFILE2_CHAR1_UUID); // Char1
GATT_BT_UUID(simpleGattProfile2_char2UUID, SIMPLEGATTPROFILE2_CHAR2_UUID); // Char2

static CONST gattAttrType_t simpleGattProfile2_Service = { ATT_BT_UUID_SIZE, simpleGattProfile2_ServUUID };



static uint8 simpleGattProfile2_Char1Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile2_Char1 = 0;
static uint8 simpleGattProfile2_Char1UserDesp[17] = "Svc2Char1";

static uint8 simpleGattProfile2_Char2Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile2_Char2 = 0;
static uint8 simpleGattProfile2_Char2UserDesp[17] = "Svc2Char2";

static gattCharCfg_t *simpleGattProfile2_Char2Config;
static gattCharCfg_t *simpleGattProfile2_Char1Config;

static gattAttribute_t simpleGattProfile2_attrTbl[] =
{
 /*------------------type-----------------*/ /*-----------permissions-----------*/ /*-----------------pValue----------------*/
 // Simple Profile Service
 GATT_BT_ATT( primaryServiceUUID,           GATT_PERMIT_READ,                      (uint8 *) &simpleGattProfile2_Service ),

 // Characteristic 1 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile2_Char1Props ),
 // Characteristic Value 1
 GATT_BT_ATT( simpleGattProfile2_char1UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile2_Char1 ),
 // Characteristic 1 configuration
  GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile2_Char1Config ),
 // Characteristic 1 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile2_Char1UserDesp ),

 // Characteristic 2 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile2_Char2Props ),
 // Characteristic Value 2
 GATT_BT_ATT( simpleGattProfile2_char2UUID,  GATT_PERMIT_READ,                      &simpleGattProfile2_Char2 ),
 // Characteristic 2 configuration
 GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile2_Char2Config ),
 // Characteristic 2 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile2_Char2UserDesp ),
};




/******************************************SERVICE 2 END***********************************************************************/



/******************************************SERVICE 3***********************************************************************/


bStatus_t SimpleGattProfile3_readAttrCB(uint16_t connHandle,
                                       gattAttribute_t *pAttr,
                                       uint8_t *pValue, uint16_t *pLen,
                                       uint16_t offset, uint16_t maxLen,
                                       uint8_t method)
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)pLen;
    (void)offset;
    (void)maxLen;
    (void)method;
    return SUCCESS;
}

bStatus_t SimpleGattProfile3_writeAttrCB( uint16_t connHandle,
                                     gattAttribute_t *pAttr,
                                     uint8_t *pValue, uint16_t len,
                                     uint16_t offset, uint8_t method )
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)offset;
    (void)len;
    (void)method;
    return SUCCESS;
}
CONST gattServiceCBs_t simpleGattProfile3_CBs =
{
  SimpleGattProfile3_readAttrCB,  // Read callback function pointer
  SimpleGattProfile3_writeAttrCB, // Write callback function pointer
  NULL                       // Authorization callback function pointer
};

#define SIMPLEGATTPROFILE3_SERV_UUID               0xD000
#define SIMPLEGATTPROFILE3_CHAR1_UUID              0xD001
#define SIMPLEGATTPROFILE3_CHAR2_UUID              0xD002


GATT_BT_UUID(simpleGattProfile3_ServUUID,  SIMPLEGATTPROFILE3_SERV_UUID); // Service
GATT_BT_UUID(simpleGattProfile3_char1UUID, SIMPLEGATTPROFILE3_CHAR1_UUID); // Char1
GATT_BT_UUID(simpleGattProfile3_char2UUID, SIMPLEGATTPROFILE3_CHAR2_UUID); // Char2

static CONST gattAttrType_t simpleGattProfile3_Service = { ATT_BT_UUID_SIZE, simpleGattProfile3_ServUUID };



static uint8 simpleGattProfile3_Char1Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile3_Char1 = 0;
static uint8 simpleGattProfile3_Char1UserDesp[17] = "Svc3Char1";

static uint8 simpleGattProfile3_Char2Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile3_Char2 = 0;
static uint8 simpleGattProfile3_Char2UserDesp[17] = "Svc3Char2";

static gattCharCfg_t *simpleGattProfile3_Char2Config;
static gattCharCfg_t *simpleGattProfile3_Char1Config;

static gattAttribute_t simpleGattProfile3_attrTbl[] =
{
 /*------------------type-----------------*/ /*-----------permissions-----------*/ /*-----------------pValue----------------*/
 // Simple Profile Service
 GATT_BT_ATT( primaryServiceUUID,           GATT_PERMIT_READ,                      (uint8 *) &simpleGattProfile3_Service ),

 // Characteristic 1 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile3_Char1Props ),
 // Characteristic Value 1
 GATT_BT_ATT( simpleGattProfile3_char1UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile3_Char1 ),
 // Characteristic 1 configuration
  GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile3_Char1Config ),
 // Characteristic 1 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile3_Char1UserDesp ),

 // Characteristic 2 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile3_Char2Props ),
 // Characteristic Value 2
 GATT_BT_ATT( simpleGattProfile3_char2UUID,  GATT_PERMIT_READ,                      &simpleGattProfile3_Char2 ),
 // Characteristic 2 configuration
 GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile3_Char2Config ),
 // Characteristic 2 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile3_Char2UserDesp ),
};


/******************************************Single SERVICE***********************************************************************/
#define SIMPLEGATTPROFILE4_SERV_UUID               0xE000
#define SIMPLEGATTPROFILE4_CHAR1_UUID              0xE001
#define SIMPLEGATTPROFILE4_CHAR2_UUID              0xE002
#define SIMPLEGATTPROFILE4_CHAR3_UUID              0xE003
#define SIMPLEGATTPROFILE4_CHAR4_UUID              0xE004
#define SIMPLEGATTPROFILE4_CHAR5_UUID              0xE005

#define SIMPLEGATTPROFILE4_CHAR6_UUID              0xE006
#define SIMPLEGATTPROFILE4_CHAR7_UUID              0xE007
#define SIMPLEGATTPROFILE4_CHAR8_UUID              0xE008
#define SIMPLEGATTPROFILE4_CHAR9_UUID              0xE009
#define SIMPLEGATTPROFILE4_CHAR10_UUID             0xE00A


GATT_BT_UUID(simpleGattProfile4_ServUUID,  SIMPLEGATTPROFILE4_SERV_UUID); // Service
GATT_BT_UUID(simpleGattProfile4_char1UUID, SIMPLEGATTPROFILE4_CHAR1_UUID); // Char1
GATT_BT_UUID(simpleGattProfile4_char2UUID, SIMPLEGATTPROFILE4_CHAR2_UUID); // Char2
GATT_BT_UUID(simpleGattProfile4_char3UUID, SIMPLEGATTPROFILE4_CHAR3_UUID); // Char3
GATT_BT_UUID(simpleGattProfile4_char4UUID, SIMPLEGATTPROFILE4_CHAR4_UUID); // Char4
GATT_BT_UUID(simpleGattProfile4_char5UUID, SIMPLEGATTPROFILE4_CHAR5_UUID); // Char5
GATT_BT_UUID(simpleGattProfile4_char6UUID, SIMPLEGATTPROFILE4_CHAR6_UUID); // Char6
GATT_BT_UUID(simpleGattProfile4_char7UUID, SIMPLEGATTPROFILE4_CHAR7_UUID); // Char7
GATT_BT_UUID(simpleGattProfile4_char8UUID, SIMPLEGATTPROFILE4_CHAR8_UUID); // Char8
GATT_BT_UUID(simpleGattProfile4_char9UUID, SIMPLEGATTPROFILE4_CHAR9_UUID); // Char9
GATT_BT_UUID(simpleGattProfile4_char10UUID, SIMPLEGATTPROFILE4_CHAR10_UUID); // Char10

static CONST gattAttrType_t simpleGattProfile4_Service = { ATT_BT_UUID_SIZE, simpleGattProfile4_ServUUID };



static uint8 simpleGattProfile4_Char1Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile4_Char1 = 0;
static uint8 simpleGattProfile4_Char1UserDesp[17] = "SvcChar1";

static uint8 simpleGattProfile4_Char2Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char2 = 0;
static uint8 simpleGattProfile4_Char2UserDesp[17] = "SvcChar2";

static uint8 simpleGattProfile4_Char3Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile4_Char3 = 0;
static uint8 simpleGattProfile4_Char3UserDesp[17] = "SvcChar3";

static uint8 simpleGattProfile4_Char4Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char4 = 0;
static uint8 simpleGattProfile4_Char4UserDesp[17] = "SvcChar4";

static uint8 simpleGattProfile4_Char5Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char5 = 0;
static uint8 simpleGattProfile4_Char5UserDesp[17] = "SvcChar5";

static uint8 simpleGattProfile4_Char6Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile4_Char6 = 0;
static uint8 simpleGattProfile4_Char6UserDesp[17] = "SvcChar6";

static uint8 simpleGattProfile4_Char7Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char7 = 0;
static uint8 simpleGattProfile4_Char7UserDesp[17] = "SvcChar7";

static uint8 simpleGattProfile4_Char8Props = GATT_PROP_NOTIFY | GATT_PROP_WRITE;
static uint8 simpleGattProfile4_Char8 = 0;
static uint8 simpleGattProfile4_Char8UserDesp[17] = "SvcChar8";

static uint8 simpleGattProfile4_Char9Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char9 = 0;
static uint8 simpleGattProfile4_Char9UserDesp[17] = "SvcChar9";

static uint8 simpleGattProfile4_Char10Props = GATT_PROP_WRITE| GATT_PROP_NOTIFY;
static uint8 simpleGattProfile4_Char10 = 0;
static uint8 simpleGattProfile4_Char10UserDesp[17] = "SvcChar10";
/********************************************************************************************/

static gattCharCfg_t *simpleGattProfile4_Char1Config;
static gattCharCfg_t *simpleGattProfile4_Char2Config;
static gattCharCfg_t *simpleGattProfile4_Char3Config;
static gattCharCfg_t *simpleGattProfile4_Char4Config;
static gattCharCfg_t *simpleGattProfile4_Char5Config;

static gattCharCfg_t *simpleGattProfile4_Char6Config;
static gattCharCfg_t *simpleGattProfile4_Char7Config;
static gattCharCfg_t *simpleGattProfile4_Char8Config;
static gattCharCfg_t *simpleGattProfile4_Char9Config;
static gattCharCfg_t *simpleGattProfile4_Char10Config;


gattAttribute_t* getHdl(void);

bStatus_t SimpleGattProfile4_readAttrCB(uint16_t connHandle,
                                       gattAttribute_t *pAttr,
                                       uint8_t *pValue, uint16_t *pLen,
                                       uint16_t offset, uint16_t maxLen,
                                       uint8_t method)
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)pLen;
    (void)offset;
    (void)maxLen;
    (void)method;
    return SUCCESS;
}

bStatus_t SimpleGattProfile4_writeAttrCB( uint16_t connHandle,
                                     gattAttribute_t *pAttr,
                                     uint8_t *pValue, uint16_t len,
                                     uint16_t offset, uint8_t method )
{
    (void)connHandle;
    (void)pAttr;
    (void)pValue;
    (void)offset;
    (void)len;
    (void)method;

    volatile gattAttribute_t*attr = NULL;
    bStatus_t status = SUCCESS;
    uint8 notifyApp = 0xFF;
    attHandleValueNoti_t noti = {0};

    if ( pAttr->type.len == ATT_BT_UUID_SIZE )
    {
      // 16-bit UUID
      uint16 uuid = BUILD_UINT16( pAttr->type.uuid[0], pAttr->type.uuid[1]);
      switch ( uuid )
      {
      case SIMPLEGATTPROFILE4_CHAR1_UUID:
      case SIMPLEGATTPROFILE4_CHAR2_UUID:
      {
          if(pAttr->pValue == &simpleGattProfile4_Char2 )
          {
              if(pValue[0] == 1){
                  notifyApp = SIMPLEGATTPROFILE4_CHAR1;
              }
              else if(pValue[0] == 2){
                  notifyApp = SIMPLEGATTPROFILE4_CHAR2;
              }
              else if(pValue[0] == 3){
                  notifyApp = SIMPLEGATTPROFILE4_CHAR3;
              }
              else if(pValue[0] == 4){
                  notifyApp = SIMPLEGATTPROFILE4_CHAR4;
              }
              else if(pValue[0] == 5){
                  notifyApp = SIMPLEGATTPROFILE4_CHAR5;
              }
              else{
                  notifyApp = 0xff;
              }

          }
          else
          {
              notifyApp = 0xff;
          }
      }break;
      case SIMPLEGATTPROFILE4_CHAR3_UUID:
      case SIMPLEGATTPROFILE4_CHAR4_UUID:
      case SIMPLEGATTPROFILE4_CHAR5_UUID:
      {
          //Write the value
          if ( status == SUCCESS )
          {
              uint8 *pCurValue = (uint8 *)pAttr->pValue;
              *pCurValue = pValue[0];

              if( pAttr->pValue == &simpleGattProfile4_Char2 )
              {
                  notifyApp = 0x06;
              }
              else
              {
                  notifyApp = SIMPLEGATTPROFILE_CHAR3;
              }
          }
      }
      break;

      case GATT_CLIENT_CHAR_CFG_UUID:
      {
          // Find the characteristic value attribute
          status = GATTServApp_ProcessCCCWriteReq( connHandle, pAttr, pValue, len,
                                                   offset, GATT_CLIENT_CFG_NOTIFY );
          //notify the App that a change has occurred in Char 4
          notifyApp = SIMPLEGATTPROFILE_CHAR4;
      }break;

      default:
          // Should never get here! (characteristics 2 and 4 do not have write permissions)
          status = ATT_ERR_ATTR_NOT_FOUND;
          break;
      }
    }
    else
    {
      // 128-bit UUID
      status = ATT_ERR_INVALID_HANDLE;
    }

    // If a characteristic value changed then callback function to notify application of change
    if ((notifyApp != 0xFF ) && simpleGattProfile_appCBs && simpleGattProfile_appCBs->pfnSimpleGattProfile_Change)
    {
        SimpleGattProfile_callback( notifyApp );
    }

    // Return status value
    return ( status );

}
CONST gattServiceCBs_t simpleGattProfile4_CBs =
{
  SimpleGattProfile4_readAttrCB,  // Read callback function pointer
  SimpleGattProfile4_writeAttrCB, // Write callback function pointer
  NULL                       // Authorization callback function pointer
};

static gattAttribute_t simpleGattProfile4_attrTbl[] =
{
 /*------------------type-----------------*/ /*-----------permissions-----------*/ /*-----------------pValue----------------*/
 // Simple Profile Service
 GATT_BT_ATT( primaryServiceUUID,           GATT_PERMIT_READ,                      (uint8 *) &simpleGattProfile4_Service ),

 // Characteristic 1 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char1Props ),
 // Characteristic Value 1
 GATT_BT_ATT( simpleGattProfile4_char1UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile4_Char1 ),
 // Characteristic 1 configuration
  GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char1Config ),
 // Characteristic 1 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char1UserDesp ),

 // Characteristic 2 Declaration
 GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char2Props ),
 // Characteristic Value 2
 GATT_BT_ATT( simpleGattProfile4_char2UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE, &simpleGattProfile4_Char2 ),
 // Characteristic 2 configuration
 GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char2Config ),
 // Characteristic 2 User Description
 GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char2UserDesp ),

  // Characteristic 3 Declaration
  GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char3Props ),
  // Characteristic Value 1
  GATT_BT_ATT( simpleGattProfile4_char3UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile4_Char3 ),
  // Characteristic 3 configuration
   GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char3Config ),
  // Characteristic 3 User Description
  GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char3UserDesp ),

  // Characteristic 4 Declaration
  GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char4Props ),
  // Characteristic Value 1
  GATT_BT_ATT( simpleGattProfile4_char4UUID,  GATT_PERMIT_READ| GATT_PERMIT_WRITE,  &simpleGattProfile4_Char4 ),
  // Characteristic 4 configuration
  GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char4Config ),
  // Characteristic 4 User Description
  GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char4UserDesp ),

  // Characteristic 5 Declaration
   GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char5Props ),
   // Characteristic Value 1
   GATT_BT_ATT( simpleGattProfile4_char5UUID,  GATT_PERMIT_READ| GATT_PERMIT_WRITE,  &simpleGattProfile4_Char5 ),
   // Characteristic 5 configuration
   GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char5Config ),
   // Characteristic 5 User Description
   GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char5UserDesp ),


   /****************************************************************************************************************************/
   // Characteristic 6 Declaration
    GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char6Props ),
    // Characteristic Value 6
    GATT_BT_ATT( simpleGattProfile4_char6UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile4_Char6 ),
    // Characteristic 6 configuration
     GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char6Config ),
    // Characteristic 6 User Description
    GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char6UserDesp ),

    // Characteristic 7 Declaration
    GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char7Props ),
    // Characteristic Value 7
    GATT_BT_ATT( simpleGattProfile4_char7UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE, &simpleGattProfile4_Char7 ),
    // Characteristic 7 configuration
    GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char7Config ),
    // Characteristic 7 User Description
    GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char7UserDesp ),

     // Characteristic 8 Declaration
     GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char8Props ),
     // Characteristic Value 1
     GATT_BT_ATT( simpleGattProfile4_char8UUID,  GATT_PERMIT_READ | GATT_PERMIT_WRITE,  &simpleGattProfile4_Char8 ),
     // Characteristic 8 configuration
      GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char8Config ),
     // Characteristic 8 User Description
     GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char8UserDesp ),

     // Characteristic 9 Declaration
     GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char9Props ),
     // Characteristic Value 1
     GATT_BT_ATT( simpleGattProfile4_char9UUID,  GATT_PERMIT_READ| GATT_PERMIT_WRITE,  &simpleGattProfile4_Char9 ),
     // Characteristic 9 configuration
     GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char9Config ),
     // Characteristic 9 User Description
     GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char9UserDesp ),

     // Characteristic 10 Declaration
      GATT_BT_ATT( characterUUID,                GATT_PERMIT_READ,                      &simpleGattProfile4_Char10Props ),
      // Characteristic Value 1
      GATT_BT_ATT( simpleGattProfile4_char10UUID,  GATT_PERMIT_READ| GATT_PERMIT_WRITE,  &simpleGattProfile4_Char10 ),
      // Characteristic 10 configuration
      GATT_BT_ATT( clientCharCfgUUID,            GATT_PERMIT_READ | GATT_PERMIT_WRITE,  (uint8 *) &simpleGattProfile4_Char10Config ),
      // Characteristic 10 User Description
      GATT_BT_ATT( charUserDescUUID,             GATT_PERMIT_READ,                      simpleGattProfile4_Char10UserDesp ),
};








///////////////////////////////////////////////////////////////////////



/****************************************************************************************/
/*********************************************************************
 * LOCAL FUNCTIONS
 */
bStatus_t SimpleGattProfile_readAttrCB( uint16_t connHandle,
                                        gattAttribute_t *pAttr,
                                        uint8_t *pValue, uint16_t *pLen,
                                        uint16_t offset, uint16_t maxLen,
                                        uint8_t method );
bStatus_t SimpleGattProfile_writeAttrCB( uint16_t connHandle,
                                         gattAttribute_t *pAttr,
                                         uint8_t *pValue, uint16_t len,
                                         uint16_t offset, uint8_t method );

/*********************************************************************
 * PROFILE CALLBACKS
 */

// Simple GATT Profile Service Callbacks
// Note: When an operation on a characteristic requires authorization and
// pfnAuthorizeAttrCB is not defined for that characteristic's service, the
// Stack will report a status of ATT_ERR_UNLIKELY to the client.  When an
// operation on a characteristic requires authorization the Stack will call
// pfnAuthorizeAttrCB to check a client's authorization prior to calling
// pfnReadAttrCB or pfnWriteAttrCB, so no checks for authorization need to be
// made within these functions.
CONST gattServiceCBs_t simpleGattProfile_CBs =
{
  SimpleGattProfile_readAttrCB,  // Read callback function pointer
  SimpleGattProfile_writeAttrCB, // Write callback function pointer
  NULL                       // Authorization callback function pointer
};

/*********************************************************************
 * PUBLIC FUNCTIONS
 */

/*********************************************************************
 * @fn      SimpleGattProfile_addService
 *
 * @brief   This function initializes the Simple GATT Server service
 *          by registering GATT attributes with the GATT server.
 *
 * @return  SUCCESS or stack call status
 */
bStatus_t SimpleGattProfile_addService( void )
{
#if 0
  uint8 status = SUCCESS;

  // Allocate Client Characteristic Configuration table
  simpleGattProfile1_Char2Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                 MAX_NUM_BLE_CONNS );
  simpleGattProfile1_Char1Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                  MAX_NUM_BLE_CONNS );

  simpleGattProfile2_Char2Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                  MAX_NUM_BLE_CONNS );

  simpleGattProfile2_Char1Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                  MAX_NUM_BLE_CONNS );

  simpleGattProfile3_Char2Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                  MAX_NUM_BLE_CONNS );
  simpleGattProfile3_Char1Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) *
                                                                    MAX_NUM_BLE_CONNS );

  if ( simpleGattProfile1_Char2Config == NULL  || simpleGattProfile1_Char1Config == NULL
       || simpleGattProfile2_Char1Config == NULL|| simpleGattProfile2_Char2Config == NULL
       || simpleGattProfile3_Char1Config == NULL|| simpleGattProfile3_Char2Config == NULL)
  {
    return ( bleMemAllocError );
  }

  // Initialize Client Characteristic Configuration attributes
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile1_Char2Config );
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile1_Char1Config );
  // Initialize Client Characteristic Configuration attributes
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile2_Char2Config );
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile2_Char1Config );
  // Initialize Client Characteristic Configuration attributes svc3
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile3_Char1Config );
  GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile3_Char2Config );
  // Register GATT attribute list and CBs with GATT Server App
  status = GATTServApp_RegisterService( simpleGattProfile_attrTbl,
                                        GATT_NUM_ATTRS( simpleGattProfile_attrTbl ),
                                        GATT_MAX_ENCRYPT_KEY_SIZE,
                                        &simpleGattProfile_CBs );

  // Register GATT attribute list and CBs with GATT Server App
  status = GATTServApp_RegisterService( simpleGattProfile2_attrTbl,
                                        GATT_NUM_ATTRS( simpleGattProfile2_attrTbl ),
                                        GATT_MAX_ENCRYPT_KEY_SIZE,
                                        &simpleGattProfile2_CBs );

  // Register GATT attribute list and CBs with GATT Server App
  status = GATTServApp_RegisterService( simpleGattProfile3_attrTbl,
                                        GATT_NUM_ATTRS( simpleGattProfile3_attrTbl ),
                                        GATT_MAX_ENCRYPT_KEY_SIZE,
                                        &simpleGattProfile3_CBs );


  // Return status value
  return ( status );

#else
  uint8 status = SUCCESS;
      // MAX_NUM_BLE_CONNS = 1
    // Allocate Client Characteristic Configuration table
    simpleGattProfile4_Char1Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char2Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char3Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char4Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char5Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );

    simpleGattProfile4_Char6Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char7Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char8Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char9Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );
    simpleGattProfile4_Char10Config = (gattCharCfg_t *)ICall_malloc( sizeof( gattCharCfg_t ) * MAX_NUM_BLE_CONNS );

    if (    simpleGattProfile4_Char1Config == NULL  || simpleGattProfile4_Char2Config == NULL
         || simpleGattProfile4_Char3Config == NULL  || simpleGattProfile4_Char4Config == NULL
         || simpleGattProfile4_Char5Config == NULL  || simpleGattProfile4_Char6Config == NULL
         || simpleGattProfile4_Char7Config == NULL  || simpleGattProfile4_Char8Config == NULL
         || simpleGattProfile4_Char9Config == NULL  || simpleGattProfile4_Char10Config == NULL)
    {
      return ( bleMemAllocError );
    }

    // Initialize Client Characteristic Configuration attributes
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char1Config );
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char2Config );
    // Initialize Client Characteristic Configuration attributes
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char3Config );
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char4Config );
    // Initialize Client Characteristic Configuration attributes svc
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char5Config );

    // Initialize Client Characteristic Configuration attributes
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char6Config );
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char7Config );
    // Initialize Client Characteristic Configuration attributes
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char8Config );
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char9Config );
    // Initialize Client Characteristic Configuration attributes svc
    GATTServApp_InitCharCfg( LINKDB_CONNHANDLE_INVALID, simpleGattProfile4_Char10Config );

    // Register GATT attribute list and CBs with GATT Server App
    status = GATTServApp_RegisterService( simpleGattProfile4_attrTbl,
                                          GATT_NUM_ATTRS( simpleGattProfile4_attrTbl ),
                                          GATT_MAX_ENCRYPT_KEY_SIZE,
                                          &simpleGattProfile4_CBs );


    // Register GATT attribute list and CBs with GATT Server App
    status = GATTServApp_RegisterService( simpleGattProfile3_attrTbl,
                                          GATT_NUM_ATTRS( simpleGattProfile3_attrTbl ),
                                          GATT_MAX_ENCRYPT_KEY_SIZE,
                                          &simpleGattProfile3_CBs );

    // Return status value
    return ( status );
#endif
}

/*********************************************************************
 * @fn      SimpleGattProfile_registerAppCBs
 *
 * @brief   Registers the application callback function. Only call
 *          this function once.
 *
 * @param   appCallbacks - pointer to application callback.
 *
 * @return  SUCCESS or INVALIDPARAMETER
 */
bStatus_t SimpleGattProfile_registerAppCBs( SimpleGattProfile_CBs_t *appCallbacks )
{
  if ( appCallbacks )
  {
    simpleGattProfile_appCBs = appCallbacks;

    return ( SUCCESS );
  }
  else
  {
    return ( bleAlreadyInRequestedMode );
  }
}

/*********************************************************************
 * @fn      SimpleGattProfile_setParameter
 *
 * @brief   Set a Simple GATT Profile parameter.
 *
 * @param   param - Profile parameter ID
 * @param   len - length of data to right
 * @param   value - pointer to data to write.  This is dependent on
 *                  the parameter ID and WILL be cast to the appropriate
 *                  data type (example: data type of uint16 will be cast to
 *                  uint16 pointer).
 *
 * @return  SUCCESS or INVALIDPARAMETER
 */
bStatus_t SimpleGattProfile_setParameter( uint8 param, uint8 len, void *value )
{
  bStatus_t status = SUCCESS;

  switch ( param )
  {
    case SIMPLEGATTPROFILE_CHAR1:
      if ( len == sizeof ( uint8 ) )
      {
        simpleGattProfile_Char1 = *((uint8*)value);
      }
      else
      {
        status = bleInvalidRange;
      }
      break;

    case SIMPLEGATTPROFILE_CHAR2:
      if ( len == sizeof ( uint8 ) )
      {
        simpleGattProfile_Char2 = *((uint8*)value);
      }
      else
      {
        status = bleInvalidRange;
      }
      break;

    case SIMPLEGATTPROFILE_CHAR3:
      if ( len == sizeof ( uint8 ) )
      {
        simpleGattProfile_Char3 = *((uint8*)value);
      }
      else
      {
        status = bleInvalidRange;
      }
      break;

    case SIMPLEGATTPROFILE_CHAR4:
      if ( len == sizeof ( uint8 ) )
      {
        simpleGattProfile_Char4 = *((uint8*)value);

        // See if Notification has been enabled
        GATTServApp_ProcessCharCfg( simpleGattProfile1_Char2Config, &simpleGattProfile_Char4, FALSE,
                                    simpleGattProfile_attrTbl, GATT_NUM_ATTRS( simpleGattProfile_attrTbl ),
                                    INVALID_TASK_ID, SimpleGattProfile_readAttrCB );
      }
      else
      {
        status = bleInvalidRange;
      }
      break;

    case SIMPLEGATTPROFILE_CHAR5:
      if ( len == SIMPLEGATTPROFILE_CHAR5_LEN )
      {
        VOID memcpy( simpleGattProfile_Char5, value, SIMPLEGATTPROFILE_CHAR5_LEN );
      }
      else
      {
        status = bleInvalidRange;
      }
      break;

    default:
      status = INVALIDPARAMETER;
      break;
  }

  // Return status value
  return ( status );
}

/*********************************************************************
 * @fn      SimpleGattProfile_getParameter
 *
 * @brief   Get a Simple Profile parameter.
 *
 * @param   param - Profile parameter ID
 * @param   value - pointer to data to put.  This is dependent on
 *          the parameter ID and WILL be cast to the appropriate
 *          data type (example: data type of uint16 will be cast to
 *          uint16 pointer).
 *
 * @return  bStatus_t
 */
bStatus_t SimpleGattProfile_getParameter( uint8 param, void *value )
{
  bStatus_t status = SUCCESS;
  switch ( param )
  {
    case SIMPLEGATTPROFILE_CHAR1:
      *((uint8*)value) = simpleGattProfile_Char1;
      break;

    case SIMPLEGATTPROFILE_CHAR2:
      *((uint8*)value) = simpleGattProfile_Char2;
      break;

    case SIMPLEGATTPROFILE_CHAR3:
      *((uint8*)value) = simpleGattProfile_Char3;
      break;

    case SIMPLEGATTPROFILE_CHAR4:
      *((uint8*)value) = simpleGattProfile_Char4;
      break;

    case SIMPLEGATTPROFILE_CHAR5:
      VOID memcpy( value, simpleGattProfile_Char5, SIMPLEGATTPROFILE_CHAR5_LEN );
      break;

    default:
      status = INVALIDPARAMETER;
      break;
  }

  // Return status value
  return ( status );
}

/*********************************************************************
 * @fn          SimpleGattProfile_readAttrCB
 *
 * @brief       Read an attribute.
 *
 * @param       connHandle - connection message was received on
 * @param       pAttr - pointer to attribute
 * @param       pValue - pointer to data to be read
 * @param       pLen - length of data to be read
 * @param       offset - offset of the first octet to be read
 * @param       maxLen - maximum length of data to be read
 * @param       method - type of read message
 *
 * @return      SUCCESS, blePending or Failure
 */
bStatus_t SimpleGattProfile_readAttrCB(uint16_t connHandle,
                                       gattAttribute_t *pAttr,
                                       uint8_t *pValue, uint16_t *pLen,
                                       uint16_t offset, uint16_t maxLen,
                                       uint8_t method)
{
  bStatus_t status = SUCCESS;

  // Make sure it's not a blob operation (no attributes in the profile are long)
  if ( offset > 0 )
  {
    return ( ATT_ERR_ATTR_NOT_LONG );
  }

  if ( pAttr->type.len == ATT_BT_UUID_SIZE )
  {
    // 16-bit UUID
    uint16 uuid = BUILD_UINT16( pAttr->type.uuid[0], pAttr->type.uuid[1]);
    switch ( uuid )
    {
      // No need for "GATT_SERVICE_UUID" or "GATT_CLIENT_CHAR_CFG_UUID" cases;
      // gattserverapp handles those reads

      // characteristics 1 and 2 have read permissions
      // characteritisc 3 does not have read permissions; therefore it is not
      //   included here
      // characteristic 4 does not have read permissions, but because it
      //   can be sent as a notification, it is included here
      case SIMPLEGATTPROFILE_CHAR1_UUID:
      case SIMPLEGATTPROFILE_CHAR2_UUID:
      case SIMPLEGATTPROFILE_CHAR4_UUID:
        *pLen = 1;
        pValue[0] = *pAttr->pValue;
        break;

      case SIMPLEGATTPROFILE_CHAR5_UUID:
        *pLen = SIMPLEGATTPROFILE_CHAR5_LEN;
        VOID memcpy( pValue, pAttr->pValue, SIMPLEGATTPROFILE_CHAR5_LEN );
        break;

      default:
        // Should never get here! (characteristics 3 and 4 do not have read permissions)
        *pLen = 0;
        status = ATT_ERR_ATTR_NOT_FOUND;
        break;
    }
  }
  else
  {
    // 128-bit UUID
    *pLen = 0;
    status = ATT_ERR_INVALID_HANDLE;
  }

  // Return status value
  return ( status );
}

/*********************************************************************
 * @fn      SimpleGattProfile_writeAttrCB
 *
 * @brief   Validate attribute data prior to a write operation
 *
 * @param   connHandle - connection message was received on
 * @param   pAttr - pointer to attribute
 * @param   pValue - pointer to data to be written
 * @param   len - length of data
 * @param   offset - offset of the first octet to be written
 * @param   method - type of write message
 *
 * @return  SUCCESS, blePending or Failure
 */
bStatus_t SimpleGattProfile_writeAttrCB( uint16_t connHandle,
                                     gattAttribute_t *pAttr,
                                     uint8_t *pValue, uint16_t len,
                                     uint16_t offset, uint8_t method )
{
  bStatus_t status = SUCCESS;
  uint8 notifyApp = 0xFF;

  if ( pAttr->type.len == ATT_BT_UUID_SIZE )
  {
    // 16-bit UUID
    uint16 uuid = BUILD_UINT16( pAttr->type.uuid[0], pAttr->type.uuid[1]);
    switch ( uuid )
    {
      case SIMPLEGATTPROFILE_CHAR1_UUID:
      case SIMPLEGATTPROFILE_CHAR3_UUID:
        {
          //Validate the value
          // Make sure it's not a blob oper
          if ( offset == 0 )
          {
            if ( len != 1 )
            {
              status = ATT_ERR_INVALID_VALUE_SIZE;
            }
          }
          else
          {
            status = ATT_ERR_ATTR_NOT_LONG;
          }

          //Write the value
          if ( status == SUCCESS )
          {
            uint8 *pCurValue = (uint8 *)pAttr->pValue;
            *pCurValue = pValue[0];

            if( pAttr->pValue == &simpleGattProfile_Char1 )
            {
              notifyApp = SIMPLEGATTPROFILE_CHAR1;
            }
            else
            {
              notifyApp = SIMPLEGATTPROFILE_CHAR3;
            }
          }
        }
        break;

      case GATT_CLIENT_CHAR_CFG_UUID:
        status = GATTServApp_ProcessCCCWriteReq( connHandle, pAttr, pValue, len,
                                                 offset, GATT_CLIENT_CFG_NOTIFY );
        //notify the App that a change has occurred in Char 4
        notifyApp = SIMPLEGATTPROFILE_CHAR4;
        break;

      default:
        // Should never get here! (characteristics 2 and 4 do not have write permissions)
        status = ATT_ERR_ATTR_NOT_FOUND;
        break;
    }
  }
  else
  {
    // 128-bit UUID
    status = ATT_ERR_INVALID_HANDLE;
  }

  // If a characteristic value changed then callback function to notify application of change
  if ((notifyApp != 0xFF ) && simpleGattProfile_appCBs && simpleGattProfile_appCBs->pfnSimpleGattProfile_Change)
  {
      SimpleGattProfile_callback( notifyApp );
  }

  // Return status value
  return ( status );
}

/*********************************************************************
 * @fn      SimpleGattProfile_callback
 *
 * @brief   This function will be called from the BLE App Util module
 *          context.
 *          Calling the application callback
 *
 * @param   pData - data
 *
 * @return  None
 */
void SimpleGattProfile_callback( uint8 paramID )
{
  char *pData = ICall_malloc(sizeof(char));

  if(pData == NULL)
  {
    return;
  }

  pData[0] = paramID;

  BLEAppUtil_invokeFunction(SimpleGattProfile_invokeFromFWContext, pData);
}

/*********************************************************************
 * @fn      SimpleGattProfile_invokeFromFWContext
 *
 * @brief   This function will be called from the BLE App Util module
 *          context.
 *          Calling the application callback
 *
 * @param   pData - data
 *
 * @return  None
 */
void SimpleGattProfile_invokeFromFWContext( char *pData )
{
  simpleGattProfile_appCBs->pfnSimpleGattProfile_Change(pData[0]);
}

/*********************************************************************
*********************************************************************/


gattAttribute_t* getHdl(void)
{
    return GATTServApp_FindAttr(simpleGattProfile4_attrTbl, GATT_NUM_ATTRS( simpleGattProfile4_attrTbl ), &simpleGattProfile4_Char2);
}


bStatus_t sendNotification(uint8 *pValue, uint16 len, uint8_t dest)
{
  bStatus_t status = SUCCESS;
  gattCharCfg_t *pItem = NULL;
  gattAttribute_t *pAttr = NULL;
  attHandleValueNoti_t noti = {0};
  linkDBInfo_t connInfo = {0};
  uint16 offset = 0;
  uint8 i = 0;
  uint8_t *search = NULL;

  // Verify input parameters
  if ( pValue == NULL )
  {
    return ( INVALIDPARAMETER );
  }

  switch(dest)
  {
  case SIMPLEGATTPROFILE4_CHAR1:
      pItem = &( simpleGattProfile4_Char1Config[0] );
      search = &simpleGattProfile4_Char1;
      break;
  case SIMPLEGATTPROFILE4_CHAR2:
      pItem = &( simpleGattProfile4_Char2Config[0] );
      search = &simpleGattProfile4_Char2;
      break;
  case SIMPLEGATTPROFILE4_CHAR3:
      pItem = &( simpleGattProfile4_Char3Config[0] );
      search = &simpleGattProfile4_Char3;
      break;
  case SIMPLEGATTPROFILE4_CHAR4:
      pItem = &( simpleGattProfile4_Char4Config[0] );
      search = &simpleGattProfile4_Char4;
      break;
  case SIMPLEGATTPROFILE4_CHAR5:
      pItem = &( simpleGattProfile4_Char5Config[0] );
      search = &simpleGattProfile4_Char5;
      break;
  default:
      return 0;
  }

  // Find the characteristic value attribute
  pAttr = GATTServApp_FindAttr(simpleGattProfile4_attrTbl, GATT_NUM_ATTRS(simpleGattProfile4_attrTbl), search);
  if ( pAttr != NULL )
  {

      // If the connection has register for notifications
      if ( ( pItem->connHandle != LINKDB_CONNHANDLE_INVALID ) &&
           ( pItem->value == GATT_CLIENT_CFG_NOTIFY) )
      {
        // Find out what the maximum MTU size is for each connection
        status = linkDB_GetInfo(pItem->connHandle, &connInfo);
        offset = 0;

        while ( status != bleTimeout && status != bleNotConnected &&  len > offset )
        {
          // Determine allocation size
          uint16_t allocLen = (len - offset);
          if ( allocLen > ( connInfo.MTU - ATT_OPCODE_SIZE + 2 ) )
          {
            // If len > MTU split data to chunks of MTU size
            allocLen = connInfo.MTU - ATT_OPCODE_SIZE + 2;
          }

          noti.len = allocLen;
          noti.pValue = (uint8 *)GATT_bm_alloc( pItem->connHandle, ATT_HANDLE_VALUE_NOTI, allocLen, 0 );
          if ( noti.pValue != NULL )
          {
            // If allocation was successful, copy out data and send it
            memcpy(noti.pValue, pValue + offset, noti.len);
            noti.handle = pAttr->handle;

            // Send the data over BLE notifications
            status = GATT_Notification( pItem->connHandle, &noti, FALSE );

            // If unable to send the data, free allocated buffers and return
            if ( status != SUCCESS )
            {
              GATT_bm_free( (gattMsg_t *)&noti, ATT_HANDLE_VALUE_NOTI );

            }
            else
            {
              // Increment data offset
              offset += allocLen;
            }
          }
          else
          {
            status = bleNoResources;
          }
        } // End of while
      }
  } // End of if

  // Return status value
  return ( status );
}
