/*
 * Copyright (c) 2015-2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== pwmled.c ========
 */

#include <stddef.h>
#include <ctype.h>
#include <strings.h>
#include <stdio.h>

/* Driver Header files */
#include <ti/drivers/PWM.h>

/* Example/Board Header files */
#include "Board.h"
#include "Raster.h"
#include "SRI_Global.h"
#include "Parse.h"
#include "EEPROM.h"
#include "sdi_task.h"

PWM_Handle Raster_pwm = NULL;

int32_t Raster_Init(void)
{
    PWM_Params params0;

    PWM_Params_init(&params0);
    params0.dutyUnits = PWM_DUTY_COUNTS;
    params0.dutyValue = 0;
    params0.periodUnits = PWM_PERIOD_COUNTS;
    params0.periodValue = RASTER_PWM_PERIOD;  // 20 KHz with main clock at 48 MHz
    params0.idleLevel = PWM_IDLE_LOW;

    Raster_pwm = PWM_open(Board_PWM0, &params0);
    if (Raster_pwm == NULL) {
        /* Raster_pwm did not open */
        return (-1);
    }
    return (0);
}


int32_t Raster_Control(uint16_t drive_level)
{
    int32_t return_val;
    uint32_t duty;

    if (drive_level > 100) return (-1);
    else if (drive_level == 0)
    {
        PWM_stop(Raster_pwm);
        return_val = 0;
    }
    else
    {
        PWM_start(Raster_pwm);
        duty = drive_level * (RASTER_PWM_PERIOD/100);
        return_val = PWM_setDuty(Raster_pwm, duty);
    }
    return (return_val);
}


void Raster_cmd(void)
{
  int raster_pwr_lvl;

  switch (parsed[parsed_tail].token)
  {
  case TOK_ON:
    Raster_Control(eeprom_base.eeprom.RasterDrive);
    break;

  case TOK_OFF:
    Raster_Control(0);
    break;

  case TEST:
    /* Test mode off means the monitor is active. */
    if (strcasecmp(parsed[parsed_tail].string[1], "ON") == 0)
    {
        eeprom_base.eeprom.RasterMonitor = ON;
    }
    else
    {
        eeprom_base.eeprom.RasterMonitor = OFF;
    }
    status.rasterStall = FALSE;
    //EEPROM_UpdateFCChecksum();
    break;

  case SET:
    sscanf(parsed[parsed_tail].string[1], "%d", &raster_pwr_lvl);
    if (raster_pwr_lvl > 100)
    {
      SDITask_Printf("\nRASTER: ERROR, NEW VALUE OUT OF RANGE (%d > 100%)", raster_pwr_lvl);
      return;
    }
    else
    {
      Raster_Control(raster_pwr_lvl);
      eeprom_base.eeprom.RasterDrive = raster_pwr_lvl;
     // EEPROM_UpdateFCChecksum();
    }
    break;

  case QUERY:
    if (status.rasterSet == ON)
    {
      SDITask_Printf("\nRASTER: ON   ");
    }
    else SDITask_Printf("\nRASTER: OFF   ");
    SDITask_Printf("Pwr: %d", eeprom_base.eeprom.RasterDrive);
    SDITask_Printf("   RPM: %5.0f",   status.rasterRPM);
    SDITask_Printf("\nSTALL MONITOR: ");
    if (eeprom_base.eeprom.RasterMonitor == OFF) SDITask_Printf("ON");
    else SDITask_Printf("OFF");
    SDITask_Printf("\nSTALL DETECTED: ");
    if (status.rasterStall == TRUE) SDITask_Printf("TRUE");
    else SDITask_Printf("FALSE");
    break;

  default:
    SDITask_Printf(" Raster_cmd Invalid Subcommand");
    break;
  }  /* end Switch on token */
}   /* end Raster_cmd() */


/************************************ Done ***********************************************/
