/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ====================== CC2640R2_LAUNCHXL.c =================================
 *  This board file is made for the 7x7 mm QFN package, to convert this board
 *  file to use for other smaller device packages please refer to the table
 *  below which lists the max IOID values supported by each package. All other
 *  unused pins should be set to IOID_UNUSED.
 *
 *  Furthermore the board file is also used
 *  to define a symbol that configures the RF front end and bias.
 *  See the comments below for more information.
 *  For an in depth tutorial on how to create a custom board file, please refer
 *  to the section "Running the SDK on Custom Boards" with in the Software
 *  Developer's Guide.
 *
 *  Refer to the datasheet for all the package options and IO descriptions:
 *  http://www.ti.com/lit/ds/symlink/cc2640r2f.pdf
 *
 *  +-----------------------+------------------+-----------------------+
 *  |     Package Option    |  Total GPIO Pins |   MAX IOID            |
 *  +=======================+==================+=======================+
 *  |     7x7 mm QFN        |     31           |   IOID_30             |
 *  +-----------------------+------------------+-----------------------+
 *  |     5x5 mm QFN        |     15           |   IOID_14             |
 *  +-----------------------+------------------+-----------------------+
 *  |     4x4 mm QFN        |     10           |   IOID_9              |
 *  +-----------------------+------------------+-----------------------+
 *  |     2.7 x 2.7 mm WCSP |     14           |   IOID_13             |
 *  +-----------------------+------------------+-----------------------+
 *  ============================================================================
 */
#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <ti/sysbios/family/arm/m3/Hwi.h>

#include <ti/devices/cc26x0r2/driverlib/ioc.h>
#include <ti/devices/cc26x0r2/driverlib/udma.h>
#include <ti/devices/cc26x0r2/inc/hw_ints.h>
#include <ti/devices/cc26x0r2/inc/hw_memmap.h>

#include "CC2640R2_LAUNCHXL.h"
#include "Interlocks.h"
/*
 *  =============================== ADC ===============================
 */
#include <ti/drivers/ADC.h>
#include <ti/drivers/adc/ADCCC26XX.h>

ADCCC26XX_Object adcCC26xxObjects[CC2640R2_LAUNCHXL_ADCCOUNT];

const ADCCC26XX_HWAttrs adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADCCOUNT] =
{
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO23_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO7,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO24_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO6,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO25_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO5,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO26_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO4,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO27_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO3,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO28_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO2,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = MINIRAMAN_LASER_THERM,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO1,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_10P9_MS,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = CC2640R2_LAUNCHXL_DIO30_ANALOG,
    .adcCompBInput       = ADC_COMPB_IN_AUXIO0,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = PIN_UNASSIGNED,
    .adcCompBInput       = ADC_COMPB_IN_DCOUPL,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_2P7_US,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = PIN_UNASSIGNED,
    .adcCompBInput       = ADC_COMPB_IN_VSS,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_10P9_MS,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
},
{
    .adcDIO              = PIN_UNASSIGNED,
    .adcCompBInput       = ADC_COMPB_IN_VDDS,
    .refSource           = ADCCC26XX_FIXED_REFERENCE,
    .samplingDuration    = ADCCC26XX_SAMPLING_DURATION_10P9_MS,
    .inputScalingEnabled = true,
    .triggerSource       = ADCCC26XX_TRIGGER_MANUAL,
    .returnAdjustedVal   = false
}
};

const ADC_Config ADC_config[CC2640R2_LAUNCHXL_ADCCOUNT] =
{
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC0], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC0]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC1], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC1]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC2], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC2]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC3], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC3]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC4], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC4]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC5], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC5]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC6], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC6]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADC7], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADC7]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADCDCOUPL], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADCDCOUPL]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADCVSS], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADCVSS]},
 {&ADCCC26XX_fxnTable, &adcCC26xxObjects[CC2640R2_LAUNCHXL_ADCVDDS], &adcCC26xxHWAttrs[CC2640R2_LAUNCHXL_ADCVDDS]},
};

const uint_least8_t ADC_count = CC2640R2_LAUNCHXL_ADCCOUNT;

/*
 *  =============================== Crypto ===============================
 */
#include <ti/drivers/crypto/CryptoCC26XX.h>

CryptoCC26XX_Object cryptoCC26XXObjects[CC2640R2_LAUNCHXL_CRYPTOCOUNT];

const CryptoCC26XX_HWAttrs cryptoCC26XXHWAttrs[CC2640R2_LAUNCHXL_CRYPTOCOUNT] = {
    {
        .baseAddr       = CRYPTO_BASE,
        .powerMngrId    = PowerCC26XX_PERIPH_CRYPTO,
        .intNum         = INT_CRYPTO_RESULT_AVAIL_IRQ,
        .intPriority    = ~0,
    }
};

const CryptoCC26XX_Config CryptoCC26XX_config[CC2640R2_LAUNCHXL_CRYPTOCOUNT] = {
    {
         .object  = &cryptoCC26XXObjects[CC2640R2_LAUNCHXL_CRYPTO0],
         .hwAttrs = &cryptoCC26XXHWAttrs[CC2640R2_LAUNCHXL_CRYPTO0]
    },
};

/*
 *  =============================== GPIO ===============================
 */
#include <ti/drivers/GPIO.h>
#include <ti/drivers/gpio/GPIOCC26XX.h>

/*
#define MINIRAMAN_ACQUIRE_BUTTON      (IOID_7)
#define MINIRAMAN_POWER_BUTTON        (IOID_21)
#define MINIRAMAN_INTERLOCK_2         (IOID_27)
#define MINIRAMAN_INTERLOCK_1         (IOID_28)
#define MINIRAMAN_INTERLOCK_3         (IOID_30)
*/
/*
 * Array of Pin configurations
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in CC2640R2_LAUNCHXL.h
 * NOTE: Pins not used for interrupts should be placed at the end of the
 *       array. Callback entries can be omitted from callbacks array to
 *       reduce memory usage.
 */
GPIO_PinConfig gpioPinConfigs[] =
{
    /* Input pins */
 MINIRAMAN_ACQUIRE_BUTTON | GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_BOTH_EDGES,
 MINIRAMAN_POWER_BUTTON  | GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_BOTH_EDGES,
 MINIRAMAN_INTERLOCK_2  | GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_BOTH_EDGES,
 MINIRAMAN_INTERLOCK_1  | GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_BOTH_EDGES,
 MINIRAMAN_INTERLOCK_3  | GPIO_CFG_IN_PU | GPIO_CFG_IN_INT_BOTH_EDGES,
};

/*
 * Array of callback function pointers
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in CC2640R2_LAUNCH.h
 * NOTE: Pins not used for interrupts can be omitted from callbacks array to
 *       reduce memory usage (if placed at end of gpioPinConfigs array).
 */
GPIO_CallbackFxn gpioCallbackFunctions[] = {
    NULL,  /* ACQUIRE_BUTTON */
    NULL,  /* POWER_BUTTON */
    NULL,  /* INTERLOCK_2 */
    NULL,  /* INTERLOCK_1 */
    NULL,  /* INTERLOCK_3 */
};

const GPIOCC26XX_Config GPIOCC26XX_config = {
    .pinConfigs         = (GPIO_PinConfig *)gpioPinConfigs,
    .callbacks          = (GPIO_CallbackFxn *)gpioCallbackFunctions,
    .numberOfPinConfigs = CC2640R2_LAUNCHXL_GPIOCOUNT,
    .numberOfCallbacks  = sizeof(gpioCallbackFunctions)/sizeof(GPIO_CallbackFxn),
    .intPriority        = (~0)
};

/*
 *  =============================== GPTimer ===============================
 *  Remove unused entries to reduce flash usage both in Board.c and Board.h
 */
#include <ti/drivers/timer/GPTimerCC26XX.h>

GPTimerCC26XX_Object gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMERCOUNT];

const GPTimerCC26XX_HWAttrs gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMERPARTSCOUNT] = {
    { .baseAddr = GPT0_BASE, .intNum = INT_GPT0A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT0, .pinMux = GPT_PIN_0A, },
    { .baseAddr = GPT0_BASE, .intNum = INT_GPT0B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT0, .pinMux = GPT_PIN_0B, },
    { .baseAddr = GPT1_BASE, .intNum = INT_GPT1A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT1, .pinMux = GPT_PIN_1A, },
    { .baseAddr = GPT1_BASE, .intNum = INT_GPT1B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT1, .pinMux = GPT_PIN_1B, },
    { .baseAddr = GPT2_BASE, .intNum = INT_GPT2A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT2, .pinMux = GPT_PIN_2A, },
    { .baseAddr = GPT2_BASE, .intNum = INT_GPT2B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT2, .pinMux = GPT_PIN_2B, },
    { .baseAddr = GPT3_BASE, .intNum = INT_GPT3A, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT3, .pinMux = GPT_PIN_3A, },
    { .baseAddr = GPT3_BASE, .intNum = INT_GPT3B, .intPriority = (~0), .powerMngrId = PowerCC26XX_PERIPH_GPT3, .pinMux = GPT_PIN_3B, },
};

const GPTimerCC26XX_Config GPTimerCC26XX_config[CC2640R2_LAUNCHXL_GPTIMERPARTSCOUNT] = {
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER0], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER0A], GPT_A },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER0], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER0B], GPT_B },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER1], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER1A], GPT_A },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER1], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER1B], GPT_B },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER2], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER2A], GPT_A },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER2], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER2B], GPT_B },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER3], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER3A], GPT_A },
    { &gptimerCC26XXObjects[CC2640R2_LAUNCHXL_GPTIMER3], &gptimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_GPTIMER3B], GPT_B },
};

/*
 *  =============================== I2C ===============================
*/
#include <ti/drivers/I2C.h>
#include <ti/drivers/i2c/I2CCC26XX.h>

I2CCC26XX_Object i2cCC26xxObjects[CC2640R2_LAUNCHXL_I2CCOUNT];

const I2CCC26XX_HWAttrsV1 i2cCC26xxHWAttrs[CC2640R2_LAUNCHXL_I2CCOUNT] = {
    {
        .baseAddr    = I2C0_BASE,
        .powerMngrId = PowerCC26XX_PERIPH_I2C0,
        .intNum      = INT_I2C_IRQ,
        .intPriority = ~0,
        .swiPriority = 0,
        .sdaPin      = CC2640R2_LAUNCHXL_I2C0_SDA0,
        .sclPin      = CC2640R2_LAUNCHXL_I2C0_SCL0,
    }
};

const I2C_Config I2C_config[CC2640R2_LAUNCHXL_I2CCOUNT] = {
    {
        .fxnTablePtr = &I2CCC26XX_fxnTable,
        .object      = &i2cCC26xxObjects[CC2640R2_LAUNCHXL_I2C0],
        .hwAttrs     = &i2cCC26xxHWAttrs[CC2640R2_LAUNCHXL_I2C0]
    },
};

const uint_least8_t I2C_count = CC2640R2_LAUNCHXL_I2CCOUNT;

/*
 *  =============================== NVS ===============================
 */
#include <ti/drivers/NVS.h>
#include <ti/drivers/nvs/NVSSPI25X.h>
#include <ti/drivers/nvs/NVSCC26XX.h>

#define SECTORSIZE 0x1000
#define NVS_REGIONS_BASE 0x1B000

static uint8_t verifyBuf[64];

/*
 * Reserve flash sectors for NVS driver use by placing an uninitialized byte
 * array at the desired flash address.
 */
#if defined(__TI_COMPILER_VERSION__)

/*
 * Place uninitialized array at NVS_REGIONS_BASE
 */
#pragma LOCATION(flashBuf, NVS_REGIONS_BASE);
#pragma NOINIT(flashBuf);
static char flashBuf[SECTORSIZE * 4];

#elif defined(__IAR_SYSTEMS_ICC__)

/*
 * Place uninitialized array at NVS_REGIONS_BASE
 */
__no_init static char flashBuf[SECTORSIZE * 4] @ NVS_REGIONS_BASE;

#elif defined(__GNUC__)

/*
 * Place the reserved flash buffers in the .nvs section.
 * The nvs section will be placed at address NVS_REGIONS_BASE by
 * the gcc linker cmd file.
 */
__attribute__ ((section (".nvs")))
static char flashBuf[SECTORSIZE * 4];

#endif

/* Allocate objects for NVS and NVS SPI */
NVSCC26XX_Object nvsCC26xxObjects[1];
NVSSPI25X_Object nvsSPI25XObjects[1];

/* Hardware attributes for NVS */
const NVSCC26XX_HWAttrs nvsCC26xxHWAttrs[1] = {
    {
        .regionBase = (void *)flashBuf,
        .regionSize = SECTORSIZE * 4,
    },
};

/* Hardware attributes for NVS SPI */
const NVSSPI25X_HWAttrs nvsSPI25XHWAttrs[1] = {
    {
        .regionBaseOffset = 0,
        .regionSize = SECTORSIZE * 4,
        .sectorSize = SECTORSIZE,
        .verifyBuf = verifyBuf,
        .verifyBufSize = 64,
        .spiHandle = NULL,
        .spiIndex = 0,
        .spiBitRate = 4000000,
        .spiCsnGpioIndex = CC2640R2_LAUNCHXL_GPIO_SPI_FLASH_CS,
    },
};

/* NVS Region index 0 and 1 refer to NVS and NVS SPI respectively */
const NVS_Config NVS_config[CC2640R2_LAUNCHXL_NVSCOUNT] = {
    {
        .fxnTablePtr = &NVSCC26XX_fxnTable,
        .object = &nvsCC26xxObjects[0],
        .hwAttrs = &nvsCC26xxHWAttrs[0],
    },
    {
        .fxnTablePtr = &NVSSPI25X_fxnTable,
        .object = &nvsSPI25XObjects[0],
        .hwAttrs = &nvsSPI25XHWAttrs[0],
    },
};

const uint_least8_t NVS_count = CC2640R2_LAUNCHXL_NVSCOUNT;

/*
 *  =============================== PIN ===============================
 */
#include <ti/drivers/PIN.h>
#include <ti/drivers/pin/PINCC26XX.h>

 const PIN_Config BoardGpioInitTable[] =
 {
     MINIRAMAN_RASTER_PWM | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MAX,
     MINIRAMAN_RASTER_FG | PIN_INPUT_EN | PIN_PULLUP | PIN_HYSTERESIS,
     MINIRAMAN_UART_TXD | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL,
     MINIRAMAN_UART_RXD | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_SCL | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_OPENDRAIN | PIN_DRVSTR_MAX,
     MINIRAMAN_SDA | PIN_INPUT_EN | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_OPENDRAIN | PIN_DRVSTR_MAX,
     MINIRAMAN_VBUS_DETECT | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_NEGEDGE | PIN_HYSTERESIS,
     MINIRAMAN_ACQUIRE_BUTTON | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_NEGEDGE | PIN_HYSTERESIS,
     MINIRAMAN_FLASH_MISO | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_FLASH_MOSI | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_FLASH_SCK | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_FPGA_MISO | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_FPGA_SCK | PIN_INPUT_EN | PIN_PULLDOWN,
     MINIRAMAN_nFPGA_CS | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_nFPGA_RST | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_FPGA_DONE | PIN_INPUT_EN | PIN_PULLDOWN | PIN_IRQ_POSEDGE | PIN_HYSTERESIS,
     MINIRAMAN_IMAGER_XCLR | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_IMAGER_XTRIG | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_UART_RTS | PIN_INPUT_EN | PIN_PULLUP | PIN_HYSTERESIS,
     MINIRAMAN_UART_CTS | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_nFLASH_CS | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_POWER_BUTTON | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_NEGEDGE | PIN_HYSTERESIS,
     MINIRAMAN_nLASER_PWM | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MAX,
     MINIRAMAN_EN_3V3 | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_LED_EN | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MIN,
     MINIRAMAN_GAS_GAUGE_ALERT | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_NEGEDGE | PIN_HYSTERESIS,
     MINIRAMAN_TE_PWR_EN | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_DRVSTR_MED,
     MINIRAMAN_INTERLOCK_2 | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_BOTHEDGES | PIN_HYSTERESIS,
     MINIRAMAN_INTERLOCK_1 | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_BOTHEDGES | PIN_HYSTERESIS,
     MINIRAMAN_LASER_THERM | PIN_INPUT_DIS,
     MINIRAMAN_INTERLOCK_3 | PIN_INPUT_EN | PIN_PULLUP | PIN_IRQ_BOTHEDGES | PIN_HYSTERESIS,
     PIN_TERMINATE
};


const PINCC26XX_HWAttrs PINCC26XX_hwAttrs = {
    .intPriority = ~0,
    .swiPriority = 0
};

/*
 *  =============================== Power ===============================
 */
#include <ti/drivers/Power.h>
#include <ti/drivers/power/PowerCC26XX.h>

const PowerCC26XX_Config PowerCC26XX_config = {
    .policyInitFxn      = NULL,
    .policyFxn          = &PowerCC26XX_standbyPolicy,
    .calibrateFxn       = &PowerCC26XX_calibrate,
    .enablePolicy       = TRUE,
#ifdef USE_RCOSC
    .calibrateRCOSC_LF  = TRUE,
#else
    .calibrateRCOSC_LF  = FALSE,
#endif
    .calibrateRCOSC_HF  = TRUE,
};

/*
 *  =============================== PWM ===============================
 *  Remove unused entries to reduce flash usage both in Board.c and Board.h
 */
#include <ti/drivers/PWM.h>
#include <ti/drivers/pwm/PWMTimerCC26XX.h>

PWMTimerCC26XX_Object pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWMCOUNT];

const PWMTimerCC26XX_HwAttrs pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWMCOUNT] = {
    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN0, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER0A },
    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN1, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER0B },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN2, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER1A },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN3, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER1B },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN4, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER2A },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN5, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER2B },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN6, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER3A },
//    { .pwmPin = CC2640R2_LAUNCHXL_PWMPIN7, .gpTimerUnit = CC2640R2_LAUNCHXL_GPTIMER3B },
};

const PWM_Config PWM_config[CC2640R2_LAUNCHXL_PWMCOUNT] = {
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM0], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM0] },
    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM1], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM1] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM2], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM2] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM3], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM3] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM4], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM4] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM5], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM5] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM6], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM6] },
//    { &PWMTimerCC26XX_fxnTable, &pwmtimerCC26xxObjects[CC2640R2_LAUNCHXL_PWM7], &pwmtimerCC26xxHWAttrs[CC2640R2_LAUNCHXL_PWM7] },
};

const uint_least8_t PWM_count = CC2640R2_LAUNCHXL_PWMCOUNT;

/*
 *  =============================== RF Driver ===============================
 *  Note: The BLE-Stack requires RF SWI priorities to be set to 5.
 */
#include <ti/drivers/rf/RF.h>

const RFCC26XX_HWAttrs RFCC26XX_hwAttrs = {
    .hwiCpe0Priority = ~0,
    .hwiHwPriority   = ~0,
    .swiCpe0Priority =  5,
    .swiHwPriority   =  5,
};

/*
 *  =============================== SPI DMA ===============================
 */
#include <ti/drivers/SPI.h>
#include <ti/drivers/spi/SPICC26XXDMA.h>

SPICC26XXDMA_Object spiCC26XXDMAObjects[CC2640R2_LAUNCHXL_SPICOUNT];

const SPICC26XXDMA_HWAttrsV1 spiCC26XXDMAHWAttrs[CC2640R2_LAUNCHXL_SPICOUNT] = {
    {
        .baseAddr           = SSI0_BASE,
        .intNum             = INT_SSI0_COMB,
        .intPriority        = ~0,
        .swiPriority        = 0,
        .powerMngrId        = PowerCC26XX_PERIPH_SSI0,
        .defaultTxBufValue  = 0,
        .rxChannelBitMask   = 1<<UDMA_CHAN_SSI0_RX,
        .txChannelBitMask   = 1<<UDMA_CHAN_SSI0_TX,
        .mosiPin            = CC2640R2_LAUNCHXL_SPI0_MOSI,
        .misoPin            = CC2640R2_LAUNCHXL_SPI0_MISO,
        .clkPin             = CC2640R2_LAUNCHXL_SPI0_CLK,
        .csnPin             = CC2640R2_LAUNCHXL_SPI0_CSN
    },
    {
        .baseAddr           = SSI1_BASE,
        .intNum             = INT_SSI1_COMB,
        .intPriority        = ~0,
        .swiPriority        = 0,
        .powerMngrId        = PowerCC26XX_PERIPH_SSI1,
        .defaultTxBufValue  = 0,
        .rxChannelBitMask   = 1<<UDMA_CHAN_SSI1_RX,
        .txChannelBitMask   = 1<<UDMA_CHAN_SSI1_TX,
        .mosiPin            = CC2640R2_LAUNCHXL_SPI1_MOSI,
        .misoPin            = CC2640R2_LAUNCHXL_SPI1_MISO,
        .clkPin             = CC2640R2_LAUNCHXL_SPI1_CLK,
        .csnPin             = CC2640R2_LAUNCHXL_SPI1_CSN
    }
};

const SPI_Config SPI_config[CC2640R2_LAUNCHXL_SPICOUNT] = {
    {
         .fxnTablePtr = &SPICC26XXDMA_fxnTable,
         .object      = &spiCC26XXDMAObjects[CC2640R2_LAUNCHXL_SPI0],
         .hwAttrs     = &spiCC26XXDMAHWAttrs[CC2640R2_LAUNCHXL_SPI0]
    },
    {
         .fxnTablePtr = &SPICC26XXDMA_fxnTable,
         .object      = &spiCC26XXDMAObjects[CC2640R2_LAUNCHXL_SPI1],
         .hwAttrs     = &spiCC26XXDMAHWAttrs[CC2640R2_LAUNCHXL_SPI1]
    },
};

const uint_least8_t SPI_count = CC2640R2_LAUNCHXL_SPICOUNT;

/*
 *  =============================== UART ===============================
 */
#include <ti/drivers/UART.h>
#include <ti/drivers/uart/UARTCC26XX.h>

UARTCC26XX_Object uartCC26XXObjects[CC2640R2_LAUNCHXL_UARTCOUNT];

const UARTCC26XX_HWAttrsV2 uartCC26XXHWAttrs[CC2640R2_LAUNCHXL_UARTCOUNT] = {
    {
        .baseAddr       = UART0_BASE,
        .powerMngrId    = PowerCC26XX_PERIPH_UART0,
        .intNum         = INT_UART0_COMB,
        .intPriority    = ~0,
        .swiPriority    = 0,
        .txPin          = CC2640R2_LAUNCHXL_UART_TX,
        .rxPin          = CC2640R2_LAUNCHXL_UART_RX,
        .ctsPin         = PIN_UNASSIGNED,
        .rtsPin         = PIN_UNASSIGNED
    }
};

const UART_Config UART_config[CC2640R2_LAUNCHXL_UARTCOUNT] = {
    {
        .fxnTablePtr = &UARTCC26XX_fxnTable,
        .object      = &uartCC26XXObjects[CC2640R2_LAUNCHXL_UART0],
        .hwAttrs     = &uartCC26XXHWAttrs[CC2640R2_LAUNCHXL_UART0]
    },
};

const uint_least8_t UART_count = CC2640R2_LAUNCHXL_UARTCOUNT;

/*
 *  =============================== UDMA ===============================
 */
#include <ti/drivers/dma/UDMACC26XX.h>

UDMACC26XX_Object udmaObjects[CC2640R2_LAUNCHXL_UDMACOUNT];

const UDMACC26XX_HWAttrs udmaHWAttrs[CC2640R2_LAUNCHXL_UDMACOUNT] = {
    {
        .baseAddr    = UDMA0_BASE,
        .powerMngrId = PowerCC26XX_PERIPH_UDMA,
        .intNum      = INT_DMA_ERR,
        .intPriority = ~0
    }
};

const UDMACC26XX_Config UDMACC26XX_config[CC2640R2_LAUNCHXL_UDMACOUNT] = {
    {
         .object  = &udmaObjects[CC2640R2_LAUNCHXL_UDMA0],
         .hwAttrs = &udmaHWAttrs[CC2640R2_LAUNCHXL_UDMA0]
    },
};



/*
 *  =============================== Watchdog ===============================
 */
#include <ti/drivers/Watchdog.h>
#include <ti/drivers/watchdog/WatchdogCC26XX.h>

WatchdogCC26XX_Object watchdogCC26XXObjects[CC2640R2_LAUNCHXL_WATCHDOGCOUNT];

const WatchdogCC26XX_HWAttrs watchdogCC26XXHWAttrs[CC2640R2_LAUNCHXL_WATCHDOGCOUNT] = {
    {
        .baseAddr    = WDT_BASE,
        .reloadValue = 1000 /* Reload value in milliseconds */
    },
};

const Watchdog_Config Watchdog_config[CC2640R2_LAUNCHXL_WATCHDOGCOUNT] = {
    {
        .fxnTablePtr = &WatchdogCC26XX_fxnTable,
        .object      = &watchdogCC26XXObjects[CC2640R2_LAUNCHXL_WATCHDOG0],
        .hwAttrs     = &watchdogCC26XXHWAttrs[CC2640R2_LAUNCHXL_WATCHDOG0]
    },
};

const uint_least8_t Watchdog_count = CC2640R2_LAUNCHXL_WATCHDOGCOUNT;



/*
 *  ========================= TRNG begin ====================================
 */
#include <TRNGCC26XX.h>

/* TRNG objects */
TRNGCC26XX_Object trngCC26XXObjects[CC2640R2_LAUNCHXL_TRNGCOUNT];

/* TRNG configuration structure, describing which pins are to be used */
const TRNGCC26XX_HWAttrs TRNGCC26XXHWAttrs[CC2640R2_LAUNCHXL_TRNGCOUNT] = {
    {
        .powerMngrId    = PowerCC26XX_PERIPH_TRNG,
    }
};

/* TRNG configuration structure */
const TRNGCC26XX_Config TRNGCC26XX_config[] = {
    {
         .object  = &trngCC26XXObjects[0],
         .hwAttrs = &TRNGCC26XXHWAttrs[0]
    },
    {NULL, NULL}
};

/*
 *  ========================= TRNG end ====================================
 */
void CC2640R2_LAUNCHXL_sendExtFlashByte(PIN_Handle pinHandle, uint8_t byte)
{
    PIN_setOutputValue(pinHandle, CC2640R2_LAUNCHXL_SPI_FLASH_CS, 0);
    uint8_t i;
    for(i = 0; i < 8; i++){
        PIN_setOutputValue(pinHandle, CC2640R2_LAUNCHXL_SPI0_CLK, 0);
        PIN_setOutputValue(pinHandle, CC2640R2_LAUNCHXL_SPI0_MOSI, (byte >> (7 - i)) & 0x01);
        PIN_setOutputValue(pinHandle, CC2640R2_LAUNCHXL_SPI0_CLK, 1);
    }
    PIN_setOutputValue(pinHandle, CC2640R2_LAUNCHXL_SPI_FLASH_CS, 1);
}


/*
 *  ======== CC2640R2_LAUNCHXL_shutDownExtFlash ========
 */

void CC2640R2_LAUNCHXL_shutDownExtFlash(void)
{
    PIN_Config extFlashPinTable[] = {
        CC2640R2_LAUNCHXL_SPI_FLASH_CS | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_INPUT_DIS | PIN_DRVSTR_MED,
        CC2640R2_LAUNCHXL_SPI0_CLK | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_INPUT_DIS | PIN_DRVSTR_MED,
        CC2640R2_LAUNCHXL_SPI0_MOSI | PIN_GPIO_OUTPUT_EN | PIN_GPIO_LOW | PIN_PUSHPULL | PIN_INPUT_DIS | PIN_DRVSTR_MED,
        CC2640R2_LAUNCHXL_SPI0_MISO | PIN_INPUT_EN | PIN_PULLDOWN,
        PIN_TERMINATE
    };
    PIN_State extFlashPinState;
    PIN_Handle extFlashPinHandle = PIN_open(&extFlashPinState, extFlashPinTable);

    uint8_t extFlashStartup = 0xAB;
    uint8_t extFlashShutdown = 0xB9;

    CC2640R2_LAUNCHXL_sendExtFlashByte(extFlashPinHandle, extFlashStartup);
    CC2640R2_LAUNCHXL_sendExtFlashByte(extFlashPinHandle, extFlashShutdown);

    PIN_close(extFlashPinHandle);
}
