/******************************************************************************

@file  app_data.c

@brief This file contains the application data functionality

Group: WCS, BTS
Target Device: cc23xx

******************************************************************************

 Copyright (c) 2022-2023, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

******************************************************************************


*****************************************************************************/

//*****************************************************************************
//! Includes
//*****************************************************************************
#include <string.h>
#include <ti/bleapp/ble_app_util/inc/bleapputil_api.h>
#include <ti/bleapp/menu_module/menu_module.h>
#include <app_main.h>

#include <ti/drivers/dpl/ClockP.h>

//*****************************************************************************
//! Defines
//*****************************************************************************

//*****************************************************************************
//! Globals
//*****************************************************************************

uint8_t display_index = APP_MENU_PROFILE_STATUS_LINE + 1;
static void GATT_EventHandler(uint32 event, BLEAppUtil_msgHdr_t *pMsgData);

/* Note that the events are of BLEAPPUTIL, but in the handler we drop the BLEAPPUTIL prefix. */
BLEAppUtil_EventHandler_t dataGATTHandler =
{
    .handlerType    = BLEAPPUTIL_GATT_TYPE,
    .pEventHandler  = GATT_EventHandler,
    .eventMask      = BLEAPPUTIL_ATT_FLOW_CTRL_VIOLATED_EVENT |
                      BLEAPPUTIL_ATT_MTU_UPDATED_EVENT |
                      BLEAPPUTIL_ATT_FIND_BY_TYPE_VALUE_RSP |
                      BLEAPPUTIL_ATT_READ_BY_TYPE_RSP
};


//*****************************************************************************
//! Functions
//*****************************************************************************

/*********************************************************************
 * @fn      GATT_EventHandler
 *
 * @brief   The purpose of this function is to handle GATT events
 *          that rise from the GATT and were registered in
 *          @ref BLEAppUtil_RegisterGAPEvent
 *
 * @param   event - message event.
 * @param   pMsgData - pointer to message data.
 *
 * @return  none
 */
static void GATT_EventHandler(uint32 event, BLEAppUtil_msgHdr_t *pMsgData)
{
  gattMsgEvent_t *gattMsg = ( gattMsgEvent_t * )pMsgData;
  switch ( gattMsg->method )
  {
    case ATT_FLOW_CTRL_VIOLATED_EVENT:
      {
          MenuModule_printf(APP_MENU_PROFILE_STATUS_LINE, 0, "GATT status: ATT flow control is violated");
      }
      break;

    case ATT_MTU_UPDATED_EVENT:
      {
          MenuModule_printf(APP_MENU_PROFILE_STATUS_LINE, 0, "GATT status: ATT MTU update to %d",
                            gattMsg->msg.mtuEvt.MTU);
      }
      break;
    case ATT_READ_BY_TYPE_RSP:
    {
        display_index += 1;
        attReadByTypeRsp_t *response = (attReadByTypeRsp_t *)&(gattMsg->msg.readByTypeRsp);
        MenuModule_printf(display_index, 0, "GATT status: Read by type %d", response->numPairs);

        for (int i = 0; i < response->numPairs; i++)
        {
            uint16_t attributeHandle = BUILD_UINT16(response->pDataList[i * response->len + 0],
                                                    response->pDataList[i * response->len + 1]);

            uint16_t endHandle = BUILD_UINT16(response->pDataList[i * response->len + 2],
                                              response->pDataList[i * response->len + 3]);

            display_index += 1;
            MenuModule_printf(display_index, 0, "GATT status: Read by type: Handle: %x | %x", attributeHandle, endHandle);
        }
        break;

    }
    case ATT_FIND_BY_TYPE_VALUE_RSP:
    {
        display_index += 1;
        attFindByTypeValueRsp_t *response = (attFindByTypeValueRsp_t *)&(gattMsg->msg.findByTypeValueRsp);
        MenuModule_printf(display_index, 0, "GATT status: Find by type: numInfo %d", response->numInfo);

        for (int i = 0; i < response->numInfo; i++)
        {
            uint16_t attributeHandle = BUILD_UINT16(response->pHandlesInfo[i * response->numInfo + 0],
                                                    response->pHandlesInfo[i * response->numInfo + 1]);

            uint16_t endHandle = BUILD_UINT16(response->pHandlesInfo[i * response->numInfo + 2],
                                              response->pHandlesInfo[i * response->numInfo + 3]);

            display_index += 1;
            MenuModule_printf(display_index, 0, "GATT Handle start: %x | end: %x", attributeHandle, endHandle);


            /*
             * Note: The call to GATT_DiscAllChars will fail if called without a delay. Here is what TI recommends:
             *      1. Store the handles, and trigger a timer which will then call GATT_DiscAllChars with the data.
             *      2. Add a delay.
             *
             *  The reason a delay or timer is needed is because the BLE stack is still processing information. Thus leading
             *  the call to GATT_DiscAllChars to return with 'bleNotReady'. For the sake of convenience the delay implementation
             *  is provided here. However, a timer that triggers an interrupt would be a more elegant solution.
             *
             */
            ClockP_sleep(1);

            bStatus_t status = GATT_DiscAllChars(gattMsg->connHandle, attributeHandle, endHandle, BLEAppUtil_getSelfEntity());

             if (status != SUCCESS)
             {
                 display_index += 1;
                 MenuModule_printf(display_index, 0, "GATT_DiscAllChars Failed: %x", status);
             }
        }



        break;

    }


    default:
      break;
  }
}

/*********************************************************************
 * @fn      Data_start
 *
 * @brief   This function is called after stack initialization,
 *          the purpose of this function is to initialize and
 *          register the specific events handlers of the data
 *          application module
 *
 * @return  SUCCESS, errorInfo
 */
bStatus_t Data_start( void )
{
  bStatus_t status = SUCCESS;

  // Register the handlers
  status = BLEAppUtil_registerEventHandler( &dataGATTHandler );

  // Return status value
  return( status );
}
