/*
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/***** Includes *****/
/* Standard C Libraries */
#include <stdlib.h>
#include <unistd.h>
#include <ti/display/Display.h>

/* TI Drivers */
#include <ti/drivers/rf/RF.h>
#include <ti/drivers/GPIO.h>
//#include <ti/drivers/pin/PINCC26XX.h>

/* Driverlib Header files */
#include DeviceFamily_constructPath(driverlib/rf_ble_mailbox.h)

/* Board Header files */
#include "ti_drivers_config.h"
#include <ti_radio_config.h>

/***** Defines *****/

/* Do power measurement */
//#define POWER_MEASUREMENT

/* Packet TX Configuration */
#define PACKET_INTERVAL     500000  /* Set packet interval to 500000us or 500ms */

/* Packet TX Configuration */
#define BLE5_ADV_ADDR_LENGTH             10
#define BLE4_ADV_ADDR_LENGTH             6

#define ADV_NONCONN_IND                  2 //Packet type for BLE4
#define AUX_ADV_IND                      7 //Packet type for BLE5

#define PROP_LEN_FIELD_LENGTH            1
#define BLE_PDU_HEADER_FIELD_LENGTH      2

/* Buffer needs to be large enough for either packet */
#define PACKET_DATA_LENGTH               27
#define PAYLOAD_LENGTH                   (PACKET_DATA_LENGTH + BLE5_ADV_ADDR_LENGTH) //Create buffer that can accept BLE4 or BLE5 packet

#define NUM_DATA_ENTRIES                 2 // NOTE: Only two data entries supported at the moment
#define NUM_APPENDED_BYTES               8 // BLE PDU Header (2 Bytes), CRC (3 Bytes), RSSI (1 Byte), Status (2 Byte)

/* Buffer needs to be large enough for either packet */
#define EXTENDED_HEADER_LENGTH            9

/***** Prototypes *****/

/***** Variable declarations *****/
static RF_Object rfObject;
static RF_Handle rfHandle;

static uint8_t packet[PAYLOAD_LENGTH];
static uint8_t seqNumber;
rfc_ble5ExtAdvEntry_t ble5ExtAdvPacket;

Display_Handle display;

/***** Function definitions *****/

void *mainThread(void *arg0)
{
    RF_EventMask terminationReason;
    RF_Params rfParams;
    RF_Params_init(&rfParams);

    display = Display_open(Display_Type_UART, NULL);
    Display_printf(display, 0, 0, "mainThread() Init");

    GPIO_setConfig(CONFIG_GPIO_GLED, GPIO_CFG_OUT_STD | GPIO_CFG_OUT_LOW);

    GPIO_write(CONFIG_GPIO_GLED, CONFIG_GPIO_LED_OFF);

    /* Request access to the ble radio */
    RF_cmdBle5AdvAux.pParams->pAdvPkt = (uint8_t *)&ble5ExtAdvPacket;
    ble5ExtAdvPacket.extHdrInfo.length = EXTENDED_HEADER_LENGTH;
    ble5ExtAdvPacket.advDataLen = PACKET_DATA_LENGTH;
    ble5ExtAdvPacket.pAdvData = packet;
    RF_cmdBle5AdvAux.startTrigger.triggerType = TRIG_NOW;
    RF_cmdBle5AdvAux.startTrigger.pastTrig = 1;
    RF_cmdBle5AdvAux.startTime = 0;
    RF_cmdBle5AdvAux.txPower = 0;

    /* Request access to the radio */
    rfHandle = RF_open(&rfObject, &RF_prop, (RF_RadioSetup*)&RF_cmdBle5RadioSetup, &rfParams);
    Display_printf(display, 0, 0, "RF_open(RF_cmdBle5RadioSetup) OK");

    RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityNormal, NULL, 0);
    Display_printf(display, 0, 0, "RF_postCmd(RF_cmdFs) OK");

    while(1)
    {
        /* Create packet with incrementing sequence number and random payload */
        packet[0] = seqNumber++;
        packet[1] = 0x01;
        packet[2] = 0x02;
        packet[3] = 0x03;
        uint8_t i;
        for (i = 4; i < PAYLOAD_LENGTH; i++)
        {
            packet[i] = rand();
        }

        /* Set the frequency */
//        RF_cmdFs.frequency = 2420; // ch8
//        RF_cmdFs.frequency = 2440; // ch17
//        RF_cmdFs.frequency = 2460; // ch27
//        RF_cmdFs.frequency = 2480; // ch39(Adv)
        RF_postCmd(rfHandle, (RF_Op*)&RF_cmdFs, RF_PriorityNormal, NULL, 0);
        Display_printf(display, 0, 0, "RF_postCmd(RF_cmdFs) OK");
        /* Send packet */
//        RF_cmdBle5AdvAux.channel = 0x78; //ch8
//        RF_cmdBle5AdvAux.channel = 0x8C; //ch17
//        RF_cmdBle5AdvAux.channel = 0xA0; //ch27
//        RF_cmdBle5AdvAux.channel = 0xB4; //ch39(Adv)
        terminationReason = RF_runCmd(rfHandle, (RF_Op*)&RF_cmdBle5AdvAux,
                                                   RF_PriorityNormal, NULL, 0);
        Display_printf(display, 0, 0, "RF_runCmd(RF_cmdBle5AdvAux) OK");
        switch(terminationReason)
        {
        case RF_EventLastCmdDone:
            // A stand-alone radio operation command or the last radio
            // operation command in a chain finished.
            break;
        case RF_EventCmdCancelled:
            // Command cancelled before it was started; it can be caused
        // by RF_cancelCmd() or RF_flushCmd().
            break;
        case RF_EventCmdAborted:
            // Abrupt command termination caused by RF_cancelCmd() or
            // RF_flushCmd().
            break;
        case RF_EventCmdStopped:
            // Graceful command termination caused by RF_cancelCmd() or
            // RF_flushCmd().
            break;
        default:
            // Uncaught error event
            while(1);
        }

        uint32_t cmdStatus = ((volatile RF_Op*)&RF_cmdBle5AdvAux)->status;
        switch(cmdStatus)
        {
        case BLE_DONE_OK:
            // Packet transmitted successfully
            break;
        case BLE_DONE_STOPPED:
            // received CMD_STOP while transmitting packet and finished
            // transmitting packetoi
            break;
        case BLE_DONE_ABORT:
            // Received CMD_ABORT while transmitting packet
            break;
        case BLE_ERROR_PAR:
            // Observed illegal parameter
            break;
        case BLE_ERROR_NO_SETUP:
            // Command sent without setting up the radio in a supported
            // mode using CMD_BLE_RADIO_SETUP or CMD_RADIO_SETUP
            break;
        case BLE_ERROR_NO_FS:
            // Command sent without the synthesizer being programmed
            break;
        case BLE_ERROR_TXUNF:
            // TX underflow observed during operation
            break;
        default:
            // Uncaught error event - these could come from the
            // pool of states defined in rf_mailbox.h
            while (1);
        }

        GPIO_toggle(CONFIG_GPIO_GLED);

        /* Power down the radio */
        RF_yield(rfHandle);

        /* Sleep for PACKET_INTERVAL us */
        sleep(1);
//        usleep(PACKET_INTERVAL);
    }
}
