/*******************************************************************************
*  Filename:       am_i2c.c
*  Revised:        $Date: $
*  Revision:       $Revision: $
*
*  Description:    I2C setup and handlers for use with the Activity Monitor
*                  application
*
*  Copyright (C) 2015 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*******************************************************************************/

/*******************************************************************************
 * INCLUDES
 */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/family/arm/cc26xx/Power.h>

#include <ti/drivers/i2c/I2CCC26XX.h>

#include <driverlib/prcm.h>

#include "board.h"
#include "am_i2c.h"

/*******************************************************************************
 * CONSTANTS
 */

/*******************************************************************************
 * GLOBAL variables
 */

/*******************************************************************************
 * LOCAL variables
 */
static volatile uint8_t slaveAddr;
static volatile uint8_t interface;
static I2C_Handle i2cHandle;
static I2C_Params i2cParams;

/*******************************************************************************
 * @fn          amI2cInit
 *
 * @brief       Initialize the RTOS I2C driver (must be called only once)
 *
 * @param       none
 *
 * @return      none
 */
void amI2cInit(void)
{
  I2C_Params_init(&i2cParams);
  i2cParams.bitRate = I2C_400kHz;
  i2cHandle = I2C_open(Board_I2C, &i2cParams);
  while(!i2cHandle) {
  }

  // Initialize local variable
  slaveAddr = 0xFF;
}

/*******************************************************************************
 * @fn          amI2cSelect
 *
 * @brief       Select an I2C slave
 *
 * @param       address - slave address
  *
 * @return      true if success
 */
bool amI2cSelect(uint8_t address)
{
  // Store new slave address
  slaveAddr = address;

  return true;
}

/*******************************************************************************
 * @fn          amI2cWrite
 *
 * @brief       Burst write to an I2C device
 *
 * @param       data - pointer to data buffer
 * @param       len - number of bytes to write
 *
 * @return      true if success
 */
bool amI2cWrite(uint8_t *data, uint8_t len)
{
  I2C_Transaction masterTransaction;

  masterTransaction.writeCount   = len;
  masterTransaction.writeBuf     = data;
  masterTransaction.readCount    = 0;
  masterTransaction.readBuf      = NULL;
  masterTransaction.slaveAddress = slaveAddr;

  return I2C_transfer(i2cHandle, &masterTransaction) == TRUE;
}

/*******************************************************************************
 * @fn          amI2cWriteSingle
 *
 * @brief       Single byte write to an I2C device
 *
 * @param       data - byte to write
 *
 * @return      true if success
 */
bool amI2cWriteSingle(uint8_t data)
{
  uint8_t d;

  d = data;

  return amI2cWrite(&d, 1);
}

/*******************************************************************************
 * @fn          amI2cRead
 *
 * @brief       Burst read from an I2C device
 *
 * @param       data - pointer to data buffer
 * @param       len - number of bytes to write
 *
 * @return      true if success
 */
bool amI2cRead(uint8_t *data, uint8_t len)
{
  I2C_Transaction masterTransaction;

  masterTransaction.writeCount   = 0;
  masterTransaction.writeBuf     = NULL;
  masterTransaction.readCount    = len;
  masterTransaction.readBuf      = data;
  masterTransaction.slaveAddress = slaveAddr;

  return I2C_transfer(i2cHandle, &masterTransaction) == TRUE;
}

/*******************************************************************************
 * @fn          amI2cWriteRead
 *
 * @brief       Burst write/read from an I2C device
 *
 * @param       wdata - pointer to write data buffer
 * @param       wlen - number of bytes to write
 * @param       rdata - pointer to read data buffer
 * @param       rlen - number of bytes to read
 *
 * @return      true if success
 */
bool amI2cWriteRead(uint8_t *wdata, uint8_t wlen, uint8_t *rdata, uint8_t rlen)
{
  I2C_Transaction masterTransaction;

  masterTransaction.writeCount   = wlen;
  masterTransaction.writeBuf     = wdata;
  masterTransaction.readCount    = rlen;
  masterTransaction.readBuf      = rdata;
  masterTransaction.slaveAddress = slaveAddr;

  return I2C_transfer(i2cHandle, &masterTransaction) == TRUE;
}

/*******************************************************************************
 * @fn          amI2cDisable
 *
 * @brief       Disable the RTOS I2C driver
 *
 * @param       none
 *
 * @return      none
 */
void amI2cDisable(void)
{
  I2C_close(i2cHandle);
}

/*******************************************************************************
 * @fn          amI2cReset
 *
 * @brief       Reset the RTOS I2C driver
 *
 * @param       none
 *
 * @return      none
 */
void amI2cReset(void)
{
  // Close the driver
  I2C_close(i2cHandle);

  // Reset the I2C controller
  HapiResetPeripheral(PRCM_PERIPH_I2C0);

  // Reset local variable
  slaveAddr = 0xFF;

  // Open driver
  i2cHandle = I2C_open(Board_I2C, &i2cParams);
}
