/******************************************************************************

 @file  simple_Spi.c

 @brief This file contains the Simple BLE spi sample application for use
        with the CC2650 Bluetooth Low Energy Protocol Stack.

 Group: WCS, BTS
 Target Device: CC2650, CC2640, CC1350

 ******************************************************************************
 
 Copyright (c) 2011-2016, Texas Instruments Incorporated
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 *  Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

 *  Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

 *  Neither the name of Texas Instruments Incorporated nor the names of
    its contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ******************************************************************************
 Release Name: ble_sdk_2_02_00_31
 Release Date: 2016-06-16 18:57:29
 *****************************************************************************/

/*********************************************************************
 * INCLUDES
 */
#include <string.h>

#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Mailbox.h>
#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>

#include "board.h"
#include "simple_spi.h"
#include "simple_observer.h"
#include "hci.h"
#include <ti/drivers/SPI.h>
#include <ti/drivers/spi/SPICC26XXDMA.h>
#include <ti/drivers/dma/UDMACC26XX.h>

/*********************************************************************
 * MACROS
 */

/*********************************************************************
 * CONSTANTS
 */
  
// Task configuration
#define SBI_TASK_PRIORITY                     1

#ifndef SBI_TASK_STACK_SIZE
#define SBI_TASK_STACK_SIZE                   660
#endif

#define SPI_CMD_READ_RSSI_VALUES 0x03

#define CMD_LENGTH 1

/*********************************************************************
 * TYPEDEFS
 */


/*********************************************************************
 * GLOBAL VARIABLES
 */

/*
 * Application output pin configuration table
 */
PIN_Config IRQPinTable[] = {
    Board_BLE_SPI_READY  | PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL,
    PIN_TERMINATE
};

/*********************************************************************
 * EXTERNAL VARIABLES
 */

/*********************************************************************
 * EXTERNAL FUNCTIONS
 */

/*********************************************************************
 * LOCAL VARIABLES
 */

static PIN_Handle IRQPinHandle;
static PIN_State IRQPinState;

static SPI_Handle spiHandle;
static SPI_Params spiParams;

static SPI_Transaction transaction;
int8_t spi_tx_buf[RSSI_TABLE_LENGTH];
uint8_t spi_rx_buf[CMD_LENGTH];

// Task configuration
static Task_Struct sboTask;
static Char sboTaskStack[SBI_TASK_STACK_SIZE];

/*********************************************************************
 * LOCAL FUNCTIONS
 */
void SimpleBLESpi_spiTransferCB(SPI_Handle spiHandle, SPI_Transaction *transaction);
static void SimpleBLESpi_init(void);
static void SimpleBLESpi_taskFxn(UArg a0, UArg a1);

/*********************************************************************
 * PUBLIC FUNCTIONS
 */

/*********************************************************************
 * @fn      SimpleBLESpi_createTask
 *
 * @brief   Task creation function for the Simple BLE Spi.
 *
 * @param   none
 *
 * @return  none
 */
void SimpleBLESpi_createTask(void)
{
  Task_Params taskParams;
    
  // Configure task
  Task_Params_init(&taskParams);
  taskParams.stack = sboTaskStack;
  taskParams.stackSize = SBI_TASK_STACK_SIZE;
  taskParams.priority = SBI_TASK_PRIORITY;
  
  Task_construct(&sboTask, SimpleBLESpi_taskFxn, &taskParams, NULL);
}

/*********************************************************************
 * @fn      SimpleBLESpi_init
 *
 * @brief   Initialization function for the Simple BLE Spi App Task.
 *          This is called during initialization and should contain
 *          any application specific initialization (ie. hardware
 *          initialization/setup, table initialization, power up
 *          notification).
 *
 * @param   none
 *
 * @return  none
 */
void SimpleBLESpi_init(void)
{

    /* Open IRQ pins */
    IRQPinHandle = PIN_open(&IRQPinState, IRQPinTable);

    // Init SPI and specify non-default parameters
    SPI_Params_init(&spiParams);
    spiParams.bitRate      = 100000;
    spiParams.frameFormat  = SPI_POL0_PHA1;
    spiParams.mode         = SPI_SLAVE;
//    spiParams.transferMode        = SPI_MODE_BLOCKING;
    spiParams.transferMode        = SPI_MODE_CALLBACK;
    spiParams.transferCallbackFxn = SimpleBLESpi_spiTransferCB;

    // Open the SPI
    spiHandle = SPI_open(Board_SPI0, &spiParams);

//    SPI_control(spiHandle, SPICC26XXDMA_CMD_RETURN_PARTIAL_ENABLE, NULL);
}

/*********************************************************************
 * @fn      SimpleBLESpi_taskFxn
 *
 * @brief   Application task entry point for the Simple BLE Spi.
 *
 * @param   none
 *
 * @return  none
 */
static void SimpleBLESpi_taskFxn(UArg a0, UArg a1)
{
    Bool ret;
    int8_t new_rssi_data[RSSI_TABLE_LENGTH];

	// Initialize application
	SimpleBLESpi_init();

	transaction.count = RSSI_TABLE_LENGTH;
	transaction.txBuf = spi_tx_buf;
	transaction.rxBuf = spi_rx_buf;
//	ret = SPI_transfer(spiHandle, &transaction);
//	if(!ret) {
//	    System_abort("Error with SPI_transfer at init\n");
//	    return;
//	}

	while(true) {

	    ret = Mailbox_pend(mailboxHandle, new_rssi_data, BIOS_WAIT_FOREVER);
	    if(ret) {
	        memcpy(spi_tx_buf, new_rssi_data, RSSI_TABLE_LENGTH);
	        // Falling edge interrupt to MCU through Board_BLE_SPI_READY
            PIN_setOutputValue(IRQPinHandle, Board_BLE_SPI_READY, 0);
            SPI_transfer(spiHandle, &transaction);
            if(!ret) {
                System_abort("Error with SPI_transfer at init\n");
                break;
            }
            PIN_setOutputValue(IRQPinHandle, Board_BLE_SPI_READY, 1);
	    } else {
	        break;
	    }
	}

}

/*********************************************************************
 * @fn      SimpleBLEObserver_spiTransferCB
 *
 * @brief   Observer SPI transfer ended callback function.
 *
 * @param   pEvent - pointer to event structure
 */
void SimpleBLESpi_spiTransferCB(SPI_Handle spiHandle, SPI_Transaction *trnsction)
{
//    Bool ret;
//    ret = SPI_transfer(spiHandle, &transaction);
//    if(!ret) {
//        System_abort("Error with SPI_transfer in SPI transfer callback\n");
//    }
}
