/*
 * linux/sound/soc/codecs/tlv320adc3101.c
 *
 * Copyright 2019 by Garmin Ltd. or its subsidiaries.
 * Louisyw Lu <Louisyw.Lu@garmin.com>
 *
 * Foundation is based on sound/soc/codecs/tlv320aic32x4.c
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/gpio.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/i2c.h>
#include <linux/cdev.h>
#include <linux/slab.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>
#include <sound/tlv.h>

#include "tlv320adc3101.h"

struct adc3101_rate_divs {
	u32 mclk;
	u32 rate;
	u8 p_val;
	u8 pll_j;
	u16 pll_d;
	u16 dosr;
	u8 ndac;
	u8 mdac;
	u8 aosr;
	u8 nadc;
	u8 madc;
	u8 blck_N;
};

struct adc3101_priv {
	struct snd_soc_codec *codec;
	struct regmap *regmap;
	u32 sysclk;
	u32 power_cfg;
	u32 micpga_routing;
	int rstn_gpio;
	unsigned int mic_differential;
	int mic_type;
};

static int adc3101_mic_type_get(struct snd_kcontrol *kcontrol,
    struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_soc_kcontrol_codec(kcontrol);
	struct adc3101_priv *adc3101 = snd_soc_codec_get_drvdata(codec);

	ucontrol->value.integer.value[0] = adc3101->mic_type;

	return 0;
};

static char const *adc3101_mic_type_text[] = {"Internal DMIC", "Internal AMIC", "External AMIC"};

static SOC_ENUM_SINGLE_EXT_DECL(adc3101_mic_type, adc3101_mic_type_text);

static const struct snd_kcontrol_new adc3101_dmic_ch_control =
	SOC_DAPM_DOUBLE("Channel Switch", ADC3101_ADCSETUP,
			2, 3, 1, 0);

/* 0dB min, 0.5dB steps */
static DECLARE_TLV_DB_SCALE(tlv_step_0_5, 0, 50, 0);

static const struct snd_kcontrol_new adc3101_snd_controls[] = {
	SOC_DOUBLE_R("ADC3101 Left Right PGA Switch", ADC3101_LMICPGAVOL,
			ADC3101_RMICPGAVOL, 7, 0x01, 1),

	SOC_SINGLE("ADC3101 ADCFGA Left Mute Switch", ADC3101_ADCFGA, 7, 1, 0),
	SOC_SINGLE("ADC3101 ADCFGA Right Mute Switch", ADC3101_ADCFGA, 3, 1, 0),

	SOC_DOUBLE_R_TLV("ADC3101 ADC Level Volume", ADC3101_LADCVOL,
			ADC3101_RADCVOL, 0, 0x28, 0, tlv_step_0_5),
	SOC_DOUBLE_R_TLV("ADC3101 PGA Level Volume", ADC3101_LMICPGAVOL,
			ADC3101_RMICPGAVOL, 0, 0x50, 0, tlv_step_0_5),

	SOC_SINGLE("ADC3101 AGC Left Switch", ADC3101_LAGC_CTL1, 7, 1, 0),
	SOC_SINGLE("ADC3101 AGC Right Switch", ADC3101_RAGC_CTL1, 7, 1, 0),
	SOC_DOUBLE_R("ADC3101 AGC Target Level", ADC3101_LAGC_CTL1, ADC3101_RAGC_CTL1,
			4, 0x07, 0),
	SOC_DOUBLE_R("ADC3101 AGC Hysteresis", ADC3101_LAGC_CTL2, ADC3101_RAGC_CTL2,
			6, 0x03, 0),
	SOC_DOUBLE_R("ADC3101 AGC Noise Threshold", ADC3101_LAGC_CTL2, ADC3101_RAGC_CTL2,
			1, 0x1F, 0),
	SOC_DOUBLE_R("ADC3101 AGC Max PGA Gain", ADC3101_LAGC_MAXGAIN, ADC3101_RAGC_MAXGAIN,
			0, 0x7F, 0),
	SOC_DOUBLE_R("ADC3101 AGC Attack Time", ADC3101_LAGC_ATK_TIME, ADC3101_RAGC_ATK_TIME,
			3, 0x1F, 0),
	SOC_DOUBLE_R("ADC3101 AGC Decay Time", ADC3101_LAGC_DECY_TIME, ADC3101_RAGC_DECY_TIME,
			3, 0x1F, 0),
	SOC_DOUBLE_R("ADC3101 AGC Noise Debounce", ADC3101_LAGC_NOISE_DBCE, ADC3101_RAGC_NOISE_DBCE,
			0, 0x1F, 0),
	SOC_DOUBLE_R("ADC3101 AGC Signal Debounce", ADC3101_LAGC_SIG_DBCE, ADC3101_RAGC_SIG_DBCE,
			0, 0x0F, 0),

	SOC_ENUM_EXT("ADC3101 MIC Type", adc3101_mic_type,
			adc3101_mic_type_get, NULL),
};

static const struct adc3101_rate_divs adc3101_divs[] = {
	/* mclk rate p  j d dosr ndac mdac aosr nadc madc blk_N */
	/* 8k rate */
	{ADC3101_FREQ_2048000,  8000, 1, 4,    0, 128, 1, 1, 128, 1,  2,  1},
	{ADC3101_FREQ_12000000, 8000, 1, 7, 6800, 768, 5, 3, 128, 5, 18, 24},
	{ADC3101_FREQ_24000000, 8000, 2, 7, 6800, 768, 15, 1, 64, 45, 4, 24},
	{ADC3101_FREQ_25000000, 8000, 2, 7, 3728, 768, 15, 1, 64, 45, 4, 24},
	/* 11.025k rate */
	{ADC3101_FREQ_12000000, 11025, 1, 7, 5264, 512, 8, 2, 128, 8, 8, 16},
	{ADC3101_FREQ_24000000, 11025, 2, 7, 5264, 512, 16, 1, 64, 32, 4, 16},
	/* 16k rate */
	{ADC3101_FREQ_4096000,  16000, 1, 4,    0, 128, 1, 1, 128, 1, 2, 1},
	{ADC3101_FREQ_12000000, 16000, 1, 7, 6800, 384, 5, 3, 128, 5, 9, 12},
	{ADC3101_FREQ_24000000, 16000, 2, 7, 6800, 384, 15, 1, 64, 18, 5, 12},
	{ADC3101_FREQ_25000000, 16000, 2, 7, 3728, 384, 15, 1, 64, 18, 5, 12},
	/* 22.05k rate */
	{ADC3101_FREQ_12000000, 22050, 1, 7, 5264, 256, 4, 4, 128, 4, 8, 8},
	{ADC3101_FREQ_24000000, 22050, 2, 7, 5264, 256, 16, 1, 64, 16, 4, 8},
	{ADC3101_FREQ_25000000, 22050, 2, 7, 2253, 256, 16, 1, 64, 16, 4, 8},
	/* 32k rate */
	{ADC3101_FREQ_8192000,  32000, 1, 4,    0, 128, 1, 1, 128, 1, 2, 1},
	{ADC3101_FREQ_12000000, 32000, 1, 7, 1680, 192, 2, 7, 64, 2, 21, 6},
	{ADC3101_FREQ_24000000, 32000, 2, 7, 1680, 192, 7, 2, 64, 7, 6, 6},
	/* 44.1k rate */
	{ADC3101_FREQ_11289600, 44100, 1, 4,    0, 128, 1, 1, 128, 1, 2, 1},
	{ADC3101_FREQ_12000000, 44100, 1, 7, 5264, 128, 2, 8, 128, 2, 8, 4},
	{ADC3101_FREQ_24000000, 44100, 2, 7, 5264, 128, 8, 2, 64, 8, 4, 4},
	{ADC3101_FREQ_25000000, 44100, 2, 7, 2253, 128, 8, 2, 64, 8, 4, 4},
	/* 48k rate */
	{ADC3101_FREQ_12000000, 48000, 1, 8, 1920, 128, 2, 8, 128, 2, 8, 4},
	{ADC3101_FREQ_12288000, 48000, 1, 4,    0, 128, 2, 1, 128, 1, 2, 1},
	{ADC3101_FREQ_24000000, 48000, 2, 8, 1920, 128, 8, 2, 64, 8, 4, 4},
	{ADC3101_FREQ_25000000, 48000, 2, 7, 8643, 128, 8, 2, 64, 8, 4, 4},
};

static const struct snd_kcontrol_new left_input_mixer_controls[] = {
	SOC_DAPM_SINGLE("IN2_L L+ IN3_L L- Switch", ADC3101_LMICPGAPIN, 6, 3, 0),
	SOC_DAPM_SINGLE("IN3_L L- Switch", ADC3101_LMICPGAPIN, 4, 3, 0),
	SOC_DAPM_SINGLE("IN2_L L+ Switch", ADC3101_LMICPGAPIN, 2, 3, 0),
	SOC_DAPM_SINGLE("IN1_L L+ Switch", ADC3101_LMICPGAPIN, 0, 3, 0),

	SOC_DAPM_SINGLE("IN2_L L+ IN3_L L- Bypass Switch", ADC3101_LMICPGANIN, 7, 1, 0),
	SOC_DAPM_SINGLE("IN1_L L+ IN1_R L- Switch", ADC3101_LMICPGANIN, 4, 3, 0),
	SOC_DAPM_SINGLE("IN2_R L+ IN3_R L- Switch", ADC3101_LMICPGANIN, 2, 3, 0),
	SOC_DAPM_SINGLE("IN1_R L- Switch", ADC3101_LMICPGANIN, 0, 3, 0),
};

static const struct snd_kcontrol_new right_input_mixer_controls[] = {
	SOC_DAPM_SINGLE("IN2_R R+ IN3_R R- Switch", ADC3101_RMICPGAPIN, 6, 3, 0),
	SOC_DAPM_SINGLE("IN3_R R- Switch", ADC3101_RMICPGAPIN, 4, 3, 0),
	SOC_DAPM_SINGLE("IN2_R R+ Switch", ADC3101_RMICPGAPIN, 2, 3, 0),
	SOC_DAPM_SINGLE("IN1_R R- Switch", ADC3101_RMICPGAPIN, 0, 3, 0),

	SOC_DAPM_SINGLE("IN2_R R+ IN3_R R- Bypass Switch", ADC3101_RMICPGANIN, 7, 1, 0),
	SOC_DAPM_SINGLE("IN1_L R+ IN1_R R- Switch", ADC3101_RMICPGANIN, 4, 3, 0),
	SOC_DAPM_SINGLE("IN2_L R+ IN3_L R- Switch", ADC3101_RMICPGANIN, 2, 3, 0),
	SOC_DAPM_SINGLE("IN1_L R+ Switch", ADC3101_RMICPGANIN, 0, 3, 0),
};

static const struct snd_soc_dapm_widget adc3101_dapm_widgets[] = {
	SND_SOC_DAPM_MIXER("Left Input Mixer", SND_SOC_NOPM, 0, 0,
			   &left_input_mixer_controls[0],
			   ARRAY_SIZE(left_input_mixer_controls)),
	SND_SOC_DAPM_MIXER("Right Input Mixer", SND_SOC_NOPM, 0, 0,
			   &right_input_mixer_controls[0],
			   ARRAY_SIZE(right_input_mixer_controls)),
	SND_SOC_DAPM_ADC("Left ADC", "Left Capture", ADC3101_ADCSETUP, 7, 0),
	SND_SOC_DAPM_ADC("Right ADC", "Right Capture", ADC3101_ADCSETUP, 6, 0),
	SND_SOC_DAPM_ADC("Digital Mic Left ADC", "DMic Left Capture", ADC3101_ADCSETUP, 3, 0),
	SND_SOC_DAPM_ADC("Digital Mic Right ADC", "DMic Right Capture", ADC3101_ADCSETUP, 2, 0),

	SND_SOC_DAPM_SWITCH("DMIC Input", SND_SOC_NOPM,
			   0, 0, &adc3101_dmic_ch_control),

	SND_SOC_DAPM_INPUT("IN1_L L+ Single"),
	SND_SOC_DAPM_INPUT("IN2_L L+ Single"),
	SND_SOC_DAPM_INPUT("IN3_L L- Single"),
	SND_SOC_DAPM_INPUT("IN2_L L+ IN3_L L- Diff"),
	SND_SOC_DAPM_INPUT("Left PGA Bypass"),
	SND_SOC_DAPM_INPUT("IN1_L L+ IN1_R L- Diff"),
	SND_SOC_DAPM_INPUT("IN2_R L+ IN3_R L- Diff"),
	SND_SOC_DAPM_INPUT("IN1_R L- Single"),

	SND_SOC_DAPM_INPUT("IN2_R R+ IN_3_R R- Diff"),
	SND_SOC_DAPM_INPUT("IN3_R R- Single"),
	SND_SOC_DAPM_INPUT("IN2_R R+ Single"),
	SND_SOC_DAPM_INPUT("IN1_R R- Single"),
	SND_SOC_DAPM_INPUT("Right PGA Bypass"),
	SND_SOC_DAPM_INPUT("IN1_L R+ IN1_R R- Diff"),
	SND_SOC_DAPM_INPUT("IN2_L R+ IN3_L R- Diff"),
	SND_SOC_DAPM_INPUT("IN1_L R+ Single"),

	SND_SOC_DAPM_INPUT("DMIC"),
};

static const struct snd_soc_dapm_route adc3101_dapm_routes[] = {
	/* Left Input */
	{"Left ADC", NULL, "Left Input Mixer"},

	{"Left Input Mixer", "IN1_L L+ Switch", "IN1_L L+ Single"},
	{"Left Input Mixer", "IN2_L L+ Switch", "IN2_L L+ Single"},
	{"Left Input Mixer", "IN3_L L- Switch", "IN3_L L- Single"},
	{"Left Input Mixer", "IN2_L L+ IN3_L L- Switch", "IN2_L L+ IN3_L L- Diff"},

	{"Left Input Mixer", "IN2_L L+ IN3_L L- Bypass Switch", "Left PGA Bypass"},
	{"Left Input Mixer", "IN1_L L+ IN1_R L- Switch", "IN1_L L+ IN1_R L- Diff"},
	{"Left Input Mixer", "IN2_R L+ IN3_R L- Switch", "IN2_R L+ IN3_R L- Diff"},
	{"Left Input Mixer", "IN1_R L- Switch", "IN1_R L- Single"},

	/* Right Input */
	{"Right ADC", NULL, "Right Input Mixer"},

	{"Right Input Mixer", "IN2_R R+ IN3_R R- Switch", "IN2_R R+ IN_3_R R- Diff"},
	{"Right Input Mixer", "IN3_R R- Switch", "IN3_R R- Single"},
	{"Right Input Mixer", "IN2_R R+ Switch", "IN2_R R+ Single"},
	{"Right Input Mixer", "IN1_R R- Switch", "IN1_R R- Single"},

	{"Right Input Mixer", "IN2_R R+ IN3_R R- Bypass Switch", "Right PGA Bypass"},
	{"Right Input Mixer", "IN1_L R+ IN1_R R- Switch", "IN1_L R+ IN1_R R- Diff"},
	{"Right Input Mixer", "IN2_L R+ IN3_L R- Switch", "IN2_L R+ IN3_L R- Diff"},
	{"Right Input Mixer", "IN1_L R+ Switch", "IN1_L R+ Single"},

	/* DMIC Input */
	{"DMIC Input", "Channel Switch", "DMIC"},
};

static const struct regmap_range_cfg adc3101_regmap_pages[] = {
	{
		.range_min = 0,
		.range_max = ADC3101_RMICPGAVOL,
		.selector_reg = ADC3101_PSEL,
		.selector_mask = 0xff,
		.selector_shift = 0,
		.window_start = 0,
		.window_len = 128,
	},
};

static const struct regmap_config adc3101_i2c_regmap = {
	.reg_bits = 8,
	.val_bits = 8,
	.ranges = adc3101_regmap_pages,
	.num_ranges = ARRAY_SIZE(adc3101_regmap_pages),
	.max_register = ADC3101_RMICPGAVOL,
};

static inline int adc3101_get_divs(int mclk, int rate)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(adc3101_divs); i++) {
		if ((adc3101_divs[i].rate == rate)
		    && (adc3101_divs[i].mclk == mclk)) {
			return i;
		}
	}

	pr_err("adc3101: master clock and sample rate is not supported\n");
	return -EINVAL;
}

static int adc3101_set_dai_sysclk(struct snd_soc_dai *codec_dai,
				  int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct adc3101_priv *adc3101 = snd_soc_codec_get_drvdata(codec);

	switch (freq) {
	case ADC3101_FREQ_12000000:
	case ADC3101_FREQ_24000000:
	case ADC3101_FREQ_25000000:
	case ADC3101_FREQ_11289600:
	case ADC3101_FREQ_12288000:
	case ADC3101_FREQ_2048000:
	case ADC3101_FREQ_4096000:
	case ADC3101_FREQ_8192000:
		adc3101->sysclk = freq;
		return 0;
	}

	dev_err(codec->dev, "adc3101: invalid frequency to set DAI system clock\n");
	return -1;
}

static int adc3101_set_dai_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct adc3101_priv *adc3101 = NULL;
	u8 iface_reg_1;
	u8 dsp_a_val;
	u8 iface_reg_2;

	adc3101 = snd_soc_codec_get_drvdata(codec);
	if (adc3101 == NULL)
		return -EINVAL;

	iface_reg_1 = snd_soc_read(codec, ADC3101_IFACE1);
	iface_reg_1 = iface_reg_1 & ~(3 << 6 | 3 << 2);
	dsp_a_val = 0;
	iface_reg_2 = snd_soc_read(codec, ADC3101_IFACE2);
	iface_reg_2 = iface_reg_2 & ~(1 << 3);

	/* set master/slave audio interface */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		dev_info(codec->dev, "%s: codec clk & FRM master \n", __func__);
		iface_reg_1 |= ADC3101_BCLKMASTER | ADC3101_WCLKMASTER;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		dev_info(codec->dev, "%s: codec clk & FRM slave \n", __func__);
		break;
	default:
		dev_err(codec->dev, "adc3101: invalid DAI master/slave interface\n");
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		dev_info(codec->dev, "%s: codec DAI format is I2S \n", __func__);
		break;
	case SND_SOC_DAIFMT_DSP_A:
		dev_info(codec->dev, "%s: codec DAI format is DSP_A \n", __func__);
		iface_reg_1 |= (ADC3101_DSP_MODE << ADC3101_PLLJ_SHIFT);
		if ((fmt & SND_SOC_DAIFMT_INV_MASK) == SND_SOC_DAIFMT_IB_NF)
		    iface_reg_2 |= (1 << 3); /* invert bit clock */
		dsp_a_val = 0x01; /* default add offset 1 */
		break;
	case SND_SOC_DAIFMT_DSP_B:
		dev_info(codec->dev, "%s: codec DAI format is DSP_B \n", __func__);
		iface_reg_1 |= (ADC3101_DSP_MODE << ADC3101_PLLJ_SHIFT);
		if ((fmt & SND_SOC_DAIFMT_INV_MASK) == SND_SOC_DAIFMT_IB_NF)
		    iface_reg_2 |= (1 << 3); /* invert bit clock */
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		dev_info(codec->dev, "%s: codec DAI format is RIGHT_J \n", __func__);
		iface_reg_1 |=
			(ADC3101_RIGHT_JUSTIFIED_MODE << ADC3101_PLLJ_SHIFT);
		break;
	case SND_SOC_DAIFMT_LEFT_J:
		dev_info(codec->dev, "%s: codec DAI format is LEFT_J \n", __func__);
		iface_reg_1 |=
			(ADC3101_LEFT_JUSTIFIED_MODE << ADC3101_PLLJ_SHIFT);
		break;
	default:
		dev_err(codec->dev, "adc3101: invalid DAI interface format\n");
		return -EINVAL;
	}

	snd_soc_write(codec, ADC3101_IFACE1, iface_reg_1);
	snd_soc_write(codec, ADC3101_DATASLOTOFFSETCTL, dsp_a_val);
	snd_soc_write(codec, ADC3101_IFACE2, iface_reg_2);

	if (adc3101->mic_type == 0) {
		/* set DMCLK to ADC_MOD_CLK */
		snd_soc_write(codec, ADC3101_DMCLKCTL, 0x28);

		/* setup DMDIN to input mode */
		snd_soc_write(codec, ADC3101_DMDINCTL, 0x04);
	}

	return 0;
}

static int adc3101_hw_params(struct snd_pcm_substream *substream,
			     struct snd_pcm_hw_params *params,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct adc3101_priv *adc3101 = snd_soc_codec_get_drvdata(codec);
	unsigned int rate = params_rate(params);
	u8 data;
	int i;

	dev_dbg(codec->dev, "%s() rate=%u \n", __func__, rate);

	i = adc3101_get_divs(adc3101->sysclk, rate);
	if (i < 0) {
		dev_err(codec->dev, "adc3101: sampling rate (%d), sysclk (%ld) not supported\n", rate, adc3101->sysclk);
		return i;
	}

	/* Use PLL as CODEC_CLKIN and DAC_MOD_CLK as BDIV_CLKIN */
	snd_soc_write(codec, ADC3101_CLKMUX, 0);

	/* We will fix R value to 1 and will make P & J=K.D as varialble */
	data = snd_soc_read(codec, ADC3101_PLLPR);
	data &= ~(7 << 4);

	snd_soc_write(codec, ADC3101_PLLPR,
		      (data | (adc3101_divs[i].p_val << 4) | 0x01));

	snd_soc_write(codec, ADC3101_PLLJ, adc3101_divs[i].pll_j);

	snd_soc_write(codec, ADC3101_PLLDMSB, (adc3101_divs[i].pll_d >> 8));
	snd_soc_write(codec, ADC3101_PLLDLSB,
		      (adc3101_divs[i].pll_d & 0xff));
	/* NADC divider value */
	data = snd_soc_read(codec, ADC3101_NADC);
	data &= ~(0x7f);
	snd_soc_write(codec, ADC3101_NADC, data | adc3101_divs[i].nadc);

	/* MADC divider value */
	data = snd_soc_read(codec, ADC3101_MADC);
	data &= ~(0x7f);
	snd_soc_write(codec, ADC3101_MADC, data | adc3101_divs[i].madc);

	/* AOSR value */
	snd_soc_write(codec, ADC3101_AOSR, adc3101_divs[i].aosr);

	/* BCLK N divider */
	data = snd_soc_read(codec, ADC3101_BCLKN);
	data &= ~(0x7f);
	snd_soc_write(codec, ADC3101_BCLKN, data | adc3101_divs[i].blck_N);

	data = snd_soc_read(codec, ADC3101_IFACE1);
	data = data & ~(3 << 4);
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		break;
	case SNDRV_PCM_FORMAT_S20_3LE:
		data |= (ADC3101_WORD_LEN_20BITS << ADC3101_DOSRMSB_SHIFT);
		break;
	case SNDRV_PCM_FORMAT_S24_3LE:
	case SNDRV_PCM_FORMAT_S24_LE:
		data |= (ADC3101_WORD_LEN_24BITS << ADC3101_DOSRMSB_SHIFT);
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		data |= (ADC3101_WORD_LEN_32BITS << ADC3101_DOSRMSB_SHIFT);
		break;
	}

	snd_soc_write(codec, ADC3101_IFACE1, data);

	return 0;
}

static int adc3101_set_bias_level(struct snd_soc_codec *codec,
				  enum snd_soc_bias_level level)
{
	struct adc3101_priv *adc3101 = snd_soc_codec_get_drvdata(codec);

	dev_info(codec->dev, "%s() level=%d\n", __func__, level);

	switch (level) {
	case SND_SOC_BIAS_ON:
		if (adc3101->mic_type == 1 || adc3101->mic_type == 2) {
			/* Set Mic Bias to AVDD */
			snd_soc_write(codec, ADC3101_MICBIAS, ADC3101_MICBIAS1_BIAS2_AVDD);
		}

		/* Switch on PLL */
		snd_soc_update_bits(codec, ADC3101_PLLPR,
				    ADC3101_PLLEN, ADC3101_PLLEN);

		/* Switch on NADC Divider */
		snd_soc_update_bits(codec, ADC3101_NADC,
				    ADC3101_NADCEN, ADC3101_NADCEN);

		/* Switch on MADC Divider */
		snd_soc_update_bits(codec, ADC3101_MADC,
				    ADC3101_MADCEN, ADC3101_MADCEN);

		/* Switch on BCLK_N Divider */
		snd_soc_update_bits(codec, ADC3101_BCLKN,
				    ADC3101_BCLKEN, ADC3101_BCLKEN);

		break;
	case SND_SOC_BIAS_PREPARE:
		break;
	case SND_SOC_BIAS_STANDBY:
		/* Switch off PLL */
		snd_soc_update_bits(codec, ADC3101_PLLPR,
				    ADC3101_PLLEN, 0);

		/* Switch off NADC Divider */
		snd_soc_update_bits(codec, ADC3101_NADC,
				    ADC3101_NADCEN, 0);

		/* Switch off MADC Divider */
		snd_soc_update_bits(codec, ADC3101_MADC,
				    ADC3101_MADCEN, 0);

		/* Switch off BCLK_N Divider */
		snd_soc_update_bits(codec, ADC3101_BCLKN,
				    ADC3101_BCLKEN, 0);

		/* Power down Mic Bias */
		snd_soc_write(codec, ADC3101_MICBIAS, ADC3101_MICBIAS1_BIAS2_PWR_DOWN);
		break;
	case SND_SOC_BIAS_OFF:
		break;
	}

	return 0;
}

#define ADC3101_RATES	SNDRV_PCM_RATE_8000_96000
#define ADC3101_FORMATS	(SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S24_3LE \
			 | SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE)

static const struct snd_soc_dai_ops adc3101_ops = {
	.hw_params = adc3101_hw_params,
	.set_fmt = adc3101_set_dai_fmt,
	.set_sysclk = adc3101_set_dai_sysclk,
};

static struct snd_soc_dai_driver adc3101_dai[] = {
	{
		.name = "tlv320adc3101-hifi",

		.capture = {
			.stream_name = "Capture",
			.channels_min = 1,
			.channels_max = 8,
			.rates = ADC3101_RATES,
			.formats = ADC3101_FORMATS,},
		.ops = &adc3101_ops,
		.symmetric_rates = 1,
	},
};

static int adc3101_readable_register_dump(struct snd_soc_codec *codec, char *buf)
{
	uint16_t start_reg = ADC3101_PSEL;
	uint16_t end_reg = ADC3101_RMICPGAVOL;
	uint16_t reg = 0;
	int val = 0;
	int len = 0;
	int retry = 3;

	for (reg = start_reg; reg <= end_reg; reg++) {
i2c_read_retry:
		val = snd_soc_read(codec, reg);
		if (val < 0 && --retry > 0) {
			dev_warn(codec->dev, "retry reading page: %d reg:%d\n", reg / 128, reg % 128);
			msleep(1);
			goto i2c_read_retry;
		}
		retry = 3;
		len += scnprintf(buf + len, PAGE_SIZE, "page: %d, %d: 0x%02x\n", reg / 128, reg % 128, val);
		dev_dbg(codec->dev, "page: %d, %d: 0x%02x\n", reg / 128, reg % 128, val);
	}

	return len;
}

static ssize_t adc3101_readable_reg_show(struct device *dev,
    struct device_attribute *attr, char *buf)
{
	struct adc3101_priv *adc3101 = dev_get_drvdata(dev);
	int len;

	len = adc3101_readable_register_dump(adc3101->codec, buf);

	return len;
}

static ssize_t adc3101_reg_store(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	struct adc3101_priv *adc3101 = dev_get_drvdata(dev);
	int reg, val;

	sscanf(buf, "%d:%x", &reg, &val);
	if (snd_soc_write(adc3101->codec, reg, val) < 0) {
		dev_err(dev, "Fail to write reg:0x%x, val:%x \n", reg, val);
		return -EINVAL;
	}

	return count;
}

static ssize_t adc3101_reset_store(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	struct adc3101_priv *adc3101 = dev_get_drvdata(dev);
	int val;

	if (adc3101->rstn_gpio > 0) {
		sscanf(buf, "%x", &val);

		if (val == 1) {
			gpio_set_value_cansleep(adc3101->rstn_gpio, 1);
		} else if (val == 0) {
			gpio_set_value_cansleep(adc3101->rstn_gpio, 0);
		} else {
			return -EINVAL;
		}
	}
	return count;
}

static ssize_t adc3101_reset_show(struct device *dev,
    struct device_attribute *attr, char *buf)
{
	struct adc3101_priv *adc3101 = dev_get_drvdata(dev);
	int val;

	if (adc3101->rstn_gpio > 0)
		val = gpio_get_value_cansleep(adc3101->rstn_gpio);

    return sprintf(buf, "%d\n", val);
}

static DEVICE_ATTR(readable_reg, S_IRUGO, adc3101_readable_reg_show, NULL);
static DEVICE_ATTR(reg, S_IWUSR, NULL, adc3101_reg_store);
static DEVICE_ATTR(reset, S_IWUSR | S_IRUSR, adc3101_reset_show, adc3101_reset_store);

static struct attribute *adc3101_attr[] = {
	&dev_attr_readable_reg.attr,
	&dev_attr_reg.attr,
	&dev_attr_reset.attr,
	NULL
};

static const struct attribute_group adc3101_attr_group = {
	.attrs = adc3101_attr,
};

static int adc3101_codec_probe(struct snd_soc_codec *codec)
{
	struct adc3101_priv *adc3101 = snd_soc_codec_get_drvdata(codec);
	u32 tmp_reg;
	int ret;

	dev_info(codec->dev, "%s() \n", __func__);

	if (gpio_is_valid(adc3101->rstn_gpio)) {
		ndelay(10);
		gpio_set_value_cansleep(adc3101->rstn_gpio, 1);
	}

	adc3101->codec = codec;
	snd_soc_write(codec, ADC3101_RESET, 0x01);

	/*
	 * Workaround: for an unknown reason, the ADC needs to be powered up
	 * and down for the first capture to work properly. It seems related to
	 * a HW BUG or some kind of behavior not documented in the datasheet.
	 */
	tmp_reg = snd_soc_read(codec, ADC3101_ADCSETUP);
	snd_soc_write(codec, ADC3101_ADCSETUP, tmp_reg |
				ADC3101_LADC_EN | ADC3101_RADC_EN);
	snd_soc_write(codec, ADC3101_ADCSETUP, tmp_reg);

	ret = sysfs_create_group(&codec->dev->kobj, &adc3101_attr_group);
	if (ret) {
		dev_err(codec->dev, "Failure %d creating sysfs group\n",
			ret);
		sysfs_remove_group(&codec->dev->kobj, &adc3101_attr_group);
	}

	return 0;
}

static int adc3101_remove(struct snd_soc_codec *codec)
{
	adc3101_set_bias_level(codec, SND_SOC_BIAS_OFF);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_adc3101 = {
	.probe = adc3101_codec_probe,
	.remove = adc3101_remove,
	.set_bias_level = adc3101_set_bias_level,

	.component_driver = {
		.controls = adc3101_snd_controls,
		.num_controls = ARRAY_SIZE(adc3101_snd_controls),
		.dapm_widgets = adc3101_dapm_widgets,
		.num_dapm_widgets = ARRAY_SIZE(adc3101_dapm_widgets),
		.dapm_routes = adc3101_dapm_routes,
		.num_dapm_routes = ARRAY_SIZE(adc3101_dapm_routes),
	}
};

static int adc3101_i2c_probe(struct i2c_client *i2c,
			     const struct i2c_device_id *id)
{
	struct adc3101_priv *adc3101 = NULL;
	struct device *dev = &i2c->dev;
	enum of_gpio_flags flags;
	int ret = 0;

	dev_info(&i2c->dev, "%s() \n", __func__);

	adc3101 = devm_kzalloc(&i2c->dev, sizeof(struct adc3101_priv),
			GFP_KERNEL);
	if (adc3101 == NULL)
		return -ENOMEM;

	adc3101->regmap = devm_regmap_init_i2c(i2c, &adc3101_i2c_regmap);
	if (IS_ERR(adc3101->regmap)) {
		dev_err(&i2c->dev, "%s failed devm_regmap_init_i2c\n", __func__);
		return PTR_ERR(adc3101->regmap);
	}
	i2c_set_clientdata(i2c, adc3101);
	dev_set_drvdata(dev, adc3101);

	adc3101->power_cfg = 0;
	adc3101->micpga_routing = 0;
	adc3101->rstn_gpio = -1;

	ret = of_get_named_gpio_flags(i2c->dev.of_node, "reset-gpios", 0, &flags);
	if (ret > 0)
		adc3101->rstn_gpio = ret;

	if (gpio_is_valid(adc3101->rstn_gpio)) {
		ret = devm_gpio_request_one(&i2c->dev,
					    adc3101->rstn_gpio,
						flags & OF_GPIO_ACTIVE_LOW ?
						GPIOF_OUT_INIT_LOW : GPIOF_OUT_INIT_HIGH,
					    "adc3101-reset-pin");
		if (ret < 0) {
			dev_err(&i2c->dev, "unable to acquire gpio\n");
			return ret;
		}
	}
	ret = of_property_read_u32(i2c->dev.of_node, "mic_differential",
		&adc3101->mic_differential);
	if (ret) {
		dev_warn(&i2c->dev, "failed to get mic_differential, set it default\n");
		adc3101->mic_differential = 0;
		ret = 0;
	}

	/*
	 * 0: internal digital mic
	 * 1: internal analog mic (omni)
	 * 2: external analog mic (directional)
	 */
	if (of_property_read_u32(i2c->dev.of_node, "mic-type", &adc3101->mic_type)) {
		dev_warn(&i2c->dev, "failed to get mic type, set it default internal dmic\n");
		adc3101->mic_type = 0;
	}

	ret = snd_soc_register_codec(&i2c->dev,
				&soc_codec_dev_adc3101, adc3101_dai, 1);

	return ret;
}

static int adc3101_i2c_remove(struct i2c_client *client)
{
	struct device *dev = &client->dev;

	dev_info(dev, "%s() \n", __func__);

	sysfs_remove_group(&dev->kobj, &adc3101_attr_group);

	snd_soc_unregister_codec(dev);

	return 0;
}

static const struct of_device_id tlv320adc3101_of_match[] = {
	{.compatible = "ti,tlv320adc3101"},
	{},
};
MODULE_DEVICE_TABLE(of, tlv320adc3101_of_match);

static const struct i2c_device_id adc3101_i2c_id[] = {
	{ "tlv320adc3101", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, adc3101_i2c_id);

static struct i2c_driver adc3101_i2c_driver = {
	.driver = {
		.name = "tlv320adc3101",
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(tlv320adc3101_of_match),
	},
	.probe =    adc3101_i2c_probe,
	.remove =   adc3101_i2c_remove,
	.id_table = adc3101_i2c_id,
};

module_i2c_driver(adc3101_i2c_driver);

MODULE_DESCRIPTION("ASoC tlv320adc3101 codec driver");
MODULE_AUTHOR("Louisyw Lu <Louisyw.Lu@garmin.com>");
MODULE_LICENSE("GPL");
