/* 
 *  Copyright 2009
 *  Texas Instruments Incorporated
 *
 *  All rights reserved.  Property of Texas Instruments Incorporated
 *  Restricted rights to use, duplicate or disclose this code are
 *  granted through contract.
 * 
 */
 /*
  *     File            : ti_load_app.c
  *     Date            : 10th Dec 2009
  *     Author          : Jaz John
  */

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <stdarg.h>

#include "ti_load_app.h"

#define DEBUG
#undef DEBUG

/*
 * power functionality
 */
int power(int base, int n)
{
	int i, p;

	if (n == 0)
		return 1;

	p = 1;
	for (i = 1; i <= n; ++i) {
		p = p * base;
	}

	return p;
}

/*
 * hexadecimal to integer
 */
int htoi(char *s)
{
	int len, value = 1, digit = 0, total = 0;
	int c, x, y, i = 0;
	char hexchars[] = "abcdef";	/* string to find hex digit values */

	/* Test for 0s, '0x', or '0X' at the beginning and move on */

	if (s[i] == '0') {
		i++;
		if (s[i] == 'x' || s[i] == 'X')
			i++;
	}

	len = strlen(s);

	for (x = i; x < len; x++) {
		c = tolower(s[x]);
		if (c >= '0' && c <= '9') {
			digit = c - '0';
		} else if (c >= 'a' && c <= 'f') {
			for (y = 0; hexchars[y] != '\0'; y++) {
				if (c == hexchars[y])
					digit = y + 10;
			}
		} else
			return 0;	/* Return 0 if we get something unexpected */

		value = power(16, len - x - 1);
		total += value * digit;
	}
	return total;
}

/*
 * split a line into tokens
 */
static int split_line(char *buf, char **token, int max_token)
{
	char *dst;
	int n, esc, quote;

	for (n = 0; n < max_token; n++) {
		while (isspace(*buf))
			buf++;
		if (!*buf || *buf == '\n')
			return n;
		/* skip comments */
		if (*buf == '#' || *buf == '!')
			return n;
		esc = 0;
		quote = 0;
		token[n] = buf;
		for (dst = buf; *buf && *buf != '\n'; buf++) {
			if (esc)
				esc = 0;
			else if (isspace(*buf) && !quote) {
				buf++;
				break;
			} else if (*buf == '\\') {
				esc = 1;
				continue;
			} else if (*buf == '\'' || *buf == '"') {
				if (!quote) {
					quote = *buf;
					continue;
				} else if (*buf == quote) {
					quote = 0;
					continue;
				}
			}
			*dst++ = *buf;
		}
		*dst = 0;
	}
	return n;
}

/* 
 * There may be instance where the same register may be programmed by the
 * default ASoC driver and when programming the registers in the .cfg 
 * configuration file. The register number in the below array indicates which
 * registers in the .cfg file will be written. Others will be ignored
 */

/* Allowed registers for page '0' */
static unsigned char page_0_allow_regs[] = { 21, 22, 27, 61, 62, 83, 84, 255 };

/* Allowed registers for page '1' */
static unsigned char page_1_allow_regs[] = { };

/*
 *----------------------------------------------------------------------------
 * Function : check_allow_list
 *
 * Purpose  : Check and return if the register is allowed to write.
 *		      Allowed registers are store in an arry per page. The register
 *            and page address passed are checked against this list and 
 *            return whether to ignore or not 
 *----------------------------------------------------------------------------
 */
int check_allow_list(unsigned char page, unsigned char reg_addr)
{
	int i;

	if (page == 0) {
		for (i = 0; i < sizeof(page_0_allow_regs); i++) {
			if (page_0_allow_regs[i] == reg_addr)
				/* allow register write */
				return (CODEC_REG_ALLOW);
		}
		/* not in allow list, ignore the register */
		return (CODEC_REG_IGNORE);
	} else if (page == 1) {
		for (i = 0; i < sizeof(page_1_allow_regs); i++) {
			if (page_1_allow_regs[i] == reg_addr)
				/* allow register write */
				return (CODEC_REG_ALLOW);
		}
		/* not in allow list, ignore the register */
		return (CODEC_REG_IGNORE);
	} else {
		/* Write to pages other than 0,1 are allowed */
		return (CODEC_REG_ALLOW);
	}
}

static char current_page = 0;

int program_minidsp_device(int fd, char *data, int count)
{
//      static char current_page = 0;
	char reg_addr;
	size_t ret;
	int i, data_index;
	unsigned char buf[2];

	/* check if page is changed */
	if (data[0] == 0x0) {
		/* update current page */
		current_page = data[1];

		buf[0] = 0;
		buf[1] = current_page;

		/* Program the page number */
		if (write(fd, buf, 2) != 2) {
			printf("TILoad: failed changing page to 0x%x\n",
			       current_page);
			return -1;
		}
		/* if no more data to write in this sequence, return */
		if (count == 2) {
			return 0;
		}

		/* we have more data to write, after 'page' register */
		if ((current_page != 0) && (current_page != 1)) {
			/* program the registers without checking, and return */
			ret = write(fd, data, count);
			if (ret != count) {
				printf("TILoad: write failed on page 0x%x\n",
				       current_page);
				return -1;
			} else {
#ifdef DEBUG
				printf("TILoad: written page 0x%x size = %d\n",
				       current_page, count);
#endif
				return 0;
			}
		} else {	/* page 0 or page 1, needs checking */
			/*  setup start reg and value for checking allow list */
			data_index = 2;	/* Index of next data */
			reg_addr = 1;	/* next register address */
			count -= data_index;	/* decr count, page is written */
		}

	} else {		/* write operation on current page */
		if ((current_page != 0) && (current_page != 1)) {
			/* program the registers without checking, and return */
			ret = write(fd, data, count);
			if (ret != count) {
				printf("TILoad: write failed on page 0x%x\n",
				       current_page);
				return -1;
			} else {
#ifdef DEBUG
				printf("TILoad:page 0x%x size = %d\n",
				       current_page, count);
#endif
				return 0;
			}
		} else {	/* page 0 or page 1, needs checking */
			/*  setup start reg and value for checking allow list */
			reg_addr = data[0];	/* start register address */
			data_index = 1;	/* Index of next data */
			count -= 1;	/* count of data bytes */
		}
	}

	/********** page 0 and 1 registers programming section *************/

	/*  check for accept list of registers 
	 *  Only listed registers are allowed to write in
	 *  page '0' and page '1'
	 */
	for (i = 0; i < count; i++) {
		if (check_allow_list(current_page, reg_addr)
		    == CODEC_REG_ALLOW) {
			/* write register */
			buf[0] = reg_addr;
			buf[1] = data[data_index];

			ret = write(fd, buf, 2);
			if (ret != 2) {
				printf("TILoad: write failed, page 0x%x\n",
				       current_page);
				return -1;
			}
#ifdef DEBUG
			printf("TILOAD....page 0x%x, [%d] = 0x%x\n",
			       current_page, buf[0], buf[1]);
#endif
		} else {
			/*  else register is ignored */
		}

		/* Move to next register and data */
		reg_addr++;
		data_index++;
	}
	return 0;
}

int compare_register_flags(char test_val, char *test_pattern)
{
	int k, test_flag;

	for (k = 0; k < 8; k++) {
		/* check for the dont care bits */
		switch (test_pattern[k]) {
		case '0':
			/* check if bit is zero */
			if ((test_val & (1 << k)) == 0) {
				test_flag = 1;
			} else {
				test_flag = 0;
			}
			break;

		case '1':
			/* check if bit is set */
			if ((test_val & (1 << k))) {
				test_flag = 1;
			} else {
				test_flag = 0;
			}
			break;
			/* case: dont care */
		default:
			/* dont care, go to next bit */
			break;
		}
		/*  if mismatch already, break out and return */
		if (test_flag == 0) {
			break;
		}
	}
	return test_flag;
}

/* If File operations ie. --file option */
int load_pps_config_file(int fd, const char *file_name)
{
	FILE *fp;
	int n, i, j, k;
	size_t count = 0, rd_count = 0;
	unsigned char buf[256];
	char *args[32];
	char ch;
	unsigned char read_buf[32];
	char test_flag, test_val, read_reg;
	char test_pattern[9];
	int read_size;
	int read_addr;
	static unsigned char data[MINIDSP_PAGE_SIZE + 1];
	/* Flag for indicating burst accumulation */
	static burst_flag = 0;

	fp = fopen(optarg, "r");
	if (fp == NULL) {
		printf("Error in openning the file %s\n", optarg);
		return -1;
	}

	/* process one line at a time */
	while (fgets(buf, sizeof(buf), fp)) {

		n = split_line(buf, args, 32);

		/* neglect comment/blank lines */
		if (n == 0)
			continue;

		sprintf(&ch, "%c", *args[0]);

		switch (ch) {

			/* CASE: write command */
		case 'w':
			/* write the buffer accumulated previously */
			if (count != 0) {
				program_minidsp_device(fd, data, count);
			}

			count = 0;
			for (i = 0; i < (n - 2); i++) {
				data[i] = htoi(args[i + 2]);
				count++;
			}
			break;

			/* CASE: next register value */
		case '>':
			/* continue buffering the data */
			for (i = 0; i < (n - 1); i++) {
				data[count + i] = htoi(args[i + 1]);
			}
			count += (n - 1);
			break;

			/* CASE: Read I2C registers */
		case 'r':
#if 1
			/* write the buffer accumulated previously */
			if (count != 0) {
				program_minidsp_device(fd, data, count);
			}
			count = 0;
#endif
			/* read register address */
			read_addr = htoi(args[1]);
			/* read amount in bytes */
			read_size = htoi(args[2]);

			printf("TiLoad:read reg=%d, size=%d\n", read_addr,
			       read_size);

			if (read_size > 32) {
				printf
				    (" can not read %d value, Maximum read size is 32\n",
				     read_size);
				return -1;
			}
			/* send read register address */
			read_buf[0] = read_addr;

			rd_count = read(fd, read_buf, read_size);
			if (rd_count != read_size) {
				printf("TiLoad: read failed, read %d\n",
				       rd_count);
				return -1;
			}
			printf
			    ("TiLoad: read, page %d start register addr = %d\n",
			     current_page, read_addr);

			for (j = 0; j < read_size; j++) {
				printf("0x%x,", read_buf[j]);
			}
			printf("\n");
			break;
		case 'd':
			usleep((atoi(args[1])) * 1000);
			break;

			/* case f: check and wait for the flag to be set */
		case 'f':
			read_reg = htoi(args[2]);
			printf("wait for flag: reg=0x%x, flag=%s\n", read_reg,
			       args[3]);

			if (strlen(args[3]) < 8) {
				printf
				    ("Insufficient pattern length to check\n");
				return -1;
			}
			/* Extract the individual bit values */
			for (k = 0; k < 8; k++) {
				test_pattern[7 - k] = args[3][k];
			}

			printf("Pattern: ");
			/* Extract the individual bit values */
			for (k = 0; k < 8; k++) {
				printf("%c,", test_pattern[k]);
			}
			printf("\n");

			/* wait for flag in a loop, repeatative read and check */
			do {
				read_buf[0] = read_reg;

				rd_count = read(fd, read_buf, 1);
				if (rd_count != 1) {
					printf("TiLoad: read failed, read %d\n",
					       rd_count);
					return -1;
				}
				test_val = read_buf[0];
				printf("read 0x%x ", test_val);

				test_flag =
				    compare_register_flags(test_val,
							   test_pattern);
				/* all pattern matched, stop waiting and return back */
				if (test_flag) {
					break;	/*  out of do--while loop */
				}
			} while (1);
			printf("Flag is set\n");
			break;	/* switch break; */

		default:
			printf("Command %c not supported\n", ch);
		}
	}

	/* write the buffer accumulated before EOF */
	if (count != 0) {
		program_minidsp_device(fd, data, count);
	}

	/* change page back to '0' before exit */
	data[0] = 0;
	data[1] = 0;

	write(fd, data, 2);

	fclose(fp);

	return 0;
}

int write_minidsp_reg(int fd, char *arg_string, int bytes)
{
	unsigned char data[4];
	int user_value;
	int ret;

	user_value = htoi(arg_string);

	switch (bytes) {
		/* 8 bit reg addr + 8 bit data */
	case TILOAD_DATA_SIZE_16:
		/* MSB is register address */
		data[0] = (user_value >> 8) & 0xFF;
		/* data, 1 byte */
		data[1] = (user_value) & 0xFF;
		break;

		/* 8 bit reg addr + 16 bit data */
	case TILOAD_DATA_SIZE_24:
		/* MSB is register address */
		data[0] = (user_value >> 16) & 0xFF;
		/* data, 2 bytes */
		data[1] = (user_value >> 8) & 0xFF;
		data[2] = (user_value) & 0xFF;
		break;

		/* 8 bit reg addr + 24 bit data */
	case TILOAD_DATA_SIZE_32:
		/* MSB is register address */
		data[0] = (user_value >> 24) & 0xFF;
		/* data, 3 bytes */
		data[1] = (user_value >> 16) & 0xFF;
		data[2] = (user_value >> 8) & 0xFF;
		data[3] = (user_value) & 0xFF;
		break;

	default:
		printf("Unknown data size\n");
		return -1;
	}

	ret = write(fd, data, bytes + 1);

	if (ret != bytes + 1) {
		printf("%s: write %d bytes failed\n", MINIDSP_DEVICE_NODE,
		       bytes + 1);
		return -1;
	}
	return 0;
}

int read_minidsp_reg(int fd, char *arg_string, int bytes)
{
	unsigned char data[4];
	int user_value;
	unsigned int reg_val;

	user_value = htoi(arg_string);

	data[0] = user_value;

	/* send read register address */
	read(fd, data, bytes);

	switch (bytes) {
	case TILOAD_DATA_SIZE_16:
		reg_val = data[0];
		break;

	case TILOAD_DATA_SIZE_24:
		reg_val = data[0];
		reg_val <<= 8;
		reg_val |= data[1];
		break;

	case TILOAD_DATA_SIZE_32:
		reg_val = data[0];
		reg_val <<= 8;
		reg_val |= data[1];
		reg_val <<= 8;
		reg_val |= data[2];
		break;

	default:
		printf("Unknown data size\n");
		return -1;
	}

	return reg_val;
}

static struct option long_option[] = {
	{"help", no_argument, NULL, 'h'},
	{"file", required_argument, NULL, 'f'},
	{"RM16", required_argument, NULL, 'b'},
	{"RM24", required_argument, NULL, 'w'},
	{"RM32", required_argument, NULL, 't'},
	{"RR16", required_argument, NULL, 'm'},
	{"RR24", required_argument, NULL, 'n'},
	{"RR32", required_argument, NULL, 'o'},
	{"MS", required_argument, NULL, 's'},
	{"MG", no_argument, NULL, 'g'},
	{NULL, 0, NULL, 0},
};

static int help(void)
{
	printf("  -h, --help this help\n");
	printf("  -f, --file cfg file to be opened\n");
	printf("  -b, --RM16 16bit_val  write one byte \n");
	printf("  -w, --RM24 24bit_val  write word(2 bytes) \n");
	printf("  -t, --RM32 32bit_val  write 3 bytes\n");
	printf("  -m, --RR16 16bit_val  read one byte \n");
	printf("  -n, --RR24 24bit_val  read word(2 bytes) \n");
	printf("  -o, --RR32 32bit_val  read 3 bytes\n");
	printf("  -s, --MS Set 4byte magic number\n");
	printf("  -g, --MG Get 4byte magic number\n");
	return 0;
}

int main(int argc, char *argv[])
{
	int ret = 0;
	int c;
	int fd;
	unsigned int magic_num;

	if (argc <= 1) {
		help();
		return -1;
	}

	fd = open(MINIDSP_DEVICE_NODE, O_RDWR, 0);
	if (fd < 0) {
		printf("Can not open the device %s\n", MINIDSP_DEVICE_NODE);
		return -1;
	}

	while (1) {
		c = getopt_long(argc, argv, "hf:b:w:t:m:n:o:s:g", long_option,
				NULL);

		/* check for the end of options */
		if (c == -1) {
			break;
		}
		//printf ( "%s: found option %c, arg %s\n", __FILE__, c, optarg );
		switch (c) {
			/* Case '-help': show help for the TILoad command */
		case 'h':
			help();
			break;
			/* Case '--file': input method is a PPS config file */
		case 'f':
			ret = load_pps_config_file(fd, optarg);
			break;

		case 'b':
			ret =
			    write_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_16);
			break;

		case 'w':
			ret =
			    write_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_24);
			break;

		case 't':
			ret =
			    write_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_32);
			break;
		case 'm':
			ret = read_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_16);
			printf("register value = 0x%x\n", ret);
			break;

		case 'n':
			ret = read_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_24);
			printf("register value = 0x%x\n", ret);
			break;

		case 'o':
			ret = read_minidsp_reg(fd, optarg, TILOAD_DATA_SIZE_32);
			printf("register value = 0x%x\n", ret);
			break;

		case 's':
			magic_num = htoi(optarg);
			ret = ioctl(fd, ADC3101_IOMAGICNUM_SET, &magic_num);
			break;

		case 'g':
			ret = ioctl(fd, ADC3101_IOMAGICNUM_GET, &magic_num);
			printf("Magic Number = 0x%x\n", magic_num);
			break;
		default:
			printf("%s: Unknown command\n", __FILE__);
			help();
			break;
		}
	}
	/* close the mini dsp device */
	close(fd);
	return ret;
}
