/*
 * omap3evm.c - SoC audio for OMAP3530 EVM.
 * 
 * Author: Sandeep S Prabhu <sandeepsp@mistralsolutions.com>
 * Copyright (C) 2009 Mistral Solutions Pvt Ltd.
 * 
 * Based on sound/soc/omap/omap3beagle.c by Steve Sakoman
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/soundcard.h>
#include <linux/clk.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/interrupt.h>

#include <mach/gpio.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>

#include <asm/mach-types.h>
#include <mach/mux.h>
#include <mach/io.h>
#include <asm/io.h>
#include <linux/delay.h>

#include "omap-pcm.h"
#include "omap-mcbsp.h"
#include "../codecs/tlv320adc3101.h"

#define ADC3101_RESET_GPIO	158
#define ADC3101_INTR_GPIO	10

/* Select BCLK & WCLK clock inputs for McBSP1 */
#define CONFIG_FSX_CLKX_PINS	1
/* #define CONFIG_FSR_CLKR_PINS	1 */

static struct clk *sys_clkout2;
static struct clk *clkout2_src_ck;
static struct clk *cm_96m_ck;

static int omap_linein_func = 1;
static int omap_micin_func = 0;
static int omap_digimicin_func = 0;

static void omap_ext_control(struct snd_soc_codec *codec)
{
	if (omap_linein_func)
		snd_soc_dapm_enable_pin(codec, "Line Input");
	else
		snd_soc_dapm_disable_pin(codec, "Line Input");

	if (omap_micin_func)
		snd_soc_dapm_enable_pin(codec, "Mic Input");
	else
		snd_soc_dapm_disable_pin(codec, "Mic Input");

	if (omap_digimicin_func)
		snd_soc_dapm_enable_pin(codec, "Digital Mic Input");
	else
		snd_soc_dapm_disable_pin(codec, "Digital Mic Input");

	snd_soc_dapm_sync(codec);
}

static int omap3evm_hw_params(struct snd_pcm_substream *substream,
			      struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->dai->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->dai->cpu_dai;
	int ret;

	/* Set codec DAI configuration */
	ret = snd_soc_dai_set_fmt(codec_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec DAI configuration\n");
		return ret;
	}

	/* Set cpu DAI configuration */
	ret = snd_soc_dai_set_fmt(cpu_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set cpu DAI configuration\n");
		return ret;
	}

	/* Set the codec system clock 12MHz */
	ret = snd_soc_dai_set_sysclk(codec_dai, 0, 12000000, SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec system clock\n");
		return ret;
	}
#ifdef CONFIG_FSR_CLKR_PINS
	/*
	 * CLKR and FSR inputs of McBSP1 are derived from CLKX and FSX pins
	 */
	ret = snd_soc_dai_set_sysclk(cpu_dai, OMAP_MCBSP_CLKR_SRC_CLKR, 0,
				     SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set MCBSP1 CLKR receiver pin\n");
		return ret;
	}

	ret = snd_soc_dai_set_sysclk(cpu_dai, OMAP_MCBSP_FSR_SRC_FSR, 0,
				     SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set MCBSP1 FSR receiver pin\n");
		return ret;
	}
#else /*CONFIG_FSX_CLKX_PINS */
	/*
	 * CLKR and FSR inputs of McBSP1 are derived from CLKX and FSX pins
	 */
	ret = snd_soc_dai_set_sysclk(cpu_dai, OMAP_MCBSP_CLKR_SRC_CLKX, 0,
				     SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set McBSP1 CLKX receiver pin\n");
		return ret;
	}

	ret = snd_soc_dai_set_sysclk(cpu_dai, OMAP_MCBSP_FSR_SRC_FSX, 0,
				     SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set MCBSP1 FSX receiver pin\n");
		return ret;
	}
#endif

	return 0;
}

static int omap3evm_startup(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_codec *codec = rtd->socdev->codec;
	omap_ext_control(codec);
	return clk_enable(sys_clkout2);
}

static void omap3evm_shutdown(struct snd_pcm_substream *substream)
{
	clk_disable(sys_clkout2);
}

static struct snd_soc_ops omap3evm_ops = {
	.hw_params = omap3evm_hw_params,
	.startup = omap3evm_startup,
	.shutdown = omap3evm_shutdown,
};

static int omap_get_linein(struct snd_kcontrol *kcontrol,
			   struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = omap_linein_func;

	return 0;
}

static int omap_set_linein(struct snd_kcontrol *kcontrol,
			   struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);

	if (omap_linein_func == ucontrol->value.integer.value[0])
		return 0;

	omap_linein_func = ucontrol->value.integer.value[0];
	omap_ext_control(codec);

	return 1;
}

static int omap_get_micin(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = omap_micin_func;

	return 0;
}

static int omap_set_micin(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);

	if (omap_micin_func == ucontrol->value.integer.value[0])
		return 0;

	omap_micin_func = ucontrol->value.integer.value[0];
	omap_ext_control(codec);

	return 1;
}

static int omap_get_digimicin(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = omap_digimicin_func;

	return 0;
}

static int omap_set_digimicin(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);

	if (omap_digimicin_func == ucontrol->value.integer.value[0])
		return 0;

	omap_digimicin_func = ucontrol->value.integer.value[0];
	omap_ext_control(codec);

	return 1;
}

static const struct snd_soc_dapm_widget omap_dapm_widgets[] = {
	/* Line Input */
	SND_SOC_DAPM_MIC("Line Input", NULL),
	/* Mic Input */
	SND_SOC_DAPM_MIC("Mic Input", NULL),
	/* Digital Mic Input */
	SND_SOC_DAPM_MIC("Digital Mic Input", NULL),
};

static const struct snd_soc_dapm_route audio_map[] = {
/* Line Input to codec dapm Inputs */
	{"IN1_L", NULL, "Line Input"},
	{"IN1_R", NULL, "Line Input"},

/* Mic Input to codec dapm Inputs */
	{"IN1_L", NULL, "Mic Input"},
	{"IN1_R", NULL, "Mic Input"},

/* Digital Mic Input to codec dapm Inputs */
	{"DMic_L", NULL, "Digital Mic Input"},
	{"DMic_R", NULL, "Digital Mic Input"},
};

static const char *linein_function[] = { "Off", "On" };
static const char *micin_function[] = { "Off", "On" };
static const char *digimicin_function[] = { "Off", "On" };

static const struct soc_enum omap_enum[] = {
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(linein_function), linein_function),
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(micin_function), micin_function),
	SOC_ENUM_SINGLE_EXT(ARRAY_SIZE(digimicin_function), digimicin_function),
};

static const struct snd_kcontrol_new adc3101_omap_controls[] = {
/* Line In Jack control */
	SOC_ENUM_EXT("Line In Jack", omap_enum[0],
		     omap_get_linein, omap_set_linein),
/* Mic In Jack control */
	SOC_ENUM_EXT("MIC In Jack", omap_enum[1],
		     omap_get_micin, omap_set_micin),
/* Digital Mic In control */
	SOC_ENUM_EXT("Digital MIC In", omap_enum[2],
		     omap_get_digimicin, omap_set_digimicin),
};

static int omap_adc3101_init(struct snd_soc_codec *codec)
{
	int i, err;

	/* Add adc3101 specific controls */
	for (i = 0; i < ARRAY_SIZE(adc3101_omap_controls); i++) {
		err = snd_ctl_add(codec->card,
				  snd_soc_cnew(&adc3101_omap_controls[i], codec,
					       NULL));
		if (err < 0)
			return err;
	}

	/* Add adc3101 specific widgets */
	snd_soc_dapm_new_controls(codec, omap_dapm_widgets,
				  ARRAY_SIZE(omap_dapm_widgets));

	/* Set up adc3101 specific audio path audio_map */
	snd_soc_dapm_add_routes(codec, audio_map, ARRAY_SIZE(audio_map));

	snd_soc_dapm_sync(codec);

	return 0;
}

/* Digital audio interface glue - connects codec <--> CPU */
static struct snd_soc_dai_link omap3evm_dai = {
	.name = "TLV320ADC3101",
	.stream_name = "ADC3101",
	.cpu_dai = &omap_mcbsp_dai[0],
	.codec_dai = &adc3101_dai,
	.init = omap_adc3101_init,
	.ops = &omap3evm_ops,
};

/* Audio machine driver */
static struct snd_soc_machine snd_soc_machine_omap3evm = {
	.name = "omap3evm",
	.dai_link = &omap3evm_dai,
	.num_links = 1,
};

/* Audio subsystem */
static struct snd_soc_device omap3evm_snd_devdata = {
	.machine = &snd_soc_machine_omap3evm,
	.platform = &omap_soc_platform,
	.codec_dev = &soc_codec_dev_adc3101,
};

static struct platform_device *omap3evm_snd_device;

static int __init omap3evm_adc3101_init(void)
{
	int ret = 0;
	struct device *dev;

	printk("machine driver initialization\n");

	/* Pin Mux for gpio_158 */
	omap_cfg_reg(V21_3430_MCBSP1_DX);

	/* Issue hardware reset to adc3101 */
	gpio_request(ADC3101_RESET_GPIO, "adc3101-reset");
	gpio_direction_output(ADC3101_RESET_GPIO, 0);
	udelay(100);
	gpio_direction_output(ADC3101_RESET_GPIO, 1);

	omap3evm_snd_device = platform_device_alloc("soc-audio", -1);
	if (!omap3evm_snd_device)
		return -ENOMEM;

	platform_set_drvdata(omap3evm_snd_device, &omap3evm_snd_devdata);
	omap3evm_snd_devdata.dev = &omap3evm_snd_device->dev;
	/* We have MCBSP1 interface */
	*(unsigned int *)omap3evm_dai.cpu_dai->private_data = 0;	/* McBSP1 */

	ret = platform_device_add(omap3evm_snd_device);

	if (ret) {
		printk(KERN_ERR "Unable to add platform device\n");
		platform_device_put(omap3evm_snd_device);
		return ret;
	}

	dev = &omap3evm_snd_device->dev;

	sys_clkout2 = clk_get(dev, "sys_clkout2");
	if (IS_ERR(sys_clkout2)) {
		printk("Could not get sys_clkout2\n");
		return -1;
	}

	sys_clkout2 = clk_get(dev, "sys_clkout2");
	if (IS_ERR(sys_clkout2)) {
		printk("Could not get sys_clkout2\n");
		return -1;
	}
	/*
	 * Configure 12 MHz output on SYS_CLKOUT2. Therefore we must use
	 * 96 MHz as its parent in order to get 12 MHz
	 */
	cm_96m_ck = clk_get(dev, "cm_96m_fck");
	if (IS_ERR(cm_96m_ck)) {
		printk("Could not get func 96M clock\n");
		ret = -1;
		goto err1;
	}
	clk_set_parent(clkout2_src_ck, cm_96m_ck);
	clk_set_rate(sys_clkout2, 12000000);

	/* Hack for getting SYS_CLKOUT2 */
	omap_writel(0x1a, 0x48004d70);

	/* Pin Muxing for McBSP1, GPIOs and clock */
#ifdef CONFIG_FSR_CLKR_PINS
	/*
	 * CLKR and FSR inputs are derived from CLKR and FSR pins
	 */
	omap_cfg_reg(Y21_3430_MCBSP1_CLKR);
	omap_cfg_reg(AA21_3430_MCBSP1_FSR);
#else /*CONFIG_FSX_CLKX_PINS */
	/*
	 * CLKR and FSR inputs are derived from CLKX and FSX pins
	 */
	omap_cfg_reg(W21_3430_MCBSP1_CLKX);
	omap_cfg_reg(K26_3430_MCBSP1_FSX);
#endif
	omap_cfg_reg(U21_3430_MCBSP1_DR);
	omap_cfg_reg(AE22_3430_SYS_CLKOUT2);

	/* apply MCLK to adc3101 */
	clk_enable(sys_clkout2);

	return 0;

err1:
	clk_put(sys_clkout2);

	return ret;
}

module_init(omap3evm_adc3101_init);

static void __exit omap3evm_adc3101_exit(void)
{
	printk("machine driver exit\n");
	platform_device_unregister(omap3evm_snd_device);
}

module_exit(omap3evm_adc3101_exit);

MODULE_AUTHOR("Sandeep S Prabhu <sandeepsp@mistralsolutions.com>");
MODULE_DESCRIPTION("ALSA SoC OMAP3530 EVM");
MODULE_LICENSE("GPL");
