/*
 * picopass.c
 *
 *  Created on: Sep 12, 2014
 *      Author: a0867749
 *
 *      {PicoPass Specific Functions}
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "picopass.h"
#include "trf7970.h"

//===============================================================

extern u08_t buf[300];
extern u08_t i_reg;
extern volatile u08_t irq_flag;
extern s08_t rxtx_state;
extern u08_t rx_error_flag;
extern u08_t stand_alone_flag;
extern u08_t remote_flag;

u08_t ASNB[8];
//===============================================================
// NAME: void PicoPass_ACS(void)
//
// BRIEF: Is used to detect PicoPass 32K(S) tags in stand alone
// mode.
//
// INPUTS:
//
// OUTPUTS:
//
// PROCESS:	[1] turn on RF driver
//			[2] do a complete ACS sequence (ACTALL, IDENTIFY and SELECT) to retrieve
//			[3] turn off RF driver
//
// NOTE: If PicoPass tag is present and identified, ISO15693, ISO14443A and ISO14443B LEDs will illuminate.
//
// CHANGE:
// DATE  			WHO	DETAIL
// Sept.12, 2014	JDW	Original Code
//===============================================================

void PicoPass_ACS(void) {

	Trf7970WriteIsoControl(0x86); //no RX CRC, ISO15693 high tag data rate, FSK, 1 of 4 data coding

	// The VCD should wait at least 1mSec after it activating the
	// powering field before sending the first request, to
	// ensure that the VICCs are ready to receive it. (ISO15693)
	McuDelayMillisecond(20);

	// detect a PicoPass tag in EAS mode,
	// then do complete ACS sequence (ACTALL, IDENTIFY and SELECT) to retrieve serial # and select tag,
	// then read single block and then 4 blocks of data
	PicoPass_DETECT();						//for EAS mode
	PicoPass_ACTALL();						//ACS step 1 (Activation)
	PicoPass_IDENTIFY();					//ACS step 2 (anti-Collision)
	PicoPass_SELECT();						//ACS step 3 (Selection)
	PicoPass_READ();						//read single block
	PicoPass_READ4();						//read four blocks
	//PicoPass_HALT();

}

//===============================================================
// NAME: void PicoPass_ACTALL(void)
//
// BRIEF: 	Is used to issue PicoPass ACTALL command: (Command Code A)
//			Function: First step in retrieving serial #, response is SOF
//
// INPUTS:	Command Code A, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  1   |  0   |  1   |  0   |
//---------------------------------------------------------
//|             A             |             A             |
//---------------------------------------------------------
//
// OUTPUTS:	SOF
//
// PROCESS:	[1] Send Command Code 0xAA
//			[2] Get SOF back from tag (ACK)
// CHANGE:
// DATE  		WHO	DETAIL
// 09/12/2014	JDW Original Code
//===============================================================

	void PicoPass_ACTALL(void) {

	buf[0] = 0x8F;					// reset FIFO
	buf[1] = 0x90;					// sending without CRC
	buf[2] = 0x3D;					// write continuous from 1D
	buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
	buf[4] = 0x10;					// lower and broken nibbles of transmit byte length
	buf[5] = 0xAA;					// ACTALL

	McuCounterSet();				// TimerA set
	COUNT_VALUE = COUNT_1ms * 30;	// 30ms
	IRQ_CLR;						// PORT2 interrupt flag clear (inside MSP430)
	IRQ_ON;

	Trf7970RawWrite(&buf[0], 6);	//issuing the ACTALL command

}

//===============================================================
// NAME: void PicoPass_IDENTIFY(void)
//
// BRIEF: 	Is used to issue PicoPass IDENTIFY command: (Command Code C)
//			Function: Second step in retrieving serial #, response is ASNB + CRC
//
// INPUTS:	Command Code C, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  1   |  1   |  0   |  0   |
//---------------------------------------------------------
//|             A             |             C             |
//---------------------------------------------------------
//
// OUTPUTS:	ASNB + CRC
//
// PROCESS:	[1] Send Command Code 0xAC
//			[2] Get ASNB + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/12/2014	JDW Original Code
//===============================================================
	void PicoPass_IDENTIFY(void) {
	u08_t i = 1, found = 0;
	u16_t k;
	McuDelayMillisecond(2); 		//delay

	buf[0] = 0x8F;					// reset FIFO
	buf[1] = 0x90;					// sending without CRC
	buf[2] = 0x3D;					// write continuous from 1D
	buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
	buf[4] = 0x10;					// lower and broken nibbles of transmit byte length
	buf[5] = 0xAC;					// IDENTIFY

	McuCounterSet();				// TimerA set
	COUNT_VALUE = COUNT_1ms * 30;	// 30ms, not 20ms
	IRQ_CLR;						// PORT2 interrupt flag clear (inside MSP430)
	IRQ_ON;

	Trf7970RawWrite(&buf[0], 6);	//issuing the IDENTIFY command

	i_reg = 0x01;
	irq_flag = 0x00;
	START_COUNTER;					//	Starting Timeout

	while (irq_flag == 0x00) {
	}								// wait for end of TX interrupt
	RESET_COUNTER;

	McuCounterSet();				// TimerA set
	COUNT_VALUE = COUNT_1ms * 20;
	START_COUNTER;					// start timer up mode

	irq_flag = 0x00;

	while (irq_flag == 0x00) {
	}								// wait for interrupt
	RESET_COUNTER;

	while (i_reg == 0x01)			// wait for RX complete
	{
		k++;

		if (k == 0xFFF0) {
			i_reg = 0x00;
			rx_error_flag = 0x00;
		}
	}

	if (i_reg == 0xFF) {					// if received block data in buffer
		if (stand_alone_flag == 1)
				{
			found = 1;
#ifdef ENABLE_HOST
			UartPutCrlf();
			UartSendCString("PicoPass Tag ");
			UartSendCString("  ASNB:      ");
			UartPutChar('[');

			for (i = 1; i < 9; i++) {
				UartPutByte(buf[i]);				// send ASNB to host
				ASNB[i-1] = buf[i];
			}
			UartPutChar(']');
			UartPutCrlf();
			UartSendCString("PicoPass Tag ");
			UartSendCString("   CRC:      ");
			UartPutChar('[');

			for (i = 9; i < 11; i++) {
				UartPutByte(buf[i]);				// send ASNB CRC to host
			}
			UartPutChar(']');
			UartPutCrlf();
#endif

			rxtx_state = 1; 						//resetting buffer pointer
		}

		if(found == 1)
					{
						LED_15693_ON;
						LED_14443B_ON;
						LED_14443A_ON;
					}

	} else
		{
			LED_15693_OFF;
			LED_14443A_OFF;
			LED_14443B_OFF;
		}
		// clear any IRQs
	Trf7970ResetIrqStatus();

	//return;
}
//===============================================================
// NAME: void PicoPass_SELECT(void)
//
// BRIEF: 	Is used to issue PicoPass SELECT command: (Command Code 1)
//			Function: Third step in retrieving serial #, response is SN + CRC
//
// INPUTS:	Command Code 1, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  0   |  0   |  0   |  1   |
//---------------------------------------------------------
//|             2             |             1             |
//---------------------------------------------------------
//
// OUTPUTS:	SN + CRC
//
// PROCESS:	[1] Send Command Code 0x21
//			[2] Get SN + CRC
// CHANGE:
// DATE  		WHO	DETAIL
// 09/12/2014	JDW Original Code
//===============================================================
		void PicoPass_SELECT(void) {
		u08_t i = 1,command[2];
		u16_t k;
		rxtx_state = 1; 				//resetting buffer pointer
		McuDelayMillisecond(2); 		//delay
		buf[0] = 0x8F;					// reset FIFO
		buf[1] = 0x90;					// sending without CRC from TRF, hardcoded here for one card to test
		buf[2] = 0x3D;					// write continuous from 1D
		buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
		buf[4] = 0x90;					// lower and broken nibbles of transmit byte length
		buf[5] = 0x21;					// SELECT
		for (i=0; i<8; i++){			// for loop takes ASNB and dynamically populates from IDENTITY command response
			buf[i+6] = ASNB[i];
		}

		McuCounterSet();				// TimerA set
		COUNT_VALUE = COUNT_1ms * 30;	// 30ms, not 20ms
		IRQ_CLR;
		IRQ_ON;				// PORT2 interrupt flag clear (inside MSP430)

		Trf7970RawWrite(&buf[0], 14);	//issuing the SELECT command

		i_reg = 0x01;
		irq_flag = 0x00;
		START_COUNTER;					//	Starting Timeout

		while (irq_flag == 0x00) {
		}								// wait for end of TX interrupt
		RESET_COUNTER;

		McuCounterSet();				// TimerA set
		COUNT_VALUE = COUNT_1ms * 20;
		START_COUNTER;					// start timer up mode

		irq_flag = 0x00;

		while (irq_flag == 0x00) {
		}								// wait for interrupt
		RESET_COUNTER;

		while (i_reg == 0x01)			// wait for RX complete
		{
			k++;

			if (k == 0xFFF0) {
				i_reg = 0x00;
				rx_error_flag = 0x00;
			}
		}

		command[0] = RSSI_LEVELS;						// read RSSI levels
		Trf7970ReadSingle(command, 1);


		if (i_reg == 0xFF) {					// if received block data in buffer
			if (stand_alone_flag == 1)
					{
	#ifdef ENABLE_HOST
				UartSendCString("PicoPass Tag ");
				UartSendCString("    SN:      ");
				UartPutChar('[');

				for (i = 8; i > 0; i--) {
					UartPutByte(buf[i]);				// send SN to host
				}
				UartPutChar(']');
				UartPutCrlf();
				UartSendCString("PicoPass Tag ");
				UartSendCString("   CRC:      ");
				UartPutChar('[');

				for (i = 10; i > 8; i--) {
					UartPutByte(buf[i]);				// send SN CRC to host
				}
				UartPutChar(']');
				UartPutCrlf();
				UartSendCString("         RSSI LEVEL:      ");
				UartPutChar('[');
				UartPutByte(command[0]);		// RSSI levels
				UartPutChar(']');
				UartPutCrlf();

	#endif


				rxtx_state = 1; 						//resetting buffer pointer

			}
			// response received
			//ui8Status = 0x01;
		} else {
			// no response
			//ui8Status = 0x00;
		}

		//Trf7970TurnRfOff();

		// clear any IRQs
		Trf7970ResetIrqStatus();

		//return;
	}
//===============================================================
// NAME: void PicoPass_READ(void)
//
// BRIEF: 	Is used to issue PicoPass READ command: (Command Code C)
//			Function: Read BLOCK DATA + CRC
//
// INPUTS:	Command Code C, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  1   |  1   |  0   |  0   |
//---------------------------------------------------------
//|             A             |             C             |
//---------------------------------------------------------
//
// OUTPUTS:	BLOCK DATA + CRC
//
// PROCESS:	[1] Send Command Code 0xAC + BLOCK # + CRC
//			[2] Get BLOCK DATA + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/13/2014	JDW Original Code
//===============================================================
void PicoPass_READ(void) {
			u08_t i = 1;
			u16_t k;
			McuDelayMillisecond(2); 		//delay

			buf[0] = 0x8F;					// reset FIFO
			buf[1] = 0x90;					// sending without CRC
			buf[2] = 0x3D;					// write continuous from 1D
			buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
			buf[4] = 0x40;					// lower and broken nibbles of transmit byte length
			buf[5] = 0xAC;					// READ
			buf[6] = 0x01;					// BLOCK #1 (HARDCODED FOR NOW)
			buf[7] = 0xFA;					// CRC (HARDCODED FOR NOW)
			buf[8] = 0x22;					// CRC (HARDCODED FOR NOW)

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 30;	// 30ms, not 20ms
			IRQ_CLR;
			IRQ_ON;						// PORT2 interrupt flag clear (inside MSP430)

			Trf7970RawWrite(&buf[0], 9);	//issuing the READ command

			i_reg = 0x01;
			irq_flag = 0x00;
			START_COUNTER;					//	Starting Timeout

			while (irq_flag == 0x00) {
			}								// wait for end of TX interrupt
			RESET_COUNTER;

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 20;
			START_COUNTER;					// start timer up mode

			irq_flag = 0x00;

			while (irq_flag == 0x00) {
			}								// wait for interrupt
			RESET_COUNTER;

			while (i_reg == 0x01)			// wait for RX complete
			{
				k++;

				if (k == 0xFFF0) {
					i_reg = 0x00;
					rx_error_flag = 0x00;
				}
			}

			if (i_reg == 0xFF) {					// if received block data in buffer
				if (stand_alone_flag == 1)
						{
		#ifdef ENABLE_HOST
					UartPutCrlf();
					//UartSendCString("PicoPass ");
					UartSendCString("PicoPass BLOCK 01 DATA:   ");
					UartPutChar('[');

					for (i = 8; i > 0; i--) {
						UartPutByte(buf[i]);				// send rotated BLOCK DATA to host
					}
					UartPutChar(']');
					UartPutCrlf();
					//UartSendCString("PicoPass ");
					UartSendCString("BLOCK 01 DATA CRC:        ");
					UartPutChar('[');

					for (i = 10; i > 8; i--) {
						UartPutByte(buf[i]);				// send rotated BLOCK DATA CRC to host
					}
					UartPutChar(']');
					UartPutCrlf();
		#endif

					rxtx_state = 1; 						//resetting buffer pointer
				}

			} else

			// clear any IRQs
			Trf7970ResetIrqStatus();

			//return;
		}
//===============================================================
// NAME: void PicoPass_READ4(void)
//
// BRIEF: 	Is used to issue PicoPass READ4 command: (Command Code 6)
//			Function: Read BLOCK DATA + CRC
//
// INPUTS:	Command Code 6, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  0   |  1   |  1   |  0   |
//---------------------------------------------------------
//|             A             |             6             |
//---------------------------------------------------------
//
// OUTPUTS:	4 BLOCKS OF DATA + CRC
//
// PROCESS:	[1] Send Command Code 0xA6 + BLOCK # TO START ON + CRC
//			[2] Get FOUR BLOCKS OF DATA + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/13/2014	JDW Original Code
//===============================================================
void PicoPass_READ4(void) {
			u08_t i = 1, j=0;
			u16_t k;
			u08_t ui8StartBlock = 3;
			McuDelayMillisecond(2); 		//delay

			buf[0] = 0x8F;					// reset FIFO
			buf[1] = 0x90;					// sending without CRC
			buf[2] = 0x3D;					// write continuous from 1D
			buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
			buf[4] = 0x40;					// lower and broken nibbles of transmit byte length
			buf[5] = 0xA6;					// READ4
			buf[6] = 0x03;					// BLOCK #3 to start on(HARDCODED FOR NOW)
			buf[7] = 0xE8;					// CRC (HARDCODED FOR NOW)
			buf[8] = 0x01;					// CRC (HARDCODED FOR NOW)

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 30;	// 30ms, not 20ms
			IRQ_CLR;
			IRQ_ON;						// PORT2 interrupt flag clear (inside MSP430)

			Trf7970RawWrite(&buf[0], 9);	//issuing the READ4 command

			i_reg = 0x01;
			irq_flag = 0x00;
			START_COUNTER;					//	Starting Timeout

			while (irq_flag == 0x00) {
			}								// wait for end of TX interrupt
			RESET_COUNTER;

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 20;
			START_COUNTER;					// start timer up mode

			irq_flag = 0x00;

			while (irq_flag == 0x00) {
			}								// wait for interrupt
			RESET_COUNTER;

			while (i_reg == 0x01)			// wait for RX complete
			{
				k++;

				if (k == 0xFFF0) {
					i_reg = 0x00;
					rx_error_flag = 0x00;
				}
			}

			if (i_reg == 0xFF) {					// if received block data in buffer
				if (stand_alone_flag == 1)
						{
		#ifdef ENABLE_HOST
				UartPutCrlf();
				for(j = 0; j < 4; j++){
					UartSendCString("PicoPass BLOCK ");
					UartPutByte(ui8StartBlock++);
					UartSendCString(" DATA:   [");
					for(i = 8+(j*8); i > 0+(j*8); i--)
					{
						UartPutByte(buf[i]);		// send block data to host
					}
					UartPutChar(']');
					UartPutCrlf();
				}


					UartSendCString("PicoPass READ4 DATA CRC:  ");
					UartPutChar('[');

					for (i = 34; i > 32; i--) {
					UartPutByte(buf[i]);				// send rotated 4 BLOCK DATA CRC to host
					}
					UartPutChar(']');
					UartPutCrlf();
					UartPutCrlf();

		#endif

					rxtx_state = 1; 						//resetting buffer pointer
				}

			} else

			// clear any IRQs
			Trf7970ResetIrqStatus();

			//return;
		}
//===============================================================
// NAME: void PicoPass_HALT(void)
//
// BRIEF: 	Is used to issue PicoPass HALT command: (Command Code 0)
//			Function: Stop tag from communicating, while its in the reader field, no response
//
// INPUTS:	Command Code 0, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  0   |  0   |  0   |  0   |
//---------------------------------------------------------
//|             A             |             0             |
//---------------------------------------------------------
//
// OUTPUTS:	SOF
//
// PROCESS:	[1] Send Command Code 0xAA
//			[2] Get SOF back from tag (ACK)
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Code
//===============================================================

	void PicoPass_HALT(void) {

	buf[0] = 0x8F;					// reset FIFO
	buf[1] = 0x90;					// sending without CRC
	buf[2] = 0x3D;					// write continuous from 1D
	buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
	buf[4] = 0x10;					// lower and broken nibbles of transmit byte length
	buf[5] = 0xA0;					// HALT

	McuCounterSet();				// TimerA set
	COUNT_VALUE = COUNT_1ms * 30;	// 30ms
	IRQ_CLR;						// PORT2 interrupt flag clear (inside MSP430)
	IRQ_ON;

	Trf7970RawWrite(&buf[0], 6);	//issuing the HALT command

}

//===============================================================
// NAME: void PicoPass_ACT(void)
//
// BRIEF: 	Is used to issue PicoPass ACT command: (Command Code E)
//			Function: shifts by 3 the ASNB each time this is issued after the ACTALL
//
// INPUTS:	Command Code E, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  1   |  1   |  1   |  0   |
//---------------------------------------------------------
//|             2             |             E             |
//---------------------------------------------------------
//
// OUTPUTS:	SOF
//
// PROCESS:	[1] Send Command Code 0x2E
//			[2] Get SOF back from tag (ACK)
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Code
//===============================================================

		void PicoPass_ACT(void) {

		buf[0] = 0x8F;					// reset FIFO
		buf[1] = 0x90;					// sending without CRC
		buf[2] = 0x3D;					// write continuous from 1D
		buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
		buf[4] = 0x10;					// lower and broken nibbles of transmit byte length
		buf[5] = 0x2E;					// ACT

		McuCounterSet();				// TimerA set
		COUNT_VALUE = COUNT_1ms * 30;	// 30ms
		IRQ_CLR;						// PORT2 interrupt flag clear (inside MSP430)
		IRQ_ON;

		Trf7970RawWrite(&buf[0], 6);	//issuing the ACT command

	}

//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_UPDATE(void)
//
// BRIEF: 	Is used to issue PicoPass UPDATE command: (Command Code 7)
//			Function: Write Single Block (Unsecured)
//
// INPUTS:	Command Code 7, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  0   |  1   |  1   |  1   |
//---------------------------------------------------------
//|             2             |             7             |
//---------------------------------------------------------
//
// OUTPUTS:	4 BLOCKS OF DATA + CRC
//
// PROCESS:	[1] Send Command Code 0x27 + BLOCK # TO WRITE + DATA to WRITE + CRC
//			[2] UPDATED BLOCK DATA + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
void PicoPass_UPDATE(void) {
	//SHELL FOR COMMAND
}

//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_DETECT(void)
//
// BRIEF: 	Is used to issue PicoPass DETECT command: (Command Code F)
//			Function: EAS operations (if EAS bit is set to 0 in tag)
//
// INPUTS:	Command Code F, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  1   |  1   |  1   |  1   |
//---------------------------------------------------------
//|             A             |             F             |
//---------------------------------------------------------
//
// OUTPUTS:	SN + CRC
//
// PROCESS:	[1] Send Command Code 0xAF
//			[2] Tag SN + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
	void PicoPass_DETECT(void) {
			u08_t i = 1,command[2];
			u16_t k;
			rxtx_state = 1; 				//resetting buffer pointer
			McuDelayMillisecond(2); 		//delay
			buf[0] = 0x8F;					// reset FIFO
			buf[1] = 0x90;					// sending without CRC from TRF, hardcoded here for one card to test
			buf[2] = 0x3D;					// write continuous from 1D
			buf[3] = 0x00;					// upper and middle nibbles of transmit byte length
			buf[4] = 0x10;					// lower and broken nibbles of transmit byte length
			buf[5] = 0xAF;					// SELECT

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 30;	// 30ms, not 20ms
			IRQ_CLR;
			IRQ_ON;				// PORT2 interrupt flag clear (inside MSP430)

			Trf7970RawWrite(&buf[0], 6);	//issuing the SELECT command

			i_reg = 0x01;
			irq_flag = 0x00;
			START_COUNTER;					//	Starting Timeout

			while (irq_flag == 0x00) {
			}								// wait for end of TX interrupt
			RESET_COUNTER;

			McuCounterSet();				// TimerA set
			COUNT_VALUE = COUNT_1ms * 20;
			START_COUNTER;					// start timer up mode

			irq_flag = 0x00;

			while (irq_flag == 0x00) {
			}								// wait for interrupt
			RESET_COUNTER;

			while (i_reg == 0x01)			// wait for RX complete
			{
				k++;

				if (k == 0xFFF0) {
					i_reg = 0x00;
					rx_error_flag = 0x00;
				}
			}

			command[0] = RSSI_LEVELS;						// read RSSI levels
			Trf7970ReadSingle(command, 1);


			if (i_reg == 0xFF) {					// if received block data in buffer
				if (stand_alone_flag == 1)
						{
		#ifdef ENABLE_HOST
					UartPutCrlf();
					UartSendCString("PicoPass EAS ");
					UartSendCString("    SN:      ");
					UartPutChar('[');

					for (i = 8; i > 0; i--) {
						UartPutByte(buf[i]);				// send SN to host
					}
					UartPutChar(']');
					UartPutCrlf();
					UartSendCString("PicoPass EAS ");
					UartSendCString("   CRC:      ");
					UartPutChar('[');

					for (i = 10; i > 8; i--) {
						UartPutByte(buf[i]);				// send SN CRC to host
					}
					UartPutChar(']');
					UartPutCrlf();
					UartSendCString("         RSSI LEVEL:      ");
					UartPutChar('[');
					UartPutByte(command[0]);		// RSSI levels
					UartPutChar(']');
					UartPutCrlf();

		#endif


					rxtx_state = 1; 						//resetting buffer pointer

				}
				// response received
				//ui8Status = 0x01;
			} else {
				// no response
				//ui8Status = 0x00;
			}

			//Trf7970TurnRfOff();

			// clear any IRQs
			Trf7970ResetIrqStatus();

			//return;
		}

//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_PAGESEL(void)
//
// BRIEF: 	Is used to issue PicoPass PAGESEL command: (Command Code 4)
//			Function: Changes the book or the page
//
// INPUTS:	Command Code 4, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  0   |  1   |  0   |  0   |
//---------------------------------------------------------
//|             2             |             4             |
//---------------------------------------------------------
//
// OUTPUTS:	Contents of Block 1 of the new page + CRC
//
// PROCESS:	[1] Send Command Code 0x24 + Book and Page # + CRC
//			[2] Get back Block 1 of new page + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
void PicoPass_PAGESEL(void) {
	//SHELL FOR COMMAND
}
//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_READCHECK(void)
//
// BRIEF: 	Is used to issue PicoPass READCHECK command: (Command Code 8)
//			Function: First command of auth procedure.
//			If K = 0, Debit Key (Kd) will be used,
//          If K = 1, Credit Key (Kc) will be used
//
// INPUTS:	Command Code 8, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  1   |  0   |  0   |  0   |
//---------------------------------------------------------
//|             2             |             8             | //using Kd
//---------------------------------------------------------
//							 or
//|-------------------------------------------------------|
//|  1   |  0   |  1   |  1   |  1   |  0   |  0   |  0   |
//---------------------------------------------------------
//|             B             |             8             | //using Kc
//---------------------------------------------------------
//
// OUTPUTS:	Contents of Block 1 of the new page + CRC
//
// PROCESS:	[1] Send Command Code 0x28 + BLOCK ADDRESS (for Kd)
//				or
//			[1] Send Command Code 0xB8 + BLOCK ADDRESS (for Kc)
//			[2] Get back contents of Block Address specified, follow with CHECK command
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
void PicoPass_READCHECK(void) {
	//SHELL FOR COMMAND
}
//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_CHECK(void)
//
// BRIEF: 	Is used to issue PicoPass CHECK command: (Command Code 5)
//			must be preceded by the READCHECK command
//			Function: Second Step of authentication procedure (mutual auth portion)
//
// INPUTS:	Command Code 5, with Command Opcode of A
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  1   |  0   |  1   |  0   |  0   |  1   |  0   |  1   |
//---------------------------------------------------------
//|             A             |             5             |
//---------------------------------------------------------
//
// OUTPUTS:	32 bit response to the challenge for reader to authenticate the tag
//
// PROCESS:	[1] Send Command Code 0xA5 + 32 bit challenge + 32 bit reader signature
//			[2] Get back results of algorithm (32 bits), which indicates tag authenticated the reader
//			[3] reader takes 32 bit response and compares it to its own calculation to authenticate the tag
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
void PicoPass_CHECK(void) {
	//SHELL FOR COMMAND
}
//==================================================================
//  *******************SHELL FOR COMMAND**************************
//
//NAME: void PicoPass_UPDATE_SECURED(void)
//
// BRIEF: 	Is used to issue PicoPass UPDATE_SECURED command: (Command Code 7)
//			Function: Write Single Block (secured)
//			issued after READCHECK, CHECK procedure successful
//
// INPUTS:	Command Code 7, with Command Opcode of 2
//
//      Command Opcodes + Instruction Code derivation:
//|-------------------------------------------------------|
//|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |
//---------------------------------------------------------
//|  P   |  M1  |  M0  |  K   |        Instruction        |
//---------------------------------------------------------
//|  0   |  0   |  1   |  0   |  0   |  1   |  1   |  1   |
//---------------------------------------------------------
//|             2             |             7             |
//---------------------------------------------------------
//
// OUTPUTS:	4 BLOCKS OF DATA + CRC
//
// PROCESS:	[1] Send Command Code 0x27 + BLOCK # TO WRITE + BLOCK DATA TO WRITE + SIGNATURE
//			[2] UPDATED BLOCK DATA + CRC
//
// CHANGE:
// DATE  		WHO	DETAIL
// 09/14/2014	JDW Original Codeshell (validated opcode and parity)
//==================================================================
void PicoPass_UPDATE_SECURED(void) {
	//SHELL FOR COMMAND
}









