/*************************************************************************************************************************************************/
/*!     DAC8760.c
 *
 *       This code is designed to perform standard command and control operations on the DAC8760 over a SPI bus. Functions exist to setup, configure
 *       and control the DAC8760.
 *
 *
 */
/**************************************************************************************************************************************************
 *       Copyright  2014 Texas Instruments Incorporated - http://www.ti.com/                                                                      *
 ***************************************************************************************************************************************************
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met: *
 *                                                                                                                                                 *
 *    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.                 *
 *                                                                                                                                                 *
 *    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the        *
 *    documentation and/or other materials provided with the distribution.                                                                         *
 *                                                                                                                                                 *
 *    Neither the name of Texas Instruments Incorporated nor the names of its contributors may be used to endorse or promote products derived      *
 *    from this software without specific prior written permission.                                                                                *
 *                                                                                                                                                 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          *
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT     *
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT         *
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY    *
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                           *
 **************************************************************************************************************************************************/
//
//
//   Thomas Schneider
//   Texas Instruments Inc.
//   July 2014
//   Built with Code Composer Studio V6.0
//***************************************************************************************************************************************************
#include <stdint.h>
#include <msp430.h>
#include "DAC8760.h"
#include "hal.h"

static unsigned char RcvData[3];

/*************************************************************************************************************************************************
 *  DAC8760_Write_Regs
 **************************************************************************************************************************************************/
/*!
 * @brief Writes registers on the DAC8760.
 *
 * This function will execute a write register command to the DAC8760. This function can be used to update one register on the DAC8760.
 * No error checking is performed, so it is the user's responsibility to make sure they do not attempt to write past the end of the DAC8760 registers.
 *
 * @param[in]   writeValues    16 bit register value to place in the DAC8760
 * @param[in]   address        Address of the register to write
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_WriteReg (uint16_t writeValues, uint8_t address)
{
    uint8_t outData[3];

    outData[0] = address;

    // Switch Endianess
    outData[1] = writeValues >> 8;
    outData[2] = writeValues & 0xff;

    DAC8760_SPIWrite (outData, RcvData);

}

/*************************************************************************************************************************************************
 *  DAC8760_Read_Regs
 **************************************************************************************************************************************************/
/*!
 * @brief Reads registers on the DAC8760.
 *
 * This function will execute a read register command to the DAC8760 and return the resultant data. This function can be used to read one or more
 * registers from the DAC8760. No error checking is performed, so it is the user's responsibility to make sure they do not attempt to read past
 * the end of the DAC8760 registers.
 *
 * @param[out]   *readValues     Pointer to place the 8 bit register values from the DAC8760
 * @param[in]    address         Address of the register to read
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_ReadReg (uint8_t *readValues, uint8_t address)
{
    uint8_t outData[3] = {0x55, 0x55, 0x55};

    // Sets the address for the read command
    outData[0] = DAC8760_READ_REGISTER;
    outData[2] = address;
    DAC8760_SPIWrite (outData, readValues);

    // Performs actual read of previous address
    outData[0] = DAC8760_WRITE_NOP_REGISTER;
    DAC8760_SPIWrite (outData, readValues);

}

/*************************************************************************************************************************************************
 *  Setup_DAC8760
 **************************************************************************************************************************************************/
/*!
 * @brief Performs the setup of the DAC8760.
 *
 * This function will configure the DAC8760.
 *
 * @param[in]     controlReg             Value written into DAC8760_CONTROL_REGISTER
 * @param[in]     configurationReg       Value written into DAC8760_CONFIGURATION_REGISTER
 * @param[in]     gainCalReg             Value written into DAC8760_GAIN_CALIBRATION_REGISTER
 * @param[in]     zeroCalReg             Value written into DAC8760_ZERO_CALIBRATION_REGISTER
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_Setup (uint16_t controlReg, uint16_t configurationReg, uint16_t gainCalReg, uint16_t zeroCalReg)
{
    DAC8760_SPISetupMaster();
    DAC8760_Reset();
    DAC8760_Nop();
    DAC8760_WriteReg (controlReg, DAC8760_WRITE_CONTROL_REGISTER);
    DAC8760_WriteReg (configurationReg, DAC8760_WRITE_CONFIGURATION_REGISTER);
    DAC8760_WriteReg (gainCalReg, DAC8760_WRITE_GAIN_CALIBRATION_REGISTER);
    DAC8760_WriteReg (zeroCalReg, DAC8760_WRITE_ZERO_CALIBRATION_REGISTER);

}


/*************************************************************************************************************************************************
 *  DAC8760_Reset
 **************************************************************************************************************************************************/
/*!
 * @brief Sends a Reset Command to the DAC8760.
 *
 * This function sends a Reset command to the DAC8760 on the SPI bus.
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_Reset (void)
{
    uint8_t outData[3];
    outData[0] = DAC8760_WRITE_RESET_REGISTER;
    outData[2] = DAC8760_RESET;

    DAC8760_SPIWrite (outData, RcvData);

}

/*************************************************************************************************************************************************
 *  DAC8760_Nop
 **************************************************************************************************************************************************/
/*!
 * @brief Sends a Nop Command to the DAC8760.
 *
 * This function sends a Nop command to the DAC8760 on the SPI bus. The DAC8760 will timeout and move into an error state if it does not receive
 * regular commands for the SPI master. This command can be used to notify the DAC8760 that the system is still operational, but no change to the
 * DAC8760 is desired.
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_Nop (void)
{
    uint8_t outData[3];
    outData[0] = DAC8760_WRITE_NOP_REGISTER;
    DAC8760_SPIWrite (outData, RcvData);

}

/*************************************************************************************************************************************************
 *  DAC8760_Set_Out_Value
 **************************************************************************************************************************************************/
/*!
 * @brief Sets the output current from the DAC8760.
 *
 * The DAC8760 is designed to be used in a 4-20mAmp system. Data is communicated by changing the current output from the DAC8760. This function sets
 * the desired output current.
 *
 * @param[in]     value         Value to output from the DAC8760
 *
 * @return  None
 *
 **************************************************************************************************************************************************/
void DAC8760_SetOutValue (uint16_t value)
{
    DAC8760_WriteReg (value, DAC8760_WRITE_DATA_REGISTER);
}

/*************************************************************************************************************************************************
 *  DAC8760_Read_Status
 **************************************************************************************************************************************************/
/*!
 * @brief Reads thE status register from the DAC8760.
 *
 * This function returns the current value in the status register of the DAC8760
 *
 * @return  DAC8760 Status     (DAC8760_FERR_STS, DAC8760_SPI_TIMEOUT_ERR, DAC8760_LOOP_STS, DAC8760_CURR_LOOP_STS)
 **************************************************************************************************************************************************/
uint8_t DAC8760_ReadStatus (void)
{
    uint8_t returnValue[3];

    DAC8760_ReadReg (returnValue, DAC8760_READ_STATUS_REGISTER);

    return (returnValue[2]);
}


