/*
 * hal.c
 *
 *
 */
/**************************************************************************************************************************************************
 *       Copyright  2014 Texas Instruments Incorporated - http://www.ti.com/                                                                      *
 ***************************************************************************************************************************************************
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met: *
 *                                                                                                                                                 *
 *    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.                 *
 *                                                                                                                                                 *
 *    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the        *
 *    documentation and/or other materials provided with the distribution.                                                                         *
 *                                                                                                                                                 *
 *    Neither the name of Texas Instruments Incorporated nor the names of its contributors may be used to endorse or promote products derived      *
 *    from this software without specific prior written permission.                                                                                *
 *                                                                                                                                                 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          *
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT     *
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT         *
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY    *
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                           *
 **************************************************************************************************************************************************/

#include <stdint.h>
#include <msp430.h>
#include "DAC8760.h"
#include "hal.h"
#include "driverlib.h"

#define GPIO_ALL    GPIO_PIN0|GPIO_PIN1|GPIO_PIN2|GPIO_PIN3| \
                    GPIO_PIN4|GPIO_PIN5|GPIO_PIN6|GPIO_PIN7

/*
* This function drives all the I/O's as output-low, to avoid floating inputs
* (which cause extra power to be consumed). This setting is compatible with
* TI FET target boards, but may not be compatible with custom hardware,
* which may have components attached to the I/Os that could be affected by these settings.
* So if using other boards, this function may need to be modified.
*/
void initPorts(void)
{
#ifdef __MSP430_HAS_PORT1_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P1, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P1, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT2_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P2, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P2, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT3_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P3, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P3, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT4_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P4, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P4, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT5_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P5, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P5, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT6_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P6, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P6, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT7_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P7, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P7, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT8_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P8, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P8, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORT9_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_P9, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_P9, GPIO_ALL);
#endif

#ifdef __MSP430_HAS_PORTJ_R__
    GPIO_setOutputLowOnPin(GPIO_PORT_PJ, GPIO_ALL);
    GPIO_setAsOutputPin(GPIO_PORT_PJ, GPIO_ALL);
#endif
}

/* Configures the system clocks:
* MCLK = SMCLK = DCO/FLL = mclkFreq (expected to be expressed in Hz)
* ACLK = FLLref = REFO=32kHz
*
* XT2 is not configured here.  Instead, the USB API automatically starts XT2
* when beginning USB communication, and optionally disables it during USB
* suspend.  It's left running after the USB host is disconnected, at which
* point you're free to disable it.  You need to configure the XT2 frequency
* in the Descriptor Tool (currently set to 4MHz in this example).
* See the Programmer's Guide for more information.
*/
void initClocks(uint32_t mclkFreq)
{
#ifndef DRIVERLIB_LEGACY_MODE
    UCS_clockSignalInit(
       UCS_FLLREF,
       UCS_REFOCLK_SELECT,
       UCS_CLOCK_DIVIDER_1);

    UCS_clockSignalInit(
       UCS_ACLK,
       UCS_REFOCLK_SELECT,
       UCS_CLOCK_DIVIDER_1);

    UCS_initFLLSettle(
        mclkFreq/1000,
        mclkFreq/32768);
#else
    UCS_clockSignalInit(
       UCS_BASE,
       UCS_FLLREF,
       UCS_REFOCLK_SELECT,
       UCS_CLOCK_DIVIDER_1);

    UCS_clockSignalInit(
       UCS_BASE,
       UCS_ACLK,
       UCS_REFOCLK_SELECT,
       UCS_CLOCK_DIVIDER_1);

    UCS_initFLLSettle(
        UCS_BASE,
        mclkFreq/1000,
        mclkFreq/32768);
#endif

}

/*************************************************************************************************************************************************
*  Setup_USCIA0_SPI
**************************************************************************************************************************************************/
void DAC8760_SPISetupMaster(void)
{
    UCA0CTL1  |= UCSWRST;                                                      // Enable SW reset
    UCA0CTL0 = (UCMST + UCMSB + UCSYNC + UCCKPH);        // SPI Master, 3 wire, synchronous mode
    UCA0CTL1 |= UCSSEL_2;
    UCA0BR0 = 8;                                                               // SMCLK/8 = SCLK (1MHz)
    UCA0BR1 = 0;
    UCA0CTL1 &= ~UCSWRST;                                                      // Clear SW reset, resume operation

    DAC8760_SPI_PORTDIR  &= ~(DAC8760_SIMO_PIN + DAC8760_SOMI_PIN + DAC8760_CLK_PIN);                                     // Assign SPI pins to Primary Peripheral of USCI_A0
    DAC8760_SPI_PORTSEL  |= (DAC8760_SIMO_PIN + DAC8760_SOMI_PIN + DAC8760_CLK_PIN);
    DAC8760_LATCH_PORTOUT &= ~DAC8760_LATCH_PIN;
    DAC8760_LATCH_PORTDIR |= DAC8760_LATCH_PIN;


}


/*************************************************************************************************************************************************
*  SPI_Write
**************************************************************************************************************************************************/
/*!
* @brief Perform a SPI communication sequence to Write/Read  3 bytes on SPI bus
*
* The SPI bus uses one clock line and 2 data lines that transfer data at the same time. The master will read data from the MISO
* (Master In Slave Out) line on one edge of the clock. The slave device will read on the MOSI (Master Out Slave In) on the other edge of the clock.
* As such, reads and writes happen together. If there is no data to send to the slave, the master must write out dummy data on the bus to get the
* clock to toggle for the slave device to communicate.
*
*
* @param[in]   outData              Pointer to an array with the data that will be written on the SPI bus
* @param[in]   inData               Pointer to an array to place the data returned from the slave device. One byte for EVERY byte sent,
*                                       even the command byte
*
**************************************************************************************************************************************************/
void DAC8760_SPIWrite (uint8_t *outData, uint8_t *inData)
{
    uint8_t i;

    for (i=0; i<3; i++)
    {
        inData[i] = DAC8760_SPIWriteByte (outData[i]);
    }

    // Load DAC registers input.
    // A rising edge on the Latch pin loads the input shift register data into the DAC data and control registers and updates the DAC outputs.
    DAC8760_LATCH_PORTOUT |= DAC8760_LATCH_PIN;
    __delay_cycles(1);          // delay min 40ns
    DAC8760_LATCH_PORTOUT &= ~DAC8760_LATCH_PIN;
}

/*************************************************************************************************************************************************
*  SPI_Write_Byte
**************************************************************************************************************************************************/
/*!
* @brief This function is used to place a single byte on the SPI bus. This function is for expert mode only.
*
* This function can be used by someone managing their own communications on the SPI bus to place one byte on the bus. It is assumed that the user
* will manage the STE pin themselves when using this command. The received byte from the MISO pin will be returned by this function.
*
* This function only exists to support users who have non-standard SPI devices on their bus. Some devices require excessive time between bytes or
* a delay between the command byte and subsequent data bytes.
*
* @param[in]   data              Pointer to an array with the data that will be written on the SPI bus
*
* @return  in_data               Character sent from SPI device on MISO line
**************************************************************************************************************************************************/
uint8_t DAC8760_SPIWriteByte (uint8_t data)
{
    while (!(UCA0IFG&UCTXIFG));               // USCI_B0 TX buffer ready?
    UCA0IFG &= ~UCRXIFG;             // Clear RX Interrupt flag

    UCA0TXBUF = data;              // Place data in TX Buffer
    while (!(UCA0IFG & UCRXIFG));    // Wait for end of data receive
    return ((uint8_t)UCA0RXBUF);            // Return the received byte from RX Buffer
}

