import ctypes
import os
import sys

os.chdir(os.path.dirname(sys.argv[0])) #Set the current working directory to the location of this script

import supportFunctions
from supportFunctions import adcWriteRegs as writeReg
from supportFunctions import adcReadRegs as readReg

# dll_path = "C:/TI_FTDI_x64/x64/TI_FTDIx64.dll" #example of absolute path
dll_path = "TI_FTDI_DLL/TI_FTDIx64.dll" # example of relative path (where DLL is within subfolder of current directory named TI_FTDI_DLL)
TI_FTDI = ctypes.cdll.LoadLibrary(dll_path) # Load DLL which is at dll_path. This is an x64 .DLL as needed by Python.

# Below are void pointers that will be used as FT_HANDLES. TI_FTDI_DLL uses these with the d2xx API internally. 
ADC12DL3200EVM_FTDI_BusA = ctypes.c_void_p() # ptr for ftdi handle bus A
ADC12DL3200EVM_FTDI_BusC = ctypes.c_void_p() # ptr for ftdi handle bus C

# Below are void pointers that will be used as SPI and I2C devices handles. They are used by TI_FTDI_DLL library.
adc_ptr = ctypes.c_void_p() # ptr for ADC12DL3200EVM spi bus
lmk_ptr = ctypes.c_void_p() # ptr for LMK04828 spi bus
lmx_ptr = ctypes.c_void_p() # ptr for LMX2582 spi bus

connectedDevices = supportFunctions.returnFTDISerialNumber(TI_FTDI,'ADC12DL3200')
EVM_serial = connectedDevices[0] # If multiple EVM are connected, ensure this is set to correct index

adc_serial = EVM_serial + 'A' # with bus appended 
lmkx_serial = EVM_serial + 'C' # with bus appended

# Setup SPI for ADC12DL3200:
TI_FTDI.setupFTDIdev(ctypes.byref(ADC12DL3200EVM_FTDI_BusA), adc_serial.encode(), 12000000) # Set up FTDI BusB at baud rate = 12e6 (max for DLL)
TI_FTDI.initSPIDev(ctypes.byref(adc_ptr), 0, 3, 1, 2, 15, 8, True, True) # SCLK=0, SEN/CSb=3, MOSI=1, MISO=2, Addr_bits=15, Data_bits=8, posedge_clk=True, usesRWbit=True

# Setup SPI for LMK04832 and LMX2582:
TI_FTDI.setupFTDIdev(ctypes.byref(ADC12DL3200EVM_FTDI_BusC), lmkx_serial.encode(), 12000000) # Set up FTDI BusB at baud rate = 12e6 (max for DLL)
# Setup SPI for LMK04828:
TI_FTDI.initSPIDev(ctypes.byref(lmk_ptr), 0, 2, 1, 5, 15, 8, True, True) # SCLK=0, SEN/CSb=2, MOSI=1, MISO=5, Addr_bits=15, Data_bits=8, posedge_clk=True, usesRWbit=True
# Setup SPI for LMX2582:
TI_FTDI.initSPIDev(ctypes.byref(lmx_ptr), 0, 3, 1, 4, 7, 16, True, True) # SCLK=0, SEN/CSb=3, MOSI=1, MISO=4, Addr_bits=7, Data_bits=16, posedge_clk=True, usesRWbit=True

# Mapping handles for easier code visibility:
ADC12DL3200 = 0
LMK04828 = 1
LMX2582 = 2

TI_FTDI.mapHandles(ADC12DL3200EVM_FTDI_BusA, adc_ptr, ADC12DL3200)
TI_FTDI.mapHandles(ADC12DL3200EVM_FTDI_BusC, lmk_ptr, LMK04828)
TI_FTDI.mapHandles(ADC12DL3200EVM_FTDI_BusC, lmx_ptr, LMX2582)

# Example function calls:

''' Writing single register: '''
# Example:
# Write to register 0x002 value 0x03
writeReg(TI_FTDI,ADC12DL3200,0x002,0x03)

''' Writing multiple registers: '''
# Example: 
# Write to register 0x002 value 0x00
# Write to register 0x002 value 0x03
# Write to register 0x002 value 0x00
# Write to register 0x030 value 0x12
# Write to register 0x031 value 0x34

addr_list = [0x002,0x002,0x002,0x030,0x031]
data_list = [0x00, 0x03, 0x00, 0x12, 0x34]
writeReg(TI_FTDI,ADC12DL3200,addr_list,data_list)

''' Reading from register: '''
# Example:
# Read value from register 0x002
value = readReg(TI_FTDI,ADC12DL3200,0x002)
print('\nreadReg(TI_FTDI,ADC12DL3200,0x002): ',value) # returns integer

''' Reading multiple registers: '''
# Example: 
# Read from register 0x002
# Read from register 0x002
# Read from register 0x002
# Read from register 0x030
# Read from register 0x031

addr_list = [0x002,0x002,0x002,0x030,0x031]
value = readReg(TI_FTDI,ADC12DL3200,addr_list) # returns list of integers
print('readReg(TI_FTDI,ADC12DL3200,addr_list): ',value)

value = supportFunctions.convertToString(value) # convert list of integers to list of strings of the hex representation
print('supportFunctions.convertToString(value): ',value)
print('\n\n')


# Reading chip type
addr_list = [0x003]
readback = readReg(TI_FTDI,ADC12DL3200,addr_list) # returns list of integers
value0x3 = readback[0]

chip_type = value0x3
print(f"CHIP_TYPE: {hex(chip_type)}")


# Reading chip id
addr_list = [0x004, 0x005]
readback = readReg(TI_FTDI,ADC12DL3200,addr_list) # returns list of integers
value0x4 = readback[0]
value0x5 = readback[1]

chip_id = (value0x5<<8)+value0x4
print(f"CHIP_ID: {hex(chip_id)}")
