/*
    <TI_FTDI.h>

    Written by Matt Kramer of Texas Instruments (A0226277)
*/
#include <stdio.h>
#include <stdint.h>
#include <windows.h>

#ifndef TI_FTDI_INCLUDED
#define TI_FTDI_INCLUDED

// Common baudrates.
#define FT245MAXBAUD    3E6
#define FT2232MAXBAUD   12E6
#define I2CBAUDSTD      8E4
#define I2CBAUDFAST     32E4

// These can only be changed at compile time of .DLL but are useful outside of .DLL.
#define SERIAL_LEN 64       // 64 should be enough for holding an FTDI serial number.
#define DEVICE_COUNT 256    // number of devices which can be indexed.

//DLL building stuff.
#ifdef BUILD_DLL
    #define DLL_EXPORT __declspec(dllexport)
#else
    #define DLL_EXPORT __declspec(dllimport)
#endif

#ifdef __cplusplus
extern "C"
{
#endif
//Handles used to point to internal data structures.
typedef void*  FT_HANDLE;
typedef void*  PROT_HANDLE;


//Print Current FTDI devices connected to PC.
int32_t DLL_EXPORT printConnectedDevices(   BOOL print2Screen
);

int32_t DLL_EXPORT getConnectedDevices(     char ** connectedDevices,
                                            uint32_t numDevices
);
//Get DeviceInfo for given Index. Update boolan needs to be true the first time the function is called or if an FTDI devices is added or removed from the PC.
int32_t DLL_EXPORT getDeviceInfo(           uint32_t index,
                                            uint32_t * flags,
                                            uint32_t * type,
                                            uint32_t * devID,
                                            uint32_t * locID,
                                            char * description,
                                            char * serialNumber,
                                            BOOL update
);

//FTDI init Wrapper
int32_t DLL_EXPORT setupFTDIdev(            FT_HANDLE* inHandle,
                                            char serial[],
                                            uint32_t baudRate
);

//SPI Initialize. Needs to be called for each SPI device that is connected. Settings will be stored in provided handle.
int32_t DLL_EXPORT initSPIDev(              PROT_HANDLE* inHandle,
                                            uint8_t CLKnum,
                                            uint8_t CSnum,
                                            uint8_t SDInum,
                                            uint8_t SDOnum,
                                            uint8_t addrBits,
                                            uint8_t dataBits,
                                            BOOL posedge,
                                            BOOL RWbit
);
//Can be used for both I2C and SPI Devices. If extra OE GPO is necessary for readback.
int32_t DLL_EXPORT setupOE(                 PROT_HANDLE* inHandle,
                                            uint8_t OEnum,
                                            BOOL OEpol
);

//WriteSPI and readSPI are SPI specific and accept FTDI and PROT handles directly.
int32_t DLL_EXPORT writeSPI(                FT_HANDLE dev,
                                            PROT_HANDLE spi,
                                            uint32_t addr,
                                            uint32_t data
);


int32_t DLL_EXPORT readSPI(                 FT_HANDLE dev,
                                            PROT_HANDLE spi,
                                            uint32_t addr
);

//I2C Initialize. Needs to be called for each I2C device that is connected. Settings will be stored in provided handle.
int32_t DLL_EXPORT initI2CDev(              PROT_HANDLE * inHandle,
                                            uint8_t SCL_num,
                                            uint8_t SDAw_num,
                                            uint8_t SDAr_num,
                                            uint8_t addr,
                                            uint8_t bytesPerAddr,
                                            uint8_t bytesPerReg
);

//WriteI2C and readI2C are SPI specific and accept FTDI and PROT handles directly.
int32_t DLL_EXPORT writeI2C(                FT_HANDLE dev,
                                            PROT_HANDLE i2c,
                                            uint32_t reg,
                                            uint32_t data
);

int32_t DLL_EXPORT readI2C(                 FT_HANDLE dev,
                                            PROT_HANDLE i2c,
                                            uint32_t reg
);

//SPI and I2C function for reading/writing multiple registers in a single API call. It is protocol agnostic as it checks the PROT handle for the type of device.
//data is updated by reference in readback.
int32_t DLL_EXPORT multiRegRW(              FT_HANDLE dev,
                                            PROT_HANDLE prot,
                                            uint32_t addr [],
                                            uint32_t data [],
                                            uint32_t numRegs,
                                            BOOL write
);
//SPI and I2C function for reading/writing in stream mode. A single address is provided followed by multiple data fields.
//data is updated by reference in readback.
int32_t DLL_EXPORT multiRegStreamRW(        FT_HANDLE dev,
                                            PROT_HANDLE prot,
                                            uint32_t addr,
                                            uint32_t data [],
                                            uint32_t numRegs,
                                            BOOL write
);


// <MapHandles> is used to map an FTDI handle and PROT handle to an index. Once this is setup, indexed functions can be used to simplify calls.
int32_t DLL_EXPORT mapHandles(              FT_HANDLE dev,
                                            PROT_HANDLE prot,
                                            uint32_t deviceID
);

//Indexed Functions. These are used after the handles have been mapped to an index using mapHandles.
//All indexed functions are protocol agnostic as the PROT handle is first checked.
int32_t DLL_EXPORT write_reg(               uint32_t deviceID,
                                            uint32_t reg,
                                            uint32_t data
);

int32_t DLL_EXPORT read_reg(                uint32_t deviceID,
                                            uint32_t reg
);

//Multi register indexed functions. These are also used after the handles have been mapped to an index using mapHandles.
int32_t DLL_EXPORT write_regs(              uint32_t deviceID,
                                            uint32_t reg [],
                                            uint32_t data [],
                                            uint32_t numRegs
);

//Read_regs updates the values in data [] directly instead of returning the values.
int32_t DLL_EXPORT read_regs(               uint32_t deviceID,
                                            uint32_t reg [],
                                            uint32_t data [],
                                            uint32_t numRegs
);

//stream register indexed functions. These are also used after the handles have been mapped to an index using mapHandles.
int32_t DLL_EXPORT write_reg_stream(        uint32_t deviceID,
                                            uint32_t reg,
                                            uint32_t data [],
                                            uint32_t numRegs
);
//Read_regs_stream updates the values in data [] directly instead of returning the values.
int32_t DLL_EXPORT read_reg_stream(         uint32_t deviceID,
                                            uint32_t reg,
                                            uint32_t data [],
                                            uint32_t numRegs
);
//These functions are for toggling GPIO. Note the FTDI BUS will be left in the state of the last byte written until another function that interacts with that BUS is called.
int32_t DLL_EXPORT toggleSingleBit(         FT_HANDLE dev,
                                            uint8_t bitNum,
                                            BOOL toggleLow
);
int32_t DLL_EXPORT writeCustomPattern(      FT_HANDLE dev,
                                            uint8_t MASK,
                                            uint8_t *byteSequence,
                                            int32_t sequenceLen
);
//For freeing up memory if necessary.
int32_t DLL_EXPORT closeFTDI(               FT_HANDLE dev
);

int32_t DLL_EXPORT deleteSPIDev(            PROT_HANDLE spi
);

int32_t DLL_EXPORT deleteI2CDev(            PROT_HANDLE i2c
);
//For Debug purposes. Print patterns to the screen using standard print function. Debug is used for single transactions only and does not work with multi-transactions or streaming.
int32_t enableDebugPrint(                   BOOL writeDB,
                                            BOOL readDB
);

int32_t setDebugDelim(                      char delim
);


#endif // TI_FTDI_INCLUDED

