import ctypes


def getConnectedFTDI(TI_FTDI):
    """Prints to console all connected FTDI devices.

    Args:
        TI_FTDI (DLL Object): TI_FTDI_DLL handle
    """
    # Get all FTDI Device and return list of them for connecting via FTDI EEPROM Description vs serial number:
    flags = ctypes.c_ulong()
    devType = ctypes.c_ulong()
    devID = ctypes.c_ulong()
    locID = ctypes.c_ulong()
    description = ctypes.create_string_buffer(64)
    serial = ctypes.create_string_buffer(64)
    idx = 0
    numDevices = TI_FTDI.getDeviceInfo(idx,ctypes.byref(flags),ctypes.byref(devType),ctypes.byref(devID),ctypes.byref(locID),description,serial,True)
    for idx in range(1,numDevices+1):
        print('\nFlags:',flags.value)
        print('Type:',devType.value)
        print('Device ID:',devID.value)
        print('Location ID:',locID.value)
        print('Description:',description.value)
        print('Serial Number:',serial.value)
        TI_FTDI.getDeviceInfo(idx,ctypes.byref(flags),ctypes.byref(devType),ctypes.byref(devID),ctypes.byref(locID),description,serial,False)


def returnFTDISerialNumber(TI_FTDI,FTDI_description):
    """This function will return the FTDI serial number when passed the FTDI Product Description. 

    Args:
        TI_FTDI (DLL Object): TI_FTDI_DLL handle
        FTDI_description (str): FTDI Product description. This can be found by using FT_Prog and choosing the USB String Descriptors tab for any FT device.

    Returns:
        list: List containing all serial numbers of connected FTDI devices which contain the FTDI_description argument as substring in FTDI Product Description field in EEPROM. 
    """
    flags = ctypes.c_ulong()
    devType = ctypes.c_ulong()
    devID = ctypes.c_ulong()
    locID = ctypes.c_ulong()
    description = ctypes.create_string_buffer(64)
    serial = ctypes.create_string_buffer(64)
    connectedDevices = []
    idx = 0
    numDevices = TI_FTDI.getDeviceInfo(idx,ctypes.byref(flags),ctypes.byref(devType),ctypes.byref(devID),ctypes.byref(locID),description,serial,True) # get first device details
    for idx in range(1,numDevices+1):
        if FTDI_description in description.value.decode():
            connectedDevices.append(serial.value.decode()[:-1]) # Remove bus label
        TI_FTDI.getDeviceInfo(idx,ctypes.byref(flags),ctypes.byref(devType),ctypes.byref(devID),ctypes.byref(locID),description,serial,False) # get next device
    connectedDevices = list(set(connectedDevices)) # Remove duplicates from return list
    return connectedDevices



def adcWriteRegs(TI_FTDI,deviceHandle,addr_list,data_list):
    """Write a list of registers to the ADC

    Args:
        TI_FTDI (DLL Object): TI_FTDI_DLL handle
        addr_list (list of integers): Addresses to write to
        data_list (list of integers): Data values to write into the addresses

    Returns:
        err: DLL_ErrorStatus
    """
    if type(addr_list) == int:
        addr_list = [addr_list]
        data_list = [data_list]
    num_regs = min(len(addr_list),len(data_list))
    addr = (ctypes.c_uint32*num_regs)()  # allocate memory for register writes
    data = (ctypes.c_uint32*num_regs)() # allocate memory for register writes
    for idx in range(0,num_regs):
        addr[idx] = addr_list[idx]
        data[idx] = data_list[idx]
    err = TI_FTDI.write_regs(deviceHandle,addr,data,num_regs)
    return err


def adcReadRegs(TI_FTDI,deviceHandle,addr_list):
    """Read a list of registers from the ADC

    Args:
        TI_FTDI (DLL Object): TI_FTDI_DLL handle
        addr_list (list of integers): Addresses to read

    Returns:
        List: List corresponding to address list containing the value at the address register.
    """
    single_read = False
    if type(addr_list) == int:
        addr_list = [addr_list]
        single_read = True
    num_regs = len(addr_list)
    addr = (ctypes.c_uint32*num_regs)()  # allocate memory for register reads
    data = (ctypes.c_uint32*num_regs)() # allocate memory for register reads
    for idx in range(0,num_regs):
        addr[idx] = addr_list[idx] # load regs into memory
    err = TI_FTDI.read_regs(deviceHandle,addr,data,num_regs)
    data_list = []
    for idx in range(0,num_regs):
        data_list.append(data[idx]) # store register data to new list
    if single_read:
        return data_list[0]
    else:
        return data_list


def writeConfigFile(TI_FTDI,deviceHandle,filename):
    addr_list=[]
    data_list=[]
    length=0
    with open(filename) as file:
        for line in file:
            addr_list.append(int(line.split(',')[0],16))
            data_list.append(int(line.split(',')[1].split(' ')[0],16))
            length+=1
    # Write multiple registers:
    addr = (ctypes.c_uint32*length)()  # allocate memory for register writes
    data = (ctypes.c_uint32*length)() # allocate memory for register writes

    for i in range(0,length):
        addr[i] = addr_list[i]
        data[i] = data_list[i]

    TI_FTDI.write_regs(deviceHandle,addr,data,len(addr))


def convertToString(data):
    """Convert list of integers into a list of hex string

    Args:
        data (list): List of integers

    Returns:
        returned_list: List of hex representation of input integers
    """
    new_data=[]
    for idx in range(0,len(data)):
        new_data.append(hex(data[idx])) # store register data to new list
    return new_data
