/* --COPYRIGHT--,BSD
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

#include <stdint.h>
#include "TI_MSP430.h"
#include "TI_MSP430_hardware_board.h"
#include "TI_MSP430_spi.h"

//******************************************************************************
// If Bit Bang is used for DAC128S085 SPI connection (Default)
//******************************************************************************
#if TI_DAC128S085_SER_INTF == TI_DAC128S085_SPI_BITBANG
uint16_t TI_DAC128S085_SPI_bitbang_inout(uint8_t);
//------------------------------------------------------------------------------
//  void TI_DAC128S085_SPISetup(void)
//
//  DESCRIPTION:
//  Configures the assigned interface to function as a SPI port and
//  initializes it.
//------------------------------------------------------------------------------
void TI_DAC128S085_SPISetup(void)
{
  TI_DAC128S085_CSn_PxOUT |= TI_DAC128S085_CSn_PIN;
  TI_DAC128S085_CSn_PxDIR |= TI_DAC128S085_CSn_PIN;                            // /CS disable

  // Disable special function on bitbang pins
  TI_DAC128S085_SPI_BITBANG_SIMO_PxSEL &= ~TI_DAC128S085_SPI_BITBANG_SIMO;
  TI_DAC128S085_SPI_BITBANG_UCLK_PxSEL &= ~TI_DAC128S085_SPI_BITBANG_UCLK;
  
  // Config bitbang pins: SIMO high and UCLK high
  TI_DAC128S085_SPI_BITBANG_PxOUT |= TI_DAC128S085_SPI_BITBANG_SIMO;
  TI_DAC128S085_SPI_BITBANG_PxOUT |= TI_DAC128S085_SPI_BITBANG_UCLK; 
  
  // Config bitbang pins direction: SIMO, UCLK as OUT & SOMI as IN
  TI_DAC128S085_SPI_BITBANG_PxDIR |= TI_DAC128S085_SPI_BITBANG_SIMO | TI_DAC128S085_SPI_BITBANG_UCLK;
}
//------------------------------------------------------------------------------
// uint16_t TI_DAC128S085_SPI_bitbang_inout(uint16_t dac_val)
// Description: Input 16-bit value using selected bit-bang pins
//------------------------------------------------------------------------------
uint16_t TI_DAC128S085_SPI_bitbang_inout(uint16_t dac_val)
{
  uint16_t x=0,shift=0;
  uint8_t y;

  // Determine how many bit positions SOMI is from least-significant bit
  x = TI_DAC128S085_SPI_BITBANG_SOMI;
  while(x>1)
  {
    shift++;
    x = x >> 1;
  };

  x = 0;

  for(y=16;y>0;y--)
  {
    if(dac_val & 0x8000)                                                       // If bit is high...
      TI_DAC128S085_SPI_BITBANG_PxOUT |= TI_DAC128S085_SPI_BITBANG_SIMO;       // Set SIMO high...
    else
      TI_DAC128S085_SPI_BITBANG_PxOUT &= ~TI_DAC128S085_SPI_BITBANG_SIMO;      // Set SIMO low...
    value = value << 1;                                                        // Rotate bits 
    
    TI_DAC128S085_SPI_BITBANG_PxOUT &= ~TI_DAC128S085_SPI_BITBANG_UCLK;        // Set clock high
    TI_DAC128S085_SPI_BITBANG_PxOUT |= TI_DAC128S085_SPI_BITBANG_UCLK;         // Set clock low   

    x = x << 1;                                                                // Make room for next bit
//    x = x | ((TI_DAC128S085_SPI_BITBANG_PxIN & TI_DAC128S085_SPI_BITBANG_SOMI) >> shift);    
                                                                               // Store next bit
  }
  TI_DAC128S085_SPI_BITBANG_PxOUT |= TI_DAC128S085_SPI_BITBANG_UCLK;           // Set clock high   
  return(x);
}
//------------------------------------------------------------------------------
//  void TI_DAC128S085_SPIWriteReg(uint8_t, uint16_t)
//
//  DESCRIPTION:
//  Reads the adc output
//------------------------------------------------------------------------------
void TI_DAC128S085_SPIWriteReg(uint8_t dac_ch, uint16_t dac_data)
{
  uint16_t wval=0;
  
  TI_DAC128S085_CSn_PxOUT &= ~TI_DAC128S085_CSn_PIN;                           // /CS enable
  dac_ch = 0x0F & dac_ch;
  wval |= dac_ch << 12;
  wval |= (dac_data & 0x0FFF);  
  TI_DAC128S085_SPI_bitbang_inout(wval);                                       // Write data 
  TI_DAC128S085_CSn_PxOUT |= TI_DAC128S085_CSn_PIN;                            // /CS disable  
  
  return x;
}
//------------------------------------------------------------------------------
#endif
//******************************************************************************
// If USCIB0_5xx is used for DAC12x SPI connection  
//******************************************************************************
#if TI_DAC128S085_SER_INTF == TI_DAC128S085_SER_INTF_USCIB0_5xx
//------------------------------------------------------------------------------
//  void TI_DAC128S085_SPISetup(void)
//
//  DESCRIPTION:
//  Configures the assigned interface to function as a SPI port and
//  initializes it.
//------------------------------------------------------------------------------
void TI_DAC128S085_SPISetup(void)
{
  TI_DAC128S085_CSn_PxOUT |= TI_DAC128S085_CSn_PIN;
  TI_DAC128S085_CSn_PxDIR |= TI_DAC128S085_CSn_PIN;                            // /CS disable

  UCB0CTL1 |= UCSWRST;                                                         // **Disable USCI state machine**
  UCB0CTL0 |= UCMST/*+UCCKPH*/+UCMSB+UCSYNC;                                       // 3-pin, 8-bit SPI master 
  UCB0CTL1 |= UCSSEL_2;                                                        // SMCLK
  UCB0BR0 = 0x02;                                                              // UCLK/2
  UCB0BR1 = 0;
  
  TI_DAC128S085_SPI_USCIB0_SIMO_PxSEL |= TI_DAC128S085_SPI_USCIB0_SIMO;        // SPI option select
  TI_DAC128S085_SPI_USCIB0_SOMI_PxSEL |= TI_DAC128S085_SPI_USCIB0_SOMI;
  TI_DAC128S085_SPI_USCIB0_UCLK_PxSEL |= TI_DAC128S085_SPI_USCIB0_UCLK;
                                                                               
  TI_DAC128S085_SPI_USCIB0_SIMO_PxDIR |= TI_DAC128S085_SPI_USCIB0_SIMO;        // SPI TXD out direction
  TI_DAC128S085_SPI_USCIB0_UCLK_PxDIR |= TI_DAC128S085_SPI_USCIB0_UCLK;  
  
  UCB0CTL1 &= ~UCSWRST;                                                        // **Initialize USCI state machine**
}
//------------------------------------------------------------------------------
//  void TI_DAC128S085_SPIWriteReg(uint8_t dac_ch, uint16_t dac_data)
//
//  DESCRIPTION:
//  
//------------------------------------------------------------------------------
void TI_DAC128S085_SPIWriteReg(uint8_t dac_ch, uint16_t dac_data)
{
  
  uint16_t wval=0;
  
  
  wval = 0x0F & dac_ch;
  wval = wval << 12;
  wval |= (0x0FFF & dac_data);  
  
  TI_DAC128S085_CSn_PxOUT &= ~TI_DAC128S085_CSn_PIN;                           // /CS enable 
  
  while (!(UCB0IFG&UCTXIFG));                                                  // Wait for TXBUF ready
  UCB0TXBUF = wval>>8;                                                         // Send ch to read    
  
  while (UCB0STAT & UCBUSY);                                                 // Wait for TX complete
  
  while (!(UCB0IFG & UCTXIFG));                                                // Wait for TXBUF ready
  UCB0TXBUF = wval;                                                            // Send dummy data to read lsb  
  
  while (UCB0STAT & UCBUSY);                                                 // Wait for TX complete
   
  TI_DAC128S085_CSn_PxOUT |= TI_DAC128S085_CSn_PIN;                            // /CS disable
      
}
//------------------------------------------------------------------------------
#endif