"""
Filename: hsdcpro_dll_wrapper_32bit.py
Author: Chase Wood
Title: Applications Engineer
Company: Texas Instruments
Created: November 9, 2023
Version: 0.0.0
Created and tested with Python 3.11 but modified (and slightly tested) for operation with 32-bit version of Python 2.7
"""

import os
import ctypes
import time

TimeoutinMs = 10000

# Open HSDC Pro DLL
hsdc_dll_path = r'C:\Program Files (x86)\Texas Instruments\High Speed Data Converter Pro\HSDCPro Automation DLL\32Bit DLL\HSDCProAutomation.dll' # running in 32 bit mode
HSDC_Pro = ctypes.cdll.LoadLibrary(hsdc_dll_path)


def cstr(msg):
    """
    Encodes a message in utf-8 format.

    Args:
        msg (str): Message to be encoded.

    Returns:
        str: Encoded message.
    """
    return msg.encode('utf-8')

############### HSDC PRO DLL FUNCTIONS ############## :

def minimize_hsdcpro():
    """
    Minimizes HSDC Pro.
    """
    HSDC_Pro.Minimize_HSDCPro(TimeoutinMs)
    time.sleep(0.1)
    return

def restore_hsdcpro():
    """
    Maximizes HSDC Pro.
    """
    HSDC_Pro.Restore_HSDCPro(TimeoutinMs)
    time.sleep(0.1)
    return

def reload_device_ini():
    """
    Reloads the device INI in HSDC Pro.
    """
    HSDC_Pro.Reload_Device_INI(TimeoutinMs)
    time.sleep(0.1)
    return

def save_hsdc_pro_screenshot(file_path,image_format):
    """
    Saves a screenshot of HSDC Pro in specified format.

    Args:
        file_path (str): Path to save image to. Must include filename and extension.
        image_format (int): Format to save image as. 0=BMP, 1=JPEG, 2=PNG.
    """
    HSDC_Pro.Save_HSDC_Pro_Screenshot(cstr(file_path),image_format)
    time.sleep(0.1)
    return

def select_adc_device(ini_file):
    """
    Selects the ADC device INI file in HSDC Pro.

    Args:
        ini_file (str): The INI file for the ADC device.
    """
    HSDC_Pro.Select_ADC_Device(cstr(ini_file),TimeoutinMs)
    time.sleep(0.1)
    return

def pass_adc_output_data_rate(Fs):
    """
    Pass new ADC output data rate to HSDC Pro.

    Args:
        Fs (float): The output data rate for the ADC.
    """
    HSDC_Pro.Pass_ADC_Output_Data_Rate(ctypes.c_double(Fs),TimeoutinMs)
    time.sleep(0.1)
    return

def set_adc_input_target_frequency(Fin):
    """
    Sets the ADC input target frequency in HSDC Pro.

    Args:
        Fin (float): The input target frequency for the ADC.
    """
    HSDC_Pro.Set_ADC_Input_Target_Frequency(ctypes.c_double(Fin),TimeoutinMs)
    time.sleep(0.1)
    return

def set_adc_2nd_input_frequency(Fin):
    """
    Sets the ADC second input frequency in HSDC Pro.

    Args:
        Fin (float): The second input frequency for the ADC.
    """
    HSDC_Pro.Set_ADC_2nd_Input_Frequency(ctypes.c_double(Fin),TimeoutinMs)
    time.sleep(0.1)
    return

def set_number_of_samples(number_of_samples):
    """
    Sets the number of samples per channel in HSDC Pro. Note: HSDC Pro will automatically coerce it to be a multiple of 4096.

    Args:
        number_of_samples (int): The number of samples per channel.
    """
    HSDC_Pro.Set_Number_of_Samples(ctypes.c_uint64(number_of_samples),TimeoutinMs)
    time.sleep(0.1)
    return

def adc_average_settings(num_avg):
    """
    Sets the ADC average settings in HSDC Pro.

    Args:
        num_avg (int): The number of averages. If less than or equal to 1, the average settings are set to 0 and 1.
    """
    if num_avg <= 1:
        HSDC_Pro.ADC_Average_Settings(0,1,TimeoutinMs)
    else:
        HSDC_Pro.ADC_Average_Settings(1,num_avg,TimeoutinMs)
    time.sleep(0.1)
    return

def adc_test_selection(test_selection):
    """
    Switches between the different Test Selection modes in HSDC Pro. Changing the Test Selection also affects the ADC Graph plot type.

    Args:
        test_selection (int): Specifies the Mode (Time Domain-0, Single Tone-1, Two Tone-2, Channel Power-3)
    """
    HSDC_Pro.ADC_Test_Selection(test_selection,TimeoutinMs)
    time.sleep(0.1)
    return

def adc_analysis_window_length(window_size):
    """
    Changes the number of samples to be considered for analysis in HSDC Pro. The analysis window length (number of samples) should be one of the following values: 4096, 8192, 16384, 32768, 65536, 131072, 262144, and 524288. If any other value is passed, it will be coerced to the next highest value in the above list.

    Args:
        window_size (int): The number of samples to be considered for analysis.
    """
    HSDC_Pro.ADC_Analysis_Window_Length(window_size,TimeoutinMs)
    time.sleep(0.1)
    return

def adc_plot_type(plot_type):
    """
    Changes the plot type (Time domain/Frequency domain) of the graph in HSDC Pro. Changing the ADC Plot type also changes the "Test Selection" by switching to Time/Frequency domain.

    Args:
        plot_type (int): ADC Graph plot Type (Codes-0, Bits-1, Real FFT-2, Complex FFT-3)
    """
    HSDC_Pro.ADC_Plot_Type(plot_type,TimeoutinMs)
    time.sleep(0.1)
    return

def fft_window(window_type):
    """
    Changes the FFT windowing technique to be applied to the active channel in HSDC Pro.

    Args:
        window_type (int): The type of FFT window to be viewed in HSDCPro GUI. (Rectangular-0 ,Hamming-1, Hanning-2, Blackman-3)
    """
    HSDC_Pro.FFT_Window(window_type,TimeoutinMs)
    time.sleep(0.1)
    return

def select_adc_channel(channel_idx):
    """
    Changes the active channel of the GUI in HSDC Pro. Note channel index is zero based rather than 1 based as shown in the GUI.

    Args:
        channel_idx (int): ChannelIndex(0-based) - Used to change the active channel in the GUI
    """
    HSDC_Pro.Select_ADC_Channel(channel_idx,TimeoutinMs)
    time.sleep(0.1)
    return

def trigger_option(trigger_mode_enable,software_trigger_enable,arm_on_next_capture_button_press,trigger_clk_delays):
    """
    Sets the Trigger mode in ADC in HSDC Pro. For normal capture, trigger mode enable, software trigger enable and "Arm On Next Capture Button Press, has to be disabled(0). For external trigger, "TriggerModeEnable" has to be enabled(1) and "SoftwareTriggerEnable" and "ArmOnNextCaptureButtonPress" has to be disabled(0). For software trigger, "TriggerModeEnable" and "SoftwareTriggerEnable" has to be enabled(1) and "ArmOnNextCaptureButtonPress" has to be disabled(0).

    Args:
        trigger_mode_enable (int): Enables(1) or disables(0) the trigger mode.
        software_trigger_enable (int): Enables(1) or disables(0) the software trigger mode.
        arm_on_next_capture_button_press (int): Arms the trigger after user presses the Capture Button. Similar to software trigger mode.
        trigger_clk_delays (int): Specifies the number of Clock delays before trigger.
    """
    HSDC_Pro.Trigger_Option(trigger_mode_enable,software_trigger_enable,arm_on_next_capture_button_press,ctypes.c_uint8(trigger_clk_delays),TimeoutinMs)
    return

def pass_capture_event():
    """
    Take new capture in HSDC Pro.
    """
    HSDC_Pro.Pass_Capture_Event(TimeoutinMs)
    time.sleep(0.1)
    return

def generate_software_trigger(wait_to_check_trigger):
    """
    Generates a software trigger in ADC mode. Before this function is called, software trigger mode needs to be set("TriggerModeEnable" and "SoftwareTriggerEnable" should be "1" and "Arm On Next Capture Button Press" have to be "0" ) by using "Trigger Option" function. This function is also used to start the capture in "Arm On Next Capture Button Press" mode ("TriggerModeEnable" is 1, "SoftwareTriggerEnable" is "0" and "Arm On Next Capture Button Press" is "1" ).

    Args:
        wait_to_check_trigger (int): Controls whether to wait and check if the trigger has occurred. If "CheckForTrigger" is disabled("0")", it will not check whether trigger has occurred.
    """
    HSDC_Pro.Generate_Software_Trigger(wait_to_check_trigger,TimeoutinMs)
    time.sleep(0.1)
    return

def read_ddr_memory(wait_to_check_trigger):
    """
    Reads the DDR memory in HSDC Pro.

    Args:
        wait_to_check_trigger (int): Controls whether to wait and check if the trigger has occurred.
    """
    HSDC_Pro.Read_DDR_Memory(wait_to_check_trigger,TimeoutinMs)
    time.sleep(0.1)
    return

def save_fft_as_png(channel_idx, file_path):
    """
    Saves the FFT of single channel as a PNG image in HSDC Pro.

    Args:
        channel_idx (int): The index of the channel to save the FFT from.
        file_path (str): The path to save the PNG image to.
    """
    HSDC_Pro.Save_FFT_As_PNG(channel_idx,cstr(file_path),TimeoutinMs)
    time.sleep(0.1)
    return

def save_raw_data_as_csv(file_path):
    """
    Saves the raw data as a CSV file in HSDC Pro. The CSV file contains the time domain data of all channels with each channel as a separate column.

    Args:
        file_path (str): Specifies the file path (with the file name and CSV extension) on where the raw data (Time domain data) needs to be saved.
    """
    HSDC_Pro.Save_Raw_Data_As_CSV(cstr(file_path),TimeoutinMs)
    time.sleep(0.1)
    return

def save_raw_data_as_binary_file(file_path):
    """
    Saves the raw data as a binary file in HSDC Pro. The binary file contains the time domain data of all channels in interleaved format.

    Args:
        file_path (str): Specifies the file path (with the file name and BIN extension) on where the raw data (Time domain data) needs to be saved.
    """
    HSDC_Pro.Save_Raw_Data_As_Binary_File(cstr(file_path),TimeoutinMs)
    time.sleep(0.1)
    return

def set_adc_bim(lower_hz,upper_hz):
    """
    Sets the ADC BIM in HSDC Pro.

    Args:
        lower_hz (float): The lower frequency for the BIM.
        upper_hz (float): The upper frequency for the BIM.
    """
    HSDC_Pro.Set_ADC_BIM(1,ctypes.c_double(lower_hz),ctypes.c_double(upper_hz),TimeoutinMs)
    time.sleep(0.1)
    return

def time_domain_parameters():
    """
    Gets the values of the required single tone parameter values in HSDC Pro.

    Returns:
        list: List of HSDC Pro parameters corresponding to [Min, Max, St Dev, Mean, Median, RMS, Peak to Peak, PAR]
    """
    output=[]
    HSDCPro_returned_data = (ctypes.c_double*8)() #Initializing the data array for the requested number of parameters
    for i,_ in enumerate(HSDCPro_returned_data): # Set each element in data array to 0.
        HSDCPro_returned_data[i] = 0
    HSDC_Pro.Time_Domain_Parameters((cstr("Min;Max;St Dev;Mean;Median;RMS;Peak to Peak;PAR")),TimeoutinMs,ctypes.byref(HSDCPro_returned_data),8)
    output += HSDCPro_returned_data
    return output

def single_tone_parameters(unit=1):
    """
    Returns all single tone parameters in HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        list: List of HSDC Pro parameters corresponding to [SNR, SFDR, THD, SINAD, ENOB, Fund., Next Spur, HD2, HD3, HD4, HD5, NSD/Hz]
    """
    output=[]
    HSDCPro_returned_data = (ctypes.c_double*12)() #Initializing the data array for the requested number of parameters
    for i,_ in enumerate(HSDCPro_returned_data): # Set each element in data array to 0.
        HSDCPro_returned_data[i] = 0
    HSDC_Pro.Single_Tone_Parameters((cstr("SNR;SFDR;THD;SINAD;ENOB;Fund.;Next Spur;HD2;HD3;HD4;HD5;NSD/Hz")),unit,TimeoutinMs,ctypes.byref(HSDCPro_returned_data),12)
    output += HSDCPro_returned_data
    return output

def two_tone_parameters():
    """This function is used to get the values of the required two tone parameter values.

    Returns:
        list: List of HSDC Pro parameters corresponding to [F1, F2, 2F1+F2, 2F2+F1, 2F1-F2, 2F2-F1, 3F2-2F1, 3F1-2F2, F1+F2, F1-F2, 3F1, 3F2, 2F1+2F2]
    """
    output=[]
    HSDCPro_returned_data = (ctypes.c_double*13)() #Initializing the data array for the requested number of parameters
    for i,_ in enumerate(HSDCPro_returned_data): # Set each element in data array to 0.
        HSDCPro_returned_data[i] = 0
    HSDC_Pro.Two_Tone_Parameters((cstr("F1;F2;2F1+F2;2F2+F1;2F1-F2;2F2-F1;3F2-2F1;3F1-2F2;F1+F2;F1-F2;3F1;3F2;2F1+2F2")),TimeoutinMs,ctypes.byref(HSDCPro_returned_data),13)
    output += HSDCPro_returned_data
    return output

def get_adc_device():
    """
    Gets the ADC device in HSDC Pro.

    Returns:
        str: The current ADC device.
    """
    current_adc = (ctypes.c_char*256)()
    HSDC_Pro.Get_ADC_Device(256,TimeoutinMs,current_adc)
    return current_adc.value.decode('utf-8')

def adc_sample_offset(sample_starting_index):
    """
    Sets the ADC sample offset in HSDC Pro.

    Args:
        sample_starting_index (int): The starting index for the samples.
    """
    HSDC_Pro.ADC_Sample_Offset(sample_starting_index,TimeoutinMs)
    time.sleep(0.1)
    return

def get_number_of_samples_per_channel():
    """
    Gets the number of samples per channel in HSDC Pro. This is not the length of the analysis window.

    Returns:
        int: The number of samples per channel.
    """
    ptr1 = ctypes.c_uint64()
    HSDC_Pro.Get_Number_of_Samples_Per_Channel(ctypes.byref(ptr1),TimeoutinMs)
    return ptr1.value

def get_active_adc_channel():
    """
    Gets the active ADC channel in the HSDC Pro GUI.

    Returns:
        int: The active channel (indexed to 0).
    """
    active_channel_idx = ctypes.c_uint16()
    HSDC_Pro.Get_Active_ADC_Channel(ctypes.byref(active_channel_idx),TimeoutinMs)
    return active_channel_idx.value

def set_adc_auto_calculation_of_coherent_frequencies(use_coherent):
    """
    Enables or disables the coherent calculation of frequencies in HSDC Pro.

    Args:
        use_coherent (int): 0 to disable, 1 to enable.
    """
    HSDC_Pro.Set_ADC_Auto_Calculation_Of_Coherent_Frequencies(use_coherent,TimeoutinMs)
    time.sleep(0.1)
    return

def set_additional_device_parameters_single_tone(Fs,Fin,NCO_frequency,decimation_factor):
    """This function can be used to set the value of the Additional Device Parameters in HSDC Pro when the Test Selection is 'Two Tone'.

    Args:
        Fs (float): Sample rate (in Hz)
        Fin (float): Input frequency (in Hz)
        NCO_frequency (float): NCO frequency (in Hz)
        decimation_factor (int): Decimation factor.
    """
    HSDC_Pro.Set_AdditionalFrequencyParameters(1,ctypes.c_double(Fs),ctypes.c_double(Fin),ctypes.c_double(NCO_frequency),ctypes.c_double(decimation_factor),TimeoutinMs)
    time.sleep(0.1)
    return

def set_additional_device_parameters_two_tone(Fs,Fin1,Fin2,NCO_frequency,decimation_factor):
    """This function can be used to set the value of the Additional Device Parameters in HSDC Pro when the Test Selection is 'Two Tone'.

    Args:
        Fs (float): Sample rate (in Hz)
        Fin1 (float): Input frequency 1 (in Hz)
        Fin2 (float): Input frequency 2 (in Hz)
        NCO_frequency (float): NCO frequency (in Hz)
        decimation_factor (int): Decimation factor.
    """
    HSDC_Pro.Set_AdditionalFrequencyParametersForTwoTone(1,ctypes.c_double(Fs),ctypes.c_double(Fin1),ctypes.c_double(Fin2),ctypes.c_double(NCO_frequency),ctypes.c_double(decimation_factor),TimeoutinMs)
    time.sleep(0.1)
    return

def set_unit_and_nyquist_for_two_tone(unit,nyquist):
    """Sets the unit and nyquist region when using two tone measurement.

    Args:
        unit (int): Specifies the required unit. Available options: dBFs (0), Hz (1).
        nyquist (int): Specifies the required Nyquist region. Available options: First Nyquist (0), Second Nyquist (1), Both (2).
    """
    HSDC_Pro.Set_UnitAndNyquistForTwoTone(unit,nyquist,TimeoutinMs)
    return

def set_phase_unit(unit):
    """Selects degrees or radians as the phase unit.

    Args:
        unit (int): Specifies the required unit. Available options: Radian (0), Degree (1).
    """
    HSDC_Pro.Set_Phase_Unit(unit,TimeoutinMs)
    return

def adc_graph_set_x_scale_range(min,max):
    """Sets the x-scale range for the ADC graph.

    Args:
        min (float): The minimum value for the x-scale.
        max (float): The maximum value for the x-scale.
    """
    HSDC_Pro.ADC_Graph_Set_X_Scale_Range(0,ctypes.c_double(min),ctypes.c_double(max),TimeoutinMs)
    time.sleep(0.1)
    return

def adc_graph_set_y_scale_range(min,max):
    """Sets the y-scale range for the ADC graph.

    Args:
        min (float): The minimum value for the y-scale.
        max (float): The maximum value for the y-scale.
    """
    HSDC_Pro.ADC_Graph_Set_Y_Scale_Range(0,ctypes.c_double(min),ctypes.c_double(max),TimeoutinMs)
    time.sleep(0.1)
    return

def export_unwrap_waveform_as_csv(file_path):
    """Exports the unwrapped waveform as a CSV file.

    Args:
        file_path (str): The path where the CSV file will be saved.
    """
    adc_test_selection(0)
    HSDC_Pro.Export_Unwrap_Waveform_As_CSV(cstr(file_path),TimeoutinMs)
    return

def set_phase_unit(unit):
    """Selects degrees or radians as the phase unit.

    Args:
        unit (int): Specifies the required unit. Available options: Radian (0), Degree (1).
    """
    HSDC_Pro.Set_Phase_Unit(unit,TimeoutinMs)
    return

def adc_graph_set_x_scale_range(min,max):
    """Sets the x-scale range for the ADC graph.

    Args:
        min (float): The minimum value for the x-scale.
        max (float): The maximum value for the x-scale.
    """
    HSDC_Pro.ADC_Graph_Set_X_Scale_Range(0,ctypes.c_double(min),ctypes.c_double(max),TimeoutinMs)
    time.sleep(0.1)
    return

def adc_graph_set_y_scale_range(min,max):
    """Sets the y-scale range for the ADC graph.

    Args:
        min (float): The minimum value for the y-scale.
        max (float): The maximum value for the y-scale.
    """
    HSDC_Pro.ADC_Graph_Set_Y_Scale_Range(0,ctypes.c_double(min),ctypes.c_double(max),TimeoutinMs)
    time.sleep(0.1)
    return

def export_unwrap_waveform_as_csv(file_path):
    """Exports the unwrapped waveform as a CSV file.

    Args:
        file_path (str): The path where the CSV file will be saved.
    """
    adc_test_selection(0)
    HSDC_Pro.Export_Unwrap_Waveform_As_CSV(cstr(file_path),TimeoutinMs)
    return

########################### CUSTOM FUNCTIONS ###########################:
def get_target_freq(twoTone=False):
    ptr1 = ctypes.c_double()
    ptr2 = ctypes.c_double()
    HSDC_Pro.Get_ADC_Input_Target_Frequencies(ctypes.byref(ptr1),ctypes.byref(ptr2),TimeoutinMs)
    if twoTone:
        return [ptr1.value,ptr2.value]
    else:
        return ptr1.value

def get_snr(unit=1):
    """Get 'SNR' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'SNR' in specified unit.
    """
    return single_tone_parameters(unit)[0]

def get_sfdr(unit=1):
    """Get 'SFDR' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'SFDR' in specified unit.
    """
    return single_tone_parameters(unit)[1]

def get_thd(unit=1):
    """Get 'THD' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'THD' in specified unit.
    """
    return single_tone_parameters(unit)[2]

def get_sinad(unit=1):
    """Get 'SINAD' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'SINAD' in specified unit.
    """
    return single_tone_parameters(unit)[3]

def get_enob(unit=1):
    """Get 'ENOB' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'ENOB' in specified unit.
    """
    return single_tone_parameters(unit)[4]

def get_fund(unit=1):
    """Get 'Fund.' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'Fund.' in specified unit.
    """
    return single_tone_parameters(unit)[5]

def get_spur(unit=1):
    """Get 'Next Spur' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'Next Spur' in specified unit.
    """
    return single_tone_parameters(unit)[6]

def get_hd2(unit=1):
    """Get 'HD2' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'HD2' in specified unit.
    """
    return single_tone_parameters(unit)[7]

def get_hd3(unit=1):
    """Get 'HD3' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'HD3' in specified unit.
    """
    return single_tone_parameters(unit)[8]

def get_hd4(unit=1):
    """Get 'HD4' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'HD4' in specified unit.
    """
    return single_tone_parameters(unit)[9]

def get_hd5(unit=1):
    """Get 'HD5' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'HD5' in specified unit.
    """
    return single_tone_parameters(unit)[10]

def get_nsd(unit=1):
    """Get 'NSD/Hz' from HSDC Pro.

    Args:
        unit (int, optional): Sets the units. 0=dBc, 1=dBFs, 2=Hz. Defaults to 1.

    Returns:
        float: 'NSD/Hz' in specified unit.
    """
    return single_tone_parameters(unit)[11]

def disable_additional_device_parameters():
    HSDC_Pro.Set_AdditionalFrequencyParameters(0,0,0,0,1,TimeoutinMs)
    return

def enable_fundamental_freq_search():
    """Enables fundamental frequency search in FFT view.
    """
    HSDC_Pro.Fundamental_Freq_Search(1,TimeoutinMs)
    return

def disable_fundamental_freq_search():
    """Disables fundamental frequency search in FFT view.
    """
    HSDC_Pro.Fundamental_Freq_Search(0,TimeoutinMs)
    return

def enable_nco_bits(nco_resolution):
    """This function is used to enable and set the value of the #NCO Bits. Additional Device Parameters and Auto Calculation of Coherent Frequencies must be enabled or this will not work. 

    Args:
        nco_resolution (int): Number of bits in NCO. (Ex. 16, 32, 48)
    """
    HSDC_Pro.Set_NCOBits(1,nco_resolution,TimeoutinMs)
    return

def disable_nco_bits():
    """This function is used to disable the #NCO Bits function.
    """
    HSDC_Pro.Set_NCOBits(0,0,TimeoutinMs)
    return

def enable_harmonic_freq_search():
    """Enables harmonic frequency search in FFT view.
    """
    HSDC_Pro.Harmonic_Freq_Search(1,TimeoutinMs)
    return

def disable_harmonic_freq_search():
    """Disables harmonic frequency search in FFT view.
    """
    HSDC_Pro.Harmonic_Freq_Search(0,TimeoutinMs)
    return