import ftd2xxController #* MUST IMPORT FOR STANDALONE USE LIKE THIS: ftd2xxController. 
# from bin.APIs.ftd2xx_controller import ftd2xxController #* MUST IMPORT FROM CORRECT LOCATION IF CALLING FROM HIGHER LEVEL SCRIPT, LIKE THIS: from bin.APIs.ftd2xx_controller import ftd2xxController

class EEPROM():
    def __init__(self, i2c_device:ftd2xxController.ftdRegCtrl):
        self.i2c_handle = i2c_device
        return
    
    def convert_to_integer_array(self,data):
        '''Converts data to an integer array.

        Args:
            data (str/char_array/int): Can be any of the following: string, character array, single integer

        Returns:
            list: Data converted as unicode representation in integer list.
        '''
        if type(data)==int:
            convertedData=[data] # convert to list of int
        if type(data)==list or type(data)==str: # convert list of characters or string
            if type(data[0])!=int: # if not already integer array:
                # convert to array of integers by taking ord()
                tempList=[]
                for each in data:
                    tempList.append(ord(each))
                convertedData = tempList
            else:
                convertedData = data # do not change array, it is already an integer array
        return convertedData 

    def write_memory(self, startIndex : int=0, data : str = ''):
        '''Write a user-specified selection of registers into this 8K EEPROM.

        Args:
            startIndex (int): Beginning location of the data to written.
            data (str): String of data to be written into EEPROM.
        '''
        intList = self.convert_to_integer_array(data)
        # old method writing single regsiters at a time
        for index in range(0,len(intList)):
            self.i2c_handle.write(startIndex+index,intList[index])

        # numBytes = len(intList) 

        # # must use uint32
        # if numBytes>1:
        #     num_chunks = numBytes/32-(numBytes%32)/32 # essential the same as np.floor(numBytes/32)
        #     for idx in range(0,int(num_chunks)): # bulk 32 byte reads in this loop:
        #         values = intList[idx*32:idx*32+32]
        #         mem_allocation=(ctypes.c_uint32*32)(*values)
        #         self.streamWrite(startIndex+(32*idx),mem_allocation)
        # else:
        #     self.write(startIndex,intList[0])
        # return

    def reset_memory(self,startIndex : int=0, numBytes : int = 0, value : int=0xFF):
        '''Sequentially reset the EEPROM memory with a user-specified character/value.

        Args:
            startIndex (int): Beginning location of data to overwrite.
            numBytes (int): Number of EEPROM registers to overwrite.
            value (int, optional): User specified value to reset into the EEPROM. Defaults to 0xFF (Unicode representation = "ÿ").
        '''
        # dataRange = ''.join([chr(value)]*numBytes)
        # self.writeMemory(startIndex,dataRange)

        # old method
        for index in range(0,numBytes):
            self.write_memory(startIndex+index, chr(value)) # chr() is used to converter an integer or hexadecimal value into its unicode representation
        return    

    def read_memory(self, startIndex : int=0,numBytes : int=0):
        '''Read a user-specified selection of registers from this 8K EEPROM.

        Args:
            startIndex (int): Beginning location of the data to read.
            numBytes (int): Number of EEPROM registers to read.

        Returns:
            list: Sequential data in EEPROM from user-specified register as integer list.
        '''

        self.i2c_handle.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        # OLD
        # memDump = [] # Create an empty list
        # for index in range(startIndex, startIndex+numBytes):
        #     memDump.append(self.read(index))
        # return memDump

        memory_dump = [] # Create an empty list
        if numBytes>1:
            num_chunks = numBytes/32-(numBytes%32)/32 # essential the same as np.floor(numBytes/32)
            for idx in range(0,int(num_chunks)): # bulk 32 byte reads in this loop:
                data = self.i2c_handle.stream_read(startIndex+(32*idx),32)
                for j in data:
                    memory_dump.append(j)
            num_remaining_bytes = int(numBytes%32)
            if num_remaining_bytes > 0: # if there are any incomplete 32 byte reads needed:
                data = self.i2c_handle.stream_read(startIndex+(32*num_chunks),num_remaining_bytes)
                for j in data:
                    memory_dump.append(j)
        else:
            for index in range(startIndex, startIndex+numBytes):
                memory_dump.append(self.i2c_handle.read(index))
        return memory_dump        

    def read_all_memory(self):
        '''Read every register in this 8K EEPROM.

        Returns:
            list: Data in EEPROM from register 0x0000 through 0x8191 as integer list.
        '''
        self.i2c_handle.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        memDump = [] # Create empty list
        for index in range(0, 2**13): # 0 -> 8191
            memDump.append(self.i2c_handle.read(index)) 
        return memDump

    def read_board_details(self):
        '''Read the memory programmed into EEPROM by contract manufacturer (SVTronics/Krypton/other).

        Returns:
            list: Sequential data in EEPROM of length equal to the sum of {val(0x0)*1000 + val(0x1)*100 + val(0x2)*10 + val(0x3)}
        '''
        lengthOfData = self.read_EEPROM_length() # find out how many bytes of data to read from EEPROM
        return self.read_memory(startIndex=0x4,numBytes=lengthOfData)

    def write_next_available_memory(self,data : str='',resetPadding : int=10):
        '''Writes data at next available register in EEPROM without erasing previous data. Updates the first four data length registers (0x0-0x03)

        Args:
            data (str): String of data to be written into EEPROM.
            resetPadding (int, optional): Amount of extra registers to reset in EEPROM at end of data before writing with new data. Defaults to 10.
        '''
        lengthOfData = self.read_EEPROM_length() # find out how many bytes of data are currently programmed in EEPROM
        newDataIndex = lengthOfData+4 # account for first 4 bytes
        self.reset_memory(newDataIndex, len(data)+resetPadding)
        self.write_memory(newDataIndex, data)
        self.update_data_length(lastIndex=newDataIndex+len(data))
        return

    def add_key_value_pairs(self, newDict : dict = {}):
        '''This function will add the dictionary's key/value pairs to EEPROM at next available memory without over-writing any existing data.

        Args:
            newDict (dict): Dictionary containing additional key/value pairs to write to EEPROM.
        '''
        dataList=[]
        for EVM_param in newDict.keys(): 
            # Add the parameter name/field:
            for charValue in EVM_param:
                dataList.append(ord(charValue))
            # Add additional characters
            dataList.append(ord(':')) # Add ":"
            # Add the parameter value:
            for charValue in newDict[EVM_param]:
                dataList.append(ord(charValue))
            # Add newline character for parsing
            dataList.append(92) # Add backslash (decimal 92)
            dataList.append(ord('n')) # Add "n"
        self.write_next_available_memory(self.convert_to_string(dataList))
        return

    def to_digits(self,value : int = 0):
        '''Takes any number and breaks down into thousandths, hundredths, tens, and ones places.

        Args:
            val (int): Integer

        Returns:
            int: thousandths
            int: hundredths
            int: tens
            int: ones
        '''
        try:
            thousandths = int(str(value)[0])
            hundredths = int(str(value)[1])
            tens = int(str(value)[2])
            ones = int(str(value)[3])
        except IndexError:
            try:
                thousandths = 0
                hundredths = int(str(value)[0])
                tens = int(str(value)[1])
                ones = int(str(value)[2])
            except IndexError:
                try:
                    thousandths = 0
                    hundredths = 0
                    tens = int(str(value)[0])
                    ones = int(str(value)[1])
                except IndexError:
                    try:
                        thousandths = 0
                        hundredths = 0
                        tens = 0
                        ones = int(str(value)[0])
                    except:
                        pass
        return thousandths, hundredths, tens, ones

    def update_data_length(self,lastIndex):
        '''Updates the first 4 registers to reflect new data length.

        Args:
            lastIndex (int): New position of last register of value in EEPROM.
        '''
        thousandths,hundredths,tens,ones = self.to_digits(lastIndex-4)
        self.write_memory(0x0000,chr(thousandths+48))
        self.write_memory(0x0001,chr(hundredths+48))
        self.write_memory(0x0002,chr(tens+48))
        self.write_memory(0x0003,chr(ones+48))
        return        


    def convert_to_string(self,data : list = []):
        '''Converts integer array into it's unicode string.

        Args:
            data (list): A list of integers.

        Returns:
            str: Representation of the input data list as single unicode string.
        '''
        string = ""
        for index in data:
            string += chr(index)
        return string


    def embed_length_into_data(self,data : list = []):
        '''Insert the length of the data list at the front of the data list. Length will be the first 4 elements of returned data.

        Args:
            data (list): A list of integers.

        Returns:
            list: A list of integers with length embedded at first 4 elements. All existing elements are shifted by 4 indexes.
        '''
        thousandths,hundredths,tens,ones = self.to_digits(len(data))
        # + 48 turns this pythonic integer into the unicode value.
        # This is to retain singular data type into or out of the EEPROM as unicode rather than mixing unicode/integer data types.
        data.insert(0,ones+48)
        data.insert(0,tens+48)
        data.insert(0,hundredths+48)
        data.insert(0,thousandths+48)
        return data


    def read_EEPROM_length(self):
        '''Read the first 4 registers of EEPROM memory and return length as integer.

        Returns:
            int: Length of EEPROM memory.
        '''
        # I know this can be done better by using only 2 registers and binary format, however I want to stick with unicode representation for data as entire rest of memory is like this...
        self.i2c_handle.read(0x0) # to avoid the bug of first readback after FTDI connection being -2^31.
        lengthOfData = chr(self.i2c_handle.read(0x0))
        lengthOfData += chr(self.i2c_handle.read(0x1))
        lengthOfData += chr(self.i2c_handle.read(0x2))
        lengthOfData += chr(self.i2c_handle.read(0x3))
        lengthOfData = int(lengthOfData) # Convert concatenated string into an integer
        
        return lengthOfData


    def mfr_programming_function(self,evmParamsDict : dict = {},resetPadding : int = 0):
            '''Write the user-specified dictionary into EEPROM. Dictionary data will start at EEPROM index 0x0004 a

            Args:
                evmParamsDict (dict): Dictionary of EVM specific parameters to be programmed into EEPROM.
                resetPadding (int, optional): Amount of extra registers to reset in EEPROM at end of data before writing with new data. Defaults to 0.
            '''
            # Building single string to write into EEPROM:
            dataList = []
            for EVM_param in evmParamsDict.keys(): 
                # Add the parameter name/field:
                for char_val in EVM_param:
                    dataList.append(ord(char_val))
                # Add additional characters
                dataList.append(ord(':')) # Add ":"
                # Add the parameter value:
                for char_val in evmParamsDict[EVM_param]:
                    dataList.append(ord(char_val))
                # Add newline character for parsing
                dataList.append(92) # Add backslash (decimal 92)
                dataList.append(ord('n')) # Add "n"
            dataList = self.embed_length_into_data(dataList) # inserts the length into the first 4 registers
            self.reset_memory(0x0000,len(dataList)+resetPadding)
            self.write_memory(0x0000,self.convert_to_string(dataList))
            return
    
    def print_eeprom_contents(self):
        data = self.convert_to_string(self.read_board_details()).strip()
        split_data = data.split(r'\n')[:-1]
        for each in split_data:
            print(each)
        return
    
if __name__ == '__main__':
    #* Example usage:
    ftdi_serial_number = 'FT8OI7GY'
    ftdi_bus_c = ftdi_serial_number + 'C'
    i2c_rate = 320e3 # 320e3 == 400kHz I2C
    dev_addr = 0x50

    _i2c_ftdi_obj = ftd2xxController.open_FT(ftdi_bus_c,baudRate=int(i2c_rate))
    _eeprom_ftdRegCtrl_obj =_i2c_ftdi_obj.init_I2C(0,1,2,dev_addr,16,8)
    eeprom = EEPROM(_eeprom_ftdRegCtrl_obj)

    eeprom.reset_memory(0,40)
    eeprom.write_memory(0,'The intergalactic EEPROM alliance.')
    print(eeprom.convert_to_string(eeprom.read_memory(0,40)))

    # SVT or Krypton will fill out (some of) these parameters using an ultra-lightweight EVM specific ROM programming GUI or through test procedure python script:
    EVM_param_fields = ['Device'   , 'Lot # '    , 'PG'   , 'CM Vendor', 'WO # '   , 'Build Date', 'EVM Board Revision', 'BOM Variant', 'Test Eng ID # ', 'Test Revision']
    EVM_param_values = ['DAC39RF10', '2255285CD2', 'PG1.4', 'SVTronics', '64589745', '11/18/2022', 'Rev. C'            , '-001'       , 'a0490213'      , 'Rev. A2']
    
    # # IIRC, a dictionary is used for everything in the GUI, otherwise, just simply zip some lists into a new dictionary and pass into function 
    data_dict = dict(zip(EVM_param_fields,EVM_param_values)) # Zip param field list and param value list into a dictionary

    eeprom.mfr_programming_function(data_dict)
    eeprom.print_eeprom_contents()
