# -*- coding: utf-8 -*-
'''FTD2xx Controller'''
'''
███████╗████████╗██████╗ ██████╗ ██╗  ██╗██╗  ██╗     ██████╗ ██████╗ ███╗   ██╗████████╗██████╗  ██████╗ ██╗     ██╗     ███████╗██████╗ 
██╔════╝╚══██╔══╝██╔══██╗╚════██╗╚██╗██╔╝╚██╗██╔╝    ██╔════╝██╔═══██╗████╗  ██║╚══██╔══╝██╔══██╗██╔═══██╗██║     ██║     ██╔════╝██╔══██╗
█████╗     ██║   ██║  ██║ █████╔╝ ╚███╔╝  ╚███╔╝     ██║     ██║   ██║██╔██╗ ██║   ██║   ██████╔╝██║   ██║██║     ██║     █████╗  ██████╔╝
██╔══╝     ██║   ██║  ██║██╔═══╝  ██╔██╗  ██╔██╗     ██║     ██║   ██║██║╚██╗██║   ██║   ██╔══██╗██║   ██║██║     ██║     ██╔══╝  ██╔══██╗
██║        ██║   ██████╔╝███████╗██╔╝ ██╗██╔╝ ██╗    ╚██████╗╚██████╔╝██║ ╚████║   ██║   ██║  ██║╚██████╔╝███████╗███████╗███████╗██║  ██║
╚═╝        ╚═╝   ╚═════╝ ╚══════╝╚═╝  ╚═╝╚═╝  ╚═╝     ╚═════╝ ╚═════╝ ╚═╝  ╚═══╝   ╚═╝   ╚═╝  ╚═╝ ╚═════╝ ╚══════╝╚══════╝╚══════╝╚═╝  ╚═╝                                                                                                                                              
'''

import ctypes
import os
import time
import random
import string

# Use TI_FTDIx64.dll located in this directory's bin folder:
dirPath = os.path.dirname(os.path.abspath(__file__))
fileName = dirPath + "\\bin\\TI_FTDI_DLL\\x64\\TI_FTDIx64.dll"

# Showing how to use an absolute path for specific TI_FTDIx64.dll version:
# fileName = "C:/TI_FTDI_x64/x64/TI_FTDIx64.dll"

ctypes.windll.kernel32.SetDllDirectoryW(None)

microCtrl = ctypes.cdll.LoadLibrary(fileName)
microCtrl = ctypes.CDLL(fileName)
mapValue = [-1]              # mutable varaible that allows user not to have to explicitly call out the mapping function

def version():
    """
    v1.0.1: Initial commit.\n
    v1.1.0: Introduces functions for programming and erasing FTDI Device EEPROM as well as other helpful functions:\n
        - get_ft_index_by_serial_number_and_port()\n
        - get_ft_indexes_by_product_description()\n
        - get_serial_numbers_by_product_description()\n
        - get_info_blank_devices()\n
        - get_num_blank_devices()\n
        - get_index_blank_devices()\n
        - __generate_random_string()\n
        - program_blank_device()\n
        - read_ft_eeprom()\n
        - cycle_port()\n
        - erase_ft_eeprom()
    """
    return 'v1.1.0'

def connected_devices(bool,allDevices):
    '''scan for connected devices and return # found

    Args:
        bool (boolean): print to screen device serial numbers

    Returns:
        int: # of devices found
    '''
    return microCtrl.printConnectedDevices(bool,allDevices)

def create_device_info_list():
    '''Create the internal device info list and return the number of devices

    Raises:
        RuntimeError: raises a runtime error and returns the in from the dll

    Returns:
        list: serial numbers of connected devices
    '''    

    devCount = connected_devices(False,True)
    if devCount:
        get_device_info_detail(0,True)

    return devCount

def get_device_info_detail(index,bool=False):
    """Get device information based on FT index location.

    Args:
        index (int): FT index location

    Returns:
        dict: Dictionary containing the following keys: 'index', 'flags', 'type', 'id', 'location', 'serial', 'description'
    """
    deviceInfo = {}
    flags = ctypes.c_ulong()
    type = ctypes.c_ulong()
    devID = ctypes.c_ulong()
    locID = ctypes.c_ulong()
    description = ctypes.create_string_buffer(64)
    serial = ctypes.create_string_buffer(64)    

    val = microCtrl.getDeviceInfo(index, ctypes.byref(flags), ctypes.byref(type), ctypes.byref(devID), ctypes.byref(locID), description, serial, bool)
    deviceInfo.update({'index' : index, 'flags' : flags.value,'type': type.value,'id': devID.value,'location': locID.value,'serial': serial.value,'description': description.value})

    return deviceInfo

def get_num_connected_devices():
    """By returning the create_device_info_list function, we return the number of connected devices."""
    return create_device_info_list()

def print_all_connected_devices():
    """
    Print information about all connected devices.

    This function retrieves the number of connected devices and prints it. 
    Then it retrieves and prints the information for each connected device.

    Prints:
        The number of connected devices and the information for each connected device.

    Returns:
        None
    """
    num_connected_devices = get_num_connected_devices()
    print(f'There are {num_connected_devices} connected devices.')
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        print(deviceInfo)
    return

def get_ft_index_by_serial_number_and_port(serial_number,port):
    """
    Get FT index by serial number and port.

    This function iterates over the connected devices and returns the FT index of the device that matches the provided serial number and port.
    If no device matches, the function returns None.

    Args:
        serial_number (str): The serial number of the device.
        port (str): The port of the device.

    Returns:
        int: The FT index of the device if found, None otherwise.
    """
    num_connected_devices = get_num_connected_devices()
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        if deviceInfo['serial'].decode() == serial_number+port:
            return deviceInfo['index']
    return

def get_ft_indexes_by_product_description(description):
    """
    Get FT indexes by product description.

    This function iterates over the connected devices and returns a list of FT indexes 
    for the devices that contain the provided description in their product description.

    Args:
        description (str): The description to search for in the product descriptions of the devices.

    Returns:
        list: A list of FT indexes for the devices that contain the provided description in their product description.
    """
    num_connected_devices = get_num_connected_devices()
    product_desc_indexes = []
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        if description in deviceInfo['description'].decode():
            product_desc_indexes.append(deviceInfo['index'])
    return product_desc_indexes

def get_serial_numbers_by_product_description(description):
    """
    Get serial numbers by product description.

    This function iterates over the connected devices and returns a list of serial numbers 
    and their corresponding FT indexes for the devices that contain the provided description 
    in their product description. The list is sorted by the FT index in ascending order.

    Args:
        description (str): The description to search for in the product descriptions of the devices.

    Returns:
        list: A list of lists where each sublist contains the serial number and the FT index 
              of a device that contains the provided description in its product description. 
              The list is sorted by the FT index in ascending order.
    """
    num_connected_devices = get_num_connected_devices()
    serial_nums = []
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        if description in deviceInfo['description'].decode():
            serial_nums.append([deviceInfo['serial'].decode(),deviceInfo['index']])
    sorted_serial_nums_by_index = sorted(serial_nums, key=lambda x: x[1])
    return sorted_serial_nums_by_index

def get_info_blank_devices():
    """
    Get information about blank devices.

    This function iterates over the connected devices and returns a dictionary containing 
    the number of blank devices and their corresponding FT indexes. A device is considered 
    blank if its description contains 'Quad RS232-HS' or if both its description and serial 
    number are empty.

    Returns:
        dict: A dictionary containing the following keys:
              - 'num_blank_devices': The number of blank devices.
              - 'indexes': A list of FT indexes for the blank devices.
    """
    blank_device_info = {}
    num_connected_devices = get_num_connected_devices()
    blank_device_indexes=[]
    for each in range(num_connected_devices):
        deviceInfo = get_device_info_detail(each)
        if b'Quad RS232-HS' in deviceInfo['description']:
            blank_device_indexes.append(each)
        elif deviceInfo['description'] == b'' and deviceInfo['serial'] == b'':
            blank_device_indexes.append(each)
    blank_device_info.update({'num_blank_devices':len(blank_device_indexes),'indexes':blank_device_indexes})
    return blank_device_info

def get_num_blank_devices():
    """Returns number of blank devices."""
    return get_info_blank_devices()['num_blank_devices']

def get_index_blank_devices():
    """Returns a list of indexes of the blank devices."""
    return get_info_blank_devices()['indexes']

def __generate_random_string(length, percentage_numbers):
    """
    Generate a random string of a specified length.

    This function generates a random string of a specified length, with a specified percentage of numbers. 
    The string does not contain the letters 'I', 'O', and 'Z' to avoid confusion with numbers. 
    The first character of the string is always a number.

    Args:
        length (int): The length of the string to generate.
        percentage_numbers (float): The percentage of numbers in the string.

    Returns:
        str: A random string of the specified length, with the specified percentage of numbers.
    """
    letters = string.ascii_letters.replace('I', '').replace('O', '').replace('Z', '') # remove letters that look like numbers
    digits = '123456789'
    
    num_numbers = int(length * percentage_numbers)
    num_letters = length - num_numbers
    
    # Ensure at least one number for the first character
    if num_numbers == 0:
        num_numbers = 1
        num_letters -= 1
    
    chosen_numbers = ''.join(random.choice(digits) for _ in range(num_numbers))
    chosen_letters = ''.join(random.choice(letters) for _ in range(num_letters))
    
    combined = chosen_numbers[1:] + chosen_letters
    random_combined = ''.join(random.sample(combined, len(combined)))
    
    # Make sure the first character is a number
    return chosen_numbers[0] + random_combined

def program_blank_device(description:str,serial_number_prefix:str='TI',serial_number:str=None):
    """
    Program a blank device with a description and a serial number.

    This function opens a blank device, generates a serial number if not provided (does not use TI_FTDI_DLL auto generation feature), 
    programs the EEPROM of the device with the description and serial number, and then cycles the port.

    Args:
        description (str): The description to program into the device.
        serial_number_prefix (str, optional): The prefix for the serial number. Defaults to 'TI'.
        serial_number (str, optional): The serial number to program into the device. 
                                        If None, a random serial number is generated. Defaults to None.

    Raises:
        ConnectionError: If there is no blank FTDI device connected to program.

    Prints:
        Information about the programmed device and its readiness for use.

    Returns:
        None
    """
    time.sleep(4) # allow time for ports to cycle if new device has just been connected via usb.
    create_device_info_list() # update device info list
    ftDevice = ctypes.c_void_p()
    num_ports = ctypes.c_void_p()
    status = microCtrl.openBlankDevice(ctypes.byref(ftDevice),ctypes.byref(num_ports))

    number_blank_devices = get_num_blank_devices()
    if number_blank_devices == 0:
        raise ConnectionError('There is no blank FTDI device connected to program.')

    if serial_number == None: # only generate serial number if serial_number arg is left as None
        serial_number = serial_number_prefix+__generate_random_string(length=6,percentage_numbers=0.75) # 6 digits long (after the serial_number_prefix and then 75% of characters will be numbers

    description_in = ctypes.create_string_buffer(64)
    serialNumber_in = ctypes.create_string_buffer(64)
    description_in.value = description.encode()
    serialNumber_in.value = serial_number.encode()
    status = microCtrl.programEEPROM_TI(ftDevice,ctypes.byref(description_in),ctypes.byref(serialNumber_in),False) # Never let autoSerial flag high.
    cycle_port(ftDevice=ftDevice) # cycle port after programming
    print(f"\nBlank FTDI device has been programmed with:\n⁍ Serial number: {serial_number}\n⁍ Product description: {description}\nPort has been cycled and device is ready for use.")
    return

def read_ft_eeprom(ftDevice):
    """
    Read the EEPROM of a device.

    This function reads the manufacturer, description, and serial number from the EEPROM of a device.

    Args:
        ftDevice (c_void_p): The device to read the EEPROM from.

    Returns:
        dict: A dictionary containing the following keys:
              - 'manufacturer': The manufacturer of the device.
              - 'description': The description of the device.
              - 'serial_number': The serial number of the device.
    """
    device_info = {}
    manufacturer = ctypes.c_uint32()
    description = ctypes.create_string_buffer(64)
    serial_number = ctypes.create_string_buffer(64)
    status = microCtrl.readEEPROM(ftDevice,ctypes.byref(manufacturer),ctypes.byref(description),ctypes.byref(serial_number))
    device_info.update({'manufacturer' : manufacturer, 'description' : description.value,'serial_number': serial_number.value})
    return device_info

def cycle_port(ftDevice):
    """
    Cycle the port of a device.

    This function cycles the port of a device and then waits for 6 seconds.

    Args:
        ftDevice (c_void_p): The device to cycle the port of.

    Returns:
        None
    """
    status = microCtrl.cyclePortFTDI(ftDevice)
    time.sleep(6) # required
    return

def erase_ft_eeprom(serial_number:str):
    """
    Erase the EEPROM of a device.

    This function connects to a device at a specified serial number, erases the EEPROM of the device, 
    and then cycles the port.

    Args:
        serial_number (str): The serial number of the device to connect to and erase the EEPROM of.

    Returns:
        None
    """
    serial_number+='A'
    ftDevice = ctypes.c_void_p()
    # Connect to ftdi at specified serial_number, but instead of returning FTDIxxxx class, just return ftDevice.
    # 3e6 baud allows for opening/resetting FT245 as well as FT2232 and FT4232 devices. 
    status = microCtrl.setupFTDIdev(ctypes.byref(ftDevice),serial_number.encode(),3000000)
    status = microCtrl.eraseEEPROM(ftDevice)
    cycle_port(ftDevice=ftDevice) # cycle port after erasing
    return

def open_FT(serialNum : str = '',port : str = '',baudRate : int=12000000):
    '''Open the device port and return a handle which will be used for subsequent accesses

    Args:
        serialNum (str): serial number of device with the port A-D. Defaults to ''.
        port (str): port number A-D, if left blank then assumes port is at the end of the serial number
        baudRate (int): maximum baud rate device can run at. Defaults to 12Mbaud

    Raises:
        RuntimeError: returns status error from dll

    Returns:
        pointer: object reference
    '''

    if port:
        serialNum = serialNum + port

    ftDevice = ctypes.c_void_p()

    status = microCtrl.setupFTDIdev(ctypes.byref(ftDevice),serialNum.encode(),baudRate)
    
    if not status:
        return FTDIxxxx(ftDevice)
        # return ftDevice
        
    else:
        raise RuntimeError('Device not found. Returned : ' + str(status))

class FTDIxxxx:

    def __init__(self,device):
        self.device = device

    def map_port(self):
        '''maps the device port to an integer for the dll to keep track of

        Raises:
            RuntimeError: _description_
        '''
        mapValue[0] += 1
        self.mapValue = mapValue[0]

        status = microCtrl.mapHandles(self.device, self.port, self.mapValue)

        if status:
            raise RuntimeError('Handling mapping error. Returned : ' + str(status))        

    def write_bytes(self, mask:int, byte:int):

        if isinstance(byte, list):
            byteLength = len(byte) 
            byte_ctype = (ctypes.c_uint8*byteLength)()
            for i in range(0, byteLength):
                byte_ctype[i] = byte[i]
        else:
            byteLength = 1
            byte_ctype = (ctypes.c_uint8*1)()
            byte_ctype[0] = byte

        microCtrl.writeCustomPattern(self.device, mask, byte_ctype, byteLength)

    def set_output_low(self, mask:int):
        """Sets the masked bits to low.

        Args:
            mask (int): Mask of bits to set low. Example, to set bits 8 and 1 low the mask should be set to 0x82 (8b'1000 0010)
        """
        self.write_bytes(mask=mask,byte=0x00)
        return
    
    def set_output_high(self, mask:int):
        """Sets the masked bits to high.

        Args:
            mask (int): Mask of bits to set high. Example, to set bits 8 and 1 high the mask should be set to 0x82 (8b'1000 0010)
        """
        self.write_bytes(mask=mask,byte=0xFF)
        return

    def pulse_output_low(self,mask:int, delay:float):
        """Pulses the masked bits low for the specified delay time.

        Args:
            mask (int): Mask of bits to pulse low. Example, to pulse bits 8 and 1 the mask should be set to 0x82 (8b'1000 0010)
            delay (float): Duration of low pulse.
        """
        self.set_output_low(mask=mask)
        time.sleep(delay)
        self.set_output_high(mask=mask)
        return
    
    def pulse_output_high(self,mask:int, delay:float):
        """Pulses the masked bits high for the specified delay time.

        Args:
            mask (int): Mask of bits to pulse high. Example, to pulse bits 8 and 1 the mask should be set to 0x82 (8b'1000 0010)
            delay (float): Duration of high pulse.
        """
        self.set_output_high(mask=mask)
        time.sleep(delay)
        self.set_output_low(mask=mask)
        return
    
    def init_SPI(self,sclk:int=0,mosi:int=1,miso:int=2,csb:int=3,addr_bits:int=7,data_bits:int=16,pos_edge:bool=True,rw_bit:bool=True, oe=None, oe_pol=True):
            '''initialize the port as a SPI controller

            Args:
                sclk (int, optional): DUT sclk line. Defaults to 0.
                mosi (int, optional): DUT data input line. Defaults to 1.
                miso (int, optional): DUT data output line. Defaults to 2.
                csb (int, optional): DUT chip select line. Defaults to 3.
                addr_bits (int, optional): number of address bits. Defaults to 7.
                data_bits (int, optional): number of data bits. Defaults to 16.
                pos_edge (bool, optional): positive edge triggered. Defaults to True.
                rw_bit (bool, optional): has a read/write bit. Defaults to True.
                oe (bool, optional): Enable output enable (for 5-wire spi cases). Defaults to None.
                oe_pol (bool optional): set output enable polarity. True = Active Low, False = Active High. Defaults to True. 

            Raises:
                RuntimeError: _description_

            Returns:
                reference: handle to device
            '''
            self.port = ctypes.c_void_p()

            status = microCtrl.initSPIDev(ctypes.byref(self.port),sclk,csb,mosi,miso,addr_bits,data_bits,pos_edge,rw_bit)    

            if isinstance(oe, int):
                if oe <= 7 and oe >= 0:
                    status = microCtrl.setupOE(ctypes.byref(self.port), oe, oe_pol)

            if status:
                raise RuntimeError('SPI initialization error. Returned : ' + str(status))
            else:
                self.map_port()
                return ftdRegCtrl(self.mapValue,self.device,self.port)
  

    def init_I2C(self,scl : int=0, sdaw : int=1, sdar : int=2, dev_addr : int=0x00, addr_bits : int=8, data_bits : int=8):
        '''initalize the device port to act as an I2C controller

        Args:
            scl (int, optional): clock line. Defaults to 0.
            sdaw (int, optional): data write line from controller. Defaults to 1.
            sdar (int, optional): data read line from controller. Defaults to 2.
            dev_addr (int, optional): device address. Defaults to 0x00.
            addr_bits (int, optional): number of address bits. Defaults to 8.
            data_bits (int, optional): number of data bits. Defaults to 8.

        Raises:
            RuntimeError: _description_

        Returns:
            reference: handle to device
        '''
        self.port = ctypes.c_void_p()

        status = microCtrl.initI2CDev(ctypes.byref(self.port),scl,sdaw,sdar,dev_addr,ctypes.c_uint8(int(addr_bits/8)),ctypes.c_uint8(int(data_bits/8)))    

        if status:
            raise RuntimeError('I2C initialization error. Returned : ' + str(status))
        else:
            self.map_port()
            return ftdRegCtrl(self.mapValue,self.device,self.port)
             

class ftdRegCtrl:
    def __init__(self,mapValue,device,port):
        self.mapValue = mapValue
        self.device = device
        self.port = port
        pass

    def close(self):
        '''closes the open FTDI device reference

        Raises:
            RuntimeError: _description_
        '''

        try:
            self.close_I2C()
        except:
            pass

        try:
            self.close_SPI() 
        except:
            pass

        status = microCtrl.closeFTDI(ctypes.byref(self.device))    

        if status:
            raise RuntimeError('FTDI close error. Returned : ' + str(status))   

    def close_SPI(self):
        status = microCtrl.deleteSPIDev(ctypes.byref(self.port))    

        if status:
            raise RuntimeError('SPI close error. Returned : ' + str(status))        

    def close_I2C(self):
        status = microCtrl.deleteI2CDev(ctypes.byref(self.port))    

        if status:
            raise RuntimeError('I2C close error. Returned : ' + str(status))    

    def read(self,*args):
        '''reads value of register at address

        Args:
            can be one of the following:
            \t1) int: single address\n
            \t2) list: of address\n
            \t3) int, int: start address, stop address


        Returns:
            int: value of the register in decimal\n
            OR\n
            list: value of register in decimal
        '''
        if len(args) == 1:
            # if it's a single read or a list of addresses
            if type(args[0]) != list:
                # single write with address, value format
                address = args[0]

                # write single value to device
                return microCtrl.read_reg(self.mapValue,address)

            else:
                
                addresses = args[0]
                values = [0xff]*len(addresses)

                if addresses:
                    addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                    valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                    status = microCtrl.read_regs(self.mapValue,addressIn,valuesIn,len(addresses))
                    
                    return [x for x in valuesIn]
        else:
            startAddress = args[0]
            stopAddress = args[1]

            addresses = []
            for address in range(startAddress,stopAddress+1):
                addresses.append(address)

            values = [0xff]*len(addresses)

            if addresses:
                addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                status = microCtrl.read_regs(self.mapValue,addressIn,valuesIn,len(addresses))
                
                return [x for x in valuesIn]            

    def stream_read(self,startAddress : int = 0x00, numRegs : int = 0):
        values = [0x0]*numRegs
        valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

        
        status = microCtrl.multiRegStreamRW(self.device, self.port,int(startAddress),valuesIn,len(values),False)
        return [x for x in valuesIn]

    def stream_write(self,startAddress : int = 0x00, values : list = []):
        valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

        status = microCtrl.multiRegStreamRW(self.device, self.port,startAddress,valuesIn,len(values),True)
        if status:
            raise RuntimeError('Stream write error.  Returned : ' + str(status))
        else:
            return status


    def write(self, *args):
        '''writes single or multiple address/value combinations to DUT in a single stream

        Args:
            input can be any of the following formats but need type int
            \t1) address,value\n
            \t2) [addesses0-X],[values0-X]\n
            \t3) [(address,value),(addressX,valueX)]

        Raises:
            RuntimeError: returns an error if the input is incorrect or dll has error
            
        '''
        if type(args[0]) == int:
            # single write with address, value format
            address = args[0]
            value = args[1]

            # write single value to device
            status = microCtrl.write_reg(self.mapValue,address,value)

            if status:
                raise RuntimeError('Register write error.  Returned : ' + str(status))
            else:
                return 0
        else:
            addresses = []
            values = []
            if type(args[0]) == list and len(args) == 1:
                # format is a list of tuples
                for each in args[0]:
                    addresses.append(each[0])
                    values.append(each[1])
            elif type(args[0]) == list and len(args) == 2:
                # format is a list of addresses followed by a list of values
                addresses = args[0]
                values = args[1]
            else:
                return 0
            
            if addresses:
                addressIn = (ctypes.c_uint32 * len(addresses))(*addresses)
                valuesIn  = (ctypes.c_uint32 * len(values))(*values) 

                status = microCtrl.write_regs(self.mapValue,addressIn,valuesIn,len(addresses))

                if status:
                    raise RuntimeError('Register write error.  Returned : ' + str(status))
                else:
                    return 0
            else:
                raise RuntimeError('Input value(s) are not in the correct format')

if __name__ == '__main__':
    # Example: Programming blank device
    program_blank_device('ADC34RF55EVM')

    # Example: Returning ftdi index location by serial number and port
    print(get_ft_index_by_serial_number_and_port('TI973C3V','C'))
    print(get_ft_index_by_serial_number_and_port('TI973C3V','D'))
    print(get_ft_index_by_serial_number_and_port('TI973C3V','A'))
    print(get_ft_index_by_serial_number_and_port('TI973C3V','B'))

    # Example: Returning ftdi index locations by product description
    print(get_ft_indexes_by_product_description('ADC34RF55EVM'))

    # Example: Returning ftdi serial numbers (with port) and index locations by product description
    print(get_serial_numbers_by_product_description('ADC34RF55EVM'))

    for i in range(3):
        print_all_connected_devices()
        program_blank_device('ADC34RF55EVM',serial_number_prefix='TI')
        print_all_connected_devices()
        serial_num = get_device_info_detail(0)['serial'].decode()[:-1] # decode and remove port 
        erase_ft_eeprom(serial_number=serial_num)