/*
    TI_FTDI.c To be compiled into a .DLL. This wrapper API works with ftd2xx API to generate SPI and I2C transactions for use with TI EVM's.
    TI_FTDI.h is the accompaning header file.
    For compiliation this should be linked to ftd2xx.lib file and ftd2xx.dll should be present.
    ftd2xx.h is also necessary as included in source.
    Written by Matt Kramer of Texas Instruments (A0226277)
*/
#include "ftd2xx.h"     // FTDI driver header file.
#include "TI_FTDI.h"

#define MAJOR 1
#define MINOR 0
#define PATCH 1
/*
    These global variables help keep track of FT_HANDLES and PROT_HANDLES. Initialize these void pointer arrays to 0.
    When mapHandles() is called these are updated to point at the respective pointers at the provided index.
    DEVICE_COUNT can be changed at compile time and is the range of indexs which can be used to map devices.
*/
FT_HANDLE       FTDI_Handles    [DEVICE_COUNT] = {0};
PROT_HANDLE     PROT_Handles    [DEVICE_COUNT] = {0};
/*
    Global variables for enabling Debug. These can be enabled via. enableDebugPrint() function
    By default debugDelim is 'n' which is interpreted as no delimiter. This can be updated with setDebugDelim() function. Useful if wanting to save as .csv file.
*/
BOOL writeDebug = FALSE;
BOOL readDebug  = FALSE;
char debugDelim = 'n';
/*
    struct <PROT_DES> A single structure that will contain all protocol information about a particular device. This will be referenced as a handle and not mutable outside of the .DLL.
    Both I2C and SPI devices share a single structure for all their respective parameters.
*/
struct PROT_DES
{
    BOOL SPI_I2C; // Boolean used to see if SPI or I2C Device. TRUE for SPI, False for I2C.
    // Variables used for SPI
    uint8_t CLKnum;         // sets the bit on FTDI bus that is used for clock
    uint8_t CSnum;          // sets the bit on FTDI bus that is used for CSb (chip select bar)
    uint8_t SDInum;         // sets the bit on FTDI bus that is used for SDI (write data)
    uint8_t SDOnum;         // sets the bit on FTDI bus that is used for SDO (readback)
    uint8_t addrBits;       // number of address bits in transaction.
    uint8_t dataBits;       // number of data bits in transaction.
    BOOL posedge;           // Set the polarity of the clock. True = positive edge triggered. False = negative edge triggered.
    BOOL RWbit;             // Set if a R/W bit is present in the transaction
    // Variables used for I2C.
    uint8_t SCLnum;         // sets the bit on the FTDI bus that is used for clock
    uint8_t SDAnumW;        // sets the bit on the FTDI bus that is used for writing data (Low Z)
    uint8_t SDAnumR;        // sets the bit on the FTDI bus that is used for reading data (HIgh Z)
    uint8_t addr;           // Sets the address for the I2C Device.
    uint8_t bytesPerAddr;   // Sets the number of bytes for each register address (Not device Address!).
    uint8_t bytesPerReg;    // Sets the number of bytes for each register access (typically 1 or 2).
    // Used both for I2C and SPI addtional optional OE line for Bus direction control.
    uint8_t OEnum;          // sets the bit on FTDI bus that is used output enable
    BOOL OE_EN;             // enable output enable (OE) GPO.
    BOOL OE_pol;            // Set the polarity of the output enable with SDO pin
};
/*
    The sequenceMode enum is used for creating byte sequences. The SPI and I2C byte sequence functions can generate all or part of a transaction.
    These portions can be appended together into a larger byte sequnce, allowing things like streaming and multi-register transactions to be packed into a single FTDI write API call.
*/
typedef enum
{
    WRITE,                  // Full write with address, register and data. Peripheral only ack's for write.
    WRITE_STREAM_WORD,      // Write single word. To be appended after WRITE transaction. Used for I2C and SPI streaming.
    READ_NACK,              // Full Read with controller NACK at end (last byte). Useful for single Read transaction.
    READ_ACK,               // Full read with controller ACK at end (last byte). Useful for starting read and appending additional bytes for streaming mode in I2C mode.
    READ_STREAM_WORD_NACK,  // Read single byte with NACK at end (last byte). Should be appended to full readback transaction with ACK.
    READ_STREAM_WORD_ACK,   // Read single byte with ACK at end (not last byte). Should be appended to full readback transaction with ACK.
}   sequenceMode;
/*
    The following function prototypes are for functions that are used internally that are not used outside of this file. These functions can be found at the end of the source file.
*/
static void         printPattern        (uint8_t *writeArray, uint32_t len, char delim);                                                    // Used for printing write byte array or readback byte array to screen when debug booleans are set true.
static uint8_t *    createSPIByteStream (struct PROT_DES spiDev, uint32_t addr, uint32_t data, sequenceMode mode, uint32_t *eventCount);    // Create SPI bit-bang sequence. Will return pointer to array in memory.
static uint8_t *    createI2CByteStream (struct PROT_DES i2cdev, uint32_t reg, uint32_t data, sequenceMode mode, uint32_t *eventCount);     // Create I2C bit-bang sequence. Will return pointer to array in memory.
static uint32_t     parseByteStream     (uint8_t * byteStream, uint32_t StrobeIndex, uint32_t strobeJump, uint32_t strobeCount);            // Strobe Data in readback byte-sequence.
static int32_t      setImpedance        (FT_HANDLE dev, PROT_HANDLE prot);                                                                  // sets input/output pins for each transaction
static int32_t      wait4RxBuffer       (FT_HANDLE dev, uint32_t count);                                                                    // simple delay function to wait for RxFifo to be full.
/*
    <int32_t printConnectedDevices>
    This function prints the serial numbers of all connected FTDI devices. It accepts one parameter.
    <Bool print2Screen>     If true, print the devices to the console. Setting this false can be useful to just get the number of open FTDI devices connected to the computer quickly.
    Returns the number of connected devices if successful. Returns -1 if not.
    If a device is already open it will not be counted towards the number of connected devices as trying to open this device will result in an error.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT printConnectedDevices(BOOL print2Screen, BOOL returnTotalCount)
{
    int32_t numDevs; // Will contain total number of devices including open devices. (Due to the way ftd2xx protocol works)
    FT_STATUS ftStatus;
    ftStatus = FT_ListDevices(&numDevs, NULL, FT_LIST_NUMBER_ONLY); // Call to D2XX API to first get the number of devices connected to the PC.
    if(ftStatus != FT_OK)
    {
        return -1; // Just return a -1 if it does not work properly.
    }
    DWORD devIndex;           // Will be used in for loop to iterate through all devices.
    int32_t numOpen = numDevs;  // Start by making numOpen = to numDevs. Decrement numOpen when a FT_DEVICE_NOT_OPEN error is returned by FT_ListDevices
    char name[SERIAL_LEN];      // Make it SERIAL_LEN bytes long.
    if(print2Screen == TRUE)
    {
        printf("Current connected and open devices by serial number:\n");
    }
    for(devIndex = 0; devIndex < numDevs; devIndex++) // iterate through the connected devices and print their respective serial number to the screen.
    {
        ftStatus = FT_ListDevices((PVOID)devIndex, name, FT_LIST_BY_INDEX | FT_OPEN_BY_SERIAL_NUMBER); // Call to D2XX API to provide the serial number of the indexed device.
        if(ftStatus == FT_DEVICE_NOT_OPENED) // This ftStatus means the device is unable to be opened. Likely meaning it is already open by another process.
        {
            numOpen--; // The device is closed and will not be listed. Trying to list it will result in an error. Decrement the numOpen integer.
        }
        else{
            if(ftStatus != FT_OK) // Any error other than FT_DEVICE_NOT_OPENED points to an unexpected error
            {
                printf("FTDI Error\n");
                return -1;
            }
            else if(print2Screen == TRUE) // Print to console.
            {
                printf("%s\n", name);
            }
        }
    }
    if(returnTotalCount == TRUE)
    {
        return numDevs; //Return total number of devices.
    }
    return numOpen; // Worked properly. Return number of open devices (Open meaning ready to be connected to).
}
/*
    <in32_t getConnectedDevices>
    This function accepts a char * pointer to pre allocated memory as well as an integer with the number of devices it can write serial numbers for.
    It is designed to play well with printConnectedDevices() as this function returns the number of open devices connected to the PC.
    The constant defined in TI_FTDI.h <SERIAL_LEN> defines the number of bytes that are allocated per serial number. This function will work with numDevices*SERIAL_LEN bytes of memory.
    The memory should be preallocated. This allows garbage collection of higher level languages to reallocate the memory after it is used or the memory to be reallocated manually.
    It accepts the following parameters.
    <char ** connectedDevices>  A pointer to a char pointer that has allocated memory to be modified with serial numbers/ Will be a 2D array.
    <uint32_t numDevices>       Number of <SERIAL_LEN> char arrays getConnectedDevices can write to. Providing this manually will prevent memory access violations.
    Returns the -1 for any ftStatus related error. Returns 0 otherwise.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT getConnectedDevices(char ** connectedDevices, uint32_t numDevices)
{
    DWORD devIndex; // To be used in for loop to keep track of current device index.
    FT_STATUS ftStatus;
    uint32_t openIndex = 0;     // Will only be incremented when ftStatus is FT_OK and not FT_DEVICE_NOT_OPENED
    uint32_t offset = 0;        // Will be incremented when ftStatus is FT_DEVICE_NOT_OPENED. This essentially means an already opened device was attempted to be open again.
    char name[SERIAL_LEN];      // Each serial number will temporarily be stored
    for(devIndex = 0; devIndex < numDevices + offset; devIndex++)
    {
        ftStatus = FT_ListDevices((PVOID)devIndex, name, FT_LIST_BY_INDEX | FT_OPEN_BY_SERIAL_NUMBER); // Call to D2XX API to provide the serial number of the indexed device.
        if(ftStatus == FT_DEVICE_NOT_OPENED) // This ftStatus means the device is unable to be opened. Likely meaning it is already open by another process.
        {
            offset++; // The device is closed and will not be listed. Trying to list it will result in an error. Decrement the numOpen integer.
        }
        else
        {
            if(ftStatus != FT_OK)
            {
                printf("FTDI Error\n"); // Device is not FT_OK or FT_DEVICE_NOT_OPENED, meaning something else went wrong. Error out.
                return -1;
            }
            else
            {
                strcpy(connectedDevices[openIndex++],name); // Copy the name that was returned by FT_ListDevices to connected Devices Array. Increment index.
            }
        }
    }
    return 0;
}
/*
    <int32_t getDeviceInto>
    This function is used to get addtional information about an FTDI device at index <index>. This addtional information first needs to be generated either when called for the first
    time or anytime an FTDI device is added or removed from the PC. If <update> is TRUE, FT_CreateDeviceInfoList() is called and the number of connected devices is updated. This process
    times time and only needs to happen once. That is why the <update> paramter should be false otherwise. This function regardless of update's state will update the
    <flags>, <type, <devID, <locID, <description, and <serialNumber> by reference.
    It accepts the following parameters
    <uint32_t   index>              index of FTDI device to pull information from.
    <uint32_t * flags>              int32_t pointer to flags. (Will be updated by reference).
    <uint32_t * type>               int32_t pointer to type. (Will be updated by reference).
    <uint32_t * devID>              int32_t pointer to devID. (Will be updated by reference).
    <uint32_t * locID>              int32_t pointer to locID. (Will be updated by reference).
    <char     * description>        char pointer to description. (Will be updated by reference and properly terminated).
    <char     * serialNumber>       char pointer to serialNumber. (Will be updated by reference and properly terminated).
    <BOOL       update>             Boolean to indicate if deviceInfo structure needs to be updated. (Takes time to do!)
    Returns the number of connected FTDI devices if update is TRUE. Otherwise returns 0 if sucessful. If an error occurs it returns -1.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT getDeviceInfo(uint32_t index, uint32_t * flags, uint32_t * type, uint32_t * devID, uint32_t * locID, char * description, char * serialNumber, BOOL update)
{
    FT_STATUS ftStatus;
    FT_HANDLE temp;
    uint32_t numDevs = 0;
    if(update) // update DeviceInfoList internal to ftd2xx API. This can be SLOW.
    {
        ftStatus = FT_CreateDeviceInfoList((DWORD*)&numDevs);
        if(ftStatus != FT_OK)
        {
            printf("FTDI Error\n");
            return -1;  // ftStatus should be FT_OK. Otherwise return error.
        }
    }
    // FTDI uses DWORDS (Windows specific 32 bit unsigned integer. I prefer using stdint data types. These are masked to surpress warnings. They are functionally equiviilent (32 bit unsigned integer)
    ftStatus = FT_GetDeviceInfoDetail((DWORD)index, (DWORD*)flags, (DWORD*)type, (DWORD*)devID, (DWORD*)locID, serialNumber, description, &temp); //Update referenced values with API call.
    if(ftStatus != FT_OK)
    {
        printf("FTDI Error\n");
        return -1;  // ftStatus should be FT_OK. Otherwise return error.
    }
    return numDevs;
}
/*
    <int32_t setupFTDIDev>
    This function sets up the FTDI handle and sets the baudrate. This is used for both SPI and I2C.
    It accepts the following parameters.
    <FT_HANDLE *inHandle>       A pointer to a FT_HANDLE. This will be modified at the end of the function call to point to the newly created FT_HANDLE inside the function.
    <char serial[]>             A character array containing the serial number of the desired FTDI device serial.
    <uint32_t baudRate>         An unsigned int that sets the baudrate of the FTDI device bus.
    Returns the FT_STATUS of the first D2XX API call that fails or FT_OK if all D2XX API calls are successful .
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT setupFTDIdev(FT_HANDLE *inHandle, char serial[], uint32_t baudRate)
{
    FT_HANDLE dev = (FT_HANDLE*)malloc(sizeof(FT_HANDLE));          // Create an FT_HANDLE object on the heap and store the address. (Note this is just a void pointer).
    FT_STATUS ftStatus;
    ftStatus = FT_OpenEx(serial, FT_OPEN_BY_SERIAL_NUMBER, &dev);   // Call to D2XX API to open device based off provided serial number.
    if(ftStatus != FT_OK)
    {
        printf("FTDI Error, Cannot open Serial %s\n", serial);
        return ftStatus;
    }
    ftStatus = FT_SetBaudRate(dev, baudRate);                       // Call to D2XX API to set the baudrate of the device.
    if(ftStatus != FT_OK)
    {
        printf("FTDI Error, invalid Baudrate\n");
        return ftStatus;
    }
    *inHandle = dev;                                                // Set the provided pointer to the FT_HANDLE to the newly created FT_HANDLE
    return ftStatus;                                                // This will always be FT_OK (0) if it made it this far.
}
/*
    <int 32_t initSPIDev>
    This function sets up the PROT_DES handle to be used for a SPI device. The OE bit is not setup in this function. For OE functionality one should call setupOE().
    It accepts the following parameters.
    <PROT_HANDLE *inHandle>     A pointer to what will become a PROT_HANDLE. Will be updated at the end of the function call.
    <uint8_t CLKnum>            A byte that has the bit number of the clock.
    <uint8_t CSnum>             A byte that has the bit number of the CS signal.
    <uint8_t SDInum>            A byte that has the bit number of the SDI signal. (Also known as Controller output).
    <uint8_t SDOnum>            A byte that has the bit number of the SDO signal. (Also known as the peripheral output).
    <uint8_t addrBits>          A byte that contains how many bits wide the SPI addresses are of the peripheral.
    <uint8_t dataBits>          A byte that contains how many bits wide the SPI data fields are of the peripheral.
    <BOOL posedge>              A boolean that sets up if the peripheral uses positive or negative edge triggered clocks. (TRUE = posedge, False = negedge).
    <BOOL RWbit>                A boolean that sets up if the peripheral has a RW bit passed before the address. The RWbit is high for read and low for write.
    Returns 0.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT initSPIDev(PROT_HANDLE *inHandle, uint8_t CLKnum, uint8_t CSnum, uint8_t SDInum, uint8_t SDOnum, uint8_t addrBits, uint8_t dataBits, BOOL posedge, BOOL RWbit)
{
    struct PROT_DES *spiDev = malloc(sizeof(struct PROT_DES));
    spiDev->SPI_I2C     = TRUE;     // Set this boolean true, meaning this handle is being used for a SPI device.
    spiDev->CLKnum      = CLKnum;   // Update each SPI parameter based off user input.
    spiDev->CSnum       = CSnum;
    spiDev->SDInum      = SDInum;
    spiDev->SDOnum      = SDOnum;
    spiDev->addrBits    = addrBits;
    spiDev->dataBits    = dataBits;
    spiDev->posedge     = posedge;
    spiDev->RWbit       = RWbit;

    // These are not used. Will create separate function for setting these up as many devices do not need OE bit.
    spiDev->OE_EN       = FALSE; // This boolean will be set true if setupSpiOE is called.
    spiDev->OEnum       = 0;
    spiDev->OE_pol      = TRUE;
    *inHandle           = spiDev;
    return 0;
}
/*
    <int32_t setupOE>
    This function modifies the PROT_DES handler for the OE bit. This function only needs to be called if a OE bit is necessary. It should always be called after initSpiDev() or initI2CDev()
    The OE bit is toggled during readback and is useful for devices that use 3 wire SPI that have a tri-state buffer for setting if the peripheral or controller is actively writing.
    It can also be used for I2C buses which use a push pull type of output drive on a tri-state buffer.
    It accepts the following parameters.
    <PROT_HANDLE *inHandle>     A pointer to the already created PROT_DES handle.
    <uint8_t OEnum>             A byte that has the bit number of the output enable GPO.
    <BOOL OEpol>                A boolean that sets up the polarity of OE.
    Returns 0.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT setupOE(PROT_HANDLE *inHandle, uint8_t OEnum, BOOL OEpol)
{
    struct PROT_DES *prot = *inHandle;
    prot->OEnum   = OEnum;
    prot->OE_pol  = OEpol;
    prot->OE_EN   = TRUE;
    return 0;
}
/*
    <int32_t writeSPI>
    This function is what is called for writing a SPI transaction on an FTDI device.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use write_reg()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t addr>             An unsigned integer containing the address data to be written.
    <uint32_t data>             An unsigned integer containing the data to be written to the given address.
    Returns 0 if successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK)
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT writeSPI(FT_HANDLE dev, PROT_HANDLE spi, uint32_t addr, uint32_t data)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); // Purge both TX and RX buffers before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    struct PROT_DES spiDev;
    memcpy(&spiDev, spi, sizeof(struct PROT_DES)); // Local PROT_DES structure now has contents of handle copied to it.
    if(spiDev.SPI_I2C == FALSE)
    {
        return -1; // I2C Device handle provided. Error out.
    }
    uint8_t* writeBytes; // Will point to a dynamically allocated array of writeBytes. This is created/allocated in createSPIByteStream.
    uint32_t eventCount; // Will keep track of events count. Passed by reference to createSPIByteStream
    uint32_t byteCount;  // FT_write will modify this value by reference. It should always match event count.
    writeBytes = createSPIByteStream(spiDev, addr, data, WRITE, &eventCount); // create Bytestream to be written to FTDI device.
    if(setImpedance(dev, spi) != 0) // Setup which physical pins on the FTDI device are input or output.
    {
        return -1; // Exit on Error.
    }
    ftStatus = FT_Write(dev, writeBytes, eventCount, (LPDWORD)&byteCount); // write bytestream to FTDI device using FTDI API.
    if(writeDebug)  // if writeDebug == TRUE, print writeBytes to screen.
    {
        printPattern(writeBytes, eventCount, debugDelim); // printPattern organizes data and prints it to screen.
    }
    free(writeBytes); // reallocate memory for dynamically created writeBytes array.
    if(ftStatus != FT_OK || byteCount != eventCount)
    {
        return -1; // Either FTDI API thew an error or the write did not execute the proper number of bytes.
    }
    return 0;
}
/*
    <int32_t readSPI>
    This function is what is called for reading a register from a SPI device.
    It first creates a byte stream of bytes to be written to the device. It then performs a FIFO readback and then strobes the respective bits to readback from the device.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use read_reg()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t addr>             An unsigned integer containing the address to read from.
    Returns the read value if successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK)
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT readSPI(FT_HANDLE dev, PROT_HANDLE spi, uint32_t addr)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); // Purge before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    struct PROT_DES spiDev;
    memcpy(&spiDev, spi, sizeof(struct PROT_DES)); // Local PROT_DES structure now has contents of handle copied to it.
    if(spiDev.SPI_I2C == FALSE)
    {
        return -1; // I2C Device handle provided. Error out.
    }
    uint8_t* writeBytes; // Will point to a dynamically allocated array of writeBytes. This is created/allocated in createSPIByteStream.
    uint32_t eventCount; // Will keep track of events count. Passed by reference to createSPIByteStream
    uint32_t byteCount;  // FT_write and FT_read will modify this value by reference. It should always match event count.
    writeBytes = createSPIByteStream(spiDev, addr, 0, READ_ACK, &eventCount); // create Bytestream to be written to FTDI device.
    if(setImpedance(dev, spi) != 0) // Setup which physical pins on the FTDI device are input or output.
    {
        return -1; // Exit on Error.
    }
    ftStatus = FT_Write(dev, writeBytes, eventCount, (LPDWORD)&byteCount);  // write bytestream to FTDI device using FTDI API.
    free(writeBytes); // reallocate memory for dynamically created writeBytes array.
    if(ftStatus != FT_OK || byteCount != eventCount)
    {
        return -1; // Either FTDI API thew an error or the write did not execute the proper number of bytes.
    }
    if(wait4RxBuffer(dev, eventCount) == -1) // Wait for RXFifo to be ready.
    {
        return -1;
    }
    uint8_t *rxBytes = (uint8_t*)malloc(eventCount * sizeof(uint8_t)); // Looks like the RxFIFO is ready to be ready from. Allocate dynamic memory to contain the data.
    ftStatus = FT_Read(dev, rxBytes, eventCount, (LPDWORD)&byteCount); // Read the data into rxBytes.
    if(ftStatus != FT_OK || byteCount != eventCount) // Just like with FT_write Check the FT_Status good and the correct number of bytes were retrieved.
    {
        free(rxBytes); // Free the rxBytes pointer
        return -1; // Leave
    }
    uint8_t clkStrobe = 0; // Keep track of first clock strobe found in readback data. Some intelligence will account for any offsets between read and write data vs. just hardcoding offsets.
    for(int i = 0; i < eventCount; i++)
    {
        if((rxBytes[i] & (0x01 << spiDev.CSnum)) == 0)  // Verify CS is low
        {
            clkStrobe = ((rxBytes[i] & (0x01 << spiDev.CLKnum)) >> spiDev.CLKnum);
            if(spiDev.posedge == FALSE)
                clkStrobe = 0x01 & ~clkStrobe; // invert clock to make finding initial clock strobe the same between rising and falling edge.
            if(clkStrobe == 0x01)
            {
                clkStrobe = i;  // First Clock Strobe found at index i. Store this into clockStrobe and leave this loop.
                i = eventCount; // Leave the loop
            }
        }
    }
    uint32_t MSBData = clkStrobe + spiDev.addrBits*2; // First index of data to strobe (After CS going low and address bits)
    if(spiDev.RWbit == TRUE)
    {
        MSBData += 2;   // Add addtional 2 strobes for RW bit
    }
    int32_t readbackData = 0;
    readbackData = parseByteStream(&rxBytes[MSBData], spiDev.SDOnum, 2, spiDev.dataBits); // Parse each bit in the readback data and store it.
    if(readDebug) // if readDebug == TRUE, print rxBytes to screen.
    {
        printPattern(rxBytes, eventCount, debugDelim); // printPattern organizes data and prints it to screen.
    }
    free(rxBytes); // free rxBytes array.
    return readbackData;
}
/*
    <int32_t initI2CDev>
    This function sets up the PROT_DES handle to be used for a I2C device. The OE bit is not setup in this function. For OE functionality one should call setupOE().
    It accepts the following parameters.
    <PROT_HANDLE *inHandle>     A pointer to what will become a PROT_HANDLE. Will be updated at the end of the function call.
    <uint8_t SCL_num>           A byte that has the bit number of the clock signal.
    <uint8_t SDAw_num>          A byte that has the bit number of the SDA write signal.
    <uint8_t SDAr_num>          A byte that has the bit number of the SDA read signal
    <uint8_t addr>              A byte that contains the 7 bit address of the I2C Device.
    <uint8_t bytesPerReg>       A byte that contains how many bytes are present in each register. (1 or 2 is common).
    Returns 0.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT initI2CDev(PROT_HANDLE * inHandle, uint8_t SCL_num, uint8_t SDAw_num, uint8_t SDAr_num, uint8_t addr, uint8_t bytesPerAddr, uint8_t bytesPerReg)
{
    struct PROT_DES *i2cdev = malloc(sizeof(struct PROT_DES));
    i2cdev->SPI_I2C = FALSE;
    i2cdev->SCLnum = SCL_num;
    i2cdev->SDAnumW = SDAw_num;
    i2cdev->SDAnumR = SDAr_num;
    i2cdev->addr = addr;
    i2cdev->bytesPerAddr = bytesPerAddr;
    i2cdev->bytesPerReg = bytesPerReg;
    // These are not used. Will create separate function for setting these up as the OE bit is typically optional.
    i2cdev->OE_EN = FALSE;
    i2cdev->OEnum = 0;
    i2cdev->OE_pol = TRUE;
    *inHandle = i2cdev;
    return 0;
}
/*
    <int32_t writeI2C>
    This function is what is called for writing a I2C transaction on an FTDI device.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use write_reg()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a Protocol description structure.
    <uint32_t addr>             An unsigned integer containing the address data to be written.
    <uint32_t data>             An unsigned integer containing the data to be written to the given address.
    Returns 0 if successful and all ACK bits are low. Returns a -1 if an error occurs (FT_STATUS != FT_OK). Will return non zero negative value if any ack bits are not 0 from device.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT writeI2C(FT_HANDLE dev, PROT_HANDLE i2c, uint32_t reg, uint32_t data)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); // Purge both TX and RX buffers before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    struct PROT_DES i2cDev;
    memcpy(&i2cDev, i2c, sizeof(struct PROT_DES)); // Local PROT_DES structure now has contents of handle copied to it.
    if(i2cDev.SPI_I2C == TRUE)
    { // A SPI prot handle was passed. Quit on error.
        return -1;
    }
    uint8_t* writeBytes; // Will point to a dynamically allocated array of writeBytes. This is created/allocated in createI2CByteStream.
    uint32_t eventCount; // Will keep track of events count. Passed by reference to createSPIByteStream
    uint32_t byteCount;  // FT_write will modify this value by reference. It should always match event count.
    writeBytes = createI2CByteStream(i2cDev, reg, data, WRITE, &eventCount);  // create Bytestream to be written to FTDI device.
    if(setImpedance(dev, i2c) != 0) // Setup which physical pins on the FTDI device are input or output.
    {
        return -1; // Exit on Error.
    }
    ftStatus = FT_Write(dev, writeBytes, eventCount, (LPDWORD)&byteCount);
    if(writeDebug) // if writeDebug == TRUE, print rxBytes to screen.
    {
        printPattern(writeBytes, eventCount, debugDelim);
    }
    free(writeBytes); // reallocate memory for dynamically created writeBytes array.
    if(ftStatus != FT_OK || byteCount != eventCount)
    {
        return -1; // Either FTDI API thew an error or the write did not execute the proper number of bytes.
    }
    /*
        Single I2C write transaction also checks ACK bits of frame and returns them as a negative integer if an error occurs. Each Ack bit is masked from LSB to MSB.
        A return of 0 indicates all ACK bits were low as expected.
    */
    if(wait4RxBuffer(dev, eventCount) == -1) // Wait for RXFifo to be ready.
    {
        return -1;
    }
    uint8_t *rxBytes = (uint8_t*)malloc(eventCount * sizeof(uint8_t)); // Looks like the RxFIFO is ready to be ready from. Allocate dynamic memory to contain the data.
    ftStatus = FT_Read(dev, rxBytes, eventCount, (LPDWORD)&byteCount); // Read the data into rxBytes.
    if(ftStatus != FT_OK || byteCount != eventCount) // Just like with FT_write Check the FT_Status good and the correct number of bytes were retrieved.
    {
        free(rxBytes); // Free the rxBytes pointer
        return -1; // Exit on error
    }
    int32_t clkStrobe = -1; // Checking ACK bits. First find the initial clock strobe.
    for(int i = 0; i < eventCount; i++)
    {
        if ((rxBytes[i] >> i2cDev.SCLnum & 0x01) == 0)
        { // First event with a low clock
            clkStrobe = i;  // Store this event in clockStrobe
            i = eventCount; // Force exit from the for loop
        }
    }
    uint8_t ackBits = 0; // Will be masked with each respective ACK bit
    uint8_t index = 0;
    for(int i = clkStrobe - 2; i < eventCount; i += 36) // Every 9 bits (4 events per clock period) an ACK will occur. Subtract 2 from clkStrobe to line up the frames.
    {
        if(i > clkStrobe) // First iteration will be at beginning of frame and not where ACK is present. This if state will bypass the first round.
        {
            ackBits |= ((rxBytes[i] >> i2cDev.SDAnumR) & 0x01) << index++; // Each ACK bit is or'd to ackBits and normalized by index. The first ACK bit (After address is bit 0).
        }
    }
    free(rxBytes);
    return -ackBits;
}
/*
    <int32_t readI2C>
    This function is what is called for reading a I2C transaction on an FTDI device.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use read_reg()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t addr>             An unsigned integer containing the address data to be written.
    Returns the read value if successful and all ACK bits are low. Returns a -1 if an error occurs (FT_STATUS != FT_OK). Will return non zero negative value if any ack bits are not 0 from device.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT readI2C(FT_HANDLE dev, PROT_HANDLE i2c, uint32_t reg)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); // Purge both TX and RX buffers before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    struct PROT_DES i2cDev;
    memcpy(&i2cDev, i2c, sizeof(struct PROT_DES)); // Local PROT_DES structure now has contents of handle copied to it.
    if(i2cDev.SPI_I2C == TRUE) // A SPI device pointer was passed to I2C function. Return on error.
    {
        return -1;
    }
    uint8_t* writeBytes; // Will point to a dynamically allocated array of writeBytes. This is created/allocated in createSPIByteStream.
    uint32_t eventCount; // Will keep track of events count. Passed by reference to createSPIByteStream
    uint32_t byteCount;  // FT_write will modify this value by reference. It should always match event count.
    writeBytes = createI2CByteStream(i2cDev, reg, 0, READ_NACK, &eventCount);
    if(setImpedance(dev, i2c) != 0)
    {
        return -1;
    }
    ftStatus = FT_Write(dev, writeBytes, eventCount, (LPDWORD)&byteCount); //
    free(writeBytes); // reallocate memory for dynamically created writeBytes array.
    if(ftStatus != FT_OK || byteCount != eventCount)
    {
        return -1; // Either FTDI API thew an error or the write did not execute the proper number of bytes.
    }
    if(wait4RxBuffer(dev, eventCount) == -1) // Wait for RXFifo to be ready.
    {
        return -1;
    }
    uint8_t *rxBytes = (uint8_t*)malloc(eventCount * sizeof(uint8_t)); // Looks like the RxFIFO is ready to be ready from. Allocate dynamic memory to contain the data.
    ftStatus = FT_Read(dev, rxBytes, eventCount, (LPDWORD)&byteCount); // Read the data into rxBytes.
    if(ftStatus != FT_OK || byteCount != eventCount) // Just like with FT_write Check the FT_Status good and the correct number of bytes were retrieved.
    {
        free(rxBytes); // Free the rxBytes pointer
        return -1; // Exit on error
    }
    int32_t clkStrobe = -1; // Checking ACK bits. First find the initial clock strobe.
    for(int i = 0; i < eventCount; i++)
    {
        if ((rxBytes[i] >> i2cDev.SCLnum & 0x01) == 0) // First event with a low clock
        {
            clkStrobe = i;  // Store this event in clockStrobe
            i = eventCount; // Force exit from the for loop
        }
    }
    uint8_t ackBits = 0; // Will be masked with each respective ACK bit
    int8_t index = 0;
    for(int i = clkStrobe - 2; i < eventCount; i += 36) // Every 9 bits (4 events per clock period) an ACK will occur. Subtract 2 from clkStrobe to line up the frames.
    {
        if(i > clkStrobe) // First iteration will be at beginning of frame and not where ACK is present. This if statement will bypass the first round.
        {
            ackBits |= ((rxBytes[i] >> i2cDev.SDAnumR) & 0x01) << index++; // Each ACK bit is or'd to ackBits and normalized by index. The first ACK bit (After address is bit 0).
        }
    }
    uint8_t mask = 0x3;
    mask |= ~(~mask << i2cDev.bytesPerAddr);
    ackBits &= mask; // Only interested in ACK bits of device and not controller readback ack bits.
    if (ackBits != 0) // Return -1 to indicate not a proper acknowledgement.
    {
        if(readDebug)
        {
            printPattern(rxBytes, eventCount, debugDelim); // print to screen even if ACK bits are not all 0.
        }
        free(rxBytes);
        return -ackBits; // return negative ackbits as to not confuse it with a proper readback.
    }
    int32_t readBack = 0;
    for(int i = 0; i < i2cDev.bytesPerReg; i++)
    {
        readBack |= (parseByteStream(&rxBytes[clkStrobe + 76 + 36*i2cDev.bytesPerAddr + i*36], i2cDev.SDAnumR, 4, 8) << 8*(i2cDev.bytesPerReg - i - 1)); // I promise it works
    }
    if(readDebug) // if readDebug == TRUE, print rxBytes to screen.
    {
        printPattern(rxBytes, eventCount, debugDelim);
    }
    free(rxBytes);
    return readBack;
}
/*
    <int32_t multiRegRW>
    This function is what is called for writing or reading multiple registers to a SPI or I2C device.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use read_regs() or write_regs()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t addr []>          An array of address to be written to.
    <uint32_t data []>          An array of the data to be written to each corresponding address for write. For readback this will be updated by reference to contain readback data.
    <uint32_t numRegs>          An unsigned integer indicating how many registers are to be written.
    <BOOL write>                A boolean used to indicate if a read or write is being performed.
    returns 0 if sucessful. Returns a -1 on error.
    Due to nature of large transaction debug printing is not used in this function.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT multiRegRW(FT_HANDLE dev, PROT_HANDLE prot, uint32_t reg [], uint32_t data [], uint32_t numRegs, BOOL write)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); // Purge both TX and RX buffers before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    uint8_t * (*byteStreamFunc [])(struct PROT_DES, uint32_t, uint32_t, sequenceMode, uint32_t*) = {createI2CByteStream, createSPIByteStream}; // A function pointer to createI2CbyteStream and createSPIByteStream.
    struct PROT_DES protDev;
    memcpy(&protDev, prot, sizeof(struct PROT_DES)); // Local PROT_DES structure now has contents of handle copied to it.
    /*
        ByteStreamFunc will either be createSPIByteStream or createI2CByteStream depending on the value of SPI_I2C. If SPI_I2C is TRUE (1) Then SPI will be called.
        Otherwise I2C will be called. Both functions accept identical input arguments making this simple.
    */
    uint8_t* writeBytes; // Will contain the entire transaction
    uint8_t* temp; // Will contain each individual transaction. To be erased and reused each transaction.
    uint32_t eventCount; // Will contain the event count for a single transaction
    uint32_t byteCount;
    sequenceMode mode = WRITE; // set sequenceMode to WRITE by default. Both I2C and SPI byte seqeunce functions accept this as standard write.
    if(write == FALSE)
    {
        mode = READ_NACK; // set sequenceMode to READ_NACK by default. Both I2C and SPI byte seqeunce functions accept this as standard read. I2C read always terminates with controller NACK on last byte.
    }
    free((byteStreamFunc)[(int)protDev.SPI_I2C](protDev, 0, 0, mode, &eventCount)); // Just for getting necessary eventCount. Returns a char pointer that needs to be free'd.
    writeBytes = (uint8_t*)malloc(eventCount*numRegs*sizeof(uint8_t)); // Create writeByte sequnece on heap.
    if(setImpedance(dev, prot) != 0) // Set input/output of pins.
    {
        return -1; // Exit on error
    }
    for (int i = 0; i < numRegs; i++)
    {
        temp = (byteStreamFunc)[(int)protDev.SPI_I2C](protDev, reg[i], data[i], mode, &eventCount); // Store the byte stream into temp
        memcpy(&writeBytes[i*eventCount], temp, eventCount); // copy contents of temp into appropriate location in writeBytes.
        free(temp);
    }
    ftStatus = FT_Write(dev, writeBytes, eventCount*numRegs, (LPDWORD)&byteCount); // Total number of bytes to be written is eventCount * (number of registers)
    free(writeBytes);
    if(ftStatus != FT_OK || byteCount != eventCount*numRegs)
    {
        return -1; // Either FTDI API thew an error or the write did not execute the proper number of bytes.
    }
    if(write == FALSE) // readback read the FIFO of the FTDI chip and interpret what was read back from the device.
    {
        if(wait4RxBuffer(dev, eventCount) == -1) // Wait for RXFifo to be ready.
        {
            return -1;
        }
        uint8_t *rxBytes = (uint8_t*)malloc(eventCount * numRegs * sizeof(uint8_t)); // Looks like the RxFIFO is ready to be ready from. Allocate dynamic memory to contain the data.
        ftStatus = FT_Read(dev, rxBytes, eventCount * numRegs, (LPDWORD)&byteCount); // Read the data into rxBytes.
        if(protDev.SPI_I2C == TRUE) // Device is a SPI device.
        {
            int32_t * csEvents = (int32_t*)malloc(numRegs*sizeof(int32_t)); // An array with each CS strobe in the readback transaction. This will be used to find each transaction in the stream
            int j = 0;
            for(int i = 1; i < eventCount*numRegs; i++) // This for loop goes through the entire transaction looking for CS strobes.
            {
                if((((rxBytes[i - 1] >> protDev.CSnum) & 0x01) == 0x01) && (((rxBytes[i] >> protDev.CSnum) & 0x01) == 0x00)) // CS went from 1 to 0
                {
                    csEvents[j++] = i; // Store the event CS went from 1 to 0 in array and increment array index.
                    if(j == numRegs) // Time to bail out of for loop. All CS events should be accounted for.
                    {
                        i = eventCount*numRegs; // This will force for loop to quit
                    }
                }
            }
            int csOffset = 2 + protDev.addrBits*2; // The actual index that will have data to be read back from.
            if(protDev.RWbit == TRUE)
            {
                csOffset += 2; // Add an extra two events to account for RW bit
            }
            for(int i = 0; i < numRegs; i++) // Read back each frame.
            {
                data[i] = parseByteStream(&rxBytes[csEvents[i] + csOffset], protDev.SDOnum, 2, protDev.dataBits); //Parse Bytes from readback.
            }
        free(csEvents);
        }
        else // protDev.SPI_I2C == FALSE. I2C Device.
        {
            int32_t * startEvents = (int32_t*)malloc(numRegs*sizeof(int32_t)); // Each repeated start bit index in the list of i2c transactions will be stored in an array.
            int j = 0;
            int clkCounter = 0; // Keep track of number of events clock is high. Clock is typically high for 2 consecutive events, however during start events it is high for 3 events.
            BOOL repeatedStart = FALSE; // First start bit is inital start. Repeated start marks where readback begins and therefore will be index of interest.
            for(int i = 0; i < eventCount*numRegs - 1; i++) // This will go through each event in the entire transaction looking for repeated start events. Last event is not checked due to [i + 1] being checked.
            {
                if((rxBytes[i] >> protDev.CLKnum & 0x01) == 0x01) //clock is high
                {
                    if(++clkCounter == 3) //The clock was high for 3 consecutive events. A start event has been found.
                    {
                        clkCounter = 0; // Reset clock counter.
                        if(repeatedStart == TRUE) // repeatedStart toggles from True to False after each start event find. It will be true on second start event of each transaction (repeated start).
                        {
                            repeatedStart = FALSE; // Next start event will be start event of next I2C transaction and therefore will not be repeated start.
                            startEvents[j++] = i; // Store event number of repeated start.
                            if((rxBytes[i + 1] >> protDev.CLKnum & 0x01) == 0x01)
                            {
                                startEvents[j - 1]--;   // If 4 clock cycles are high (typical of first byte due to FTDI putting last byte first in FIFO, not sure why it does this!) Decrement startEvents by 1 to make it line up with other transactions.
                            }                           // It is important to note if the FTDI chip does not place last byte in first index this code will still work fine! This line will never be reached and everything will line up.
                        }
                        else
                        {
                            repeatedStart = TRUE; // repeatedStart was FALSE, therefore next start will be repeatedStart.
                        }
                    }
                }
                else
                {
                    clkCounter = 0; // Clock went low too quickly, indicating it was just strobing a normal bit and not initiating a start condition.
                }
            }
            for(int i = 0; i < numRegs; i++) // Only need to loop numRegs time. Each time starting at the location of the repeated start condution.
            {
                data[i] = 0; // Data was passed by reference. First clear data.
                for(int j = 0; j < protDev.bytesPerReg; j++) // Called for each byte in the stream. If a single Byte is to be readback this will execute only once.
                {
                    data[i] |= (parseByteStream(&rxBytes[startEvents[i] + 39 + j*36], protDev.SDAnumR, 4, 8) << 8*(protDev.bytesPerReg - j - 1)); // This will parse a single byte. The ACK bits are skipped.
                }
            }
            free(startEvents);
            }
    free(rxBytes);
    }
    return 0;
}
/*
    <int32_t multiRegStreamRW>
    This function is what is called for writing or reading a stream transaction. It is used for both I2C and SPI.
    It accepts handles directly as input. If a user wants to map handles to a device integer they should use read_reg_stream() or write_reg_stream()
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t addr>             An unsigned integer contraining the address. Only a single address is used for streaming mode.
    <uint32_t data []>          An array of the data to be written to each corresponding address for write. For readback this will be updated by reference to contain readback data.
    <uint32_t numRegs>          An unsigned integer indicating how many registers are to be written.
    <BOOL write>                A boolean used to indicate if a read or write is being performed.
    Returns 0 if sucessful. Returns a -1 on error.
    returns 0 if sucessful. Returns a -1 on error.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT multiRegStreamRW(FT_HANDLE dev, PROT_HANDLE prot, uint32_t reg, uint32_t data [], uint32_t numRegs, BOOL write)
{
    FT_STATUS ftStatus;
    ftStatus = FT_Purge(dev, FT_PURGE_RX | FT_PURGE_TX); //Purge both TX and RX buffers before write.
    if(ftStatus != FT_OK)
    {
        return -1;
    }
    struct PROT_DES protDev;
    memcpy(&protDev, prot, sizeof(struct PROT_DES)); //Local PROT_DES structure now has contents of handle copied to it.
    uint8_t* writeBytes;    // Will contain the entire transaction
    uint8_t* temp;          // Will contain each individual transaction
    uint32_t eventCount;    // Will contain the event count for a single transaction
    uint32_t totalEvents = 0;
    uint32_t byteCount;
    if(setImpedance(dev, prot) != 0)
    {
        return -1;
    }
    if (protDev.SPI_I2C == TRUE) // SPI Device.
    {
        if(write == TRUE) //Write.
        {
            temp = createSPIByteStream(protDev, reg, data[0], WRITE, &eventCount); // First byte also has address in from of it. Treat it like a normal SPI transaction.
        }
        else
        {
            temp = createSPIByteStream(protDev, reg, 0, READ_ACK, &eventCount); // First byte also has address in from of it. Treat it like a normal SPI transaction.
        }
        uint8_t stopFrame [3]; // The last 3 events contain the CS going back up. Store these events to a local variable to append at the end of the transaction later.
        stopFrame[0] = temp[eventCount - 3];
        stopFrame[1] = temp[eventCount - 2];
        stopFrame[2] = temp[eventCount - 1];
        totalEvents = eventCount + protDev.dataBits*2*(numRegs - 1); // totalEvent count will contain single SPI event (eventCount contains this) + (2*(number of data bits)*(total number of bytes to be written - 1))
        writeBytes = (uint8_t*)malloc(totalEvents*sizeof(uint8_t));  // Create writeBytes which will contain entire transaction.
        int32_t writeIndex = 0;
        memcpy(&writeBytes[writeIndex], temp, eventCount - 3); // Copy contents of temp into appropriate location in writeBytes. Do not include CS event at end.
        free(temp);
        writeIndex = eventCount - 3; // Start at loication of end of last byte without CS event at end.
        for(int i = 1; i < numRegs; i++) // Each addtional byte that needs to be written or read back from.
        {
            if(write == TRUE)
            {
                temp = createSPIByteStream(protDev, reg, data[i], WRITE_STREAM_WORD, &eventCount); // WRITE_STREAM_WORD means only include data portion of frame and bypass Address.
            }
            else
            {
                temp = createSPIByteStream(protDev, reg, 0, READ_STREAM_WORD_ACK, &eventCount); // READ_STREAM_WORD_ACK means only include data portion of frame and bypass Address.
            }
            memcpy(&writeBytes[writeIndex], temp, eventCount); // copy contents of temp into appropriate location in writeBytes.
            free(temp);
            writeIndex += eventCount;
        }
        writeBytes[writeIndex] = stopFrame[0]; //a ppend the last three bytes of writeBytes with CS stop event.
        writeBytes[writeIndex + 1] = stopFrame[1];
        writeBytes[writeIndex + 2] = stopFrame[2];
        ftStatus = FT_Write(dev, writeBytes, writeIndex + 3, (LPDWORD)&byteCount); // Total number of bytes to be written is eventCount * (number of registers)
        if(writeDebug) // Print debug to screen.
        {
            printPattern(writeBytes, writeIndex + 3, debugDelim);
        }
        free(writeBytes);
        if(write == FALSE) // Readback. Need to parse each data byte frame to readback.
        {
            uint8_t *rxBytes = (uint8_t*)malloc(sizeof(uint8_t)*writeIndex + 3);
            ftStatus = FT_Read(dev, rxBytes, writeIndex + 3, (LPDWORD)&byteCount); // Read the data into rxBytes.
            if(ftStatus != FT_OK || byteCount != writeIndex + 3) // Just like with FT_write Check the FT_Status good and the correct number of bytes were retrieved.
            {
                free(rxBytes); // Free the rxBytes pointer
                return -1; // Leave
            }
            int32_t clkStrobe = -1; // To contain location of initial clock strobe.
            for(int i = 0; i < eventCount; i++)
            {
                if((rxBytes[i] & (0x01 << protDev.CSnum)) == 0)  // Verify CS is low
                {
                    clkStrobe = ((rxBytes[i] & (0x01 << protDev.CLKnum)) >> protDev.CLKnum);
                    if(protDev.posedge == FALSE)
                    {
                         clkStrobe = 0x01 & ~clkStrobe; // invert clock to make finding initial clock strobe the same between rising and falling edge.
                    }
                    if(clkStrobe == 0x01) // clock strobe found! Take note of the value and exit loop.
                    {
                        clkStrobe = i; // First Clock Strobe found at index i. Store this into clockStrobe and leave this loop.
                        i = eventCount;
                    }
                }
            }
            uint32_t offset = protDev.addrBits*2 + clkStrobe; // First data frame is after address portion of transaction
            if(protDev.RWbit == TRUE)
            {
                offset += 2; // Add two events for RW bit.
            }
            for(int i = 0; i < numRegs; i++) // Read back each frame. This will be called for each data frame.
            {
                data[i] = parseByteStream(&rxBytes[offset + 2*i*protDev.dataBits], protDev.SDOnum, 2, protDev.dataBits); // Parse through each frame
            }
            if(readDebug) // readDebug is supported for streaming.
            {
                printPattern(rxBytes, writeIndex + 3, debugDelim); // writeIndex is still before CS event. Add 3 to print entire transaction.
            }
            free(rxBytes);
        }
    }
    else // I2C device.
    {
        if(write == TRUE)
        {
            temp = createI2CByteStream(protDev, reg, data[0], WRITE, &eventCount); // First frame of I2C Transaction, including device address, register and first data frame.
        }
        else
        {
            if(numRegs == 1)
            {
                temp = createI2CByteStream(protDev, reg, 0, READ_NACK, &eventCount); // First (and only) frame of I2C Transaction, including device address, register and first (and only) data frame.
            }
            else
            {
                temp = createI2CByteStream(protDev, reg, 0, READ_ACK, &eventCount); // First frame of I2C Transaction, including device address, register and first  data frame. ACK is provided at end due to continued frames after.
            }
        }
        uint8_t stopFrame [3]; // Just like with SPI the last three events contain a STOP event. Store these to append later.
        stopFrame[0] = temp[eventCount - 3];
        stopFrame[1] = temp[eventCount - 2];
        stopFrame[2] = temp[eventCount - 1];
        totalEvents = eventCount + protDev.bytesPerReg*36*(numRegs - 1); // eventCount
        writeBytes = (uint8_t*)malloc(totalEvents*sizeof(uint8_t));
        int32_t writeIndex = 0;
        memcpy(&writeBytes[writeIndex], temp, eventCount - 3); // copy contents of temp into appropriate location in writeBytes.
        writeIndex += eventCount - 3; // The last three bytes will be overwritten.
        free(temp);
        for(int i = 1; i < numRegs; i++) // First byte written. Set i = 1.
        {
            if(write == TRUE)
            {
                temp = createI2CByteStream(protDev, reg, data[i], WRITE_STREAM_WORD, &eventCount); // This will only write data frame including necessary ACK bits.
            }
            else
            {
                if(i == (numRegs - 1))
                {
                    temp = createI2CByteStream(protDev, reg, 0, READ_STREAM_WORD_NACK, &eventCount); // This will only write data frame including necessary ACK bits. This is the last byte therefore controller should NACK at the end.
                }
                else
                {
                    temp = createI2CByteStream(protDev, reg, 0, READ_STREAM_WORD_ACK, &eventCount); // This will only write data frame including necessary ACK bits.
                }
            }
            memcpy(&writeBytes[writeIndex], temp, eventCount); // copy contents of temp into appropriate location in writeBytes.
            free(temp);
            writeIndex += eventCount; //keep track of writeIndex
        }
        writeBytes[writeIndex] = stopFrame[0]; //append final stop event to end of transaction.
        writeBytes[writeIndex + 1] = stopFrame[1];
        writeBytes[writeIndex + 2] = stopFrame[2];
        ftStatus = FT_Write(dev, writeBytes, writeIndex + 3, (LPDWORD)&byteCount); // Total number of bytes to be written is eventCount * (number of registers)
        if(writeDebug) // Stream mode supports debug print
        {
            printPattern(writeBytes, writeIndex + 3, debugDelim);
        }
        free(writeBytes);
        if(write == FALSE) // need to parse for readback.
        {
            uint8_t *rxBytes = (uint8_t*)malloc(sizeof(uint8_t)*writeIndex + 3);
            ftStatus = FT_Read(dev, rxBytes, writeIndex + 3, (LPDWORD)&byteCount); // Read the data into rxBytes.
            if(ftStatus != FT_OK || byteCount != writeIndex + 3) // Just like with FT_write Check the FT_Status good and the correct number of bytes were retrieved.
            {
                free(rxBytes); // Free the rxBytes pointer
                return -1; // Exit on error.
            }
            int32_t clkStrobe = -1; // initialize variable to contain inital clock strobe location.
            for(int i = 0; i < eventCount; i++)
            {
                if ((rxBytes[i] >> protDev.SCLnum & 0x01) == 0) // First event with a low clock
                {
                    clkStrobe = i;  // Store this event in clockStrobe
                    i = eventCount; // Force exit from the for loop
                }
            }
            for(int i = 0; i < numRegs; i++) // Each register transaction.
            {
                data[i] = 0;
                for(int j = 0; j < protDev.bytesPerReg; j++) //Each byte within register frame. Skip the ACK bits.
                {
                    data[i] |= (parseByteStream(&rxBytes[clkStrobe + 76 + 36*protDev.bytesPerAddr + i*36*protDev.bytesPerReg + j*36], protDev.SDAnumR, 4, 8) << 8*(protDev.bytesPerReg - j - 1));
                }
            }
            if(readDebug) // Streaming supports printDebug.
            {
                printPattern(rxBytes, writeIndex + 3, debugDelim);
            }
            free(rxBytes);
        }
    }
    return 0;
}
/*
    The next several functions are for using device indexing. Essentially setup the device index using mapHandles and then
    after that <write_reg>, <read_reg>, <write_regs>, <read_regs>, <write_reg_stream> and <read_reg_stream> can all be called using the appropriate index.
    can all be called for each corresponding deviceID
*/
/*
    <int32_t mapHandles>
    This function is used to map handles to an index. It modifies the global arrays FTDI_Handles[] and PROT_handles[] at index deviceID to contain pointers to the respective handles.
    It accepts the following parameters.
    <FT_HANDLE dev>             A handle (void pointer) to an FTDI device structure.
    <PROT_HANDLE spi>           A handle (void pointer) to a protocol description structure.
    <uint32_t deviceID>         An unsigned integer containing the address to be used to map the device. This needs to be less than global variable DEVICE_COUNT which is defined at compile time.
    Returns a 0 successful. Returns a -1 if an invalid index is provided.
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT mapHandles(FT_HANDLE dev, PROT_HANDLE prot, uint32_t deviceID)
{
    if(deviceID > DEVICE_COUNT) //Out of bounds index. Return -1 as error.
    {
        return -1;
    }
    FTDI_Handles[deviceID] = dev;   //point FTDI_Handles at index deviceID to point to dev.
    PROT_Handles[deviceID] = prot;  //point PROT_Handles at index deviceID to point to prot.
    return 0;
}
/*
    <int32_t write_reg>
    This function is used as a way to write to a device based off its deviceID. It can be called once mapHandles has been called to map the FT_HANDLE and PROT_HANDLE to a particular index.
    It automatically calls writeSPI or writeI2C depending on the PROT_HANDLE that is indexed.
    It accepts the following parameters
    <uint32_t deviceID>         An unsigned integer that provides the index.
    <uint32_t reg>              An unsigned integer that provides the register to write to.
    <uint32_t data>             An unsigned integer that provides the data to be written.
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK). For I2C transactions it will return the ACK bits mapped to each bit of the returned integer. (0 if successful).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT write_reg(uint32_t deviceID, uint32_t reg, uint32_t data)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        struct PROT_DES prot;
        memcpy(&prot, PROT_Handles[deviceID], sizeof(struct PROT_DES)); // Load prot structure into local variable to check if I2C or SPI device.
        if(prot.SPI_I2C == TRUE)
        {
            return writeSPI(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data);
        }
        else
        {
            return writeI2C(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data);
        }
    }
}
/*
    <int32_t read_reg>
    This function used as a way to read from a device based off its deviceID. It can be called once mapHandles has been called to map the FT_HANDLE and PROT_HANDLE to a particular index.
    It automatically calls readSPI or readI2C depending on the PROT_HANDLE that is indexed.
    It accepts the following parameters
    <uint32_t deviceID>         An unsigned integer that provides the index.
    <uint32_t reg>              An unsigned integer that provides the register to write to.
    Returns read value if successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT read_reg(uint32_t deviceID, uint32_t reg)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        struct PROT_DES prot;
        memcpy(&prot, PROT_Handles[deviceID], sizeof(struct PROT_DES)); // Load prot structure into local variable to check if I2C or SPI device
        if(prot.SPI_I2C == TRUE)
        {
            return readSPI(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg);
        }
        else
        {
            return readI2C(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg);
        }
    }
}
/*
    <int32_t write_regs>
    This function is used as a way to write multiple registers to a device based off its deviceID. It can be called once mapHandles has been called to map the FT_HANDLE and PROT_HANDLE to a particular index.
    It is a simple function that calls multiRegRW in write mode with the corresponding handles pass to it based of the index given.
    It accepts the following parameters. Each transaction is complete, no streaming is used.
    <uint32_t       deviceID>      An unsigned integer that provides the index.
    <uint32_t []    reg>           An unsigned integer array that provides the registers to write to.
    <uint32_t []    data>          An unsigned integer array that provides the data to be written to each corresponding register.
    <uint32_t       numRegs>       The number of registers to be read.
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK). For I2C transactions it will return the ACK bits mapped to each bit of the returned integer. (0 if successful).
    This function is intended to be called outside of this source file. */
int32_t DLL_EXPORT write_regs(uint32_t deviceID, uint32_t reg [], uint32_t data [], uint32_t numRegs)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        return multiRegRW(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data, numRegs, TRUE);
    }
}
/*
    <int32_t read_regs>
    This function is used as a way to read multiple registers to a device based off its deviceID. It can be called once mapHandles has been called to map the FT_HANDLE and PROT_HANDLE to a particular index.
    It is a simple function that calls multiRegRW in read mode with the corresponding handles pass to it based of the index given.
    It accepts the following parameters. Each transaction is complete, no streaming is used.
    <uint32_t       deviceID>      An unsigned integer that provides the index.
    <uint32_t []    reg>           An unsigned integer array that provides the registers to write to.
    <uint32_t []    data>          An unsigned integer array that will be updated by reference with readback data.
    <uint32_t       numRegs>       The number of registers to be read.
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT read_regs(uint32_t deviceID, uint32_t reg [], uint32_t data [], uint32_t numRegs)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        return multiRegRW(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data, numRegs, FALSE);
    }
}
/*
    <int32_t write_reg_stream>
    This function is a way to write I2C or SPI streaming transactions. Both SPI and I2C devices can support streaming and there is a fairly standard way to do it.
    For SPI devices, after the first data portion is written, the clock continues to strobe and the next data frame is written. This continues until the last register is written and then CS is pulled high to end the frame.
    For I2C devices it is similar, the STOP condition is not presented until the last byte is written.
    The Peripheral must support streaming mode for it to work. Some devices do not support it.
    It accepts the following parameters. Each transaction is complete.
    <uint32_t       deviceID>         An unsigned integer that provides the index.
    <uint32_t       reg>           An unsigned integer that provides the register to start the stream write at.
    <uint32_t []    data>          An unsigned integer array that provides the data to be written to each corresponding register.
    <uint32_t       numRegs>       The number of registers to be read.
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT write_reg_stream(uint32_t deviceID, uint32_t reg, uint32_t data [], uint32_t numRegs)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        return multiRegStreamRW(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data, numRegs, TRUE);
    }
}
/*
    <int32_t read_reg_stream>
    This function is a way to read I2C or SPI streaming transactions. Both SPI and I2C devices can support streaming and there is a fairly standard way to do it.
    For SPI devices, after the first data portion is read, the clock continues to strobe and the next data frame is readback. This continues until the last register is readback and then CS is pulled high to end the frame.
    For I2C devices it is similar, the STOP condition is not presented until the last byte is readback.
    The Peripheral must support streaming mode for it to work. Some devices do not support it.
    It accepts the following parameters. Each transaction is complete.
    <uint32_t       deviceID>      An unsigned integer that provides the index.
    <uint32_t       reg>           An unsigned integer that provides the register to start the stream read at.
    <uint32_t []    data>          An unsigned integer array that will be updated by reference with readback data.
    <uint32_t       numRegs>       The number of registers to be read.
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT read_reg_stream(uint32_t deviceID, uint32_t reg, uint32_t data [], uint32_t numRegs)
{
    if(FTDI_Handles[deviceID] == 0 || PROT_Handles[deviceID] == 0) //Verify the index has been set. Both handle arrays are initialized to 0. 0 is not a valid memory address.
    {
        return -1;
    }
    else
    {
        return multiRegStreamRW(FTDI_Handles[deviceID], PROT_Handles[deviceID], reg, data, numRegs, FALSE);
    }
}
/*
    The Next two functions are for GPIO bit banging.
*/
/*
    <int32_t toggleSingleBit>
    This function is useful for generating a single pulse on a GPIO (like reset).
    It can generate both high-low-high and low-high-low pulses with a single call.
    It accepts the following parameters.
    <FT_HANDLE  dev>            FTDI Handle of target FTDI device.
    <int8_t     bitNum>         An unsigned 8 bit integer indicating which GPIO to toggle. (0-7)
    <BOOL       toggleLow>      A BOOL used to indicate if the pattern should go high-low-high (TRUE) or low-high-low (FALSE).
    Returns a 0 successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT toggleSingleBit(FT_HANDLE dev, uint8_t bitNum, BOOL toggleLow, BOOL singleEdge)
{
    // These #defines can be changed only at compile time.
    #define FRONT_PAD 10    // how many events to hold low initially. (Invert for toggleLow =  TRUE)
    #define TOGGLE_LEN 20   // how many events to hold high. (Invert for toggleLow = TRUE)
    #define BACK_PAD 10     // how may events to hold low after going high. (Invert for toggleLow =  TRUE)
    #define TOTAL_LEN FRONT_PAD+TOGGLE_LEN+BACK_PAD
    if(bitNum > 7)
    {
        return -1; // invalid number.
    }
    uint8_t bitMask = 0x01 << bitNum; // bit mask contains which bit to toggle.
    if(FT_SetBitMode(dev, bitMask, FT_BITMODE_SYNC_BITBANG) != FT_OK)
    {
        return -1; // FTDI ERROR. Exit.
    }
    uint8_t writeArray[TOTAL_LEN] = {0};
    for(int i = 0; i < TOTAL_LEN; i++)
    {
        if(i >= FRONT_PAD && (singleEdge == TRUE || i < (TOTAL_LEN - BACK_PAD)))
        {
            writeArray[i] = bitMask; // set bits high.
        }
        if(toggleLow)
        {
            writeArray[i] ^= bitMask; // invert every bit if toggleLow is TRUE
        }
    }
    int32_t bytes; // to store number of bytes written. Needed for FT_write call.
    if(FT_Write(dev, writeArray, TOTAL_LEN, (LPDWORD)&bytes) != FT_OK)
    {
        return -1;
    }
    return 0;
}
/*
    <int32_t writeCustomPattern>
    This function is useful for loading custom byte sequnce and sending it to the FTDI chip.
    It accepts the following parameters.
    <FT_HANDLE  dev>                FTDI Handle of target FTDI device.
    <int8_t     MASK>               An unsigned 8 bit integer masking which bits are drivers.
    <uint8_t    *byteSequence>      An unsigned 8 bit integer array pointer that points to the first element of the byte sequence.
    <uint32_t   sequenceLen>        An unsigned 32 bit integer indicating how long the sequence is.
    Returns a the number of bytes written if successful. Returns a -1 if an error occurs (FT_STATUS != FT_OK).
    This function is intended to be called outside of this source file.
*/
int32_t DLL_EXPORT writeCustomPattern(FT_HANDLE dev, uint8_t MASK, uint8_t *byteSequence, int32_t sequenceLen)
{
    if(FT_SetBitMode(dev, MASK, FT_BITMODE_SYNC_BITBANG) != FT_OK) // set which bits are outputs on the 8 bit FTDI bus.
    {
        return -1;
    }
    int32_t bytes; // to store number of bytes written. Needed for FT_write call.
    if(FT_Write(dev, byteSequence, sequenceLen, (LPDWORD)&bytes) != FT_OK)
    {
        return -1;
    }
    return bytes;
}
/*
The next three functions are simple inline wrapper functions that can be called at the end to close and free up memory.
*/
inline DLL_EXPORT int32_t closeFTDI(FT_HANDLE dev)
{
    return FT_Close(dev);
}

inline DLL_EXPORT int32_t deleteSPIDev(PROT_HANDLE spi)
{
    free(spi);
    return 0;
}
inline DLL_EXPORT int32_t deleteI2CDev(PROT_HANDLE i2c)
{
    free(i2c);
    return 0;
}
// This function simply enables the global flags that are used for printing debug information.
inline DLL_EXPORT int32_t enableDebugPrint(BOOL writeDB, BOOL readDB)
{
    writeDebug = writeDB;
    readDebug = readDB;
    return 0;
}
// This function simply sets the global debugDelim varaible equal to the passed delim character.
inline DLL_EXPORT int32_t setDebugDelim(char delim)
{
    debugDelim = delim;
    return 0;
}
// Very basic version control. Just to see what DLL is being worked with.
inline DLL_EXPORT void getDLLVersion(){

    printf("TI_FTDI DLL Version %d.%d.%d", MAJOR, MINOR, PATCH);

}
// Extra DLL stuff.
#ifdef BUILD_DLL
DLL_EXPORT BOOL APIENTRY DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
            // attach to process
            // return FALSE to fail DLL load
            break;

        case DLL_PROCESS_DETACH:
            // detach from process
            break;

        case DLL_THREAD_ATTACH:
            // attach to thread
            break;

        case DLL_THREAD_DETACH:
            // detach from thread
            break;
    }
    return TRUE; // successful
}
#endif
/*
    Begin Internal functions. These functions are NOT intended to be called outside of this source file and are NOT exported to the DLL.
*/
/*
    <void printPattern>
    This function is useful for printing input and output data for debug. It can used by enabling writeDebug or readDebug using enableDebugPrint function.
    It accepts the following parameters.
    <uint8_t *writeArray>       A PROT_DES structure that contains all SPI device information
    <int len>                   An unsigned integer of how long the byte array is.
    <char delim>                The delimiter character that is inserted between each element. 'n' is used for none.
    This function does not return anything and only prints to the screen.
    This function is NOT intended to be called outside of this source file. It is not present in the header and accepts a PROT_DES struct directly as a parameter.
    It is called in both writeSPI and readSPI function calls.
*/
static void printPattern(uint8_t *writeArray, uint32_t len, char delim)
{
    uint8_t** patternMatrix; //2D array consisting of byte array. Each bit is represented as a uint8_t as working with bytes directly is a lot of extra work.
    patternMatrix = malloc(len*sizeof(*patternMatrix));
    for (int i = 0; i < len; i++)   //
    {
        patternMatrix[i] = malloc(8*sizeof(patternMatrix[0]));
    }
    for (int i = 0; i < 8; i++)
    {
        for(int j = 0; j < len; j++)
        {
            patternMatrix[j][i] = 0;
            if(((writeArray[j] >> i) & 0x01) == 0x01)
            {
                patternMatrix[j][i] = 1;
            }
            if (delim == 'n')
            {
                printf("%d", patternMatrix[j][i]);
            }
            else
            {
                printf("%d%c", patternMatrix[j][i], delim);
            }
        }
    printf("\n");
    }
    for (int i = 0; i < len; i++)
    {
        free(patternMatrix[i]);
    }
    printf("\n");
    free(patternMatrix);
}
/*
    <uint8_t * creatSPIByteStream>
    This function creates the byte stream that is sent to the FT_write API call. It is used for both write and read transactions.
    It accepts the following parameters.
    <struct PROT_DES spiDev>    A PROT_DES structure that contains all SPI device information
    <uint32_t addr>             An unsigned integer containing the address to be written to.
    <uint32_t data>             An unsigned integer containing the data to be written to the provided address.
    <enum sequenceMode mode>    enum used to tell function what type of bytestream is being created. (Full or partial stream as well as read/write). ACK and NACK are treated the same as these are only useful for I2C transactions.
    <uint32_t *eventCount>      A pointer to an integer that will contain the number of events present in the transaction. There are two events per clock strobe.
    Returns a pointer to the byte array that will be written.
    This function is NOT intended to be called outside of this source file. It is not present in the header and accepts a PROT_DES struct directly as a parameter.
    It is called in both writeSPI and readSPI function calls.
*/
static uint8_t * createSPIByteStream(struct PROT_DES spiDev, uint32_t addr, uint32_t data, sequenceMode mode, uint32_t *eventCount)
{
    *eventCount = spiDev.addrBits*2 + spiDev.dataBits*2 + 6; //Each bit that is written require two events. In addtion 6 events are used for CSbar padding.
    uint8_t RWoffset = 0;       //Will set to 2 for RW bit
    if (spiDev.RWbit == TRUE)
    {
        *eventCount += 2;       //An additional bit needs to be transfered (RWbit) so two more events are necessary.
        RWoffset = 2;           //increase offset two extra events to account for RWbit
    }
    if (mode == READ_NACK) //READ_ACK is used for read in SPI mode.
    {
        mode = READ_ACK;
    }
    uint32_t events = *eventCount;                      //create a local variable that is equal to the value at the eventCount pointer.
    uint8_t *writeArray;                                //Create pointer to writeArray
    writeArray = malloc (sizeof (uint8_t) * (events));  //allocate memory on the heap for the byte Array which will be eventCount long.
    uint8_t clkBit = 0x01 << spiDev.CLKnum;             //Create the masks for each respective bit.
    uint8_t csBit = 0x01 << spiDev.CSnum;
    uint8_t SDIbit = 0x01 << spiDev.SDInum;
    uint8_t OEbit = 0x01 << spiDev.OEnum;
    int32_t addrIndex = spiDev.addrBits;                //Indexes will keep track of address and data index during bit banging.
    int32_t dataIndex = spiDev.dataBits;
    if(mode == WRITE || mode == READ_ACK)
    {
        for (int i = 0; i < events; i++) //This for loop will create each event. Byte by byte.
        {
            writeArray[i] = 0;                              //initialize writeArray element to 0 as all bits are or'd as necessary.
            if(spiDev.OE_EN == TRUE && spiDev.OE_pol == FALSE) //If OE is used and the polarity if false, the OE pin should always be high unless read back is being performed.
            {
                writeArray[i] |= OEbit;                     //Set all the OE bit high at all events. Set it low later for the read back events.
            }
            //Clock Stuff
            if(i > 3 && (i % 2 == 0) && i < events - 2) //Toggle the clock during these events. There is some padding for CSbar framing.
            {
                writeArray[i] |= clkBit; //Note that the clock is set high every other bit thanks to the % 2 operation.
            }
            if (spiDev.posedge == FALSE)
            {
                writeArray[i] ^= clkBit;                //Inverse of posedge clock. Clock will be high at beginning and end of transaction as expected with negedge clock.
            }
            //CSB
            if((i < 2 || i > events - 3)) //CSbar framing. CSbar is high at beginning and end.
            {
                writeArray[i] |= csBit;
            }
            //SDI
            if((i == 3 || i == 4) && mode == READ_ACK) //RWbit set high for readback.
            {
                if(spiDev.RWbit == TRUE)
                {
                    writeArray[i] |= SDIbit;
                }
            }
            if(i > (2 + RWoffset) && i < spiDev.addrBits*2 + (3 + RWoffset)) //Address portion of frame.
            {
                if (i % 2 == 1)
                {
                    addrIndex--; //move index down by one every other event (single clock period)
                }
                if(((addr >> addrIndex) & 0x01) == 0x01)
                {
                    writeArray[i] |= SDIbit; //The respective address bit needs to be set high.
                }
            }
            else if(i >= spiDev.addrBits*2 + 3 + RWoffset && i < events - 3) //Data portion of frame
            {
                if(mode == WRITE) //Write is true so load data into SDI line
                {
                    if (i % 2 == 1)
                    {
                        dataIndex--; //move index down by every other event (single clock period)
                    }
                    if(((data >> dataIndex) & 0x01) == 0x01)
                    {
                        writeArray[i] |= SDIbit; //set respective data bit high
                    }
                }
                else
                {
                    if (spiDev.OE_EN == TRUE)
                    {
                        writeArray[i] ^= OEbit; //XOR the OEbit so the peripheral has control of the line. No need to worry about polarity as OE was set already depending on polarity.
                    }
                }
            }
        }
    }
    else
    {
        events = spiDev.dataBits*2;
        *eventCount = events;
        for(int i = 0; i < events; i++)
        {
            writeArray[i] = 0;
            if(spiDev.OE_EN == TRUE && spiDev.OE_pol == FALSE) //If OE is used and the polarity if false, the OE pin should always be high unless read back is being performed.
            {
                writeArray[i] |= OEbit;                     //Set all the OE bit high at all events. Set it low later for the read back events.
            }
            if(spiDev.OE_EN == TRUE && mode == READ_STREAM_WORD_ACK) //Readback mode. Since this is read byte only all bits will need to be in readback mode.
            {
                writeArray[i] ^= OEbit;
            }
            if(i % 2 == 1)
            {
                writeArray[i] |= clkBit; //Note that the clock is set high every other bit thanks to the % 2 operation.
            }
            if (spiDev.posedge == FALSE)
            {
                writeArray[i] ^= clkBit;                //Inverse of posedge clock. Clock will be high at beginning and end of transaction as expected with negedge clock.
            }
            if(mode == WRITE_STREAM_WORD)
            {
                if (i % 2 == 0)
                {
                    dataIndex--; //move index down by every other event (single clock period)
                }
                if(((data >> dataIndex) & 0x01) == 0x01)
                {
                    writeArray[i] |= SDIbit; //set respective data bit high
                }
            }
        }
    }
    return writeArray; //return the array
}
/*
    <uint8_t * creaI2CByteStream>
    This function creates the byte stream that is sent to the FT_write API call. It is used for both write and read transactions in I2C.
    It accepts the following parameters.
    <struct PROT_DES i2cDev>    A PROT_DES structure that contains all protocol information
    <uint32_t reg>              An unsigned integer containing the register address to be written to.
    <uint32_t data>             An unsigned integer containing the data to be written to the provided register address.
    <uint32_t write_Mode>       An unsigned integer that is 0 for read (with NACK at end), 1 for write, 2 for read (with ACK at end), 3 for read byte (with ACK), 4 for read byte (with NACK) and 5 for write byte.
    <uint32_t *eventCount>      A pointer to an integer that will contain the number of events present in the transaction. There are four events per clock strobe in an I2C transaction.
    Returns a pointer to the byte array that will be written.
    This function is NOT intended to be called outside of this source file. It is not present in the header and accepts a PROT_DES struct directly as a parameter.
    It is called in both writeI2C and readI2C function calls.
*/
static uint8_t * createI2CByteStream(struct PROT_DES i2cdev, uint32_t reg, uint32_t data, sequenceMode mode, uint32_t *eventCount)
{
    typedef enum //I2C is easiest to implement as a FSM. This enum will be used for state names.
    {
        START,
        ADDR,
        REG,
        DATA,
    }   i2cState;
    BOOL read = TRUE;
    if (mode == WRITE || mode == WRITE_STREAM_WORD)
    {
        read = FALSE;
    }
    //First determine the number of events needed based off sequenceMode. Each event will be a single byte that is passed to FTDI device.
    i2cState currState = START; //initialize current and next state to START
    i2cState nextState = START;
    int32_t index = 6; //i2c addresses are always 7 bits long (6:0) with trailing R/W bit
    int32_t indexdata;   //Used during data portion of FSM. A second index to keep track data while the index variable keeps track of ACK bits.
    if(mode == WRITE || mode == READ_ACK || mode == READ_NACK) //Full I2C transaction including address, register data fields.
    {
        if(mode == WRITE)
        {
            *eventCount = 42; //includes 8 bits of address and 8 bits of register as well as ack bits and start/stop bit
        }
        else
        {
            *eventCount = 81; //Includes ST ADDR REG ST ADDR and STOP bits.
        }
        *eventCount += 36*i2cdev.bytesPerAddr;
        *eventCount += 36*i2cdev.bytesPerReg; //Data portion of transaction which will grow with more bytes. 36 bits (9x4 bits) per byte (8 bits data + 1 bit ACK)
    }
    else
    {
        *eventCount = 36*i2cdev.bytesPerReg; //Single word being written to or read from. 9 bits per byte (including ACK) with 4 events per bit.
        currState = DATA;
        nextState = DATA;
        index = 9*i2cdev.bytesPerReg - 1; //index is decremented every clock strobe. There are 9 clock strobes per frame (8 bits + ACK bit)
        indexdata = 8*i2cdev.bytesPerReg - 1; //indexdata is not decremented during ACK bit and is used to index actual data being sent.
    }
    uint32_t events = *eventCount; //local variable to use.
    uint8_t *writeArray; //Will return writeArray address
    writeArray = malloc(sizeof (uint8_t) * events); //dynamically allocate writeArray memory on the heap. free will need to be called after this function executes.
    uint8_t SCLbit = 0x01 << i2cdev.SCLnum;
    uint8_t SDAWbit = 0x01 << i2cdev.SDAnumW;
    uint8_t OEbit = 0x01 << i2cdev.OEnum;
    uint8_t clkCounter = 0; //Used throughout. Essentially counts 1,2,3,4 and repeats. When it is equal to 2 or 3 clock is set high. During 1 and 4 the clock is set low. (With the exception of START state).
    BOOL readBackPass = FALSE;
    for(int i = 0; i < events; i++) //Will leave state machine when the correct number of events have been processed.
    {
        writeArray[i] = 0;
        if(i2cdev.OE_EN == TRUE && i2cdev.OE_pol == FALSE) //If OE is used and the polarity if false, the OE pin should always be high unless read back is being performed.
        {
            writeArray[i] |= OEbit;  //Set all the OE bit high at all events. Set it low later for the read back events.
        }
        clkCounter += 1; //clkcounter is used for keeping track of where we are in the clock cycle. It is reset to 0 when a state changes. Otherwise it counts 1,2,3,4,1,2,3,4
        switch(currState) //Each byte that is transmitted has its own state.
        {
            case(START): //This state generates the start bit
                if(clkCounter == 1)
                {
                    writeArray[i] |= (SCLbit + SDAWbit); //First event make SCL and SDA high
                }
                if(clkCounter == 2)
                {
                    writeArray[i] |= SCLbit; //Next two events pull SDA low and leave clock high.
                }
                if(clkCounter == 3)
                {
                    writeArray[i] |= SCLbit;
                    nextState = ADDR; //The next state is ADDR where the devices I2C address is sent along with R/W bit.
                    clkCounter = 0; //Set clkCounter to 0. It will immediately be incremented to 1 on the next event.
                }
                break;
            case(ADDR):
                if(index >= 0) //Index counts 6,5,4,3,2,1,0,-1,-2. When it is >= 0 the address is being shifted out. -1 means R/W bit and -2 means ACK bit.
                {
                    if(((i2cdev.addr >> index) & 0x01))
                    {
                        writeArray[i] |= SDAWbit; //The corresponding address bit at index is 1.
                    }
                }
                else //Index is either -1 or -2.
                {
                    if(read == TRUE && index == -1 && readBackPass == TRUE) //During a write or during the first pass of a read readBackpass will be False.
                    {
                        writeArray[i] |= SDAWbit; //on the second address frame of a read, the RW bit should be high indicating a read.
                    }
                    if(index == -2) //ACK bit. let the bus go high so the peripheral can pull it low for an ACK. OE should also be toggled if necessary for a peripheral read.
                    {
                        writeArray[i] |= SDAWbit;
                        if(i2cdev.OE_EN == TRUE)
                        {
                            writeArray[i] ^= OEbit; //All OEbits were set high if OEpol is FALSE. flip the bits that need to be low for readback.
                        }
                    }
                }
                if(clkCounter == 2 || clkCounter == 3) //Clock Stuff. A clock cycle is 4 events total with the clock being high in the middle two events.
                {
                    writeArray[i] |= SCLbit; //Clock should be high.
                }
                if(clkCounter == 4) //The last event of a clock period. Pull the clock low and decrement the index.
                {
                    clkCounter = 0; //Clock counter is set to 0 but immediately be incremented to 1 at the beginning of the next event.
                    index--;
                }
                if(index == -3) //Time to go the next event.
                {
                    if(readBackPass == FALSE) //This means the address field was either the first field of a readback or the address field of a write. The next state is the REG portion
                    {
                        nextState = REG;
                        clkCounter = 0;
                        index = 9*i2cdev.bytesPerAddr - 1;
                        indexdata = 8*i2cdev.bytesPerAddr - 1;
                        readBackPass = TRUE; //Set readbackpass to true, if the ADDR state is executed again this pass, a read is being performed.
                    }
                    else //readbackpass was true, meaning the next state is data and not reg.
                    {
                        index = 9*i2cdev.bytesPerReg - 1; //Two index variables are used to simplify multibyte devices. One keeps track of the index including the ACK bits. The other keeps track of the bits of the data itself.
                        indexdata = 8*i2cdev.bytesPerReg - 1;
                        nextState = DATA;
                    }
                }
                break;
            case(REG):
                if(index % 9 == 0) //If index %9 = 0, This is ack bit
                {
                    writeArray[i] |= SDAWbit; //set open drain bus high to allow peripheral to pull BUS low for acknowledge.
                    if(i2cdev.OE_EN == TRUE)
                    {
                        writeArray[i] ^= OEbit; //invert OE as well if enabled.
                    }
                }
                else //Regular data bit.
                {
                    if((reg >> indexdata) & 0x01)
                    {
                        writeArray[i] |= SDAWbit; //set open drain bus high.
                        if(i2cdev.OE_EN == TRUE && read == TRUE)
                        {
                            writeArray[i] ^= OEbit; //invert OE as well if enabled.
                        }
                    }
                }
                if(clkCounter == 2 || clkCounter == 3) //Same clock code.
                {
                    writeArray[i] |= SCLbit;
                }
                if(clkCounter == 4) //end of clock period, decrement address and reset clock counter.
                {
                    clkCounter = 0;
                    if(index % 9 != 0) //dont want to decrement indexdata during a ACK bit.
                    {
                        indexdata--;
                    }
                    index--; //index is always decremented regardless.
                }
                if(index == -1) //Next frame because index is -2.
                {
                    if(read == FALSE)
                    {
                        nextState = DATA; //Time to go to data for writing.
                        index = 9*i2cdev.bytesPerReg - 1; //index is decremented every clock strobe. There are 9 clock strobes per frame (8 bits + ACK bit)
                        indexdata = 8*i2cdev.bytesPerReg - 1; //indexdata is not decremented during ACK bit and is used to index actual data being sent.
                    }
                    else
                    {
                        nextState = START; //Readback requires an additional start bit. Go back to START.
                        index = 6; //Prepare index for address portion.
                    }
                }
                break;
            case(DATA):
                if(index % 9 == 0) //If index %9 = 0, This is ack bit
                {
                    if(read == FALSE || (indexdata == -1 && (mode == READ_NACK || mode == READ_STREAM_WORD_NACK)))
                    {
                        writeArray[i] |= SDAWbit; //set open drain bus high to allow peripheral to pull BUS low for acknowledge. Or let it high for a readback NACK (End of transaction).
                        if(i2cdev.OE_EN == TRUE)
                        {
                            writeArray[i] ^= OEbit; //invert OE as well if enabled.
                        }
                    }
                }
                else //Regular data bit.
                {
                    if(((data >> indexdata) & 0x01) || read == TRUE) //for readback SDAW should always be high to allow the peripheral to pull the BUS low. Also let bus go high if a data bit is 1.
                    {
                        writeArray[i] |= SDAWbit; //set open drain bus high.
                        if(i2cdev.OE_EN == TRUE)
                        {
                            writeArray[i] ^= OEbit; //invert OE as well if enabled.
                        }
                    }
                }
                if(clkCounter == 2 || clkCounter == 3)
                {
                    writeArray[i] |= SCLbit; //strobe clock as usual
                }
                if(clkCounter == 4) //Decrement index
                {
                    clkCounter = 0;
                    if(index % 9 != 0) //dont want to decrement indexdata during a ACK bit.
                    {
                        indexdata--;
                    }
                    index--;
                }
                break; //No need for additional exit state due to finite run time of for loop.
        }
        currState = nextState;
        if(i == events - 1)
        {
            writeArray[i] |= SDAWbit;
        }
    }
    return writeArray; //return the array
}
/*
    <uint32_t parseByteStream>
    This function parses through a byte stream and returns the readback data. It goes MSB first as both I2C and SPI start with MSB and count down.
    It accepts the following parameters.
    <uint8_t *byteStream>       A uint8_t pointer to the byteArray to be strobed. Typically this will NOT be the first index of the byte array.
    <uint32_t strobeIndex>      An unsigned integer containing which bit (0-7) is being strobed.
    <uint32_t strobeJump>       An unsigned integer containing how many events to skip between strobes. For SPI this is 2, for I2C this is 4.
    <uint32_t strobeCount>      An unsigned integer containing how many strobes to take.
    This function returns the data that was strobed.
    This function is NOT intended to be used outside of this source file.
*/
static uint32_t parseByteStream(uint8_t * byteStream, uint32_t StrobeIndex, uint32_t strobeJump, uint32_t strobeCount)
{
    uint32_t currentStrobe = 0;
    uint32_t readBackData = 0;
    uint32_t dataIndex = strobeCount - 1;
    for(int i = 0; i < strobeCount; i++)
    {
        if(((byteStream[currentStrobe] >> StrobeIndex) & 0x01) == 0x01) //A "1" was readback. Update the data variable accordingly.
        {
            readBackData |= (0x01 << dataIndex);
        }
        dataIndex--; //Decrement dataIndex each strobe.
        currentStrobe += strobeJump; //Increment each strobe (register address).
    }
    return readBackData;
}
/*
    <int32_t setImpedance>
    This function sets the impedance (Driver or receiver) for each pin of an FTDI device based off the contents of the PROT_DEV HANDLE.
    For SPI the Clock, CS and SDI bits are all set as low impedance (drivers) while the rest are left as high Z (receivers).
    For I2C the Clock and SDAnumW are set as low impedance (drivers) while the rest are left as high Z (receivers).
    It accepts the following parameters.
    <FT_HANDLE dev>             FT_HANDLE for FTDI device.
    <PROT_HANDLE prot>          PROT_HANDLE for SPI/I2C Device.
    This function returns a 0 if successful and a -1 if FT_SetBitMode does not return FT_OK
    This function is NOT intended to be used outside of this source file.
*/
static int32_t setImpedance(FT_HANDLE dev, PROT_HANDLE prot)
{
    struct PROT_DES protDev;
    uint8_t impedance = 0;
    memcpy(&protDev, prot, sizeof(struct PROT_DES)); //Local PROT_DES structure now has contents of handle copied to it.
    if(protDev.SPI_I2C == TRUE)
    {
        impedance |= (0x01 << protDev.CLKnum);             //Create the masks for each respective bit.
        impedance |= (0x01 << protDev.CSnum);
        impedance |= (0x01 << protDev.SDInum);
    }
    else
    {
        impedance |= (0x01 << protDev.SCLnum);             //Create the masks for each respective bit.
        impedance |= (0x01 << protDev.SDAnumW);
    }
    if(protDev.OE_EN == TRUE)
    {
        impedance |= (0x01 << protDev.OEnum);
    }
    if(FT_SetBitMode(dev, impedance, FT_BITMODE_SYNC_BITBANG) != FT_OK)
    {
        return -1;
    }
    return 0;
}
/*
    <int32_t wait4RxBuffer>
    This function waits for the RxBuffer to fill. It is used throughout the program for this purpose.
    It accepts the following parameters.
    <FT_HANDLE dev>         FT_HANDLE for FTDI device.
    <uint32_t count>        Unsigned 32 bit integer that contains the value the buffer needs to fill to to exit.
    This function returns a 0 if successful and a -1 if not.
    This function is NOT intended to be used outside of this source file.
*/
static int32_t wait4RxBuffer(FT_HANDLE dev, uint32_t count)
{
    FT_STATUS ftStatus;
    DWORD RxFIFO;
    ftStatus = FT_GetQueueStatus(dev, (DWORD*)&RxFIFO); // Call to D2XX API call that returns the number of bytes in the RX FIFO.
    while(RxFIFO < count) // This while loop will run until the RxFIFO is not full.
    {
        if(ftStatus != FT_OK)
            return -1;
        ftStatus = FT_GetQueueStatus(dev, &RxFIFO);
    }
    return 0;
}
